package org.apache.turbine.torque.engine.intake.model;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and 
 *    "Apache Turbine" must not be used to endorse or promote products 
 *    derived from this software without prior written permission. For 
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    "Apache Turbine", nor may "Apache" appear in their name, without 
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

// JDK
import java.io.*;
import java.util.*;

// Xerces
import org.xml.sax.*;
import org.xml.sax.helpers.*;

/**
 * A Class for holding data about a grouping of inputs used in an Application.
 *
 * @author <a href="mailto:jmcnally@collab.net>John McNally</a>
 * @version $Id: InputGroup.java,v 1.3 2001/03/09 11:54:22 jmcnally Exp $
 */
public class InputGroup
{
    private Vector fields;
    private Vector mapToObjects;
    private String defaultMapToObject;
    private AppData parent;
    private String classname;
    private String key;

    /**
     * Constructs a input group object
     */
    public InputGroup()
    {
        fields = new Vector();
        mapToObjects = new Vector(2);
    }

    /**
     * Load the input group object from an xml tag.
     */
    public void loadFromXML (Attributes attrib)
    {
        classname = attrib.getValue("class");
        key = attrib.getValue("key");
        
        String objName = attrib.getValue("mapToObject");
        if ( objName != null ) 
        {
            defaultMapToObject = objName;
            mapToObjects.add(objName);
        }
    }

    /**
     * Get the class name that handles this group
     */
    public String getClassName()
    {
        return classname;
    }

    /**
     * Set the class name that handles this group
     */
    public void setClassName(String newClass)
    {
        classname = newClass;
    }

    /**
     * Get the key used to reference this group in input (form)
     */
    public String getKey()
    {
        return key;
    }

    /**
     * Set the key used to reference this group in input (form)
     */
    public void setKey(String newKey)
    {
        key = newKey;
    }

    /**
     * A utility function to create a new field
     * from attrib and add it to this input group.
     */
    public XmlField addField(Attributes attrib)
    {
        XmlField field = new XmlField();
        field.loadFromXML(attrib);
        addField(field);

        return field;
    }

    /**
     * Adds a new field to the fields vector and set the
     * parent group of the field to the current group
     */
    public void addField(XmlField field)
    {
        field.setInputGroup (this);

        // if this field has an object defined for mapping,
        // add it to the list
        if ( field.getMapToObject() != null ) 
        {
            boolean isNewObject = true;
            for ( int i=0; i<mapToObjects.size(); i++ ) 
            {
                if ( mapToObjects.get(i).equals(field.getMapToObject()) ) 
                {
                    isNewObject = false;
                    break;
                }
            }
            if ( isNewObject ) 
            {
                mapToObjects.add(field.getMapToObject());
            }
        }
        // if a mapToProperty exists, set the object to this group's default
        else if( field.getMapToProperty() != null 
                 && defaultMapToObject != null )
        {
            field.setMapToObject(defaultMapToObject);
        }
        
        fields.addElement(field);
    }

    /**
     * Returns a collection of fields in this input group
     */
    public Vector getFields()
    {
        return fields;
    }

    /**
     * Utility method to get the number of fields in this input group
     */
    public int getNumFields()
    {
        return fields.size();
    }


    /**
     * Returns a Specified field.
     * @return Return a XmlField object or null if it does not exist.
     */
    public XmlField getField(String name)
    {
        String curName;

        for (Enumeration e = fields.elements() ; e.hasMoreElements() ;)
        {
            XmlField field = (XmlField) e.nextElement();
            curName = field.getRawName();
            if (curName.equals(name))
            {
                return field;
            }
        }
        return null;
    }

    /**
     * Returns true if the input group contains a spesified field
     */
    public boolean containsField(XmlField field)
    {
        return fields.contains (field);
    }

    /**
     * Returns true if the input group contains a specified field
     */
    public boolean containsField(String name)
    {
        return (getField (name) != null);
    }

    public Vector getMapToObjects()
    {
        return mapToObjects;
    }

    /**
     * Set the parent of the database
     */
    public void setAppData(AppData parent)
    {
        this.parent = parent;
    }

    /**
     * Get the parent of the input group
     */
    public AppData getAppData()
    {
        return parent;
    }

    /**
     * A String which might be used as a variable of this class
     */
    public String getVariable()
    {
        String firstChar = getClassName().substring(0,1).toLowerCase();
        return firstChar + getClassName().substring(1);
    }


    /**
     * Creates a string representation of this input group. This
     * is an xml representation.
     */
    public String toString()
    {
        StringBuffer result = new StringBuffer();

        result.append ("<group class=\"").append(classname);
        result.append(" key=\""+key+"\"");
        result.append(">\n");

        if (fields != null)
        {
            for (Enumeration e = fields.elements() ; e.hasMoreElements() ;)
            {
                result.append(e.nextElement());
            }
        }        

        result.append ("</group>\n");

        return result.toString();
    }


}
