-- `Topal': GPG/Pine integration
--
-- Copyright (C) 2001--2003  Phillip J. Brooke
--
--     This program is free software; you can redistribute it and/or modify
--     it under the terms of the GNU General Public License as published by
--     the Free Software Foundation; either version 2 of the License, or
--     (at your option) any later version.
--
--     This program is distributed in the hope that it will be useful,
--     but WITHOUT ANY WARRANTY; without even the implied warranty of
--     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--     GNU General Public License for more details.
--
--     You should have received a copy of the GNU General Public License
--     along with this program; if not, write to the Free Software
--     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

with Ada.IO_Exceptions;
with Ada.Strings.Fixed;
with Ada.Strings.Unbounded;
with Ada.Text_IO;
with Expanding_Array;
with Externals;             use Externals;
with Externals.GPG;
with Menus;                 use Menus;
with Misc;                  use Misc;
with Readline;

package body Keys is

   Key_List_Initial_Size : constant Positive := 200;

   type Menu_Array_Ptr is access all Keylist_Menus.MNA;

   -- Add a key to the list....
   procedure Add_Key (Key  : in     UBS;
                      List : in out Key_List) is
      Match : Boolean := False;
   begin
      Debug("+Add_Key");
      -- Is it a duplicate?
      Debug("Add_Key: Checking for duplicates");
      declare
         use type UBS;
      begin
         Debug("Add_Key: Approaching duplicates loop");
         for I in 1..List.Count loop
            Debug("Add_Key: In duplicates loop");
            Match := Match or Key = UAP.Value(List.KA, I);
         end loop;
      end;
      if not Match then
         -- Add the key.
         Debug("Add_Key: Adding a key");
         List.Count := List.Count + 1;
         UAP.Set(List.KA, List.Count, Key);
         Debug("Add_Key: Done adding a key");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Keys.Add_Key");
         raise;
   end Add_Key;

   -- Remove a key.  We then shuffle the array down.
   -- It will only remove the last key if there are multiple instances.
   procedure Remove_Key (Key  : in     UBS;
                         List : in out Key_List) is
      Key_Num : Integer;
      Key_Found : Boolean := False;
      use type UBS;
   begin
      for I in 1..List.Count loop
         if Key = UAP.Value(List.KA, I) then
            Key_Found := True;
            Key_Num := I;
         end if;
      end loop;
      if Key_Found then
         -- Shuffle the array over it.
         for I in Key_Num .. List.Count - 1 loop
            UAP.Set(List.KA, I, UAP.Value(List.KA, I+1));
         end loop;
         -- Reduce the number of keys.
         List.Count := List.Count - 1;
      else
         raise Key_Not_Found;
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Keys.Remove_Key");
         raise;
   end Remove_Key;

   -- Turn the list of keys to send into `-r <key>' string.
   function Processed_Recipient_List (List : in Key_List) return String is
      Result : UBS := Ada.Strings.Unbounded.Null_Unbounded_String;
      use type UBS;
   begin
      for I in 1..List.Count loop
         Result := Result & ToUBS(" -r ") & UAP.Value(List.KA, I) & ToUBS(" ");
      end loop;
      return ToStr(Result);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Keys.Processed_Recipient_List");
         raise;
   end Processed_Recipient_List;

   -- Get key fingerprint(s) for a key.  Add them.
   procedure Add_Keys_By_Fingerprint (Key_In : in     UBS;
                                      List   : in out Key_List;
                                      Found  :    out Boolean) is
      K          : String := ToStr(Key_In);
      Tempfile   : String := Temp_File_Name("findkey-akbf");
      TFH        : Ada.Text_IO.File_Type;
      L          : UBS;
      Match      : Boolean := False;
   begin
      Debug("Add_Keys_By_Fingerprint (A): starting...");
      -- First, try finding some keys for this Key_In.
      GPG.Findkey(Key    => K,
                  Target => Tempfile);
      Debug("Add_Keys_By_Fingerprint (A): GPG.Findkey returned okay");
      -- Now, open the file, and read in each line as a key for Add_Key.
      Ada.Text_IO.Open(File => TFH,
                       Mode => Ada.Text_IO.In_File,
                       Name => Tempfile);
      Debug("Add_Keys_By_Fingerprint (A): opened file successfully");
  Fingerprint_Loop:
      loop
         begin
            L := Unbounded_Get_Line(TFH);
            Debug("Add_Keys_By_Fingerprint: Adding " & ToStr(L));
            Add_Key(L, List);
            Match := True;
         exception
            when Ada.IO_Exceptions.End_Error =>
               exit Fingerprint_Loop;
         end;
      end loop Fingerprint_Loop;
      Ada.Text_IO.Close(TFH);
      Found := Match;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Keys.Add_Keys_By_Fingerprint (A)");
         raise;
   end Add_Keys_By_Fingerprint;

   procedure Add_Keys_By_Fingerprint (Key_In : in     UBS;
                                      List   : in out Key_List) is
      Found : Boolean;
   begin
      Add_Keys_By_Fingerprint(Key_In, List, Found);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Keys.Add_Keys_By_Fingerprint (B)");
         raise;
   end Add_Keys_By_Fingerprint;

   -- Add the secret keys to this key list.
   procedure Add_Secret_Keys (Key_In : in     UBS;
                              List   : in out Key_List) is
      K          : String := ToStr(Key_In);
      Tempfile   : String := Temp_File_Name("findkey-ask");
      TFH        : Ada.Text_IO.File_Type;
      L          : UBS;
      Match      : Boolean := False;
   begin
      -- First, try finding some _secret_ keys for this Key_In.
      GPG.Findkey_Secret(Key    => K,
                         Target => Tempfile);
      -- Now, open the file, and read in each line as a key for
      -- Add_Keys_By_Fingerprint (so we don't need to go hunting for the
      -- long codes.
      Ada.Text_IO.Open(File => TFH,
                       Mode => Ada.Text_IO.In_File,
                       Name => Tempfile);
  Fingerprint_Loop:
      loop
         begin
            L := Unbounded_Get_Line(TFH);
            Debug("Add_Keys_By_Fingerprint: Adding " & ToStr(L));
            Add_Keys_By_Fingerprint(L, List);
            Match := True;
         exception
            when Ada.IO_Exceptions.End_Error =>
               exit Fingerprint_Loop;
         end;
      end loop Fingerprint_Loop;
      Ada.Text_IO.Close(TFH);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Keys.Add_Secret_Keys");
         raise;
   end Add_Secret_Keys;

   -- List the keys and edit them as appropriate (include removing a key
   -- from that list, and adding one from the keyring.
   procedure List_Keys (List : in out Key_List) is
      Key_Head_Length : constant Positive := 65;
      subtype Key_Head is String(1..Key_Head_Length);
      package Key_Head_Array_Package
      is new Expanding_Array (Item => Key_Head);
      package KHA renames Key_Head_Array_Package;

      subtype Key_Head_Array is Key_Head_Array_Package.Big_Array;

      Key_Heads : Key_Head_Array;

      Key_Menu : Menu_Array_Ptr;

      procedure Generate_Key_List is
      begin
         Debug("+Generate_Key_List");
         -- First, we generate a current list of info files for the keys.
         for I in 1..List.Count loop
            declare
               Key_File_Name : String
                 := Temp_File_Name("key"
                                   & Trim_Leading_Spaces(Integer'Image(I)));
            begin
               GPG.Listkey(Key    => ToStr(UAP.Value(List.KA, I)),
                           Target => Key_File_Name);
               -- Now, dump into our `head' array the first line of each.
               declare
                  File_Handle : Ada.Text_IO.File_Type;
                  First_Line  : UBS;
               begin
                  Ada.Text_IO.Open(File => File_Handle,
                                   Mode => Ada.Text_IO.In_File,
                                   Name => Key_File_Name);
                  First_Line := Unbounded_Get_Line(File_Handle);
                  Ada.Text_IO.Close(File_Handle);
                  declare
                     FL  : String := ToStr(First_Line);
                     use Ada.Strings.Fixed;
                  begin
                     if FL'Length >= Key_Head_Length then
                        KHA.Set(Key_Heads, I, FL(FL'First..
                                             FL'First+Key_Head_Length-1));
                     else
                        KHA.Set(Key_Heads, I, FL
                            & (Key_Head_Length - FL'Length) * ' ');
                     end if;
                  end;
               end;
            end;
         end loop;
         -- Now, create a menu with those key_heads.
         Key_Menu := new Keylist_Menus.MNA(1..List.Count);
         for I in 1..List.Count loop
            Key_Menu(I) := ToUBS("Details: " & KHA.Value(Key_Heads, I));
         end loop;
         Debug("-Generate_Key_List");
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Keys.List_Keys.Generate_Key_List");
            raise;
      end Generate_Key_List;

      Selection    : Keylist_Menus.CLM_Return;
      Selected     : Integer;
      SK_Selection : Specific_Key_Index;
   begin
      Debug("+List_Keys");
      KHA.Create(Key_Heads, Key_List_Initial_Size);
      Generate_Key_List;
  List_Keys_Loop:
      loop
         Selection := Keylist_Menu(Key_Menu.all);
         if not Selection.Is_Num then
            case Selection.I is
               when Done =>
                  exit List_Keys_Loop;
               when AddPattern => -- Add key from main keyring.
                  declare
                     New_Key : UBS;
                     use Ada.Text_IO;
                  begin
                     New_Line(1);
                     Put_Line("Main key ring access:");
                     New_Key
                       := ToUBS(Readline.Get_String("Type GPG search pattern: "));
                     Add_Keys_By_Fingerprint(New_Key, List);
                     Generate_Key_List;
                  end;
               when AddSearch => -- Do a search, then select a key.
                  declare
                     Search_List : Key_List;
                     Pattern     : UBS;
                     The_Key     : UBS;
                     Aborted     : Boolean;
                  begin
                     Empty_Keylist(Search_List);
                     Pattern := ToUBS(Readline.Get_String("Type GPG search pattern: "));
                     Add_Keys_By_Fingerprint(Pattern, Search_List);
                     Select_Key_From_List(Search_List, The_Key, Aborted);
                     if not Aborted then
                        Add_Key(The_Key, List);
                        Generate_Key_List;
                     end if;
                  end;
            end case;
         else
            -- Selection key menu.
            Selected := Selection.N;
        Specific_Key_Loop:
            loop
               SK_Selection := Specific_Key_Menu1("Key: "
                                                  & ToStr(Key_Menu(Selected))
                                                  & NL
                                                  & "(d) Display details of key with less, (v) Verbosely"
                                                  & NL
                                                  & "(r) Remove key from list   (kql) Return to key list  "
                                                  & NL);
               case SK_Selection is
                  when Done =>
                     exit Specific_Key_Loop;
                  when Display =>
                     GPG.Viewkey(ToStr(UAP.Value(List.KA, Selected)));
                  when DisplayVerbose =>
                     GPG.Viewkey(ToStr(UAP.Value(List.KA, Selected)),
                                 Verbose => True);
                  when Remove =>
                     Remove_Key(UAP.Value(List.KA, Selected),
                                List);
                     Generate_Key_List;
                     exit Specific_Key_Loop;
                  when SSelect =>
                     Error("Menu should not have allowed SSelect here");
               end case;
            end loop Specific_Key_Loop;
         end if;
      end loop List_Keys_Loop;
      Debug("-List_Keys");
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Keys.List_Keys");
         raise;
   end List_Keys;

   -- List the keys.  Either return with Aborted true, or
   -- The_Fingerprint set to the chosen fingerprint.
   procedure Select_Key_From_List (List            : in out Key_List;
                                   The_Fingerprint :    out UBS;
                                   Aborted         :    out Boolean) is
      Key_Head_Length : constant Positive := 65;
      subtype Key_Head is String(1..Key_Head_Length);
      package Key_Head_Array_Package
      is new Expanding_Array (Item => Key_Head);
      package KHA renames Key_Head_Array_Package;

      subtype Key_Head_Array is Key_Head_Array_Package.Big_Array;

      Key_Heads : Key_Head_Array;

      Key_Menu : Menu_Array_Ptr;

      procedure Generate_Key_List is
      begin
         Debug("+Generate_Key_List");
         -- First, we generate a current list of info files for the keys.
         for I in 1..List.Count loop
            declare
               Key_File_Name : String
                 := Temp_File_Name("key"
                                   & Trim_Leading_Spaces(Integer'Image(I)));
            begin
               GPG.Listkey(Key    => ToStr(UAP.Value(List.KA, I)),
                           Target => Key_File_Name);
               -- Now, dump into our `head' array the first line of each.
               declare
                  File_Handle : Ada.Text_IO.File_Type;
                  First_Line  : UBS;
               begin
                  Ada.Text_IO.Open(File => File_Handle,
                                   Mode => Ada.Text_IO.In_File,
                                   Name => Key_File_Name);
                  First_Line := Unbounded_Get_Line(File_Handle);
                  Ada.Text_IO.Close(File_Handle);
                  declare
                     FL  : String := ToStr(First_Line);
                     use Ada.Strings.Fixed;
                  begin
                     if FL'Length >= Key_Head_Length then
                        KHA.Set(Key_Heads, I, FL(FL'First..
                                             FL'First+Key_Head_Length-1));
                     else
                        KHA.Set(Key_Heads, I, FL
                            & (Key_Head_Length - FL'Length) * ' ');
                     end if;
                  end;
               end;
            end;
         end loop;
         -- Now, create a menu with those key_heads.
         Key_Menu := new Keylist_Menus.MNA(1..List.Count);
         for I in 1..List.Count loop
            Key_Menu(I) := ToUBS("Details: " & KHA.Value(Key_Heads, I));
         end loop;
         Debug("-Generate_Key_List");
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Keys.Select_Key_From_List.Generate_Key_List");
            raise;
      end Generate_Key_List;

      Selection        : Keylist_Menus.CLM_Return;
      Selected         : Integer;
      SK_Selection     : Specific_Key_Index;
      Abort_Real       : Boolean := False;
      Fingerprint_Real : UBS;
   begin
      Debug("+List_Keys");
      KHA.Create(Key_Heads, Key_List_Initial_Size);
      Generate_Key_List;
  List_Keys_Loop:
      loop
         Selection := Keylist_Menu2(Key_Menu.all);
         if (not Selection.Is_Num) then
            case Selection.I is
               when Done =>
                  Abort_Real := True;
                  exit List_Keys_Loop;
               when AddPattern =>
                  Error("Menu should not have allowed AddPattern here");
               when AddSearch =>
                  Error("Menu should not have allowed AddSearch here");
            end case;
         else
            -- Selection key menu.
            Selected := Selection.N;
        Specific_Key_Loop:
            loop
               SK_Selection := Specific_Key_Menu2("Key: "
                                                  & ToStr(Key_Menu(Selected))
                                                  & NL
                                                  & "(d) Display details of key with less, (v) Verbosely"
                                                  & NL
                                                  & "(s) Select this key         (kql) Return to key list  "
                                                  & NL);
               case SK_Selection is
                  when Done =>
                     exit Specific_Key_Loop;
                  when Display =>
                     GPG.Viewkey(ToStr(UAP.Value(List.KA, Selected)));
                  when DisplayVerbose =>
                     GPG.Viewkey(ToStr(UAP.Value(List.KA, Selected)),
                                 Verbose => True);
                  when SSelect =>
                    Fingerprint_Real := UAP.Value(List.KA, Selected);
                    exit List_Keys_Loop;
                  when Remove =>
                     Error("Menu should not have allowed Remove here");
                  end case;
               end loop Specific_Key_Loop;
            end if;
         end loop List_Keys_Loop;
      The_Fingerprint := Fingerprint_Real;
      Aborted := Abort_Real;
      Debug("-List_Keys");
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Keys.List_Keys");
         raise;
   end Select_Key_From_List;

   -- Use keylist.
   procedure Use_Keylist (Recipients : in     UBS_Array;
                          List       : in out Key_List) is
      use Ada.Text_IO;
      Found_Key : array (Recipients'Range) of Boolean
        := (others => False);
   begin
      Debug("+Use_Keylist");
      -- New key list.
      -- Do something with the keylist.  Given a list of email addresses,
      -- open the relevant file (er, look in the array), and keep those key
      -- ids.
      for J in 1..Config.AKE_Count loop
         for I in Recipients'Range loop
            declare
               use type UBS;
               Found : Boolean;
            begin
               if UAP.Value(Config.AKE_Email, J) = Recipients(I) then
                  Add_Keys_By_Fingerprint(UAP.Value(Config.AKE_Key, J),
                                          List,
                                          Found);
                  Found_Key(I) := Found_Key(I) or Found;
               end if;
            end;
         end loop;
      end loop;
      -- Now, run through the recipients that we _haven't_ found a key for.
      for I in Recipients'Range loop
         if Found_Key(I) then
            Put_Line("Info: Added keylist key(s) for recipient: `"
                     & ToStr(Recipients(I)) & "'");
         else
            declare
               Search_List : Key_List;
               use type UBS;
            begin
               Empty_Keylist(Search_List);
               Add_Keys_By_Fingerprint(Recipients(I), Search_List);
               -- Remove any on the XK list.  Dead easy: simply run
               -- through the entire XK list saying `Remove Key'.
               for J in 1 .. Config.XK_Count loop
                  Remove_Key(UAP.Value(Config.XK_Key, J), Search_List);
               end loop;
               if Count(Search_List) > 0 then
                  Put_Line("Info: Adding non-keylist key(s) for recipient: `"
                           & ToStr(Recipients(I)) & "'");
                  -- Add the keys in Search_List to List.
                  for J in 1 .. Search_List.Count loop
                     Add_Key(UAP.Value(Search_List.KA, J), List);
                  end loop;
               else
                  Put_Line("Warning: Cannot find key for recipient: `"
                           & ToStr(Recipients(I)) & "'");
               end if;
            end;
         end if;
      end loop;
      Debug("-Use_Keylist");
   exception
      when Ada.IO_Exceptions.Name_Error =>
         Put_Line("Warning: Can't open keylist file");
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Keys.Use_Keylist");
         raise;
   end Use_Keylist;

   procedure Empty_Keylist (List : in out Key_List) is
   begin
      UAP.Create(List.KA, Key_List_Initial_Size);
      List.Count := 0;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Keys.Empty_Keylist");
         raise;
   end;

   function Count (List : in Key_List) return Natural is
   begin
      return List.Count;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Keys.Count");
         raise;
   end Count;

end Keys;


