-- `Topal': GPG/Pine integration
--
-- Copyright (C) 2001,2002  Phillip J. Brooke
--
--     This program is free software; you can redistribute it and/or modify
--     it under the terms of the GNU General Public License as published by
--     the Free Software Foundation; either version 2 of the License, or
--     (at your option) any later version.
--
--     This program is distributed in the hope that it will be useful,
--     but WITHOUT ANY WARRANTY; without even the implied warranty of
--     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--     GNU General Public License for more details.
--
--     You should have received a copy of the GNU General Public License
--     along with this program; if not, write to the Free Software
--     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

with Ada.IO_Exceptions;
with Ada.Text_IO;
with Misc;

package body Externals.Simple is

   procedure Cat (D : in String) is
      use Ada.Text_IO;
      use Misc;
      F1 : File_Type;
      TL : UBS;
   begin
      Open(File => F1,
           Mode => In_File,
           Name => D);
  File_Loop:
      loop
         begin
            TL := Unbounded_Get_Line(F1);
            Put_Line(ToStr(TL));
         exception
            when Ada.IO_Exceptions.End_Error =>
               exit File_Loop;
         end;
      end loop File_Loop;
      Close(F1);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Cat");
         raise;
   end Cat;

   procedure Cat_Out (D1 : in String;
                      D2 : in String) is
      use Ada.Text_IO;
      use Misc;
      F1, F2 : File_Type;
      TL : UBS;
   begin
      Open(File => F1,
           Mode => In_File,
           Name => D1);
      Create(File => F2,
             Mode => Out_File,
             Name => D2);
  File_Loop:
      loop
         begin
            TL := Unbounded_Get_Line(F1);
            Put_Line(F2, ToStr(TL));
         exception
            when Ada.IO_Exceptions.End_Error =>
               exit File_Loop;
         end;
      end loop File_Loop;
      Close(F1);
      Close(F2);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Cat_Out");
         raise;
   end Cat_Out;

   -- Equivalent of `cat > D'.
   procedure Cat_Stdin_Out (D : in String) is
      use Ada.Text_IO;
      use Misc;
      F2 : File_Type;
      TL : UBS;
   begin
      Create(File => F2,
             Mode => Out_File,
             Name => D);
  File_Loop:
      loop
         begin
            TL := Unbounded_Get_Line;
            Put_Line(F2, ToStr(TL));
         exception
            when Ada.IO_Exceptions.End_Error =>
               exit File_Loop;
         end;
      end loop File_Loop;
      Close(F2);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Cat_Stdin_Out");
         raise;
   end Cat_Stdin_Out;

   procedure Cat_Append (D1 : in String;
                         D2 : in String) is
      use Ada.Text_IO;
      use Misc;
      F1, F2 : File_Type;
      TL : UBS;
   begin
      Open(File => F1,
           Mode => In_File,
           Name => D1);
      begin
         Open(File => F2,
              Mode => Append_File,
              Name => D2);
      exception
         when Ada.IO_Exceptions.Name_Error =>
            -- Try again with a create.
            Create(File => F2,
                   Mode => Append_File,
                   Name => D2);
      end;
  File_Loop:
      loop
         begin
            TL := Unbounded_Get_Line(F1);
            Put_Line(F2, ToStr(TL));
         exception
            when Ada.IO_Exceptions.End_Error =>
               exit File_Loop;
         end;
      end loop File_Loop;
      Close(F1);
      Close(F2);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Cat_Append");
         raise;
   end Cat_Append;

   procedure Chmod (P : in String;
                    D : in String) is
   begin
      if ForkExec(Misc.Value_Nonempty(Config.Chmod_Binary),
                  UBS_Array'(0 => ToUBS("chmod"),
                             1 => ToUBS(P),
                             2 => ToUBS(D))) /= 0 then
         Misc.Error("`chmod " & P & " " & D & "' failed.");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Chmod");
         raise;
   end Chmod;

   procedure Clear is
   begin
      if ForkExec(Misc.Value_Nonempty(Config.Clear_Binary),
                  UBS_Array'(0 => ToUBS("clear"))) /= 0 then
         Misc.Error("`clear' failed.");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Clear");
         raise;
   end Clear;

   procedure Date_Append (D : in String) is
   begin
      if ForkExec_Append(Misc.Value_Nonempty(Config.Date_Binary),
                         UBS_Array'(0 => ToUBS("date")),
                         D) /= 0 then
         Misc.Error("`date >> " & D & "' failed.");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Date_Append");
         raise;
   end Date_Append;

   function Date_String return String is
      use Ada.Text_IO;
      use Misc;
      Target : String := Temp_File_Name("thedate");
      TL     : UBS;
      F1     : File_Type;
   begin
      if ForkExec_Out(Misc.Value_Nonempty(Config.Date_Binary),
                      UBS_Array'(0 => ToUBS("date"),
                                 1 => ToUBS("+%Y%m%d%H%M%S")),
                      Target) /= 0 then
         Misc.Error("`date '+%Y%m%d%H%M%S' >" & Target & "' failed.");
      end if;
      -- Now, extract that string.
      Open(File => F1,
           Mode => In_File,
           Name => Target);
      -- There should be exactly one line that has the date in it.
      TL := Unbounded_Get_Line(F1);
      Close(F1);
      -- Return that string.
      return ToStr(TL);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Date_String");
         raise;
   end Date_String;

   function Diff_Brief (F1 : in String;
                        F2 : in String) return Boolean is
      E : Integer;
   begin
      E := ForkExec_Out(Misc.Value_Nonempty(Config.Diff_Binary),
                        UBS_Array'(0 => ToUBS("diff"),
                                   1 => ToUBS("--brief"),
                                   2 => ToUBS(F1),
                                   3 => ToUBS(F2)),
                        Target => "/dev/null");
      case E is
         when 0 =>  -- No changes.
            return False;
         when 1 =>  -- There are changes.
            return True;
         when others => -- There's a problem.
            Misc.Error("`diff --brief " & F1 & "  " & F2 & " >/dev/null' failed.");
            return False; -- Won't be executed.
      end case;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Diff_Brief");
         raise;
   end Diff_Brief;

   procedure Dos2Unix_U (D : in String) is
   begin
      if ForkExec(Misc.Value_Nonempty(Config.Dos2Unix_Binary),
                  UBS_Array'(0 => ToUBS("dos2unix"),
                             1 => ToUBS("-u"),
                             2 => ToUBS(D))) /= 0 then
         Misc.Error("`dos2unix -u " & D & "' failed.");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Dos2Unix_U");
         raise;
   end Dos2Unix_U;

   procedure Dos2Unix (D : in String) is
   begin
      if ForkExec(Misc.Value_Nonempty(Config.Dos2Unix_Binary),
                  UBS_Array'(0 => ToUBS("dos2unix"),
                             1 => ToUBS(D))) /= 0 then
         Misc.Error("`dos2unix " & D & "' failed.");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Dos2Unix");
         raise;
   end Dos2Unix;

   procedure Echo_Append (E : in String;
                          D : in String) is
      use Ada.Text_IO;
      use Misc;
      F : File_Type;
   begin
      begin
         Open(File => F,
              Mode => Append_File,
              Name => D);
      exception
         when Ada.IO_Exceptions.Name_Error =>
            -- Try again with a create.
            Create(File => F,
                   Mode => Append_File,
                   Name => D);
      end;
      Put_Line(F, E);
      Close(F);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Echo_Append");
         raise;
   end Echo_Append;

   procedure Echo_Out (E : in String;
                       D : in String) is
      use Ada.Text_IO;
      use Misc;
      F : File_Type;
   begin
      Create(File => F,
             Mode => Out_File,
             Name => D);
      Put_Line(F, E);
      Close(F);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Echo_Out");
         raise;
   end Echo_Out;

   procedure Echo_Out_N (E : in String;
                            D : in String) is
      use Misc;
      use Character_IO;
      F : File_Type;
   begin
      Create(File => F,
             Mode => Out_File,
             Name => D);
      Character_IO_Put(F, E);
      Close(F);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Echo_Out_N");
         raise;
   end Echo_Out_N;

   function Grep_I_InOut (T : in String;
                          D1 : in String;
                          D2 : in String) return Integer is
   begin
      return ForkExec_InOut(Misc.Value_Nonempty(Config.Grep_Binary),
                            UBS_Array'(0 => ToUBS("grep"),
                                 1 => ToUBS("-i"),
                                       2 => ToUBS(T)),
                            Source => D1,
                            Target => D2);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Grep_I_InOut");
         raise;
   end Grep_I_InOut;

   procedure Mkdir_P (D : in String) is
   begin
      if ForkExec(Misc.Value_Nonempty(Config.Mkdir_Binary),
                  UBS_Array'(0 => ToUBS("mkdir"),
                             1 => ToUBS("-p"),
                             2 => ToUBS(D))) /= 0 then
         Misc.Error("`mkdir -p " & D & "' failed.");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Mkdir_P");
         raise;
   end Mkdir_P;

   procedure Mv_F (D1 : in String;
                   D2 : in String) is
   begin
      if not Test_F(D1) then
         Misc.Error("`" & D1 & "' does not exist to be moved!");
      end if;
      if ForkExec(Misc.Value_Nonempty(Config.Mv_Binary),
                  UBS_Array'(0 => ToUBS("mv"),
                             1 => ToUBS("-f"),
                             2 => ToUBS(D1),
                             3 => ToUBS(D2))) /= 0 then
         Misc.Error("`mv -f " & D1 & " " & D2 & "' failed.");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Mv_F");
         raise;
   end Mv_F;

   -- View a file with a pager.
   procedure Pager (F : in String) is
   begin
      if ForkExec(Misc.Value_Nonempty(Config.Less_Binary),
                  UBS_Array'(0 => ToUBS("less"),
                             1 => ToUBS(F))) /= 0 then
         Misc.Error("less failed! (ff4)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Pager");
         raise;
   end Pager;

   procedure Rm_File (F : in String) is
      Dummy : Integer;
   begin
      -- We ignore Rm's return code.
      Dummy := ForkExec(Misc.Value_Nonempty(Config.Rm_Binary),
                        UBS_Array'(0 => ToUBS("rm"),
                                   1 => ToUBS("-f"),
                                   2 => ToUBS(F)));
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Rm_File");
         raise;
   end Rm_File;

   procedure Rm_Glob (Pattern : in String) is
      Files : UBS_Array := Glob(Pattern);
      Dummy : Integer;
   begin
      -- Now, rather than deleting all of these at once, we'll give the list
      -- to a single instantiation of rm.
      Misc.Debug("Rm_Glob: start: count of files is "
                 & Integer'Image(Files'Length));
      -- Do nothing if there are no files....
      if Files'Length /= 0 then
      -- Construct a suitable array.
         declare
            FEA : UBS_Array(0 .. Files'Length + 2);
         begin
            FEA(0) := ToUBS("rm");
            FEA(1) := ToUBS("-f");
            for I in 1 .. Files'Length loop
               -- When I = 1, we refer to FEA(2 = I + 1)
               -- and Files(Files'First = Files''First + I - 1).
               FEA(I + 1) := Files(Files'First + I -1);
               -- We ignore Rm's return code.
               Dummy := ForkExec(Misc.Value_Nonempty(Config.Rm_Binary), FEA);
            end loop;
         end;
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Rm_Glob");
         raise;
   end Rm_Glob;

   procedure Rm_Tempfiles is
   begin
      -- We ignore Rm's return code.
      Rm_Glob(ToStr(Topal_Directory) & "/temp*");
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Rm_Tempfiles");
         raise;
   end Rm_Tempfiles;

   procedure Rm_Tempfiles_PID is
   begin
      -- We ignore Rm's return code.
      Rm_Glob(ToStr(Topal_Directory) & "/temp"
              & Misc.Trim_Leading_Spaces(Integer'Image(Our_PID))
              & "*");
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Rm_Tempfiles_PID");
         raise;
   end Rm_Tempfiles_PID;

   procedure Rm_Cachefiles is
   begin
      -- We ignore Rm's return code.
      Rm_Glob(ToStr(Topal_Directory) & "/cache/*");
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Rm_Cachefiles");
         raise;
   end Rm_Cachefiles;

   -- General sed invocation.  Only accepts a SINGLE argument.
   procedure Sed_InOut (A      : in String;
                        Source : in String;
                        Target : in String) is
   begin
      if ForkExec_InOut(Misc.Value_Nonempty(Config.Sed_Binary),
                        UBS_Array'(0 => ToUBS("sed"),
                                   1 => ToUBS(A)),
                        Source => Source,
                        Target => Target) /=  0 then
         Misc.Error("Sed failed! (ff6)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Sed_InOut");
         raise;
   end Sed_InOut;

   procedure Stty_Sane is
   begin
      if ForkExec(Misc.Value_Nonempty(Config.Stty_Binary),
               UBS_Array'(0 => ToUBS("stty"),
                          1 => ToUBS("sane"))) /= 0 then
         Misc.Error("`stty sane' failed.");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Stty_Sane");
         raise;
   end;

   function Test_D (D : in String)  return Boolean is
   begin
      return ForkExec(Misc.Value_Nonempty(Config.Test_Binary),
                      UBS_Array'(0 => ToUBS("test"),
                                 1 => ToUBS("-d"),
                                 2 => ToUBS(D))) = 0;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Test_D");
         raise;
   end Test_D;

   function Test_F (D : in String)  return Boolean is
   begin
      return ForkExec(Misc.Value_Nonempty(Config.Test_Binary),
                      UBS_Array'(0 => ToUBS("test"),
                                 1 => ToUBS("-f"),
                                 2 => ToUBS(D))) = 0;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Test_F");
         raise;
   end Test_F;

   function Test_R (D : in String) return Boolean is
   begin
      return ForkExec(Misc.Value_Nonempty(Config.Test_Binary),
                      UBS_Array'(0 => ToUBS("test"),
                                 1 => ToUBS("-r"),
                                 2 => ToUBS(D))) = 0;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Test_R");
         raise;
   end Test_R;

   function Test_S (D : in String) return Boolean is
   begin
      return ForkExec(Misc.Value_Nonempty(Config.Test_Binary),
                      UBS_Array'(0 => ToUBS("test"),
                                 1 => ToUBS("-S"),
                                 2 => ToUBS(D))) = 0;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.Test_S");
         raise;
   end Test_S;

   -- View a MIME file.
   procedure View_MIME (F : String) is
   begin
      if ForkExec(Misc.Value_Nonempty(Config.Metamail_Binary),
                  UBS_Array'(0 => ToUBS("metamail"),
                             1 => ToUBS(F))) /= 0 then
         Misc.Error("metamail failed! (ff5)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Simple.View_MIME");
         raise;
   end View_MIME;

end Externals.Simple;
