/*
 * snes9express
 * profiler.cpp
 * (C) 1998-1999, David Nordlund
 * Licensed under the "Artistic" license
 * To view the details of the Artistic license, see the file
 * ./ARTISTIC, or visit http://www.opensource.org/artistic-license.html
 */

#include <iostream>
using namespace std;
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>

#ifdef FR_GNOME
# include <gnome.h>
#else
# include <glib.h>
# include "pix/ball.xpm"
# include "pix/del.xpm"
# include "pix/def.xpm"
# include "pix/tmp.xpm"
#endif
#include "profiler.h"
#include "pix/logo.xpm"

char *Default="default:", *Saved="save:", *Tmp="tmp:", *Deleted="delete:";
static char*NewListData[] = {
	"error:", "[ Anonymous Profile ]", PROG " error!", (char*)0
};

/// Create the Profiler GUI
s9x_Profiler::s9x_Profiler(s9x_Interface*parent):
fr_Window(parent, PROG " Profiler"),
#ifdef FR_GNOME
ImgApply(this, GNOME_STOCK_BUTTON_APPLY),
ImgSave(this, GNOME_STOCK_PIXMAP_SAVE),
ImgDel(this, GNOME_STOCK_PIXMAP_TRASH),
ImgDef(this, GNOME_STOCK_PIXMAP_JUMP_TO),
ImgTmp(this, GNOME_STOCK_PIXMAP_NEW),
#else
ImgApply(this, ball_xpm),
ImgSave(this, ball_xpm),
ImgDel(this, del_xpm),
ImgDef(this, def_xpm),
ImgTmp(this, tmp_xpm),
#endif
Logo(this, "[logo]", logo_xpm),
ProfName(this, (char*)0, "", false, S9X_PROFILE_NAME_LEN),
BtnOK(this, "Ok"),
BtnCancel(this, "Cancel"),
BtnApply(this, "Apply", ImgApply),
BtnSave(this, "Save", ImgSave),
BtnDef(this, "Default", ImgDef),
BtnDel(this, "Delete", ImgDel),
BtnBox(this),
ProfileList(this, 3, false),
ProfileFile("profiles")
{
   fr_Separator *Separator;
   fr_Box *Box;
	
   InLoading = false;
   SetIcon("Profiler", logo_xpm);
   SetGridSize(9, 9, false);
   SetPadding(9, 9);
   Pad();

   NewProfileName = "[ Current ]";

   CreateProfileList();
   Pack(ProfileList, 2, 0, 9, 5);

   CreateButtonBar();
   Pack(BtnBox, 0, 1, 2, 4);

   ProfName.AddListener(this);
   ProfName.SetTooltip("To rename the selected profile, type a new name here");
   ProfName.AddLabel("Profile Name:");
   Pack(ProfName, 2, 5, 9, 6);

   SetPadding(3, 3);
   Separator = new fr_Separator(this, fr_Horizontal);
   SetStretch(Fill, Fill);
   Pack(*Separator, 1, 6, 9, 7);
   SetStretch(Grow, Grow);

   Box = new fr_Box(this);
   Box->SetGridSize(1, 1, false);
   Box->Pack(Logo);
   Box->AddBorderBevelIn();
   Box->SetVisibility(true);
   Pack(*Box, 0, 6, 1, 9);

   BtnOK.SetTooltip("Save profile changes, Apply the selected profile, and close the Profiler");
   BtnCancel.SetTooltip("Close the Profiler, but do not save profile changes");
   BtnOK.AddListener(this);
   BtnCancel.AddListener(this);
   fr_ButtonBox ButtonBox(this);
   ButtonBox.AddButton(BtnOK, true);
   ButtonBox.AddButton(BtnCancel);
   Pack(ButtonBox, 4, 7, 9, 9);
}

s9x_Profiler::~s9x_Profiler() {
}

/// Create the list of profiles
void s9x_Profiler::CreateProfileList() {	
   static char*titles[] = {
      "", "Profile", "Profile Options", (char*)0
   };
   ProfileList.SetHeaders(titles);
   ProfileList.SetRowHeight(22);
   ProfileList.SetColumnAlign(0, fr_DataTable::Center);
   ProfileList.SetColumnWidth(0,  40);
   ProfileList.SetColumnWidth(1, 300);
   ProfileList.SetColumnWidth(2);
   ProfileList.AllowReorder(true);
   ProfileList.SetSize(360, 240);
   ProfileList.SetVisibility(true);
   ProfileList.AddListener(this);
}

/// Create the button box
void s9x_Profiler::CreateButtonBar() {
   BtnApply.SetTooltip("Apply the selected profile(s) to the snes9express"
		       " interface.  If multiple profiles are selected,"
		       " they will be merged.");
   BtnApply.AddListener(this);

   BtnSave.SetTooltip("Mark the selected profile(s) to be saved");
   BtnSave.AddListener(this);
   BtnDef.SetTooltip("Mark the selected profile to be the Default "
		      "profile.  The Default profile will be loaded "
		      "automatically when " PROG " is started");
   BtnDef.AddListener(this);
   BtnDel.SetTooltip("Mark the selected profile(s) not to be saved");
   BtnDel.SetEditable(false);
   BtnDel.AddListener(this);	

   BtnBox.SetGridSize(1, 4, true);
   BtnBox.SetPadding(3, 3);
   BtnBox.Pack(BtnApply);
   BtnBox.Pack(BtnSave);
   BtnBox.Pack(BtnDef);
   BtnBox.Pack(BtnDel);
   BtnBox.Pad();
   BtnBox.AddBorderBevelIn();
   BtnBox.SetVisibility(true);
}

/// Load the list of profiles from disk into memory
int s9x_Profiler::LoadList() {
   int DefaultRow, fl;
   char *bufvar, *bufval, fbuf[256];
   char NameHolder[S9X_PROFILE_NAME_LEN];
   char DefaultProfileName[S9X_PROFILE_NAME_LEN];
	
   ProfileList.RemoveAll();

   NameHolder[0] = 0;
   DefaultProfileName[0] = 0;
   DefaultRow = -1;
   fl = 0;
   
   try {
      ProfileFile.open(ios::in);
   
      ProfileList.Freeze(true);
      ProfileList.SetEditable(false);
   
      InLoading = true;
      while(ProfileFile.getline(fbuf, 255)) {
	 fl++;
	 bufvar = strtok(fbuf, " :=\t");
	 bufval = strtok(NULL, "#\n\r");
	 if(!bufvar)
	   continue;
	 if(!g_strcasecmp(bufvar, "PROFILE")) {
	    strcpy(NameHolder, bufval);
	 } else if(!g_strcasecmp(bufvar, "DEFAULT")) {
	    if(DefaultProfileName[0])
	      g_warning("Duplicate %s, in %s:%d",
			bufvar, ProfileFile.GetFilename(), fl);
	    strcpy(DefaultProfileName, bufval);
	 } else if(!g_strcasecmp(bufvar, "OPTIONS")) {
	    if((!bufval)||(bufval[0]==0))
	      continue;
	    if(!g_strcasecmp(NameHolder, DefaultProfileName))
	      DefaultRow = ProfileList.CountRows();
	    ProfileList.AddRow(NewListData);
	    SetProfile(ProfileList.CountRows()-1, S9X_PROFILE_STATE_SAVED,
		       NameHolder, bufval);
	    strcpy(NameHolder, NewListData[1]);
	 };
      };
      ProfileFile.close();
   } catch (...) {
      fr_Mesg("error opening profiles");
   };
   time(&LastModified);
   if(DefaultRow>-1)
     SetRowState(DefaultRow, S9X_PROFILE_STATE_DEFAULT);
   ProfileList.Freeze(false);
   ProfileList.SetEditable(true);
   InLoading = false;
   return 0;
}

/// Set the profile data for a given row
void s9x_Profiler::SetProfile(int r, s9x_ProfileState s, char*N, char*A) {
   SetRowState(r, s);
   ProfileList.SetCell(r, 1, N);
   ProfileList.SetCell(r, 2, A);
}

/// Set the state of a given row
void s9x_Profiler::SetRowState(int row, s9x_ProfileState s) {
   if((row<0)||(row>=ProfileList.CountRows()))
     return;
   switch(s) {
    case S9X_PROFILE_STATE_DEFAULT:
      ProfileList.SetCell(row, 0, Default, &ImgDef);
      break;
    case S9X_PROFILE_STATE_SAVED:
      ProfileList.SetCell(row, 0, Saved, &ImgSave);
      break;
    case S9X_PROFILE_STATE_DELETED:
      ProfileList.SetCell(row, 0, Deleted, &ImgDel);
      break;
    case S9X_PROFILE_STATE_TEMP:
      ProfileList.SetCell(row, 0, Tmp, &ImgTmp);
      break;
   };
}

/// Get the state of a given row
s9x_ProfileState s9x_Profiler::GetRowState(int row) {
   if((row<0)||(row>=ProfileList.CountRows()))
     return S9X_PROFILE_STATE_TEMP;

   fr_Image *p = ProfileList.GetCellPic(row, 0);
   if(p == &ImgDef)
     return S9X_PROFILE_STATE_DEFAULT;
   if(p == &ImgSave)
     return S9X_PROFILE_STATE_SAVED;
   if(p == &ImgDel)
     return S9X_PROFILE_STATE_DELETED;

   return S9X_PROFILE_STATE_TEMP;
}

/// Get the name of a given row
char*s9x_Profiler::GetRowName(int row) {
   if((row<0)||(row>=ProfileList.CountRows()))
     return (char*)0;
   return ProfileList.GetCell(row, 1);
}

/// Get the args of a given row
char*s9x_Profiler::GetRowArgsEncoded(int row) {
   if((row<0)||(row>=ProfileList.CountRows()))
     return (char*)0;
   return ProfileList.GetCell(row, 2);
}

/// Get the ArgList for a given row
fr_ArgList*s9x_Profiler::GetRowArgList(int row) {	
   if((row<0)||(row>=ProfileList.CountRows()))
     return (fr_ArgList*)0;
   char *a = ProfileList.GetCell(row, 2);
   if(!a)
     return (fr_ArgList*)0;
   return new fr_ArgList(a, true);
}

/// Find the row with the default profile
int s9x_Profiler::GetDefaultRow() {
   int i, CountProfiles = ProfileList.CountRows();
   for(i=0; i<CountProfiles; i++)
     if(GetRowState(i)==S9X_PROFILE_STATE_DEFAULT)
       return i;
   return -1;
}

/// Save the Profiles to disk
int s9x_Profiler::SaveList() {
   int i, DefaultRow = -1, CountProfiles = ProfileList.CountRows();
   char *n;
   struct stat statbuf;
   s9x_ProfileState s;

   if(CountProfiles>0) {
      try {
	 ProfileFile.open(ios::out);
	 ProfileFile
	   << "# snes9express profiles\n"
	   << "# format:\n"
	   << "# PROFILE Name of profile\n"
	   << "# OPTIONS snes9x options\n\n";
	 DefaultRow = GetDefaultRow();
	 if(DefaultRow>-1) {
	    n = GetRowName(DefaultRow);
	    if(n)
	      ProfileFile << "DEFAULT " << n << "\n\n";
	 };
	 n = NULL;
	 for(i=0; i<CountProfiles; i++) {
	    s = GetRowState(i);
	    if((s==S9X_PROFILE_STATE_SAVED)||(s==S9X_PROFILE_STATE_DEFAULT)) {
	       n = GetRowName(i);
	       ProfileFile << "PROFILE " << n << "\n";
	       n = GetRowArgsEncoded(i);
	       ProfileFile << "OPTIONS " << n << "\n\n";
	    };
	 };
	 ProfileFile.close();
	 if(ProfileFile.stat(statbuf)==0)
	   LastModified = statbuf.st_mtime;
      } catch(...) {
	 fr_Mesg("Error saving profiles!  :(");
	 return 1;
      }
   } else
     ProfileFile.erase();
   return 0;
}


/// Show or hide the entire Profiler window
void s9x_Profiler::SetVisibility(bool s) {
   int i = -1, match = -1, CountProfiles = ProfileList.CountRows();
   char *CurrentArgs;
   struct stat statbuf;

   /* unselect all rows */
   ProfileList.DeselectAll();
   if(s) {
      //reload profiles if file is more recent than whats in mem
      if((ProfileFile.stat(statbuf)==0)
	 &&(statbuf.st_mtime > LastModified))
	LoadList();
      CountProfiles = ProfileList.CountRows();
      //match current interface to a profile, or add new entry
      fr_ArgList ArgList(20);
      ((s9x_Interface*)Parent)->CompileArgs(ArgList);
      CurrentArgs = ArgList.GetEncoded(false);
      if((CurrentArgs)&&(CurrentArgs[0])) {
	 for(i=0; i<CountProfiles; i++)
	   if(!strcmp(CurrentArgs, GetRowArgsEncoded(i))) {
	      match = i;
	      break;
	   };
	 if(match<0) {
	    ProfileList.AddRow(NewListData);
	    match = CountProfiles++;
	    SetProfile(match, S9X_PROFILE_STATE_TEMP,
		       NewProfileName, CurrentArgs);
	    i = -2;
	 };
      } else match = 0;
   } else for(i=0; i<CountProfiles; i++) { // get rid of temp entries
      switch(GetRowState(i)) {
       case S9X_PROFILE_STATE_TEMP:
       case S9X_PROFILE_STATE_DELETED:
	 ProfileList.RemoveRow(i--);
	 CountProfiles--;
       default: break;
      };
   };
   fr_Window::SetVisibility(s);
   if(s && CountProfiles<1) {
      fr_Mesg("You do not have any profiles.\n"
	      "\tTo make a profile, change some options\n"
	      "\tin the main interface, then open the profiler\n"
	      "\tto save that profile.");
      BtnBox.SetEditable(false);
      BtnOK.SetEditable(false);
      ProfName.SetEditable(false);
      BtnCancel.GrabFocus();
   } else if(s) {
      BtnBox.SetEditable(true);
      BtnOK.SetEditable(true);
      ProfileList.Select(match);
      ProfileList.ShowRow(match);
      if(i==-2) {
	 ProfName.SelectText();
	 ProfName.GrabFocus();
      } else
	ProfileList.GrabFocus();
   };
}

/**
 * Get the number of the selected row, if ONE row is selected.
 * If there are multiple rows are selected, return a negative number
 */
int s9x_Profiler::GetSelectedRow() {
   if(ProfileList.CountSelectedRows()!=1)
     return -1;
   return(ProfileList.GetSelected(0));
}

/// Apply row to the main interface
void s9x_Profiler::ApplyRow(int row) {
   fr_ArgList *ArgList;
   if((row>-1)
      &&((ArgList = GetRowArgList(row))!=NULL)) {
      ((s9x_Interface*)Parent)->SiftArgs(*ArgList);
      delete ArgList;
   };
}

/// Apply the selected rows to the main interface
void s9x_Profiler::Apply() {
   int i, p, CountSelected;
   char *SavedName;
   ((s9x_Interface*)Parent)->SetToDefaults();
   CountSelected = ProfileList.CountSelectedRows();
   for(i=0; i<CountSelected; i++) {
      p = ProfileList.GetSelected(i);
      if(p>=0)
	ApplyRow(p);
   };
   if(CountSelected > 1) {
      SavedName = NewProfileName;
      NewProfileName = "[ Merged ]";
      SetVisibility(true);
      NewProfileName = SavedName;
   };
}

/// Apply the default profile to the main interface
void s9x_Profiler::ApplyDefaultProfile() {
   if(ProfileList.CountRows() < 1)
     LoadList();
   ApplyRow(GetDefaultRow());
}

/// A row in the list has been clicked
void s9x_Profiler::RowSelected(int row) {
   int SelectedRow, CountRows;
   SelectedRow = ProfileList.GetSelected(0);
   InLoading = true;
   if(SelectedRow>=0) {
      ProfName.SetEditable(true);
      ProfName.SetText(GetRowName(SelectedRow));
      BtnApply.SetEditable(true);
   } else {
      ProfName.SetText("");
      ProfName.SetEditable(false);
      CountRows = ProfileList.CountSelectedRows();
      BtnApply.SetEditable(CountRows>1);
      BtnSave.SetEditable(CountRows>1);
      BtnDef.SetEditable(false);
      BtnDel.SetEditable(CountRows>1);
      InLoading = false;
      return;
   };
   InLoading = false;
   switch(GetRowState(SelectedRow)) {
    case S9X_PROFILE_STATE_DELETED:
      BtnDel.SetEditable(false);
      BtnSave.SetEditable(true);
      BtnDef.SetEditable(false);
      ProfName.SetEditable(false);
      break;
    case S9X_PROFILE_STATE_TEMP:
      BtnDel.SetEditable(false);
      BtnSave.SetEditable(true);
      BtnDef.SetEditable(false);
      break;
    case S9X_PROFILE_STATE_SAVED:
      BtnDel.SetEditable(true);
      BtnSave.SetEditable(false);
      BtnDef.SetEditable(true);
      break;
    case S9X_PROFILE_STATE_DEFAULT:
      BtnDel.SetEditable(true);
      BtnSave.SetEditable(true);
      BtnDef.SetEditable(false);
      break;
   };
}

/// Change the state of the selected profile(s)
void s9x_Profiler::ChangeState(s9x_ProfileState s) {
   int i, CountSelected, p=-1;

   CountSelected = ProfileList.CountSelectedRows();
   if(CountSelected<1) return;

   if(s==S9X_PROFILE_STATE_DEFAULT) {
      if(CountSelected!=1) return;
      i = GetDefaultRow();
      if(i>=0)
	SetRowState(i, S9X_PROFILE_STATE_SAVED);
      SetRowState(GetSelectedRow(), s);
   } else for(i=0; i<CountSelected; i++) {
      p = ProfileList.GetSelected(i);
      if(p>=0)
	SetRowState(p, s);
   };
   if(p>=0)
     RowSelected(p);
}

/// The name of the selected profile has been updated
void s9x_Profiler::NameUpdated() {
   int SelectedRow;
   char*n;
	
   if(InLoading)
     return;
   SelectedRow = GetSelectedRow();
   if(SelectedRow>=0) {
      if(!(n = ProfName.GetText())) n = "";
      ProfileList.SetCell(SelectedRow, 1, n);
      if(GetRowState(SelectedRow)!=S9X_PROFILE_STATE_DEFAULT)
	ChangeState(S9X_PROFILE_STATE_SAVED);
   };
}

/// Handle events generated within the Profiler
void s9x_Profiler::EventOccurred(fr_Event*e) {
   int SelectedRow;
	
   SelectedRow = GetSelectedRow();
   if(e->Is(BtnDel))
     ChangeState(S9X_PROFILE_STATE_DELETED);
   else if(e->Is(BtnDef))
     ChangeState(S9X_PROFILE_STATE_DEFAULT);
   else if(e->Is(BtnSave))
     ChangeState(S9X_PROFILE_STATE_SAVED);
   else if(e->Is(ProfileList, fr_Click))
     ProfileList.Sort();
   else if(e->Is(ProfileList, fr_Select))
     RowSelected(e->intArg);
   else if(e->Is(ProfName, fr_Changed))
     NameUpdated();
   //else if(e->Is(ProfName))
   //  BtnOK.GrabFocus();  // not sure why, but this goes insane.
   else if(e->Is(BtnApply)||e->Is(BtnOK)||e->Is(ProfileList, fr_DoubleClick))
     Apply();
   else if(e->Is(BtnCancel)||e->Is(this, fr_Destroy)) {
      LastModified = 0;
      ProfileList.RemoveAll();
      SetVisibility(false);
   };
   if(e->Is(BtnOK)||e->Is(ProfileList, fr_DoubleClick)) {
      if(SaveList()==0)
	SetVisibility(false);
   };
}
