/*
 * snes9express
 * interface.cpp
 * (C) 1998-1999, David Nordlund
 * Licensed under the "Artistic" license
 * To view the details of the Artistic license, see the file ./ARTISTIC,
 * or go to http://www.opensource.org/artistic-license.html
 */

#include <iostream>
#ifdef FR_GNOME
# include <gnome.h>
#else
# include <gtk/gtk.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <unistd.h>
#include "interface.h"
#ifdef S9X_INCLUDE_ESD
/* normally I would just #include <esd.h>,
 * but for some reason, that gives me a compiler parse error in audiofile.h
 * So instead, I'll just do the function prototypes myself.
 */
  extern "C" {
	int esd_standby(int);
	int esd_resume(int);
  };
#endif

#include "pix/power.xpm"
#include "pix/eject.xpm"
#include "pix/reset.xpm"
#include "pix/logo.xpm"
#include "pix/nordlund.xpm"
#include "pix/snestex.xpm"
#include "pix/rom.xpm"
#include "pix/camera.xpm"
#ifndef FR_GNOME
# include "pix/ball.xpm"
# include "pix/def.xpm"
# include "pix/del.xpm"
#endif

using namespace std;

const char*Authors[] = {
	"David Nordlund <" SNES9EXPRESS_EMAIL ">",
	(char*)0
};
char *Copyright = " 1998-1999, David Nordlund";
char *Comments = PROG " is a front-end for snes9x,\n"
                 "the Super Nintendo emulator.\n\n"
                 PROG " website: " SNES9EXPRESS_WEBSITE "\n"
  		 "snes9x website: " SNES9X_WEBSITE "\n\n"
		 "Built on " __DATE__ ",\n with " FR_GTK
#ifdef FR_GNOME
		" and " FR_GNOME
#endif
		"\n";
;

fr_ArgList*CommandLineArgs;

s9x_Interface *Interface;
int main(int argc, char*argv[]) {
   CommandLineArgs = &fr_init(PROG, &argc, &argv);
   Interface = new s9x_Interface();
   fr_ArgList ExecArgs = fr_main(Interface, *CommandLineArgs);
   int r = fr_exec(ExecArgs);
   cerr << ExecArgs.GetString();
   return r;
}

s9x_Interface::s9x_Interface():
fr_MainProgram(PROG, logo_xpm),
Nordlund(this, "(Nordlund)", nordlund_xpm),
BannerBox(this),
Panel(this),
Power(this, new fr_Label(this, "Power", power_xpm)),
Reset(this, new fr_Label(this, "Reset", reset_xpm)),
Eject(this, new fr_Label(this, "Eject", eject_xpm)),
NoteBook(this),
Quit(this),
Tip(this),
ImgTex(this, snestex_xpm),
ImgRom(this, rom_xpm),
ImgSnap(this, camera_xpm),
#ifdef FR_GNOME
ImgPref(this, GNOME_STOCK_MENU_PREF),
ImgQuit(this, GNOME_STOCK_MENU_QUIT),
ImgHelp(this, GNOME_STOCK_MENU_ABOUT),
ImgPower(this, GNOME_STOCK_MENU_EXEC),
ImgReset(this, GNOME_STOCK_MENU_UNDO),
#else
ImgPref(this, ball_xpm),
ImgQuit(this, del_xpm),
ImgHelp(this, ball_xpm),
ImgPower(this, def_xpm),
ImgReset(this, def_xpm),
#endif	
SNESstyle(this, 0x330066, 0xCDCDCD, &ImgTex),
LastArgs(0),
LastFile("last"),
Prefs(this),
ROM(this),
Sound(this),
Video(this),
CPU(this),
Joystick(this),
NetPlay(this),
Snapshot(this),
Extra(this),
Profiler(this),
About(this, VER "-" REL, Copyright, Authors, Comments)
{
   SetGridSize(1, 3);

   BannerBox.SetGridSize(5, 4, true);
   BannerBox.SetPadding(3, 3);
   BannerBox.SetStretch(Grow, Fill);
   BannerBox.SetVisibility(false);
   BannerBox.Pack(Nordlund, 4, 0, 5, 1);
	
   SetStretch(Normal, Normal);
   Pack(BannerBox);

   CreateNotebook();
   SetPadding(2, 0);
   SetStretch(Fill, Grow);
   Pack(NoteBook);
	
   CreatePanel();
   Panel.SetVisibility(false);
   SetPadding(0, 0);
   SetStretch(Normal, Normal);
   Pack(Panel, 0, 2, 1, 3);

   CreateMenu();
   Set9xVersion(SNES9X_VERSION);
   Prefs.Apply();
   SetVisibility(true);
   AddListener(this);
   
   try {
      char lastbuf[FR_PATH_MAX];
      LastFile.open(ios::in);
      if(!(LastFile >> lastbuf)) throw "garbage";
      if(LastFile.getline(lastbuf, FR_PATH_MAX-1))
	LastArgs = new fr_ArgList(lastbuf, false);
   } catch(...) {
      // no big deal
   }
}

s9x_Interface::~s9x_Interface() {
   delete LastArgs;
}

void s9x_Interface::CreateMenu() {
   static fr_MenuBarItem FileMenu[] = {
	{"Open Profiler  (Eject) ...", &Eject, &ImgPref},
	{"Search for ROM ...", &ROM, &ImgRom},
	{"Search for Snapshot ...", &Snapshot, &ImgSnap},
	{"-", (fr_Element*)0, 0},
	{"Quit", &Quit, &ImgQuit},
	{(char*)0, (fr_Element*)0, 0}
   };
   static fr_MenuBarItem ConsoleMenu[] = {
	{"Preferences ...", &Prefs, &ImgPref},
	{"-", (fr_Element*)0, 0},
	{"Reset options to defaults  (Reset)", &Reset, &ImgReset},
	{"Run snes9x  (Power)", &Power, &ImgPower},
	{(char*)0, (fr_Element*)0, 0}
   };
   static fr_MenuBarItem HelpMenu[] = {
	{"About " PROG " ...", &About, &ImgHelp},
	{"-", (fr_Element*)0, 0},
	{"Random tip ...", &Tip, &ImgHelp},
	{(char*)0, (fr_Element*)0, 0}
   };
   Quit.AddListener(this);
   Tip.AddListener(this);
   About.AddListener(this);

   AddMenu("File",    FileMenu);
   AddMenu("Console", ConsoleMenu);
   AddMenu("Help",    HelpMenu);
}

void s9x_Interface::CreateNotebook() {
   NoteBook.SetPadding(7, 7);
   NoteBook.AddPage(ROM);
   NoteBook.AddPage(Sound);
   NoteBook.AddPage(Video);
   NoteBook.AddPage(Joystick);
   NoteBook.AddPage(CPU);
   NoteBook.AddPage(NetPlay);
   NoteBook.AddPage(Snapshot);
   NoteBook.AddPage(Extra);
   NoteBook.SetScrollable(true);
}

void s9x_Interface::CreatePanel() {
   fr_Label *SidePadding;
   fr_Box *box;

   Panel.SetGridSize(5, 1, false);
	
   Power.SetTooltip("Run snes9x");
   Eject.SetTooltip("Use the " PROG " profiler...");
   Reset.SetTooltip("Set all options to their default values");

   Panel.SetPadding(0, 0);
   Panel.SetStretch(Shrink, Grow);

   box = new fr_Box(this);
   box->SetPadding(0, 0);
   box->SetGridSize(1, 6, true);
   box->Pack(Power, 0, 1, 2, 3);
   box->AddBorderBevelOut();
   box->ApplyStyle(&SNESstyle);
   box->SetVisibility(true);
   Panel.Pack(*box, 1, 0, 2, 1);

   box = new fr_Box(this);
   box->SetPadding(0, 0);
   box->SetGridSize(1, 6, false);
   box->Pack(Eject, 0, 1, 2, 5);
   box->SetVisibility(true);
   Panel.Pack(*box, 2, 0, 3, 1);

   box = new fr_Box(this);
   box->SetPadding(0, 0);
   box->SetGridSize(1, 6, true);
   box->Pack(Reset, 0, 1, 2, 3);
   box->AddBorderBevelOut();
   box->ApplyStyle(&SNESstyle);
   box->SetVisibility(true);
   Panel.Pack(*box, 3, 0, 4, 1);

   Panel.SetStretch(Normal, Normal);
   SidePadding = new fr_Label(this, " ");
   Panel.Pack(*SidePadding, 0, 0, 1, 1);
   SidePadding = new fr_Label(this, " ");
   Panel.Pack(*SidePadding, 4, 0, 5, 1);

   Panel.ApplyStyle(&SNESstyle);
   
   Power.AddListener(this);
   Eject.AddListener(this);
   Reset.AddListener(this);
}

void s9x_Interface::SetToDefaults() {
   ROM.SetToDefaults();
   Sound.SetToDefaults();
   Video.SetToDefaults();
   CPU.SetToDefaults();
   Joystick.SetToDefaults();
   NetPlay.SetToDefaults();
   Snapshot.SetToDefaults();
   Extra.SetToDefaults();
}

void s9x_Interface::Set9xVersion(double version) {
   ROM.Set9xVersion(version);
   Sound.Set9xVersion(version);
   Video.Set9xVersion(version);
   CPU.Set9xVersion(version);
   Joystick.Set9xVersion(version);
   NetPlay.Set9xVersion(version);
}

void s9x_Interface::SiftArgs(fr_ArgList& L) {
   static bool BeenHereBefore = false;

   L.ClearMarks();

   if(!BeenHereBefore) {
      BeenHereBefore = true;
      if(Prefs.StartToProfile())
	Profiler.ApplyDefaultProfile();
      if(Prefs.StartToLast())
	SiftArgs(*LastArgs);
   };
   Prefs.SiftArgs(L);
   Prefs.Apply();
   Joystick.SiftArgs(L);
   Sound.SiftArgs(L);
   Video.SiftArgs(L);
   CPU.SiftArgs(L);
   NetPlay.SiftArgs(L);
   Snapshot.SiftArgs(L);
   ROM.SiftArgs(L);		//ROM should be 2nd last
   Extra.SiftArgs(L);		//And Extra must be last
}

void s9x_Interface::CompileArgs(fr_ArgList& L) {
   Sound.CompileArgs(L);
   Video.CompileArgs(L);
   Joystick.CompileArgs(L);
   CPU.CompileArgs(L);
   NetPlay.CompileArgs(L);
   Snapshot.CompileArgs(L);
   ROM.CompileArgs(L);
   Extra.CompileArgs(L);
}

void s9x_Interface::ApplyStyle(fr_Style*S) {
   fr_Window::ApplyStyle(S);
   NoteBook.ApplyStyle(S);
   ROM.ApplyStyle(S);
   Sound.ApplyStyle(S);
   Video.ApplyStyle(S);
   Joystick.ApplyStyle(S);
   CPU.ApplyStyle(S);
   NetPlay.ApplyStyle(S);
   Snapshot.ApplyStyle(S);
   Extra.ApplyStyle(S);
}

void s9x_Interface::ViewExpanded(bool b) {
   static bool l = false;
   if(l==b)
     return;
   if(b) {
      ApplyStyle(&SNESstyle);
      Panel.SetVisibility(true);
      BannerBox.SetVisibility(true);
   } else {
      ResetStyle();
      Panel.SetVisibility(false);
      BannerBox.SetVisibility(false);
   };
   l = b;
}

void s9x_Interface::EventOccurred(fr_Event*e) {
   if(e->Is(Power)) {
      if(fr_FileExists(ROM.GetFileName())) {
	 if(Prefs.Reincarnate())
	   s9x_Interface::Run();
	 else
	   fr_MainProgram::Run();
      } else
	fr_Mesg("You need to select a ROM file.");
   } else if(e->Is(Eject))
	Profiler.SetVisibility(true);
   else if(e->Is(Reset)) {
      SetToDefaults();
      if(Prefs.ResetToProfile())
	Profiler.ApplyDefaultProfile();
      if(Prefs.ResetToArgs())
	SiftArgs(*CommandLineArgs);
      if(Prefs.ResetToLast())
	SiftArgs(*LastArgs);
   } else if(e->Is(About))
     About.Splash();
   else if(e->Is(Tip))
     s9x_RandomTip();
   else if(e->Is(Quit)||e->Is(this, fr_Destroy))
     Exit(0);
}

void s9x_Interface::Run() {
   int status, winx, winy;
   pid_t snes9xpid;
   char **Args, Msg[80], *snapdir;
   mode_t m;
	
   snapdir = Prefs.GetSnapDir();
   m = S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH;
   delete LastArgs;
   LastArgs = new fr_ArgList();
   *LastArgs << GetExecutable();
   LastArgs->Mark(0);
   CompileArgs(*LastArgs);
   Args = LastArgs->GetArgs();
   if(!Args) {
      fr_Mesg("I can't figure out what to run. Something is wrong");
      return;
   };
#ifdef S9X_INCLUDE_ESD
   if(Prefs.DisableESD())
     esd_standby(gnome_sound_connection);
#endif
   if(snapdir) // make the dir in case it does not exist
     mkdir(snapdir, m);
   switch((snes9xpid = fork())) {
    case -1:	// error
      fr_Mesg("Error: could not fork!");
#ifdef S9X_INCLUDE_ESD
      if(Prefs.DisableESD())
	esd_resume(gnome_sound_connection);
#endif		  
      break;
    case 0:	// snes9x
      //log the command we are about to run
      try {
	 LastFile.open(ios::out);
	 LastFile << LastArgs->GetEncoded(false) << endl;
	 // log snes9x output, redirect standard output streams.
	 LastFile.funnel(cout);
	 LastFile.funnel(cerr);
      } catch(...) {
	 cerr << PROG " warning: could not log output." << endl;
      };
      execvp(Args[0], Args);
      _exit(255);
    default:	// snes9express
      gdk_window_get_root_origin(Window->window, &winx, &winy);
      gdk_window_hide(Window->window);
      fr_Flush();
      waitpid(snes9xpid, &status, 0);
      if(snapdir) // remove the snapdir IF it is empty
	rmdir(snapdir);
      gdk_window_show(Window->window);
      gdk_window_move(Window->window, winx, winy);
#ifdef S9X_INCLUDE_ESD
      if(Prefs.DisableESD())
	esd_resume(gnome_sound_connection);
#endif		  
      if(!WIFEXITED(status))
	fr_Mesg("Error: snes9x terminated abnormally");
      else if(WEXITSTATUS(status)==255) {
	 sprintf(Msg, "Error: could not exec %s",
		 strlen(Args[0])<50?Args[0]:"snes9x");
	 fr_Mesg(Msg);
      } else if((status)&&(status!=768)) {
	 sprintf(Msg, "%s returned error code %d",
		 strlen(Args[0])<50?Args[0]:"snes9x",
		 WEXITSTATUS(status));
	 fr_Mesg(Msg);
      };
   };
}
