package Ora2Pg;
#------------------------------------------------------------------------------
# Project  : Oracle to PostgreSQL database schema converter
# Name     : Ora2Pg.pm
# Language : Perl
# Authors  : Gilles Darold, gilles _AT_ darold _DOT_ net
# Copyright: Copyright (c) 2000-2010 : Gilles Darold - All rights reserved -
# Function : Main module used to export Oracle database schema to PostgreSQL
# Usage    : See documentation in this file with perldoc.
#------------------------------------------------------------------------------
#
#        This program is free software: you can redistribute it and/or modify
#        it under the terms of the GNU General Public License as published by
#        the Free Software Foundation, either version 3 of the License, or
#        any later version.
# 
#        This program is distributed in the hope that it will be useful,
#        but WITHOUT ANY WARRANTY; without even the implied warranty of
#        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#        GNU General Public License for more details.
# 
#        You should have received a copy of the GNU General Public License
#        along with this program. If not, see < http://www.gnu.org/licenses/ >.
# 
#------------------------------------------------------------------------------

use vars qw($VERSION $PSQL %Config);
use Carp qw(confess);
use DBI;
use POSIX qw(locale_h);
use IO::File;

#set locale to LC_NUMERIC C
setlocale(LC_NUMERIC,"C");


$VERSION = '6.2';
$PSQL = $ENV{PLSQL} || 'psql';


=head1 NAME

Ora2Pg - Oracle to PostgreSQL database schema converter


=head1 SYNOPSIS

Ora2pg has a companion script called ora2pg. When use in
conjonction with a custom version of ora2pg.conf they perform what
I'm trying to explain bellow. See content of the ora2pg.conf
file for more explanation on configuration directives.

	use Ora2Pg;

	# Create an instance of the Ora2Pg Perl module
	my $schema = new Ora2Pg (config => './ora2pg.conf');

	# Create a PostgreSQL representation of Oracle export
	# you've defined in ora2pg.conf.
	$schema->export_schema();

	exit(0);

You can always overwrite any configuration option set in ora2pg.conf
by passing a hash where keys are the same that in configuration file
but in lower case. For example, if you only want to extract only a
selection of tables:

	my @tables = ('t1', 't2', 't3');
	my $schema = new Ora2Pg (
		datasource => $dbsrc,   # Oracle DBD datasource
		user => $dbuser,        # Database user
		password => $dbpwd,     # Database password
		tables => \@tables,
	# or				
	#	tables => [('tab1','tab2')],  # Tables to extract
		debug => 1		      # Verbose running.
	);

or if you only want to extract only the first 10 tables:

	my $schema = new Ora2Pg (
		datasource => $dbsrc,   # Oracle DBD datasource
		user => $dbuser,        # Database user
		password => $dbpwd,     # Database password
		max => 10		# First 10 tables to extract
	);

or if you only want to extract tables 10 to 20:

	my $schema = new Ora2Pg (
		datasource => $dbsrc,   # Database DBD datasource
		user => $dbuser,        # Database user
		password => $dbpwd,     # Database password
		min => 10,		# Begin extraction at rank 10
		max => 20		# End extraction at rank 20
	);

Setting showtableid to 1 will display a table and its ranks without any
extraction. This will help you to set values of min/max options.

To choose a particular Oracle schema to export just set the following option
to your schema name:

	schema => 'APPS'

This schema definition can also be needed when you want to export data. If
export fails and complaining that the table doesn't exists use this directive
to prefix the table name by the schema name.

If you want to use PostgreSQL 7.4+ schema support set the init option
'export_schema' set to 1. Default is no schema export.

To know at which ranks tables can be found during the extraction use the option:

	showtableid => 1

You can process multiple types of extractions at the same time by setting the
value to a space separated list of the following keywords.

To extract all views set the type option as follows:

	type => 'VIEW'

To extract all grants set the type option as follows:

	type => 'GRANT'

To extract all sequences set the type option as follows:

	type => 'SEQUENCE'

To extract all triggers set the type option as follows:

	type => 'TRIGGER'

To extract all functions set the type option as follows:

	type => 'FUNCTION'

To extract all procedures set the type option as follows:

	type => 'PROCEDURE'

To extract all packages and packages bodies set the type option as follows:

	type => 'PACKAGE'

Default is table extraction:

	type => 'TABLE'

To extract tablespaces (PostgreSQL >= v8):

	type => 'TABLESPACE'

To extract table range or list partition (PostgreSQL >= v8.4):

	type => 'PARTITION'

To extract user defined Oracle type

	type => 'TYPE'

To extract table datas as INSERT statements use:

	type => 'DATA'

To extract table datas as COPY statements use:

	type => 'COPY'

and set data_limit => n to specify the max tuples to return. If you set
this options to 0 or nothing, no limitation are used. Additional options
'table', 'min' and 'max' can also be used. This is useful only when data
is send to Pg backend directly, but not when when dumping to file.

Oracle export is done by calling method:

	$schema->export_schema();

The extracted data is dumped to filename specified in the OUTPUT configuration
directive or to stdout if it's set to nothing. You can always overwrite this
configuration value by specifying a filename as argument of this function.

You can also send the data directly to a PostgreSQL backend by setting PG_DSN,
PG_USER and PG_PWD configuration directives. This feature is only available for
COPY or DATA export types. The data will not be sent via DBD::Pg but will be
loaded to the PG database using the psql command.
Edit the $PSQL environment variable to specify the path of your psql command
(nothing to edit if psql is in your path).

When copying tables, Ora2Pg normally exports constraints as they are;
if they are non-deferrable they will be exported as non-deferrable.
However, non-deferrable constraints will probably cause problems when
attempting to import data to PostgreSQL. The option:

       fkey_deferrable => 1

will cause all foreign key constraints to be exported as deferrable,
even if they are non-deferrable.
In addition, setting:

       defer_fkey => 1

when exporting data will add a command to actually defer all constraints
before importing data.

To non perl gurus, you can use the configuration file and run ora2pg as is.
You will find all information into the ora2pg.conf to be able to set it
correctly.


=head1 DESCRIPTION

Ora2Pg is a perl OO module used to export an Oracle database schema
to a PostgreSQL compatible schema.

It simply connects to your Oracle database, extracts its structures and
generates an SQL script that you can load into your PostgreSQL database.

Ora2Pg.pm dumps the database schema (tables, views, sequences, indexes,
grants, etc.), with primary, unique and foreign keys into PostgreSQL syntax
without need to edit the SQL code generated.

It can also dump Oracle data into a PostgreSQL database 'on the fly'. Also
you can choose a selection of columns to be exported for each table.

The SQL and/or PL/SQL code generated for functions, procedures and triggers
has to be reviewed to match the PostgreSQL syntax. You find some useful
recommandations on porting Oracle PL/SQL code to PostgreSQL PL/PGSQL at
"http://techdocs.postgresql.org/" under the topic "Converting from other
Databases to PostgreSQL", Oracle.

Notice that the trunc() function in Oracle is the same for number and date
types. Be carefull when porting to PostgreSQL to use trunc() for numbers
and date_trunc() for dates.


=head1 ABSTRACT

The goal of the Ora2Pg Perl module is to cover everything needed to export
an Oracle database to a PostgreSQL database without other thing than providing
the parameters needed for connecting to the Oracle database.

Features include:

	- Exporting the database schema (tables, views, sequences, indexes),
	  with unique, primary and foreign key and check constraints.
	- Exporting grants/privileges for users and groups.
	- Exporting range and list table partition.
	- Exporting a table selection (by specifying the table names or max
	  tables).
	- Exporting the Oracle schema to a PostgreSQL 7.3+ schema.
	- Exporting predefined functions/triggers/procedures/packages.
	- Exporting user defined data type.
	- Exporting table data.
	- Exporting Oracle views as PG tables.
	- Providing basic help for converting PLSQL code to PLPGSQL (needs
	  manual work).

See ora2pg.conf for more information on use.

My knowledge about database is really poor especially for Oracle RDBMS.
Any contributions, particularly in this matter, are welcome.


=head1 REQUIREMENTS

You just need the DBI, DBD::Pg and DBD::Oracle Perl module to be installed.
DBD::Pg is optional and needed only for 'on the fly' migration. The PostgreSQL
client (psql) must also be installed on the host running Ora2Pg.

If you want to compress output as a gzip file you need Compress::Zlib Perl
module. And if you want to use bzip2 compression, program bzip2 must be
available.


=head1 PUBLIC METHODS

=head2 new HASH_OPTIONS

Creates a new Ora2Pg object.

The only required option is:

    - config : Path to the configuration file (required).

All directives found in the configuration file can be overwritten in the
instance call by passing them in lowercase as arguments. These supported
options are (See ora2pg.conf for more details):

    - datasource : Oracle DBD datasource (required)
    - user : Oracle DBD user (optional with public access)
    - password : Oracle DBD password (optional with public access)
    - schema : Oracle internal schema to extract (optional)
    - type : Type of data to extract, can be TABLE,VIEW,GRANT,SEQUENCE,
      TRIGGER,FUNCTION,PROCEDURE,DATA,COPY,PACKAGE,TABLESPACE,PARTTION
      or a combinaison of these keywords separated by blanks.
    - debug : verbose mode.
    - export_schema : Export Oracle schema to PostgreSQL >7.3 schema
    - tables : Extract only the specified tables (arrayref) and set the
      extracting order
    - exclude : Exclude the specified tables from extraction (arrayref)
    - showtableid : Display only the table ranks during extraction
    - min : Table rank to begin extraction at. Defaults to 0
    - max : Table rank to end extraction at. Defaults to 0 meaning no limits
    - data_limit : Max number of tuples to return during data extraction
      (defaults to 0 meaning no limit)
    - case_sensitive: Allow to preserve Oracle object names as they are
      written. Default is not.
    - skip_fkeys : Skip foreign key constraints extraction. Defaults to 0
      (extraction)
    - skip_pkeys : Skip primary keys extraction. Defaults to 0 (extraction)
    - skip_ukeys : Skip unique column constraints extraction. Defaults to 0
      (extraction)
    - skip_indices : Skip all other index types extraction. Defaults to 0
      (extraction)
    - skip_checks : Skip check constraints extraction. Defaults to 0
      (extraction)
    - keep_pkey_names : By default, primary key names in the source database
      are ignored, and default key names are created in the target database.
      If this is set to true, primary key names are preserved.
    - bzip2: Path to the Bzip2 program to compress data export. Default
      /usr/bin/bzip2
    - gen_user_pwd : When set to 1 this will replace the default password
      'change_my_secret' with a random string.
    - fkey_deferrable: Force foreign key constraints to be exported as
      deferrable. Defaults to 0: export as is.
    - defer_fkey : Force all foreign key constraints to be deferred during
      data import. Defaults to 0: export as is.
    - pg_numeric_type: Convert the Oracle NUMBER data type to adequate PG data
      types instead of using the slow numeric(p,s) data type.
    - default_numeric: By default the NUMBER(x)type without precision is
      converted to float. You can overwrite this data type by any PG type.
    - keep_pkey_names: Preserve oracle primary key names. The default is to
      ignore and use PostgreSQl defaults.
    - pg_supports_inout: Allow PG support of in/out/inout function parameters
      Must be used with PostgreSQL > 8.1. Defaults to none support (backward
      compatibility).
    - pg_supports_role: Allow PG support of roles instead of user/group.
      Defaults to none support (backward compatibility).
    - disable_triggers: Disable triggers on all tables in COPY and
      DATA mode.
    - disable_sequence: Disables alter sequence on all tables in COPY or
      DATA mode.
    - noescape: Disable character escaping during data export.
    - datatype: Redefine Oracle to PostgreSQl data type conversion.
    - binmode: Force Perl to use the specified binary mode for output. The
      default is ':raw';
    - sysusers: Add other system users to the default exclusion list
      (SYS,SYSTEM,DBSNMP,OUTLN,PERFSTAT,CTXSYS,XDB,WMSYS,SYSMAN,SQLTXPLAIN,
      MDSYS,EXFSYS,ORDSYS,DMSYS,OLAPSYS).
    - ora_sensitive: Force the use of Oracle case sensitive table/view names.
    - plsql_pgsql: Enable plsql to plpgsql conversion.
    - pg_schema: Allow to specify a coma delimited list of PostgreSQL schema.

Beware that this list may grow longer because all initialization is
performed this way.


Special configuration options to handle character encoding:
-----------------------------------------------------------

NLS_LANG

If you experience any issues where mutibyte characters are being substituted
with replacement characters during the export try to set the NLS_LANG
configuration directive to the Oracle encoding. This may help a lot especially
with UTF8 encoding.

BINMODE

If you experience the Perl warning: "Wide character in print", it means
that you tried to write a Unicode string to a non-unicode file handle.
You can force Perl to use binary mode for output by setting the BINMODE
configuration option to the specified encoding. If you set it to 'utf8', it
will force printing like this: binmode OUTFH, ":utf8"; By default Ora2Pg opens
the output file in 'raw' binary mode.

Exporting Oracle views as PostgreSQL tables:
--------------------------------------------

Since version 4.10 you can export Oracle views as PostgreSQL tables simply
by setting TYPE configuration option to TABLE and COPY or DATA and specifying
your views in the TABLES configuration option.
Then if Ora2Pg does not find the name in Oracle table names it automatically
deduces that it must search for it in the view names, and if it finds
the view it will extract its schema (if TYPE=TABLE) into a PG create table form,
then it will extract the data (if TYPE=COPY or DATA) following the view schema.

Case sensitive table names in Oracle:
-------------------------------------

Since version 4.10 you can extract/export Oracle databases with case sensitive
table/view names. This requires the use of quoted table/view names during
Oracle querying. Set the configuration option ORA_SENSITIVE to 1 to enable this
feature. By default it is off.

=cut

sub new
{
	my ($class, %options) = @_;

	# This create an OO perl object
	my $self = {};
	bless ($self, $class);

	# Initialize this object
	$self->_init(%options);
	
	# Return the instance
	return($self);
}


=head2 export_data FILENAME

OBSOLETE: you must use export_schema instead. Still here
for backward compatibility. It simply callback export_schema().

=cut

sub export_data
{
	my ($self, $outfile) = @_;

	$self->export_schema($outfile);
}


=head2 export_schema FILENAME

Print SQL data output to a file name or
to STDOUT if no file name is specified.

=cut

sub export_schema
{
	my ($self, $outfile) = @_;

	# Init with configuration OUTPUT filename
	$outfile ||= $self->{output};

	if ($outfile) {
		# Send output to the specified file
		if ($outfile =~ /\.gz$/) {
			use Compress::Zlib;
			$self->{compress} = 'Zlib';
			$self->{zlib_hdl} = gzopen($outfile, "wb") or $self->logit("FATAL: Can't create deflation file $outfile\n", 0, 1);
		} elsif ($outfile =~ /\.bz2$/) {
			$self->logit("FATAL: can't run bzip2\n",0,1) if (!-x $self->{bzip2});
			$self->{fhout} = new IO::File;
			$self->{fhout}->open("|$self->{bzip2} --stdout >$outfile") or $self->logit("FATAL: Can't open pipe to $self->{bzip2} --stdout >$outfile: $!\n", 0, 1);
		} else {
			$self->{fhout} = new IO::File;
			$self->{fhout}->open(">$outfile") or $self->logit("FATAL: Can't open $outfile: $!\n", 0, 1);
			binmode($self->{fhout},$self->{'binmode'});
		}
		foreach my $t (@{$self->{export_type}}) {
			$self->{type} = $t;
			# Return data as string
			$self->_get_sql_data();
		}
		if ($outfile =~ /\.gz$/) {
			$self->{zlib_hdl}->gzclose();
		} else {
			$self->{fhout}->close();
		}

	} else {

		foreach my $t (@{$self->{export_type}}) {
			$self->{type} = $t;
			# Return data as string
			$self->_get_sql_data();
		}

	}

}


=head2 export_file FILENAME

Open a file handle to a given filename.

=cut

sub export_file
{
	my ($self, $outfile) = @_;

	my $filehdl = undef;

	if ($outfile) {
		# If user request data compression
		if ($self->{output} =~ /\.gz$/) {
			use Compress::Zlib;
			$self->{compress} = 'Zlib';
			$filehdl = gzopen("$outfile.gz", "wb") or $self->logit("FATAL: Can't create deflation file $outfile.gz\n",0,1);
		} elsif ($self->{output} =~ /\.bz2$/) {
			$self->logit("Error: can't run bzip2\n",0,1) if (!-x $self->{bzip2});
			$filehdl = new IO::File;
			$filehdl->open("|$self->{bzip2} --stdout >$outfile.bz2") or $self->logit("FATAL: Can't open pipe to $self->{bzip2} --stdout >$outfile.bz2: $!\n", 0,1);
		} else {
			$filehdl = new IO::File;
			$filehdl->open(">$outfile") or $self->logit("FATAL: Can't open $outfile: $!\n", 0, 1);
			binmode($filehdl, $self->{'binmode'});
		}

	}
	return $filehdl;
}


=head2 close_export_file FILEHANDLE

Close a file handle.

=cut

sub close_export_file
{
	my ($self, $filehdl) = @_;


	if ($self->{output} =~ /\.gz$/) {
		$filehdl->gzclose();
	} else {
		$filehdl->close();
	}

}

=head2 modify_struct TABLE_NAME ARRAYOF_FIELDNAME

Modify the table structure during the export. Only the specified columns
will be exported. 

=cut

sub modify_struct
{
	my ($self, $table, @fields) = @_;

	if (!$self->{case_sensitive}) {
		map { $_ = lc($_) } @fields;
		$table = lc($table);
	}

	push(@{$self->{modify}{$table}}, @fields);

}

=head2 replace_tables HASH

Modify table names during the export.

=cut

sub replace_tables
{
	my ($self, %tables) = @_;

	foreach my $t (keys %tables) {
		$tables{$t} = lc($tables{$t}) if (!$self->{case_sensitive});
		$self->{replaced_tables}{"\L$t\E"} = $tables{$t};
	}

}

=head2 replace_cols HASH

Modify column names during the export.

=cut

sub replace_cols
{
	my ($self, %cols) = @_;

	foreach my $t (keys %cols) {
		foreach my $c (keys %{$cols{$t}}) {
			$cols{$t}{$c} = lc($cols{$t}{$c}) if (!$self->{case_sensitive});
			$self->{replaced_cols}{"\L$t\E"}{"\L$c\E"} = $cols{$t}{$c};
		}
	}

}

=head2 set_where_clause HASH

Add a WHERE clause during data export on specific tables or on all tables

=cut

sub set_where_clause
{
	my ($self, $global, %table_clause) = @_;

	$self->{global_where} = $global;
	foreach my $t (keys %table_clause) {
		$self->{where}{"\L$t\E"} = $table_clause{$t};
	}

}



#### Private subroutines ####

=head1 PRIVATE METHODS

=head2 _init HASH_OPTIONS

Initialize an Ora2Pg object instance with a connexion to the
Oracle database.

=cut

sub _init
{
	my ($self, %options) = @_;

	# Read configuration file
	&read_config($options{config}) if ($options{config});

	# Those are needed by DBI
	$ENV{ORACLE_HOME} = $Config{'ORACLE_HOME'} if ($Config{'ORACLE_HOME'});
	$ENV{NLS_LANG} = $Config{'NLS_LANG'} if ($Config{'NLS_LANG'});

	# Init arrays
	$self->{limited} = ();
	$self->{excluded} = ();
	$self->{modify} = ();
	$self->{replaced_tables} = ();
	$self->{replaced_cols} = ();
	$self->{where} = ();
	@{$self->{sysusers}} = ('SYSTEM','SYS','DBSNMP','OUTLN','PERFSTAT','CTXSYS','XDB','WMSYS','SYSMAN','SQLTXPLAIN','MDSYS','EXFSYS','ORDSYS','DMSYS','OLAPSYS');
	$self->{ora_reserved_words} = ();

	# Init PostgreSQL DB handle
	$self->{dbhdest} = undef;

	# Initialyze following configuration file
	foreach my $k (keys %Config) {
		if (lc($k) eq 'tables') {
			$self->{limited} = $Config{TABLES};
		} elsif (lc($k) eq 'exclude') {
			$self->{excluded} = $Config{EXCLUDE};
		} else {
			$self->{lc($k)} = $Config{uc($k)};
		}
	}

	# Set transaction isolation level
	if ($self->{transaction} eq 'readonly') {
		$self->{transaction} = 'SET TRANSACTION READ ONLY';
	} elsif ($self->{transaction} eq 'readwrite') {
		$self->{transaction} = 'SET TRANSACTION READ WRITE';
	} elsif ($self->{transaction} eq 'committed') {
		$self->{transaction} = 'SET TRANSACTION ISOLATION LEVEL READ COMMITTED';
	} else {
		$self->{transaction} = 'SET TRANSACTION ISOLATION LEVEL SERIALIZABLE';
	}

	# Overwrite configuration with all given parameters
	# and try to preserve backward compatibility
	foreach my $k (keys %options) {
		if ($options{tables} && (lc($k) eq 'tables')) {
			$self->{limited} = $options{tables};
		} elsif ($options{exclude} && (lc($k) eq 'exclude')) {
			$self->{excluded} = $options{exclude};
		} elsif ($options{datasource} && (lc($k) eq 'datasource')) {
			$self->{oracle_dsn} = $options{datasource};
		} elsif ($options{user} && (lc($k) eq 'user')) {
			$self->{oracle_user} = $options{user};
		} elsif ($options{password} && (lc($k) eq 'password')) {
			$self->{oracle_pwd} = $options{password};
		} elsif ($options{$k} ne '') {
			$self->{lc($k)} = $options{$k};
		}
	}
	# Log file handle
	$self->{fhlog} = undef;
	if ($self->{logfile}) {
		$self->{fhlog} = new IO::File;
		$self->{fhlog}->open(">>$self->{logfile}") or $self->logit("FATAL: can't log to $self->{logfile}, $!\n", 0, 1);
	}

	# Free memory
	%options = ();
	%Config = ();

        # Connect the database
	$self->logit("Trying to connect to database: $self->{oracle_dsn}\n", 1);
        $self->{dbh} = DBI->connect($self->{oracle_dsn}, $self->{oracle_user}, $self->{oracle_pwd});

        # Check for connection failure
        if (!$self->{dbh}) {
		$self->logit("FATAL: $DBI::err ... $DBI::errstr\n", 0, 1);
	}

	# Set some default
	$self->{global_where} = '';
	$self->{min} ||= 0;
	$self->{max} ||= 0;
	$self->{dbh}->{LongReadLen} = 0;
	#$self->{dbh}->{LongTruncOk} = 1;
	$self->{data_current} = 0;
	$self->{prefix} = 'DBA';
	if ($self->{user_grants}) {
		$self->{prefix} = 'ALL';
	}
	$self->{bzip2} ||= '/usr/bin/bzip2';
	$self->{default_numeric} ||= 'float';
	# backward compatibility
	if ($self->{disable_table_triggers}) {
		$self->{disable_triggers} = $self->{disable_table_triggers};
	}
	$self->{binmode} ||= ':raw';
	$self->{binmode} =~ s/^://;
	$self->{binmode} = ':' . lc($self->{binmode});

	# Allow multiple or chained extraction export type
	$self->{export_type} = ();
	if ($self->{type}) {
		@{$self->{export_type}} = split(/[\s\t,;]+/, $self->{type});
	} else {
		push(@{$self->{export_type}}, 'TABLE');
	}

	# If you decide to autorewrite PLSQL code, this load the dedicated
	# Perl module
	if ($self->{plsql_pgsql}) {
		use Ora2Pg::PLSQL;
	}

        # Use consistent reads for concurrent dumping...
        $self->{dbh}->begin_work || $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	if ($self->{debug}) {
		$self->logit("Isolation level: $self->{transaction}\n", 1);
	}
        my $sth = $self->{dbh}->prepare($self->{transaction}) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
        $sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
        undef $sth;

	# Check validity of the requested schema/role
	if ($self->{schema}) {
		if (!$self->_get_schema()) {
			$self->{dbh}->disconnect() if ($self->{dbh});
			$self->logit("FATAL: Schema $self->{schema} not found !\n", 0, 1);
		}
	}
	$self->{fhout} = undef;
	$self->{compress} = '';
	$self->{zlib_hdl} = undef;

	# Retreive all table informations
        foreach my $t (@{$self->{export_type}}) {
                $self->{type} = $t;
		if (($self->{type} eq 'TABLE') || ($self->{type} eq 'DATA') || ($self->{type} eq 'COPY')) {
			$self->{dbh}->{LongReadLen} = 100000;
			$self->_tables();
		} elsif ($self->{type} eq 'VIEW') {
			$self->{dbh}->{LongReadLen} = 100000;
			$self->_views();
		} elsif ($self->{type} eq 'GRANT') {
			$self->_grants();
		} elsif ($self->{type} eq 'SEQUENCE') {
			$self->_sequences();
		} elsif ($self->{type} eq 'TRIGGER') {
			$self->{dbh}->{LongReadLen} = 100000;
			$self->_triggers();
		} elsif (($self->{type} eq 'FUNCTION') || ($self->{type} eq 'PROCEDURE')) {
			$self->{dbh}->{LongReadLen} = 100000;
			$self->_functions($self->{type});
		} elsif ($self->{type} eq 'PACKAGE') {
			$self->{dbh}->{LongReadLen} = 100000;
			$self->_packages();
		} elsif ($self->{type} eq 'TYPE') {
			$self->{dbh}->{LongReadLen} = 100000;
			$self->_types();
		} elsif ($self->{type} eq 'TABLESPACE') {
			$self->_tablespaces();
		} elsif ($self->{type} eq 'PARTITION') {
			$self->{dbh}->{LongReadLen} = 100000;
			$self->_partitions();
		} else {
			warn "type option must be TABLE, VIEW, GRANT, SEQUENCE, TRIGGER, PACKAGE, FUNCTION, PROCEDURE, PARTITION, DATA, COPY or TABLESPACE\n";
		}
		# Mofify export structure if required
		if ($self->{type} =~ /^(DATA|COPY)$/) {
			for my $t (keys %{$self->{'modify_struct'}}) {
				$self->modify_struct($t, @{$self->{'modify_struct'}{$t}});
			}
			$self->replace_tables(%{$self->{'replace_tables'}});
			$self->replace_cols(%{$self->{'replace_cols'}});
			$self->set_where_clause($self->{"global_where"}, %{$self->{'where'}});
		}
	}
	# Disconnect from the database
	$self->{dbh}->disconnect() if ($self->{dbh});

	exit(0) if ($self->{showtableid});

	$self->_send_to_pgdb() if ($self->{pg_dsn} && !$self->{dbhdest});
}


# We provide a DESTROY method so that the autoloader doesn't
# bother trying to find it. We also close the DB connexion
sub DESTROY { }


=head2 _send_to_pgdb DEST_DATASRC DEST_USER DEST_PASSWD

Open a DB handle to a PostgreSQL database

=cut

sub _send_to_pgdb
{
	my ($self, $destsrc, $destuser, $destpasswd) = @_;

	# Init with configuration options if no parameters
	$destsrc ||= $self->{pg_dsn};
	$destuser ||= $self->{pg_user};
	$destpasswd ||= $self->{pg_pwd};

        # Then connect the destination database
        $self->{dbhdest} = DBI->connect($destsrc, $destuser, $destpasswd);

	$destsrc =~ /dbname=([^;]*)/;
	$self->{dbname} = $1;
	$destsrc =~ /host=([^;]*)/;
	$self->{dbhost} = $1;
	$self->{dbhost} = 'localhost' if (!$self->{dbhost});
	$destsrc =~ /port=([^;]*)/;
	$self->{dbport} = $1;
	$self->{dbport} = 5432 if (!$self->{dbport});
	$self->{dbuser} = $destuser;

        # Check for connection failure
        if (!$self->{dbhdest}) {
		$self->logit("FATAL: $DBI::err ... $DBI::errstr\n", 0, 1);
	}

}

# Backward Compatibility
sub send_to_pgdb
{
	&_send_to_pgdb(@_);

}

=head2 _grants

This function is used to retrieve all privilege information.

It extracts all Oracle's ROLES to convert them to Postgres groups (or roles)
and searches all users associated to these roles.

=cut

sub _grants
{
	my ($self) = @_;

	if (!$self->{pg_supports_role}) {
		$self->logit("Retrieving users/roles/grants information...\n", 1);
		$self->{users} = $self->_get_users();
		$self->{groups} = $self->_get_roles();
		$self->{grants} = $self->_get_all_grants();
	} else {
		$self->{roles} = $self->_get_all_roles();
	}
}


=head2 _sequences

This function is used to retrieve all sequences information.

=cut

sub _sequences
{
	my ($self) = @_;

	$self->logit("Retrieving sequences information...\n", 1);
	$self->{sequences} = $self->_get_sequences();

}


=head2 _triggers

This function is used to retrieve all triggers information.

=cut

sub _triggers
{
	my ($self) = @_;

	$self->logit("Retrieving triggers information...\n", 1);
	$self->{triggers} = $self->_get_triggers();

}


=head2 _functions

This function is used to retrieve all functions information.

=cut

sub _functions
{
	my ($self, $type) = @_;

	$self->logit("Retrieving functions information...\n", 1);
	$self->{functions} = $self->_get_functions($type);

}


=head2 _packages

This function is used to retrieve all packages information.

=cut

sub _packages
{
	my ($self) = @_;

	$self->logit("Retrieving packages information...\n", 1);
	$self->{packages} = $self->_get_packages();

}


=head2 _types

This function is used to retrieve all custom types information.

=cut

sub _types
{
	my ($self) = @_;

	$self->logit("Retrieving user defined types information...\n", 1);
	$self->{types} = $self->_get_types();

}

=head2 _tables

This function is used to retrieve all table information.

Sets the main hash of the database structure $self->{tables}.
Keys are the names of all tables retrieved from the current
database. Each table information is composed of an array associated
to the table_info key as array reference. In other way:

    $self->{tables}{$class_name}{table_info} = [(OWNER,TYPE)];

DBI TYPE can be TABLE, VIEW, SYSTEM TABLE, GLOBAL TEMPORARY, LOCAL TEMPORARY,
ALIAS, SYNONYM or a data source specific type identifier. This only extracts
the TABLE type.

It also gets the following information in the DBI object to affect the
main hash of the database structure :

    $self->{tables}{$class_name}{field_name} = $sth->{NAME};
    $self->{tables}{$class_name}{field_type} = $sth->{TYPE};

It also calls these other private subroutines to affect the main hash
of the database structure :

    @{$self->{tables}{$class_name}{column_info}} = $self->_column_info($class_name, $owner);
    %{$self->{tables}{$class_name}{unique_key}}  = $self->_unique_key($class_name, $owner);
    @{$self->{tables}{$class_name}{foreign_key}} = $self->_foreign_key($class_name, $owner);
    %{$self->{tables}{$class_name}{check_constraint}}  = $self->_check_constraint($class_name, $owner);

=cut

sub _tables
{
	my ($self) = @_;

	# Get all tables information specified by the DBI method table_info
	$self->logit("Retrieving table information...\n", 1);

	my $sth = $self->_table_info()  or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	my @tables_infos = $sth->fetchall_arrayref();

	if ($self->{showtableid}) {
		foreach my $table (@tables_infos) {
			for (my $i=0; $i<=$#{$table};$i++) {
				my $nt = $i+1;
				$self->logit("[$nt] ${$table}[$i]->[2]\n", 1);
			}
		}
		return;
	}

	my @done = ();
	my $id = 0;
	foreach my $table (@tables_infos) {
		# Set the table information for each class found
		my $i = 1;
		$self->logit("Min table dump set to $self->{min}.\n", 1) if ($self->{min});
		$self->logit("Max table dump set to $self->{max}.\n", 1) if ($self->{max});
		foreach my $t (@$table) {
			# Jump to desired extraction
			if (grep(/^$t->[2]$/, @done)) {
				$self->logit("Duplicate entry found: $t->[0] - $t->[1] - $t->[2]\n", 1);
			} else {
				push(@done, $t->[2]);
			}
			$i++, next if ($self->{min} && ($i < $self->{min}));
			last if ($self->{max} && ($i > $self->{max}));
			next if (($#{$self->{limited}} >= 0) && !grep($t->[2] =~ /^$_$/i, @{$self->{limited}}));
			next if (($#{$self->{excluded}} >= 0) && grep($t->[2] =~ /^$_$/i, @{$self->{excluded}}));

			$self->logit("[$i] Scanning $t->[2] (@$t)...\n", 1);
			
			# Check of uniqueness of the table
			if (exists $self->{tables}{$t->[2]}{field_name}) {
				$self->logit("Warning duplicate table $t->[2], SYNONYME ? Skipped.\n", 1);
				next;
			}
			# Try to respect order specified in the TABLES limited extraction array
			$self->{tables}{$t->[2]}{internal_id} = 0;
			if ($#{$self->{limited}} >= 0) {
				for (my $j = 0; $j <= $#{$self->{limited}}; $j++) {
					if (uc($self->{limited}->[$j]) eq uc($t->[2])) {
						$self->{tables}{$t->[2]}{internal_id} = $j;
						last;
					}
				}
			}
			# usually OWNER,TYPE. QUALIFIER is omitted until I know what to do with that
			$self->{tables}{$t->[2]}{table_info} = [($t->[1],$t->[3])];
			# Set the fields information
			my $query = "SELECT * FROM $t->[1].$t->[2] WHERE 1=0";
			if ($self->{ora_sensitive}) {
				$query = "SELECT * FROM $t->[1].\"$t->[2]\" WHERE 1=0";
			}
			my $sth = $self->{dbh}->prepare($query);
			if (!defined($sth)) {
				warn "Can't prepare statement: $DBI::errstr";
				next;
			}
			$sth->execute;
			if ($sth->err) {
				warn "Can't execute statement: $DBI::errstr";
				next;
			}
			$self->{tables}{$t->[2]}{field_name} = $sth->{NAME};
			$self->{tables}{$t->[2]}{field_type} = $sth->{TYPE};

			@{$self->{tables}{$t->[2]}{column_info}} = $self->_column_info($t->[2],$t->[1]);
                        # We don't check for skip_ukeys/skip_pkeys here; this is taken care of inside _unique_key
			%{$self->{tables}{$t->[2]}{unique_key}} = $self->_unique_key($t->[2],$t->[1]);
			($self->{tables}{$t->[2]}{foreign_link}, $self->{tables}{$t->[2]}{foreign_key}) = $self->_foreign_key($t->[2],$t->[1]) if (!$self->{skip_fkeys});
			($self->{tables}{$t->[2]}{uniqueness}, $self->{tables}{$t->[2]}{indexes}) = $self->_get_indexes($t->[2],$t->[1]) if (!$self->{skip_indices});
			%{$self->{tables}{$t->[2]}{check_constraint}} = $self->_check_constraint($t->[2],$t->[1]) if (!$self->{skip_checks});
			$i++;
		}
	}

	# Try to search requested TABLE names in the VIEW names if not found in
	# real TABLE names
	if ($#{$self->{limited}} >= 0) {
		my $search_in_view = 0;
		foreach (@{$self->{limited}}) {
			if (not exists $self->{tables}{$_}) {
				$self->logit("Found view extraction for $_\n", 1);
				$search_in_view = 1;
				last;
			}
		}
		if ($search_in_view) {
			my %view_infos = $self->_get_views();
			foreach my $table (sort keys %view_infos) {
				# Set the table information for each class found
				# Jump to desired extraction
				next if (!grep($table =~ /^$_$/i, @{$self->{limited}}));
				$self->logit("Scanning view $table...\n", 1);

				$self->{views}{$table}{text} = $view_infos{$table};
				$self->{views}{$table}{alias}= $view_infos{$table}{alias};
				my $realview = $table;
				if ($self->{ora_sensitive}) {
					$realview = "\"$table\"";
				}
				if ($self->{schema}) {
					$realview = $self->{schema} . ".$realview";
				}
				# Set the fields information
				my $sth = $self->{dbh}->prepare("SELECT * FROM $realview WHERE 1=0");
				if (!defined($sth)) {
					warn "Can't prepare statement: $DBI::errstr";
					next;
				}
				$sth->execute;
				if ($sth->err) {
					warn "Can't execute statement: $DBI::errstr";
					next;
				}
				$self->{views}{$table}{field_name} = $sth->{NAME};
				$self->{views}{$table}{field_type} = $sth->{TYPE};
				@{$self->{views}{$table}{column_info}} = $self->_column_info($table);
			}
		}
	}
}


=head2 _views

This function is used to retrieve all views information.

Sets the main hash of the views definition $self->{views}.
Keys are the names of all views retrieved from the current
database and values are the text definitions of the views.

It then sets the main hash as follows:

    # Definition of the view
    $self->{views}{$table}{text} = $view_infos{$table};

=cut

sub _views
{
	my ($self) = @_;

	# Get all views information
	$self->logit("Retrieving views information...\n", 1);
	my %view_infos = $self->_get_views();

	if ($self->{showtableid}) {
		my $i = 1;
		foreach my $table (sort keys %view_infos) {
			$self->logit("[$i] $table\n", 1);
			$i++;
		}
		return;
	}

	$self->logit("Min view dump set to $self->{min}.\n", 1) if ($self->{min});
	$self->logit("Max view dump set to $self->{max}.\n", 1) if ($self->{max});
	my $i = 1;
	foreach my $table (sort keys %view_infos) {
		# Set the table information for each class found
		# Jump to desired extraction
		next if ($table =~ /\$/);
		$i++, next if ($self->{min} && ($i < $self->{min}));
		last if ($self->{max} && ($i > $self->{max}));
		next if (($#{$self->{limited}} >= 0) && !grep($table =~ /^$_$/i, @{$self->{limited}}));
		next if (($#{$self->{excluded}} >= 0) && grep($table =~ /^$_$/i, @{$self->{excluded}}));

		$self->logit("[$i] Scanning $table...\n", 1);

		$self->{views}{$table}{text} = $view_infos{$table};
                ## Added JFR : 3/3/02 : Retrieve also aliases from views
                $self->{views}{$table}{alias}= $view_infos{$table}{alias};
		$i++;
	}

}

=head2 _tablespaces

This function is used to retrieve all Oracle Tablespaces information.

Sets the main hash $self->{tablespaces}.

=cut

sub _tablespaces
{
	my ($self) = @_;

	$self->logit("Retrieving tablespaces information...\n", 1);
	$self->{tablespaces} = $self->_get_tablespaces();

}

=head2 _partitions

This function is used to retrieve all Oracle partition information.

Sets the main hash $self->{partition}.

=cut

sub _partitions
{
	my ($self) = @_;

	$self->logit("Retrieving partitions information...\n", 1);
	($self->{partitions}, $self->{partitions_default}) = $self->_get_partitions();

}


=head2 _get_sql_data

Returns a string containing the entire PostgreSQL compatible SQL Schema
definition.

=cut

sub _get_sql_data
{
	my ($self, $outfile) = @_;

	my $sql_header = "-- Generated by Ora2Pg, the Oracle database Schema converter, version $VERSION\n";
	$sql_header .= "-- Copyright 2000-2009 Gilles DAROLD. All rights reserved.\n";
	$sql_header .= "-- DATASOURCE: $self->{oracle_dsn}\n\n";
	if ($self->{export_schema} && ($self->{type} ne 'TABLE')) {
		if ($self->{pg_schema}) {
			$sql_header .= "SET search_path = $self->{pg_schema};\n\n";
		} else {
			if (!$self->{case_sensitive}) {
				$sql_header .= "SET search_path = \L$self->{schema}\E, pg_catalog;\n\n" if ($self->{schema});
			} else {
				$sql_header .= "SET search_path = $self->{schema}, pg_catalog;\n\n" if ($self->{schema});
			}
		}
	}
	if ($self->{type} eq 'DATA') {
		$sql_header .= "BEGIN TRANSACTION;\n\n";
	} else {
		$sql_header .= "\\set ON_ERROR_STOP ON\n\n";
	}

	my $sql_output = "";

	# Process view only
	if ($self->{type} eq 'VIEW') {
		$self->logit("Add views definition...\n", 1);
		foreach my $view (sort keys %{$self->{views}}) {
			$self->logit("\tAdding view $view...\n", 1);
			$self->{views}{$view}{text} =~ s/\s*\bWITH\b\s+.*$//s;
			$self->{views}{$view}{text} = $self->_format_view($self->{views}{$view}{text});
			if (!@{$self->{views}{$view}{alias}}) {
				if (!$self->{case_sensitive}) {
					$sql_output .= "CREATE VIEW \"\L$view\E\" AS ";
				} else {
					$sql_output .= "CREATE VIEW \"$view\" AS ";
				}
				$sql_output .= $self->{views}{$view}{text} . ";\n";
			} else {
				if (!$self->{case_sensitive}) {
					$sql_output .= "CREATE VIEW \"\L$view\E\" (";
				} else {
					$sql_output .= "CREATE VIEW \"$view\" (";
				}
				my $count = 0;
				foreach my $d (@{$self->{views}{$view}{alias}}) {
					if ($count == 0) {
						$count = 1;
					} else {
						$sql_output .= ", ";
					}
					if (!$self->{case_sensitive}) {
						$sql_output .= "\"\L$d->[0]\E\"";
					} else {
						$sql_output .= "\"$d->[0]\"";
					}
				}
				$sql_output .= ") AS " . $self->{views}{$view}{text} . ";\n";
			}
		}

		if (!$sql_output) {
			$sql_output = "-- Nothing found of type $self->{type}\n";
		} else {
			$sql_output .= "\n";
		}

		$self->dump($sql_header . $sql_output);

		return;
	}

	# Process grant only
	if ($self->{type} eq 'GRANT') {

		$self->logit("Add users/roles/grants privileges...\n", 1);
		my $grants = '';
		if (!$self->{pg_supports_role}) {
			# Add groups definition
			my $groups = '';
			my @users = ();
			my @grps = ();
			foreach my $u (@{$self->{users}}) {
				next if (exists $self->{groups}{"$u"});
				my $secret = 'change_my_secret';
				if ($self->{gen_user_pwd}) {
					$secret = &randpattern("CccnCccn");
				}
				$sql_header .= "CREATE USER $u WITH PASSWORD '$secret';\n";
			}
			foreach my $role (sort keys %{$self->{groups}}) {
				push(@grps, $role);
				$groups .= "CREATE GROUP $role WITH USER " . join(',', @{$self->{groups}{$role}}) . ";\n";
			}
			$sql_header .= "\n" . $groups . "\n";

			# Add privilege definition
			foreach my $table (sort keys %{$self->{grants}}) {
				$grants .= "REVOKE ALL ON $table FROM PUBLIC;\n";
				foreach my $priv (sort keys %{$self->{grants}{$table}}) {
					my $usr = '';
					my $grp = '';
					foreach my $user (@{$self->{grants}{$table}{$priv}}) {
						if (grep(/^$user$/, @grps)) {
							$grp .= "$user,";
						} else {
							$usr .= "$user,";
						}
					}
					$grp =~ s/,$//;
					$usr =~ s/,$//;
					if ($grp) {
						$grants .= "GRANT $priv ON $table TO GROUP $grp;\n";
					} else {
						$grants .= "GRANT $priv ON $table TO $usr;\n";
					}
				}
			}
		} else {
			foreach my $r (sort keys %{$self->{roles}}) {
				$self->logit("Extracting role $r\n",1);
				next if ($r eq 'CONNECT');
				$grants .= "CREATE ROLE $r WITH";
				if ($#{$self->{roles}{$r}{users}} >= 0) {
					$grants .= " USER " . join(',', @{$self->{roles}{$r}{users}});
				}
				if ($#{$self->{roles}{$r}{roles}} >= 0) {
					$grants .= " ROLE " . join(',', @{$self->{roles}{$r}{roles}});
				}
				# It's seems difficult to parse all kind of oracle privilege. So if one admin option is set we set all PG admin option.
				if (grep(/YES|1/, @{$self->{roles}{$r}{admin_option}})) {
					$grants .= " CREATEDB CREATEROLE CREATEUSER";
				}
				# if this role is found in the connect role allow login
				if (exists $self->{roles}{'CONNECT'} && grep(/^$r$/, @{$self->{roles}{'CONNECT'}{roles}})) {
					$grants .= " LOGIN";
				}
				$grants .= ";\n";
				# Now export all object granted to this role
				foreach my $t ( sort keys %{$self->{roles}{$r}{table}}) {
					if (grep(/EXECUTE/, @{$self->{roles}{$r}{table}{$t}})) {
						$grants .= "GRANT " . join(',', @{$self->{roles}{$r}{table}{$t}}) . " ON FUNCTION $t TO ROLE $r;\n";
					} else {
						$grants .= "GRANT " . join(',', @{$self->{roles}{$r}{table}{$t}}) . " ON $t TO ROLE $r;\n";
					}
				} 
			}
			# Now extract simple user allowed to connect
			if (exists $self->{roles}{'CONNECT'}) {
				foreach my $u (@{$self->{roles}{'CONNECT'}{users}}) {
					my $secret = 'change_my_secret';
					if ($self->{gen_user_pwd}) {
						$secret = &randpattern("CccnCccn");
					}
					$sql_header .= "CREATE USER $u WITH PASSWORD '$secret';\n";
				}
			}
		}
		if (!$grants) {
			$grants = "-- Nothing found of type $self->{type}\n";
		}

		$sql_output .= "\n" . $grants . "\n";

		$self->dump($sql_header . $sql_output);

		return;
	}

	# Process sequences only
	if ($self->{type} eq 'SEQUENCE') {
		$self->logit("Add sequences definition...\n", 1);
		foreach my $seq (@{$self->{sequences}}) {
			my $cache = 1;
			$cache = $seq->[5] if ($seq->[5]);
			my $cycle = '';
			$cycle = ' CYCLE' if ($seq->[6] eq 'Y');
			if ($seq->[2] > 2147483646) {
				$seq->[2] = 2147483646;
			}
			if ($seq->[1] < -2147483647) {
				$seq->[1] = -2147483647;
			}
			if (!$self->{case_sensitive}) {
				$sql_output .= "CREATE SEQUENCE \"\L$seq->[0]\E\" INCREMENT $seq->[3] MINVALUE $seq->[1] MAXVALUE $seq->[2] START $seq->[4] CACHE $cache$cycle;\n";
			} else {
				$sql_output .= "CREATE SEQUENCE \"$seq->[0]\" INCREMENT $seq->[3] MINVALUE $seq->[1] MAXVALUE $seq->[2] START $seq->[4] CACHE $cache$cycle;\n";
			}
		}

		if (!$sql_output) {
			$sql_output = "-- Nothing found of type $self->{type}\n";
		}

		$self->dump($sql_header . $sql_output);
		return;
	}

	# Process triggers only. PL/SQL code is pre-converted to PL/PGSQL following
	# the recommendation of Roberto Mello, see http://techdocs.postgresql.org/
	# Oracle's PL/SQL to PostgreSQL PL/pgSQL HOWTO  
	if ($self->{type} eq 'TRIGGER') {
		$self->logit("Add triggers definition...\n", 1);
		foreach my $trig (@{$self->{triggers}}) {
			$trig->[1] =~ s/\s*EACH ROW//is;
			chop($trig->[4]);
			chomp($trig->[4]);
			$self->logit("\tDumping trigger $trig->[0]...\n", 1);
			# Check if it's like a pg rule
			if ($trig->[1] =~ /INSTEAD OF/) {
				if (!$self->{case_sensitive}) {
					$sql_output .= "CREATE RULE \"\L$trig->[0]\E\" AS\n\tON \L$trig->[3]\E\n\tDO INSTEAD\n(\n\t$trig->[4]\n);\n\n";
				} else {
					$sql_output .= "CREATE RULE \"$trig->[0]\" AS\n\tON $trig->[3]\n\tDO INSTEAD\n(\n\t$trig->[4]\n);\n\n";
				}
			} else {
				if ($self->{plsql_pgsql}) {
					$trig->[4] = Ora2Pg::PLSQL::plsql_to_plpgsql($trig->[4]);
					$trig->[4] =~ s/\b(END[;]*)$/RETURN NEW;\n$1/igs;
				}
				if (!$self->{case_sensitive}) {
					$sql_output .= "CREATE FUNCTION pg_fct_\L$trig->[0]\E () RETURNS trigger AS \$BODY\$\n$trig->[4]\n\$BODY\$\n LANGUAGE 'plpgsql';\n\n";
					$sql_output .= "CREATE TRIGGER \L$trig->[0]\E\n\t$trig->[1] $trig->[2] ON \"\L$trig->[3]\E\" FOR EACH ROW\n\tEXECUTE PROCEDURE pg_fct_\L$trig->[0]\E();\n\n";
				} else {
					$sql_output .= "CREATE FUNCTION pg_fct_$trig->[0] () RETURNS trigger AS \$BODY\$\n$trig->[4]\n\$BODY\$ LANGUAGE 'plpgsql';\n\n";
					$sql_output .= "CREATE TRIGGER $trig->[0]\n\t$trig->[1] $trig->[2] ON \"$trig->[3]\" FOR EACH ROW\n\tEXECUTE PROCEDURE pg_fct_$trig->[0]();\n\n";
				}
			}
		}

		if (!$sql_output) {
			$sql_output = "-- Nothing found of type $self->{type}\n";
		}

		$self->dump($sql_header . $sql_output);
		return;
	}

	# Process functions only
	if (($self->{type} eq 'FUNCTION') || ($self->{type} eq 'PROCEDURE')) {
		use constant SQL_DATATYPE => 2;
		$self->logit("Add functions definition...\n", 1);
		foreach my $fct (sort keys %{$self->{functions}}) {
			if ($self->{plsql_pgsql}) {
				$sql_output .= $self->_convert_function($self->{functions}{$fct}) . "\n";
			} else {
				$sql_output .= $self->{functions}{$fct} . "\n";
			}
		}

		if (!$sql_output) {
			$sql_output = "-- Nothing found of type $self->{type}\n";
		}

		$self->dump($sql_header . $sql_output);
		return;
	}

	# Process packages only
	if ($self->{type} eq 'PACKAGE') {
		$self->logit("Add packages definition...\n", 1);
		foreach my $pkg (sort keys %{$self->{packages}}) {
			$self->logit("Dumping package $pkg...\n", 1);
			$sql_output .= "-- Oracle package '$pkg' declaration, please edit to match PostgreSQL syntax.\n";
			if ($self->{plsql_pgsql}) {
				$sql_output .= $self->_convert_package($self->{packages}{$pkg}) . "\n";
			} else {
				$sql_output .= $self->{packages}{$pkg} . "\n";
			}
			$sql_output .= "-- End of Oracle package '$pkg' declaration\n\n";
		}

		if (!$sql_output) {
			$sql_output = "-- Nothing found of type $self->{type}\n";
		}

		$self->dump($sql_header . $sql_output);
		return;
	}

	# Process types only
	if ($self->{type} eq 'TYPE') {
		$self->logit("Add custom types definition...\n", 1);
		foreach my $tpe (sort keys %{$self->{types}}) {
			$self->logit("Dumping type $tpe...\n", 1);
			$sql_output .= "-- Oracle type '$tpe' declaration, please edit to match PostgreSQL syntax.\n";
			if ($self->{plsql_pgsql}) {
				$sql_output .= $self->_convert_type($self->{types}{$tpe}) . "\n";
			} else {
				$sql_output .= $self->{types}{$tpe} . "\n";
			}
			$sql_output .= "-- End of Oracle type '$tpe' declaration\n\n";
		}

		if (!$sql_output) {
			$sql_output = "-- Nothing found of type $self->{type}\n";
		}

		$self->dump($sql_header . $sql_output);
		return;
	}

	# Process TABLESPACE only
	if ($self->{type} eq 'TABLESPACE') {
		$self->logit("Add tablespaces definition...\n", 1);
		$sql_header .= "-- Oracle tablespaces export, please edit path to match your filesystem.\n";
		$sql_header .= "-- In PostgreSQl the path must be a directory and is expected to already exists\n";
		my $create_tb = '';
		foreach my $tb_type (sort keys %{$self->{tablespaces}}) {
			# TYPE - TABLESPACE_NAME - FILEPATH - OBJECT_NAME
			foreach my $tb_name (sort keys %{$self->{tablespaces}{$tb_type}}) {
				foreach my $tb_path (sort keys %{$self->{tablespaces}{$tb_type}{$tb_name}}) {
					$create_tb .= "CREATE TABLESPACE $tb_name LOCATION '$tb_path';\n" if ($create_tb !~ /CREATE TABLESPACE $tb_name LOCATION/s);
					foreach my $obj (@{$self->{tablespaces}{$tb_type}{$tb_name}{$tb_path}}) {
						$sql_output .= "ALTER $tb_type $obj SET TABLESPACE $tb_name;\n";
					}
				}
			}
		}

		if (!$sql_output) {
			$sql_output = "-- Nothing found of type $self->{type}\n";
		}
		
		$self->dump($sql_header . "$create_tb\n" . $sql_output);

		return;
	}

	# Extract data only
	if (($self->{type} eq 'DATA') || ($self->{type} eq 'COPY')) {
		# Connect the database
		$self->{dbh} = DBI->connect($self->{oracle_dsn}, $self->{oracle_user}, $self->{oracle_pwd});
		# Check for connection failure
		if (!$self->{dbh}) {
			$self->logit("FATAL: $DBI::err ... $DBI::errstr\n", 0, 1);
		}

		if (!$self->{dbhdest}) {
			$self->dump($sql_header);
		} else {
			if ($self->{type} eq 'COPY') {
				if ($self->{dbuser}) {
					open(DBH, "| $PSQL -h $self->{dbhost} -p $self->{dbport} -d $self->{dbname} -U $self->{dbuser}") or $self->logit("FATAL: Can't open $PSQL command, $!\n", 0, 1);
				} else {
					# Executed as current user
					open(DBH, "| $PSQL -h $self->{dbhost} -p $self->{dbport} -d $self->{dbname}") or $self->logit("FATAL: Can't open $PSQL command, $!\n", 0, 1);
				}
			}
		}

		if ($self->{dbhdest} && $self->{export_schema} &&  $self->{schema}) {
			my $search_path = "SET search_path = $self->{schema}, pg_catalog";
			if ($self->{pg_schema}) {
				$search_path = "SET search_path = $self->{pg_schema}";
			}
			if ($self->{type} ne 'COPY') {
				my $s = $self->{dbhdest}->prepare($search_path) or $self->logit("FATAL: " . $self->{dbhdest}->errstr . "\n", 0, 1);
				$s->execute or $self->logit("FATAL: " . $s->errstr . "\n", 0, 1);
			} else {
				print DBH "$search_path;\n";
			}
		}
		if ($self->{defer_fkey}) {
			if ($self->{dbhdest}) {
				print DBH "SET CONSTRAINTS ALL DEFERRED;\n";
			} else {
				$self->dump("SET CONSTRAINTS ALL DEFERRED;\n");
			}
		}
		my $seq_dump = 0;
		foreach my $table (sort { $self->{tables}{$a}{internal_id} <=> $self->{tables}{$b}{internal_id} } keys %{$self->{tables}}) {
			$seq_dump = 1 if ($#{$self->{limited}} == -1);
			my $start_time = time();
			my $fhdl = undef;
			$self->logit("Dumping table $table...\n", 1);
			if ($self->{file_per_table} && !$self->{dbhdest}) {
				$self->dump("\\i ${table}__$self->{output}\n");
				$self->logit("Dumping to one file per table : ${table}_$self->{output}\n", 1);
				$fhdl = $self->export_file("${table}_$self->{output}");
			}
			## disable triggers of current table if requested
			if ($self->{disable_triggers}) {
				my $tmptb = $table;
				if (exists $self->{replaced_tables}{"\L$table\E"} && $self->{replaced_tables}{"\L$table\E"}) {
					$self->logit("\tReplacing table $table as " . $self->{replaced_tables}{lc($table)}, "...\n", 1);
					$tmptb = $self->{replaced_tables}{lc($table)};
				}
			    if ($self->{dbhdest}) {
				print DBH "ALTER TABLE $tmptb DISABLE TRIGGER $self->{disable_triggers};\n";
			    } else {
				if ($self->{file_per_table}) {
					$self->dumptofile($fhdl, "ALTER TABLE $tmptb DISABLE TRIGGER $self->{disable_triggers};\n");
				} else {
					$self->dump("ALTER TABLE $tmptb DISABLE TRIGGER $self->{disable_triggers};\n");
				}
			    }

			}

			my @tt = ();
			my @nn = ();
			my $total_record = 0;
			my $s_out = "INSERT INTO \"\L$table\E\" (";
			$s_out = "INSERT INTO \"$table\" (" if ($self->{case_sensitive});
			if ($self->{type} eq 'COPY') {
				$s_out = "\nCOPY \"\L$table\E\" ";
				$s_out = "\nCOPY \"$table\" " if ($self->{case_sensitive});
			}
			my @fname = ();
			foreach my $i ( 0 .. $#{$self->{tables}{$table}{field_name}} ) {
				my $fieldname = ${$self->{tables}{$table}{field_name}}[$i];
				if (!$self->{case_sensitive}) {
					if (exists $self->{modify}{"\L$table\E"}) {
						next if (!grep(/$fieldname/i, @{$self->{modify}{"\L$table\E"}}));
					}
				} else {
					if (exists $self->{modify}{"$table"}) {
						next if (!grep(/$fieldname/i, @{$self->{modify}{"$table"}}));
					}
				}
				if (!$self->{case_sensitive}) {
					push(@fname, lc($fieldname));
				} else {
					push(@fname, $fieldname);
				}

				foreach my $f (@{$self->{tables}{$table}{column_info}}) {
					next if ($f->[0] ne "$fieldname");
					my $type = $self->_sql_type($f->[1], $f->[2], $f->[5], $f->[6]);
					$type = "$f->[1], $f->[2]" if (!$type);
					push(@tt, $type);
					push(@nn, $f->[0]);
					if ($self->{type} ne 'COPY') {
						if (!$self->{case_sensitive}) {
							$s_out .= "\"\L$f->[0]\E\",";
						} else {
							$s_out .= "\"$f->[0]\",";
						}
					}
					last;
				}
			}
			if ($self->{type} eq 'COPY') {
				map { $_ = '"' . $_ . '"' } @fname;
				$s_out .= '(' . join(',', @fname) . ") FROM stdin;\n";
			}

			if ($self->{type} ne 'COPY') {
				$s_out =~ s/,$//;
				$s_out .= ") VALUES (";
			}

			# Change table name
			if (exists $self->{replaced_tables}{"\L$table\E"} && $self->{replaced_tables}{"\L$table\E"}) {
				$self->logit("\tReplacing table $table as " . $self->{replaced_tables}{lc($table)} . "...\n", 1);
				$s_out =~ s/INSERT INTO "$table"/INSERT INTO "$self->{replaced_tables}{lc($table)}"/si;
				$s_out =~ s/COPY "$table"/COPY "$self->{replaced_tables}{lc($table)}"/si;
			}
			# Change column names
			if (exists $self->{replaced_cols}{"\L$table\E"} && $self->{replaced_cols}{"\L$table\E"}) {
				foreach my $c (keys %{$self->{replaced_cols}{"\L$table\E"}}) {
					$self->logit("\tReplacing column $c as " . $self->{replaced_cols}{lc($table)}{$c} . "...\n", 1);
					$s_out =~ s/"$c"/"$self->{replaced_cols}{lc($table)}{$c}"/si;
					$s_out =~ s/"$table"/"$self->{replaced_cols}{lc($table)}{$c}"/si;
				}
			}
			# Extract all data from the current table
			$self->{data_current} = 0;
			$self->{data_end} = 0;
			while ( !$self->{data_end} ) {
				my $inter_time = time();
				my $sth = $self->_get_data($table, \@nn, \@tt);
				$self->{data_end} = 1 if (!$self->{data_limit});
				my $sql = '';
				if ($self->{type} eq 'COPY') {
					if ($self->{dbhdest}) {
						$sql = $s_out;
					} else {
						if ($self->{file_per_table}) {
							$self->dumptofile($fhdl, $s_out);
						} else {
							$self->dump($s_out);
						}
					}
				}
				my $count = 0;
				while (my $row = $sth->fetch) {
					if ($count % 1000 == 0) {
						$self->logit(".",1);
					}
					if ($self->{type} ne 'COPY') {
						if ($self->{dbhdest}) {
							$sql .= $s_out;
						} else {
							if ($self->{file_per_table}) {
								$self->dumptofile($fhdl, $s_out);
							} else {
								$self->dump($s_out);
							}
						}
					}
					for (my $i = 0; $i <= $#{$row}; $i++) {
						if ($self->{type} ne 'COPY') {
							if ($tt[$i] =~ /(char|date|time|text|bytea)/) {
								$row->[$i] =~ s/'/''/gs; # escape single quote
								$row->[$i] =~ s/\\/\\\\/g;
								$row->[$i] =~ s/\0/\\0/gs;
								$row->[$i] =~ s/([\1-\10])/sprintf("\\%03o", ord($1))/egs;
								$row->[$i] =~ s/([\13-\14])/sprintf("\\%03o", ord($1))/egs;
								$row->[$i] =~ s/([\16-\37])/sprintf("\\%03o", ord($1))/egs;
								if ($row->[$i] ne '') {
									$row->[$i] = "'$row->[$i]'";
								} else {
									$row->[$i] = 'NULL';
								}
								if ($self->{dbhdest}) {
									$sql .= $row->[$i];
								} else {
									if ($self->{file_per_table}) {
										$self->dumptofile($fhdl, $row->[$i]);
									} else {
										$self->dump($row->[$i]);
									}
								}
							} else {
								# Convert local decimal separator
								$row->[$i] =~ s/,/./;
								if ($row->[$i] eq '') {
									$row->[$i] = 'NULL';
								}
								if ($self->{dbhdest}) {
									$sql .= $row->[$i];
								} else {
									if ($self->{file_per_table}) {
										$self->dumptofile($fhdl, $row->[$i]);
									} else {
										$self->dump($row->[$i]);
									}
								}
							}
							if ($i < $#{$row}) {
								if ($self->{dbhdest}) {
									$sql .= ",";
								} else {
									if ($self->{file_per_table}) {
										$self->dumptofile($fhdl, ",");
									} else {
										$self->dump(",");
									}
								}
							}
						} else {
							$row->[$i] = &escape($row->[$i], $tt[$i], $self->{type}) if (!$self->{noescape});
							if ($tt[$i] eq 'bytea') {
								$row->[$i] =~ s/(.)/sprintf("\\\\%03o",ord($1))/ges;
							}
							if ($tt[$i] !~ /(char|date|time|text|bytea)/) {
								$row->[$i] =~ s/,/./;
							} elsif ($self->{noescape}) {
								# Even if noescape is set to on we always need to
								# take care for embedded characters which need to be
								# escaped. We have to look at characters which are likely
								# to destroy the copy format, e.g backslashes, \r, \t and \n
								$row->[$i] =~ s/\\/\\\\/g;
								$row->[$i] =~ s/\r/\\r/g;
								$row->[$i] =~ s/\n/\\n/g;
								$row->[$i] =~ s/\t/\\t/g;
							}
							if ($row->[$i] eq '') {
								$row->[$i] = '\N';
							}
							if ($self->{dbhdest}) {
								$sql .= $row->[$i];
							} else {
								if ($self->{file_per_table}) {
									$self->dumptofile($fhdl, $row->[$i]);
								} else {
									$self->dump($row->[$i]);
								}
							}
							if ($i < $#{$row}) {
								if ($self->{dbhdest}) {
									$sql .= "\t";
								} else {
									if ($self->{file_per_table}) {
										$self->dumptofile($fhdl, "\t");
									} else {
										$self->dump("\t");
									}
								}
							} else {
								if ($self->{dbhdest}) {
									$sql .= "\n";
								} else {
									if ($self->{file_per_table}) {
										$self->dumptofile($fhdl, "\n");
									} else {
										$self->dump("\n");
									}
								}
							}
						}
					}
					if ($self->{type} ne 'COPY') {
						if ($self->{dbhdest}) {
							$sql .= ");\n";
						} else {
							if ($self->{file_per_table}) {
								$self->dumptofile($fhdl, ");\n");
							} else {
								$self->dump(");\n");
							}
						}
					}
					$count++;
				}
				$total_record += $count;
				if ($self->{type} eq 'COPY') {
					if ($self->{dbhdest}) {
						$sql .= "\\.\n";
					} else {
						if ($self->{file_per_table}) {
							$self->dumptofile($fhdl, "\\.\n");
						} else {
							$self->dump("\\.\n");
						}
					}
				}
				if ($self->{data_limit}) {
					$self->{data_end} = 1 if ($count+1 < $self->{data_limit});
				}
				# Insert data if we are in online processing mode
				if ($self->{dbhdest}) {
					if ($self->{type} ne 'COPY') {
						my $s = $self->{dbhdest}->prepare($sql) or $self->logit("FATAL: " . $self->{dbhdest}->errstr . "\n", 0, 1);
						$s->execute or $self->logit("FATAL: " . $s->errstr . "\n", 0, 1);
					} else {
						print DBH "$sql";
					}
				}
				my $end_time = time();
				my $dt = $end_time - $inter_time;
				my $rps = sprintf("%2.1f", $count / ($dt+.0001));
				$self->logit("\n$count records in $dt secs = $rps recs/sec\n", 1);
			}

                        ## don't forget to enable all triggers if needed...
			if ($self->{disable_triggers}) {
				my $tmptb = $table;
				if (exists $self->{replaced_tables}{"\L$table\E"} && $self->{replaced_tables}{"\L$table\E"}) {
					$self->logit("\tReplacing table $table as " . $self->{replaced_tables}{lc($table)} . "...\n", 1);
					$tmptb = $self->{replaced_tables}{lc($table)};
				}
			    if ($self->{dbhdest}) {
				print DBH "ALTER TABLE $tmptb ENABLE TRIGGER $self->{disable_triggers};\n";
			    } else {
				if ($self->{file_per_table}) {
					$self->dumptofile($fhdl, "ALTER TABLE $tmptb ENABLE TRIGGER $self->{disable_triggers};\n");
				} else {
					$self->dump("ALTER TABLE $tmptb ENABLE TRIGGER $self->{disable_triggers};\n");
				}
			    }

			}
			if ($self->{file_per_table} && !$self->{dbhdest}) {
				$self->close_export_file($fhdl);
			}
			$self->logit("Total extracted records from table $table: $total_record\n", 1);
			my $end_time = time();
			my $dt = $end_time - $start_time;
			my $rps = sprintf("%.1f", $total_record / ($dt+.0001));
			$self->logit("in $dt secs = $rps recs/sec\n", 1);
		}

		# extract sequence information
		if ($seq_dump && !$self->{disable_sequence}) {
			$self->dump($self->_extract_sequence_info($self->{schema}));
		}

		# Extract data from view if requested
		my $search_in_view = 0;
		foreach (@{$self->{limited}}) {
			if (not exists $self->{tables}{$_} && not exists $self->{tables}{lc($_)}) {
				$self->logit("Found view data export for $_\n", 1);
				$search_in_view = 1;
				last;
			}
		}
		if ($search_in_view) {
			foreach my $table (sort keys %{$self->{views}}) {
				my $start_time = time();
				$self->logit("Dumping view $table...\n", 1);

				my $fhdl = undef;
				if ($self->{file_per_table} && !$self->{dbhdest}) {
					$self->dump("\\i ${table}_$self->{output}\n");
					$self->logit("Dumping to one file per table/view : ${table}_$self->{output}\n", 1);
					$fhdl = $self->export_file("${table}_$self->{output}");
				}

				## disable triggers of current table if requested
				if ($self->{disable_triggers}) {
					my $tmptb = $table;
					if (exists $self->{replaced_tables}{"\L$table\E"} && $self->{replaced_tables}{"\L$table\E"}) {
						$self->logit("\tReplacing table $table as " . $self->{replaced_tables}{lc($table)} . "...\n", 1);
						$tmptb = $self->{replaced_tables}{lc($table)};
					}
				    if ($self->{dbhdest}) {
					print DBH "ALTER TABLE $tmptb DISABLE TRIGGER $self->{disable_triggers};\n";
				    } else {
					if ($self->{file_per_table}) {
						$self->dumptofile($fhdl, "ALTER TABLE $tmptb DISABLE TRIGGER $self->{disable_triggers};\n");
					} else {
						$self->dump("ALTER TABLE $tmptb DISABLE TRIGGER $self->{disable_triggers};\n");
					}
				    }

				}

				my @tt = ();
				my @nn = ();
				my $total_record = 0;
				my $s_out = "INSERT INTO \"\L$table\E\" (";
				$s_out = "INSERT INTO \"$table\" (" if ($self->{case_sensitive});
				if ($self->{type} eq 'COPY') {
					$s_out = "\nCOPY \"\L$table\E\" ";
					$s_out = "\nCOPY \"$table\" " if ($self->{case_sensitive});
				}
				my @fname = ();
				foreach my $i ( 0 .. $#{$self->{views}{$table}{field_name}} ) {
					my $fieldname = ${$self->{views}{$table}{field_name}}[$i];
					if (!$self->{case_sensitive}) {
						if (exists $self->{modify}{"\L$table\E"}) {
							next if (!grep(/$fieldname/i, @{$self->{modify}{"\L$table\E"}}));
						}
					} else {
						if (exists $self->{modify}{"$table"}) {
							next if (!grep(/$fieldname/i, @{$self->{modify}{"$table"}}));
						}
					}
					if (!$self->{case_sensitive}) {
						push(@fname, lc($fieldname));
					} else {
						push(@fname, $fieldname);
					}

					foreach my $f (@{$self->{views}{$table}{column_info}}) {
						next if ($f->[0] ne "$fieldname");

						my $type = $self->_sql_type($f->[1], $f->[2], $f->[5], $f->[6]);
						$type = "$f->[1], $f->[2]" if (!$type);
						push(@tt, $type);
						push(@nn, $f->[0]);
						if ($self->{type} ne 'COPY') {
							if (!$self->{case_sensitive}) {
								$s_out .= "\"\L$f->[0]\E\",";
							} else {
								$s_out .= "\"$f->[0]\",";
							}
						}
						last;
					}
				}
				if ($self->{type} eq 'COPY') {
					map { $_ = '"' . $_ . '"' } @fname;
					$s_out .= '(' . join(',', @fname) . ") FROM stdin;\n";
				}

				if ($self->{type} ne 'COPY') {
					$s_out =~ s/,$//;
					$s_out .= ") VALUES (";
				}

				# Change table name
				if (exists $self->{replaced_tables}{"\L$table\E"} && $self->{replaced_tables}{"\L$table\E"}) {
					$self->logit("\tReplacing table $table as " . $self->{replaced_tables}{lc($table)} . "...\n", 1);
					$s_out =~ s/INSERT INTO "$table"/INSERT INTO "$self->{replaced_tables}{lc($table)}"/si;
					$s_out =~ s/COPY "$table"/COPY "$self->{replaced_tables}{lc($table)}"/si;
				}
				# Change column names
				if (exists $self->{replaced_cols}{"\L$table\E"} && $self->{replaced_cols}{"\L$table\E"}) {
					foreach my $c (keys %{$self->{replaced_cols}{"\L$table\E"}}) {
						$self->logit("\tReplacing column $c as " . $self->{replaced_cols}{lc($table)}{$c} . "...\n", 1);
						$s_out =~ s/"$c"/"$self->{replaced_cols}{lc($table)}{$c}"/si;
						$s_out =~ s/"$table"/"$self->{replaced_cols}{lc($table)}{$c}"/si;
					}
				}
				# Extract all data from the current table
				$self->{data_current} = 0;
				$self->{data_end} = 0;
				while ( !$self->{data_end} ) {
					my $inter_time = time();
					my $sth = $self->_get_data($table, \@nn, \@tt);
					$self->{data_end} = 1 if (!$self->{data_limit});
					my $sql = '';
					if ($self->{type} eq 'COPY') {
						if ($self->{dbhdest}) {
							$sql = $s_out;
						} else {
							if ($self->{file_per_table}) {
								$self->dumptofile($fhdl, $s_out);
							} else {
								$self->dump($s_out);
							}
						}
					}
					my $count = 0;
					while (my $row = $sth->fetch) {
						if ($count % 1000 == 0) {
							$self->logit(".", 1);
						}
						if ($self->{type} ne 'COPY') {
							if ($self->{dbhdest}) {
								$sql .= $s_out;
							} else {
								if ($self->{file_per_table}) {
									$self->dumptofile($fhdl, $s_out);
								} else {
									$self->dump($s_out);
								}
							}
						}
						for (my $i = 0; $i <= $#{$row}; $i++) {
							if ($self->{type} ne 'COPY') {
								if ($tt[$i] =~ /(char|date|time|text|bytea)/) {
									$row->[$i] =~ s/'/''/gs; # escape single quote
									$row->[$i] =~ s/\\/\\\\/g;
									$row->[$i] =~ s/\0/\\0/gs;
									$row->[$i] =~ s/([\1-\10])/sprintf("\\%03o", ord($1))/egs;
									$row->[$i] =~ s/([\13-\14])/sprintf("\\%03o", ord($1))/egs;
									$row->[$i] =~ s/([\16-\37])/sprintf("\\%03o", ord($1))/egs;
									if ($row->[$i] ne '') {
										$row->[$i] = "'$row->[$i]'";
									} else {
										$row->[$i] = 'NULL';
									}
									if ($self->{dbhdest}) {
										$sql .= $row->[$i];
									} else {
										if ($self->{file_per_table}) {
											$self->dumptofile($fhdl, $row->[$i]);
										} else {
											$self->dump($row->[$i]);
										}
									}
								} else {
									# Convert local decimal separator
									$row->[$i] =~ s/,/./;
									if ($row->[$i] eq '') {
										$row->[$i] = 'NULL';
									}
									if ($self->{dbhdest}) {
										$sql .= $row->[$i];
									} else {
										if ($self->{file_per_table}) {
											$self->dumptofile($fhdl, $row->[$i]);
										} else {
											$self->dump($row->[$i]);
										}
									}
								}
								if ($i < $#{$row}) {
									if ($self->{dbhdest}) {
										$sql .= ",";
									} else {
										if ($self->{file_per_table}) {
											$self->dumptofile($fhdl, ",");
										} else {
											$self->dump(",");
										}
									}
								}
							} else {
								$row->[$i] = &escape($row->[$i], $tt[$i], $self->{type}) if (!$self->{noescape});
								if ($tt[$i] eq 'bytea') {
									$row->[$i] =~ s/(.)/sprintf("\\\\%03o",ord($1))/ges;
								}
								if ($tt[$i] !~ /(char|date|time|text|bytea)/) {
									$row->[$i] =~ s/,/./;
								} elsif ($self->{noescape}) {
									# Even if noescape is set to on we always need to
									# take care for embedded characters which need to be
									# escaped. We have to look at characters which are likely
									# to destroy the copy format, e.g backslashes, \r, \t and \n
									$row->[$i] =~ s/\\/\\\\/g;
									$row->[$i] =~ s/\r/\\r/g;
									$row->[$i] =~ s/\n/\\n/g;
									$row->[$i] =~ s/\t/\\t/g;
								}
								if ($row->[$i] eq '') {
									$row->[$i] = '\N';
								}
								if ($self->{dbhdest}) {
									$sql .= $row->[$i];
								} else {
									if ($self->{file_per_table}) {
										$self->dumptofile($fhdl, $row->[$i]);
									} else {
										$self->dump($row->[$i]);
									}
								}
								if ($i < $#{$row}) {
									if ($self->{dbhdest}) {
										$sql .= "\t";
									} else {
										if ($self->{file_per_table}) {
											$self->dumptofile($fhdl, "\t");
										} else {
											$self->dump("\t");
										}
									}
								} else {
									if ($self->{dbhdest}) {
										$sql .= "\n";
									} else {
										if ($self->{file_per_table}) {
											$self->dumptofile($fhdl, "\n");
										} else {
											$self->dump("\n");
										}
									}
								}
							}
						}
						if ($self->{type} ne 'COPY') {
							if ($self->{dbhdest}) {
								$sql .= ");\n";
							} else {
								if ($self->{file_per_table}) {
									$self->dumptofile($fhdl, ");\n");
								} else {
									$self->dump(");\n");
								}
							}
						}
						$count++;
					}
					$total_record += $count;
					if ($self->{type} eq 'COPY') {
						if ($self->{dbhdest}) {
							$sql .= "\\.\n";
						} else {
							if ($self->{file_per_table}) {
								$self->dumptofile($fhdl, "\\.\n");
							} else {
								$self->dump("\\.\n");
							}
						}
					}
					if ($self->{data_limit}) {
						$self->{data_end} = 1 if ($count+1 < $self->{data_limit});
					}
					# Insert data if we are in online processing mode
					if ($self->{dbhdest}) {
						if ($self->{type} ne 'COPY') {
							my $s = $self->{dbhdest}->prepare($sql) or $self->logit("FATAL: " . $self->{dbhdest}->errstr . "\n", 0, 1);
							$s->execute or $self->logit("FATAL: " . $s->errstr . "\n");
						} else {
							print DBH "$sql";
						}
					}
					my $end_time = time();
					my $dt = $end_time - $inter_time;
					my $rps = sprintf("%2.1f", $count / ($dt+.0001));
					$self->logit("\n$count records in $dt secs = $rps recs/sec\n", 1);
				}

				## don't forget to enable all triggers if needed...
				if ($self->{disable_triggers}) {
					my $tmptb = $table;
					if (exists $self->{replaced_tables}{"\L$table\E"} && $self->{replaced_tables}{"\L$table\E"}) {
						$self->logit("\tReplacing table $table as " . $self->{replaced_tables}{lc($table)} . "...\n", 1);
						$tmptb = $self->{replaced_tables}{lc($table)};
					}
				    if ($self->{dbhdest}) {
					print DBH "ALTER TABLE $tmptb ENABLE TRIGGER $self->{disable_triggers};\n";
				    } else {
					if ($self->{file_per_table}) {
						$self->dumptofile($fhdl, "ALTER TABLE $tmptb ENABLE TRIGGER $self->{disable_triggers};\n");
					} else {
						$self->dump("ALTER TABLE $tmptb ENABLE TRIGGER $self->{disable_triggers};\n");
					}
				    }

				}

				$self->close_export_file($fhdl);
				$self->logit("Total extracted records from table $table: $total_record\n", 1);
				my $end_time = time();
				my $dt = $end_time - $start_time;
				my $rps = sprintf("%.1f", $total_record / ($dt+.0001));
				$self->logit("in $dt secs = $rps recs/sec\n", 1);
			}
		}

		if ( ($self->{type} eq 'DATA') && !$self->{dbhdest} ) {
			$self->dump("\nEND TRANSACTION;\n");
		}

		# Disconnect from the database
		$self->{dbh}->disconnect() if ($self->{dbh});
		$self->{dbhdest}->disconnect() if ($self->{dbhdest});

		if ($self->{type} eq 'COPY') {
			close DBH;
		}
		return;
	}

	# Process PARTITION only
	if ($self->{type} eq 'PARTITION') {
		$self->logit("Add partitions definition...\n", 1);
		$sql_header .= "-- Oracle partitions export.\n";
		$sql_header .= "-- Please take a look at the export to see if order and default table match your need\n";

		foreach my $table (sort keys %{$self->{partitions}}) {
			my $function = qq{
CREATE OR REPLACE FUNCTION ${table}_insert_trigger()
RETURNS TRIGGER AS \$\$
BEGIN
};
			my $cond = 'IF';
			my $funct_cond = '';
			my %create_table = ();
			foreach my $part (sort keys %{$self->{partitions}{$table}}) {
				$create_table{$table}{table} .= "CREATE TABLE ${table}_${part} ( CHECK (\n";
				my @condition = ();
				for (my $i = 0; $i <= $#{$self->{partitions}{$table}{$part}{column}}; $i++) {
					my $op = '<';
					$op = '=' if ($self->{partitions}{$table}{$part}{type}[$i] eq 'LIST');
					$create_table{$table}{table} .= "\t$self->{partitions}{$table}{$part}{column}[$i] $op $self->{partitions}{$table}{$part}{value}[$i]";
					$create_table{$table}{table} .= " AND" if ($i < $#{$self->{partitions}{$table}{$part}{column}});
					$create_table{$table}{'index'} .= "CREATE INDEX ${table}_${part}_$self->{partitions}{$table}{$part}{column}[$i] ON ${table}_${part} ($self->{partitions}{table}{$part}{column}[$i]);\n";
					push(@condition, "NEW.$self->{partitions}{$table}{$part}{column}[$i] $op $self->{partitions}{$table}{$part}{value}[$i]");
				}
				$create_table{$table}{table} .= "\n) ) INHERITS $table;\n";
				$funct_cond .= "\t$cond ( " . join(' AND ', @condition) . " ) THEN INSERT INTO ${table}_${part} VALUES (NEW.*);\n";
				$cond = 'ELSIF';
			}
			if (!$self->{partitions_default}{$table}) {
				$function .= $funct_cond . qq{
        ELSE
                INSERT INTO $table VALUES (NEW.*);
};
			} else {
				$function .= $funct_cond . qq{
        ELSE
                INSERT INTO ${table}_$self->{partitions_default}{$table} VALUES (NEW.*);
};
			}
			$function .= qq{
                -- Or if you prefer raising an exception
                -- RAISE EXCEPTION 'Value out of range. Fix the ${table}_insert_trigger() function!';
        END IF;
        RETURN NULL;
END;
\$\$
LANGUAGE plpgsql;
};

			$sql_output .= qq{
$create_table{$table}{table}
};
			$sql_output .= qq{
-- Create default table, where datas are inserted if no condition match
CREATE TABLE $table$self->{partitions_default}{$table} () INHERITS $table;
} if ($self->{partitions_default}{$table});
			$sql_output .= qq{
-- Create indexes on each partition table
$create_table{$table}{'index'}

$function

CREATE TRIGGER insert_${table}_trigger
    BEFORE INSERT ON $table
    FOR EACH ROW EXECUTE PROCEDURE ${table}_insert_trigger();

-------------------------------------------------------------------------------
};
		}

		if (!$sql_output) {
			$sql_output = "-- Nothing found of type $self->{type}\n";
		}
	
		$self->dump($sql_header . $sql_output);

		return;
	}

	# Dump the database structure
	if ($self->{export_schema} &&  $self->{schema}) {
		if (!$self->{case_sensitive}) {
			$sql_output .= "CREATE SCHEMA \"\L$self->{schema}\E\";\n\n";
			if ($self->{pg_schema}) {
				$sql_output .= "SET search_path = $self->{pg_schema};\n\n";
			} else {
				$sql_output .= "SET search_path = \L$self->{schema}\E, pg_catalog;\n\n";
			}
		} else {
			$sql_output .= "CREATE SCHEMA \"$self->{schema}\";\n\n";
			if ($self->{pg_schema}) {
				$sql_output .= "SET search_path = $self->{pg_schema};\n\n";
			} else {
				$sql_output .= "SET search_path = $self->{schema}, pg_catalog;\n\n";
			}
		}
	}

	foreach my $table (sort { $self->{tables}{$a}{internal_id} <=> $self->{tables}{$b}{internal_id} } keys %{$self->{tables}}) {
		$self->logit("Dumping table $table...\n", 1);
		if (!$self->{case_sensitive}) {
			$sql_output .= "CREATE ${$self->{tables}{$table}{table_info}}[1] \"\L$table\E\" (\n";
		} else {
			$sql_output .= "CREATE ${$self->{tables}{$table}{table_info}}[1] \"$table\" (\n";
		}
		foreach my $i ( 0 .. $#{$self->{tables}{$table}{field_name}} ) {
			foreach my $f (@{$self->{tables}{$table}{column_info}}) {
				next if ($f->[0] ne "${$self->{tables}{$table}{field_name}}[$i]");
				my $type = $self->_sql_type($f->[1], $f->[2], $f->[5], $f->[6]);
				$type = "$f->[1], $f->[2]" if (!$type);
				if (!$self->{case_sensitive}) {
					$sql_output .= "\t\"\L$f->[0]\E\" $type";
				} else {
					$sql_output .= "\t\"$f->[0]\" $type";
				}
				if ($f->[4] ne "") {
					$f->[4] =~  s/SYSDATE/CURRENT_TIMESTAMP/ig;
					$sql_output .= " DEFAULT $f->[4]";
				} elsif (!$f->[3] || ($f->[3] eq 'N')) {
					$sql_output .= " NOT NULL";
				}
				$sql_output .= ",\n";
				last;
			}
		}
		$sql_output =~ s/,$//;
		$sql_output .= ");\n";

		# Set the unique (and primary) key definition 
                my $newtabname = $self->{case_sensitive} ? $table : lc($table);
		foreach my $consname (keys %{$self->{tables}{$table}{unique_key}}) {
                        my $newconsname = $self->{case_sensitive} ? $consname : lc($consname);
                        my $constype =   $self->{tables}{$table}{unique_key}{$consname}{type};
                        my @conscols = @{$self->{tables}{$table}{unique_key}{$consname}{columns}};
                        my %constypenames = ('U' => 'UNIQUE', 'P' => 'PRIMARY KEY');
                        my $constypename = $constypenames{$constype};
                        my $columnlist = join(',', map(qq{"$_"}, @conscols));
                        $columnlist = lc($columnlist) unless ($self->{case_sensitive});
                        if(($constype ne 'P') || $self->{keep_pkey_names}) {
                            $sql_output .= qq{ALTER TABLE "$newtabname" ADD CONSTRAINT "$newconsname" $constypename ($columnlist);\n};
                        } else {
                            $sql_output .= qq{ALTER TABLE "$newtabname" ADD PRIMARY KEY ($columnlist);\n};
                        }
		}

		# Set the check constraint definition 
		foreach my $k (keys %{$self->{tables}{$table}{check_constraint}}) {
			if (!$self->{case_sensitive}) {
				my $first_letter = substr($self->{tables}{$table}{check_constraint}{$k},0,1);
				if($first_letter eq '"') {
					$sql_output .= "ALTER TABLE \"\L$table\E\" ADD CONSTRAINT \"\L$k\E\" CHECK (" . lc($self->{tables}{$table}{check_constraint}{$k}) . ");\n";
				} else {
					$sql_output .= "ALTER TABLE \"\L$table\E\" ADD CONSTRAINT \"\L$k\E\" CHECK (" . $self->{tables}{$table}{check_constraint}{$k} . ");\n";
				}
			} else {
				$sql_output .= "ALTER TABLE \"$table\" ADD CONSTRAINT \"$k\" CHECK (" . $self->{tables}{$table}{check_constraint}{$k} . ");\n";
			}
		}

		# Set the index definition
		foreach my $idx (keys %{$self->{tables}{$table}{indexes}}) {
			map { if ($_ !~ /\(.*\)/) { s/^/"/; s/$/"/; } } @{$self->{tables}{$table}{indexes}{$idx}};
			my $columns = join(',', @{$self->{tables}{$table}{indexes}{$idx}});
			my $unique = '';
			$unique = ' UNIQUE' if ($self->{tables}{$table}{uniqueness}{$idx} eq 'UNIQUE');
			if (!$self->{case_sensitive}) {
				$sql_output .= "CREATE$unique INDEX \L$idx\E ON \"\L$table\E\" (\L$columns\E);\n";
			} else {
				$sql_output .= "CREATE$unique INDEX $idx ON \"$table\" ($columns);\n";
			}
		}
		$sql_output .= "\n";
	}

	# Extract data from view if requested
	my $search_in_view = 0;
	foreach (@{$self->{limited}}) {
		if (not exists $self->{tables}{$_}) {
			$self->logit("Found view data export for $_\n", 1);
			$search_in_view = 1;
			last;
		}
	}
	if ($search_in_view) {
		foreach my $table (sort keys %{$self->{views}}) {
			$self->logit("Dumping views as table $table...\n", 1);
			if (!$self->{case_sensitive}) {
				$sql_output .= "CREATE TABLE \"\L$table\E\" (\n";
			} else {
				$sql_output .= "CREATE TABLE \"$table\" (\n";
			}
			foreach my $i ( 0 .. $#{$self->{views}{$table}{field_name}} ) {
				foreach my $f (@{$self->{views}{$table}{column_info}}) {
					next if ($f->[0] ne "${$self->{views}{$table}{field_name}}[$i]");
					my $type = $self->_sql_type($f->[1], $f->[2], $f->[5], $f->[6]);
					$type = "$f->[1], $f->[2]" if (!$type);
					if (!$self->{case_sensitive}) {
						$sql_output .= "\t\"\L$f->[0]\E\" $type";
					} else {
						$sql_output .= "\t\"$f->[0]\" $type";
					}
					if ($f->[4] ne "") {
						$f->[4] =~  s/SYSDATE/CURRENT_TIMESTAMP/ig;
						$sql_output .= " DEFAULT $f->[4]";
					} elsif (!$f->[3] || ($f->[3] eq 'N')) {
						$sql_output .= " NOT NULL";
					}
					$sql_output .= ",\n";
					last;
				}
			}
			$sql_output =~ s/,$//;
			$sql_output .= ");\n";
		}
	}

	foreach my $table (keys %{$self->{tables}}) {
		next if ($#{$self->{tables}{$table}{foreign_key}} < 0);
		$self->logit("Dumping RI $table...\n", 1);

		# Add constraint definition
		my @done = ();
		foreach my $h (@{$self->{tables}{$table}{foreign_key}}) {
			next if (grep(/^$h->[0]$/, @done));
			my $desttable = '';
			foreach (keys %{$self->{tables}{$table}{foreign_link}{$h->[0]}{remote}}) {
				$desttable .= "$_";
			}
			push(@done, $h->[0]);
			map { $_ = '"' . $_ . '"' } @{$self->{tables}{$table}{foreign_link}{$h->[0]}{local}};
			map { $_ = '"' . $_ . '"' } @{$self->{tables}{$table}{foreign_link}{$h->[0]}{remote}{$desttable}};
			if (!$self->{case_sensitive}) {
				$sql_output .= "ALTER TABLE \"\L$table\E\" ADD CONSTRAINT \"\L$h->[0]\E\" FOREIGN KEY (" . lc(join(',', @{$self->{tables}{$table}{foreign_link}{$h->[0]}{local}})) . ") REFERENCES \"\L$desttable\E\" (" . lc(join(',', @{$self->{tables}{$table}{foreign_link}{$h->[0]}{remote}{$desttable}})) . ")";
			} else {
				$sql_output .= "ALTER TABLE \"$table\" ADD CONSTRAINT \"$h->[0]\" FOREIGN KEY (" . join(',', @{$self->{tables}{$table}{foreign_link}{$h->[0]}{local}}) . ") REFERENCES \"$desttable\" (" . join(',', @{$self->{tables}{$table}{foreign_link}{$h->[0]}{remote}{$desttable}}) . ")";
			}
			$sql_output .= " MATCH $h->[2]" if ($h->[2]);
			$sql_output .= " ON DELETE $h->[3]";
			$sql_output .= " $h->[4]";
			$sql_output .= " INITIALLY $h->[5];\n";
		}
	}

	if (!$sql_output) {
		$sql_output = "-- Nothing found of type TABLE\n";
	}

	if ($self->{type} eq 'DATA') {
		$sql_output .= "\nEND TRANSACTION;\n";
	}
	$self->dump($sql_header . $sql_output);
}

=head2 _extract_sequence_info

This function retrieves the last value returned from the sequences in the
Oracle database. The result is a SQL script assigning the new start values
to the sequences found in the Oracle database.

=cut
sub _extract_sequence_info
{
	my ($self, $schema) = @_;

	my $sql = "SELECT SEQUENCE_NAME, MIN_VALUE, MAX_VALUE, INCREMENT_BY, CYCLE_FLAG, ORDER_FLAG, CACHE_SIZE, LAST_NUMBER FROM ";
	$sql .= "$self->{prefix}_SEQUENCES WHERE SEQUENCE_OWNER=? AND SEQUENCE_OWNER NOT IN ('" . join("','", @{$self->{sysusers}}) . "')";
	my $script = '';

	my $sth = $self->{dbh}->prepare($sql) or $self->logit("FATAL: " . $self->{dbh}->errstr ."\n", 0, 1);
	$sth->execute($schema) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);    

	while (my $seq_info = $sth->fetchrow_hashref) {
		my $nextvalue = $seq_info->{LAST_NUMBER} + $seq_info->{INCREMENT_BY};
		my $alter ="ALTER SEQUENCE $seq_info->{SEQUENCE_NAME} RESTART WITH $nextvalue;";
		$script .="$alter\n";
		$self->logit("Extracted sequence information for sequence \"$seq_info->{SEQUENCE_NAME}\"\n", 1);

	}

	$sth->finish();
	return $script;

}


=head2 _get_data TABLE

This function implements an Oracle-native data extraction.

Returns a list of array references containing the data

=cut

sub _get_data
{
	my ($self, $table, $name, $type) = @_;

	my $str = "SELECT ";
	my $tmp = "SELECT ";
	my $extraStr = "";
	for my $k (0 .. $#{$name}) {
		if (grep(/^$name->[$k]$/i, @{$self->{ora_reserved_words}})) {
			$name->[$k] = '"' . $name->[$k] . '"';
		}
		if ( $type->[$k] =~ /(date|time)/) {
			$str .= "to_char($name->[$k], 'YYYY-MM-DD HH24:MI:SS'),";
		} else {
			$str .= "$name->[$k],";
		}
		$tmp .= "$name->[$k],";
	}
	$str =~ s/,$//;
	$tmp =~ s/,$//;
	my $tmp2 = $tmp;
	$tmp2 =~ s/SELECT /SELECT ROWNUM as noline,/;

	# Fix a problem when the table need to be prefixed by the schema
	my $realtable = $table;
	if ($self->{ora_sensitive}) {
		$realtable = "\"$table\"";
	}
	if ($self->{schema}) {
		$realtable = $self->{schema} . ".$realtable";
	}
	# Fix a problem when using data_limit AND where clause
	if (exists $self->{where}{"\L$table\E"} && $self->{where}{"\L$table\E"}) {
		$extraStr .= ' AND (' . $self->{where}{"\L$table\E"} . ')';
	} elsif ($self->{global_where}) {
		$extraStr .= ' AND (' . $self->{global_where} . ')';
	}
	if ($self->{data_limit}) {
		$str = $str . " FROM ( $tmp2 FROM ( $tmp FROM $realtable) ";
		$str .= " WHERE ROWNUM < ($self->{data_limit} + $self->{data_current}) $extraStr) ";
		$str .= " WHERE noline >= $self->{data_current}";
	} else {
		$str .= " FROM $realtable";
	}
	if (exists $self->{where}{"\L$table\E"} && $self->{where}{"\L$table\E"}) {
		if ($str =~ / WHERE /) {
			$str .= ' AND ';
		} else {
			$str .= ' WHERE ';
		}
		$str .= '(' . $self->{where}{"\L$table\E"} . ')';
		$self->logit("\tApplying WHERE clause on table: " . $self->{where}{"\L$table\E"} . "\n", 1);
	} elsif ($self->{global_where}) {
		if ($str =~ / WHERE /) {
			$str .= ' AND ';
		} else {
			$str .= ' WHERE ';
		}
		$str .= '(' . $self->{global_where} . ')';
		$self->logit("\tApplying WHERE global clause: " . $self->{global_where} . "\n", 1);
	}

	$self->{data_current} += $self->{data_limit};

	# Fix a problem when exporting type LONG and LOB
	$self->{dbh}->{'LongReadLen'} = 1023*1024;
	$self->{dbh}->{'LongTruncOk'} = 1;

	my $sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);

	return $sth;	

}


=head2 _sql_type INTERNAL_TYPE LENGTH PRECISION SCALE

This function returns the PostgreSQL datatype corresponding to the
Oracle data type.

=cut

sub _sql_type
{
        my ($self, $type, $len, $precision, $scale) = @_;

	my $data_type = '';


        my %TYPE = (
		# Oracle only has one flexible underlying numeric type, NUMBER.
		# Without precision and scale it is set to the PG type float8
		# to match all needs
                'NUMBER' => 'numeric',
		# CHAR types limit of 2000 bytes with defaults to 1 if no length
		# is specified. PG char type has max length set to 8104 so it
		# should match all needs
                'CHAR' => 'char',
                'NCHAR' => 'char',
		# VARCHAR types the limit is 2000 bytes in Oracle 7 and 4000 in
		# Oracle 8. PG varchar type has max length iset to 8104 so it
		# should match all needs
                'VARCHAR' => 'varchar',
                'NVARCHAR' => 'varchar',
                'VARCHAR2' => 'varchar',
                'NVARCHAR2' => 'varchar',
		# The DATE data type is used to store the date and time
		# information. PG type timestamp should match all needs.
                'DATE' => 'timestamp',
		# Type LONG is like VARCHAR2 but with up to 2Gb. PG type text
		# should match all needs or if you want you could use blob
                'LONG' => 'text', # Character data of variable length
                'LONG RAW' => 'bytea',
		# Types LOB and FILE are like LONG but with up to 4Gb. PG type
		# text should match all needs or if you want you could use blob
		# (large object)
                'CLOB' => 'text', # A large object containing single-byte characters
                'NCLOB' => 'text', # A large object containing national character set data
                'BLOB' => 'bytea', # Binary large object
                'BFILE' => 'text', # Locator for external large binary file
		# The RAW type is presented as hexadecimal characters. The
		# contents are treated as binary data. Limit of 2000 bytes
		# PG type text should match all needs or if you want you could
		# use blob (large object)
                'RAW' => 'bytea',
                'ROWID' => 'oid',
                'FLOAT' => 'double precision',
		'DEC' => 'decimal',
		'DECIMAL' => 'decimal',
		'DOUBLE PRECISION' => 'double precision',
		'INT' => 'integer',
		'INTEGER' => 'integer',
		'REAL' => 'real',
		'SMALLINT' => 'smallint',
		'BINARY_FLOAT' => 'double precision',
		'BINARY_DOUBLE' => 'double precision',
		'TIMESTAMP' => 'timestamp',
		'BOOLEAN' => 'boolean',
        );
	# Simplify timestamp type
	$type =~ s/TIMESTAMP.*/TIMESTAMP/i;

	# Set user defined datatype translation
	if ($self->{datatype}) {
		my @transl = split(/[,;]/, $self->{datatype});
		foreach my $t (@transl) {
			my ($typ, $val) = split(/:/, $t);
			$typ =~ s/^\s+//;
			$typ =~ s/\s+$//;
			$val =~ s/^\s+//;
			$val =~ s/\s+$//;
			$TYPE{$typ} = $val if ($val);
		}
	}

        # Overide the length
        $len = $precision if ( ($type eq 'NUMBER') && $precision );

        if (exists $TYPE{uc($type)}) {
		$type = uc($type); # Force uppercase
		if ($len) {

			if ( ($type eq "CHAR") || ($type =~ /VARCHAR/) ) {
				# Type CHAR have default length set to 1
				# Type VARCHAR(2) must have a specified length
				$len = 1 if (!$len && ($type eq "CHAR"));
                		return "$TYPE{$type}($len)";
			} elsif ($type eq "NUMBER") {
				# This is an integer
				if (!$scale) {
					if ($precision) {
						if ($self->{pg_numeric_type}) {
							if ($precision < 5) {
								return 'smallint';
							} elsif ($precision <= 10) {
								return 'integer'; # The speediest in PG
							} else {
								return 'bigint';
							}
						}
						return "numeric($precision)";
					} elsif ($self->{pg_numeric_type}) {
						# Most of the time interger should be enought?
						return $self->{default_numeric} || 'float';
					}
				} else {
					if ($precision) {
						if ($self->{pg_numeric_type}) {
							if ($precision < 6) {
								return 'real';
							} else {
								return 'float';
							}
						}
						return "decimal($precision,$scale)";
					}
				}
			}
			return "$TYPE{$type}";
		} else {
			if (($type eq 'NUMBER') && $self->{pg_numeric_type}) {
				return $self->{default_numeric};
			} else {
				return $TYPE{$type};
			}
		}
        }

        return $type;
}


=head2 _column_info TABLE OWNER

This function implements an Oracle-native column information.

Returns a list of array references containing the following information
elements for each column the specified table

[(
  column name,
  column type,
  column length,
  nullable column,
  default value
)]

=cut

sub _column_info
{
	my ($self, $table, $owner) = @_;

	$owner = "AND OWNER='$owner' " if ($owner);
	my $sth = $self->{dbh}->prepare(<<END) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
SELECT COLUMN_NAME, DATA_TYPE, DATA_LENGTH, NULLABLE, DATA_DEFAULT, DATA_PRECISION, DATA_SCALE, CHAR_LENGTH
FROM $self->{prefix}_TAB_COLUMNS
WHERE TABLE_NAME='$table' $owner
ORDER BY COLUMN_ID
END
	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	my $data = $sth->fetchall_arrayref();
	foreach my $d (@$data) {
		$self->logit("\t$d->[0] => type:$d->[1] , length:$d->[2] (char_length:$d->[7]), precision:$d->[5], scale:$d->[6], nullable:$d->[3] , default:$d->[4]\n", 1);
		$d->[2] = $d->[7] if $d->[1] =~ /char/i;
	}

	return @$data;	

}


=head2 _unique_key TABLE OWNER

This function implements an Oracle-native unique (including primary)
key column information.

Returns a hash of hashes in the following form:
    ( constraintname => (type => 'PRIMARY',
                         columns => ('a', 'b', 'c')),
      constraintname => (type => 'UNIQUE',
                         columns => ('b', 'c', 'd')),
      etc.
    )

=cut

sub _unique_key
{
	my($self, $table, $owner) = @_;

	my %result = ();
        my @accepted_constraint_types = ();
        push @accepted_constraint_types, "'P'" unless($self->{skip_pkeys});
        push @accepted_constraint_types, "'U'" unless($self->{skip_ukeys});
        return %result unless(@accepted_constraint_types);
        my $cons_types = '('. join(',', @accepted_constraint_types) .')';
	$owner = "AND OWNER='$owner'" if ($owner);
	my $sth = $self->{dbh}->prepare(<<END) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
SELECT CONSTRAINT_NAME,R_CONSTRAINT_NAME,SEARCH_CONDITION,DELETE_RULE,DEFERRABLE,DEFERRED,R_OWNER,CONSTRAINT_TYPE
FROM $self->{prefix}_CONSTRAINTS
WHERE CONSTRAINT_TYPE IN $cons_types
AND STATUS='ENABLED'
AND TABLE_NAME='$table' $owner
END
	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);

	while (my $row = $sth->fetch) {
		my %constraint = (type => $row->[7], columns => ());
		my $sql = "SELECT DISTINCT COLUMN_NAME,POSITION FROM $self->{prefix}_CONS_COLUMNS WHERE CONSTRAINT_NAME='$row->[0]' $owner ORDER BY POSITION";
		my $sth2 = $self->{dbh}->prepare($sql) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
		$sth2->execute or $self->logit("FATAL: " . $sth2->errstr . "\n", 0, 1);
		while (my $r = $sth2->fetch) {
			push @{$constraint{'columns'}}, $r->[0];
		}
		$result{$row->[0]} = \%constraint;
	}

	return %result;
}

=head2 _check_constraint TABLE OWNER

This function implements an Oracle-native check constraint
information.

Returns a hash of lists of all column names defined as check constraints
for the specified table and constraint name.

=cut

sub _check_constraint
{
	my($self, $table, $owner) = @_;

	$owner = "AND OWNER='$owner'" if ($owner);
	my $sth = $self->{dbh}->prepare(<<END) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
SELECT CONSTRAINT_NAME,R_CONSTRAINT_NAME,SEARCH_CONDITION,DELETE_RULE,DEFERRABLE,DEFERRED,R_OWNER
FROM $self->{prefix}_CONSTRAINTS
WHERE CONSTRAINT_TYPE='C'
AND STATUS='ENABLED'
AND GENERATED != 'GENERATED NAME'
AND TABLE_NAME='$table' $owner
END
	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);

	my %data = ();
	while (my $row = $sth->fetch) {
		$data{$row->[0]} = $row->[2];
	}

	return %data;
}


=head2 _foreign_key TABLE OWNER

This function implements an Oracle-native foreign key reference
information.

Returns a list of hash of hash of array references. Ouf! Nothing very difficult.
The first hash is composed of all foreign key names. The second hash has just
two keys known as 'local' and 'remote' corresponding to the local table where
the foreign key is defined and the remote table referenced by the key.

The foreign key name is composed as follows:

    'local_table_name->remote_table_name'

Foreign key data consists in two arrays representing at the same index for the
local field and the remote field where the first one refers to the second one.
Just like this:

    @{$link{$fkey_name}{local}} = @local_columns;
    @{$link{$fkey_name}{remote}} = @remote_columns;

=cut

sub _foreign_key
{
	my ($self, $table, $owner) = @_;

	$owner = "AND OWNER='$owner'" if ($owner);
	my $deferrable = $self->{fkey_deferrable} ? "'DEFERRABLE' AS DEFERRABLE" : "DEFERRABLE";
	my $sth = $self->{dbh}->prepare(<<END) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
SELECT CONSTRAINT_NAME,R_CONSTRAINT_NAME,SEARCH_CONDITION,DELETE_RULE,$deferrable,DEFERRED,R_OWNER
FROM $self->{prefix}_CONSTRAINTS
WHERE CONSTRAINT_TYPE='R'
AND STATUS='ENABLED'
AND TABLE_NAME='$table' $owner
END
	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);

	my @data = ();
	my %link = ();
	my @tab_done = ();
	while (my $row = $sth->fetch) {
		next if (grep(/^$row->[0]$/, @tab_done));
		push(@data, [ @$row ]);
		push(@tab_done, $row->[0]);
		my $sql = "SELECT DISTINCT COLUMN_NAME,POSITION FROM $self->{prefix}_CONS_COLUMNS WHERE CONSTRAINT_NAME='$row->[0]' $owner ORDER BY POSITION";
		my $sth2 = $self->{dbh}->prepare($sql) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
		$sth2->execute or $self->logit("FATAL: " . $sth2->errstr . "\n", 0, 1);
		my @done = ();
		while (my $r = $sth2->fetch) {
			if (!grep(/^$r->[0]$/, @done)) {
				push(@{$link{$row->[0]}{local}}, $r->[0]);
				push(@done, $r->[0]);
			}
		}
		$owner = "AND OWNER = '$row->[6]'" if ($owner);
		$sql = "SELECT DISTINCT TABLE_NAME,COLUMN_NAME,POSITION FROM $self->{prefix}_CONS_COLUMNS WHERE CONSTRAINT_NAME='$row->[1]' $owner ORDER BY POSITION";
		$sth2 = $self->{dbh}->prepare($sql) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
		$sth2->execute or $self->logit("FATAL: " . $sth2->errstr . "\n", 0, 1);
		@done = ();
		while (my $r = $sth2->fetch) {
			if (!grep(/^$r->[1]$/, @done)) {
				push(@{$link{$row->[0]}{remote}{$r->[0]}}, $r->[1]);
				push(@done, $r->[1]);
			}

		}
	}

	return \%link, \@data;
}


=head2 _get_users

This function implements an Oracle-native users information.

Returns a hash of all users as an array.

=cut

sub _get_users
{
	my($self) = @_;

	# Retrieve all USERS defined in this database
	my $str = "SELECT USERNAME FROM $self->{prefix}_USERS";
	$str .= " ORDER BY USERNAME";
	my $sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);

	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	my @users = ();
	while (my $row = $sth->fetch) {
		push(@users, $row->[0]);
	}

	return \@users;
}


=head2 _get_roles

This function implements an Oracle-native roles information.

Returns a hash of all groups (roles) as an array of associated users.

=cut

sub _get_roles
{
	my($self) = @_;

	# Retrieve all ROLES defined in this database
	my $str = "SELECT GRANTED_ROLE,GRANTEE FROM $self->{prefix}_ROLE_PRIVS WHERE GRANTEE NOT IN (SELECT DISTINCT role FROM $self->{prefix}_ROLES)";
	my $sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);

	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	my %roles = ();
	while (my $row = $sth->fetch) {
		push(@{$roles{"$row->[0]"}}, $row->[1]);
	}

	return \%roles;
}

=head2 _get_all_roles

This function retrieves all Oracle roles information.

Returns a hash of all roles as an array of associated users.

=cut

sub _get_all_roles
{
	my($self) = @_;

	# Retrieve all ROLES defined in this database
	my $str = "SELECT ROLE FROM $self->{prefix}_ROLES WHERE ";
	if ($self->{schema}) {
		$str .= " ROLE='\U$self->{schema}\E'";
	} else {
		$str .= " ROLE NOT IN ('" . join("','", @{$self->{sysusers}}) . "')";
	}
	$str .= " ORDER BY ROLE";
	my $sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);

	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	my @roles = ();
	while (my $row = $sth->fetch) {
		push(@roles, $row->[0]);
	}
	$sth->finish();

	my %ROLES = ();

	# Get all user + roles + privilege granted to a role
	foreach my $r (@roles) {
		$str = "SELECT GRANTEE FROM $self->{prefix}_ROLE_PRIVS WHERE GRANTED_ROLE = '$r' AND GRANTEE IN (SELECT USERNAME FROM $self->{prefix}_USERS) AND GRANTEE NOT IN ('" . join("','", @{$self->{sysusers}}) . "')";
		$sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
		$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
		while (my $row = $sth->fetch) {
			push(@{$ROLES{$r}{users}}, $row->[0]);
		}
		$sth->finish();
		$str = "SELECT GRANTEE FROM $self->{prefix}_ROLE_PRIVS WHERE GRANTED_ROLE = '$r' AND GRANTEE IN (SELECT ROLE FROM $self->{prefix}_ROLES)";
		$sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
		$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
		while (my $row = $sth->fetch) {
			push(@{$ROLES{$r}{roles}}, $row->[0]);
		}
		$sth->finish();
		$str = "SELECT PRIVILEGE,ADMIN_OPTION FROM $self->{prefix}_SYS_PRIVS WHERE GRANTEE = '$r' ORDER BY PRIVILEGE";
		$sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
		$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
		while (my $row = $sth->fetch) {
			push(@{$ROLES{$r}{privilege}}, $row->[0]);
			push(@{$ROLES{$r}{admin_option}}, $row->[1]);
		}
		$sth->finish();
		# Object privileges granted to this role
		$str = "SELECT PRIVILEGE,TABLE_NAME FROM $self->{prefix}_TAB_PRIVS WHERE GRANTEE = '$r' ORDER BY GRANTOR, TABLE_NAME";
		$sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
		$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
		while (my $row = $sth->fetch) {
			push(@{$ROLES{$r}{table}{$row->[1]}}, $row->[0]);
		}
		$sth->finish();

	}
	return \%ROLES;
}



=head2 _get_all_grants

This function implements an Oracle-native user privilege information.

Returns a hash of all grants as an array of associated users.

=cut

sub _get_all_grants
{
	my($self) = @_;

	# Retrieve all ROLES defined in this database
	my $schema_clause = $self->{schema} ?  "AND GRANTOR = '\U$self->{schema}\E'" : '';
	my $str = "SELECT p.table_name, p.privilege, NVL(u.username, 'GROUP '||r.role) ".
		"FROM $self->{prefix}_TAB_PRIVS p, $self->{prefix}_USERS u, $self->{prefix}_ROLES r ".
		"WHERE p.PRIVILEGE IN ('DELETE', 'INSERT', 'SELECT', 'UPDATE') ".
		"      AND u.USERNAME (+)= p.GRANTEE ".
		"      AND r.ROLE (+)= p.GRANTEE ".
		"      $schema_clause ".
		"ORDER BY p.TABLE_NAME";

	my $sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);

	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	my %grants = ();
	while (my $row = $sth->fetch) {
		if ($row->[2] && ($row->[2] ne 'GROUP ')) {
			push(@{$grants{"$row->[0]"}{"$row->[1]"}}, $row->[2]);
		}
	}

	return \%grants;
}


=head2 _get_indexes TABLE OWNER

This function implements an Oracle-native indexes information.

Returns a hash of an array containing all unique indexes and a hash of
array of all indexe names which are not primary keys for the specified table.

=cut

sub _get_indexes
{
	my ($self, $table, $owner) = @_;

	my $idxowner = '';
	if ($owner) {
		$idxowner = "AND IC.TABLE_OWNER = '$owner'";
	}
	my $sub_owner = '';
	if ($owner) {
		$owner = "AND $self->{prefix}_INDEXES.OWNER='$owner' AND $self->{prefix}_IND_COLUMNS.INDEX_OWNER=$self->{prefix}_INDEXES.OWNER";
		$sub_owner = "AND OWNER=$self->{prefix}_INDEXES.TABLE_OWNER";
	}
	# Retrieve all indexes 
	my $sth = $self->{dbh}->prepare(<<END) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
SELECT DISTINCT $self->{prefix}_IND_COLUMNS.INDEX_NAME,$self->{prefix}_IND_COLUMNS.COLUMN_NAME,$self->{prefix}_INDEXES.UNIQUENESS,$self->{prefix}_IND_COLUMNS.COLUMN_POSITION
FROM $self->{prefix}_IND_COLUMNS, $self->{prefix}_INDEXES
WHERE $self->{prefix}_IND_COLUMNS.TABLE_NAME='$table' $owner
AND $self->{prefix}_INDEXES.INDEX_NAME=$self->{prefix}_IND_COLUMNS.INDEX_NAME
AND $self->{prefix}_IND_COLUMNS.INDEX_NAME NOT IN (SELECT CONSTRAINT_NAME FROM $self->{prefix}_CONSTRAINTS WHERE TABLE_NAME='$table' $sub_owner)
ORDER BY $self->{prefix}_IND_COLUMNS.COLUMN_POSITION
END
	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	my $idxnc = qq{SELECT IE.COLUMN_EXPRESSION FROM $self->{prefix}_IND_EXPRESSIONS IE, $self->{prefix}_IND_COLUMNS IC
WHERE  IE.INDEX_OWNER = IC.INDEX_OWNER
AND    IE.INDEX_NAME = IC.INDEX_NAME
AND    IE.TABLE_OWNER = IC.TABLE_OWNER
AND    IE.TABLE_NAME = IC.TABLE_NAME
AND    IE.COLUMN_POSITION = IC.COLUMN_POSITION
AND    IC.COLUMN_NAME = ?
AND    IE.TABLE_NAME = '$table'
$idxowner
};
	my $sth2 = $self->{dbh}->prepare($idxnc);
	my %data = ();
	my %unique = ();
	while (my $row = $sth->fetch) {
		$unique{$row->[0]} = $row->[2];
		# Replace function based index type
		if ($row->[1] =~ /^SYS_NC/i) {
			$sth2->execute($row->[1]) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
			my $nc = $sth2->fetch();
			$row->[1] = $nc->[0];
		}
		push(@{$data{$row->[0]}}, $row->[1]);
	}

	return \%unique, \%data;
}


=head2 _get_sequences

This function implements an Oracle-native sequences information.

Returns a hash of an array of sequence names with MIN_VALUE, MAX_VALUE,
INCREMENT and LAST_NUMBER for the specified table.

=cut

sub _get_sequences
{
	my($self) = @_;

	# Retrieve all indexes 
	my $str = "SELECT DISTINCT SEQUENCE_NAME, MIN_VALUE, MAX_VALUE, INCREMENT_BY, LAST_NUMBER, CACHE_SIZE, CYCLE_FLAG FROM $self->{prefix}_SEQUENCES";
	if (!$self->{schema}) {
		$str .= " WHERE SEQUENCE_OWNER NOT IN ('" . join("','", @{$self->{sysusers}}) . "')";
	} else {
		$str .= " WHERE SEQUENCE_OWNER = '\U$self->{schema}\E'";
	}
	my $sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);

	my @seqs = ();
	while (my $row = $sth->fetch) {
		push(@seqs, [ @$row ]);
	}

	return \@seqs;
}


=head2 _get_views

This function implements an Oracle-native views information.

Returns a hash of view names with the SQL queries they are based on.

=cut

sub _get_views
{
	my($self) = @_;

	# Retrieve all views
	my $str = "SELECT VIEW_NAME,TEXT FROM $self->{prefix}_VIEWS";
	if (!$self->{schema}) {
		$str .= " WHERE OWNER NOT IN ('" . join("','", @{$self->{sysusers}}) . "')";
	} else {
		$str .= " WHERE OWNER = '\U$self->{schema}\E'";
	}
	my $sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);

	my %data = ();
	while (my $row = $sth->fetch) {
		$data{$row->[0]} = $row->[1];
		@{$data{$row->[0]}{alias}} = $self->_alias_info ($row->[0]);
	}

	return %data;
}

=head2 _alias_info

This function implements an Oracle-native column information.

Returns a list of array references containing the following information
for each alias of the specified view:

[(
  column name,
  column id
)]

=cut

sub _alias_info
{
        my ($self, $view) = @_;

	my $str = "SELECT COLUMN_NAME, COLUMN_ID FROM $self->{prefix}_TAB_COLUMNS WHERE TABLE_NAME='$view'";
	if ($self->{schema}) {
		$str .= " AND OWNER = '\U$self->{schema}\E'";
	}
	$str .= " ORDER BY COLUMN_ID ASC";
        my $sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
        $sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
        my $data = $sth->fetchall_arrayref();
	foreach my $d (@$data) {
		$self->logit("\t$d->[0] =>  column id:$d->[1]\n", 1);
	}

        return @$data; 

}

=head2 _get_triggers

This function implements an Oracle-native triggers information. 

Returns an array of refarray of all triggers information.

=cut

sub _get_triggers
{
	my($self) = @_;

	# Retrieve all indexes 
	my $str = "SELECT TRIGGER_NAME, TRIGGER_TYPE, TRIGGERING_EVENT, TABLE_NAME, TRIGGER_BODY FROM $self->{prefix}_TRIGGERS WHERE STATUS='ENABLED'";
	if (!$self->{schema}) {
		$str .= " AND OWNER NOT IN ('" . join("','", @{$self->{sysusers}}) . "')";
	} else {
		$str .= " AND OWNER = '\U$self->{schema}\E'";
	}
	my $sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);

	my @triggers = ();
	while (my $row = $sth->fetch) {
		push(@triggers, [ @$row ]);
	}

	return \@triggers;
}


=head2 _get_functions

This function implements an Oracle-native functions information.

Returns a hash of all function names with their PLSQL code.

=cut

sub _get_functions
{
	my($self, $type) = @_;

	# Retrieve all indexes 
	my $str = "SELECT DISTINCT OBJECT_NAME,OWNER FROM $self->{prefix}_OBJECTS WHERE OBJECT_TYPE='$type' AND STATUS='VALID'";
	if (!$self->{schema}) {
		$str .= " AND OWNER NOT IN ('" . join("','", @{$self->{sysusers}}) . "')";
	} else {
		$str .= " AND OWNER = '\U$self->{schema}\E'";
	}
	my $sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);

	my %functions = ();
	my @fct_done = ();
	while (my $row = $sth->fetch) {
		next if (grep(/^$row->[0]$/, @fct_done));
		push(@fct_done, $row->[0]);
		my $sql = "SELECT TEXT FROM $self->{prefix}_SOURCE WHERE OWNER='$row->[1]' AND NAME='$row->[0]' ORDER BY LINE";
		my $sth2 = $self->{dbh}->prepare($sql) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
		$sth2->execute or $self->logit("FATAL: " . $sth2->errstr . "\n", 0, 1);
		while (my $r = $sth2->fetch) {
			$functions{"$row->[0]"} .= $r->[0];
		}
	}

	return \%functions;
}


=head2 _get_packages

This function implements an Oracle-native packages information.

Returns a hash of all package names with their PLSQL code.

=cut

sub _get_packages
{
	my ($self) = @_;

	# Retrieve all indexes 
	my $str = "SELECT DISTINCT OBJECT_NAME,OWNER FROM $self->{prefix}_OBJECTS WHERE OBJECT_TYPE='PACKAGE' AND STATUS='VALID'";
	if (!$self->{schema}) {
		$str .= " AND OWNER NOT IN ('" . join("','", @{$self->{sysusers}}) . "')";
	} else {
		$str .= " AND OWNER = '\U$self->{schema}\E'";
	}

	my $sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);

	my %packages = ();
	my @fct_done = ();
	while (my $row = $sth->fetch) {
		$self->logit("\tFound Package: $row->[0]\n", 1);
		next if (grep(/^$row->[0]$/, @fct_done));
		push(@fct_done, $row->[0]);
		my $sql = "SELECT TEXT FROM $self->{prefix}_SOURCE WHERE OWNER='$row->[1]' AND NAME='$row->[0]' AND (TYPE='PACKAGE' OR TYPE='PACKAGE BODY') ORDER BY TYPE, LINE";
		my $sth2 = $self->{dbh}->prepare($sql) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
		$sth2->execute or $self->logit("FATAL: " . $sth2->errstr . "\n", 0, 1);
		while (my $r = $sth2->fetch) {
			$packages{"$row->[0]"} .= $r->[0];
		}
	}

	return \%packages;
}

=head2 _get_types

This function implements an Oracle custom types information.

Returns a hash of all type names with their code.

=cut

sub _get_types
{
	my ($self) = @_;

	# Retrieve all indexes 
	my $str = "SELECT DISTINCT OBJECT_NAME,OWNER FROM $self->{prefix}_OBJECTS WHERE OBJECT_TYPE='TYPE' AND STATUS='VALID'";
	if (!$self->{schema}) {
		# We need to remove SYSTEM from the exclusion list
		shift(@{$self->{sysusers}});
		$str .= " AND OWNER NOT IN ('" . join("','", @{$self->{sysusers}}) . "')";
		unshift(@{$self->{sysusers}},'SYSTEM');
	} else {
		$str .= " AND (OWNER = '\U$self->{schema}\E' OR OWNER = 'SYSTEM')";
	}

	my $sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);

	my %types = ();
	my @fct_done = ();
	while (my $row = $sth->fetch) {
		$self->logit("\tFound Type: $row->[0]\n", 1);
		next if (grep(/^$row->[0]$/, @fct_done));
		push(@fct_done, $row->[0]);
		my $sql = "SELECT TEXT FROM $self->{prefix}_SOURCE WHERE OWNER='$row->[1]' AND NAME='$row->[0]' AND (TYPE='TYPE' OR TYPE='TYPE BODY') ORDER BY TYPE, LINE";
		my $sth2 = $self->{dbh}->prepare($sql) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
		$sth2->execute or $self->logit("FATAL: " . $sth2->errstr . "\n", 0, 1);
		while (my $r = $sth2->fetch) {
			$types{"$row->[0]"} .= $r->[0];
		}
	}

	return \%types;
}

=head2 _table_info

This function retrieves all Oracle-native tables information.

Returns a handle to a DB query statement.

=cut

sub _table_info
{
	my $self = shift;

	my $sql = "SELECT
                NULL            TABLE_CAT,
                at.OWNER        TABLE_SCHEM,
                at.TABLE_NAME,
                tc.TABLE_TYPE,
                tc.COMMENTS     REMARKS
            from ALL_TABLES at, ALL_TAB_COMMENTS tc
            where at.OWNER = tc.OWNER
            and at.TABLE_NAME = tc.TABLE_NAME
	";

	if ($self->{schema}) {
		$sql .= " and at.OWNER='\U$self->{schema}\E'";
	} else {
            $sql .= "AND at.OWNER NOT IN ('" . join("','", @{$self->{sysusers}}) . "')";
	}
        $sql .= " order by tc.TABLE_TYPE, at.OWNER, at.TABLE_NAME";
        my $sth = $self->{dbh}->prepare( $sql ) or return undef;
        $sth->execute or return undef;
        $sth;
}


=head2 _get_tablespaces

This function implements an Oracle-native tablespaces information.

Returns a hash of an array of tablespace names with their system file path.

=cut

sub _get_tablespaces
{
	my($self) = @_;

	# Retrieve all object with tablespaces.
my $str = qq{
SELECT a.SEGMENT_NAME,a.TABLESPACE_NAME,a.SEGMENT_TYPE,c.FILE_NAME
FROM $self->{prefix}_SEGMENTS a,$self->{prefix}_OBJECTS b, $self->{prefix}_DATA_FILES c
WHERE a.SEGMENT_TYPE IN ('INDEX', 'TABLE')
AND a.SEGMENT_NAME = b.OBJECT_NAME
AND a.SEGMENT_TYPE = b.OBJECT_TYPE
AND a.OWNER = b.OWNER
AND a.TABLESPACE_NAME = c.TABLESPACE_NAME
};
	if ($self->{schema}) {
		$str .= " AND a.OWNER='\U$self->{schema}\E'";
	} else {
		$str .= " AND a.TABLESPACE_NAME NOT IN ('SYSTEM','TOOLS')";
	}
	my $sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);

	my %tbs = ();
	while (my $row = $sth->fetch) {
		# TYPE - TABLESPACE_NAME - FILEPATH - OBJECT_NAME
		push(@{$tbs{$row->[2]}{$row->[1]}{$row->[3]}}, $row->[0]);
		$self->logit(".",1);
	}
	$sth->finish;
	$self->logit("\n", 1);

	return \%tbs;
}

=head2 _get_schema

This function returns 1 if the requested schema is found in the
database, otherwise 0.

=cut

sub _get_schema
{
	my($self) = @_;

	# Retrieve all USERS defined in this database
	my $str = "SELECT USERNAME FROM $self->{prefix}_USERS";
	$str .= " WHERE upper(USERNAME) = '\U$self->{schema}\E'";
	my $sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);

	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	while (my $row = $sth->fetch) {
		$sth->finish;
		return 1;
	}

	return 0;
}

=head2 _get_partitions

This function implements an Oracle-native partitions information.
Return two hash ref with partition details and partition default.
=cut

sub _get_partitions
{
	my($self) = @_;

	# Retrieve all partitions.
	my $str = qq{
SELECT
	a.table_name,
	a.partition_position,
	a.partition_name,
	a.high_value,
	a.tablespace_name,
	b.partitioning_type,
	c.name,
	c.column_name,
	c.column_position
FROM
	dba_tab_partitions a,
	dba_part_tables b,
	dba_part_key_columns c
WHERE
	a.table_name = b.table_name AND
	(b.partitioning_type = 'RANGE' OR b.partitioning_type = 'LIST')
	AND a.table_name = c.name
};
	if ($self->{prefix} ne 'USER') {
		if ($self->{schema}) {
			$str .= "\tAND a.table_owner ='\U$self->{schema}\E'\n";
		} else {
			$str .= "\tAND a.table_owner NOT IN ('" . join("','", @{$self->{sysusers}}) . "')\n";
		}
	}
	$str .= "ORDER BY a.partition_position\n";

	my $sth = $self->{dbh}->prepare($str) or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);
	$sth->execute or $self->logit("FATAL: " . $self->{dbh}->errstr . "\n", 0, 1);

	my %parts = ();
	my %default = ();
	while (my $rows = $sth->fetch) {
		if ( ($rows->[3] eq 'MAXVALUE') || ($rows->[3] eq 'DEFAULT')) {
			$default{$rows->[0]} = $rows->[2];
			next;
		}
		push(@{$parts{$rows->[0]}{$rows->[2]}{type}}, $rows->[5]);
		push(@{$parts{$rows->[0]}{$rows->[2]}{value}}, $rows->[3]);
		push(@{$parts{$rows->[0]}{$rows->[2]}{column}}, $rows->[7]);
		push(@{$parts{$rows->[0]}{$rows->[2]}{tablespace}}, $rows->[4]);
		$self->logit(".",1);
	}
	$sth->finish;
	$self->logit("\n", 1);

	return \%parts, \%default;
}

=head2 escape

This function escape all special characters in a specified string.

Returns the escaped string.

=cut

sub escape
{
	my ($str, $data_type, $action) = @_;

	return $str if ($data_type !~ /(char|date|time|text)/);

	$str =~ s/\\/\\\\/gs;
	$str =~ s/
/\\r/gs;
	my $CTRLM=chr(13);
	$str =~ s/$CTRLM//gs;
	$str =~ s/\n/\\n/gs;
	$str =~ s/\t/\\t/gs;
	$str =~ s/\f/\\f/gs;
	$str =~ s/\0/\\0/gs;
	$str =~ s/([\1-\10])/sprintf("\\%03o", ord($1))/egs;
	$str =~ s/([\13-\14])/sprintf("\\%03o", ord($1))/egs;
	$str =~ s/([\16-\37])/sprintf("\\%03o", ord($1))/egs;
	

	return $str;
}


=head2 dump

This function dump data to the right output (gzip file, file or stdout).

=cut

sub dump
{
	my ($self, $data) = @_;

	if ($self->{compress} eq 'Zlib') {
		$self->{zlib_hdl}->gzwrite($data) or $self->logit("FATAL: error writing compressed data\n", 0, 1);
	} else {
		if (defined $self->{fhout}) {
			$self->{fhout}->print($data);
		} else {
			print $data;
		}
	}

}


=head2 dumptofile

This function dump data to a given file handle.

=cut

sub dumptofile
{
	my ($self, $fh, $data) = @_;

	if ($self->{compress} eq 'Zlib') {
		$fh->gzwrite($data) or $self->logit("FATAL: error writing compressed data\n", 0, 1);
	} else {
		$fh->print($data);
	}

}

=head2 read_config

This function read the specified configuration file.

=cut

sub read_config
{
	my ($file) = @_;

	my $fh = new IO::File;
	$fh->open($file) or $self->logit("FATAL: can't read configuration file $file, $!\n", 0, 1);
	while (my $l = <$fh>) {
		chomp($l);
		$l =~ s/
//gs;
		$l =~ s/\#.*$//g;
		next if (!$l || ($l =~ /^[\s\t]+$/));
		$l =~ s/^\s*//; $l =~ s/\s*$//;
		my ($var, $val) = split(/[\s\t]+/, $l, 2);
		$var = uc($var);
                if ($var eq 'IMPORT') {
			if ($val) {
				$self->logit("Importing $val...\n", 1);
				&read_conf($val);
				$self->logit("Done importing $val.\n",1);
			}
		} elsif ($var =~ /^SKIP/) {
			if ($val) {
				$self->logit("No extraction of \L$val\E\n",1);
				my @skip = split(/[\s\t,]+/, $val);
				foreach (@skip) {
					$Config{"skip_\L$_\E"} = 1;
				}
			}
		} elsif (!grep(/^$var$/, 'TABLES', 'MODIFY_STRUCT', 'REPLACE_TABLES', 'REPLACE_COLS', 'WHERE', 'EXCLUDE', 'ORA_RESERVED_WORDS','SYSUSERS')) {
			$Config{$var} = $val;
		} elsif ( ($var eq 'TABLES') || ($var eq 'EXCLUDE') ) {
			push(@{$Config{$var}}, split(/\s+/, $val) );
		} elsif ( $var eq 'SYSUSERS' ) {
			push(@{$Config{$var}}, split(/[\s\t;,]+/, $val) );
		} elsif ( $var eq 'ORA_RESERVED_WORDS' ) {
			push(@{$Config{$var}}, split(/[\s\t;,]+/, $val) );
		} elsif ($var eq 'MODIFY_STRUCT') {
			while ($val =~ s/([^\(\s\t]+)[\t\s]*\(([^\)]+)\)[\t\s]*//) {
				my $table = $1;
				my $fields = $2;
				$fields =~ s/^\s+//;
				$fields =~ s/\s+$//;
				push(@{$Config{$var}{$table}}, split(/[\s,]+/, $fields) );
			}
		} elsif ($var eq 'REPLACE_COLS') {
			while ($val =~ s/([^\(\s\t]+)\s*\(([^\)]+)\)\s*//) {
				my $table = $1;
				my $fields = $2;
				$fields =~ s/^\s+//;
				$fields =~ s/\s+$//;
				my @rel = split(/[\t\s,]+/, $fields);
				foreach my $r (@rel) {
					my ($old, $new) = split(/:/, $r);
					$Config{$var}{$table}{$old} = $new;
				}
			}
		} elsif ($var eq 'REPLACE_TABLES') {
			my @replace_tables = split(/[\s\t]+/, $val);
			foreach my $r (@replace_tables) { 
				my ($old, $new) = split(/:/, $r);
				$Config{$var}{$old} = $new;
			}
		} elsif ($var eq 'WHERE') {
			while ($val =~ s/([^\[\s\t]+)[\t\s]*\[([^\]]+)\][\s\t]*//) {
				my $table = $1;
				my $where = $2;
				$where =~ s/^\s+//;
				$where =~ s/\s+$//;
				$Config{$var}{$table} = $where;
			}
			if ($val) {
				$Config{"GLOBAL_WHERE"} = $val;
			}
		}
	}
	$fh->close();
}


=head2 _convert_package

This function is used to rewrite Oracle PACKAGE code to
PostgreSQL SCHEMA. Called only if PLSQL_PGSQL configuration directive
is set to 1.

=cut

sub _convert_package
{
	my ($self, $plsql) = @_;

	my $content = '';
	if ($plsql =~ /PACKAGE[\s\t]+BODY[\s\t]*([^\s\t]+)[\s\t]*(AS|IS)\s*(.+)END[^;]*;/is) {
		my $pname = $1;
		my $type = $2;
		$content = $3;
		$pname =~ s/"//g;
		my %comments = $self->_remove_comments(\$content);
		my @functions = split(/\bFUNCTION|PROCEDURE\b/i, $content);
		$content = "-- PostgreSQL does not recognize PACKAGES. Use SCHEMA instead\n";
		$content .= "DROP SCHEMA IF EXISTS $pname CASCADE;\n";
		$content .= "CREATE SCHEMA $pname;\n" . shift(@functions);
		foreach my $f (@functions) {
			$content .= $self->_convert_function('FUNCTION'.$f);
		}
		$self->_restore_comments(\$content, \%comments);
	}
	return $content;
}

=head2 _restore_comments

This function is used to restore comments into SQL code previously
remove for easy parsing

=cut

sub _restore_comments
{
	my ($self, $content, $comments) = @_;

	foreach my $k (keys %$comments) {
		$$content =~ s/$k/$comments->{$k}/;
	}

}

=head2 _remove_comments

This function is used to remove comments from SQL code
to allow easy parsing

=cut

sub _remove_comments
{
	my ($self, $content) = @_;

	my %comments = ();
	my $i = 0;

	while ($$content =~ s/(\/\*(.*?)\*\/)/ORA2PG_COMMENT$i\%/s) {
		$comments{"ORA2PG_COMMENT$i%"} = $1;
		$i++;
	}
	my @lines = split(/\n/, $$content);
	for (my $j = 0; $j <= $#lines; $j++) {
		if ($lines[$j] =~ s/(--.*)$/ORA2PG_COMMENT$i\%/) {
			$comments{"ORA2PG_COMMENT$i%"} = "$1";
			$i++;
		}
	}
	$$content = join("\n", @lines);

	return %comments;
}

=head2 _convert_function

This function is used to rewrite Oracle FUNCTION code to
PostgreSQL. Called only if PLSQL_PGSQL configuration directive               
is set to 1.

=cut

sub _convert_function
{
	my ($self, $plsql) = @_;

	my $func_name = '';
	my @pg_args = ();
	my $func_ret_type = 'OPAQUE';
	my $hasreturn = 0;

	# Split data into declarative and code part
	my ($func_declare, $func_code) = split(/BEGIN/i,$plsql,2);
	if ( $func_declare =~ s/.*(FUNCTION|PROCEDURE)[\s\t]*([^\s\t\(]+)[\s\t]*(\([^\)]*\)|[\s\t]*)//is ) {
		$func_name = $2;
		$self->logit("\tDumping procedure $func_name...\n",1);
		my $func_args = $3;
		my $clause = '';
		my $code = '';
		$func_name =~ s/"//g;
		if ($func_declare =~ s/(.*)RETURN[\s\t]+([^\s\t]+)//is) {
			$func_args .= $1;
			$hasreturn = 1;
			$func_ret_type = $self->_sql_type($2) || 'OPAQUE';
		}
		if ($func_declare =~ s/(.*?)(USING|AS|IS)//is) {
			$func_args .= $1 if (!$hasreturn);
			$clause = $2;
		}
		# Clear argument syntax for better parsing
		$func_args =~ s/.*\([\t\s]*//s;
		$func_args =~ s/[\t\s]*\).*//s;
		foreach my $temp_arg (split(/[\s\t]*,[\s\t]*/s, $func_args)) {
			# Don't know what to do with default values, they are
			# not supported in PG. We just mark it into the
			# declaration for manual editing.
			my $default = '';
			if ($temp_arg =~ s/[\s\t]*(DEFAULT.*)//s) {
				$default = " UNSUPPORTED[ $1 ]";
			}
			# NOCOPY not supported
			$temp_arg =~ s/[\s\t]*NOCOPY//s;
			# IN OUT should be INOUT
			$temp_arg =~ s/IN[\s\t]+OUT/INOUT/s;
			@_ = split (/[\s\t]+/, $temp_arg);
			if ($#_ >= 2) {
				if (!$self->{pg_supports_inout}) {
					push(@pg_args, "$_[-3] " . $self->_sql_type($_[-1]) . $default);
				} else {
					# IN,OUT,INOUT mode are supported since
					#version 8.1
					push(@pg_args, "$_[-2] $_[-3] " . $self->_sql_type($_[-1]) . $default);
				}
			}
		}
		$func_declare = $self->_convert_declare($func_declare);
		if ($func_code) {
			# PLSQL is a Perl module that converts PLSQL code to PLPGSQL
			$func_code = Ora2Pg::PLSQL::plsql_to_plpgsql("BEGIN".$func_code);
		}
	} else {
		return $plsql;
	}
	if ($func_code) {
		$func_name= "\"$func_name\"" if ($self->{case_sensitive});
		my $function = "CREATE OR REPLACE FUNCTION $func_name (" . join(',', @pg_args) . ")";
		if ($hasreturn) {
			$function .= " RETURNS $func_ret_type AS \$body\$\n";
		} else {
			my @nout = grep(/OUT /i, @pg_args);
			if ($#nout > 0) {
				$function .= " RETURNS RECORD AS \$body\$\n";
			} elsif ($#nout == 0) {
				$nout[0] =~ s/.*OUT //;
				$function .= " RETURNS $nout[0] AS \$body\$\n";
			} else {
				$function .= " RETURNS VOID AS \$body\$\n";
			}
		}
		$function .= "DECLARE\n$func_declare\n" if ($func_declare);
		$function .= $func_code;
		$function .= "\n\$body\$\nLANGUAGE PLPGSQL;\n";
		return $function;
	}
	return $func_declare;
}

=head2 _convert_declare

This function is used to rewrite Oracle FUNCTION declaration code
to PostgreSQL. Called only if PLSQL_PGSQL configuration directive
is set to 1.

=cut

sub _convert_declare
{
	my ($self, $declare) = @_;

	$declare =~ s/^[\s\t]+//s;
	$declare =~ s/[\s\t]+$//s;

	return if (!$declare);

	my @allwithcomments = split(/(ORA2PG_COMMENT\d+\%\n*)/s, $declare);
	for (my $i = 0; $i <= $#allwithcomments; $i++) {
		next if ($allwithcomments[$i] =~ /ORA2PG_COMMENT/);
		my @pg_declare = ();
		foreach my $tmp_var (split(/;/,$allwithcomments[$i])) {
			# Not cursor declaration
			if ($tmp_var !~ /\bcursor\b/is) {
				# Extract default assignment
				my $tmp_assign = '';
				if ($tmp_var =~ s/[\s\t]*(:=|DEFAULT)(.*)$//is) {
					$tmp_assign = " $1$2";
				}
				# Extract variable name and type
				my $tmp_pref = '';
				my $tmp_name = '';
				my $tmp_type = '';
				if ($tmp_var =~ /([\s\t]*)([^\s\t]+)[\s\t]+(.*?)$/s) {
					$tmp_pref = $1;
					$tmp_name = $2;
					$tmp_type = $3;
					$tmp_type =~ s/[\s\t]+//gs;
					if ($tmp_type =~ /([^\(]+)\(([^\)]+)\)/) {
						my $type_name = $1;
						my ($prec, $scale) = split(/,/, $2);
						$scale ||= 0;
						my $len = $prec;
						$prec = 0 if (!$scale);
						$tmp_type = $self->_sql_type($type_name,$len,$prec,$scale);
					} else {
						$tmp_type = $self->_sql_type($tmp_type);
					}
					push(@pg_declare, "$tmp_pref$tmp_name $tmp_type$tmp_assign;");
				}
			} else {
				push(@pg_declare, "$tmp_var;");
			}
		}
		$allwithcomments[$i] = join("", @pg_declare);
	}

	return join("", @allwithcomments);

}


=head2 _format_view

This function is used to rewrite Oracle VIEW declaration code
to PostgreSQL.

=cut

sub _format_view
{
	my ($self, $sqlstr) = @_;

	# Add missing AS in column alias => optional in Oracle
	# and requiered in PostgreSQL
	if ($sqlstr =~ /(.*?)\bFROM\b(.*)/is) {
		my $item = $1;
		my $tmp = $2;
		# Disable coma between brackets
		my $i = 0;
		my @repstr = ();
		while ($item =~ s/(\([^\(\)]+\))/REPLACEME${i}HERE/s) {
			push(@repstr, $1);
			$i++;
		}
		$item =~ s/([a-z0-9_\$]+)([\t\s]+[a-z0-9_\$]+,)/$1 AS$2/igs;
		$item =~ s/([a-z0-9_\$]+)([\t\s]+[a-z0-9_\$]+)$/$1 AS$2/is;
		$sqlstr = $item . ' FROM ' . $tmp;
		while($sqlstr =~ /REPLACEME(\d+)HERE/) {
			$i = $1;
			$sqlstr =~ s/REPLACEME${i}HERE/$repstr[$i]/s;
		}
	}

	my @tbs = ();
	# Retrieve all tbs names used in view
	if ($sqlstr =~ /\bFROM\b(.*)/is) {
		my $tmp = $1;
		$tmp =~ s/\bWHERE.*//is;
		# Remove all SQL reserved words of FROM STATEMENT
		$tmp =~ s/(LEFT|RIGHT|INNER|OUTER|NATURAL|CROSS|JOIN|\(|\))//igs;
		# Remove all ON join, if any
		$tmp =~ s/\bON\b[A-Z_\.\s\t]*=[A-Z_\.\s\t]*//igs;
		# Sub , with whitespace
		$tmp =~ s/,/ /g;
		@tbs = split(/\s+/, $tmp);
	}
	foreach my $tb (@tbs) {
		next if (!$tb);
		my $regextb = $tb;
		$regextb =~ s/\$/\\\$/g;
		if (!$self->{case_sensitive}) {
			# Escape column name
			$sqlstr =~ s/["']*\b$regextb\b["']*\.["']*([A-Z_0-9\$]+)["']*(,?)/\L$tb\E.\L$1\E$2/igs;
			# Escape table name
			$sqlstr =~ s/(^=\s?)["']*\b$regextb\b["']*/\L$tb\E/igs;
			# Escape AS names
			$sqlstr =~ s/(\bAS[\s\t]*)["']*([A-Z_0-9]+)["']*/$1\L$2\E/igs;
		} else {
			# Escape column name
			$sqlstr =~ s/["']*\b${regextb}["']*\.["']*([A-Z_0-9\$]+)["']*(,?)/"$tb"."$1"$2/igs;
			# Escape table name
			$sqlstr =~ s/(^=\s?)["']*\b$regextb\b["']*/"$tb"/igs;
			# Escape AS names
			$sqlstr =~ s/(\bAS[\s\t]*)["']*([A-Z_0-9]+)["']*/$1"$2"/igs;
			if ($tb =~ /(.*)\.(.*)/) {
				my $prefx = $1;
				my $sufx = $2;
				$sqlstr =~ s/"$regextb"/"$prefx"\."$sufx/g;
			}
		}
	}

	return $sqlstr;
}

=head2 randpattern

This function is used to replace the use of perl module String::Random
and is simply a cut & paste from this module.

=cut

sub randpattern
{
	my $patt = shift;

	my $string = '';

	my @upper=("A".."Z");
	my @lower=("a".."z");
	my @digit=("0".."9");
	my %patterns = (
	    'C' => [ @upper ],
	    'c' => [ @lower ],
	    'n' => [ @digit ],
	);
	for my $ch (split(//, $patt)) {
		if (exists $patterns{$ch}) {
			$string .= $patterns{$ch}->[int(rand(scalar(@{$patterns{$ch}})))];
		} else {
			$string .= $ch;
		}
	}

	return $string;
}

=head2 logit

This function log information to STDOUT or to a logfile
following a debug level. If critical is set, it dies after
writing to log.

=cut

sub logit
{
	my ($self, $message, $level, $critical) = @_;

	$level ||= 0;

	if ($self->{debug} >= $level) {
		if (defined $self->{fhlog}) {
			$self->{fhlog}->print($message);
		} else {
			print "$message\n";
		}
	}
	if ($critical) {
		$self->{fhlog}->close() if (defined $self->{fhlog});
		die "Aborting export...\n";
	}
}

=head2 _convert_type

This function is used to rewrite Oracle PACKAGE code to
PostgreSQL SCHEMA. Called only if PLSQL_PGSQL configuration directive
is set to 1.

=cut

sub _convert_type
{
	my ($self, $plsql) = @_;

	my $content = '';
	if ($plsql =~ /TYPE[\t\s]+([^\t\s]+)[\t\s]+(IS|AS) TABLE OF[\t\s]+(.*)/is) {
		my $type_name = $1;
		my $tmp_type = $3;
		$tmp_type =~ /([^\(]+)\(([^\)]+)\)/;
		$tmp_type = $1;
		my ($prec, $scale) = split(/,/, $2);
		$tmp_type =~ s/[\s\t]//g;
		$scale ||= 0;
		my $len = $prec;
		my $type = $self->_sql_type($tmp_type,$len,$prec,$scale);
		$content = qq{
--
CREATE TYPE \"\L$type_name\E\";
CREATE FUNCTION \L$type_name\E_in_function(cstring) RETURNS \L$type_name\E AS
 ... CODE HERE WHAT TO DO WHEN INSERTING A VALUE ... ;
CREATE FUNCTION \L$type_name\E_out_function(\L$type_name\E) RETURNS cstring AS
 ... CODE HERE WHAT TO DO WHEN QUERYING THE VALUE ... ;
CREATE TYPE \"\L$type_name\E\" (
        INTERNALLENGTH = VARIABLE,
        INPUT = \L$type_name\E_in_function,
        OUTPUT = \L$type_name\E_out_function,
        ELEMENT = $type
);
};
	} elsif ($plsql =~ /TYPE[\t\s]+([^\t\s]+)[\t\s]+(AS|IS) OBJECT[\t\s]+\((.*)(TYPE BODY.*)/is) {
		my $type_name = $1;
		my $description = $3;
		my $body = $4;
		$description =~ s/[\s\t]*(MAP MEMBER |MEMBER )(FUNCTION|PROCEDURE).*//is;
		my $declar = '';
		my @attributes = split(/,/s, $description);
		map { s/[\r\n]+// } @attributes;
		foreach my $tmp_type (@attributes) {
			$tmp_type =~ s/^[\t\s]*([^\s\t]+)[\s\t]+//s;
			my $attrname = $1;
			my $type = '';
			if ($tmp_type =~ /([^\(]+)\(([^\)]+)\)/) {
				$tmp_type = $1;
				my $tmp = $2;
				$tmp =~ s/[^\d,]+//g;
				my ($prec, $scale) = split(/,/, $tmp);
				$tmp_type =~ s/[\s\t]//sg;
				$scale ||= 0;
				my $len = $prec;
				$type = $self->_sql_type($tmp_type,$len,$prec,$scale);
			} elsif ($tmp_type =~ /^([^\(]+)/) {
				$tmp_type = $1;
				$type = $self->_sql_type($tmp_type);
			}
			$declar .= "\t$attrname $type,\n";
		}
		$declar =~ s/,\n$//s;
		$content = qq{
CREATE TYPE \"\L$type_name\E\" AS (
$declar
);
};
		if ($body =~ /TYPE BODY[\s\t]+$type_name[\s\t]+AS[\s\t]*(.*)END;/is) {
			my $content2 = $1;
			my %comments = $self->_remove_comments(\$content2);
			my @functions = split(/(MAP MEMBER |MEMBER )(FUNCTION|PROCEDURE)/i, $content2);
			$content2 = '';
			foreach my $f (@functions) {
				$content .= $self->_convert_function('FUNCTION'.$f);
			}
			$content =~ s/FUNCTIONCREATE/CREATE/g;
			$self->_restore_comments(\$content, \%comments);

		}
	} elsif ($plsql =~ /TYPE[\t\s]+([^\t\s]+)[\t\s]+(AS|IS) OBJECT[\t\s]+\((.*)/is) {
		my $type_name = $1;
		my $description = $3;
		$description =~ s/[\s\t]*(MAP MEMBER |MEMBER )(FUNCTION|PROCEDURE).*//is;
		my $declar = '';
		my @attributes = split(/,/s, $description);
		map { s/[\r\n]+// } @attributes;
		foreach my $tmp_type (@attributes) {
			$tmp_type =~ s/^[\t\s]*([^\s\t]+)[\s\t]+//s;
			my $attrname = $1;
			my $type = '';
			if ($tmp_type =~ /([^\(]+)\(([^\)]+)\)/) {
				$tmp_type = $1;
				my $tmp = $2;
				$tmp =~ s/[^\d,]+//g;
				my ($prec, $scale) = split(/,/, $tmp);
				$tmp_type =~ s/[\s\t]//sg;
				$scale ||= 0;
				my $len = $prec;
				$type = $self->_sql_type($tmp_type,$len,$prec,$scale);
			} elsif ($tmp_type =~ /^([^\(]+)/) {
				$tmp_type = $1;
				$type = $self->_sql_type($tmp_type);
			}
			$declar .= "\t$attrname $type,\n";
		}
		$declar =~ s/,\n$//s;
		$content = qq{
CREATE TYPE \"\L$type_name\E\" AS (
$declar
);
};
	}
	return $content;
}

1;

__END__


=head1 AUTHOR

Gilles Darold <gilles _AT_ darold _DOT_ net>


=head1 COPYRIGHT

Copyright (c) 2000-2010 Gilles Darold - All rights reserved.

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program. If not, see < http://www.gnu.org/licenses/ >.


=head1 SEE ALSO

L<DBD::Oracle>, L<DBD::Pg>


=cut


