/***************************************************************************/
/* 		This code is part of Nscache - viewer of Netscape(tm)	   */
/*		browsers disk cache					   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <stdlib.h>
#include <ctype.h>
#include <gtk/gtk.h>

#ifdef HAVE_FNMATCH
#include <fnmatch.h>
#else
#include "fnmatch.h"
#endif

#include "indexdb.h"
#include "apassign.h"
#include "nls.h"
#include "gaccel.h"
#include "gprop.h"
#include "gui_tools.h"
#include "nscache.h"

#include "pixmaps/stock_close.xpm"
#include "pixmaps/stock_open.xpm"
#include "pixmaps/stock_expand.xpm"
#include "pixmaps/stock_colapse.xpm"

#include "pixmaps/nscache_logo.xpm"

#include "icons/audio.xpm"
#include "icons/binary.xpm"
#include "icons/html.xpm"
#include "icons/image.xpm"
#include "icons/text.xpm"
#include "icons/video.xpm"

enum {
	SEL_TYPE_NONE,
	COMPOUND_TEXT,
	STRING,
	TEXT,
	LAST_SEL_TYPE
};

enum {
	STYPE_URL ,
	STYPE_SIZE ,
	STYPE_ATM ,
	STYPE_MIME
} sort_type;

GtkWidget *tl;
GtkWidget *tlist;
GtkWidget *flow_menu;

static GtkWidget *slist,*status,*entry;
static GtkWidget *iw,*ilabel;
static GtkWidget *smi,*rmi,*err,*omi,*wmi,*umi,*fmi,*cmi,*nmi;
static GtkWidget *smi2,*rmi2,*omi2,*wmi2,*umi2,*fmi2,*cmi2,*nmi2;
static GtkWidget *sort_by;
GSList *elist = NULL;
static GHashTable *proth = NULL;
static GHashTable *servh = NULL;
static GSList *topnodes = NULL;
static guint tid = 0, etid = 0;
char *db_name = NULL;
static int recnum;
static int sitenum;
static long totsize;
static char *url_selection;
static int cvt_to_lower;

#ifndef O_BINARY
#define O_BINARY	0
#endif

static void Open();
static char* get_node_info(char* pom);
static void clist_col_set_visibility();

static struct {
	char *label;
	char *accel;
	char *prop;
	char *twprop;
	char *lwprop;
	char *colnrprop;
	int dwidth;
	int colnr;
	GtkJustification justify;
} colinfos[] = {
	{gettext_nop("Cache URL entry"), "config/col_url", "see_col_url", "tcol_url_width", "lcol_url_width", "url_colnr", 200, 0, GTK_JUSTIFY_LEFT},
	{gettext_nop("Local file"), "config/col_file", "see_col_file", "tcol_file_width", "lcol_file_width", "file_colnr", 150, 1, GTK_JUSTIFY_LEFT}, 
	{gettext_nop("Type"), "config/col_type" , "see_col_type", "tcol_type_width", "lcol_type_width", "type_colnr", 90, 2, GTK_JUSTIFY_LEFT},
	{gettext_nop("Size"), "config/col_size", "see_col_size", "tcol_size_width", "lcol_size_width", "size_colnr", 70, 3, GTK_JUSTIFY_RIGHT},
	{gettext_nop("Encoding"), "config/col_enc", "see_col_enc", "tcol_enc_width", "lcol_enc_width", "enc_colnr", 90, 4, GTK_JUSTIFY_LEFT},
	{gettext_nop("Charset"), "config/col_charset" , "see_col_charset", "tcol_charset_width", "lcol_charset_width", "charset_colnr", 90, 5, GTK_JUSTIFY_LEFT},
	{gettext_nop("Mod. time"), "config/col_mdtm" , "see_col_mdtm", "tcol_mdtm_width", "lcol_mdtm_width", "mdtm_colnr", 90, 6, GTK_JUSTIFY_LEFT},
	{gettext_nop("Access time"), "config/col_atm" , "see_col_atm", "tcol_atm_width", "lcol_atm_width", "atm_colnr", 90, 7, GTK_JUSTIFY_LEFT},
	{gettext_nop("Expires"), "config/col_exp" , "see_col_exp", "tcol_exp_width", "tcol_exp_width", "exp_colnr", 90, 8, GTK_JUSTIFY_LEFT},
};

struct nscache_icon_t nscache_icons[] = {
	{"text/html", html_xpm, NULL, NULL},
	{"text/*", text_xpm, NULL, NULL},
	{"image/*", image_xpm, NULL, NULL},
	{"audio/*", audio_xpm, NULL, NULL},
	{"video/*", video_xpm, NULL, NULL},
	{"*/*", binary_xpm, NULL, NULL},
	{NULL, NULL, NULL, NULL},
};

static void nscache_icons_load()
{
	int i;

	for (i = 0; nscache_icons[i].mimetype ; i++)
	{
		nscache_icons[i].pixmap = gdk_pixmap_create_from_xpm_d(
			GTK_WIDGET(tl)->window, &nscache_icons[i].mask,
			NULL, nscache_icons[i].icondata);
	}
}

struct nscache_icon_t *nscache_icon_get(mimetype)
char *mimetype;
{
	int i;

	for (i = 0; nscache_icons[i].mimetype ; i++)
	{
		if (!fnmatch(nscache_icons[i].mimetype, mimetype, 0))
			return &nscache_icons[i];
	}

	return NULL;
}

static char *str_to_lower(str)
char *str;
{
	char *p = str;

	while(*p)
	{
		*p = tolower(*p);
		p++;
	}

	return str;
}

static int ClrMsg(data)
gpointer data;
{
	gtk_label_set_text(GTK_LABEL(data) , "");
	gtk_timeout_remove(tid);
	tid = 0;

	return FALSE;
}

void gui_msg(str)
char *str;
{
	gtk_label_set_text(GTK_LABEL(status), str);
	if (tid) gtk_timeout_remove(tid);
	tid = gtk_timeout_add(10000 , (GtkFunction) ClrMsg , status);
}

static int ClrErr(data)
gpointer data;
{
	gtk_label_set_text(GTK_LABEL(data) , "");
	gtk_timeout_remove(etid);
	etid = 0;

	return FALSE;
}

void gui_err(str)
char *str;
{
	gtk_label_set_text(GTK_LABEL(err), str);
	if (etid) gtk_timeout_remove(etid);
	etid = gtk_timeout_add(10000 , (GtkFunction) ClrErr , err);
	gdk_beep();
}

static void Quit()
{
	save_rc();
	gtk_exit(0);
}

static void PopdownW(w , data)
GtkWidget *w;
gpointer data;
{
	gtk_widget_hide(GTK_WIDGET(data));
}

static void Help(w , data)
GtkWidget *w;
gpointer *data;
{
	static GtkWidget *sw = NULL;

	if (!sw)
	{
		char pom[512];
		GtkWidget *box,*label,*button,*frame,*pbox,*pmap;
		GdkPixmap *pix;
		GdkBitmap *mask;

		sw = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_window_set_title(GTK_WINDOW(sw), gettext("NScache: About"));
		gtk_container_set_border_width(GTK_CONTAINER(sw), 4);
		gtk_signal_connect(GTK_OBJECT(sw), "destroy",
			GTK_SIGNAL_FUNC(gtk_widget_destroyed), &sw);

		box = gtk_vbox_new(FALSE, 1);
		gtk_container_add(GTK_CONTAINER(sw) , box);
		gtk_widget_show(box);

		frame = gtk_frame_new(NULL);
		gtk_box_pack_start(GTK_BOX(box), frame, TRUE, FALSE, 10);
		gtk_widget_show(frame);

		pbox = gtk_hbox_new(FALSE, 1);
		gtk_container_add(GTK_CONTAINER(frame), pbox);
		gtk_widget_show(pbox);

		pix = gdk_pixmap_create_from_xpm_d(GTK_WIDGET(tl)->window, &mask, NULL, nscache_logo_xpm);
		pmap = gtk_pixmap_new(pix, mask);
		if (pix) gdk_pixmap_unref(pix);
		if (mask) gdk_bitmap_unref(mask);
		gtk_box_pack_start(GTK_BOX(pbox), pmap, TRUE, FALSE, 5);
		gtk_widget_show(pmap);

		sprintf(pom, gettext("Cache viewer for Netscape browsers disk cache\n\nVersion: %s\nAuthor: Stefan Ondrejicka <ondrej@idata.sk>\nURL: http://www.idata.sk/~ondrej/nscache/"), VERSION);
		label = gtk_label_new(pom);
		gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
		gtk_misc_set_alignment(GTK_MISC(label), 0.1, 0.5);
		gtk_misc_set_padding(GTK_MISC(label), 15, 15);
		gtk_box_pack_start(GTK_BOX(pbox), label, TRUE, FALSE, 5);
		gtk_widget_show(label);

		button = guitl_pixmap_button(stock_close_xpm, gettext("Close"));
		gtk_box_pack_start(GTK_BOX(box), button , FALSE , FALSE , 1);
		gtk_widget_show(button);

		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(PopdownW) , sw);
	}

	gtk_widget_show(sw);
	if (GTK_WIDGET_REALIZED(sw))
		gdk_window_raise(sw->window);
}

static void BrowseOK(w , data)
GtkWidget *w;
gpointer *data;
{
	char *fn = gtk_file_selection_get_filename(GTK_FILE_SELECTION(data));

	gtk_entry_set_text(GTK_ENTRY(entry) , fn);
	Open(NULL, entry);
}

static void Browse(w , data)
GtkWidget *w;
gpointer *data;
{
	static GtkWidget *fs = NULL;

	if (!fs)
	{
		char *fn;

		fs = gtk_file_selection_new(gettext("Select cache index ..."));

		gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
        	        "clicked", GTK_SIGNAL_FUNC(PopdownW), fs);

		gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
        	        "clicked", GTK_SIGNAL_FUNC(BrowseOK), fs);

		gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
        	        "clicked", GTK_SIGNAL_FUNC(PopdownW), fs);

		fn = gtk_entry_get_text(GTK_ENTRY(entry));

		if (fn && *fn)
			gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs), fn);
	}
	gtk_widget_show(fs);
	if (GTK_WIDGET_REALIZED(fs))
		gdk_window_raise(fs->window);
}

static int copy_file(sf , df)
char *sf;
char *df;
{
	char pom[8192];
	int len;
	int dfd,sfd;

	if ((sfd = open(sf , O_BINARY | O_RDONLY | O_CREAT , 0644)) < 0)
	{
		gui_err(gettext("Error opening source file"));
		return -1;
	}

	if ((dfd = open(df , O_BINARY | O_WRONLY | O_CREAT , 0644)) < 0)
	{
		gui_err(gettext("Error opening destination file"));
		close(sfd);
		return -1;
        }

	while((len = read(sfd , pom , sizeof(pom))) > 0)
	{
		if (len != write(dfd , pom , len))
		{
			gui_err(gettext("Error writing to destinantion file"));
			return -1;
		}
	}

	close(dfd);
	close(sfd);

	return 0;
}

static void Save(w, rec)
GtkWidget *w;
ns_cache_record *rec;
{
	GtkWidget *fs;
	char pom[2048];
	char *fn,*p;

	fs = gtk_widget_get_toplevel(w);
	fn = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs));

	strcpy(pom , ns_cache_get_db_name());

	p = strrchr(pom , '/');

	if (p)
	{
		strcpy(p+1 , rec->filename);
		if (cvt_to_lower)
			str_to_lower(p+1);
	}
	else
	{
		strcpy(pom , rec->filename);
		if (cvt_to_lower)
			str_to_lower(pom);
	}

	if (access(pom , R_OK))
	{
		gui_err(gettext("Source file can't be readed"));
	}
	else if (link(pom , fn))
	{
		if (errno == EPERM || errno == EXDEV)
		{
			copy_file(pom , fn);
			gtk_widget_destroy(fs);
		}
		else
		{
			gui_err(gettext("Error linking files"));
		}
	}
	else
		gtk_widget_destroy(fs);
}

static void SaveAs()
{
	ns_cache_record *rec;
	char *p;
	GtkWidget *fs;

	fs = gtk_file_selection_new(gettext("Save cache file as ..."));

	rec = (ns_cache_record *)gtk_ctree_node_get_row_data(GTK_CTREE(tlist),
		(GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data));

	p = strrchr(rec->urlstr, '/');
	if (p) p++;

	if (!p || !(*p)) p = "index.html";

	gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs) , p);

	gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
		"clicked", GTK_SIGNAL_FUNC(guitl_CloseWin), fs);

	gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
		"clicked", GTK_SIGNAL_FUNC(Save), rec);

	gtk_window_set_modal(GTK_WINDOW(fs) , TRUE);
	gtk_widget_show(fs);
}

static void unlink_file(name)
char *name;
{
	if (unlink(name))
	{
		char pom2[2348];

		sprintf(pom2 , gettext("Error removing cache file - %s") , name);
		gui_err(pom2);
	}
}

static void Remove()
{
	char pom[2048];
	char *p;
	ns_cache_record *rec;
	struct stat estat;
	GtkCTreeNode *prn = (GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data);
	GtkCTreeRow *rd = (GtkCTreeRow *) prn->list.data;
	gint row;


	rec = (ns_cache_record *)gtk_ctree_node_get_row_data(GTK_CTREE(tlist), prn);

	if (ns_cache_open_db(db_name))
	{
		sprintf(pom , gettext("Error opening cache index file : %s"), strerror(errno));
		gui_err(pom);
		return;
	}

	if (!rec)
	{
		if (rd->level == 2)
		{
			GtkCTreeNode *rn = rd->children;
			GtkCTreeNode *nrn;
			gboolean allrm = TRUE;

			gtk_clist_freeze(GTK_CLIST(tlist));
			while(rn)
			{
				rec = (ns_cache_record *)gtk_ctree_node_get_row_data(GTK_CTREE(tlist), rn);
				rd = rn->list.data;
				nrn = rd->sibling;

				strcpy(pom , db_name);

				p = strrchr(pom , '/');

				if (!p) p = pom;
				else p++;

				strcpy(p , rec->filename);
				if (cvt_to_lower)
					str_to_lower(p);
				if (!stat(pom , &estat))
				{
					if (estat.st_mode & S_IWUSR)
					{
						if (ns_cache_del_rec(rec->urlstr))
						{
							sprintf(pom, gettext("Unable to remove cache index entry - %s") , rec->urlstr);
							allrm = FALSE;
							gui_err(pom);
						}
						else
						{
							gtk_ctree_remove_node(GTK_CTREE(tlist), rn);
							row = gtk_clist_find_row_from_data(GTK_CLIST(slist), rec);
							if (row >= 0)
								gtk_clist_remove(GTK_CLIST(slist) , row);
							totsize -= rec->content_length;
							recnum --;
							unlink_file(pom);
						}
					}
					else
					{
						char pom2[2048];

						sprintf(pom2 , gettext("read only file : %s") , pom);
						gui_err(pom2);
					}
				}
				else
				{
					char pom2[2048];

					sprintf(pom2 , "%s : %s" , pom , strerror(errno));
					gui_err(pom2);
				}
				rn = nrn;
			}
			if (allrm)
			{
				sitenum --;
				gtk_ctree_remove_node(GTK_CTREE(tlist), prn);
				row = gtk_clist_find_row_from_data(GTK_CLIST(slist), rec);
				if (row >= 0)
					gtk_clist_remove(GTK_CLIST(slist) , row);
				gtk_clist_thaw(GTK_CLIST(tlist));
			}
		}
	}
	else
	{
		GtkCTreeNode *parent = rd->parent;

		rec = (ns_cache_record *)gtk_ctree_node_get_row_data(GTK_CTREE(tlist), prn);
		gtk_clist_freeze(GTK_CLIST(tlist));
		strcpy(pom , db_name);

		p = strrchr(pom , '/');

		if (!p) p = pom;
		else p++;

		strcpy(p , rec->filename);
		if (cvt_to_lower)
			str_to_lower(p);
		if (!stat(pom , &estat))
		{
			if (estat.st_mode & S_IWUSR)
			{
				if (ns_cache_del_rec(rec->urlstr))
				{
					sprintf(pom, gettext("Unable to remove cache index entry - %s") , rec->urlstr);
					gui_err(pom);
				}
				else
				{
					gtk_ctree_remove_node(GTK_CTREE(tlist), prn);
					row = gtk_clist_find_row_from_data(GTK_CLIST(slist), rec);
					if (row >= 0)
						gtk_clist_remove(GTK_CLIST(slist) , row);
					totsize -= rec->content_length;
					recnum --;
					unlink_file(pom);
				}
			}
			else
			{
				char pom2[2048];

				sprintf(pom2 , gettext("read only file : %s") , pom);
				gui_err(pom2);
			}
		}
		else
		{
			char pom2[2048];

			sprintf(pom2 , "%s : %s" , pom , strerror(errno));
			gui_err(pom2);
		}

		if (parent)
		{
			rd = (GtkCTreeRow *)parent->list.data;

			if (!rd->children)
			{
				sitenum --;
				gtk_ctree_remove_node(GTK_CTREE(tlist), parent);
			}
		}
		gtk_clist_thaw(GTK_CLIST(tlist));

	}
	ns_cache_close_db();
	gui_msg(get_node_info(pom));
}

static void compute_info(tree , node , data)
GtkCTree *tree;
GtkCTreeNode *node;
gpointer data;
{
	ns_cache_record *rec;
	struct {
		long sz;
		int num;
	} *nfo;

	nfo = data;

	rec = (ns_cache_record *)gtk_ctree_node_get_row_data(GTK_CTREE(tree), node);

	if (rec)
	{
		nfo->num ++;
		nfo->sz += rec->content_length;
	}
}

static char* get_node_info(char* pom)
{
	char* str_ptr;
	struct {
		long sz;
		int num;
	} nfo;
	long size;
	
	pom[0] = '\0';

	if (!GTK_CLIST(tlist)->selection)
		return pom;

	nfo.sz = 0;
	nfo.num = 0;

	gtk_ctree_post_recursive(GTK_CTREE(tlist) , 
		(GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data) ,
		GTK_CTREE_FUNC(compute_info) , &nfo);


	if (nfo.sz >= 1024)
	{
		size = nfo.sz / 1024;
		str_ptr = gettext("kB");
	}
	else
	{
		size = nfo.sz;
		str_ptr = gettext("Bytes");
	}

	sprintf(pom , gettext("Records: %d , Size: %ld %s"),
		nfo.num , size, str_ptr);

	return pom;
}

static void NodeInfo()
{
	char    pom[256];

	gui_msg(get_node_info(pom));
}


static void chmod_recursive(tree , node , data)
GtkCTree *tree;
GtkCTreeNode *node;
gpointer data;
{
	ns_cache_record *rec;
	struct {
		int set;
		mode_t mask;
	} *nfo;

	nfo = data;

	rec = (ns_cache_record *)gtk_ctree_node_get_row_data(GTK_CTREE(tree), node);

	if (rec)
	{
		char pom[2048];
		char *p;
		struct stat estat;

		strcpy(pom , db_name);
		p = strrchr(pom , '/');

		if (!p) p = pom;
		else p++;

		strcpy(p , rec->filename);
		if (cvt_to_lower)
			str_to_lower(p);

		if (!stat(pom , &estat))
		{
			if (nfo->set)
				estat.st_mode |= nfo->mask;
			else
				estat.st_mode &= ~nfo->mask;

			if (chmod(pom, estat.st_mode))
			{
				gui_err(gettext("Can't change mode for any of cache files"));
			}
		}
		else
		{
			gui_err(gettext("Some of cache file doesn't exist"));
		}
	}
}
	
static void RdOnly()
{
	struct {
		int set;
		mode_t mask;
	} nfo;
	nfo.set = 0;
	nfo.mask = S_IWUSR;

	gtk_ctree_post_recursive(GTK_CTREE(tlist) , 
		(GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data) ,
		GTK_CTREE_FUNC(chmod_recursive) , &nfo);

	gtk_widget_set_sensitive(omi, FALSE);
	gtk_widget_set_sensitive(omi2, FALSE);
	gtk_widget_set_sensitive(wmi, TRUE);
	gtk_widget_set_sensitive(wmi2, TRUE);
}

static void Writable()
{
	struct {
		int set;
		mode_t mask;
	} nfo;
	nfo.set = 1;
	nfo.mask = S_IWUSR;

	gtk_ctree_post_recursive(GTK_CTREE(tlist) , 
		(GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data) ,
		GTK_CTREE_FUNC(chmod_recursive) , &nfo);

	gtk_widget_set_sensitive(wmi, FALSE);
	gtk_widget_set_sensitive(wmi2, FALSE);
	gtk_widget_set_sensitive(omi, TRUE);
	gtk_widget_set_sensitive(omi2, TRUE);
}

static void _start_url_viewer(e, rec)
apassign *e;
ns_cache_record *rec;
{
	if (e && e->uapplication)
	{
		char pom[4098];
		char *p,*r;

		for (p = e->uapplication, r = pom; *p ; p++)
		{
			if (*p == '%' && *(p+1) == 'u')
			{
				p++;
				*r = '\"';
				r++;
				strcpy(r , rec->urlstr);
				r += strlen(r);
				*r = '\"';
				r++;
			}
			else
			{
				*r = *p;
				r++;
			}
		}
		*r = '&';
		r++;
		*r = '\0';

		system(pom);
		gui_msg(gettext("URL viewer launched"));
	}
}

static void ViewURL(w, e)
GtkWidget *w;
apassign *e;
{
	ns_cache_record *rec = (ns_cache_record *)
		gtk_ctree_node_get_row_data(GTK_CTREE(tlist), (GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data));

	_start_url_viewer(e, rec);
}

static void ViewURLDefault()
{
	ns_cache_record *rec = (ns_cache_record *)
		gtk_ctree_node_get_row_data(GTK_CTREE(tlist), (GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data));
	apassign *e = NULL;

	if (rec && rec->content_type)
	{
		GSList *ptr = apassign_get_matching_viewers(rec->content_type);
		if (ptr) e = (apassign *) ptr->data;
		_free_slist(ptr);
	}

	_start_url_viewer(e, rec);

}

static void _start_file_viewer(e, rec)
apassign *e;
ns_cache_record *rec;
{
	if (e && e->fapplication)
	{
		char pom[4098];
		char *p,*r;

		for (p = e->fapplication, r = pom; *p ; p++)
		{
			if (*p == '%' && *(p+1) == 'f')
			{
				char *pt;

				p++;
				*r = '\"';
				r++;

				pt = strrchr(db_name , '/');

				if (pt)		
				{
					strncpy(r , db_name , pt - db_name + 1);
					r += pt - db_name + 1;
					strcpy(r , rec->filename);
				}
				else strcpy(r , rec->filename);
				if (cvt_to_lower)
					str_to_lower(r);
				r += strlen(r);
				*r = '\"';
				r++;
			}
			else
			{
				*r = *p;
				r++;
			}
		}
		*r = '&';
		r++;
		*r = '\0';

		system(pom);
		gui_msg(gettext("File viewer launched"));
	}
}

static void ViewFile(w, e)
GtkWidget *w;
apassign *e;
{
	ns_cache_record *rec = (ns_cache_record *)
		gtk_ctree_node_get_row_data(GTK_CTREE(tlist), (GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data));

	_start_file_viewer(e, rec);
}

static void ViewFileDefault()
{
	ns_cache_record *rec = (ns_cache_record *)
		gtk_ctree_node_get_row_data(GTK_CTREE(tlist), (GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data));
	apassign *e = NULL;

	if (rec && rec->content_type)
	{
		GSList *ptr = apassign_get_matching_viewers(rec->content_type);
		if (ptr) e = (apassign *) ptr->data;
		_free_slist(ptr);
	}

	_start_file_viewer(e, rec);
}

static void GenViewerMenu(smenu, fu)
GtkWidget *smenu;
int fu;
{
	GtkWidget *mi;
	GSList *ptr;
	ns_cache_record *rec = (ns_cache_record *)
		gtk_ctree_node_get_row_data(GTK_CTREE(tlist), (GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data));

	if (!rec || !rec->content_type)
		return;

	while (GTK_MENU_SHELL(smenu)->children)
		gtk_widget_destroy(GTK_WIDGET(GTK_MENU_SHELL(smenu)->children->data));

	ptr = apassign_get_matching_viewers(rec->content_type);

	while (ptr)
	{
		apassign *e = ptr->data;
		char *p;

		p = fu ? e->fapplication : e->uapplication;

		if (p && *p)
		{
			p = g_strndup(p, strcspn(p, " "));

			mi = gtk_menu_item_new_with_label(p);
			gtk_menu_append(GTK_MENU(smenu), mi);
			gtk_widget_show(mi);

			gtk_signal_connect(GTK_OBJECT(mi), "activate",
				fu ? GTK_SIGNAL_FUNC(ViewFile) :
				     GTK_SIGNAL_FUNC(ViewURL), e);

			g_free(p);
		}

		ptr = g_slist_remove_link(ptr, ptr);
	}
}

static void SetSelectionURL()
{
	ns_cache_record *rec = (ns_cache_record *)
		gtk_ctree_node_get_row_data(GTK_CTREE(tlist), (GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data));

	if (!url_selection)
		gtk_selection_owner_set(tl , GDK_SELECTION_PRIMARY ,GDK_CURRENT_TIME);

	if (rec && rec->urlstr)
	{
		g_free(url_selection);
		url_selection = g_strdup(rec->urlstr);
	}
}

static gint SelectionClear(widget , event)
GtkWidget *widget;
GdkEventSelection *event;
{
	g_free(url_selection);
	url_selection = NULL;
	return TRUE;
}

static void SendSelection(widget , selection_data , info , time , data)
GtkWidget *widget;
GtkSelectionData *selection_data;
guint      info;
guint      time;
gpointer   data;
{
	GdkAtom type = GDK_NONE;

	switch (info)
	{
		case COMPOUND_TEXT:
			type = gdk_atom_intern("COMPOUND_TEXT",FALSE);
			break;
		case TEXT:
			type = gdk_atom_intern("TEXT",FALSE);
			break;
		case STRING:
			type = gdk_atom_intern("STRING",FALSE);
			break;
	}
	gtk_selection_data_set(selection_data , type , 8*sizeof(gchar) ,
		url_selection , strlen(url_selection)); 
}

static void UpdateEInfo()
{
	char pom[256];
	ns_cache_record *rec = NULL;


	if (GTK_CLIST(tlist)->selection)
	{
		rec = (ns_cache_record *) gtk_ctree_node_get_row_data(GTK_CTREE(tlist),
			 (GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data));
	}

	if (rec && iw)
	{
		char *p,*p1;

		p = g_strconcat(gettext("Source URL: "), rec->urlstr, "\n",
			gettext("Local cache file: "), rec->filename, "\n", NULL);

		if (rec->content_type)
		{	p1 = p;
			p = g_strconcat(p, gettext("MIME type: "), 
				rec->content_type, "\n", NULL);
			g_free(p1);
		}
			
		sprintf(pom, gettext("%d Bytes"), rec->content_length);
		p1 = p;
		p = g_strconcat(p, gettext("Size:"), pom, "\n", NULL);
		g_free(p1);

		if (rec->content_encoding)
		{
			p1 = p;
			p = g_strconcat(p, gettext("Encoding: "), 
				rec->content_encoding, "\n", NULL);
			g_free(p1);
		}

		if (rec->charset)
		{
			p1 = p;
			p = g_strconcat(p, gettext("Charset: "), 
				rec->charset, "\n", NULL);
			g_free(p1);
		}

		if (rec->last_modified)
		{
			struct tm *tm =  localtime(&rec->last_modified);
			strftime(pom , sizeof(pom) , "%H:%M %d.%m.%Y" , tm);
			p1 = p;
			p = g_strconcat(p, gettext("Modification time: "), 
				pom, "\n", NULL);
			g_free(p1);
		}

		if (rec->last_accessed)
		{
			char pom[100];
			struct tm *tm =  localtime(&rec->last_accessed);
			strftime(pom , sizeof(pom) , "%H:%M %d.%m.%Y" , tm);
			p1 = p;
			p = g_strconcat(p, gettext("Access time: "), 
				pom, "\n", NULL);
			g_free(p1);
		}

		if (rec->expires)
		{
			char pom[100];
			struct tm *tm =  localtime(&rec->expires);
			strftime(pom , sizeof(pom) , "%H:%M %d.%m.%Y" , tm);
			p1 = p;
			p = g_strconcat(p, gettext("Expiration time: "), 
				pom, "\n", NULL);
			g_free(p1);
		}

		gtk_label_set_text(GTK_LABEL(ilabel), p);

		g_free(p);
	}
	else if (iw)
	{
		gtk_label_set_text(GTK_LABEL(ilabel), get_node_info(pom));
	}
}

static void ViewInfo()
{
	if (!iw)
	{
		GtkWidget *box,*button;

		iw = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_window_set_title(GTK_WINDOW(iw) , gettext("NScache: entry info window"));
		gtk_signal_connect(GTK_OBJECT(iw), "destroy",
			GTK_SIGNAL_FUNC(gtk_widget_destroyed), &iw);

		box = gtk_vbox_new(FALSE , 5);
		gtk_container_add(GTK_CONTAINER(iw) , box);
		gtk_widget_show(box);

		ilabel = gtk_label_new("");
		gtk_misc_set_alignment(GTK_MISC(ilabel) , 0.0, 0.0);
		gtk_misc_set_padding(GTK_MISC(ilabel), 5, 5);
		gtk_label_set_justify(GTK_LABEL(ilabel), GTK_JUSTIFY_LEFT);
		gtk_widget_set_usize(ilabel, 400, 150);
		gtk_box_pack_start(GTK_BOX(box), ilabel, TRUE, TRUE, 5);
		gtk_widget_show(ilabel);

		button = guitl_pixmap_button(stock_close_xpm, gettext("Close"));
		gtk_box_pack_start(GTK_BOX(box), button, FALSE, FALSE, 2);
		gtk_widget_show(button);
	
		gtk_signal_connect(GTK_OBJECT(button) , "clicked" ,
			GTK_SIGNAL_FUNC(PopdownW) , iw);

		UpdateEInfo();
	}
	gtk_widget_show(iw);
	if (GTK_WIDGET_REALIZED(iw)) gdk_window_raise(iw->window);
}

GtkCTreeNode *get_parent_node(urlstr , dnamep)
char *urlstr;
char **dnamep;
{
	GtkCTreeNode *pn;
	GtkCTreeNode *sn;
	char *proto;
	char *serv;
	char *p,*p2;
	char *centry[9];

	p = strchr(urlstr , ':');

	if (!p) return NULL;

	proto = g_strndup(urlstr , p - urlstr);

	pn = (GtkCTreeNode *)g_hash_table_lookup(proth , proto);

	if (!pn)
	{
		centry[0] = proto;
		centry[1] = "";
		centry[2] = "";
		centry[3] = "";
		centry[4] = "";
		centry[5] = "";
		centry[6] = "";
		centry[7] = "";
		centry[8] = "";
		pn = gtk_ctree_insert_node(GTK_CTREE(tlist) , NULL , NULL , centry ,
			8 , NULL , NULL , NULL , NULL , FALSE , TRUE);

		g_hash_table_insert(proth , proto , pn);

		topnodes = g_slist_append(topnodes, pn);
	}

	p += 3;
	p2 = strchr(p , '/');

	if (!p2) return pn;

	serv = g_strndup(p , p2 - p);

	sn = (GtkCTreeNode *)g_hash_table_lookup(servh , serv);

	if (!sn)
	{
		centry[0] = serv;
		centry[1] = "";
		centry[2] = "";
		centry[3] = "";
		centry[4] = "";
		centry[5] = "";
		centry[6] = "";
		centry[7] = "";
		centry[8] = "";
		sn = gtk_ctree_insert_node(GTK_CTREE(tlist) , pn , NULL , centry ,
			8 , NULL , NULL , NULL , NULL , FALSE , FALSE);

		g_hash_table_insert(servh , serv , sn);

		sitenum++;
	}

	*dnamep = p2;
	return sn;
}

static gint list_compare(clist, r1, r2)
GtkCList *clist;
GtkCListRow *r1;
GtkCListRow *r2;
{
	gint rv = 0;
	ns_cache_record *rec1,*rec2;

	rec1 = r1->data;
	rec2 = r2->data;

	/* SIZE */
	if  (clist->sort_column == colinfos[3].colnr)
	{
			rv = rec1->content_length - rec2->content_length;
	}
	/* ATM */
	else if  (clist->sort_column == colinfos[7].colnr)
	{
		rv = rec1->last_accessed - rec2->last_accessed;
	}

	return rv;
}

static void list_sort(w , type)
GtkWidget *w;
gint type;
{
	gprop_set_int("sort_type", type);
	switch(type)
	{
		case STYPE_URL:
			gtk_clist_set_sort_column(GTK_CLIST(slist), colinfos[0].colnr);
			gtk_clist_set_compare_func(GTK_CLIST(slist), NULL);
		break;
		case STYPE_SIZE:
			gtk_clist_set_sort_column(GTK_CLIST(slist), colinfos[3].colnr);
			gtk_clist_set_compare_func(GTK_CLIST(slist), list_compare);
		break;
		case STYPE_ATM:
			gtk_clist_set_sort_column(GTK_CLIST(slist), colinfos[7].colnr);
			gtk_clist_set_compare_func(GTK_CLIST(slist), list_compare);
		break;
		case STYPE_MIME:
			gtk_clist_set_sort_column(GTK_CLIST(slist), colinfos[2].colnr);
			gtk_clist_set_compare_func(GTK_CLIST(slist), NULL);
		break;
	}
	gtk_clist_sort(GTK_CLIST(slist));
}

static void Open(w , data)
GtkWidget *w;
gpointer data;
{
	char *fname = gtk_entry_get_text(GTK_ENTRY(data));
	GSList *ptr;
	GtkCTreeNode *parent;
	char pom[2048];
	int sort_type;

	if (ns_cache_open_db(fname))
	{
		sprintf(pom , gettext("Error opening cache index file : %s"), strerror(errno));
		gui_err(pom);
		AdjustMenu();
		return;
	}

	if (!gprop_get_int("sort_type", &sort_type))
		sort_type = STYPE_URL;

	totsize = 0;
	recnum = 0;
	sitenum = 0;

	if (db_name) g_free(db_name);
	db_name = g_strdup(fname);

	gtk_clist_freeze(GTK_CLIST(tlist));
	gtk_clist_clear(GTK_CLIST(tlist));

	gtk_clist_freeze(GTK_CLIST(slist));
	gtk_clist_clear(GTK_CLIST(slist));

	while(elist)
	{
		ns_cache_free_record((ns_cache_record *)elist->data);
		elist = g_slist_remove_link(elist , elist);
	}

	if (proth)  g_hash_table_destroy(proth);
	if (servh)  g_hash_table_destroy(servh);
	while (topnodes) topnodes = g_slist_remove_link(topnodes, topnodes);

	proth = g_hash_table_new(g_str_hash , g_str_equal);
	servh = g_hash_table_new(g_str_hash , g_str_equal);

	ptr = elist = ns_cache_read_db();

	ns_cache_close_db();
	parent = NULL;
	while(ptr)
	{
		char *centry[10];
		char *dname;
		GtkCTreeNode *n;
		char size[15];
		char mdtm[30];
		char atm[30];
		char exptm[30];
		struct tm *tm;
		int row;
		struct nscache_icon_t *icon_info;
		ns_cache_record *rec = (ns_cache_record *)ptr->data;

		icon_info = nscache_icon_get(rec->content_type);

		recnum ++;
		totsize += rec->content_length;

		parent = get_parent_node(rec->urlstr, &dname);

		centry[colinfos[0].colnr] = dname;
		centry[colinfos[1].colnr] = rec->filename;
		centry[colinfos[2].colnr] = rec->content_type;
		sprintf(size , "%d" , rec->content_length);
		centry[colinfos[3].colnr] = size;
		centry[colinfos[4].colnr] = rec->content_encoding;
		centry[colinfos[5].colnr] = rec->charset;
		tm =  localtime(&rec->last_modified); 
		strftime(mdtm , sizeof(mdtm) , "%H:%M %d.%m.%Y" , tm);
		centry[colinfos[6].colnr] = mdtm;
		tm = localtime(&rec->last_accessed); 
		strftime(atm , sizeof(atm) , "%H:%M %d.%m.%Y" , tm);
		centry[colinfos[7].colnr] = atm;
		if (rec->expires)
		{
			tm = localtime(&rec->expires); 
			strftime(exptm , sizeof(exptm) , "%H:%M %d.%m.%Y" , tm);
		}
		else
			exptm[0] = '\0';
		centry[colinfos[8].colnr] = exptm;

		n = gtk_ctree_insert_node(GTK_CTREE(tlist), parent, NULL,
			centry, 8 , icon_info->pixmap, icon_info->mask,
			icon_info->pixmap, icon_info->mask, FALSE, FALSE);
		gtk_ctree_node_set_row_data(GTK_CTREE(tlist) , n , ptr->data);
		rec->tree_node = n;

		centry[0] = rec->urlstr;
		row = gtk_clist_append(GTK_CLIST(slist), centry);
		gtk_clist_set_pixtext(GTK_CLIST(slist), row, 0, centry[0],
			3, icon_info->pixmap, icon_info->mask);
		gtk_clist_set_row_data(GTK_CLIST(slist), row, ptr->data);
		rec->list_row = row;

		ptr = ptr->next;
	}

	gtk_ctree_pre_recursive_to_depth(GTK_CTREE(tlist) , NULL , 1 ,
		GTK_CTREE_FUNC(gtk_ctree_sort_recursive) , NULL);

	list_sort(NULL, sort_type);

	gtk_clist_thaw(GTK_CLIST(slist));
	gtk_clist_thaw(GTK_CLIST(tlist));

	gui_msg(get_node_info(pom));

	if(iw)
		UpdateEInfo();   /* records and size may have changed */

	AdjustMenu();           /* enable or disable the menu entries */
}

static void ToggleBool(w , fdata)
GtkWidget *w;
gpointer fdata;
{
	int *v = (int *)fdata;

	*v = GTK_CHECK_MENU_ITEM(w)->active;
}

static void SaveRC()
{
	save_rc();
}

void build_menu(w)
GtkWidget *w;
{
	GtkWidget *mbar,*menu,*mbb,*mi,*smenu;
	int i,rv;

	mbar = gtk_menu_bar_new();
	gtk_box_pack_start(GTK_BOX(w) , mbar , FALSE , TRUE , 1);
	gtk_box_reorder_child(GTK_BOX(w) , mbar , 0);
	gtk_widget_show(mbar);

	menu = gtk_menu_new();
	gtk_widget_realize(menu);

	mi = gtk_menu_item_new_with_label(gettext("Open cache index file ..."));
	gaccel_bind_widget("file/open" , "activate" , mi, NULL, tl);
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	gtk_signal_connect(GTK_OBJECT(mi) , "activate" ,
		GTK_SIGNAL_FUNC(Browse) , NULL);

	mi = gtk_menu_item_new_with_label(gettext("Save rc file"));
	gaccel_bind_widget("file/saverc" , "activate" , mi, NULL, tl);
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	gtk_signal_connect(GTK_OBJECT(mi) , "activate" ,
		GTK_SIGNAL_FUNC(SaveRC) , NULL);

	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	mi = gtk_menu_item_new_with_label(gettext("Quit"));
	gaccel_bind_widget("file/quit" , "activate" , mi, NULL, tl);
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	gtk_signal_connect(GTK_OBJECT(mi) , "activate" ,
		GTK_SIGNAL_FUNC(Quit) , NULL);

	mbb = gtk_menu_item_new_with_label(gettext("File"));
	gtk_menu_bar_append(GTK_MENU_BAR(mbar) , mbb);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(mbb) , menu);
	gtk_widget_show(mbb);

	menu = gtk_menu_new();
	gtk_widget_realize(menu);

	mi = gtk_menu_item_new_with_label(gettext("Entry informations ..."));
	gaccel_bind_widget("view/entry_infowin" , "activate" , mi, NULL, tl);
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	gtk_signal_connect(GTK_OBJECT(mi) , "activate" ,
		GTK_SIGNAL_FUNC(ViewInfo) , NULL);

	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

#define SBSET_ME(cn)\
	mi = gtk_check_menu_item_new_with_label(gettext(colinfos[cn].label));\
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mi), TRUE);\
	gtk_menu_append(GTK_MENU(smenu) , mi);\
	gtk_widget_show (mi);\
	gtk_signal_connect(GTK_OBJECT(mi), "activate",\
			GTK_SIGNAL_FUNC(clist_col_set_visibility),\
			(gpointer)cn);\
	gaccel_bind_widget(colinfos[cn].accel , "activate" , mi, NULL, tl);\
	if (gprop_get_bool(colinfos[cn].prop, &rv))\
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mi), rv);\
	clist_col_set_visibility(mi, cn);

	mi = gtk_menu_item_new_with_label(gettext("Columns"));
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	smenu = gtk_menu_new();
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi), smenu);
	gtk_widget_show(menu);

	for (i = 0; i < (sizeof(colinfos)/sizeof(colinfos[0])); i++)
	{
		SBSET_ME(i);
	}

	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	smenu = gtk_menu_new();
	gtk_signal_connect(GTK_OBJECT(smenu), "show",
		GTK_SIGNAL_FUNC(GenViewerMenu), (gpointer)TRUE);
	gtk_widget_realize(smenu);

	fmi2 = gtk_menu_item_new_with_label(gettext("View file with"));
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(fmi2), smenu);
	gaccel_bind_widget("view/file" , "activate" , fmi2, NULL, tl);
	gtk_menu_append(GTK_MENU(menu) , fmi2);
	gtk_widget_show(fmi2);

	smenu = gtk_menu_new();
	gtk_signal_connect(GTK_OBJECT(smenu), "show",
		GTK_SIGNAL_FUNC(GenViewerMenu), (gpointer)FALSE);
	gtk_widget_realize(smenu);

	umi2 = gtk_menu_item_new_with_label(gettext("View URL with"));
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(umi2), smenu);
	gaccel_bind_widget("view/url", "activate", umi2, NULL, tl);
	gtk_menu_append(GTK_MENU(menu), umi2);
	gtk_widget_show(umi2);

	mi = gtk_menu_item_new_with_label(gettext("Find ..."));
	gaccel_bind_widget("view/find", "activate", mi, NULL, tl);
	gtk_menu_append(GTK_MENU(menu), mi);
	gtk_widget_show(mi);

	gtk_signal_connect(GTK_OBJECT(mi), "activate",
		GTK_SIGNAL_FUNC(search_dlg_open), NULL);

	mbb = gtk_menu_item_new_with_label(gettext("View"));
	gtk_menu_bar_append(GTK_MENU_BAR(mbar) , mbb);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(mbb) , menu);
	gtk_widget_show(mbb);


	menu = gtk_menu_new();
	gtk_widget_realize(menu);

	smi2 = gtk_menu_item_new_with_label(gettext("Save to file ..."));
	gaccel_bind_widget("edit/save" , "activate" , smi2, NULL, tl);
	gtk_menu_append(GTK_MENU(menu) , smi2);
	gtk_widget_show(smi2);

	gtk_signal_connect(GTK_OBJECT(smi2) , "activate" ,
		GTK_SIGNAL_FUNC(SaveAs) , NULL);

	rmi2 = gtk_menu_item_new_with_label(gettext("Delete file"));
	gaccel_bind_widget("edit/delete" , "activate" , rmi2, NULL, tl);
	gtk_menu_append(GTK_MENU(menu) , rmi2);
	gtk_widget_show(rmi2);

	gtk_signal_connect(GTK_OBJECT(rmi2) , "activate" ,
		GTK_SIGNAL_FUNC(Remove) , NULL);

	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	cmi2 = gtk_menu_item_new_with_label(gettext("Copy url to clipboard"));
	gaccel_bind_widget("edit/clipboard" , "activate" , cmi2, NULL, tl);
	gtk_menu_append(GTK_MENU(menu) , cmi2);
	gtk_widget_show(cmi2);

	gtk_signal_connect(GTK_OBJECT(cmi2) , "activate" ,
		GTK_SIGNAL_FUNC(SetSelectionURL) , NULL);

	nmi2 = gtk_menu_item_new_with_label(gettext("Show entry informations"));
	gaccel_bind_widget("edit/showinfo" , "activate" , nmi2, NULL, tl);
	gtk_menu_append(GTK_MENU(menu) , nmi2);
	gtk_widget_show(nmi2);

	gtk_signal_connect(GTK_OBJECT(nmi2) , "activate" ,
		GTK_SIGNAL_FUNC(NodeInfo) , NULL);

	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	omi2 = gtk_menu_item_new_with_label(gettext("read-only permissions"));
	gaccel_bind_widget("edit/readonly" , "activate" , omi2, NULL, tl);
	gtk_menu_append(GTK_MENU(menu) , omi2);
	gtk_widget_show(omi2);

	gtk_signal_connect(GTK_OBJECT(omi2) , "activate" ,
		GTK_SIGNAL_FUNC(RdOnly) , NULL);

	wmi2 = gtk_menu_item_new_with_label(gettext("read/write permissions"));
	gaccel_bind_widget("edit/readwrite" , "activate" , wmi2, NULL, tl);
	gtk_menu_append(GTK_MENU(menu) , wmi2);
	gtk_widget_show(wmi2);

	gtk_signal_connect(GTK_OBJECT(wmi2) , "activate" ,
		GTK_SIGNAL_FUNC(Writable) , NULL);

	mbb = gtk_menu_item_new_with_label(gettext("Edit"));
	gtk_menu_bar_append(GTK_MENU_BAR(mbar) , mbb);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(mbb) , menu);
	gtk_widget_show(mbb);

	menu = gtk_menu_new();
	gtk_widget_realize(menu);

	mi = gtk_menu_item_new_with_label(gettext("Viewers setup ..."));
	gaccel_bind_widget("opt/viewers" , "activate" , mi, NULL, tl);
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	gtk_signal_connect(GTK_OBJECT(mi) , "activate" ,
		GTK_SIGNAL_FUNC(apassign_run) , NULL);

	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	mi = gtk_check_menu_item_new_with_label(gettext("Convert filenames to lowercase"));
	gaccel_bind_widget("opt/case" , "activate" , mi, NULL, tl);
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	gtk_signal_connect(GTK_OBJECT(mi) , "activate" ,
		GTK_SIGNAL_FUNC(ToggleBool) , &cvt_to_lower);

	mbb = gtk_menu_item_new_with_label(gettext("Options"));
	gtk_menu_item_right_justify(GTK_MENU_ITEM(mbb));
	gtk_menu_bar_append(GTK_MENU_BAR(mbar) , mbb);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(mbb) , menu);
	gtk_widget_show(mbb);

	menu = gtk_menu_new();
	gtk_widget_realize(menu);

	mi = gtk_menu_item_new_with_label(gettext("About ..."));
	gaccel_bind_widget("help/about" , "activate" , mi, NULL, tl);
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	gtk_signal_connect(GTK_OBJECT(mi) , "activate" ,
		GTK_SIGNAL_FUNC(Help) , NULL);

	mbb = gtk_menu_item_new_with_label(gettext("Help"));
	gtk_menu_item_right_justify(GTK_MENU_ITEM(mbb));
	gtk_menu_bar_append(GTK_MENU_BAR(mbar) , mbb);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(mbb) , menu);
	gtk_widget_show(mbb);
}

void build_seldb(w)
GtkWidget *w;
{
	GtkWidget *box,*button,*label;

	box = gtk_hbox_new(FALSE , 5);
	gtk_box_pack_start(GTK_BOX(w) , box , FALSE , TRUE , 1);
	gtk_widget_show(box);

	label = gtk_label_new(gettext("Cache index file: "));
	gtk_box_pack_start(GTK_BOX(box) , label , FALSE , FALSE , 5);
	gtk_widget_show(label);

	entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(box) , entry , TRUE , TRUE , 1);
	gtk_widget_show(entry);

	gtk_signal_connect(GTK_OBJECT(entry), "activate" ,
		GTK_SIGNAL_FUNC(Open) , entry);

	button = guitl_pixmap_button(stock_open_xpm, gettext("Browse ..."));
	gtk_box_pack_start(GTK_BOX(box) , button , FALSE , FALSE , 1);
	gtk_widget_show(button);

	gtk_signal_connect(GTK_OBJECT(button), "clicked" ,
		GTK_SIGNAL_FUNC(Browse) , NULL);
}

void AdjustMenu()
{
	if (GTK_CLIST(tlist)->selection)
	{
		gboolean have_fv = FALSE, have_uv = FALSE;
		ns_cache_record *rec = (ns_cache_record *)
			gtk_ctree_node_get_row_data(GTK_CTREE(tlist), (GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data));

		if (rec && rec->content_type)
		{
			GSList *ptr = apassign_get_matching_viewers(rec->content_type);
			while (ptr)
			{
				apassign *e = (apassign *) ptr->data;

				have_uv |= (e->uapplication != NULL &&
					    *e->uapplication);
				have_fv |= (e->fapplication != NULL &&
					    *e->fapplication);

				ptr = g_slist_remove_link(ptr, ptr);
			}
		}

		gtk_widget_set_sensitive(smi2 , (int)rec);
		gtk_widget_set_sensitive(cmi2 , (int)rec);
		gtk_widget_set_sensitive(rmi2 , (int)rec ||
			((GtkCTreeRow *)((GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data))->list.data)->level == 2);
		gtk_widget_set_sensitive(fmi2 , have_fv);
		gtk_widget_set_sensitive(umi2 , have_uv);

		gtk_widget_set_sensitive(smi , (int)rec);
		gtk_widget_set_sensitive(cmi , (int)rec);
		gtk_widget_set_sensitive(rmi , (int)rec ||
			((GtkCTreeRow *)((GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data))->list.data)->level == 2);
		gtk_widget_set_sensitive(fmi , have_fv);
		gtk_widget_set_sensitive(umi , have_uv);


		if (rec)
		{
			char pom[2048];
			char *p;
			struct stat estat;

			strcpy(pom , db_name);
			p = strrchr(pom , '/');
			if (!p) p = pom;
			else p++;
			strcpy(p , rec->filename);

			if (!stat(pom, &estat))
			{
				gtk_widget_set_sensitive(omi, estat.st_mode & S_IWUSR);
				gtk_widget_set_sensitive(wmi, !(estat.st_mode & S_IWUSR));
				gtk_widget_set_sensitive(omi2, estat.st_mode & S_IWUSR);
				gtk_widget_set_sensitive(wmi2, !(estat.st_mode & S_IWUSR));
			}
			else
			{
				gtk_widget_set_sensitive(omi , !access(pom, F_OK));
				gtk_widget_set_sensitive(wmi , !access(pom, F_OK));
				gtk_widget_set_sensitive(omi2 , !access(pom, F_OK));
				gtk_widget_set_sensitive(wmi2 , !access(pom, F_OK));
			}
		}
		else
		{
			gtk_widget_set_sensitive(omi , 
				((GtkCTreeRow *)((GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data))->list.data)->level == 2);
			gtk_widget_set_sensitive(wmi ,
				((GtkCTreeRow *)((GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data))->list.data)->level == 2);
			gtk_widget_set_sensitive(omi2 , 
				((GtkCTreeRow *)((GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data))->list.data)->level == 2);
			gtk_widget_set_sensitive(wmi2 ,
				((GtkCTreeRow *)((GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data))->list.data)->level == 2);
		}
		gtk_widget_set_sensitive(nmi2 , TRUE);
		gtk_widget_set_sensitive(nmi , TRUE);
	}
	else
	{
		gtk_widget_set_sensitive(omi2 , FALSE);
		gtk_widget_set_sensitive(nmi2 , FALSE);
		gtk_widget_set_sensitive(wmi2 , FALSE);
		gtk_widget_set_sensitive(smi2 , FALSE);
		gtk_widget_set_sensitive(cmi2 , FALSE);
		gtk_widget_set_sensitive(rmi2 , FALSE);
		gtk_widget_set_sensitive(fmi2 , FALSE);
		gtk_widget_set_sensitive(umi2 , FALSE);
		gtk_widget_set_sensitive(omi , FALSE);
		gtk_widget_set_sensitive(nmi , FALSE);
		gtk_widget_set_sensitive(wmi , FALSE);
		gtk_widget_set_sensitive(smi , FALSE);
		gtk_widget_set_sensitive(cmi , FALSE);
		gtk_widget_set_sensitive(rmi , FALSE);
		gtk_widget_set_sensitive(fmi , FALSE);
		gtk_widget_set_sensitive(umi , FALSE);
	}
}

gint popup_tmenu(widget , event , data)
GtkWidget *widget;
GdkEvent  *event;
gpointer data;
{
	GdkEventButton *bevent;

	switch (event->type)
	{
	    case GDK_2BUTTON_PRESS:
		bevent = (GdkEventButton *) event;
		if (bevent->button == 1)
		{
			ViewFileDefault();
		}
		else if (bevent->button == 2)
		{
			ViewURLDefault();
		}
	    case GDK_BUTTON_PRESS:
		bevent = (GdkEventButton *) event;
		if (bevent->button == 3)
		{
			gtk_menu_popup(GTK_MENU(data), NULL, NULL,
					 NULL, NULL, 3, bevent->time);
		}
	    break;
	    default: break;
	}

	return FALSE;
}

static void list_select(object , row , col , event , func_data)
GtkObject *object;
int row;
int col;
GdkEvent *event;
gpointer func_data;
{
	ns_cache_record *nsr;

	nsr = gtk_clist_get_row_data (GTK_CLIST(slist), row);

	if (nsr)
	{
#if 0
		GtkCTreeNode *tn = nsr->tree_node;

		while(tn)
		{
			GtkCTreeRow *tr = (GtkCTreeRow *) tn->list.data;

			gtk_ctree_expand(GTK_CTREE(tlist), tn);

			tn = tr->parent;
		}
		gtk_ctree_node_moveto(GTK_CTREE(tlist), nsr->tree_node, 0 , 0.0, 0.0);
#endif
		gtk_ctree_select(GTK_CTREE(tlist), nsr->tree_node);
	}
}


static void list_resize_column(w, col, width, fdata)
GtkWidget *w;
int col;
int width;
gpointer fdata;
{
	int rcol;

	if (w == slist)
	{
		rcol = (int) gtk_object_get_data(GTK_OBJECT(
			gtk_clist_get_column_widget(GTK_CLIST(slist), col)),
			"real_column_nr");
		gprop_set_int(colinfos[rcol].lwprop, width);
	}
	else if (w == tlist)
	{
		rcol = (int) gtk_object_get_data(GTK_OBJECT(
			gtk_clist_get_column_widget(GTK_CLIST(tlist), col)),
			"real_column_nr");
		gprop_set_int(colinfos[rcol].twprop, width);
	}
}

static int clist_get_col_index(col)
int col;
{
	int i;

	for (i = 0; i <= 8; i++)
	{
		if (col == colinfos[i].colnr)
			return i;
	}
	return -1;
}

static int clist_col_is_left_most(col)
int col;
{
	gboolean is_visible;
	int rcol;

	do
	{
		col --;

		rcol = clist_get_col_index(col);
		if (rcol <= 0)
			return TRUE;

		if (!gprop_get_bool(colinfos[rcol].prop, &is_visible))
			is_visible = TRUE;
	} while (!is_visible && col > 0);

	return !col;
}

static int clist_col_is_right_most(col)
int col;
{
	gboolean is_visible;
	int rcol;

	if (col >= 8)
		return TRUE;

	do
	{
		col ++;

		rcol = clist_get_col_index(col);
		if (rcol <= 0)
			return TRUE;

		if (!gprop_get_bool(colinfos[rcol].prop, &is_visible))
			is_visible = TRUE;
	} while (!is_visible && col < 8);

	return !is_visible;
}

static void clist_col_arrows_set_sensitivity(clist, col)
GtkWidget *clist;
int col;
{
	GtkWidget *a1, *a2, *tab;

	tab = gtk_clist_get_column_widget(GTK_CLIST(clist), col);
	a1 = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(tab), "left_arrow");
	a2 = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(tab), "right_arrow");

	gtk_widget_set_sensitive(a1, TRUE);
	gtk_widget_show(a1);
	gtk_widget_set_sensitive(a2, TRUE);
	gtk_widget_show(a2);

	if (clist_col_is_left_most(col))
	{
		gtk_widget_set_sensitive(a1, FALSE);
		gtk_widget_hide(a1);
	}

	if (clist_col_is_right_most(col))
	{
		gtk_widget_set_sensitive(a2, FALSE);
		gtk_widget_hide(a2);
	}
}

static int clist_get_right_visible_col(col)
int col;
{
	gboolean is_visible;

	if (col >= 8)
		return FALSE;

	do
	{
		col ++;

		if (!gprop_get_bool(colinfos[clist_get_col_index(col)].prop, &is_visible))
			is_visible = TRUE;

		if (is_visible)
			return col;
	} while (col < 8);

	return -1;
}

static int clist_get_left_visible_col(col)
int col;
{
	gboolean is_visible;

	do
	{
		col --;

		if (!gprop_get_bool(colinfos[clist_get_col_index(col)].prop, &is_visible))
			is_visible = TRUE;

		if (is_visible)
			return col;
	} while (col > 0);

	return -1;
}


static void clist_col_set_visibility(object, rcol)
GtkObject *object;
int rcol;
{
	int col;

	gtk_clist_set_column_visibility(GTK_CLIST(slist),
		colinfos[rcol].colnr, GTK_CHECK_MENU_ITEM(object)->active);
	gtk_clist_set_column_visibility(GTK_CLIST(tlist),
		 colinfos[rcol].colnr, GTK_CHECK_MENU_ITEM(object)->active);
	gprop_set_bool(colinfos[rcol].prop,
		GTK_CHECK_MENU_ITEM(object)->active);

	if (!rcol)
		return;

	clist_col_arrows_set_sensitivity(slist, colinfos[rcol].colnr);
	clist_col_arrows_set_sensitivity(tlist, colinfos[rcol].colnr);

	col = clist_get_right_visible_col(colinfos[rcol].colnr);

	if (col > 0)
	{
		clist_col_arrows_set_sensitivity(slist, col);
		clist_col_arrows_set_sensitivity(tlist, col);
	}

	col = clist_get_left_visible_col(colinfos[rcol].colnr);

	if (col > 0)
	{
		clist_col_arrows_set_sensitivity(slist, col);
		clist_col_arrows_set_sensitivity(tlist, col);
	}
}

static void clist_move_columns_real(clist, col1, col2, rcol1, rcol2)
GtkWidget *clist;
int col1;
int col2;
int rcol1;
int rcol2;
{
	GtkWidget *tab,*tab2, *a1, *a2;
	int w1,w2;

	tab = gtk_clist_get_column_widget(GTK_CLIST(clist), col1);
	tab2 = gtk_clist_get_column_widget(GTK_CLIST(clist), col2);

	if (clist == slist)
	{
		if (!gprop_get_int(colinfos[rcol1].lwprop, &w1))
			w1 = colinfos[rcol1].dwidth;
		if (!gprop_get_int(colinfos[rcol2].lwprop, &w2))
			w2 = colinfos[rcol2].dwidth;

		guitl_clist_swap_cols(clist, col1, col2);
	}
	else
	{
		if (!gprop_get_int(colinfos[rcol1].twprop, &w1))
			w1 = colinfos[rcol1].dwidth;
		if (!gprop_get_int(colinfos[rcol2].twprop, &w2))
			w2 = colinfos[rcol2].dwidth;

		guitl_tlist_swap_cols(clist, topnodes, col1, col2);
	}

	gtk_object_set_data(GTK_OBJECT(tab), "column_nr", (gpointer) col2);
	a1 = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(tab), "left_arrow");
	a2 = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(tab), "right_arrow");
	clist_col_arrows_set_sensitivity(clist, col2);
	gtk_clist_set_column_width(GTK_CLIST(clist), col2, w1);
	gtk_clist_set_column_justification(GTK_CLIST(clist), col2, colinfos[rcol1].justify);

	gtk_object_set_data(GTK_OBJECT(tab2), "column_nr", (gpointer) col1);
	a1 = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(tab2), "left_arrow");
	a2 = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(tab2), "right_arrow");
	clist_col_arrows_set_sensitivity(clist, col1);
	gtk_clist_set_column_width(GTK_CLIST(clist), col1, w2);
	gtk_clist_set_column_justification(GTK_CLIST(clist), col1, colinfos[rcol2].justify);
}

static void clist_move_columns(col1, dif)
int col1;
int dif;
{
	GtkWidget *tab1,*tab2;
	int col2,rcol1,rcol2;

	if (dif > 0)
		col2 = clist_get_right_visible_col(col1);
	else
		col2 = clist_get_left_visible_col(col1);

	if (col2 < 0)
		return;

	tab1 = gtk_clist_get_column_widget(GTK_CLIST(slist), col1);
	tab2 = gtk_clist_get_column_widget(GTK_CLIST(slist), col2);

	rcol1 = (int) gtk_object_get_data(GTK_OBJECT(tab1), "real_column_nr");
	rcol2 = (int) gtk_object_get_data(GTK_OBJECT(tab2), "real_column_nr");

	clist_move_columns_real(slist, col1, col2, rcol1, rcol2);
	clist_move_columns_real(tlist, col1, col2, rcol1, rcol2);

	colinfos[rcol1].colnr = col2;
	colinfos[rcol2].colnr = col1;
	gprop_set_int(colinfos[rcol2].colnrprop, col1);
	gprop_set_int(colinfos[rcol1].colnrprop, col2);
}

static void clist_move_colum_right(w, clist)
GtkWidget *w;
GtkWidget *clist;
{
	GtkWidget *tab;
	int col;

	tab = GTK_WIDGET(w)->parent;
	col = (int) gtk_object_get_data(GTK_OBJECT(tab), "column_nr");

	clist_move_columns(col, +1);
}

static void clist_move_colum_left(w, clist)
GtkWidget *w;
GtkWidget *clist;
{
	GtkWidget *tab;
	int col;

	tab = GTK_WIDGET(w)->parent;
	col = (int) gtk_object_get_data(GTK_OBJECT(tab), "column_nr");

	clist_move_columns(col, -1);
}

static int clist_col_btn_unhighlight(btn)
GtkWidget *btn;
{
	gtk_button_leave(GTK_BUTTON(btn));
	return FALSE;
}

static void clist_col_reparent(w, oldp, btn)
GtkWidget *w;
GtkWidget *oldp;
GtkWidget *btn;
{
	if (GTK_WIDGET_IS_SENSITIVE(btn) && GTK_WIDGET_STATE(btn))
	{
		gtk_timeout_add(50,
			(GtkFunction)clist_col_btn_unhighlight, btn);
	}
}

static void make_clist_col_widget(clist, col, rcol, label)
GtkWidget *clist;
int col;
int rcol;
char *label;
{
	GtkWidget *tab, *b1, *b2, *a1, *a2, *lbl;

	tab = gtk_table_new(3, 1, FALSE);
	gtk_widget_show(tab);

	b1 = gtk_button_new();
	gtk_button_set_relief(GTK_BUTTON(b1), GTK_RELIEF_NONE);
	gtk_table_attach(GTK_TABLE(tab), b1, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(b1);

	gtk_signal_connect(GTK_OBJECT(b1), "clicked",
		GTK_SIGNAL_FUNC(clist_move_colum_left), clist);
	 gtk_signal_connect(GTK_OBJECT(tab), "parent_set",
		  GTK_SIGNAL_FUNC(clist_col_reparent), b1);

	a1 = gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_OUT);
	gtk_container_add(GTK_CONTAINER(b1), a1);
	gtk_widget_show(a1);


	lbl = gtk_label_new(label);
	gtk_table_attach(GTK_TABLE(tab), lbl, 1, 2, 0, 1, GTK_EXPAND, GTK_EXPAND, 0, 0);
	gtk_widget_show(lbl);

	b2 = gtk_button_new();
	gtk_button_set_relief(GTK_BUTTON(b2), GTK_RELIEF_NONE);
	gtk_table_attach(GTK_TABLE(tab), b2, 2, 3, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(b2);

	gtk_signal_connect(GTK_OBJECT(b2), "clicked",
		GTK_SIGNAL_FUNC(clist_move_colum_right), clist);
	 gtk_signal_connect(GTK_OBJECT(tab), "parent_set",
		  GTK_SIGNAL_FUNC(clist_col_reparent), b2);

	a2 = gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT);
	gtk_container_add(GTK_CONTAINER(b2), a2);
	gtk_widget_show(a2);

	gtk_clist_set_column_widget(GTK_CLIST(clist), col, tab);

	gtk_object_set_data(GTK_OBJECT(tab), "column_nr", (gpointer) col);
	gtk_object_set_data(GTK_OBJECT(tab), "real_column_nr", (gpointer) rcol);
	gtk_object_set_data(GTK_OBJECT(tab), "left_arrow", (gpointer) b1);
	gtk_object_set_data(GTK_OBJECT(tab), "right_arrow", (gpointer) b2);
	clist_col_arrows_set_sensitivity(clist, col);
}

void build_slist(notebook)
GtkWidget *notebook;
{
	GtkWidget *swin, *box, *label, *mi, *menu, *pbox;
	gint i,sort_type;
	struct {char *label; gint type;} stp[] = {
		{gettext_nop("URL") , STYPE_URL} ,
		{gettext_nop("Size") , STYPE_SIZE} ,
		{gettext_nop("Access time") , STYPE_ATM} ,
		{gettext_nop("Mime type") , STYPE_MIME} ,
	};

	if (!gprop_get_int("sort_type", &sort_type))
		sort_type = STYPE_URL;

	box = gtk_vbox_new(FALSE,5);
	gtk_widget_show(box);
	label = gtk_label_new(gettext("Sorted view"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), box , label);

	pbox = gtk_hbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(box) , pbox , FALSE , FALSE , 1);
	gtk_widget_show(pbox);

	label = gtk_label_new(gettext("Sort by: "));
	gtk_box_pack_start(GTK_BOX(pbox) , label, FALSE, FALSE, 1);
	gtk_widget_show(label);
	
	sort_by = gtk_option_menu_new();
	gtk_box_pack_start(GTK_BOX(pbox) , sort_by, FALSE, FALSE , 1);

	menu = gtk_menu_new();

	for (i = 0 ; i < sizeof(stp)/sizeof(*stp) ; i++)
	{
		mi = gtk_menu_item_new_with_label(gettext(stp[i].label));
		gtk_menu_append(GTK_MENU(menu), mi);
		gtk_widget_show(mi);

		gtk_signal_connect(GTK_OBJECT(mi), "activate" ,
			GTK_SIGNAL_FUNC(list_sort), (gpointer)stp[i].type); 
	}

	gtk_option_menu_set_menu(GTK_OPTION_MENU(sort_by), menu);
	gtk_widget_show(sort_by);
	gtk_option_menu_set_history(GTK_OPTION_MENU(sort_by), sort_type);

	swin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin),
		GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
	gtk_widget_set_usize(swin , 600 , 300);
	gtk_box_pack_start(GTK_BOX(box) , swin , TRUE , TRUE , 1);
	gtk_widget_show (swin);

	slist = gtk_clist_new(9);
	gtk_clist_set_row_height(GTK_CLIST(slist), MAX(
		slist->style->font->ascent + slist->style->font->descent + 1,
		14));
	for (i = 0; i < (sizeof(colinfos)/sizeof(colinfos[0])); i++)
	{
		int w;

		if (!i)
		{
			gtk_clist_set_column_title(GTK_CLIST(slist), i,
				gettext(colinfos[i].label));
		}
		else
		{
			gprop_get_int(colinfos[i].colnrprop, &colinfos[i].colnr);
			make_clist_col_widget(slist, colinfos[i].colnr, i,
				gettext(colinfos[i].label));
		}

		gtk_clist_set_column_justification(GTK_CLIST(slist),
			colinfos[i].colnr, colinfos[i].justify);

		gtk_clist_set_column_width(GTK_CLIST(slist), colinfos[i].colnr,
			gprop_get_int(colinfos[i].lwprop, &w) ? w : colinfos[i].dwidth);
	}

	gtk_signal_connect(GTK_OBJECT(slist), "resize_column", 
		GTK_SIGNAL_FUNC(list_resize_column), NULL);

	gtk_clist_column_titles_show(GTK_CLIST(slist));
	gtk_clist_set_selection_mode(GTK_CLIST(slist), GTK_SELECTION_BROWSE);
	gtk_container_add(GTK_CONTAINER(swin) , slist);
	gtk_clist_column_titles_passive(GTK_CLIST(slist));
	gtk_widget_show(slist);

	gtk_signal_connect(GTK_OBJECT(slist) , "button_press_event" , 
		GTK_SIGNAL_FUNC(popup_tmenu) , flow_menu);

	gtk_signal_connect(GTK_OBJECT(slist) , "select_row" ,
		GTK_SIGNAL_FUNC(list_select) , NULL);
}

static void tlist_expand_colapse(w, colp)
GtkWidget *w;
int colp;
{
	GSList *ptr;

	for (ptr = topnodes; ptr; ptr = ptr->next)
	{
		if (colp)
			gtk_ctree_collapse_to_depth(GTK_CTREE(tlist),
				(GtkCTreeNode *) ptr->data, 2);
		else
			gtk_ctree_expand_to_depth(GTK_CTREE(tlist),
				(GtkCTreeNode *) ptr->data, 2);
	}
}

static void make_tlist_first_col(tlist, label)
GtkWidget *tlist;
char *label;
{
	GtkWidget *tab, *btn,*lbl;

	tab = gtk_table_new(3, 1, FALSE);

	btn = guitl_pixmap_button(stock_expand_xpm, NULL);
	gtk_button_set_relief(GTK_BUTTON(btn), GTK_RELIEF_NONE);
	gtk_table_attach(GTK_TABLE(tab), btn, 0, 1, 0, 1,
			GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(btn);

	gtk_signal_connect(GTK_OBJECT(btn), "clicked",
		GTK_SIGNAL_FUNC(tlist_expand_colapse), (gpointer)FALSE);

	btn = guitl_pixmap_button(stock_colapse_xpm, NULL);
	gtk_button_set_relief(GTK_BUTTON(btn), GTK_RELIEF_NONE);
	gtk_table_attach(GTK_TABLE(tab), btn, 1, 2, 0, 1,
			GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(btn);

	gtk_signal_connect(GTK_OBJECT(btn), "clicked",
		GTK_SIGNAL_FUNC(tlist_expand_colapse), (gpointer)TRUE);
 
	lbl = gtk_label_new(label);
	gtk_table_attach(GTK_TABLE(tab), lbl, 2, 3, 0, 1,
			GTK_FILL, GTK_FILL, 1, 1);
	gtk_widget_show(lbl);

	gtk_clist_set_column_widget(GTK_CLIST(tlist), 0, tab);
}

void build_tlist(notebook)
GtkWidget *notebook;
{
	GtkWidget *swin, *mi, *box, *label, *smenu;
	int i;

	box = gtk_hbox_new(FALSE,5);
	gtk_widget_show(box);
	label = gtk_label_new(gettext("Tree view"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), box , label);

	swin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (swin),
		GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
	gtk_widget_set_usize(swin , 600 , 300);
	gtk_box_pack_start(GTK_BOX(box) , swin , TRUE , TRUE , 1);
	gtk_widget_show (swin);

	tlist = gtk_ctree_new(9 , 0);
	gtk_clist_set_row_height(GTK_CLIST(tlist), MAX(
		tlist->style->font->ascent + tlist->style->font->descent + 1,
		14));
	for (i = 0; i < (sizeof(colinfos)/sizeof(colinfos[0])); i++)
	{
		int w;

		if (!i)
		{
			make_tlist_first_col(GTK_CLIST(tlist),
				gettext(colinfos[i].label));
		}
		else
		{
			make_clist_col_widget(tlist, colinfos[i].colnr, i,
				gettext(colinfos[i].label));
		}

		gtk_clist_set_column_justification(GTK_CLIST(tlist),
			colinfos[i].colnr, colinfos[i].justify);

		gtk_clist_set_column_width(GTK_CLIST(tlist), colinfos[i].colnr,
			gprop_get_int(colinfos[i].twprop, &w) ? w : colinfos[i].dwidth);
	}

	gtk_signal_connect(GTK_OBJECT(tlist), "resize_column", 
		GTK_SIGNAL_FUNC(list_resize_column), NULL);

	gtk_clist_column_titles_show(GTK_CLIST(tlist));
	gtk_ctree_set_line_style(GTK_CTREE(tlist), GTK_CTREE_LINES_DOTTED);
	gtk_clist_set_selection_mode(GTK_CLIST(tlist), GTK_SELECTION_BROWSE);
	gtk_container_add(GTK_CONTAINER(swin) , tlist);
	gtk_clist_column_titles_passive(GTK_CLIST(tlist));
	gtk_widget_show(tlist);

	gtk_signal_connect(GTK_OBJECT(tlist) , "tree_select_row" , 
		GTK_SIGNAL_FUNC(UpdateEInfo) , NULL);

	gtk_signal_connect(GTK_OBJECT(tlist) , "tree_select_row" , 
		GTK_SIGNAL_FUNC(AdjustMenu) , NULL);

	flow_menu = gtk_menu_new();
	gtk_widget_realize(flow_menu);

	smi = gtk_menu_item_new_with_label(gettext("Save to file ..."));
	gaccel_bind_widget("rmb/save" , "activate" , smi, NULL, tl);
	gtk_menu_append(GTK_MENU(flow_menu) , smi);
	gtk_widget_show(smi);

	gtk_signal_connect(GTK_OBJECT(smi) , "activate" ,
		GTK_SIGNAL_FUNC(SaveAs) , NULL);

	rmi = gtk_menu_item_new_with_label(gettext("Delete file"));
	gaccel_bind_widget("rmb/delete" , "activate" , rmi, NULL, tl);
	gtk_menu_append(GTK_MENU(flow_menu) , rmi);
	gtk_widget_show(rmi);

	gtk_signal_connect(GTK_OBJECT(rmi) , "activate" ,
		GTK_SIGNAL_FUNC(Remove) , NULL);

	cmi = gtk_menu_item_new_with_label(gettext("Copy url to clipboard"));
	gaccel_bind_widget("rmb/copy_clipboard" , "activate" , cmi, NULL, tl);
	gtk_menu_append(GTK_MENU(flow_menu) , cmi);
	gtk_widget_show(cmi);

	gtk_signal_connect(GTK_OBJECT(cmi) , "activate" ,
		GTK_SIGNAL_FUNC(SetSelectionURL) , NULL);

	nmi = gtk_menu_item_new_with_label(gettext("Show entry informations"));
	gaccel_bind_widget("rmb/showinfo" , "activate" , nmi, NULL, tl);
	gtk_menu_append(GTK_MENU(flow_menu) , nmi);
	gtk_widget_show(nmi);

	gtk_signal_connect(GTK_OBJECT(nmi) , "activate" ,
		GTK_SIGNAL_FUNC(NodeInfo) , NULL);

	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(flow_menu) , mi);
	gtk_widget_show(mi);

	omi = gtk_menu_item_new_with_label(gettext("read only permisions"));
	gaccel_bind_widget("rmb/read_only" , "activate" , omi, NULL, tl);
	gtk_menu_append(GTK_MENU(flow_menu) , omi);
	gtk_widget_show(omi);

	gtk_signal_connect(GTK_OBJECT(omi) , "activate" ,
		GTK_SIGNAL_FUNC(RdOnly) , NULL);

	wmi = gtk_menu_item_new_with_label(gettext("read/write permissions"));
	gaccel_bind_widget("rmb/read_write" , "activate" , wmi, NULL, tl);
	gtk_menu_append(GTK_MENU(flow_menu) , wmi);
	gtk_widget_show(wmi);

	gtk_signal_connect(GTK_OBJECT(wmi) , "activate" ,
		GTK_SIGNAL_FUNC(Writable) , NULL);

	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(flow_menu) , mi);
	gtk_widget_show(mi);

	smenu = gtk_menu_new();
	gtk_signal_connect(GTK_OBJECT(smenu), "show",
		GTK_SIGNAL_FUNC(GenViewerMenu), (gpointer)TRUE);
	gtk_widget_realize(smenu);

	fmi = gtk_menu_item_new_with_label(gettext("View file with"));
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(fmi), smenu);
	gaccel_bind_widget("rmb/view_file" , "activate" , fmi, NULL, tl);
	gtk_menu_append(GTK_MENU(flow_menu) , fmi);
	gtk_widget_show(fmi);

	smenu = gtk_menu_new();
	gtk_signal_connect(GTK_OBJECT(smenu), "show",
		GTK_SIGNAL_FUNC(GenViewerMenu), (gpointer)FALSE);
	gtk_widget_realize(smenu);

	umi = gtk_menu_item_new_with_label(gettext("View URL with"));
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(umi), smenu);
	gaccel_bind_widget("rmb/view_url" , "activate" , umi, NULL, tl);
	gtk_menu_append(GTK_MENU(flow_menu) , umi);
	gtk_widget_show(umi);

	mi = gtk_menu_item_new_with_label(gettext("Find ..."));
	gaccel_bind_widget("view/find", "activate", mi, NULL, tl);
	gtk_menu_append(GTK_MENU(flow_menu), mi);
	gtk_widget_show(mi);

	gtk_signal_connect(GTK_OBJECT(mi), "activate",
		GTK_SIGNAL_FUNC(search_dlg_open), NULL);

	gtk_signal_connect(GTK_OBJECT(tlist) , "button_press_event" , 
		GTK_SIGNAL_FUNC(popup_tmenu) , flow_menu);
}

static void tl_resize(w, sallocation, fdata)
GtkWidget *w;
GtkAllocation *sallocation;
{
	gprop_set_int("tl_width", sallocation->width);
	gprop_set_int("tl_height", sallocation->height);
}

static void _noop()
{
	return;
}

int main(argc , argv)
int argc;
char **argv;
{
	GtkWidget *box,*sep,*frame,*notebook;
	GdkColor rc;
	int w,h,i;
	GtkStyle *rs,*my_style;
	static GtkTargetEntry targetlist[] = {
		{"STRING", 0, STRING},
                {"TEXT", 0, TEXT},
                {"COMPOUND_TEXT", 0, COMPOUND_TEXT}
        };

	INIT_LOCALE

	load_rc();

	if (cmd_setup(argc, argv)) return 1;

	if (!gprop_get_bool("have_viewers", &w))
		assign_default_viewers();

	gtk_init(&argc , &argv);

	g_log_set_handler("Gdk", G_LOG_LEVEL_MASK, (GLogFunc) _noop, NULL);

	tl = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(tl), gettext("NScache"));
	gtk_window_set_policy(GTK_WINDOW(tl), TRUE, TRUE, TRUE);
	w = -1;
	h = -1;
	gprop_get_int("tl_width", &w);
	gprop_get_int("tl_height", &h);
	gtk_window_set_default_size(GTK_WINDOW(tl), w, h);
	gtk_signal_connect (GTK_OBJECT (tl), "destroy",
		GTK_SIGNAL_FUNC(Quit), NULL);
	gtk_signal_connect(GTK_OBJECT(tl) , "selection_clear_event" ,
		GTK_SIGNAL_FUNC(SelectionClear) , NULL);
	gtk_selection_add_targets (tl, GDK_SELECTION_PRIMARY,
		targetlist , sizeof(targetlist)/sizeof(targetlist[0]));
	gtk_signal_connect(GTK_OBJECT(tl) , "selection_get" ,
		GTK_SIGNAL_FUNC(SendSelection) , NULL);
	gtk_signal_connect(GTK_OBJECT(tl) , "size_allocate" ,
		GTK_SIGNAL_FUNC(tl_resize) , NULL);
	gtk_widget_realize(tl);

	box = gtk_vbox_new(FALSE , 1);
	gtk_container_add(GTK_CONTAINER(tl) , box);
	gtk_widget_show(box);

	sep = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(box), sep , FALSE, TRUE, 4);
	gtk_widget_show(sep);

	build_seldb(box);

	notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(box), notebook , TRUE, TRUE, 1);
	gtk_widget_show(notebook);

	for (i = 0; i < 9; i++)
		gprop_get_int(colinfos[i].colnrprop, &colinfos[i].colnr);

	build_tlist(notebook);
	build_slist(notebook);

	build_menu(box);

	frame = gtk_frame_new(NULL);
	gtk_box_pack_start(GTK_BOX (box), frame , FALSE, TRUE, 4);
	gtk_widget_show(frame);

	status = gtk_label_new("");
	gtk_misc_set_alignment(GTK_MISC(status) , 0.0 , 0.5);
	gtk_container_add(GTK_CONTAINER(frame) , status);
	gtk_widget_show(status);

	frame = gtk_frame_new(NULL);
	gtk_box_pack_start(GTK_BOX (box), frame , FALSE, TRUE, 4);
	gtk_widget_show(frame);

	err = gtk_label_new("");
	my_style = gtk_rc_get_style(err);
	if (!my_style)
		my_style = gtk_widget_get_style(err);
	rs = gtk_style_copy(my_style);
	gdk_color_parse("#ff0000" , &rc);
	gdk_color_alloc(gdk_colormap_get_system() , &rc);
	rs->fg[GTK_STATE_NORMAL] = rc;
	gtk_widget_set_style(err , rs);
	gtk_misc_set_alignment(GTK_MISC(err) , 0.0 , 0.5);
	gtk_container_add(GTK_CONTAINER(frame) , err);
	gtk_widget_show(err);

	nscache_icons_load();

	if (db_name)
	{
		gtk_entry_set_text(GTK_ENTRY(entry) , db_name);
		Open(NULL , entry);
	}

	gtk_widget_show(tl);

	gtk_main();
	return 0;
}

