/*
 * Copyright 2006 Ricoh Corporation.
 * 
 * 
 * APACHE LICENSE VERSION 2.0
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * 
 * RICOH DEVELOPER PROGRAM SUPPORT:
 * 
 * Support for this software is available only to "Premier Plus" members
 * of the Ricoh Developer Program (RiDP).  You may find out more 
 * information about the Program at
 * 
 *      http://americas.ricoh-developer.com
 * 
 * Premier plus members may find answers and ask questions through the
 * RiDP customer help website at
 * 
 *      https://ridp.custhelp.com
 * 
 * Developers who are not RiDP members may still use this software as
 * stipulated in the license terms given above.
 *
 */ 

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import jp.co.ricoh.dsdk.panel.Label;
import jp.co.ricoh.dsdk.panel.Button;
import jp.co.ricoh.dsdk.panel.Component;
import jp.co.ricoh.dsdk.panel.Font;
import jp.co.ricoh.dsdk.panel.Frame;
import jp.co.ricoh.dsdk.panel.Text;
import jp.co.ricoh.dsdk.panel.event.ActionListener;

import ricoh.dsdk.uiwidgets.*;

public class WidgetsFromFormFactory
{
	public static class Coordinate
	{
		int x;
		int y;
		
		public Coordinate(int xCoord, int yCoord)
		{
			x = xCoord;
			y = yCoord;
		}
		
		int getX()
		{
			return x;
		}
		
		int getY()
		{
			return y;
		}
	}
	
	public interface WidgetLayoutHandler
	{
		public Frame getFrame();
		public Coordinate reserveSpace(int width, int height);
	}
	
	public interface WidgetFactory
	{
		public void createWidget(final XHTMLForm theForm, final String itemName, final XHTMLForm.IFormItem formItem, WidgetLayoutHandler layoutHandler);
	}
	

	public static void createWidgetsFromFormFactory(final Frame frame, XHTMLForm theForm, final int x, final int y, final int areaWidth, final int areaHeight)
	{
		WidgetsFromFormFactory.createWidgetsFromFormFactory(theForm, new WidgetsFromFormFactory.WidgetLayoutHandler() 
			{
				int itemX = y;
				int itemY = y;
				int rowWidth = 0;
				static final int spacing = 8;

				public Frame getFrame()
				{
					return frame;
				}
				
				public WidgetsFromFormFactory.Coordinate reserveSpace(int itemWidth, int itemHeight)
				{
					if(rowWidth < itemWidth)
						rowWidth = itemWidth;
					
					WidgetsFromFormFactory.Coordinate result = new WidgetsFromFormFactory.Coordinate(itemX, itemY);
					itemY = itemY + itemHeight + spacing;
					if(itemY + itemHeight > y + areaHeight)
					{
						itemX = itemX + rowWidth + spacing;
						itemY = y;
					}
					
					return result;
				}
			} );
	}
	
	public static void createWidgetsFromFormFactory(XHTMLForm theForm, WidgetLayoutHandler layoutHandler)
	{
		//
		// Iterate over the items in the form.
		//
		Iterator iter = theForm.getFormNameSet().iterator();
		while(iter.hasNext())
		{
			String formItemName = (String)iter.next();
			
			XHTMLForm.IFormItem oneFormItem = theForm.getFormItem(formItemName);
			WidgetFactory widget = WidgetsFromFormFactory.getWidgetFactory(oneFormItem.getType().toString());
			if(widget != null)
			{
				widget.createWidget(theForm, formItemName, oneFormItem, layoutHandler);
			}
		}

	}

	static Map		widgetFactory = null;
	static
	{
		//
		// Register the various kinds of recognized html form
		// input items.  Note that we will never create a widget
		// for items whose type is 'hidden', 'readonly' or 'file'.
		// Hidden and readonly represent non-user-editable types,
		// whereas file indicates the scanned document to upload.
		//
		// Similarly, "submit" items are stored in a separate list
		// from other input item types and are not handled by
		// widget factories.
		//
		WidgetsFromFormFactory.addWidgetFactory("text", new TextWidgetFactory(false));
		WidgetsFromFormFactory.addWidgetFactory("password", new TextWidgetFactory(true));
		WidgetsFromFormFactory.addWidgetFactory("checkbox", new CheckboxWidgetFactory());
		WidgetsFromFormFactory.addWidgetFactory("radio", new RadioWidgetFactory());
		WidgetsFromFormFactory.addWidgetFactory("select", null);

	}
	
	public static void addWidgetFactory(String factoryName, WidgetFactory theFactory)
	{
		if(widgetFactory == null)
		{
			widgetFactory = new HashMap();
		}
		
		widgetFactory.put(factoryName, theFactory);
	}

	protected static WidgetFactory getWidgetFactory(String widgetType)
	{
		WidgetFactory result = null;
		
		if(widgetFactory.containsKey(widgetType))
			result = (WidgetFactory) widgetFactory.get(widgetType);
		
		return result;
	}

	static class TextWidgetFactory implements WidgetFactory
	{
		boolean password;
		
		public TextWidgetFactory(boolean usePasswordMode)
		{
			password = usePasswordMode;
		}
		
		public void createWidget(final XHTMLForm theForm, final String itemName, final XHTMLForm.IFormItem formItem, WidgetLayoutHandler layoutHandler)
		{
			Frame frame = layoutHandler.getFrame();
			Coordinate reservedSpace = layoutHandler.reserveSpace(176, 16);

			String label = formItem.getLabel();
			Text description = new Text("Form value");
			if(label != null)
			{
				description = new Text(label);
			}
			
			Button theDescription = new Button(description);
        		theDescription.setLocation(reservedSpace.getX(), reservedSpace.getY());
        		theDescription.setSize(80, 16);
        		theDescription.setFont(Font.F12);
        		theDescription.setWink(true);
						
        	Label theCurrentValue = new Label(formItem.getValue());
        		theCurrentValue.setLocation(reservedSpace.getX() + 80, reservedSpace.getY());
        		theCurrentValue.setSize(80, 16);
        		theCurrentValue.setFont(Font.F12);

			ActionListener action = UserInput.actionListener(frame, description, password, theCurrentValue, new UserInput.Handler()
				{
					public void processInput(String userInput)
					{
						theForm.setItemValue(itemName, userInput);
					}
				});
			
			theDescription.addActionListener(action);
			
			frame.add(theDescription);
			frame.add(theCurrentValue);
			
		}
	}

	static class CheckboxWidgetFactory implements WidgetFactory
	{
		public void createWidget(final XHTMLForm theForm, final String itemName, final XHTMLForm.IFormItem formItem, WidgetLayoutHandler layoutHandler)
		{
			Frame frame = layoutHandler.getFrame();
			String[] descriptions = formItem.getDescriptionList();
			String[] values = formItem.getValueList();
			Coordinate reservedSpace = layoutHandler.reserveSpace(160, 16);

			final String theTrueValue = (values[0].length() > 0) ? values[0] : "on";
			final String theFalseValue = (values[0].length() > 0) ? "" : "off";
			boolean checked = formItem.getValue().equals(theTrueValue);

			Checkbox theCheckbox = Checkbox.create(frame, descriptions[0], checked, reservedSpace.getX(), reservedSpace.getY(), 160, 16, new CheckboxControl.Handler()
					{
						public void valueChanged(boolean checkedValue)
						{
							theForm.setItemValue(itemName, checkedValue ? theTrueValue : theFalseValue);
						}
					});
			
			/*
			Button theButton = new Button();

        		theButton.setLocation(reservedSpace.getX(), reservedSpace.getY());
        		theButton.setSize(16, 16);
        		theButton.setFont(Font.F12);
        		theButton.setShape(Button.Shape.NONE);
        		theButton.setWink(true);

			Label theLabel = new Label(descriptions[0]);
        		theLabel.setLocation(reservedSpace.getX() + 20, reservedSpace.getY());
        		theLabel.setSize(76, 16);
        		theLabel.setFont(Font.F12);

			CheckboxControl checkboxControl = new CheckboxControl(checked, theButton);
			checkboxControl.addHandler(new CheckboxControl.Handler()
					{
						public void valueChanged(boolean checkedValue)
						{
							theForm.setItemValue(itemName, checkedValue ? theTrueValue : theFalseValue);
						}
					});
			
			theButton.addActionListener(checkboxControl);
			frame.add(theButton);
			frame.add(theLabel);
			*/
		}
	}

	static class RadioWidgetFactory implements WidgetFactory
	{
		public void createWidget(final XHTMLForm theForm, final String itemName, final XHTMLForm.IFormItem formItem, WidgetLayoutHandler layoutHandler)
		{
			Frame frame = layoutHandler.getFrame();
			String[] descriptions = formItem.getDescriptionList();
			String[] values = formItem.getValueList();
			Coordinate reservedSpace = layoutHandler.reserveSpace(160, 16 * values.length);

			RadioButtons theRadioGroup = RadioButtons.create(frame, descriptions, values, formItem.getValue(), reservedSpace.getX(), reservedSpace.getY(), 160, 16, new RadioGroupControl.Handler()
				{
					public void valueChanged(String controlValue)
					{
						theForm.setItemValue(itemName, controlValue);
					}
				});

		/*			
			//
			// This works pretty much the same way as the checkbox control;
			// one main difference is that the radio control contains a list
			// of buttons, and the action performed method will go through 
			// and un-set all of the sibling radio buttons (any button in
			// the list that is not the target of the ActionEvent).
			//
			RadioGroupControl radioControl = new RadioGroupControl(formItem.getValue());
			
			for(int i=0; i<values.length; ++i)
			{
				Button theButton = new Button("");
			
        			theButton.setLocation(reservedSpace.getX(), reservedSpace.getY() + (16 * i));
        			theButton.setSize(16, 16);
        			theButton.setFont(Font.F12);
        			theButton.setShape(Button.Shape.NONE);
        			theButton.setWink(true);
				theButton.setActionCommand(values[i]);
				theButton.addActionListener(radioControl);
				frame.add(theButton);

				Label theLabel = new Label(descriptions[i]);
        			theLabel.setLocation(reservedSpace.getX() + 20, reservedSpace.getY() + (16 * i));
        			theLabel.setSize(76, 16);
        			theLabel.setFont(Font.F12);

				frame.add(theLabel);

				radioControl.addRadioButton(theButton);

			}
			radioControl.addHandler(new RadioGroupControl.Handler()
				{
					public void valueChanged(String controlValue)
					{
						theForm.setItemValue(itemName, controlValue);
					}
				});
		*/
		}
	}
	
	static class ComboBoxWidgetFactory implements WidgetFactory
	{
		public void createWidget(final XHTMLForm theForm, final String itemName, final XHTMLForm.IFormItem formItem, WidgetLayoutHandler layoutHandler)
		{
			Frame frame = layoutHandler.getFrame();
			String[] descriptions = formItem.getDescriptionList();
			String[] values = formItem.getValueList();
			Coordinate reservedSpace = layoutHandler.reserveSpace(96, 16 * values.length);
			
			//
			// This works pretty much the same way as the checkbox control;
			// one main difference is that the radio control contains a list
			// of buttons, and the action performed method will go through 
			// and un-set all of the sibling radio buttons (any button in
			// the list that is not the target of the ActionEvent).
			//
			ComboBox comboBox = new ComboBox(frame);
			
			comboBox.setBounds(reservedSpace.getX(), reservedSpace.getY(), comboBox.getWidth(), comboBox.getHeight());
			
			for(int i=0; i<values.length; ++i)
			{
				comboBox.addItem(descriptions[i]);
			}
			
			comboBox.setValue(formItem.getValue());

			comboBox.addHandler(new ComboBox.Handler()
					{
						public void valueChanged(String controlValue)
						{
							theForm.selectItemValue(itemName, controlValue);
						}
					});
			
			comboBox.create();
		}
	}
}

