/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2007 Sun Microsystems, Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package org.netbeans.microedition.databinding.el;

import java.util.Hashtable;
import java.util.Vector;

/**
 * @author David Kaspar
 */
public class ELUtils {

    public static boolean coerceToBoolean (Object object) {
        return Coercions.coerceObjectToBoolean (object, null).booleanValue ();
    }

    public static boolean applyEqualityOperator (Object value1, Object value2, int operator) {
        switch (operator) {
            case OPERATOR_EQUAL:
            case OPERATOR_NON_EQUAL:
                return Coercions.applyEqualityOperator (value1, value2, operator, null).booleanValue ();
            default:
                throw new IllegalStateException ();
        }
    }

    public static boolean applyRelationalOperator (Object value1, Object value2, int operator) {
        switch (operator) {
            case OPERATOR_LT:
            case OPERATOR_GT:
                if (value1 == value2)
                    return false;
                else if (value1 == null || value2 == null)
                    return false;
                return Coercions.applyRelationalOperator (value1, value2, operator, null).booleanValue ();
            case OPERATOR_GE:
            case OPERATOR_LE:
                if (value1 == value2)
                    return true;
                else if (value1 == null || value2 == null)
                    return false;
                return Coercions.applyRelationalOperator (value1, value2, operator, null).booleanValue ();
            default:
                throw new IllegalStateException ();
        }
    }

    public static Object applyArithmeticOperator (Object value1, Object value2, int operator) {
        switch (operator) {
            case OPERATOR_PLUS:
            case OPERATOR_MINUS:
            case OPERATOR_MULTIPLY:
                return Coercions.applyArithmeticOperator (value1, value2, operator, null);
            case OPERATOR_DIVIDE:
                return applyDivideOperator (value1, value2, null);
            case OPERATOR_MODULUS:
                return applyModuleOperator (value1, value2, null);
            default:
                throw new IllegalStateException ();
        }
    }

    private static Object applyModuleOperator (Object pLeft, Object pRight, Logger pLogger) {
        if (pLeft == null && pRight == null) {
//            if (pLogger.isLoggingWarning ()) {
//                pLogger.logWarning (Constants.ARITH_OP_NULL, getOperatorSymbol ());
//            }
            return new Integer (0);
        }

        if ((pLeft != null && (Coercions.isFloatingPointType (pLeft) || Coercions.isFloatingPointString (pLeft))) ||
            (pRight != null && (Coercions.isFloatingPointType (pRight) || Coercions.isFloatingPointString (pRight)))) {
            double left = Coercions.doubleValue (Coercions.coerceObjectToPrimitiveNumber (pLeft, Double.class, pLogger));
            double right = Coercions.doubleValue (Coercions.coerceObjectToPrimitiveNumber (pRight, Double.class, pLogger));

            try {
                return new Double (left % right);
            } catch (Exception exc) {
//                if (pLogger.isLoggingError ()) {
//                    pLogger.logError (Constants.ARITH_ERROR, getOperatorSymbol (), "" + left, "" + right);
//                }
                return new Integer (0);
            }
        } else {
            long left = Coercions.longValue (Coercions.coerceObjectToPrimitiveNumber (pLeft, Long.class, pLogger));
            long right = Coercions.longValue (Coercions.coerceObjectToPrimitiveNumber (pRight, Long.class, pLogger));

            try {
                return new Long (left % right);
            } catch (Exception exc) {
//                if (pLogger.isLoggingError ()) {
//                    pLogger.logError (Constants.ARITH_ERROR, getOperatorSymbol (), "" + left, "" + right);
//                }
                return new Integer (0);
            }
        }
    }

    public static Object applyUnaryOperation (Object value, int operator) {
        switch (operator) {
            case OPERATOR_UNARY_MINUS:
                return applyUnaryMinusOperator (value, null);
            case OPERATOR_UNARY_NOT:
                return applyUnaryNotOperator (value, null);
            case OPERATOR_EMPTY:
                return applyEmptyOperator (value, null);
            default:
                throw new IllegalStateException ();
        }
    }

    private static Object applyUnaryNotOperator (Object pValue, Logger pLogger) {
        boolean val = Coercions.coerceObjectToBoolean (pValue, pLogger).booleanValue ();
        return !val ? Boolean.TRUE : Boolean.FALSE;
    }

    private static Object applyEmptyOperator (Object pValue, Logger pLogger) {
        if (pValue == null)
            return Boolean.TRUE;
        if ("".equals (pValue))
            return Boolean.TRUE;
        if (pValue.getClass ().isArray () && ((Object[]) pValue).length == 0)// Array.getLength (pValue) == 0)
            return Boolean.TRUE;
        if (pValue instanceof Vector && ((Vector) pValue).isEmpty ())
            return Boolean.TRUE;
        if (pValue instanceof Hashtable && ((Hashtable) pValue).isEmpty ())
            return Boolean.TRUE;
        return Boolean.FALSE;
    }

    private static Object applyUnaryMinusOperator (Object pValue, Logger pLogger) {
        if (pValue == null) {
//            if (pLogger.isLoggingWarning ()) {
//                pLogger.logWarning (Constants.ARITH_OP_NULL);
//            }
            return new Integer (0);
        } else if (pValue instanceof String) {
            if (Coercions.isFloatingPointString (pValue)) {
                double dval = Coercions.doubleValue (Coercions.coerceObjectToPrimitiveNumber (pValue, Double.class, pLogger));
                return new Double (-dval);
            } else {
                long lval = Coercions.longValue (Coercions.coerceObjectToPrimitiveNumber (pValue, Long.class, pLogger));
                return new Long (-lval);
            }
        } else if (pValue instanceof Byte) {
            return new Byte ((byte) -(((Byte) pValue).byteValue ()));
        } else if (pValue instanceof Short) {
            return new Short ((short) -(((Short) pValue).shortValue ()));
        } else if (pValue instanceof Integer) {
            return new Integer (-(((Integer) pValue).intValue ()));
        } else if (pValue instanceof Long) {
            return new Long (-(((Long) pValue).longValue ()));
        } else if (pValue instanceof Float) return new Float ((float) -(((Float) pValue).floatValue ())); else if (pValue instanceof Double) {
            return new Double (-(((Double) pValue).doubleValue ()));
        } else {
            if (pLogger.isLoggingError ()) {
                pLogger.logError (Constants.UNARY_OP_BAD_TYPE, pValue.getClass ().getName ());
            }
            return new Integer (0);
        }
    }

    private static Object applyDivideOperator (Object pLeft, Object pRight, Logger pLogger) {
        if (pLeft == null && pRight == null) {
//            if (pLogger.isLoggingWarning ()) {
//                pLogger.logWarning (Constants.ARITH_OP_NULL, getOperatorSymbol ());
//            }
            return new Integer (0);
        }

        double left = Coercions.doubleValue (Coercions.coerceObjectToPrimitiveNumber (pLeft, Double.class, pLogger));
        double right = Coercions.doubleValue (Coercions.coerceObjectToPrimitiveNumber (pRight, Double.class, pLogger));

        try {
            return new Double (left / right);
        } catch (Exception exc) {
//            if (pLogger.isLoggingError ()) {
//                pLogger.logError (Constants.ARITH_ERROR, getOperatorSymbol (), "" + left, "" + right);
//            }
            return new Integer (0);
        }
    }

    public static final int OPERATOR_EQUAL = '=';
    public static final int OPERATOR_NON_EQUAL = '!';

    public static final int OPERATOR_LT = '<';
    public static final int OPERATOR_GT = '>';
    public static final int OPERATOR_GE = 'l';
    public static final int OPERATOR_LE = 'g';

    public static final int OPERATOR_PLUS = '+';
    public static final int OPERATOR_MINUS = '-';

    public static final int OPERATOR_MULTIPLY = '*';
    public static final int OPERATOR_DIVIDE = '/';
    public static final int OPERATOR_MODULUS = '%';

    public static final int OPERATOR_UNARY_NOT = '!';
    public static final int OPERATOR_UNARY_MINUS = '-';
    public static final int OPERATOR_EMPTY = ' ';

}
