/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.j2ee.websphere6.config;

import java.io.*;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.netbeans.modules.j2ee.websphere6.*;

import javax.enterprise.deploy.model.*;
import javax.enterprise.deploy.spi.*;
import javax.enterprise.deploy.spi.exceptions.*;
import javax.enterprise.deploy.spi.DeploymentManager;
import javax.enterprise.deploy.shared.ModuleType;
import org.netbeans.modules.j2ee.deployment.common.api.*;
import org.netbeans.modules.j2ee.deployment.plugins.api.*;
import org.netbeans.modules.j2ee.deployment.devmodules.api.*;
import org.netbeans.modules.j2ee.websphere6.util.WSDebug;
import org.netbeans.modules.j2ee.websphere6.util.WSUtil;
import org.netbeans.modules.schema2beans.BaseBean;
import org.openide.filesystems.FileLock;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileSystem;
import org.openide.filesystems.FileUtil;
import javax.enterprise.deploy.spi.exceptions.BeanNotFoundException;
import javax.enterprise.deploy.spi.exceptions.ConfigurationException;
import javax.enterprise.deploy.spi.exceptions.InvalidModuleException;
import org.openide.util.NbBundle;
import org.openide.loaders.DataObject;


/**
 * A wrapper for the server's DeploymentConfiguration implementation. It is
 * used mostly to solve the classloading problems, i.e. to substitute the
 * thread's context classloader with the WebSphere loader and back when
 * WebSphere method returns
 *
 * @author Kirill Sorokin
 */
public class WSDeploymentConfiguration implements DeploymentConfiguration {
    
    private DeploymentManager dm;
    
    /**
     * The WS implementation of DeploymentConfiguration, all calls are
     * delegated to it
     */
    private DeploymentConfiguration configuration;
    
    /**
     * Wrapper for j2eeserver's DeployableObject, it is used for correctly
     * getting the DConfigBeanRoot
     */
    private WSDeployableObject deployableObject;
    
    /**
     * Instance properties for the current server instance
     */
    private InstanceProperties instanceProperties;
    
    /**
     * Current clasloader used for WS classes
     */
    private WSClassLoader loader;
    
    
    protected DataObject [] dataObjects;
    
    
    public DataObject [] getDataObject() {
        return dataObjects;
    }
    /**
     * Creates a new instance of WSDeploymentConfiguration.
     *
     * @param dm the DeploymentManager
     * @param deployableObject j2eeserver's DeployableObject
     * @param instanceProperties instance properties for the current server
     *      instance
     *
     * @throws InvalidModuleException never thrown in this implementation
     */
    public WSDeploymentConfiguration(DeploymentManager dm,
            DeployableObject deployableObject,
            InstanceProperties instanceProperties)
            throws InvalidModuleException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "WSDeploymentConfiguration()"); // NOI18N
        
        // save the instance properties
        this.instanceProperties = instanceProperties;
        
        // get the correct loader instance
        loader = WSClassLoader.getInstance(instanceProperties.getProperty(
                WSDeploymentFactory.SERVER_ROOT_ATTR),
                instanceProperties.getProperty(
                WSDeploymentFactory.DOMAIN_ROOT_ATTR));
        
        this.dm = dm;
        
        // create a proper wrapper for the supplied deployable object - either
        // a general deployable object or an application object
        if(deployableObject.getType()==ModuleType.EAR) {
            this.deployableObject=new WSJ2eeApplicationObject(deployableObject);
        } else {
            this.deployableObject = new WSDeployableObject(deployableObject);
        }
        
        updateDeploymentConfiguration();
    }
    
    private void updateDeploymentConfiguration() throws InvalidModuleException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "updateDeploymentConfiguration()"); // NOI18N
        
        try {
            configuration = dm.createConfiguration(deployableObject);
        } catch (InvalidModuleException e) {
            if (WSDebug.isEnabled()) // debug output
                WSDebug.notify(e);
            throw e;
        }
    }
    
    /**
     * Delegates the call to the server's DeploymentConfiguration, handling
     * the classloading issues, if necessary.
     */
    public DConfigBeanRoot getDConfigBeanRoot(DDBeanRoot bean)
    throws ConfigurationException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getDConfigBeanRoot(" + bean +  // NOI18N
                    ")");                                              // NOI18N
        
        try {
            updateDeploymentConfiguration();
        } catch (InvalidModuleException e) {
            throw (ConfigurationException) new ConfigurationException().
                    initCause(e);
        }
        
        // update the context classloader
        loader.updateLoader();
        
        // call the server's class
        DConfigBeanRoot dConfigBeanRoot = configuration.getDConfigBeanRoot(
                deployableObject.findDDBeanRoot(bean));
        
        // restore the context classloader
        loader.restoreLoader();
        
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "returning: " +                 // NOI18N
                    dConfigBeanRoot);
        
        // return
        return dConfigBeanRoot;
    }
    
    /**
     * Delegates the call to the server's DeploymentConfiguration, handling
     * the classloading issues, if necessary.
     */
    public DeployableObject getDeployableObject() {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getDeployableObject()");       // NOI18N
        return deployableObject;
        //return configuration.getDeployableObject();
    }
    
    /**
     * Delegates the call to the server's DeploymentConfiguration, handling
     * the classloading issues, if necessary.
     */
    public void removeDConfigBean(DConfigBeanRoot bean)
    throws BeanNotFoundException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "removeDConfigBean(" + bean +   // NOI18N
                    ")");                                              // NOI18N
        
        try {
            updateDeploymentConfiguration();
        } catch (InvalidModuleException e) {
            throw (BeanNotFoundException) new BeanNotFoundException("").
                    initCause(e);
        }
        
        try {
            configuration.removeDConfigBean(bean);
        } catch (BeanNotFoundException e) {
            if (WSDebug.isEnabled()) // debug output
                WSDebug.notify(e);
            
            throw e;
        }
    }
    
    /**
     * Delegates the call to the server's DeploymentConfiguration, handling
     * the classloading issues, if necessary.
     */
    public void restore(InputStream inputArchive) throws ConfigurationException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "restore(" + inputArchive +     // NOI18N
                    ")");                                              // NOI18N
        
        try {
            updateDeploymentConfiguration();
        } catch (InvalidModuleException e) {
            throw (ConfigurationException) new ConfigurationException().
                    initCause(e);
        }
        
        try {
            configuration.restore(inputArchive);
        } catch (ConfigurationException e) {
            if (WSDebug.isEnabled()) // debug output
                WSDebug.notify(e);
            
            throw e;
        }
    }
    
    /**
     * Delegates the call to the server's DeploymentConfiguration, handling
     * the classloading issues, if necessary.
     */
    public DConfigBeanRoot restoreDConfigBean(InputStream inputArchive,
            DDBeanRoot bean) throws ConfigurationException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "restoreDConfigBean(" +         // NOI18N
                    inputArchive + ", " + bean + ")");                 // NOI18N
        
        try {
            updateDeploymentConfiguration();
        } catch (InvalidModuleException e) {
            throw (ConfigurationException) new ConfigurationException().
                    initCause(e);
        }
        
        try {
            return configuration.restoreDConfigBean(inputArchive, bean);
        } catch (ConfigurationException e) {
            if (WSDebug.isEnabled()) // debug output
                WSDebug.notify(e);
            
            throw e;
        }
    }
    
    /**
     * Delegates the call to the server's DeploymentConfiguration, handling
     * the classloading issues, if necessary.
     */
    public void save(OutputStream outputArchive) throws ConfigurationException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "save(" + outputArchive + ")"); // NOI18N
        
        try {
            updateDeploymentConfiguration();
        } catch (InvalidModuleException e) {
            throw (ConfigurationException) new ConfigurationException().
                    initCause(e);
        }
        
        // updte the context classloader
        loader.updateLoader();
        
        try {
            configuration.save(outputArchive);
        } catch (ConfigurationException e) {
            if (WSDebug.isEnabled())  { // debug output
                WSDebug.notify(e);
            }
            throw e;
        }
        
        // restore the context classloader
        loader.restoreLoader();
    }
    
    /**
     * Delegates the call to the server's DeploymentConfiguration, handling
     * the classloading issues, if necessary.
     */
    public void saveDConfigBean(OutputStream outputArchive,
            DConfigBeanRoot bean) throws ConfigurationException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "saveDConfigBean(" +            // NOI18N
                    outputArchive + ", " + bean + ")");                // NOI18N
        
        try {
            updateDeploymentConfiguration();
        } catch (InvalidModuleException e) {
            throw (ConfigurationException) new ConfigurationException().
                    initCause(e);
        }
        
        try {
            configuration.saveDConfigBean(outputArchive, bean);
        } catch (ConfigurationException e) {
            if (WSDebug.isEnabled()) // debug output
                WSDebug.notify(e);
            
            throw e;
        }
    }
    
// Formerly DeploymentPlanSplitter read/write config files.
    public void readDeploymentPlanFiles() throws ConfigurationException {
        ; //TODO
    }
    
    public void writeDeploymentPlanFiles(DeployableObject module, File[] files)
    throws ConfigurationException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(files);
        
        // perform the per-module type tuning: web modules
        
        if (module.getType().equals(ModuleType.WAR)) {
            // get the file handle for the web.xml
            File file = new WSDeployableObject(module).
                    getEntryFile("WEB-INF/web.xml");                   // NOI18N
            
            // get the file's contents
            String contents = WSUtil.readFile(file);
            
            // write the updated contents
            WSUtil.writeFile(file, contents.replaceFirst("<web-app x", // NOI18N
                    "<web-app id=\"WebApp\" \n	 x"));                 // NOI18N
        }
        
        // iterate over the files array, writing stub entries basing on files'
        // names
        for (int i = 0; i < files.length; i++) {
            // if the file exists - skip it
            if (files[i].exists()) {
                continue;
            }
            
            // save the file's name
            String fileName = files[i].getPath();
            
            // init the string to be output to the file
            String output = "";

            // write stub for IBM's web binding descriptor
            if (fileName.endsWith("ibm-web-bnd.xmi")) {                // NOI18N
                output =
                        "<webappbnd:WebAppBinding xmi:version=\"2.0\" xmlns:xmi=\"http://www.omg.org/XMI\" xmlns:webappbnd=\"webappbnd.xmi\" xmlns:webapplication=\"webapplication.xmi\"  xmlns:commonbnd=\"commonbnd.xmi\" xmlns:common=\"common.xmi\" xmi:id=\"WebAppBinding\" virtualHostName=\"default_host\">\n" +
                        "    <webapp href=\"WEB-INF/web.xml#WebApp\"/>\n" +
                        "</webappbnd:WebAppBinding>";
            }
            
            // write stub for IBM's web extensions descriptor
            if (fileName.endsWith("ibm-web-ext.xmi")) {                // NOI18N
                output =
                        "<webappext:WebAppExtension xmi:version=\"2.0\" xmlns:xmi=\"http://www.omg.org/XMI\" xmlns:webappext=\"webappext.xmi\" xmlns:webapplication=\"webapplication.xmi\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xmi:id=\"WebAppExtension\" reloadInterval=\"3\" reloadingEnabled=\"true\" additionalClassPath=\"\" fileServingEnabled=\"true\" directoryBrowsingEnabled=\"false\" serveServletsByClassnameEnabled=\"true\">" + // NOI18N
                        "    <webApp href=\"WEB-INF/web.xml#WebApp\"/>\n" + // NOI18N
                        "</webappext:WebAppExtension>"; // NOI18N
            }
            
            // write the output
            try {
                PrintWriter writer =
                        new PrintWriter(new FileOutputStream(files[i]));
                writer.write(output);
                writer.close();
            } catch (FileNotFoundException e) {
                Logger.getLogger("global").log(Level.WARNING, null, e);
            }
        }
    }
    
    protected void writefile(final File file, final BaseBean bean) throws ConfigurationException {
        try {
            FileObject cfolder = FileUtil.toFileObject(file.getParentFile());
            if (cfolder == null) {
                File parentFile = file.getParentFile();
                try {
                    cfolder = FileUtil.toFileObject(parentFile.getParentFile()).createFolder(parentFile.getName());
                } catch (IOException ioe) {
                    throw new ConfigurationException(NbBundle.getMessage(WSDeploymentConfiguration.class, "MSG_FailedToCreateConfigFolder", parentFile.getAbsolutePath()));
                }
            }
            final FileObject folder = cfolder;
            FileSystem fs = folder.getFileSystem();
            fs.runAtomicAction(new FileSystem.AtomicAction() {
                public void run() throws IOException {
                    OutputStream os = null;
                    FileLock lock = null;
                    try {
                        String name = file.getName();
                        FileObject configFO = folder.getFileObject(name);
                        if (configFO == null) {
                            configFO = folder.createData(name);
                        }
                        lock = configFO.lock();
                        os = new BufferedOutputStream(configFO.getOutputStream(lock), 4086);
                        // TODO notification needed
                        if (bean != null) {
                            bean.write(os);
                        }
                    } finally {
                        if (os != null) {
                            try { os.close(); } catch(IOException ioe) {}
                        }
                        if (lock != null)
                            lock.releaseLock();
                    }
                }
            });
        } catch (IOException e) {
            throw new ConfigurationException(e.getLocalizedMessage());
        }
    }
}