/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.scripting.php.dbginterface;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.Socket;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.swing.SwingUtilities;
import org.netbeans.api.debugger.Breakpoint;
import org.netbeans.api.debugger.DebuggerManager;
import org.netbeans.modules.scripting.php.dbginterface.api.DbgSourceMap;
import org.netbeans.modules.scripting.php.dbginterface.breakpoints.PhpBreakpoint;
import org.netbeans.modules.scripting.php.dbginterface.models.DebugError;
import org.netbeans.modules.scripting.php.dbginterface.models.DebugFrame;
import org.netbeans.modules.scripting.php.dbginterface.models.Variable;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.text.Line;
import org.openide.util.Utilities;

/**
 *
 * @author marcow
 */
public class DbgServerHandler extends DbgDebuggerImpl.Context implements Runnable {
    private Socket socket;
    private InputStream in;
    private int flags;
    private boolean stop;
    private List<DbgPacket> packetsToSend = new ArrayList<DbgPacket>();
    private Thread handlerThread;
    private String sessionId;
    private DbgSourceMap sourceMap;
    private DebuggerAnnotation currentLineAnnotation;
    private Line currentLine;
    private Set<DebuggerAnnotation> errorAnnotations = new HashSet<DebuggerAnnotation>();
    private boolean suspended;
    private List<DebugFrame> callStack;
    // modNo -> FileName
    private Map<Integer, String> sourceFileMap = new HashMap<Integer, String>();
    
    public DbgServerHandler(DbgDebuggerImpl impl, Socket socket) {
        super(impl);
        sourceMap = impl.getSourceMap();
        this.socket = socket;
        stop = false;
        
        try {
            in = new BufferedInputStream(socket.getInputStream());
        }
        catch (IOException ioe) {
            ErrorManager.getDefault().notify(ErrorManager.WARNING, ioe);
        }
    }
    
    public void sendPacket(DbgPacket packet) {
        if (handlerThread == null) {
            return;
        }
        
        synchronized (packetsToSend) {
            packetsToSend.add(packet);
        }
        
        handlerThread.interrupt();
    }
    
    public void stop() {
        Thread.dumpStack();
        synchronized (this) {
            stop = true;
        
            if (handlerThread != null) {
                handlerThread.interrupt();
            }
        }
    }
    
    public void run() {
        System.err.println("mw DbgServerHandler.run() started!");
    
        synchronized (this) {
            handlerThread = Thread.currentThread();
        }
        
        while (!stop && !socket.isClosed()) {
            try {
                List<DbgPacket> toSend = null;
                
                synchronized (packetsToSend) {
                    if (packetsToSend != null && packetsToSend.size() > 0) {
                        toSend = new ArrayList<DbgPacket>(packetsToSend);
                        packetsToSend.clear();
                    }
                }
                
                if (toSend != null) {
                    for (DbgPacket p : toSend) {
                        if (p.isWaitAck()) {
                            // Are there packets from the debuggee to handle first?
                            handleIncomingPackets(false);
                        }
                        
                        p.send(socket.getOutputStream());
                        
                        if (p.isWaitAck()) {
                            // Wait for the acknowledgement packet
                            handleIncomingPackets(true);
                        }
                    }
                }

                handleIncomingPackets(false);
            }
            catch (IOException ioe) {
                ErrorManager.getDefault().notify(ErrorManager.WARNING, ioe);
            }
            
            try {
                // Wake up every 100 milliseconds and see if the debuggee has something to say.
                // The IDE side can interrupt the sleep to send new packets to the
                // debuggee.
                Thread.sleep(100); 
            }
            catch (InterruptedException ie) {
                // OK, run the look again.
            }
        }
        
        if (!socket.isClosed()) {
            try {
                socket.close();
            } catch (IOException ioe) {
                ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, ioe);
            }
        }
        
        synchronized (this) {
            handlerThread = null;
        }
        
        callStack = null;
        getServer().getCallStackModel().refresh(true);
        getServer().getVariablesModel().clearModel();
        hideCurrentLine();
        
        for (DebuggerAnnotation a : errorAnnotations) {
            a.detach();
        }
    }
    
    public boolean isSuspended() {
        return suspended;
    }

    public void setSuspended(boolean s) {
        suspended = s;
        
        getServer().setEnableContActions(s);
    }
    
    public void setCurrentLine(Line line) {
        currentLine =line;
    }
    
    public void finish() {
        getServer().closeScriptContext(this);
    }
    
    public DbgSourceMap getSourceMap() {
        return sourceMap;
    }
    
    public void addError(DebugError e) {
        String aType = null;
        if (e.isError()) {
            aType = DebuggerAnnotation.ERROR_ANNOTATION_TYPE;
        }
        else if (e.isNotice()) {
            aType = DebuggerAnnotation.NOTICE_ANNOTATION_TYPE;
        }
        else if (e.isWarning()) {
            aType = DebuggerAnnotation.WARNING_ANNOTATION_TYPE;
        }
        
        errorAnnotations.add(new DebuggerAnnotation(aType, e.getLine(), e.getString()));
    }
    
    public void hideCurrentLine() {
        if (currentLineAnnotation != null) {
            currentLineAnnotation.detach();
            currentLineAnnotation = null;
        }
    }
    
    public void handleCurrentLine() {
        if (currentLineAnnotation != null) {
            currentLineAnnotation.detach();
        }

        if (currentLine != null) {
            final Line line = currentLine;

            currentLineAnnotation = new DebuggerAnnotation(DebuggerAnnotation.CURRENT_LINE_ANNOTATION_TYPE,
                    line);

            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    line.show(Line.SHOW_GOTO);
                }
            });
        }
    }
    
    public void setCallStack(List<DebugFrame> list) {
        callStack = list;
        getServer().getCallStackModel().setNeedsRefresh();
        getServer().getVariablesModel().setStackFrame(callStack.get(0));
        getServer().getWatchesModel().setStackFrame(callStack.get(0));
    }
    
    public List<DebugFrame> getCallStack() {
        return callStack;
    }

    public void addSourceFileInfo(int modNo, String filename) {
        sourceFileMap.put(new Integer(modNo), filename);
    }
    
    public String getSourceFileInfo(int modNo) {
        return sourceFileMap.get(new Integer(modNo));
    }
    
    public Variable getScopeVariables(DebugFrame frame) {
        // Initialize the $this if the scope has one.
        // That's done by asking for it?!
        DbgPacket packet = new DbgPacket(DbgConstants.DBGA_REQUEST);
                    
        packet.setFlags(DbgConstants.DBGF_WAITACK);
        DbgPacket.FrameRawdata f = new DbgPacket.FrameRawdata(packet, "$this");
        
        packet.addFrame(f);
        packet.addFrame(new DbgPacket.FrameEval(f.getRawId(), frame.getScope()));
        
        try {
            packet.send(socket.getOutputStream());
                // Wait for the response
            packet = DbgPacket.receive(in);
            packet.handle(this);
        }
        catch (IOException ioe) {
            ErrorManager.getDefault().notify(ErrorManager.WARNING, ioe);
        }
        
        // OK, now get the scope variables;-)
        packet = new DbgPacket(DbgConstants.DBGA_REQUEST);
                    
        packet.setFlags(DbgConstants.DBGF_WAITACK);
        packet.addFrame(new DbgPacket.FrameEval(0, frame.getScope()));
        
        try {
            packet.send(socket.getOutputStream());
                // Wait for the response
            packet = DbgPacket.receive(in);
            packet.handle(this);
        }
        catch (IOException ioe) {
            ErrorManager.getDefault().notify(ErrorManager.WARNING, ioe);
        }
        
        List<Variable> vars = packet.getEvalValue();
        Variable ret = null;
        
        if (vars != null) {
            ret = vars.get(0);
        }
        return ret;
    }

    public void setVariableValue(DebugFrame frame, Variable v, Object value) {
        DbgPacket packet = new DbgPacket(DbgConstants.DBGA_REQUEST);
                    
        packet.setFlags(DbgConstants.DBGF_WAITACK);
        DbgPacket.FrameRawdata f = new DbgPacket.FrameRawdata(packet, v.getDisplayName() +
                "=" + value + ";");
        
        packet.addFrame(f);
        packet.addFrame(new DbgPacket.FrameEval(f.getRawId(), frame.getScope()));
        
        try {
            packet.send(socket.getOutputStream());
                // Wait for the response
            packet = DbgPacket.receive(in);
            packet.handle(this);
        }
        catch (IOException ioe) {
            ErrorManager.getDefault().notify(ErrorManager.WARNING, ioe);
        }

    }
    
    public Variable evaluateExpr(DebugFrame frame, String expr) {
        if (expr == null) {
            return null;
        }
        
        DbgPacket packet = new DbgPacket(DbgConstants.DBGA_REQUEST);
                    
        packet.setFlags(DbgConstants.DBGF_WAITACK);
        DbgPacket.FrameRawdata f = new DbgPacket.FrameRawdata(packet, expr);
        
        packet.addFrame(f);
        packet.addFrame(new DbgPacket.FrameEval(f.getRawId(), frame.getScope()));
        
        try {
            packet.send(socket.getOutputStream());
                // Wait for the response
            packet = DbgPacket.receive(in);
            packet.handle(this);
        }
        catch (IOException ioe) {
            ErrorManager.getDefault().notify(ErrorManager.WARNING, ioe);
        }
        
        List<Variable> vars = packet.getEvalValue();
        Variable ret = null;
        
        if (vars != null) {
            ret = vars.get(0);
        }
        return ret;
    }

    public void setSessionId(String sId) {
        sessionId = sId;
    }
    
    public void setBreakpoints() {
        Breakpoint[] breakpoints = DebuggerManager.getDebuggerManager().getBreakpoints();
        
        for (int i = 0; i < breakpoints.length; i++) {
            System.err.println("mw DbgServerHandler.setBreakpoints() breakpoints[" + i + "]= " + breakpoints[i]);
            
            if (breakpoints[i] instanceof PhpBreakpoint) {
                PhpBreakpoint b = (PhpBreakpoint)breakpoints[i];
                
                Line line = b.getLine();
                FileObject sourceFile = line.getLookup().lookup(FileObject.class);
                String targetFile = sourceMap.mapToServerPath(sourceFile);
                
                if (targetFile != null) {

                    System.err.println("mw addBP! line= " + sourceFile + ":" + (line.getLineNumber() + 1));
                    System.err.println("mw targetFile= " + targetFile);
                    
                    DbgPacket packet = new DbgPacket(DbgConstants.DBGA_REQUEST);
                    
                    packet.setFlags(DbgConstants.DBGF_WAITACK);
            
                    DbgPacket.FrameRawdata mod_name = new DbgPacket.FrameRawdata(packet, targetFile);
                    packet.addFrame(mod_name);
                    DbgPacket.FrameRawdata condition = new DbgPacket.FrameRawdata(packet, b.getCondition());
                    packet.addFrame(condition);
                    
                    packet.addFrame(new DbgPacket.FrameBps(0, line.getLineNumber() + 1, 0, mod_name.getRawId(),
                            condition.getRawId(), b.isTemp(), 0,
                            (b.isEnabled() ? DbgConstants.BPS_ENABLED : DbgConstants.BPS_DISABLED)));
                    
                    try {
                        packet.send(socket.getOutputStream());
                            // Wait for the response
                        packet = DbgPacket.receive(in);
                        DbgPacket.FrameBpl f = (DbgPacket.FrameBpl)packet.getFirstFrame();
                        b.setBpNo(f.getBpNo());
                    }
                    catch (IOException ioe) {
                        ErrorManager.getDefault().notify(ErrorManager.WARNING, ioe);
                    }
                }
           }
        }   
    }
    
    public void updateSourceTree() {
        DbgPacket packet = new DbgPacket(DbgConstants.DBGA_REQUEST);
                    
        packet.setFlags(DbgConstants.DBGF_WAITACK);
            
        DbgPacket.FrameSrcTree f = new DbgPacket.FrameSrcTree();
        packet.addFrame(f);
        
        try {
            packet.send(socket.getOutputStream());
                // Wait for the response
            packet = DbgPacket.receive(in);
            
            packet.handle(this);
        }
        catch (IOException ioe) {
            ErrorManager.getDefault().notify(ErrorManager.WARNING, ioe);
        }
    }
    
    public void resume() {
        callStack = null;
        getServer().getCallStackModel().setNeedsRefresh();
        getServer().getVariablesModel().clearModel();
        
        cont(DbgConstants.DBGA_CONTINUE);
    }

    public void stepInto() {
        cont(DbgConstants.DBGA_STEPINTO);
    }
    
    public void stepOut() {
        cont(DbgConstants.DBGA_STEPOUT);
    }
    
    public void stepOver() {
        cont(DbgConstants.DBGA_STEPOVER);
    }
    
    private void cont(int step) {
        currentLine = null;
        hideCurrentLine();

        System.err.println("mw DbgServerHandler.cont(" + step + ")");
        
        sendPacket(new DbgPacket(step));
        setSuspended(false);
    }
    
    private void handleIncomingPackets(boolean block) throws IOException {
        while (block || in.available() > 0) {
            DbgPacket packet = DbgPacket.receive(in);
        
            block = false;
            packet.handle(this);
        }
    }
    
    public DebuggerAnnotation getCurrentLineAnnotation() {
        return currentLineAnnotation;
    }
    
    public void setCurrentLineAnnotation(DebuggerAnnotation annotation) {
        this.currentLineAnnotation = annotation;
    }
/*
                packet = new DbgPacket(DbgConstants.DBGA_REQUEST);
            packet.addFrame(new DbgPacket.FrameVer());
            packet.setFlags(flags | DbgConstants.DBGF_WAITACK);
            System.err.println("mw packet to send= " + packet);
            
            packet.send(socket.getOutputStream());

            packet = DbgPacket.receive(in);
            
            System.err.println("mw packet returned= " + packet);

            packet = new DbgPacket(DbgConstants.DBGA_REQUEST);
            packet.setFlags(flags | DbgConstants.DBGF_WAITACK);
            packet.addFrame(new DbgPacket.FrameSrcTree());
           
            System.err.println("mw packet to send= " + packet);
            packet.send(socket.getOutputStream());

            packet = DbgPacket.receive(in);
            
            System.err.println("mw packet returned= " + packet);
            
            packet = new DbgPacket(DbgConstants.DBGA_REQUEST);
            packet.setFlags(flags | DbgConstants.DBGF_WAITACK);
            packet.addFrame(new DbgPacket.FrameSrcLinesInfo(1));
           
            System.err.println("mw packet to send= " + packet);
            packet.send(socket.getOutputStream());

            packet = DbgPacket.receive(in);
            
            System.err.println("mw packet returned= " + packet);

            packet = new DbgPacket(DbgConstants.DBGA_REQUEST);
            packet.setFlags(flags | DbgConstants.DBGF_WAITACK);
            packet.addFrame(new DbgPacket.FrameSrcCtxInfo(1));
           
            System.err.println("mw packet to send= " + packet);
            packet.send(socket.getOutputStream());

            packet = DbgPacket.receive(in);
            
            System.err.println("mw packet returned= " + packet);
            
            packet = new DbgPacket(DbgConstants.DBGA_REQUEST);
            packet.setFlags(flags | DbgConstants.DBGF_WAITACK);
            packet.addFrame(new DbgPacket.FrameSource(1, 10));
           
            System.err.println("mw packet to send= " + packet);
            packet.send(socket.getOutputStream());

            packet = DbgPacket.receive(in);
            
            System.err.println("mw packet returned= " + packet);

            packet = new DbgPacket(DbgConstants.DBGA_CONTINUE);
            System.err.println("mw packet to send= " + packet);
            packet.send(socket.getOutputStream());
            
            try {
                Thread.sleep(1000);
            }
            catch (InterruptedException ie) {
                
            }
            
            while (in.available() > 0) {
                packet = DbgPacket.receive(in);
            
                System.err.println("mw packet returned= " + packet);
            }
            
            packet = new DbgPacket(DbgConstants.DBGA_REQUEST);
            packet.setFlags(flags | DbgConstants.DBGF_WAITACK);
            packet.addFrame(new DbgPacket.FrameStack());
            
            System.err.println("mw packet to send= " + packet);
            packet.send(socket.getOutputStream());

            packet = DbgPacket.receive(in);
            
            System.err.println("mw packet returned= " + packet);

            packet = new DbgPacket(DbgConstants.DBGA_REQUEST);
            packet.setFlags(flags | DbgConstants.DBGF_WAITACK);
            DbgPacket.FrameRawdata f = new DbgPacket.FrameRawdata(packet, "$i");
            packet.addFrame(f);
            packet.addFrame(new DbgPacket.FrameEval(f.getRawId(), 1));
           
            System.err.println("mw packet to send= " + packet);
            packet.send(socket.getOutputStream());

            packet = DbgPacket.receive(in);
            
            System.err.println("mw packet returned= " + packet);
            
            packet = new DbgPacket(DbgConstants.DBGA_REQUEST);
            packet.setFlags(flags | DbgConstants.DBGF_WAITACK);
            f = new DbgPacket.FrameRawdata(packet, "$GLOBALS");
            packet.addFrame(f);
            packet.addFrame(new DbgPacket.FrameEval(f.getRawId(), 1));
           
            System.err.println("mw packet to send= " + packet);
            packet.send(socket.getOutputStream());

            packet = DbgPacket.receive(in);
            
            System.err.println("mw packet returned= " + packet);
            
            packet = new DbgPacket(DbgConstants.DBGA_REQUEST);
            packet.setFlags(flags | DbgConstants.DBGF_WAITACK);
            f = new DbgPacket.FrameRawdata(packet, "");
            packet.addFrame(f);
            packet.addFrame(new DbgPacket.FrameEval(f.getRawId(), 1));
           
            System.err.println("mw packet to send= " + packet);
            packet.send(socket.getOutputStream());

            packet = DbgPacket.receive(in);
            
            System.err.println("mw packet returned= " + packet);
 */
}
