package it.imolinfo.jbi4ejb.nbm.util;

import it.imolinfo.jbi4ejb.exception.EJBWSDLGenerationException;
import it.imolinfo.jbi4ejb.nbm.*;
import it.imolinfo.jbi4ejb.webservice.generator.WSDLGenerator;
import java.io.File;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;
import org.openide.DialogDisplayer;
import org.openide.NotifyDescriptor;
import org.openide.util.Exceptions;
import org.openide.util.NbBundle;

/**
 * This class provides the utility methods for wsdl generation.
 */
public final class WsdlGeneratorUtil implements Serializable {

    /** logger */
    private static final transient Logger LOG = Logger.getLogger(WsdlGeneratorUtil.class.getName());

    /** Default costructor */
    private WsdlGeneratorUtil() {}

    /**
     * Gets the EJB remote interfaces.
     * This method must:
     * - unzip the ear
     * - tests if a ejb-jar.xml exists and gets all the interfaces
     * - tests if ejb30 annotated classes exist and get all the ejb30interfaces
     */
    public static List getEjbRemoteInterfacesList(File earPath) {
        try {

            LOG.info("looking for ejb remote interfaces in file: " + earPath.getAbsolutePath());

            java.util.List<java.lang.String> remoteInterfaces = new java.util.ArrayList<java.lang.String>();
            remoteInterfaces = it.imolinfo.jbi4ejb.configuration.InterfaceExtractorUtil.extractRemoteInterfacesFromEar(earPath.getAbsolutePath());
            return remoteInterfaces;
        } catch (EJBWSDLGenerationException ex) {
            Exceptions.printStackTrace(ex);
            String msg = NbBundle.getMessage(WsdlGeneratorUtil.class, "MSG_Error", ex.getMessage());            
            LOG.info(msg);               ;
            NotifyDescriptor nd = new NotifyDescriptor.Message(
                                    msg, NotifyDescriptor.ERROR_MESSAGE);
            DialogDisplayer.getDefault().notify(nd);
            return new ArrayList<String>();
        }
    }


    /**
     * Generates the Wsdl
     */
    public static File generateWsdl(WsdlGeneratorContext wsdlContext) {

        java.io.File wsdlFile = null;
        java.lang.String interfaceClassName = wsdlContext.getRemoteInterfaceName();
        java.lang.String earPath = wsdlContext.getEarFile().getAbsolutePath();
        java.lang.String wsdlFileName = wsdlContext.getWsdlFileAbsoluteName();


        java.lang.String connectionType = null;
        if (wsdlContext.getEjbLookupMode().equals(wsdlContext.getEjbLookupMode().CORBANAME)) {
            connectionType = "corbaname";
        } else if (wsdlContext.getEjbLookupMode().equals(wsdlContext.getEjbLookupMode().JNDI)) {
            connectionType = "jndi";
        }
        java.util.Properties ejbLookupProperties = new java.util.Properties();
        java.util.Map<java.lang.String, java.lang.String> props = wsdlContext.getEjbLookupProperties();
        java.util.Iterator propIt = props.entrySet().iterator();
        while (propIt.hasNext()) {
            java.util.Map.Entry pair = (java.util.Map.Entry) propIt.next();
            ejbLookupProperties.put(pair.getKey(), pair.getValue());
        }
        it.imolinfo.jbi4ejb.webservice.generator.WSDLDescriptor wsdlDescription = new it.imolinfo.jbi4ejb.webservice.generator.WSDLDescriptor(wsdlContext.getCorbaName(), connectionType);
        wsdlDescription.setOrbProperties(ejbLookupProperties);

        /*
         * We change the current thread context ClassLoader, because NetBeans set it
         *       as the "system class loader", able to load classes from any
         *       enabled module, as well as the JDK and all startup libraries
         */
        Thread thread = Thread.currentThread();
        ClassLoader origClassLoader = thread.getContextClassLoader();

        try {
            thread.setContextClassLoader(WSDLGenerator.class.getClassLoader());
            wsdlFile = it.imolinfo.jbi4ejb.webservice.generator.WSDLGenerator.createWsdlFromEar(interfaceClassName, earPath, wsdlFileName, wsdlDescription);
        } catch (EJBWSDLGenerationException ex) {
            Exceptions.printStackTrace(ex);            
            String msg = NbBundle.getMessage(WsdlGeneratorUtil.class, "MSG_Error", ex.getMessage());            
            LOG.info(msg);
        } finally {
            thread.setContextClassLoader(origClassLoader);
        }
        return wsdlFile;
    }
}
