/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.sql.framework.highlighter;

/**
 * A SQLToken is a token that is returned by a lexer that is lexing an SQL source file. It
 * has several attributes describing the token: The type of token, the text of the token,
 * the line number on which it occurred, the number of characters into the input at which
 * it started, and similarly, the number of characters into the input at which it ended.
 * <br>
 * 
 * @author Ahimanikya Satapathy
 * @version $Revision: 1.2 $
 */
public class SQLToken extends Token {

    /**
     * a -- to end of line comment.
     */
    public final static int COMMENT_END_OF_LINE = 0xD10;

    /**
     * An column/table qualifier
     */
    public final static int DOT = 0x800;

    /**
     * An error
     */
    public final static int ERROR = 0xF00;

    /**
     * An comment start embedded in an operator
     */
    public final static int ERROR_UNCLOSED_STRING = 0xF03;

    /**
     * A variable, name, or other identifier
     */
    public final static int IDENTIFIER = 0x200;

    /**
     * A string/number literal
     */
    public final static int LITERAL = 0x300;

    /**
     * An operator
     */
    public final static int OPERATOR = 0x500;

    /**
     * An column/table qualifier
     */
    public final static int QUALIFIER = 0x600;
    /**
     * A reserved word (keyword)
     */
    public final static int RESERVED_WORD = 0x100;

    /**
     * An runtime argument
     */
    public final static int RUNTIME_ARGUMENT = 0x700;

    /**
     * A separator
     */
    public final static int SEPARATOR = 0x400;

    /**
     * White space
     */
    public final static int WHITE_SPACE = 0xE00;
    private int charBegin;
    private int charEnd;
    private String contents;

    private int ID;
    private int lineNumber;
    private int state;

    /**
     * Create a new token. The constructor is typically called by the lexer
     * 
     * @param ID the id number of the token
     * @param contents A string representing the text of the token
     * @param lineNumber the line number of the input on which this token started
     * @param charBegin the offset into the input in characters at which this token
     *        started
     * @param charEnd the offset into the input in characters at which this token ended
     */
    public SQLToken(int ID, String contents, int lineNumber, int charBegin, int charEnd) {
        this(ID, contents, lineNumber, charBegin, charEnd, Token.UNDEFINED_STATE);
    }

    /**
     * Create a new token. The constructor is typically called by the lexer
     * 
     * @param ID the id number of the token
     * @param contents A string representing the text of the token
     * @param lineNumber the line number of the input on which this token started
     * @param charBegin the offset into the input in characters at which this token
     *        started
     * @param charEnd the offset into the input in characters at which this token ended
     * @param state the state the tokenizer is in after returning this token.
     */
    public SQLToken(int ID, String contents, int lineNumber, int charBegin, int charEnd, int state) {
        this.ID = ID;
        this.contents = new String(contents);
        this.lineNumber = lineNumber;
        this.charBegin = charBegin;
        this.charEnd = charEnd;
        this.state = state;
    }

    /**
     * get a String that explains the error, if this token is an error.
     * 
     * @return a String that explains the error, if this token is an error, null
     *         otherwise.
     */
    public String errorString() {
        String s;
        if (isError()) {
            s = "Error on line " + lineNumber + ": ";
            switch (ID) {
                case ERROR:
                    s += "Unexpected token: " + contents;
                    break;
                case ERROR_UNCLOSED_STRING:
                    s += "Unclosed string literal: " + contents;
                    break;
            }

        } else {
            s = null;
        }
        return (s);
    }

    /**
     * get the offset into the input in characters at which this token started
     * 
     * @return the offset into the input in characters at which this token started
     */
    public int getCharBegin() {
        return charBegin;
    }

    /**
     * get the offset into the input in characters at which this token ended
     * 
     * @return the offset into the input in characters at which this token ended
     */
    public int getCharEnd() {
        return charEnd;
    }

    /**
     * get the contents of this token
     * 
     * @return A string representing the text of the token
     */
    public String getContents() {
        return (new String(contents));
    }

    /**
     * A description of this token. The description should be appropriate for syntax
     * highlighting. For example "comment" is returned for a comment.
     * 
     * @return a description of this token.
     */
    public String getDescription() {
        if (isReservedWord()) {
            return (StyleHelper.RESERVED_WORD);
        } else if (isIdentifier()) {
            return (StyleHelper.IDENTIFIER);
        } else if (isLiteral()) {
            return (StyleHelper.LITERAL);
        } else if (isSeparator()) {
            return (StyleHelper.SEPARATOR);
        } else if (isDot()) {
            return (StyleHelper.DOT);
        } else if (isQualifier()) {
            return (StyleHelper.QUALIFIER);
        } else if (isRuntimeArgument()) {
            return (StyleHelper.RUNTIME_ARGUMENT);
        } else if (isOperator()) {
            return (StyleHelper.OPERATOR);
        } else if (isComment()) {
            return (StyleHelper.COMMENT);
        } else if (isWhiteSpace()) {
            return (StyleHelper.WHITESPACE);
        } else if (isError()) {
            return (StyleHelper.ERROR);
        } else {
            return ("unknown");
        }
    }

    /**
     * get the ID number of this token
     * 
     * @return the id number of the token
     */
    public int getID() {
        return ID;
    }

    /**
     * get the line number of the input on which this token started
     * 
     * @return the line number of the input on which this token started
     */
    public int getLineNumber() {
        return lineNumber;
    }

    /**
     * Get an integer representing the state the tokenizer is in after returning this
     * token. Those who are interested in incremental tokenizing for performance reasons
     * will want to use this method to figure out where the tokenizer may be restarted.
     * The tokenizer starts in Token.INITIAL_STATE, so any time that it reports that it
     * has returned to this state, the tokenizer may be restarted from there.
     */
    public int getState() {
        return state;
    }

    /**
     * Checks this token to see if it is a comment.
     * 
     * @return true if this token is a comment, false otherwise
     */
    public boolean isComment() {
        return ((ID >> 8) == 0xD);
    }

    /**
     * Checks this token to see if it is a DOT.
     * 
     * @return true if this token is a Dot, false otherwise
     */
    public boolean isDot() {
        return ((ID >> 8) == 0x8);
    }

    /**
     * Checks this token to see if it is an Error. Unfinished comments, numbers that are
     * too big, unclosed strings, etc.
     * 
     * @return true if this token is an Error, false otherwise
     */
    public boolean isError() {
        return ((ID >> 8) == 0xF);
    }

    /**
     * Checks this token to see if it is an identifier.
     * 
     * @return true if this token is an identifier, false otherwise
     */
    public boolean isIdentifier() {
        return ((ID >> 8) == 0x2);
    }

    /**
     * Checks this token to see if it is a literal.
     * 
     * @return true if this token is a literal, false otherwise
     */
    public boolean isLiteral() {
        return ((ID >> 8) == 0x3);
    }

    /**
     * Checks this token to see if it is a Operator.
     * 
     * @return true if this token is a Operator, false otherwise
     */
    public boolean isOperator() {
        return ((ID >> 8) == 0x5);
    }

    /**
     * Checks this token to see if it is a Qualifier.
     * 
     * @return true if this token is a Qualifier, false otherwise
     */
    public boolean isQualifier() {
        return ((ID >> 8) == 0x6);
    }

    /**
     * Checks this token to see if it is a reserved word.
     * 
     * @return true if this token is a reserved word, false otherwise
     */
    public boolean isReservedWord() {
        return ((ID >> 8) == 0x1);
    }

    /**
     * Checks this token to see if it is a RuntimeArgument.
     * 
     * @return true if this token is a RuntimeArgument, false otherwise
     */
    public boolean isRuntimeArgument() {
        return ((ID >> 8) == 0x7);
    }

    /**
     * Checks this token to see if it is a Separator.
     * 
     * @return true if this token is a Separator, false otherwise
     */
    public boolean isSeparator() {
        return ((ID >> 8) == 0x4);
    }

    /**
     * Checks this token to see if it is White Space. Usually tabs, line breaks, form
     * feed, spaces, etc.
     * 
     * @return true if this token is White Space, false otherwise
     */
    public boolean isWhiteSpace() {
        return ((ID >> 8) == 0xE);
    }

    /**
     * get a representation of this token as a human readable string. The format of this
     * string is subject to change and should only be used for debugging purposes.
     * 
     * @return a string representation of this token
     */
    public String toString() {
        return ("Token #" + Integer.toHexString(ID) + ": " + getDescription() + " Line " + lineNumber + " from " + charBegin + " to " + charEnd
            + " : " + contents);
    }

}

