/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.bpel.model.validation.references;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.netbeans.modules.bpel.model.api.BpelContainer;
import org.netbeans.modules.bpel.model.api.BpelEntity;
import org.netbeans.modules.bpel.model.api.BpelModel;
import org.netbeans.modules.bpel.model.api.ContentElement;
import org.netbeans.modules.bpel.model.api.Process;
import org.netbeans.modules.bpel.model.api.Variable;
import org.netbeans.modules.bpel.model.api.VariableDeclaration;
import org.netbeans.modules.bpel.model.api.VariableDeclarationScope;
import org.netbeans.modules.bpel.model.api.references.BpelReference;
import org.netbeans.modules.bpel.model.api.references.ReferenceCollection;
import org.netbeans.modules.bpel.model.api.references.SchemaReference;
import org.netbeans.modules.bpel.model.api.references.WSDLReference;
import org.netbeans.modules.bpel.model.impl.references.MappedReference;
import org.netbeans.modules.bpel.model.impl.services.ExpressionUpdater;
import org.netbeans.modules.bpel.model.validation.AbstractValidator;
import org.netbeans.modules.xml.xam.Component;
import org.netbeans.modules.xml.xam.Model;
import org.netbeans.modules.xml.xam.Reference;
import org.netbeans.modules.xml.xam.dom.Attribute;
import org.netbeans.modules.xml.xam.spi.Validation;
import org.netbeans.modules.xml.xam.spi.ValidationResult;
import org.netbeans.modules.xml.xam.spi.Validator;
import org.netbeans.modules.xml.xam.spi.Validation.ValidationType;
import org.openide.util.NbBundle;
/**
 * Broken references validator.
 * @author ads
 */
public class ReferencesValidator implements Validator {
    
    
    
    private static final String FIX_VARIABLES = "FIX_Variables";    // NOI18N
    
    private static final String FIX_VARIABLE = "FIX_Variable";      // NOI18N
    
    private static final String FIX_CORRECTION_EXTERNAL =
            "FIX_Correction_External";                                  // NOI18N
    
    private static final String XSD = "xsd";                        // NOI18N
    
    private static final String WSDL = "WSDL";                      // NOI18N
    
    private static final String FIX_REFERENCE_EXTERNAL =
            "FIX_Reference_External";                                   // NOI18N
    
    private static final String FIX_CORRECTION = "FIX_Correction";  // NOI18N
    
    private static final String FIX_UNKNOWN = "FIX_Unknown";        // NOI18N
    
    private static final String FIX_REFERENCE = "FIX_Reference";    // NOI18N
    
    /** {@inheritDoc} */
    public String getName() {
        return getClass().getName();
    }
    
    /* (non-Javadoc)
     * @see org.netbeans.modules.xml.xam.spi.Validator#validate(org.netbeans.modules.xml.xam.Model, org.netbeans.modules.xml.xam.spi.Validation, org.netbeans.modules.xml.xam.spi.Validation.ValidationType)
     */
    /** {@inheritDoc} */
    public ValidationResult validate( Model model, Validation validation,
            ValidationType validationType ) {
        if(!(model instanceof BpelModel)) {
            return AbstractValidator.EMPTY_RESULT;
        }
        final BpelModel bpelModel = (BpelModel) model;
        

        if (bpelModel.getState() == Model.State.NOT_WELL_FORMED) {
            return AbstractValidator.EMPTY_RESULT;
        }

        // Initialize our result object
        final ArrayList<Set<ResultItem>> collection = 
            new ArrayList<Set<ResultItem>>(1);
        Set<Model> models = Collections.singleton((Model)bpelModel);
        
        Runnable run = new Runnable() {

            public void run() {
                collection.add( getResults(bpelModel) );
            }
            
        };

        bpelModel.invoke( run );
        Set<ResultItem> results = collection.get(0);
        
        return new ValidationResult(results, models);

    }
    
    
    private Set<ResultItem> getResults( BpelModel model ) {
        Set<ResultItem> result = new HashSet<ResultItem>();
        
        Process process = model.getProcess();
        collectResults( process , result );
        return result;
    }
    
    
    private void collectResults( BpelEntity entity, Set<ResultItem> result ) {
        checkReferenceCollection(entity, result);
        
        checkExpressions( entity , result );
        
        List<BpelEntity> children = entity.getChildren();
        for (BpelEntity child : children) {
            collectResults( child , result );
        }
    }
    
    private void checkExpressions( BpelEntity entity, Set<ResultItem> result ) {
        if ( entity instanceof ContentElement ){
            String expression = ((ContentElement) entity).getContent();
            Collection<String> collection = ExpressionUpdater.getInstance().
                    getUsedVariables( expression );
            Set<String> set = new HashSet<String>( collection );
            findDeclarationsAscendant( entity , set );
            if ( set.size() >0 ){
                StringBuilder builder = new StringBuilder();
                for (String string : set) {
                    builder.append( string );
                    builder.append(", ");                                   // NOI18N
                }
                String str ;
                if ( set.size() >1 ){
                    str = NbBundle.getMessage(getClass(), FIX_VARIABLES );
                } else {
                    str = NbBundle.getMessage(getClass(), FIX_VARIABLE );
                }
                str = MessageFormat.format( str, builder.substring( 0,
                        builder.length()-2 ) , expression.trim() );
                ResultItem resultItem = new ResultItem( this,
                        ResultType.ERROR,
                        (Component)entity,
                        str );
                result.add( resultItem );
            }
        }
    }
    
    private void findDeclarationsAscendant( BpelEntity entity, Set<String> set ) {
        if ( set.size() == 0 ){
            return;
        }
        if (entity instanceof VariableDeclarationScope) {
            findDeclarationsDescendant(entity, set);
        }
        // Fix for #81027 - we need to proceed up to root event if current entity is VariableDeclarationScope
        BpelContainer parent = entity.getParent();
        if (parent != null) {
            findDeclarationsAscendant(parent, set);
        }
    }
    
    private void findDeclarationsDescendant( BpelEntity entity, Set<String> set ) {
        if ( entity instanceof VariableDeclaration ){
            String name = ((VariableDeclaration)entity).getVariableName();
            set.remove( name );
        }
        
        List<Variable> list = entity.getChildren( Variable.class );
        if ( list != null ) {
            for (Variable variable : list) {
                String name = variable.getVariableName();
                set.remove( name );
            }
        }
        
        if ( entity instanceof VariableDeclarationScope ){
            List<VariableDeclarationScope> scopes =
                    entity.getChildren(VariableDeclarationScope.class);
            if ( scopes == null ){
                return;
            }
            for (VariableDeclarationScope scope : scopes) {
                findDeclarationsDescendant( scope , set );
            }
        }
    }
    
    private void checkReferenceCollection( BpelEntity entity,
            Set<ResultItem> result ) {
        if ( entity instanceof ReferenceCollection ){
            ReferenceCollection collection = (ReferenceCollection) entity ;
            Reference[] refs = collection.getReferences();
            for (Reference reference : refs) {
                if ( reference == null ){
                    continue;
                }
                if ( reference.isBroken() ){
                    String str = getMessage( entity , reference );
                    ResultItem resultItem = new ResultItem( this,
                            ResultType.ERROR,
                            (Component)entity,
                            str );
                    result.add( resultItem );
                }
            }
        }
    }
    
    /**
     * @return
     */
    private String getMessage( BpelEntity entity , Reference ref ) {
        String str = null;
        String tagName = entity.getPeer().getLocalName();
        Attribute attr = null;
        if ( ref instanceof MappedReference) {
            attr = ((MappedReference) ref).getAttribute();
        } else {
            /* Model should not have references that is not MappedReferences.
             * we need to use this interface for determening attribute name.
             * This is imlementation issue.
             */
            str = NbBundle.getMessage(getClass(), FIX_UNKNOWN );
            str = MessageFormat.format( str, tagName );
            assert false;
            return str;
        }
        
        if ( ref instanceof BpelReference ){
            str = NbBundle.getMessage(getClass(), FIX_REFERENCE );
            str = MessageFormat.format( str, tagName , attr.getName() ) +
                    NbBundle.getMessage(getClass(),FIX_CORRECTION);
        } else if ( ref instanceof WSDLReference ){
            str = NbBundle.getMessage(getClass(), FIX_REFERENCE_EXTERNAL );
            str = MessageFormat.format( str, tagName , attr.getName() , WSDL ) +
                    NbBundle.getMessage(getClass(),FIX_CORRECTION_EXTERNAL);
        } else if ( ref instanceof SchemaReference ){
            str = NbBundle.getMessage(getClass(), FIX_REFERENCE_EXTERNAL );
            str = MessageFormat.format( str, tagName , attr.getName() , XSD ) +
                    NbBundle.getMessage(getClass(),FIX_CORRECTION_EXTERNAL);
        }
        return str;
    }
    
}
