/* 
 *   Creation Date: <1999/03/06 13:37:20 samuel>
 *   Time-stamp: <2000/12/31 03:27:17 samuel>
 *   
 *	<pci.c>
 *	
 *   PCI top-level driver (used by bandit, chaos etc)
 *   
 *   Copyright (C) 1999, 2000 Samuel Rydh (samuel@ibrium.se)
 *   
 *   pci_device_loc() was taken from pmac_pci.c:
 *   Copyright (C) 1997 Paul Mackerras (paulus@cs.anu.edu.au)
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"

/* #define VERBOSE */

#include <linux/pci.h>

#include "verbose.h"
#include "promif.h"
#include "ioports.h"
#include "debugger.h"
#include "pci.h"
#include "res_manager.h"
#include "driver_mgr.h"

SET_VERBOSE_NAME("PCI")

typedef struct pci_device pci_device_t;
typedef struct basereg basereg_t;
typedef struct pci_bus pci_bus_t;

static void pci_cleanup( void );

static void do_write_config( struct pci_device *dev, int offs, unsigned char val );
static unsigned char do_read_config( struct pci_device *dev, int offs);

static void update_base_registers( struct pci_device *dev );
static void update_base_register( struct pci_device *dev, int reg );
static void set_base( pci_device_t *dev, int reg, ulong val );

static void map_rom( pci_device_t *dev, ulong base, int map );
static ulong make_mask( ulong size );
static ulong rom_read( ulong addr, int len, void *usr );
static void  rom_write( ulong addr, ulong data, int len, void *usr );

static inline struct pci_bus *find_bus( int bus_id );

driver_interface_t pci_driver = {
	"pci", 
	pci_init, 
	pci_cleanup
};


struct pci_bus
{
	int 			bus_id;		/* id of this bus */
	struct pci_device	*first_dev;

	struct pci_bus 		*next;
};

struct basereg
{
	ulong	r;		/* B.E. register value */
	ulong	old_r;		/* value used in last mapping */
	ulong	mask;		/* decoded bits (e.g. 0xffff0000) */
	int	flags;
};

/* flags field in basereg */
enum {
	bf_mapped=1,		/* register is currently mapped */
};


struct pci_device
{
	int		dev_fn;		/* device/function address */
	int		bus;		/* for verbose info */

	void		*usr;		/* user private pointer */

	int		config_len;	/* length of config_data (>=PCI_BASE_ADDRESS_0) */
	char		*config_data;	/* config data (VENDOR etc) */

	char		*rombuf;
	int		rom_map_id;
	int		rom_fd;
	ulong		rom_size;
	ulong		mapped_rombase;

	/* read/write registers (config space) */
	short		cmd;		/* B.E. cmd register */
	basereg_t	base[6];	/* B.E. base register */
	basereg_t	rombase;	/* B.E. rom base register */

	/* procs */
	pci_map_base_fp	map_proc[6];

	struct pci_device 	*next;	
};

static io_ops_t rom_ops={
	NULL,
	rom_read,
	rom_write, 
	NULL
};


static struct pci_bus *all_buses;


/************************************************************************/
/*	FUNCTIONS							*/
/************************************************************************/


int
pci_init( void )
{
	VPRINT("Initing PCI bus services\n");
	all_buses = NULL;
	
	pci_bridges_init();
	
	return 1;
}

static void
pci_cleanup( void )
{
	pci_bus_t *bt, *next;

	pci_bridges_cleanup();

	for( bt=all_buses; bt; bt=next ){
		pci_device_t *dev, *next_dev;

		for( dev = bt->first_dev; dev; dev=next_dev ) {
			next_dev = dev->next;
			if( dev->rom_fd != -1 )
				close( dev->rom_fd );
			if( dev->rombuf != NULL )
				free( dev->rombuf );
			if( dev->rom_map_id )
				remove_io_range( dev->rom_map_id );
			free( dev );
		}
		next = bt->next;
		free( bt );
	}
	all_buses = NULL;
}

static inline pci_bus_t *
find_bus( int bus_id )
{
	pci_bus_t *bt;
	for( bt = all_buses; bt; bt=bt->next )
		if( bt->bus_id == bus_id )
			return bt;
	return NULL;
}

static inline pci_device_t *
find_device( int bus_id, int dev_fn )
{
	pci_bus_t 	*bt;
	pci_device_t	*dev;

	if( (bt=find_bus(bus_id))==NULL )
		return NULL;
	for( dev = bt->first_dev ; dev; dev=dev->next )
		if( dev->dev_fn == dev_fn )
			return dev;
	return NULL;
}	

int
add_pci_bus( int bus_id )
{
	pci_bus_t *bt;

	if( find_bus( bus_id ) != NULL ) {
		printm("add_pci_bus: Bus already exists!\n");
		return -1;
	}
	bt = calloc( sizeof(struct pci_bus), 1 );
	bt->next = all_buses;
	all_buses = bt;

	bt->bus_id = bus_id;
	return 0;
}

int
add_pci_device( int bus_id, int dev_fn, pci_dev_info_t *info, void *usr )
{
	pci_bus_t 	*bt;
	pci_device_t 	*dev;
	char		*p;
	
	VPRINT("Adding pci device (%d:%02x)\n", bus_id, dev_fn );
	
	if( (bt = find_bus( bus_id )) == NULL ) {
		printm("add_pci_device: no such bus (%d)\n", bus_id );
		return -1;
	}
	if( (dev=find_device( bus_id, dev_fn )) != NULL ) {
		printm("add_pci_device: device already exists (%d:%02X)\n",bus_id,dev_fn );
		return -1;
	}

	dev = calloc( sizeof(struct pci_device),1 );

	/* by default, allocate only first part of page */ 
	dev->config_len = PCI_BASE_ADDRESS_0;
	p = dev->config_data = calloc( dev->config_len, 1 );

	st_le16( (short*)(p + PCI_VENDOR_ID), info->vendor_id );
	st_le16( (short*)(p + PCI_DEVICE_ID), info->device_id );
	st_le16( (short*)(p + PCI_CLASS_DEVICE), info->device_class );
	*(p + PCI_REVISION_ID) = info->revision;
	*(p + PCI_CACHE_LINE_SIZE) = 0x08;
	*(p + PCI_LATENCY_TIMER) = 0x20;

	/* Auto-detect multi-function cards and set the header bit */
	if( (dev_fn & 7) ) {
		pci_device_t *dev2 = find_device( bus_id, dev_fn & ~7 );
		if( dev2 )
			dev2->config_data[ PCI_HEADER_TYPE ] |= 0x80;
		VPRINT("Adding mutli-function PCI card\n");
	} else {
		int i;
		for(i=1; i<7; i++) {
			if( find_device( bus_id, dev_fn + i ) ) {
				dev->config_data[ PCI_HEADER_TYPE ] |= 0x80;
				break;
			}
		}
	}

	dev->next = bt->first_dev;
	bt->first_dev = dev;

	/* setup device */
	dev->dev_fn = dev_fn;
	dev->bus = bus_id;
	dev->usr = usr;
	dev->rom_fd = -1;
	return 0;
}

void 
set_pci_dev_usr_info( int bus, int dev_fn, void *usr ) 
{
	pci_device_t *dev = find_device( bus, dev_fn );
	if( !dev )
		printm("set_pci_user_info: Bad bus/devfn\n");
	else 
		dev->usr = usr;
}

int 
set_pci_base( int bus, int dev_fn, int ind, ulong bmask, pci_map_base_fp proc )
{
	pci_device_t *dev;

	if( ind < 0 || ind > 5 || (dev=find_device( bus, dev_fn )) == NULL ) {
		LOG("Error in set_pci_base_reg (%d:%d)\n", bus, dev_fn>>3 );
		return 1;
	}
	dev->base[ind].mask = bmask;
	set_base( dev, ind, 0 );
	dev->map_proc[ind] = proc;

	update_base_registers( dev );
	return 0;
}

static void
set_base( pci_device_t *dev, int i, ulong val )
{
	val &= dev->base[i].mask;
	if( (dev->base[i].mask & PCI_BASE_ADDRESS_SPACE) == PCI_BASE_ADDRESS_SPACE_IO ) {
		val &= PCI_BASE_ADDRESS_IO_MASK;
		dev->base[i].r = val | (dev->base[i].mask & ~PCI_BASE_ADDRESS_IO_MASK);
	} else {
		val &= PCI_BASE_ADDRESS_MEM_MASK;
		dev->base[i].r = val | (dev->base[i].mask & ~PCI_BASE_ADDRESS_MEM_MASK);
	}
}


int
use_pci_rom( char *res_name, char *fallback_name, int bus_id, int dev_fn )
{
	pci_device_t *dev;
	int fd;
	char *name = NULL;
	
	if( (dev=find_device( bus_id, dev_fn )) == NULL || dev->rom_fd != -1 ) {
		LOG("Can't find device (%d:%d)\n", bus_id, dev_fn>>3 );
		return 1;
	}
	if( res_name )
		name = get_str_res( res_name );
	if( !name )
		name = fallback_name;
	if( !name ){
		LOG("----> The entry '%s' is missing from the /etc/molrc file!\n",
		    res_name );
		return 1;
	}
	if( (fd = open( name, O_RDONLY )) < 0 ) {
		LOG_ERR("opening '%s'", name);
		return 1;
	}

	dev->rom_fd = fd;
	dev->rom_size = lseek( fd, 0, SEEK_END );

	dev->rombase.mask = make_mask( dev->rom_size );
	dev->rombase.mask |= PCI_ROM_ADDRESS_ENABLE;

	update_base_registers( dev );
	return 0;
}

static void
map_rom( pci_device_t *dev, ulong base, int map )
{
	if( !map ) {
		if( dev->rom_map_id )
			remove_io_range( dev->rom_map_id );
		if( dev->rombuf )
			free( dev->rombuf );
		dev->rombuf = NULL;
		dev->rom_map_id = 0;
		return;
	}
	if( dev->rom_fd == -1 || (dev->rombuf = malloc( dev->rom_size )) == NULL )
		return;

	lseek( dev->rom_fd, 0, SEEK_SET );
	read( dev->rom_fd, dev->rombuf, dev->rom_size );
	
	dev->mapped_rombase = base;
	dev->rom_map_id = add_io_range( base, dev->rom_size, "PCI-ROM", 0, &rom_ops, dev );
}

static ulong 
rom_read( ulong addr, int len, void *udev )
{
	pci_device_t *dev = (pci_device_t*) udev;

	if( addr + len > dev->mapped_rombase + dev->rom_size || addr < dev->mapped_rombase ) {
		LOG("rom_read: read outside ROM (%08lX, %08lX, %08lX)\n", addr, dev->mapped_rombase, dev->rom_size );
		return 0;
	}
	return read_mem( dev->rombuf + addr - dev->mapped_rombase, len );
}

static void 
rom_write( ulong addr, ulong data, int len, void *usr )
{
	LOG("PCI-ROM write\n");
}


/* return 0xffff000 style mask */
static ulong
make_mask( ulong size )
{
	ulong b;

	for( b=(1<<31); b; b=b>>1 )
		if( size & b )
			break;
	if( size & ~b )
		b=b << 1;

	return ~(b-1);
}


void
write_pci_config( int bus, int dev_fn, int offs, ulong val, int len )
{
	pci_device_t 	*dev;
	int 		i;

	if( (dev=find_device( bus, dev_fn )) == NULL )
		return;

	VPRINT(" [%d] write %d:%02X+%02X: %08lX\n",len,bus,dev_fn,offs,val );

	/* For simplicity, we access byte for byte to circumvent
	 * alignment problems.
	 */
	for( i=len-1; i>=0; i--, val=val>>8 )
		do_write_config( dev, offs+i, val & 0xff ); 

	/* Handle value written */
	if( (offs & ~3) != ((offs+len-1)&~3 ))
		LOG("alignment problems!\n");

	/* This function is only called a few times in the boot process
	 * - we can afford ome extra overhead here.
	 */
	update_base_registers( dev );
}

ulong
read_pci_config( int bus, int dev_fn, int offs, int len )
{
	pci_device_t 	*dev;
	ulong		val, i;

	if( (dev=find_device( bus, dev_fn )) == NULL ) {
		/* what should we return according to the standard? */
		return 0xffffffff;
	}

	for( val=0, i=0; i<len; i++ ) {
		val=val<<8;
		val |= do_read_config( dev, offs+i ) & 0xff;
	}
	
	VPRINT(" [%d] read  %d:%02X+%02X: %08lX\n",len,bus,dev_fn,offs,val );
	return val;
}


/************************************************************************/
/*	Config access							*/
/************************************************************************/

#define ALIGN_INS_BYTE( dest, val8, offs ) \
	dest &= ~(0xffL << ((offs)&3)*8); \
	dest |= (val8 & 0xff) << ((offs)&3)*8;

#define ALIGN_READ_BYTE( val, offs ) \
	(((val) >> ((offs)&3)*8) & 0xff)


static void
do_write_config( pci_device_t *dev, int offs, unsigned char val )
{
	
	int 		rr = offs >>2;
	ulong		old;
	
	/* XXX possibly call hook here */

	/* Base registers & rom */
	if( rr >= (PCI_BASE_ADDRESS_0 >> 2) && rr <= (PCI_BASE_ADDRESS_5 >> 2) ) {
		int ind = rr - (PCI_BASE_ADDRESS_0 >> 2);
		old = dev->base[ind].r;

		ALIGN_INS_BYTE( old, val, offs & 3 );
		set_base( dev, ind, old );

	} else if( rr == (PCI_ROM_ADDRESS >> 2) ) {
		old = dev->rombase.r;
		ALIGN_INS_BYTE( old, val, offs & 3 );
		dev->rombase.r = old & dev->rombase.mask;
		dev->rombase.r &= PCI_ROM_ADDRESS_MASK | PCI_ROM_ADDRESS_ENABLE;
	} else if ( (offs & ~1) == PCI_COMMAND ){
		/* Command register */
		ALIGN_INS_BYTE( dev->cmd, val, offs & 1 );
		return;
	}
}

static unsigned char
do_read_config( pci_device_t *dev, int offs )
{
	int 		rr = offs >>2;
	basereg_t 	*bp = NULL;

	/* base registers & rom */
	if( rr >= (PCI_BASE_ADDRESS_0>>2) && rr <= (PCI_BASE_ADDRESS_5>>2))
		bp = &dev->base[rr-(PCI_BASE_ADDRESS_0>>2)];
	else if( rr == (PCI_ROM_ADDRESS>>2) ) {
		bp = &dev->rombase;
	}

	if( bp ){
		return ALIGN_READ_BYTE( bp->r, offs & 3 );
	}

	/* Command register */
	if( (offs & ~1) == PCI_COMMAND )
		return ALIGN_READ_BYTE( dev->cmd, offs & 1 );

	/* Read from static config data */
	if( dev->config_data && offs < dev->config_len )
		return dev->config_data[ offs ];
#if 0
	if( offs == PCI_INTERRUPT_LINE ) {
		printm("PCI_INTERRUPT_LINE\n");
		return 1;
	}
	if( offs == PCI_INTERRUPT_PIN ) {
		printm("PCI_INTERRUPT_PIN\n");
		return 1;
	}
#endif
	return 0;
}


/* 
 * This function sets handles mapping and unmapping of
 * the base registers and the ROM.
 */
static void
update_base_registers( pci_device_t *dev )
{
	basereg_t *rb = &dev->rombase;
	int i, map;
	
	for(i=0; i<6; i++ )
		update_base_register(dev, i );

	/* Map ROM */
	map = ((rb->r & PCI_ROM_ADDRESS_ENABLE) && (dev->cmd & PCI_COMMAND_MEMORY)) ? 1:0;
	if( map && (rb->flags & bf_mapped) && rb->r != rb->old_r ) {
		map_rom( dev, 0, 0 );
		rb->flags &= ~bf_mapped;
	}
	if( map != ((rb->flags & bf_mapped)? 1:0) )
		map_rom( dev, rb->r & PCI_ROM_ADDRESS_MASK, map );
	rb->flags &= ~bf_mapped;
	rb->flags |= map ? bf_mapped : 0;
}

/* reg == -1 for ROM, 0-5 for the registers PCI_BASE_ADDRESS_0..5 */
static void 
update_base_register( pci_device_t *dev, int reg )
{
	basereg_t *br = &dev->base[reg];
	int 	do_map=0;
	ulong 	map_addr=0;
	
	if( (dev->cmd & PCI_COMMAND_IO) && (br->r & PCI_BASE_ADDRESS_SPACE_IO) ){
		/* Map I/O - how is this done? */
		LOG("Don't know how to map I/O space\n");
		do_map = 1;
		map_addr = br->r & PCI_BASE_ADDRESS_IO_MASK;
	} else if( (dev->cmd & PCI_COMMAND_MEMORY) && !(br->r & PCI_BASE_ADDRESS_SPACE_IO) ){
		/* Map MEMORY - if >1MB */
		if( br->r > 0x100000 ) {
			do_map = 1;
			map_addr = br->r & PCI_BASE_ADDRESS_MEM_MASK;
		}
	}

	if( !do_map && (br->flags & bf_mapped) ){
		br->flags &= ~bf_mapped;
		VPRINT("PCI base address unmapped (%d:%02X) %08lX\n", dev->bus, dev->dev_fn,br->r);
		if( dev->map_proc[reg] )
			dev->map_proc[reg]( reg, 0,  0 /*unmap*/, dev->usr );
	}

	if( do_map ){
		/* already mapped? */
		if( (br->flags & bf_mapped) && br->r == br->old_r )
			return;

		br->old_r = br->r;
		br->flags |= bf_mapped;
		
		/* Map */
		VPRINT("PCI base address mapped (%d:%02X) %08lX\n", dev->bus, dev->dev_fn,br->r);
		if( dev->map_proc[reg] )
			dev->map_proc[reg]( reg, map_addr, 1 /*map*/, dev->usr );
	}
}


/************************************************************************/
/*	MISC								*/
/************************************************************************/

int 
pci_device_loc( mol_device_node_t *dev, int *bus_ptr, int *devfn_ptr)
{
        unsigned int *reg;
        int len;

        reg = (unsigned int *) prom_get_property(dev, "reg", &len);
        if (reg == NULL || len < 5 * sizeof(unsigned int)) {
		printm("pci.c: Doesn't look lika a PCI device\n");
                *bus_ptr = 0xff;
                *devfn_ptr = 0xff;
                return -1;
        }
        *bus_ptr = (reg[0] >> 16) & 0xff;
        *devfn_ptr = (reg[0] >> 8) & 0xff;
        return 0;
}


int
pci_get_bus_number( mol_device_node_t *dev, int *first_bus, int *last_bus )
{
	int len;
	ulong *busr;

	busr = (ulong*)prom_get_property( dev, "bus-range", &len );
	if( !busr || len < 2 *sizeof(ulong) ){
		printm("pci.c: Doesn't look lika a PCI bridge\n");
		*first_bus = *last_bus = 0xff;
		return -1;
	}
	*first_bus = busr[0] & 0xff;
	*last_bus = busr[1] & 0xff;
	return 0;
}


/*
 * This function reads the 'assigned-addresses' property from the device tree
 * and write corresponding values to the PCI registers. This is used in the
 * bootx/newworld setting. 
 */
void
pci_assign_addresses( void )
{
	mol_device_node_t *dn;
	int len;

	/* Map in PCI devices */
	for( dn=prom_get_root(); dn ; dn=dn->allnext ){
		ulong *reg, *orgreg;
		int i, bus, devfn, r;
		
		if( !(reg = (ulong*)prom_get_property( dn, "assigned-addresses", &len )) )
			continue;
		if( !len || len % 5*sizeof(int) != 0 )
			continue;

		/* XXX: Is the assumption correct that devfn is constant? */

		/* printm("%-14s: (%ld:%02lx)\n", dn->name, 
			   (reg[0] >> 16) & 0xff, (reg[0] >> 8) & 0xff );  */

		orgreg = reg;
		for( i=0; i < len; i+=5*sizeof(long), reg+=5 ){
			ulong breg;

			bus = (reg[0] >> 16) & 0xff;
			devfn = (reg[0] >> 8) & 0xff;
			r = reg[0] & 0xff;			
			breg = reg[2];

			if( ((reg[0] >> 24) & 3) == 2 ){ 
				/* memory */
				if( r != PCI_ROM_ADDRESS )
					breg &= PCI_BASE_ADDRESS_MEM_MASK;
				else {
					breg &= PCI_ROM_ADDRESS_MASK;
					breg |= PCI_ROM_ADDRESS_ENABLE;
				}
			} else {
				breg |= 1;
			}
			/* non-reloctable? */
			if( r==0 )
				continue;
			write_pci_config( bus, devfn, r, ld_le32(&breg), 4 );
		}

		/* set enable bits */
		reg = orgreg;
		for( i=0; i < len; i+=5*sizeof(long), reg+=5 ) {
			unsigned short cmd;

			bus = (reg[0] >> 16) & 0xff;
			devfn = (reg[0] >> 8) & 0xff;
			r = reg[0] & 0xf;

			/* little endian? */
			cmd = read_pci_config( bus, devfn, PCI_COMMAND, 2 );
			cmd = ld_le16(&cmd);
			cmd |= (((reg[0] >> 24) & 3) == 2) ? PCI_COMMAND_MEMORY : PCI_COMMAND_IO;
			write_pci_config( bus, devfn, PCI_COMMAND, ld_le16(&cmd), 2 );
		}
	}
}
