/* 
 *   MOL adaption of BootX, 11/01/2201
 *
 *   Samuel Rydh, <samuel@ibrium.se>
 *
 */
/*
 * Copyright (c) 2000 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * The contents of this file constitute Original Code as defined in and
 * are subject to the Apple Public Source License Version 1.1 (the
 * "License").  You may not use this file except in compliance with the
 * License.  Please obtain a copy of the License at
 * http://www.apple.com/publicsource and read it before using this file.
 * 
 * This Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */

#include "mol_config.h"
#include "debugger.h"
#include "memory.h"
#include "res_manager.h"

#include "booter.h"
#include "bootx.h"
#include "sl.h"
#include "mac_registers.h"
#include "boot_args.h"
#include "promif.h"
#include "video.h"
#include "../drivers/include/pci.h"

/* Memory allocations
 *
 *	0x4000		kImageAddr		decoded kernel
 *			Kernel allocations (drivers etc)
 *			---
 *			BootX data
 *	0x01400000	kLoadAddr		mach-o image
 */

bootx_globals_t gBootX;				// BootX globals

typedef struct 
{
	int imageLastKernelAddr;		// linux addresses!
	int imageFirstBootXAddr;	
} bootx_private_t;

static bootx_private_t xb;
static int initialized=0;

static void fail_to_boot( char *reason );
static void SetUpBootArgs( void );
static void bootx_startup( void );
static void bootx_cleanup( void );

/************************************************************************/
/*	F U N C T I O N S						*/
/************************************************************************/

void
bootx_init( void )
{
	// Called very early, but after resource manager has initialized
	gPE.booter_startup = bootx_startup;
	gPE.booter_cleanup = bootx_cleanup;
}

void
bootx_startup( void ) 
{
	char *kernelname = get_str_res("macosx_kernel");
	char *drivers = get_str_res("macosx_drivers");
	mol_device_node_t *dn;
	int fd, s;
#if 1
	// REMOVE ME REMOVE ME REMOVE ME REMOVE ME REMOVE ME
	static void MOLHacks(void);
	MOLHacks();
#endif
	
	if( ram.size < 32 * 1024 * 1024 )
		fail_to_boot("At least 32 MB of RAM is needed!\n");
	if( !kernelname )
		fail_to_boot("no macosx_kernel specified\n");
	
	// Prepare for allocations
	xb.imageLastKernelAddr = (ulong)ram.lvbase;
	xb.imageFirstBootXAddr = (ulong)(kLoadAddr + ram.lvbase);
	initialized = 1;

	// Initialize memory map
	if( !(dn = prom_create_node("/chosen/memory-map/")) )
		fail_to_boot("Failed to create /chosen/memory-map\n");
	gBootX.memoryMapPH = prom_dn_to_phandle(dn);

	// Load kernel to kLoadAddr
	if( (fd = open( kernelname, O_RDONLY )) < 0 ) {
		perrorm("open %s", kernelname );
		exit(1);
	}
	if( (s=read( fd, ram.lvbase + kLoadAddr, kImageSize )) < 0 ){
		perrorm("reading kernel");
		close(fd);
		exit(1);
	}
	close(fd);
	printm("MacOS X kernel image '%s' read (%d bytes)\n", kernelname, s );
	DecodeMachO();

	// Load mkext drivers
	if( (fd = open( drivers, O_RDONLY )) < 0 ) {
		perrorm("open %s", drivers );
		exit(1);
	}
	if( (s=read( fd, ram.lvbase + kLoadAddr, kImageSize )) < 0 ){
		perrorm("reading mkext drivers");
		close(fd);
		exit(1);
	}
	close(fd);
	printm("MacOS X mkext driver archive '%s' read (%d bytes)\n", drivers, s );
	if( AddDriverMKext() ) {
		printm("Failed to add drivers package!\n");
		exit(1);
	}

	// Fill in the BootArgs structure
	SetUpBootArgs();

	// Setup register
	mregs->nip = gBootX.kernelEntryPoint;
	mregs->msr = 0x1000;
	mregs->gpr[3] = TO_MAC(gBootX.bootArgsAddr);
	mregs->gpr[4] = kMacOSXSignature;

	// Map in PCI ranges according to the 'assigned-addresses' property
	// pci_assign_addresses();
}


void
bootx_cleanup( void )
{
	if( !initialized )
		return;
	initialized=0;
}



/************************************************************************/
/*	Memory allocations						*/
/************************************************************************/

static void
fail_to_boot( char *reason )
{
	printm("-----> BootX failed due to an error: %s\n", reason ? reason : "Unspecified error" );
	exit(1);
}


/* start is a LV address */
long 
AllocateMemoryRange( char *rangeName, long start, long length )
{
	long *buffer;
	
	// printm("AllocateMemoryRange '%s' @%08lX (len %lx)\n", rangeName, start, length );

	buffer = AllocateBootXMemory(2 * sizeof(long));
	if( !buffer ) 
		return -1;
  
	buffer[0] = TO_MAC(start);
	buffer[1] = length;

	prom_add_property(prom_phandle_to_dn(gBootX.memoryMapPH), rangeName, (char *)buffer, 2 * sizeof(long));
	return 0;
}

void *
AllocateBootXMemory( long size )
{
	long addr = xb.imageFirstBootXAddr - size;

	if( addr < xb.imageLastKernelAddr) 
		fail_to_boot(NULL);
  
	xb.imageFirstBootXAddr = addr;

	// printm("AllocateBootXMemory, %lx bytes @%08lX\n", size ,addr);
	return (void *)addr;
}

long 
AllocateKernelMemory( long size )
{
	long addr = xb.imageLastKernelAddr;
  
	xb.imageLastKernelAddr += (size + 0xFFF) & ~0xFFF;
  
	if( xb.imageLastKernelAddr > xb.imageFirstBootXAddr )
		fail_to_boot(NULL);

	// printm("AllocateKernelMemory %lx @%lx\n",size, addr );
	return addr;
}

static void
SetUpBootArgs(void)
{
	long		sym=0;
	boot_args_ptr	args;
	long		cnt;
	char		*str;
	Boot_Video 	*v;

	// Allocate some memory for the BootArgs.
	gBootX.bootArgsSize = sizeof(boot_args);
	gBootX.bootArgsAddr = AllocateKernelMemory(gBootX.bootArgsSize);
	
	// Add the BootArgs to the memory-map.
	AllocateMemoryRange("BootArgs", gBootX.bootArgsAddr, gBootX.bootArgsSize);
	
	args = (boot_args_ptr)gBootX.bootArgsAddr;
	
	args->Revision = kBootArgsRevision;
	args->Version = kBootArgsVersion;
	args->machineType = 0;
	
	// Copy kernel args from resource
	if( get_str_res("macosx_kernel_args") ) {
		char *dest = args->CommandLine;
		int ind = 0;
		while( (str = get_str_res_ind( "macosx_kernel_args",0, ind++ )) ){
			char ch, *p = str;
			while( str[0] == '-' && (ch=*p++)) {
				switch( ch ) {
				case 'y':
					sym = 1;
					break;
				}
			}
			strcpy( dest, str );
			dest += strlen(str);
			*dest++ = ' ';
		}
		if( sym )
			sprintf(dest, "symtab=%d", TO_MAC(gBootX.symbolTableAddr) );
		else if( args )
			*(--dest)=0;
	} else {
		sym = 1;
		sprintf(args->CommandLine, "-v -y symtab=%d io=65535", TO_MAC(gBootX.symbolTableAddr));
	}

	if( !sym ) 
		gBootX.symbolTableAddr = 0;		

	// Video
	v = &args->Video;
	get_video_info( &v->v_baseAddr, (int*)&v->v_width, (int*)&v->v_height, (int*)&v->v_rowBytes, (int*)&v->v_depth );
	// Set this to 1 to get the spinning cursor (TODO: draw a nice MOL penguin...)
	v->v_display = 0;

	// Memory size
	for (cnt = 0; cnt < kMaxDRAMBanks; cnt++) {
		args->PhysicalDRAM[cnt].base = 0;
		args->PhysicalDRAM[cnt].size = 0;
	}
	args->PhysicalDRAM[0].base = 0;
	args->PhysicalDRAM[0].size = ram.size;
	
	// The video parameters are initialized in bootx_startup()
	
	// Add the DeviceTree to the memory-map.
	// The actuall address and size must be filled in later.
	AllocateMemoryRange("DeviceTree", 0, 0);
	
	FlattenDeviceTree();

	// Fill in the address and size of the device tree.
	gBootX.deviceTreeMMTmp[0] = TO_MAC( gBootX.deviceTreeAddr );
	gBootX.deviceTreeMMTmp[1] = gBootX.deviceTreeSize;
	
	args->deviceTreeP = (void *)TO_MAC( gBootX.deviceTreeAddr );
	args->deviceTreeLength = gBootX.deviceTreeSize;
	args->topOfKernelData = TO_MAC( AllocateKernelMemory(0) );
}



/************************************************************************/
/*	HACKs - These should go away					*/
/************************************************************************/

#include "os_interface.h"

static int	osip_get_adb_key( int sel, int *params );
static void	DiskHack( void );

#define KEY_BUF_SIZE	16
#define KEY_BUF_MASK	15
static int key_bot=0, key_top=0;
static char key_buf[ KEY_BUF_SIZE ];

static void
key_event( unsigned char key )
{
	if( (key_top + 1 - key_bot) & KEY_BUF_MASK ) {
		key_buf[key_top] = key;
		key_top = (key_top + 1) & KEY_BUF_MASK;
	}
}

static void MOLHacks( void )
{
	os_interface_add_proc( OSI_GET_ADB_KEY, osip_get_adb_key );
	gPE.adb_key_event = key_event;

	DiskHack();
}

static int
osip_get_adb_key( int sel, int *params )
{
	int ret=-1;
	if( key_top != key_bot ) {
		ret = key_buf[key_bot];
		key_bot = (key_bot + 1) & KEY_BUF_MASK;
	}
	return ret;
}


/************************************************************************/
/*	Disk HACK							*/
/************************************************************************/

#include "llseek.h"
static int hack_fd=-1;

static int	osipx_blk_read( int sel, int *params );
static int	osipx_blk_write( int sel, int *params );
static int	osipx_blk_size( int sel, int *params );

static void DiskHack( void )
{
	char *volume = get_str_res("blkdev_x");

	if( !volume ) {
		printm("No blkdev_x resource found!\n");
		exit(1);
	}
	if( (hack_fd = open( volume, O_RDWR )) < 0 ) {
		perrorm("Failed to open '%s'!\n", volume);
		exit(1);
	}

	os_interface_add_proc( OSI_X_BLK_READ, osipx_blk_read );
	os_interface_add_proc( OSI_X_BLK_WRITE, osipx_blk_write );
	os_interface_add_proc( OSI_X_BLK_SIZE, osipx_blk_size );
}

// mphys, blk, offs, size
static int 
osipx_blk_read( int sel, int *params )
{
	char *lvptr;

	//printm("osipx_blk_read: %08X %d %d %d\n", params[0], params[1], params[2], params[3] );
	if( mphys_to_lvptr( params[0], &lvptr ) ) {
		printm("osipx_blk_read: Not in RAM access");
		return -1;
	}
	
	if( hack_fd != -1 ) {
		int blk = params[1] + (params[2] >> 9);
		int offs = params[2] & 0x1ff;
		int size = params[3];
		
		// printm("blk %d, offs %d, size %d\n", blk, offs, size );

		if( blk_lseek( hack_fd, blk, offs ) ) {
			printm("XXXXXXXXXXXXX blk_lseek error XXXXXXXXXXXXX\n");
			exit(1);
		}
		if( read( hack_fd, lvptr, size ) != size ) {
			printm("XXXXXXXXXXXXX read error XXXXXXXXXXXXX\n");
			exit(1);
		}
	}
	return 0;
}

static int
osipx_blk_write( int sel, int *params )
{
	char *lvptr;
	
	//printm("osipx_blk_write: %08X %d %d %d\n", params[0], params[1], params[2], params[3] );
	
	if( mphys_to_lvptr( params[0], &lvptr ) ) {
		printm("osipx_blk_write: Not in RAM access");
		return -1;
	}

	if( hack_fd != -1 ) {
		int blk = params[1] + (params[2] >> 9);
		int offs = params[2] & 0x1ff;
		int size = params[3];

		// printm("blk %d, offs %d, size %d\n", blk, offs, size );

		if( blk_lseek( hack_fd, blk, offs ) ) {
			printm("XXXXXXXXXXXXX blk_lseek error XXXXXXXXXXXXX\n");
			exit(1);
		}
		if( write( hack_fd, lvptr, size ) != size ) {
			printm("XXXXXXXXXXXXX write error XXXXXXXXXXXXX\n");
			exit(1);
		}
	}
	return 0;
}

/* &blk_size --- num_blocks */
static int
osipx_blk_size( int sel, int *params )
{
	return get_file_size_in_blks( hack_fd );
}
