/********************************************************************
 *
 *      Copyright (C) 1999-2002 Nathan Fiedler
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * PROJECT:     Abstract Data Types
 * MODULE:      Disjoint Set
 * FILE:        DisjointSet.java
 *
 * AUTHOR:      Nathan Fiedler
 *
 * REVISION HISTORY:
 *      Name    Date            Description
 *      ----    ----            -----------
 *      nf      05/06/97        Initial version
 *      nf      08/17/97        Added ability to store objects
 *      nf      11/06/97        Changed marker value from 0 to -1 to
 *                              fix a serious flaw
 *      nf      01/20/01        Removed some of the synchronization
 *      nf      01/22/01        Fixed some of the stupid API.
 *      nf      01/04/02        Implemented List interface
 *
 * DESCRIPTION:
 *      Implements a Disjoint set ADT. It uses both the union-by-rank
 *      and path-compression heuristics to improve the run-time
 *      performance of the ADT. This ADT can also associate data
 *      objects with the elements in the set.
 *
 * $Id: DisjointSet.java,v 1.6 2002/01/06 09:33:03 nfiedler Exp $
 *
 *******************************************************************/

package com.bluemarsh.adt;

import java.lang.reflect.Array;
import java.util.*;

/**
 * Implements a disjoint set data type. The elements of the set begin
 * unconnected with each element in its own set. You can join elements
 * together into larger sets of elements, where every element has a root
 * element. You can check if two elements are in the same set by testing
 * if their root elements are the same, using the find method. Each element
 * in the set can have an object associated with it. Use the set() and get()
 * methods to store objects in the set. Each object will be associated with
 * an element based on the offset at which the object is stored.
 *
 * <p><b>Note that this implementation is not synchronized.</b> If multiple
 * threads access a set concurrently, and at least one of the threads modifies
 * the set, it <i>must</i> be synchronized externally. This is typically
 * accomplished by synchronizing on some object that naturally encapsulates
 * the set. If no such object exists, the set should be "wrapped" using the
 * <code>Collections.synchronizedList</code> method. This is best done at
 * creation time, to prevent accidental unsynchronized access to the set:</p>
 *<pre>
 *     List l = Collections.synchronizedList(new DisjointSet(...));
 *</pre>
 *
 * @author  Nathan Fiedler
 */
public class DisjointSet implements Cloneable, List {
    /** The disjoint set of elements. The numbers stored in this
     * array point to the root element of the array element. If
     * the number is less than zero then it is a root and it's
     * absolute value is the height of the tree. */
    protected int[] disjointSet;
    /** The set of objects represented in the disjoint set. */
    protected Object[] objectSet;
    /** Number of disjoint trees in the set. */
    protected int numTrees;
    /** Number of non-null elements stored in the set. */
    protected int numElements;

    /**
     * One-arg constructor for DisjointSet. Allocates the necessary
     * memory and creates the initial sets with each element being
     * in its own set.
     *
     * @param  size  number of elements to be stored in the set
     */
    public DisjointSet(int size) {
        if (size < 0) {
            throw new IllegalArgumentException("size cannot be negative");
        }
        init(size);
    } // DisjointSet

    /**
     * This method is not supported.
     *
     * <em>
     * Appends the specified element to the end of this list (optional
     * operation).<p>
     *
     * Lists that support this operation may place limitations on what
     * elements may be added to this list. In particular, some
     * lists will refuse to add null elements, and others will impose
     * restrictions on the type of elements that may be added. List
     * classes should clearly specify in their documentation any restrictions
     * on what elements may be added.
     * </em>
     *
     * @param  o  element to be appended to this list.
     * @return  <tt>true</tt> (as per the general contract of the
     *          <tt>Collection.add</tt> method).
     * 
     * @exception  UnsupportedOperationException
     *             if the <tt>add</tt> method is not supported by this list.
     * @exception  ClassCastException
     *             if the class of the specified element prevents it from
     *             being added to this list.
     * @exception  IllegalArgumentException
     *             if some aspect of this element prevents it from being
     *             added to this collection.
     */
    public boolean add(Object o) {
        throw new UnsupportedOperationException();
    } // add

    /**
     * This method is not supported.
     *
     * <em>
     * Inserts the specified element at the specified position in this list
     * (optional operation). Shifts the element currently at that position
     * (if any) and any subsequent elements to the right (adds one to their
     * indices).
     * </em>
     *
     * @param  index    index at which the specified element is to be inserted.
     * @param  element  element to be inserted.
     * 
     * @exception  UnsupportedOperationException
     *             if the <tt>add</tt> method is not supported by this list.
     * @exception  ClassCastException
     *             if the class of the specified element prevents it from
     *             being added to this list.
     * @exception  IllegalArgumentException
     *             if some aspect of the specified element prevents it from
     *             being added to this list.
     * @exception  IndexOutOfBoundsException
     *             if the index is out of range (index &lt; 0 ||
     *             index &gt; size()).
     */
    public void add(int index, Object element) {
        throw new UnsupportedOperationException();
    } // add

    /**
     * This method is not supported.
     *
     * <em>
     * Appends all of the elements in the specified collection to the end of
     * this list, in the order that they are returned by the specified
     * collection's iterator (optional operation). The behavior of this
     * operation is unspecified if the specified collection is modified while
     * the operation is in progress. (Note that this will occur if the
     * specified collection is this list, and it's nonempty.)
     * </em>
     *
     * @param  c  collection whose elements are to be added to this list.
     * @return  <tt>true</tt> if this list changed as a result of the call.
     * 
     * @exception  UnsupportedOperationException
     *             if the <tt>addAll</tt> method is not supported by this list.
     * @exception  ClassCastException
     *             if the class of an element in the specified collection
     *             prevents it from being added to this list.
     * @exception  IllegalArgumentException
     *             if some aspect of an element in the specified collection
     *             prevents it from being added to this list.
     * @exception  NullPointerException
     *             if the specified collection is <tt>null</tt>.
     * @see #add(Object)
     */
    public boolean addAll(Collection c) {
        throw new UnsupportedOperationException();
    } // addAll

    /**
     * This method is not supported.
     *
     * <em>
     * Inserts all of the elements in the specified collection into this
     * list at the specified position (optional operation). Shifts the
     * element currently at that position (if any) and any subsequent
     * elements to the right (increases their indices). The new elements
     * will appear in this list in the order that they are returned by the
     * specified collection's iterator. The behavior of this operation is
     * unspecified if the specified collection is modified while the
     * operation is in progress. (Note that this will occur if the specified
     * collection is this list, and it's nonempty.)
     * </em>
     *
     * @param  index  index at which to insert first element from the
     *                specified collection.
     * @param  c      elements to be inserted into this list.
     * @return  <tt>true</tt> if this list changed as a result of the call.
     * 
     * @exception  UnsupportedOperationException
     *             if the <tt>addAll</tt> method is not supported by this list.
     * @exception  ClassCastException
     *             if the class of one of elements of the specified
     *             collection prevents it from being added to this list.
     * @exception  IllegalArgumentException
     *             if some aspect of one of elements of the specified
     *             collection prevents it from being added to this list.
     * @exception  IndexOutOfBoundsException
     *             if the index is out of range (index &lt; 0 ||
     *             index &gt; size()).
     * @exception  NullPointerException
     *             if the specified collection is <tt>null</tt>.
     */
    public boolean addAll(int index, Collection c) {
        throw new UnsupportedOperationException();
    } // addAll

    /**
     * Removes all of the elements from this list (optional operation).
     * This list will be empty after this call returns (unless it throws
     * an exception).
     *
     * @exception  UnsupportedOperationException
     *             if the <tt>clear</tt> method is not supported by this list.
     */
    public void clear() {
        init(objectSet.length);
    } // clear

    /**
     * Creates a shallow copy of this object. The objects contained
     * in this set are not copied, only the set object itself.
     *
     * @return  the clone of this set
     */
    public Object clone() {
        // First make a new disjoint set of the same length.
        DisjointSet copy = new DisjointSet(disjointSet.length);
        // Then copy the array to the new set.
        for (int i = 0; i < disjointSet.length; i++) {
            copy.disjointSet[i] = disjointSet[i];
            copy.objectSet[i] = objectSet[i];
        }
        copy.numTrees = numTrees;
        copy.numElements = numElements;
        return copy;
    } // clone

    /**
     * Returns <tt>true</tt> if this list contains the specified element.
     * More formally, returns <tt>true</tt> if and only if this list contains
     * at least one element <tt>e</tt> such that
     * <tt>(o==null&nbsp;?&nbsp;e==null&nbsp;:&nbsp;o.equals(e))</tt>.
     *
     * @param  o  element whose presence in this list is to be tested.
     * @return  <tt>true</tt> if this list contains the specified element.
     */
    public boolean contains(Object o) {
        for (int ii = 0; ii < objectSet.length; ii++) {
            if (objectSet[ii] != null && o.equals(objectSet[ii])) {
                return true;
            }
        }
        return false;
    } // contains

    /**
     * Returns <tt>true</tt> if this list contains all of the elements of
     * the specified collection.
     *
     * @param  c  collection to be checked for containment in this list.
     * @return  <tt>true</tt> if this list contains all of the elements of
     *          the specified collection.
     * @exception  NullPointerException
     *             if the specified collection is <tt>null</tt>.
     * @see #contains(Object)
     */
    public boolean containsAll(Collection c) {
        Iterator iter = c.iterator();
        while (iter.hasNext()) {
            if (!contains(iter.next())) {
                return false;
            }
        }
        return true;
    } // containsAll

    /**
     * Compares the specified object with this list for equality. Returns
     * <tt>true</tt> if and only if the specified object is also a list, both
     * lists have the same size, and all corresponding pairs of elements in
     * the two lists are <i>equal</i>. (Two elements <tt>e1</tt> and
     * <tt>e2</tt> are <i>equal</i> if <tt>(e1==null ? e2==null :
     * e1.equals(e2))</tt>.)  In other words, two lists are defined to be
     * equal if they contain the same elements in the same order. This
     * definition ensures that the equals method works properly across
     * different implementations of the <tt>List</tt> interface.
     *
     * @param  o  the object to be compared for equality with this list.
     * @return  <tt>true</tt> if the specified object is equal to this list.
     */
    public boolean equals(Object o) {
        return o == this;
    } // equals

    /**
     * Find the root of the given element. You can use this to test
     * if two elements are in the same set by testing if their roots
     * are the same.
     * Running time: O(n)
     *
     * @param  index  element to find the root of
     * @return  root element of given element
     * @exception  IndexOutOfBoundsException
     *             Thrown if 'index' is out of bounds.
     */
    public int find(int index) {
        // Handle case where index is out of bounds.
        if ((index < 0) || (index > disjointSet.length)) {
            throw new IndexOutOfBoundsException();
        }
        // If the element value is less than zero then we've found the
        // root. Otherwise keep looking by calling ourselves recursively.
        // This will automatically shorten the tree.
        if (disjointSet[index] < 0) {
            return(index);
        } else {
            disjointSet[index] = find(disjointSet[index]);
            return(disjointSet[index]);
        }
    } // find

    /**
     * Returns the element at the specified position in this list.
     *
     * @param  index  index of element to return.
     * @return  the element at the specified position in this list.
     * @exception  IndexOutOfBoundsException
     *             if the index is out of range (index &lt; 0 ||
     *             index &gt;= size()).
     */
    public Object get(int index) {
        if ((index < 0) || (index >= objectSet.length)) {
            throw new IndexOutOfBoundsException();
        }
        return objectSet[index];
    } // get

    /**
     * Returns the hash code value for this list. The hash code of a list
     * is defined to be the result of the following calculation:
     * <pre>
     *  hashCode = 1;
     *  Iterator i = list.iterator();
     *  while (i.hasNext()) {
     *      Object obj = i.next();
     *      hashCode = 31*hashCode + (obj==null ? 0 : obj.hashCode());
     *  }
     * </pre>
     * This ensures that <tt>list1.equals(list2)</tt> implies that
     * <tt>list1.hashCode()==list2.hashCode()</tt> for any two lists,
     * <tt>list1</tt> and <tt>list2</tt>, as required by the general
     * contract of <tt>Object.hashCode</tt>.
     *
     * @return  the hash code value for this list.
     * @see Object#hashCode()
     * @see Object#equals(Object)
     * @see #equals(Object)
     */
    public int hashCode() {
        return System.identityHashCode(this);
    } // hashCode

    /**
     * Returns the index in this list of the first occurrence of the specified
     * element, or -1 if this list does not contain this element.
     * More formally, returns the lowest index <tt>i</tt> such that
     * <tt>(o==null ? get(i)==null : o.equals(get(i)))</tt>,
     * or -1 if there is no such index.
     *
     * @param  o  element to search for.
     * @return  the index in this list of the first occurrence of the specified
     *          element, or -1 if this list does not contain this element.
     */
    public int indexOf(Object o) {
        return indexOf(o, 0);
    } // indexOf

    /**
     * Find the index in the set of the given object.
     *
     * @param  o  object to look for in set
     * @param  i  starting offset to begin searching from
     * @return  the zero-based offset into set where object was first
     *          found after starting index index, or -1 if not found
     */
    public int indexOf(Object o, int i) {
        for (int idx = i; idx < objectSet.length; idx++) {
            if (objectSet[idx].equals(o)) {
                return idx;
            }
        }
        return -1;
    } // indexOf

    /**
     * Initialize the data structure to hold <code>size</code> elements.
     *
     * @param  size  number of elements to hold.
     */
    protected void init(int size) {
        // Simply allocate an integer array of the given size.
        // Be sure to clear out the set, making all the elements
        // the roots of their own individual sets (i.e. they are
        // in trees of height one).
        disjointSet = new int[size];
        objectSet = new Object[size];
        for (int i = 0; i < size; i++) {
            disjointSet[i] = -1;
        }
        numTrees = size;
        numElements = 0;
    } // init

    /**
     * Returns <tt>true</tt> if this list contains no elements.
     *
     * @return  <tt>true</tt> if this list contains no elements.
     */
    public boolean isEmpty() {
        return numElements == 0;
    } // isEmpty

    /**
     * Returns an iterator over the elements in this list in proper sequence.
     *
     * @return  an iterator over the elements in this list in proper sequence.
     */
    public Iterator iterator() {
        return new Iter(objectSet);
    } // iterator

    /**
     * Returns the index in this list of the last occurrence of the specified
     * element, or -1 if this list does not contain this element.
     * More formally, returns the highest index <tt>i</tt> such that
     * <tt>(o==null ? get(i)==null : o.equals(get(i)))</tt>,
     * or -1 if there is no such index.
     *
     * @param  o  element to search for.
     * @return  the index in this list of the last occurrence of the specified
     *          element, or -1 if this list does not contain this element.
     */
    public int lastIndexOf(Object o) {
        for (int ii = objectSet.length - 1; ii > -1; ii--) {
            if (objectSet[ii].equals(o)) {
                return ii;
            }
        }
        return -1;
    } // lastIndexOf

    /**
     * Returns a list iterator of the elements in this list (in proper
     * sequence).
     *
     * @return  a list iterator of the elements in this list (in proper
     *          sequence).
     */
    public ListIterator listIterator() {
        return new ListIter(objectSet);
    } // listIterator

    /**
     * Returns a list iterator of the elements in this list (in proper
     * sequence), starting at the specified position in this list. The
     * specified index indicates the first element that would be returned by
     * an initial call to the <tt>next</tt> method. An initial call to
     * the <tt>previous</tt> method would return the element with the
     * specified index minus one.
     *
     * @param  index  index of first element to be returned from the
     *                list iterator (by a call to the <tt>next</tt> method).
     * @return  a list iterator of the elements in this list (in proper
     *          sequence), starting at the specified position in this list.
     * @exception  IndexOutOfBoundsException
     *             if the index is out of range (index &lt; 0 ||
     *             index &gt; size()).
     */
    public ListIterator listIterator(int index) {
        if (index < 0 || index >= objectSet.length) {
            throw new IndexOutOfBoundsException();
        }
        return new ListIter(objectSet, index);
    } // listIterator

    /**
     * Returns the total number of elements in this set. This number
     * is the same throughout the life of the set and is equal to the
     * number passed to the constructor.
     *
     * <p>This method has O(n) running time due to the fact that it
     * iterates the set looking for non-null elements.</p>
     *
     * @return  number of elements in the set
     */
    public int numberOfElements() {
        return numElements;
    } // numberOfElements

    /**
     * Returns the number of disjoint elements in the set. If all the
     * elements are separate then this method will return the number of
     * elements in the entire set. If all the elements are joined into
     * one tree then this method will return 1.
     *
     * @return  number of trees in the set
     */
    public int numberOfTrees() {
        return numTrees;
    } // numberOfTrees

    /**
     * This method is not supported.
     *
     * <em>
     * Removes the element at the specified position in this list (optional
     * operation). Shifts any subsequent elements to the left (subtracts one
     * from their indices). Returns the element that was removed from the
     * list.
     * </em>
     *
     * @param index the index of the element to removed.
     * @return the element previously at the specified position.
     * 
     * @throws UnsupportedOperationException if the <tt>remove</tt> method is
     *            not supported by this list.
     * 
     * @throws IndexOutOfBoundsException if the index is out of range (index
     *            &lt; 0 || index &gt;= size()).
     */
    public Object remove(int index) {
        throw new UnsupportedOperationException();
    } // remove

    /**
     * This method is not supported.
     *
     * <em>
     * Removes the first occurrence in this list of the specified element 
     * (optional operation). If this list does not contain the element, it is
     * unchanged. More formally, removes the element with the lowest index i
     * such that <tt>(o==null ? get(i)==null : o.equals(get(i)))</tt> (if
     * such an element exists).
     * </em>
     *
     * @param  o  element to be removed from this list, if present.
     * @return  <tt>true</tt> if this list contained the specified element.
     * @exception  UnsupportedOperationException
     *             if the <tt>remove</tt> method is not supported by this list.
     */
    public boolean remove(Object o) {
        throw new UnsupportedOperationException();
    } // remove

    /**
     * This method is not supported.
     *
     * <em>
     * Removes from this list all the elements that are contained in the
     * specified collection (optional operation).
     * </em>
     *
     * @param  c  collection that defines which elements will be removed from
     *            this list.
     * @return  <tt>true</tt> if this list changed as a result of the call.
     * 
     * @exception  UnsupportedOperationException
     *             if the <tt>removeAll</tt> method is not supported by
     *             this list.
     * @exception  NullPointerException
     *             if the specified collection is <tt>null</tt>.
     * @see #remove(Object)
     * @see #contains(Object)
     */
    public boolean removeAll(Collection c) {
        throw new UnsupportedOperationException();
    } // removeAll

    /**
     * This method is not supported.
     *
     * <em>
     * Retains only the elements in this list that are contained in the
     * specified collection (optional operation). In other words, removes
     * from this list all the elements that are not contained in the specified
     * collection.
     * </em>
     *
     * @param  c  collection that defines which elements this set will retain.
     * @return  <tt>true</tt> if this list changed as a result of the call.
     * @exception  UnsupportedOperationException
     *             if the <tt>retainAll</tt> method is not supported by
     *             this list.
     * @exception  NullPointerException
     *             if the specified collection is <tt>null</tt>.
     * @see #remove(Object)
     * @see #contains(Object)
     */
    public boolean retainAll(Collection c) {
        throw new UnsupportedOperationException();
    } // retainAll

    /**
     * Replaces the element at the specified position in this list with
     * the specified element (optional operation).
     *
     * @param  index    index of element to replace.
     * @param  element  element to be stored at the specified position.
     * @return  the element previously at the specified position.
     * @exception  UnsupportedOperationException
     *             if the <tt>set</tt> method is not supported by this list.
     * @exception  ClassCastException
     *             if the class of the specified element prevents it from
     *             being added to this list.
     * @exception  IllegalArgumentException
     *             if some aspect of the specified element prevents it
     *             from being added to this list.
     * @exception  IndexOutOfBoundsException
     *             if the index is out of range (index &lt; 0 ||
     *             index &gt;= size()).
     */
    public Object set(int index, Object element) {
        if (index < 0 || index >= objectSet.length) {
            throw new IndexOutOfBoundsException();
        }
        Object old = objectSet[index];
        objectSet[index] = element;
        if (element != null) {
            // Increment the number of non-null elements.
            numElements++;
        } else if (old != null) {
            // Decrement the number of non-null elements.
            numElements--;
        }
        return old;
    } // set

    /**
     * Returns the number of elements in this list. If this list contains
     * more than <tt>Integer.MAX_VALUE</tt> elements, returns
     * <tt>Integer.MAX_VALUE</tt>.
     *
     * @return  the number of elements in this list.
     */
    public int size() {
        return numberOfElements();
    } // size

    /**
     * Returns a view of the portion of this list between the specified
     * <tt>fromIndex</tt>, inclusive, and <tt>toIndex</tt>, exclusive. (If
     * <tt>fromIndex</tt> and <tt>toIndex</tt> are equal, the returned list is
     * empty.)  The returned list is backed by this list, so non-structural
     * changes in the returned list are reflected in this list, and vice-versa.
     * The returned list supports all of the optional list operations supported
     * by this list.<p>
     *
     * This method eliminates the need for explicit range operations (of
     * the sort that commonly exist for arrays). Any operation that expects
     * a list can be used as a range operation by passing a subList view
     * instead of a whole list. For example, the following idiom
     * removes a range of elements from a list:
     * <pre>
     *      list.subList(from, to).clear();
     * </pre>
     * Similar idioms may be constructed for <tt>indexOf</tt> and
     * <tt>lastIndexOf</tt>, and all of the algorithms in the
     * <tt>Collections</tt> class can be applied to a subList.<p>
     *
     * The semantics of the list returned by this method become undefined if
     * the backing list (i.e., this list) is <i>structurally modified</i> in
     * any way other than via the returned list. (Structural modifications are
     * those that change the size of this list, or otherwise perturb it in such
     * a fashion that iterations in progress may yield incorrect results.)
     *
     * @param  fromIndex  low endpoint (inclusive) of the subList.
     * @param  toIndex    high endpoint (exclusive) of the subList.
     * @return  a view of the specified range within this list.
     * @exception  IndexOutOfBoundsException
     *             for an illegal endpoint index value (fromIndex &lt; 0 ||
     *             toIndex &gt; size || fromIndex &gt; toIndex).
     */
    public List subList(int fromIndex, int toIndex) {
        ArrayList list = new ArrayList(toIndex - fromIndex);
        for (int ii = fromIndex; ii < toIndex; ii++) {
            list.add(objectSet[ii]);
        }
        return list;
    } // subList

    /**
     * Returns an array containing all of the elements in this list in
     * proper sequence. Obeys the general contract of the
     * <tt>Collection.toArray</tt> method.
     *
     * @return  an array containing all of the elements in this list in proper
     *          sequence.
     * @see Arrays#asList(Object[])
     */
    public Object[] toArray() {
        Object[] copy = new Object[objectSet.length];
        System.arraycopy(objectSet, 0, copy, 0, objectSet.length);
        return copy;
    } // toArray

    /**
     * Returns an array containing all of the elements in this list in proper
     * sequence; the runtime type of the returned array is that of the
     * specified array. Obeys the general contract of the
     * <tt>Collection.toArray(Object[])</tt> method.
     *
     * @param  a  the array into which the elements of this list are to
     *            be stored, if it is big enough; otherwise, a new array of
     *            the same runtime type is allocated for this purpose.
     * @return  an array containing the elements of this list.
     * @exception  ArrayStoreException
     *             if the runtime type of the specified array is not a
     *             supertype of the runtime type of every element in this list.
     * @exception  NullPointerException
     *             if the specified array is <tt>null</tt>.
     */
    public Object[] toArray(Object a[]) {
        // Get the elements into an array.
        Object[] result = toArray();

        // Copy the elements into the array of the desired type.
        int size = size();
        if (a.length < size) {
            a = (Object[]) Array.newInstance(
                a.getClass().getComponentType(), size);
        }
        for (int ii = 0; ii < size; ii++) {
            a[ii] = result[ii];
        }
        if (a.length > size) {
            a[size] = null;
        }
        return a;
    } // toArray

    /**
     * Return the string representation of this set.
     *
     * @return  string representing this set
     */
    public String toString() {
        StringBuffer sb = new StringBuffer("DisjointSet=[");
        sb.append(objectSet[0]);
        for (int i = 1; i < objectSet.length; i++) {
            sb.append(", ");
            sb.append(objectSet[i]);
        }
        sb.append("]");
        return sb.toString();
    } // toString

    /**
     * Join two trees together within the set. It makes the tree
     * that is less deep become a child of the deeper tree. This
     * keeps the resulting tree as shallow as possible thus
     * improving the runtime of the find operation. This operation
     * decrements the number of disjoint trees by one.
     *
     * @param  root1  root of the first tree
     * @param  root2  root of the second tree
     */
    public void union(int root1, int root2) {
        // Check for invalid input.
        if ((root1 < 0) || (root1 > disjointSet.length)) {
            throw new IndexOutOfBoundsException("root1 out of bounds");
        }
        if ((root2 < 0) || (root2 > disjointSet.length)) {
            throw new IndexOutOfBoundsException("root2 out of bounds");
        }
        // Make sure the passed indices are in fact roots.
        // Exit immediately if the roots are the same.
        root1 = find(root1);
        root2 = find(root2);
        if (root1 == root2) {
            return;
        }

        // If second tree is deeper, make it the root of the
        // first tree. Else, if the two trees are of equal
        // depth, make the first tree the root of the second
        // tree and update the depth appropriately.
        if (disjointSet[root2] < disjointSet[root1]) {
            // Root2 is deeper set, make it the new root. This does
            // not require any changes to the height of either tree.
            disjointSet[root1] = root2;
        } else {
            // They're the same height, so update appropriately.
            if (disjointSet[root2] == disjointSet[root1]) {
                // Set the height of the tree by decrementing by one.
                disjointSet[root1]--;
            }
            // Make root1 the new root.
            disjointSet[root2] = root1;
        }
        // Decrement number of disjoint trees in set.
        numTrees--;
    } // union

    /**
     * Iterates a disjoint set. This iterator does not care if the
     * elements within the disjoint set change. It really does not
     * matter anyway.
     */
    protected class Iter implements Iterator {
        /** Reference to the object set. */
        protected Object[] set;
        /** Index within the set. */
        protected int index;

        /**
         * Constructs an iterator for the given object array.
         *
         * @param  set  object set to iterate over.
         */
        public Iter(Object[] set) {
            this.set = set;
        } // Iter

        /**
         * Returns <tt>true</tt> if the iteration has more elements.
         * (In other words, returns <tt>true</tt> if <tt>next</tt> would
         * return an element rather than throwing an exception.)
         *
         * @return  <tt>true</tt> if the iterator has more elements.
         */
        public boolean hasNext() {
            return index < set.length;
        } // hasNext

        /**
         * Returns the next element in the iteration.
         *
         * @return  the next element in the iteration.
         * @exception  NoSuchElementException
         *             iteration has no more elements.
         */
        public Object next() {
            if (hasNext()) {
                return set[index++];
            } else {
                throw new NoSuchElementException();
            }
        } // next

        /**
         * This method is not supported.
         *
         * <em>Removes from the underlying collection the last element returned
         * by the iterator (optional operation). This method can be called
         * only once per call to <tt>next</tt>. The behavior of an iterator
         * is unspecified if the underlying collection is modified while the
         * iteration is in progress in any way other than by calling this
         * method.</em>
         *
         * @exception  UnsupportedOperationException
         *             if the <tt>remove</tt> operation is not supported by
         *             this Iterator.
         * @exception  IllegalStateException
         *             if the <tt>next</tt> method has not yet been called,
         *             or the <tt>remove</tt> method has already been called
         *             after the last call to the <tt>next</tt> method.
         */
        public void remove() {
            throw new UnsupportedOperationException();
        } // remove
    } // Iterator

    /**
     * Iterates a disjoint set. This iterator does not care if the
     * elements within the disjoint set change. It really does not
     * matter anyway.
     */
    protected class ListIter extends Iter implements ListIterator {

        /**
         * Constructs an iterator for the given object array.
         *
         * @param  set  object set to iterate over.
         */
        public ListIter(Object[] set) {
            this(set, 0);
        } // ListIter

        /**
         * Constructs an iterator for the given object array.
         *
         * @param  set  object set to iterate over.
         * @param  idx  initial index into the set.
         */
        public ListIter(Object[] set, int idx) {
            super(set);
            index = idx;
        } // ListIter

        /**
         * This method is not supported.
         *
         * <em>
         * Inserts the specified element into the list (optional
         * operation). The element is inserted immediately before the
         * next element that would be returned by <tt>next</tt>, if
         * any, and after the next element that would be returned by
         * <tt>previous</tt>, if any. (If the list contains no
         * elements, the new element becomes the sole element on the
         * list.) The new element is inserted before the implicit
         * cursor: a subsequent call to <tt>next</tt> would be
         * unaffected, and a subsequent call to <tt>previous</tt>
         * would return the new element. (This call increases by one
         * the value that would be returned by a call to
         * <tt>nextIndex</tt> or <tt>previousIndex</tt>.)
         * </em>
         *
         * @param  o  the element to insert.
         * @exception  UnsupportedOperationException
         *             if the <tt>add</tt> method is not supported by this
         *             list iterator.
         * @exception  ClassCastException
         *             if the class of the specified element prevents it
         *             from being added to this list.
         * @exception  IllegalArgumentException
         *             if some aspect of this element prevents it from
         *             being added to this list.
         */
        public void add(Object o) {
            throw new UnsupportedOperationException();
        } // add

        /**
         * Returns <tt>true</tt> if this list iterator has more elements when
         * traversing the list in the reverse direction. (In other words,
         * returns <tt>true</tt> if <tt>previous</tt> would return an element
         * rather than throwing an exception.)
         *
         * @return  <tt>true</tt> if the list iterator has more elements when
         *          traversing the list in the reverse direction.
         */
        public boolean hasPrevious() {
            return index > 0;
        } // hasPrevious

        /**
         * Returns the index of the element that would be returned by a
         * subsequent call to <tt>next</tt>. (Returns list size if the list
         * iterator is at the end of the list.)
         *
         * @return  the index of the element that would be returned by a
         *          subsequent call to <tt>next</tt>, or list size if list
         *          iterator is at end of list.
         */
        public int nextIndex() {
            return index;
        } // nextIndex

        /**
         * Returns the previous element in the list. This method may be called
         * repeatedly to iterate through the list backwards, or intermixed with
         * calls to <tt>next</tt> to go back and forth. (Note that alternating
         * calls to <tt>next</tt> and <tt>previous</tt> will return the same
         * element repeatedly.)
         *
         * @return  the previous element in the list.
         * @exception  NoSuchElementException
         *             if the iteration has no previous element.
         */
        public Object previous() {
            if (hasPrevious()) {
                return set[--index];
            } else {
                throw new NoSuchElementException();
            }
        } // previous

        /**
         * Returns the index of the element that would be returned by a
         * subsequent call to <tt>previous</tt>. (Returns -1 if the list
         * iterator is at the beginning of the list.)
         *
         * @return  the index of the element that would be returned by a
         *          subsequent call to <tt>previous</tt>, or -1 if list
         *          iterator is at beginning of list.
         */ 
        public int previousIndex() {
            return index - 1;
        } // previousIndex

        /**
         * This method is not supported.
         *
         * <em>Removes from the underlying collection the last element returned
         * by the iterator (optional operation). This method can be called
         * only once per call to <tt>next</tt>. The behavior of an iterator
         * is unspecified if the underlying collection is modified while the
         * iteration is in progress in any way other than by calling this
         * method.</em>
         *
         * @exception  UnsupportedOperationException
         *             if the <tt>remove</tt> operation is not supported by
         *             this Iterator.
         * @exception  IllegalStateException
         *             if the <tt>next</tt> method has not yet been called,
         *             or the <tt>remove</tt> method has already been called
         *             after the last call to the <tt>next</tt> method.
         */
        public void remove() {
            throw new UnsupportedOperationException();
        } // remove

        /**
         * Replaces the last element returned by <tt>next</tt> or
         * <tt>previous</tt> with the specified element (optional operation).
         * This call can be made only if neither <tt>ListIterator.remove</tt>
         * nor <tt>ListIterator.add</tt> have been called after the last call
         * to <tt>next</tt> or <tt>previous</tt>.
         *
         * @param  o  the element with which to replace the last element
         *            returned by <tt>next</tt> or <tt>previous</tt>.
         * @exception  UnsupportedOperationException
         *             if the <tt>set</tt> operation is not supported by this
         *             list iterator.
         * @exception  ClassCastException
         *             if the class of the specified element prevents it from
         *             being added to this list.
         * @exception  IllegalArgumentException
         *             if some aspect of the specified element prevents it
         *             from being added to this list.
         * @exception  IllegalStateException
         *             if neither <tt>next</tt> nor <tt>previous</tt> have
         *             been called, or <tt>remove</tt> or <tt>add</tt> have
         *             been called after the last call to <tt>next</tt> or
         *             <tt>previous</tt>.
         */
        public void set(Object o) {
            set[index] = o;
        } // set
    } // ListIter
} // DisjointSet
