//
// File:        Argument.java
// Package:     gov.llnl.babel.symbols
// Revision:    @(#) $Id: Argument.java 4434 2005-03-17 17:05:29Z epperly $
// Description: this class represents a sidl method argument
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.symbols;

import gov.llnl.babel.symbols.SymbolUtilities;
import gov.llnl.babel.symbols.Type;

/**
 * The <code>Argument</code> class defines a SIDL argument, which consists
 * of a parameter passing mode (IN, INOUT, or OUT with an optional COPY),
 * a type, and a formal parameter name.
 */
public class Argument {
   public static final int IN    = 0; 
   public static final int INOUT = 1; 
   public static final int OUT   = 2; 

   private static final String[] s_mode_string = { "in", "inout", "out" };

   private boolean d_copy;
   private String  d_formal_name;
   private int     d_mode;
   private Type    d_type;

   /**
    * Create an argument object.  
    *
    * @param  copy  The copy flag.
    * @param  mode  The parameter passing mode (IN, OUT, or INOUT).
    * @param  type  The SIDL type of the argument.
    * @param  name  The formal parameter name.
    */
   public Argument(boolean copy, int mode, Type type, String name) {
      d_copy        = copy;
      d_formal_name = name;
      d_mode        = mode;
      d_type        = type;
   }

   /**
    * Return TRUE if the copy flag has been set for this method argument; 
    * otherwise, return FALSE.
    */
   public boolean isCopy() {
      return d_copy;
   }

   /**
    * Return the value of the argument parameter passing mode.
    */
   public int getMode() {
      return d_mode;
   }

   /**
    * Return a string representing the argument mode.
    */
   public final String getModeString() {
     return s_mode_string[d_mode];
   }

   /**
    * Return the formal name of the parameter identifier.
    */
   public String getFormalName() {
      return d_formal_name;
   }

   /**
    * Return the type of the argument.
    */
   public Type getType() {
      return(d_type);
   }

  /**
   * Return TRUE if the type of this argument is an array with an with an 
   * ordering specification. For example, <code>in array&lt;int, 2, 
   * column-major&gt; x</code> would return TRUE.  <code>out array&lt;int, 
   * 2&gt; x</code> would return FALSE because it does not have an ordering
   * specification.  Non-array arguments return FALSE.
   */
  public boolean hasArrayOrderSpec() {
    return d_type.hasArrayOrderSpec();
  }


   /**
    * Return a string representation of the argument for outputting the
    * arguments in a method signature, for example.  Optionally abbreviate 
    * the type if it is declared in the specified package.
    */
   public String getArgumentString(String abbrev_pkg) {
      StringBuffer arg = new StringBuffer();
      
      if (d_copy) {
         arg.append("copy ");
      }
      arg.append(getModeString() + " ");

      String rtype = d_type.getTypeString();
      if (abbrev_pkg != null) {
        arg.append(SymbolUtilities.getSymbolName(rtype, abbrev_pkg));
      } else {
        arg.append(rtype);
      }
      arg.append(" ");
      arg.append(d_formal_name);
      
      if(d_type.isRarray()) {
        arg.append("(" + d_type.getIndexString() + ")");
      }

      return arg.toString();
   }

   /**
    * Return a string representation of the argument for outputting the
    * arguments in a method signature, for example.
    */
   public String getArgumentString() {
      return getArgumentString(null);
   }

  public int hashCode() {
    return d_mode*7 + (d_copy ? 1 : 0) + d_type.hashCode();
  }

   /**
    * Return TRUE if the specified object is considered the same as this
    * object; otherwise, return FALSE.  Note that the formal name does 
    * not need to match for equality.
    *
    * @param  object  The object being used for comparison.
    */
   public boolean equals(Object object) {
      boolean eq = false;
      if ((object != null) && (object instanceof Argument)) {
         Argument arg = (Argument) object;
         eq = ((d_copy == arg.d_copy)
            && (d_mode == arg.d_mode)
            && (d_type.equals(arg.d_type)));
      }
      return eq;
   }
}
