/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.vfs;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;

/**
 * Accessor for vanilla directorys VFS.
 *
 * @author <a href="mailto:peter@apache.org">Peter Donald</a>
 */
public final class DirectoryVFileAccessor
    implements VFileAccessor
{
    /**
     * Get size of resource in bytes.
     *
     * @param resource the resource designator
     * @return the size of resource in bytes
     */
    public long getSize( final VFile file, final Object resource )
    {
        try
        {
            final File realFile = getFile( file, resource );
            return realFile.length();
        }
        catch( final Exception e )
        {
            return -1;
        }
    }

    /**
     * Get InputStream for resource.
     *
     * @param resource the resource designator
     * @return the input stream
     * @exception IOException if an error occurs
     */
    public InputStream getInputStream( final VFile file, final Object resource )
    {
        try
        {
            final File realFile = getFile( file, resource );
            return new FileInputStream( realFile );
        }
        catch( final Exception e )
        {
            return null;
        }
    }

    /**
     * Retrieve underlying file for resource entry.
     *
     * @param file the VFile representing resource entry
     * @param resource the resource
     * @return the File
     */
    private File getFile( final VFile file, final Object resource )
    {
        final File baseFile = (File)resource;
        final String name = file.toString().substring( 1 ).replace( '/', File.separatorChar );
        return new File( baseFile, name );
    }
}
