/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.profile;

import java.io.File;
import java.io.IOException;

/**
 * The Profiler is used to determine numeric values for specific parts of the
 * Avalon environment.  The idea is to add references to Profilable objects to
 * the Profiler.  The Profiler takes periodic snapshots of the running system
 * so that performance or resource usage can be assertained.  The sample duration
 * is dependant on the Profiler's settings, and should never change during the
 * time the Profiler is running.
 *
 * <p>
 *   Please do respect that for more accurate statistical information, the first
 *   and last sample must be thrown out.  The first sample may have residual
 *   information from before the test, and the last sample may be from an
 *   incomplete timeslice.  For instance, if the Profiler obtains a sample once
 *   every second, and it stops itself 500ms after the previous sample, the last
 *   sample will only represent 1/2 the typical timeslice.
 * </p>
 * <p>
 *   Please also bear in mind that Java has a non-deterministic scheduler, and
 *   samples may not be taken exactly on the specified interval.  This sample
 *   <em>jitter</em> is acceptable when averaged over a longer period of time.
 * </p>
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 */
public interface Profiler
{
    /**
     * Adds a target to profile, along with a name for the target.  Good names
     * include what the expected samples are.  For instance "ThreadController:
     * number of threads" or "EventQueue: events processed per second".  The real
     * results come from the Profilable object itself, but the name is so humans
     * have a reference for the values.  NOTE: if the Profilable class does
     * not expose any ProfilePoints, it is excluded from the list of Profilable
     * classes that are notified when the Profiler is active.
     *
     * @parameter  profileSource  The actual source of the samples
     */
    void add( Profilable profileSource );

    /**
     * Reports the results of the profiling to a ProfileReport.  The actual
     * format depends on the ProfileReport given.  It can be simple Comma
     * Separated Values (CSV) with the columns representing a unique Profilable.
     * Most spreadsheet programs can import this and generate meaningful graphs
     * from it.  Another alternative is to output the information in a tool
     * specific format.  The actual format depends on the ProfileReport in
     * question, and the Profiler merely needs to reference to the object.
     *
     * @parameter  outputInfo  The handle of the file the profiler serializes the
     *                         results to.
     *
     * @throws <code>NullPointerException</code> If the ProfileReport is null.
     */
    void report( ProfileReport outputInfo );
}
