/***************************************************************************

 Linux SVGALib adaptation by Phillip Ezolt pe28+@andrew.cmu.edu
  
  linux.c

  OS dependant stuff (display handling, keyboard scan...)
  This is the only file which should me modified in order to port the
  emulator to a different system.

***************************************************************************/
#ifdef svgalib

#define __SVGALIB_C

#include <vga.h>
#include <vgagl.h>
#include <vgakeyboard.h>
#include <signal.h>

#include "xmame.h"
#include "osdepend.h"
#include "driver.h"
#ifndef linux_powerpc
#include "twkuser.c"
#endif

/* screen is the used part of, well the real screen */
static int screen_width, screen_height;
static int screen_startx, screen_starty;
static int planar_mode = FALSE;

static unsigned char *video_mem;

void svgalib_mouse_init(void);

#ifndef linux_powerpc
Register scr224x288[] =
{
        { 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x5f},{ 0x3d4, 0x01, 0x37},
        { 0x3d4, 0x02, 0x38},{ 0x3d4, 0x03, 0x82},{ 0x3d4, 0x04, 0x4a},
        { 0x3d4, 0x05, 0x9a},{ 0x3d4, 0x06, 0x55},{ 0x3d4, 0x07, 0xf0},
        { 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x61},{ 0x3d4, 0x10, 0x40},
        { 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0x3f},{ 0x3d4, 0x13, 0x1c},
        { 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x40},{ 0x3d4, 0x16, 0x4a},
        { 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e},
        { 0x3ce, 0x05, 0x40},{ 0x3ce, 0x06, 0x05},{ 0x3c0, 0x10, 0x41},
        { 0x3c0, 0x13, 0x0}
};

Register scr224x288scanlines[] =
{
     { 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x5f},{ 0x3d4, 0x01, 0x37},
             { 0x3d4, 0x02, 0x38},{ 0x3d4, 0x03, 0x82},{ 0x3d4, 0x04, 0x4a},
             { 0x3d4, 0x05, 0x9a},{ 0x3d4, 0x06, 0x43},{ 0x3d4, 0x07, 0x1f},
             { 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x60},{ 0x3d4, 0x10, 0x2a},
             { 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0x1f},{ 0x3d4, 0x13, 0x1c},
             { 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x27},{ 0x3d4, 0x16, 0x3a},
             { 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e},
             { 0x3ce, 0x05, 0x40},{ 0x3ce, 0x06, 0x05},{ 0x3c0, 0x10, 0x41},
             { 0x3c0, 0x13, 0x00}
};

Register scr256x256[] =
{
     { 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x5f},{ 0x3d4, 0x01, 0x3f},
             { 0x3d4, 0x02, 0x40},{ 0x3d4, 0x03, 0x82},{ 0x3d4, 0x04, 0x4A},
             { 0x3d4, 0x05, 0x9A},{ 0x3d4, 0x06, 0x23},{ 0x3d4, 0x07, 0xb2},
             { 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x61},{ 0x3d4, 0x10, 0x0a},
             { 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0xff},{ 0x3d4, 0x13, 0x20},
             { 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x07},{ 0x3d4, 0x16, 0x1a},
             { 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e},
             { 0x3ce, 0x05, 0x40},{ 0x3ce, 0x06, 0x05},{ 0x3c0, 0x10, 0x41},
             { 0x3c0, 0x13, 0x00}
};

Register scr256x256scanlines[] =
{
     { 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x5f},{ 0x3d4, 0x01, 0x3f},
             { 0x3d4, 0x02, 0x40},{ 0x3d4, 0x03, 0x82},{ 0x3d4, 0x04, 0x4a},
             { 0x3d4, 0x05, 0x9a},{ 0x3d4, 0x06, 0x23},{ 0x3d4, 0x07, 0x1d},
             { 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x60},{ 0x3d4, 0x10, 0x0a},
             { 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0xff},{ 0x3d4, 0x13, 0x20},
             { 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x07},{ 0x3d4, 0x16, 0x1a},
             { 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e},
             { 0x3ce, 0x05, 0x40},{ 0x3ce, 0x06, 0x05},{ 0x3c0, 0x10, 0x41},
             { 0x3c0, 0x13, 0x00}
};

Register scr288x224[] =
{
        { 0x3c2, 0x0, 0xe3},{ 0x3d4, 0x0, 0x5f},{ 0x3d4, 0x1, 0x47},
        { 0x3d4, 0x2, 0x50},{ 0x3d4, 0x3, 0x82},{ 0x3d4, 0x4, 0x50},
        { 0x3d4, 0x5, 0x80},{ 0x3d4, 0x6, 0xb},{ 0x3d4, 0x7, 0x3e},
        { 0x3d4, 0x8, 0x0},{ 0x3d4, 0x9, 0x41},{ 0x3d4, 0x10, 0xda},
        { 0x3d4, 0x11, 0x9c},{ 0x3d4, 0x12, 0xbf},{ 0x3d4, 0x13, 0x24},
        { 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0xc7},{ 0x3d4, 0x16, 0x4},
        { 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x1, 0x1},{ 0x3c4, 0x4, 0xe},
        { 0x3ce, 0x5, 0x40},{ 0x3ce, 0x6, 0x5},{ 0x3c0, 0x10, 0x41},
        { 0x3c0, 0x13, 0x0}
};

Register scr288x224scanlines[] =
{
        { 0x3c2, 0x0, 0xe3},{ 0x3d4, 0x0, 0x5f},{ 0x3d4, 0x1, 0x47},
        { 0x3d4, 0x2, 0x47},{ 0x3d4, 0x3, 0x82},{ 0x3d4, 0x4, 0x50},
        { 0x3d4, 0x5, 0x9a},{ 0x3d4, 0x6, 0xb},{ 0x3d4, 0x7, 0x19},
        { 0x3d4, 0x8, 0x0},{ 0x3d4, 0x9, 0x40},{ 0x3d4, 0x10, 0xf5},
        { 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0xdf},{ 0x3d4, 0x13, 0x24},
        { 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0xc7},{ 0x3d4, 0x16, 0x4},
        { 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x1, 0x1},{ 0x3c4, 0x4, 0xe},
        { 0x3ce, 0x5, 0x40},{ 0x3ce, 0x6, 0x5},{ 0x3c0, 0x10, 0x41},
        { 0x3c0, 0x13, 0x0}
};

Register scr320x204[] =
{
     { 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x5f},{ 0x3d4, 0x01, 0x4f},
             { 0x3d4, 0x02, 0x50},{ 0x3d4, 0x03, 0x82},{ 0x3d4, 0x04, 0x54},
             { 0x3d4, 0x05, 0x80},{ 0x3d4, 0x06, 0xbf},{ 0x3d4, 0x07, 0x1f},
             { 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x41},{ 0x3d4, 0x10, 0x9c},
             { 0x3d4, 0x11, 0x8e},{ 0x3d4, 0x12, 0x97},{ 0x3d4, 0x13, 0x28},
             { 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x96},{ 0x3d4, 0x16, 0xb9},
             { 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e},
             { 0x3ce, 0x05, 0x40},{ 0x3ce, 0x06, 0x05},{ 0x3c0, 0x10, 0x41},
             { 0x3c0, 0x13, 0x00}
};

Register scr240x272[] =
{
        { 0x3c2, 0x0, 0xe3}, { 0x3d4, 0x0, 0x5f}, { 0x3d4, 0x1, 0x3b},
        { 0x3d4, 0x2, 0x38}, { 0x3d4, 0x3, 0x82}, { 0x3d4, 0x4, 0x4a},
        { 0x3d4, 0x5, 0x9a}, { 0x3d4, 0x6, 0x55}, { 0x3d4, 0x7, 0xf0},
        { 0x3d4, 0x8, 0x0},  { 0x3d4, 0x9, 0x61}, { 0x3d4, 0x10, 0x40},
        { 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0x20},{ 0x3d4, 0x13, 0x1e},
        { 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x40},{ 0x3d4, 0x16, 0x4a},
        { 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x1, 0x1},  { 0x3c4, 0x4, 0xe},
        { 0x3ce, 0x5, 0x40}, { 0x3ce, 0x6, 0x5},  { 0x3c0, 0x10, 0x41},
        { 0x3c0, 0x13, 0x0}
};

Register scr200x320[] =
{
	{ 0x3c2, 0x00, 0xe3}, { 0x3d4, 0x00, 0x5f}, { 0x3d4, 0x01, 0x31},
	{ 0x3d4, 0x02, 0x38}, { 0x3d4, 0x03, 0x82}, { 0x3d4, 0x04, 0x4a},
	{ 0x3d4, 0x05, 0x9a}, { 0x3d4, 0x06, 0x4e}, { 0x3d4, 0x07, 0x1f},
	{ 0x3d4, 0x08, 0x00}, { 0x3d4, 0x09, 0x40}, { 0x3d4, 0x10, 0x40},
	{ 0x3d4, 0x11, 0x90}, { 0x3d4, 0x12, 0x3f}, { 0x3d4, 0x13, 0x19},
	{ 0x3d4, 0x14, 0x40}, { 0x3d4, 0x15, 0x80}, { 0x3d4, 0x16, 0x40},
	{ 0x3d4, 0x17, 0xa3}, { 0x3c4, 0x01, 0x01}, { 0x3c4, 0x04, 0x0e},
	{ 0x3ce, 0x05, 0x40}, { 0x3ce, 0x06, 0x05}, { 0x3c0, 0x10, 0x41},
	{ 0x3c0, 0x13, 0x00}
};

void set_video_mode(void)
{
	Register *reg = 0;
	int reglen = 0;
	
	if (screen_width == 224 && screen_height == 288)
          {
	    if (video_mode==2)
	      {
	        reg = scr224x288;
	        reglen = sizeof(scr224x288)/sizeof(Register);
	      }
	    else
	      {
	        reg = scr224x288scanlines;
	        reglen = sizeof(scr224x288scanlines)/sizeof(Register);
	      }
	  }
	else if (screen_width == 256 && screen_height == 256)
	  {
	    if (video_mode==2)
	      {
	        reg = scr256x256;
	        reglen = sizeof(scr256x256)/sizeof(Register);
	      }
	    else
	      {
	        reg = scr256x256scanlines;
	        reglen = sizeof(scr256x256scanlines)/sizeof(Register);
	      }
	  }
	else if (screen_width == 288 && screen_height == 224)
	  {
	    if (video_mode==2)
	      {
	        reg = scr288x224;
	        reglen = sizeof(scr288x224)/sizeof(Register);
	      }
	    else
	      {
	        reg = scr288x224scanlines;
	        reglen = sizeof(scr288x224scanlines)/sizeof(Register);
	      }
	  }
	else if (screen_width == 240 && screen_height == 272)
	  {
	    reg = scr240x272;
	    reglen = sizeof(scr240x272)/sizeof(Register);
	  }
	else if (screen_width == 320 && screen_height == 204)
	  {
	    reg = scr320x204;
	    reglen = sizeof(scr320x204)/sizeof(Register);
	  }
	else if (screen_width == 200 && screen_height == 320)
	  {
	    reg = scr200x320;
	    reglen = sizeof(scr200x320)/sizeof(Register);
	  }
	outRegArray(reg,reglen);
}
#endif /* ifndef linux_powerpc */


static struct sigaction sig2handler;
static struct sigaction oldsig2handler;
#if (__GLIBC__ > 1)
static sigset_t sig2sigset;
#endif

void Sig2HandlerFunction(int n)
{
   keyboard_clearstate();
   (*(oldsig2handler.sa_handler))(n);
#ifndef linux_powerpc   
   if (video_mode) set_video_mode();
#endif
   sigaction(SIGUSR2, &sig2handler, NULL);
}


/* Create display */
int sysdep_create_display(void)
{
   int a;
   
#ifndef linux_powerpc
   /* try tweaked modes */
   if (video_mode != 0)
   {
      if ((visual_width <= 320 && visual_height <= 204))
      {
         screen_width =320;
         screen_height=204;
      }
      else if ((visual_width <= 288 && visual_height <= 224))
      {
         screen_width =288;
         screen_height=224;
      }
      else if ((visual_width <= 256 && visual_height <= 256))
      {
         screen_width =256;
         screen_height=256;
      }
      else if ((visual_width <= 240 && visual_height <= 272))
      {
         screen_width =240;
         screen_height=272;
      }
      else if ((visual_width <= 224 && visual_height <= 288))
      {
         screen_width =224;
         screen_height=288;
      }
      else if ((visual_width <= 200 && visual_height <= 320))
      {
         screen_width =200;
         screen_height=320;
      }
      else
         video_mode=0;
   }
   
   if (video_mode)

   {
        if (!vga_hasmode(G320x200x256))
        {
           fprintf(stderr_file,"Svgalib: 320x200x256colors not available\n");
           return OSD_NOT_OK;
        }    
	vga_setmode(G320x200x256);
	gl_setcontextvga(G320x200x256);
	video_mem  = vga_getgraphmem();
	video_mem += ((screen_height-visual_height) /2) * screen_width;
	video_mem +=  (screen_width -visual_width ) /2;
	set_video_mode();
   }
   else  
   {
        if (visual_width <= 320 && visual_height <= 240)
        {
           if (!vga_hasmode(G320x240x256))
           {
              fprintf(stderr_file,"Svgalib: 320x240x256colors not available\n");
              return OSD_NOT_OK;
           }    
	   vga_setmode(G320x240x256);
	   gl_setcontextvga(G320x240x256);
           screen_width=320;
           screen_height=240;
           planar_mode=TRUE;
        }
        else if (visual_width <= 320 && visual_height <= 400)
        {
           if (!vga_hasmode(G320x400x256))
           {
              fprintf(stderr_file,"Svgalib: 320x400x256colors not available\n");
              return OSD_NOT_OK;
           }    
	   vga_setmode(G320x400x256);
	   gl_setcontextvga(G320x400x256);
           screen_width=320;
           screen_height=400;
           planar_mode=TRUE;
        }
        else if (visual_width <= 640 && visual_height <= 480)
#else  /* ifndef linux_powerpc */
        if (visual_width <= 640 && visual_height <= 480)
#endif /* ifndef linux_powerpc */
        {
           if (!vga_hasmode(G640x480x256))
           {
              fprintf(stderr_file,"Svgalib: 640x480x256colors not available\n");
              return OSD_NOT_OK;
           }    
	   vga_setmode(G640x480x256);
	   gl_setcontextvga(G640x480x256);
           screen_width=640;
           screen_height=480;
        }
        else if (visual_width <= 800 && visual_height <= 600)
        {
           if (!vga_hasmode(G800x600x256))
           {
              fprintf(stderr_file,"Svgalib: 800x600x256colors not available\n");
              return OSD_NOT_OK;
           }    
	   vga_setmode(G800x600x256);
	   gl_setcontextvga(G800x600x256);
           screen_width=800;
           screen_height=600;
        }
        else if (visual_width <= 1024 && visual_height <= 768)
        {
           if (!vga_hasmode(G1024x768x256))
           {
              fprintf(stderr_file,"Svgalib: 1024x768x256colors not available\n");
              return OSD_NOT_OK;
           }    
	   vga_setmode(G1024x768x256);
	   gl_setcontextvga(G1024x768x256);
           screen_width=1024;
           screen_height=768;
        }
        else
        {
           fprintf(stderr_file,"Svgalib: requested size too big: %d x %d\n",visual_width,visual_height);
           return OSD_NOT_OK;
        }
#ifndef linux_powerpc        
   }
#endif
   
   screen_startx = (screen_width - visual_width) / 2;
   screen_starty = (screen_height - visual_height) / 2;
   
#if (__GLIBC__ > 1)
   memset(&sig2sigset,0,sizeof(sigset_t));
   sig2handler.sa_mask=sig2sigset;
#else	
   sig2handler.sa_mask=0;
#endif
   sig2handler.sa_handler=Sig2HandlerFunction;
   sig2handler.sa_flags=0;
	
   sigaction(SIGUSR2, &sig2handler, &oldsig2handler);
   
   /* set the palette to black */
   for (a=0; a < 256; a++) gl_setpalettecolor(a, 0, 0, 0);

   /* init the keyboard */
   keyboard_init();
   keyboard_translatekeys (TRANSLATE_CURSORKEYS | TRANSLATE_DIAGONAL);
   key=keyboard_getstate();

   /* 
      not sure if this is the best site but mouse init routine must be 
      called after video initialization...
   */   
   if(use_mouse)
   {
	vga_setmousesupport(TRUE);
	svgalib_mouse_init();
   }
   
#ifdef USE_TIMER
    /* svgalib catches the timer alarm. so we have to arm it after all
       other svgalib initialising. */
    if (play_sound)
    {
       if(start_timer()==OSD_NOT_OK)
       {
          osd_close_display();
          return OSD_NOT_OK;
       }
    }
#endif
   
   return OSD_OK;
}

/*
 * Alloc global symbol totalcolors colors, not needed under
 * svgalib, under svgalib we always have access to all available colors
 */
void sysdep_alloc_palette(void)
{
}

/* shut up the display */
void osd_close_display(void)
{
   sigaction(SIGUSR2, &oldsig2handler, NULL);
   vga_setmode(TEXT);
   osd_free_bitmap(bitmap);
}

void sysdep_modify_pen(int pen,unsigned char red, unsigned char green, unsigned char blue)
{
   gl_setpalettecolor(pen,(red>>2),(green>>2),(blue>>2));
}

void osd_mark_dirty(int x1, int y1, int x2, int y2, int ui)
{
	if (use_layer)	/* GfxLayer dirty handling */
	   layer_mark_rectangle_dirty_norotate(Machine->dirtylayer,x1,x2,y1,y2);
}

/* Update the display. */
void sysdep_update_display(void)
{
#ifndef linux_powerpc
   static int page=0;
   if(video_mode)
   {
     int i;
     unsigned char *dest = video_mem;
     for (i=visual.min_y; i<=visual.max_y; i++, dest+=screen_width)
        memcpy(dest, bitmap->line[i] + visual.min_x, visual_width);
   }  
   else if (planar_mode) /* bummer */
   {
     /* use page flipping otherwise the screen tears in planar modes */
     if (page) page=0;
      else page=131072;
     vga_copytoplanar256(bitmap->line[visual.min_y] + visual.min_x,
        ((bitmap->width + 7) & ~7) + 16,
        (screen_starty*screen_width + screen_startx + page)/4, 
         screen_width/4, visual_width, visual_height);
     vga_setdisplaystart(page);
   }
   else
#endif /* ifndef linux_powerpc */
     gl_putboxpart(screen_startx, screen_starty, visual_width, visual_height,
                   ((bitmap->width + 7) & ~7) + 16, bitmap->height + 16,
                   bitmap->_private, visual.min_x + 8 , visual.min_y  + 8 );
                /* Note: we use rounded bitmap->width, as used in 
                   osd_create_bitmap, this fixes the scewing bug in 
                   tempest & others */
}

void sysdep_clear_screen(void)
{
#ifndef linux_powerpc
   if (video_mode)
      memset(vga_getgraphmem(), 0, screen_width*screen_height);
   else if (planar_mode)
      return; /* FIXME: clear_screen not yet supported in planar mode */
   else
#endif   
         gl_fillbox( 0, 0, screen_width, screen_height, 0);
}

int sysdep_keyboard_init(void)
{
   /* don't do anything here instead init the keyboard in osd_create_display */
   /* this is done to make sure we won't exit unexpectedly (when failed to   */
   /* create the display without restoring the keyboard ) (dirty hack)       */
   return OSD_OK;   
}

void sysdep_keyboard_close(void)
{
   keyboard_close();
}

int sysdep_mapkey(int from, int to) 
{
	return OSD_OK;
}

#endif /* ifdef svgalib */
