/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB

   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.

   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.	The License grants you the right to
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/*
  The priviliges are saves in the following tables:
  mysql/user	 ; super user who are allowed to do almoust anything
  mysql/host	 ; host priviliges. This is used if host is empty in mysql/db.
  mysql/db	 ; database privileges / user

  data in tables is sorted according to how many not-wild-cards there is
  in the relevant fields. Empty strings comes last.
*/

#include "mysql_priv.h"
#include <m_ctype.h>
#include <stdarg.h>

/*
 ACL_HOST is used if no host is specified
 */

class ACL_ACCESS {
public:
  ulong sort;
  uint access;
};

class ACL_HOST :public ACL_ACCESS
{
public:
  char *host,*db;
};

class ACL_USER :public ACL_ACCESS
{
public:
  char *host,*user,*password;
  ulong salt[2];
};

class ACL_DB :public ACL_ACCESS
{
public:
  char *host,*user,*db;
};


static DYNAMIC_ARRAY acl_hosts,acl_users,acl_dbs;
static MEM_ROOT mem;
static bool initilized=0;

static uint get_access(TABLE *form,uint fieldnr);
static int acl_compare(ACL_ACCESS *a,ACL_ACCESS *b);
static ulong get_sort(uint count,...);

int acl_init(void)
{
  THD  *thd;
  TABLE_LIST tables[3];
  int error;
  TABLE *table;
  READ_RECORD read_record_info;
  DBUG_ENTER("acl_init");

  if (!(thd=new THD))
    DBUG_RETURN(1);
  thd->version=refresh_version;
  thd->current_tablenr=0;
  thd->open_tables=0;
  thd->db=my_strdup("mysql",MYF(0));
  tables[0].name=tables[0].real_name="host";
  tables[1].name=tables[1].real_name="user";
  tables[2].name=tables[2].real_name="db";
  tables[0].next=tables+1;
  tables[1].next=tables+2;
  tables[2].next=0;
  tables[0].flags=tables[1].flags=tables[2].flags=0; // Open for read
  tables[0].db=tables[1].db=tables[2].db=0;

  if (open_tables(thd,tables))
  {
    close_thread_tables(thd);
    delete thd;
    DBUG_RETURN(1);
  }
  TABLE *ptr[3];				// Lock tables for quick update
  ptr[0]= tables[0].table;
  ptr[1]= tables[1].table;
  ptr[2]= tables[2].table;
  MYSQL_LOCK *lock=mysql_lock_tables(thd,ptr,3);
  if (!lock)
  {
    close_thread_tables(thd);
    delete thd;
    DBUG_RETURN(1);
  }

  init_sql_alloc(&mem,1024);
  init_read_record(&read_record_info,table= tables[0].table,NULL);
  VOID(init_dynamic_array(&acl_hosts,sizeof(ACL_HOST),20,50));
  while ((error=read_record_info.read_record(&read_record_info)) <= 0)
  {
    if (!error)
    {
      ACL_HOST host;
      host.host=get_field(&mem, table,0);
      host.db=get_field(&mem, table,1);
      host.access=get_access(table,2);
      host.sort=get_sort(2,host.host,host.db);
      VOID(push_dynamic(&acl_hosts,(gptr) &host));
    }
  }
  qsort((gptr) dynamic_element(&acl_hosts,0,ACL_HOST*),acl_hosts.elements,
	sizeof(ACL_HOST),(qsort_cmp) acl_compare);
  end_read_record(&read_record_info);
  freeze_size(&acl_hosts);

  init_read_record(&read_record_info,table=tables[1].table,NULL);
  VOID(init_dynamic_array(&acl_users,sizeof(ACL_USER),50,100));
  if (table->field[2]->field_length == 8 &&
      protocol_version == PROTOCOL_VERSION)
  {
    fprintf(stderr,
	    "Old 'user' table. (Check README or the Reference manual). Continuing --old-protocol\n");
    protocol_version=9;
  }

  while ((error=read_record_info.read_record(&read_record_info)) <= 0)
  {
    if (!error)
    {
      ACL_USER user;
      uint length=0;
      user.host=get_field(&mem, table,0);
      user.user=get_field(&mem, table,1);
      user.password=get_field(&mem, table,2);
      if (user.password && (length=strlen(user.password)) == 8 &&
	  protocol_version == PROTOCOL_VERSION)
      {
	fprintf(stderr,
		"Found old style password for user '%s'. Restart using --old-protocol\n",
		user.user ? user.user : "");
	{
	  close_thread_tables(thd);
	  delete thd;
	  DBUG_RETURN(1);
	}
      }
      if (length % 8)			// This holds true for passwords
      {
	fprintf(stderr,
		"Found wrong password for user: '%s@%s'; Removing user\n",
		user.user ? user.user : "",
		user.host ? user.host : "");
	continue;
      }
      get_salt_from_password(user.salt,user.password);
      user.access=get_access(table,3);
      user.sort=get_sort(2,user.host,user.user);
      VOID(push_dynamic(&acl_users,(gptr) &user));
    }
  }
  qsort((gptr) dynamic_element(&acl_users,0,ACL_USER*),acl_users.elements,
	sizeof(ACL_USER),(qsort_cmp) acl_compare);
  end_read_record(&read_record_info);
  freeze_size(&acl_users);

  init_read_record(&read_record_info,table=tables[2].table,NULL);
  VOID(init_dynamic_array(&acl_dbs,sizeof(ACL_DB),50,100));
  while ((error=read_record_info.read_record(&read_record_info)) <= 0)
  {
    if (error == 0)
    {
      ACL_DB db;
      db.host=get_field(&mem, table,0);
      db.db=get_field(&mem, table,1);
      db.user=get_field(&mem, table,2);
      db.access=get_access(table,3);
      db.sort=get_sort(3,db.host,db.db,db.user);
      VOID(push_dynamic(&acl_dbs,(gptr) &db));
    }
  }
  qsort((gptr) dynamic_element(&acl_dbs,0,ACL_DB*),acl_dbs.elements,
	sizeof(ACL_DB),(qsort_cmp) acl_compare);
  end_read_record(&read_record_info);
  freeze_size(&acl_dbs);

  mysql_unlock_tables(lock);
  thd->version--;		/* Force close of tables to free memory */
  close_thread_tables(thd);
  delete thd;
  initilized=1;
  DBUG_RETURN(0);
}


void acl_free(void)
{
  free_root(&mem);
  delete_dynamic(&acl_hosts);
  delete_dynamic(&acl_users);
  delete_dynamic(&acl_dbs);
}

	/* Reload acl list if possible */

void acl_reload(void)
{
  DYNAMIC_ARRAY old_acl_hosts,old_acl_users,old_acl_dbs;
  MEM_ROOT old_mem;

  if (current_thd && current_thd->locked_tables)
  {					// Can't have locked tables here
    current_thd->lock=current_thd->locked_tables;
    current_thd->locked_tables=0;
    close_thread_tables(current_thd);
  }
  VOID(pthread_mutex_lock(&LOCK_Acl));

  old_acl_hosts=acl_hosts;
  old_acl_users=acl_users;
  old_acl_dbs=acl_dbs;
  old_mem=mem;

  if (acl_init())
  {					/* Error. Revert to old list */
    acl_free();
    acl_hosts=old_acl_hosts;
    acl_users=old_acl_users;
    acl_dbs=old_acl_dbs;
    mem=old_mem;
  }
  else
  {
    free_root(&old_mem);
    delete_dynamic(&old_acl_hosts);
    delete_dynamic(&old_acl_users);
    delete_dynamic(&old_acl_dbs);
  }
  VOID(pthread_mutex_unlock(&LOCK_Acl));
}


/* Get all access bits from table after fieldnr */

static uint get_access(TABLE *form,uint fieldnr)
{
  uint access=0,bit;
  char buff[2];
  String res(buff,sizeof(buff));
  Field **pos;

  for (pos=form->field+fieldnr,bit=1 ; *pos ; pos++ , bit<<=1)
  {
    (*pos)->val_str(&res);
    if (toupper(res[0]) == 'Y')
      access|=bit;
  }
  return access;
}


/*
 return a number with if sorted put string in this order:
 no wildcards
 wildcards
 empty string
 */

static ulong get_sort(uint count,...)
{
  va_list args;
  va_start(args,count);
  ulong sort=0;

  while (count--)
  {
    char *str=va_arg(args,char*);
    uint chars=0,wild=0;

    if (str)
    {
      for (; *str ; str++)
      {
	if (*str == wild_many || *str == wild_one || *str == wild_prefix)
	  wild++;
	else
	  chars++;
      }
    }
    sort= (sort << 8) + (wild ? 1 : chars ? 2 : 0);
  }
  va_end(args);
  return sort;
}


static int acl_compare(ACL_ACCESS *a,ACL_ACCESS *b)
{
  if (a->sort > b->sort)
    return -1;
  if (a->sort < b->sort)
    return 1;
  return 0;
}


/* Get master privilges for user (priviliges for all tables) */


uint acl_getroot(const char *host, const char *ip, const char *user,
		 const char *password,const char *message,char **priv_user,
		 bool old_ver)
{
  uint user_access=NO_ACCESS;
  *priv_user=(char*) user;

  if (!initilized)
    return (uint) ~NO_ACCESS;			// If no data allow anything
  VOID(pthread_mutex_lock(&LOCK_Acl));

  /*
    Get possible access from user_list. This is or:ed to others not
    fully specified
  */
  for (uint i=0 ; i < acl_users.elements ; i++)
  {
    ACL_USER *acl_user=dynamic_element(&acl_users,i,ACL_USER*);
    if (!acl_user->user || !strcmp(user,acl_user->user))
    {
      if (!acl_user->host ||
	  (host && !wild_case_compare(host,acl_user->host)) ||
	  (ip && !wild_compare(ip,acl_user->host)))
      {
	if (!acl_user->password && !*password ||
	    (acl_user->password && *password &&
	     !check_scramble(password,message,acl_user->salt,
			     (my_bool) old_ver)))
	{
	  user_access=acl_user->access;
	  if (!acl_user->user)
	    *priv_user="";		// Change to anonymious user
	  break;
	}
#ifndef ALLOW_DOWNGRADE_OF_USERS
	break;				// Wrong password breaks loop
#endif
      }
    }
  }
  VOID(pthread_mutex_unlock(&LOCK_Acl));
  return user_access;
}


uint acl_get(const char *host, const char *ip, const char *user,
	     const char *db)
{
  uint host_access,db_access,i;
  db_access=0; host_access= ~0;

  VOID(pthread_mutex_lock(&LOCK_Acl));

  /*
    Check if there is some access rights for database and user
  */
  for (i=0 ; i < acl_dbs.elements ; i++)
  {
    ACL_DB *acl_db=dynamic_element(&acl_dbs,i,ACL_DB*);
    if (!acl_db->user || !strcmp(user,acl_db->user))
    {
      if (!acl_db->host ||
	  (host && !wild_case_compare(host,acl_db->host)) ||
	  (ip && !wild_compare(ip,acl_db->host)))
      {
	if (!acl_db->db || !wild_compare(db,acl_db->db))
	{
	  db_access=acl_db->access;
	  if (acl_db->host)
	    goto exit;				// Fully specified. Take it
	  break;
	}
      }
    }
  }
  if (!db_access)
    goto exit;					// Can't be better

  /*
    No host specified for user. Get hostdata from host table
  */
  host_access=0;				// Host must be found
  for (i=0 ; i < acl_hosts.elements ; i++)
  {
    ACL_HOST *acl_host=dynamic_element(&acl_hosts,i,ACL_HOST*);
    if (!acl_host->host ||
	(host && !wild_case_compare(host,acl_host->host)) ||
	(ip && !wild_compare(ip,acl_host->host)))
    {
      if (!acl_host->db || !wild_compare(db,acl_host->db))
      {
	host_access=acl_host->access;		// Fully specified. Take it
	break;
      }
    }
  }
exit:
  VOID(pthread_mutex_unlock(&LOCK_Acl));
  return (db_access & host_access);
}

/* check if there are any possible matching entries for this host */

uint acl_getconnectright(const char *host, const char *ip)
{
  uint user_access=NO_ACCESS;

  if (!initilized)
    return (uint) ~NO_ACCESS;			// If no data allow anything
  VOID(pthread_mutex_lock(&LOCK_Acl));

  /*
    Get possible access from user_list based on host name/ip
  */
  for (uint i=0 ; i < acl_users.elements ; i++)
  {
    ACL_USER *acl_user=dynamic_element(&acl_users,i,ACL_USER*);
    if (!acl_user->host ||
	(host && !wild_case_compare(host,acl_user->host)) ||
	(ip && !wild_compare(ip,acl_user->host)))
    {
      user_access=(uint) ~NO_ACCESS;
      break;
    }
  }
  VOID(pthread_mutex_unlock(&LOCK_Acl));
  return user_access;
}

int wild_case_compare(const char *str,const char *wildstr)
{
  reg3 int flag;
  DBUG_ENTER("wild_case_compare");

  while (*wildstr)
  {
    while (*wildstr && *wildstr != wild_many && *wildstr != wild_one)
    {
      if (*wildstr == wild_prefix && wildstr[1])
	wildstr++;
      if (toupper(*wildstr++) != toupper(*str++)) DBUG_RETURN(1);
    }
    if (! *wildstr ) DBUG_RETURN (*str != 0);
    if (*wildstr++ == wild_one)
    {
      if (! *str++) DBUG_RETURN (1);	/* One char; skipp */
    }
    else
    {						/* Found '*' */
      if (!*wildstr) DBUG_RETURN(0);		/* '*' as last char: OK */
      flag=(*wildstr != wild_many && *wildstr != wild_one);
      do
      {
	if (flag)
	{
	  char cmp;
	  if ((cmp= *wildstr) == wild_prefix && wildstr[1])
	    cmp=wildstr[1];
	  cmp=toupper(cmp);
	  while (*str && toupper(*str) != cmp)
	    str++;
	  if (!*str) DBUG_RETURN (1);
	}
	if (wild_case_compare(str,wildstr) == 0) DBUG_RETURN (0);
      } while (*str++);
      DBUG_RETURN(1);
    }
  }
  DBUG_RETURN (*str != '\0');
}
