/* file: "os.c" */

/* Copyright (C) 1994-1998 by Marc Feeley, All Rights Reserved. */

/*
 * This module implements the operating system specific routines
 * including:
 *
 *  - low-level memory allocation
 *  - user interrupt handling
 *  - timer interrupt handling
 *  - OS event handling
 *  - access to environment variables
 *  - shell command
 *  - dynamic loading
 *  - dynamic C compilation
 *  - elapsed real time
 *  - elapsed cpu time (user time and system time)
 *  - virtual memory statistics
 *  - filesystem path expansion
 *  - formatting of source code position
 *  - low-level I/O
 *  - fatal error handling
 */

#define ___INCLUDED_FROM_OS
#define ___VERSION 21
#include "gambit.h"

#include "os.h"
#include "setup.h"
#include "mem.h"
#include "c_intf.h"

/*---------------------------------------------------------------------------*/

#ifdef USE_MACOS

/* Which features are available. */

___HIDDEN int has_IdleUpdate = 0;
___HIDDEN int has_WaitNextEvent = 0;
___HIDDEN int has_OSDispatch = 0;
___HIDDEN int has_FindFolder = 0;
___HIDDEN int has_AliasMgr = 0;
___HIDDEN int has_AppleEvents = 0;

#define test_bit(n,i) (n&(1<<i))

___HIDDEN TrapType get_trap_type ___P((short trap_num),(trap_num)
short trap_num;)
{
  /* OS traps start with A0, Tool traps with A8 or AA. */

  if (trap_num & 0x0800)
    return ToolTrap;
  else
    return OSTrap;
}

___HIDDEN short nb_toolbox_traps ___PVOID
{
  /* InitGraf (trap $A86E) is always implemented. */

  if (NGetTrapAddress (0xA86E, ToolTrap) == NGetTrapAddress (0xAA6E, ToolTrap))
    return (0x200);
  else
    return (0x400);
}

___HIDDEN Boolean trap_exists ___P((short trap_num),(trap_num)
short trap_num;)
{
  TrapType typ = get_trap_type (trap_num);
  if ((typ == ToolTrap) && ((trap_num &= 0x07FF) >= nb_toolbox_traps ()))
    return false;
  return (NGetTrapAddress (_Unimplemented, ToolTrap) !=
          NGetTrapAddress (trap_num, typ));
}

/* String conversion utilities for MACOS. */

___HIDDEN Boolean c2pascal
   ___P((char *cstr, StringPtr pstr, int max_length),
        (cstr, pstr, max_length)
char *cstr;
StringPtr pstr;
int max_length;)

{
  StringPtr p1 = pstr+1;
  char *p2 = cstr;
  while (max_length > 0 && *p2 != '\0')
    {
       *p1++ = *p2++;
       max_length--;
    }
  if (*p2 != '\0')
    return 0;
  else
    {
      pstr[0] = p2-cstr;
      return 1;
    }
}

___HIDDEN Boolean pascal2c
   ___P((StringPtr pstr, char *cstr, int max_length),
        (pstr, cstr, max_length)
StringPtr pstr;
char *cstr;
int max_length;)
{
  char *p1 = cstr;
  StringPtr p2 = pstr+1;
  int len = pstr[0];
  if (len > max_length)
    return 0;
  else
    {
      while (len-- > 0)
        *p1++ = *p2++;
      *p1++ = '\0';
      return 1;
    }
}

#define DIR_SEPARATOR1 ':'
#define PARENT_HOP ":"

#define DIR_SEPARATOR(c)((c) == DIR_SEPARATOR1)
#define SEPARATOR(c)DIR_SEPARATOR(c)

___HIDDEN OSErr make_ResolvedFSSpec
   ___P((short vol, long dir, ConstStr255Param path, FSSpec *spec),
        (vol, dir, path, spec)
short vol;
long dir;
ConstStr255Param path;
FSSpec *spec;)
{
  OSErr err;
  Str255 name;
  StringPtr start = (StringPtr)path+1;
  StringPtr end = start + path[0];
  StringPtr p1 = start;
  StringPtr p2 = name+1;
  CInfoPBRec pb;
  Boolean is_folder, is_aliased;

  if (!has_AliasMgr)
    return unimpErr;

  spec->vRefNum = vol;
  spec->parID = dir;

  do
    {
      p2 = name+1;
      while (p1 < end && DIR_SEPARATOR(*p1))  /* copy leading ':'s */
        *p2++ = *p1++;
      while (p1 < end && !DIR_SEPARATOR(*p1)) /* copy name that follows */
        *p2++ = *p1++;
      if (p1 < end && DIR_SEPARATOR(*p1))     /* end with a ':' if folder */
        *p2++ = DIR_SEPARATOR1;
      name[0] = p2 - (name+1);

      err = FSMakeFSSpec (spec->vRefNum, spec->parID, name, spec);
      if (err == fnfErr && p1 == end)
        return noErr;
      if (err != noErr)
        return err;

      if ((err = ResolveAliasFile (spec, 1, &is_folder, &is_aliased)) != noErr)
        return err;
      if (is_folder)
        {
          pb.dirInfo.ioNamePtr = spec->name;
          pb.dirInfo.ioVRefNum = spec->vRefNum;
          pb.dirInfo.ioDrDirID = spec->parID;
          pb.dirInfo.ioFDirIndex = 0;
          if ((err = PBGetCatInfoSync (&pb)) != noErr)
            return err;
          spec->parID = pb.hFileInfo.ioDirID;
          spec->name[0] = 0;
        }
      else if (p1 < end)
        return dirNFErr;
    } while (p1 < end);

  return noErr;
}

___HIDDEN OSErr ResolvedFSSpec_to_fullpath
   ___P((FSSpec *spec, StringPtr fullpath),
        (spec, fullpath)
FSSpec *spec;
StringPtr fullpath;)
{
  OSErr err;
  int i;
  Str255 result;
  StringPtr p = result + sizeof(result);
  CInfoPBRec pb;
  Str31 name;

  for (i = spec->name[0]; i > 0; i--)
    *--p = spec->name[i];

  pb.dirInfo.ioNamePtr = name;
  pb.dirInfo.ioVRefNum = spec->vRefNum;
  pb.dirInfo.ioDrParID = spec->parID;
  pb.dirInfo.ioFDirIndex = -1;

  do
    {
      pb.dirInfo.ioDrDirID = pb.dirInfo.ioDrParID;
      if ((err = PBGetCatInfoSync (&pb)) != noErr)
        return err;
      if (p-name[0]-1 < result)
        return bdNamErr; /* file name is too long */
      *--p = DIR_SEPARATOR1;
      for (i = name[0]; i > 0; i--)
        *--p = name[i];
    } while (pb.dirInfo.ioDrDirID != fsRtDirID);

  i = result + sizeof(result) - p;
  *fullpath++ = i;
  while (i > 0)
    {
      *fullpath++ = *p++;
      i--;
    }

  return noErr;
}

___HIDDEN int path_expand_to_absolute
   ___P((char *path, char *new_path, long max_length),
        (path, new_path, max_length)
char *path;
char *new_path;
long max_length)
{
  int result = 0;
  FSSpec spec;
  short vol;
  long dir;
  char tmp[___PATH_MAX_LENGTH+1];
  Str255 ppath;

  if (path[0] == '~')
    {
      if (path[1] == '~')
        {
          /* "~~" or "~~:xxx..." */

          int i = 0;
          int j = 0;
          int sep = 0;
          char *gambc_dir;

          if (!has_FindFolder)
            goto ret;

          if (path[2]!='\0' && !DIR_SEPARATOR(path[2]))
            goto ret;

#ifdef ___GAMBCDIR
          gambc_dir = ___GAMBCDIR;
#else
          gambc_dir = ":Gambit-C:gambc";
#endif

          i += 2;

          while (*gambc_dir != '\0')
            if (j < ___PATH_MAX_LENGTH)
              {
                tmp[j] = *gambc_dir++;
                j++;
              }
            else
              goto ret;

          while (path[i] != '\0')
            if (j < ___PATH_MAX_LENGTH)
              {
                if (DIR_SEPARATOR(path[i]))
                  sep = 1;
                tmp[j++] = path[i++];
              }
            else
              goto ret;

          if (!sep)
            if (j < ___PATH_MAX_LENGTH)
              tmp[j++] = DIR_SEPARATOR1;
            else
              goto ret;

          tmp[j] = '\0';
          path = tmp;

          if (FindFolder (kOnSystemDisk, kPreferencesFolderType, 0, &vol, &dir) != noErr)
            goto ret;
        }
      else if (path[1]!='\0' && !DIR_SEPARATOR(path[1]))
        {
          /* "~user" or "~user:xxx..." */

          goto ret; /* no equivalent on Macintosh */
        }
      else
        {
          /* "~" or "~:xxx..." */

          path++;
          vol = 0; /* use default volume and directory
                      (folder containing application) */
          dir = 0;
        }
    }
  else
    {
      vol = 0; /* use default volume and directory
                  (folder containing application) */
      dir = 0;
    }

  if (!c2pascal (path, ppath, 255) ||
      make_ResolvedFSSpec (vol, dir, ppath, &spec) != noErr ||
      ResolvedFSSpec_to_fullpath (&spec, ppath) != noErr ||
      !pascal2c (ppath, new_path, max_length))
    goto ret;

  result = 1;

 ret:

  return result;
}

___HIDDEN OSErr copy_file_sectors
   ___P((short src_refnum, short dst_refnum),(src_refnum, dst_refnum)
short src_refnum;
short dst_refnum;)
{
  OSErr err, err2;
  char buf[2048];
  long count, count2;

  do
    {
      count = sizeof (buf);
      err = FSRead (src_refnum, &count, buf);
      if (err != noErr && err != eofErr)
        return err;
      count2 = count;
      err2 = err;
      err = FSWrite (dst_refnum, &count2, buf);
      if (err != noErr || count != count2)
        return err;
    } while (err2 != eofErr);
}

___HIDDEN OSErr copy_file
   ___P((FSSpec src_spec, FSSpec dst_spec),(src_spec, dst_spec)
FSSpec src_spec;
FSSpec dst_spec;)
{
  OSErr err, err2;
  short src_refnum, dst_refnum;
  FInfo src_info;

  if (((err = FSpDelete (&dst_spec)) == noErr || err == fnfErr) &&
      (err = FSpGetFInfo (&src_spec, &src_info)) == noErr &&
      (err = FSpCreate (&dst_spec, '????', '????', 0)) == noErr)
    {
      src_info.fdFlags = src_info.fdFlags & ~kHasBeenInited;
      if ((err = FSpSetFInfo (&dst_spec, &src_info) == noErr) &&
          (err = FSpOpenRF (&src_spec, fsRdPerm, &src_refnum) == noErr))
        {
          if ((err = FSpOpenRF (&dst_spec, fsWrPerm, &dst_refnum)) == noErr)
            {
              err = copy_file_sectors (src_refnum, dst_refnum);
              err2 = FSClose (dst_refnum);
              if (err == noErr)
                err = err2;
            }
          err2 = FSClose (src_refnum);
          if (err == noErr)
            err = err2;
          if (err == noErr &&
              (err = FSpOpenDF (&src_spec, fsRdPerm, &src_refnum) == noErr))
            {
              if ((err = FSpOpenDF (&dst_spec, fsWrPerm, &dst_refnum)) == noErr)
                {
                  err = copy_file_sectors (src_refnum, dst_refnum);
                  err2 = FSClose (dst_refnum);
                  if (err == noErr)
                    err = err2;
                }
              err2 = FSClose (src_refnum);
              if (err == noErr)
                err = err2;
            }
        }
      if (err != noErr)
        FSpDelete (&dst_spec);
    }

  return err;
}

#endif


/*---------------------------------------------------------------------------*/

/* Low-level memory allocation. */

void *___alloc_mem ___P((unsigned long bytes),(bytes)
unsigned long bytes;)
{
#ifdef USE_MACOS
  if (has_OSDispatch)
    {
      OSErr err;
      Ptr p;
      Handle h = TempNewHandle (sizeof (Handle) + bytes, &err);
      if (err != noErr || h == 0)
        return 0;
      HLock (h);
      p = *h;
      *(Handle*)p = h;
      return p + sizeof (Handle);
    }
  else
    return malloc (bytes);
#else
  return malloc (bytes);
#endif
}

void ___free_mem ___P((void *ptr),(ptr)
void *ptr;)
{
#ifdef USE_MACOS
  if (has_OSDispatch)
    {
      OSErr err;
      Handle h = *(Handle*)((Ptr)ptr - sizeof (Handle));
      HUnlock (h);
      TempDisposeHandle (h, &err);
    }
  else
    free (ptr);
#else
  free (ptr);
#endif
}


/*---------------------------------------------------------------------------*/

/* User interrupt handling. */

#ifndef USE_sigaction_SIGINT
#ifndef USE_signal_SIGINT
#ifndef USE_dos_setvect_23h

___HIDDEN void setup_user_interrupt_handling ___PVOID
{
}

___HIDDEN void cleanup_user_interrupt_handling ___PVOID
{
}

___HIDDEN void disable_user_interrupt ___PVOID
{
}

___HIDDEN void enable_user_interrupt ___PVOID
{
}

#endif
#endif
#endif

/*   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   */

#ifdef USE_sigaction_SIGINT

___HIDDEN void user_interrupt_handler ___P((int sig),(sig)
int sig;)
{
  ___raise_interrupt (___INTR_USER);
}

___HIDDEN void setup_user_interrupt_handling ___PVOID
{
  struct sigaction act;

  act.sa_handler = user_interrupt_handler;
  act.sa_flags = 0;
#ifdef SA_INTERRUPT
  act.sa_flags |= SA_INTERRUPT;
#endif
  sigemptyset (&act.sa_mask);
  sigaction (SIGINT, &act, 0);
}

___HIDDEN void cleanup_user_interrupt_handling ___PVOID
{
  struct sigaction act;

  act.sa_handler = SIG_DFL;
  act.sa_flags = 0;
  sigemptyset (&act.sa_mask);
  sigaction (SIGINT, &act, 0);
}

___HIDDEN void disable_user_interrupt ___PVOID
{
  sigset_t mask;

  sigemptyset (&mask);
  sigaddset (&mask, SIGINT);
  sigprocmask (SIG_BLOCK, &mask, 0);
}

___HIDDEN void enable_user_interrupt ___PVOID
{
  sigset_t mask;

  sigemptyset (&mask);
  sigaddset (&mask, SIGINT);
  sigprocmask (SIG_UNBLOCK, &mask, 0);
}

#endif

/*   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   */

#ifdef USE_signal_SIGINT

___HIDDEN void user_interrupt_handler ___P((int sig),(sig)
int sig;)
{
  signal (SIGINT, user_interrupt_handler);
  ___raise_interrupt (___INTR_USER);
}

___HIDDEN void setup_user_interrupt_handling ___PVOID
{
  signal (SIGINT, user_interrupt_handler);
}

___HIDDEN void cleanup_user_interrupt_handling ___PVOID
{
  signal (SIGINT, SIG_DFL);
}

___HIDDEN void disable_user_interrupt ___PVOID
{
  signal (SIGINT, SIG_IGN);
}

___HIDDEN void enable_user_interrupt ___PVOID
{
  signal (SIGINT, user_interrupt_handler);
}

#endif

/*   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   */

#ifdef USE_dos_setvect_23h

/* 23h = ctrl-c vector */
___HIDDEN void (__interrupt __far *prev_vector_23h) ___PVOID;

/* 1Bh = ctrl-break vector */
___HIDDEN void (__interrupt __far *prev_vector_1Bh) ___PVOID;

___HIDDEN void __interrupt __far user_interrupt_handler ___PVOID
{
  ___raise_interrupt (___INTR_USER);
}

___HIDDEN void setup_user_interrupt_handling ___PVOID
{
  prev_vector_23h = _dos_getvect (0x23);
  prev_vector_1Bh = _dos_getvect (0x1B);
  _dos_setvect (0x23, user_interrupt_handler);
  _dos_setvect (0x1B, user_interrupt_handler);
}

___HIDDEN void cleanup_user_interrupt_handling ___PVOID
{
  _dos_setvect (0x23, prev_vector_23h);
  _dos_setvect (0x1B, prev_vector_1Bh);
}

___HIDDEN void disable_user_interrupt ___PVOID
{
}

___HIDDEN void enable_user_interrupt ___PVOID
{
}

#endif


/*---------------------------------------------------------------------------*/

/* Timer interrupt handling. */

/*
 * Use the process-time timer unless only the real-time timer is
 * available (e.g. DJGPP).  Note that on some systems (e.g. MkLinux)
 * ITIMER_VIRTUAL is an enum type, not a macro.
 */

#ifdef ITIMER_VIRTUAL
#define TIMER_ITIMER ITIMER_VIRTUAL
#define TIMER_SIG SIGVTALRM
#else
#ifdef ITIMER_REAL
#define TIMER_ITIMER ITIMER_REAL
#define TIMER_SIG SIGALRM
#else
#define TIMER_ITIMER ITIMER_VIRTUAL
#define TIMER_SIG SIGVTALRM
#endif
#endif

#ifndef USE_sigaction_setitimer
#ifndef USE_signal_setitimer
#ifndef USE_dos_setvect_1Ch
#ifndef USE_DosStartTimer
#ifndef USE_VInstall
#ifndef USE_CreateThread
#ifndef USE_timeSetEvent

void ___set_timer_interval ___P((unsigned long msecs),(msecs)
unsigned long msecs;)
{
}

___HIDDEN void setup_timer_interrupt_handling ___PVOID
{
}

___HIDDEN void cleanup_timer_interrupt_handling ___PVOID
{
}

___HIDDEN void disable_timer_interrupt ___PVOID
{
}

___HIDDEN void enable_timer_interrupt ___PVOID
{
}

#endif
#endif
#endif
#endif
#endif
#endif
#endif

/*   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   */

#ifdef USE_sigaction_setitimer

___HIDDEN void timer_interrupt_handler ___P((int sig),(sig)
int sig;)
{
  ___raise_interrupt (___INTR_TIMER);
}

void ___set_timer_interval ___P((unsigned long msecs),(msecs)
unsigned long msecs;)
{
  struct sigaction act;
  struct itimerval tv;
  int secs = msecs/1000;
  int usecs = (msecs%1000)*1000;
  tv.it_interval.tv_sec  = secs;
  tv.it_interval.tv_usec = usecs;
  tv.it_value.tv_sec     = secs;
  tv.it_value.tv_usec    = usecs;
  setitimer (TIMER_ITIMER, &tv, 0); /* msecs==0 will turn timer off */
  if (msecs == 0)
    {
      act.sa_handler = SIG_DFL;
      act.sa_flags = 0;
      sigemptyset (&act.sa_mask);
      sigaction (TIMER_SIG, &act, 0);
    }
  else
    {
      act.sa_handler = timer_interrupt_handler;
      act.sa_flags = 0;
#ifdef SA_INTERRUPT
      act.sa_flags |= SA_INTERRUPT;
#endif
      sigemptyset (&act.sa_mask);
      sigaction (TIMER_SIG, &act, 0);
    }
}

___HIDDEN void setup_timer_interrupt_handling ___PVOID
{
}

___HIDDEN void cleanup_timer_interrupt_handling ___PVOID
{
  ___set_timer_interval (0);
}

___HIDDEN void disable_timer_interrupt ___PVOID
{
  sigset_t mask;

  sigemptyset (&mask);
  sigaddset (&mask, TIMER_SIG);
  sigprocmask (SIG_BLOCK, &mask, 0);
}

___HIDDEN void enable_timer_interrupt ___PVOID
{
  sigset_t mask;

  sigemptyset (&mask);
  sigaddset (&mask, TIMER_SIG);
  sigprocmask (SIG_UNBLOCK, &mask, 0);
}

#endif

/*   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   */

#ifdef USE_signal_setitimer

___HIDDEN void timer_interrupt_handler ___P((int sig),(sig)
int sig;)
{
  signal (TIMER_SIG, timer_interrupt_handler);
  ___raise_interrupt (___INTR_TIMER);
}

void ___set_timer_interval ___P((unsigned long msecs),(msecs)
unsigned long msecs;)
{
  struct itimerval tv;
  int secs = msecs/1000;
  int usecs = (msecs%1000)*1000;
  tv.it_interval.tv_sec  = secs;
  tv.it_interval.tv_usec = usecs;
  tv.it_value.tv_sec     = secs;
  tv.it_value.tv_usec    = usecs;
  setitimer (TIMER_ITIMER, &tv, 0); /* msecs==0 will turn timer off */
  if (msecs == 0)
    signal (TIMER_SIG, SIG_DFL);
  else
    signal (TIMER_SIG, timer_interrupt_handler);
}

___HIDDEN void setup_timer_interrupt_handling ___PVOID
{
}

___HIDDEN void cleanup_timer_interrupt_handling ___PVOID
{
  ___set_timer_interval (0);
}

___HIDDEN void disable_timer_interrupt ___PVOID
{
  signal (TIMER_SIG, SIG_IGN);
}

___HIDDEN void enable_timer_interrupt ___PVOID
{
  signal (TIMER_SIG, timer_interrupt_handler);
}

#endif

/*   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   */

#ifdef USE_dos_setvect_1Ch

/* 1Ch = timer tick vector */
___HIDDEN void (__interrupt __far *prev_vector_1Ch) ___PVOID;
___HIDDEN int timer_enabled;
___HIDDEN long timer_interval;
___HIDDEN long timer_countdown;

___HIDDEN void __interrupt __far timer_interrupt_handler ___PVOID
{
  if (timer_countdown > 1)
    timer_countdown--;
  else if (timer_countdown == 1 && timer_enabled)
    {
      timer_countdown = timer_interval;
      ___raise_interrupt (___INTR_TIMER);
    }
  _chain_intr (prev_vector_1Ch);
}

void ___set_timer_interval ___P((unsigned long msecs),(msecs)
unsigned long msecs;)
{
  timer_enabled = 0;
  timer_interval = (msecs*91+5000-1)/5000; /* 182 timer ticks = 10000 msecs */
  timer_countdown = timer_interval;
  if (msecs == 0)
    _dos_setvect (0x1C, prev_vector_1Ch);
  else
    {
      timer_enabled = 1;
      _dos_setvect (0x1C, timer_interrupt_handler);
    }
}

___HIDDEN void setup_timer_interrupt_handling ___PVOID
{
  prev_vector_1Ch = _dos_getvect (0x1C);
}

___HIDDEN void cleanup_timer_interrupt_handling ___PVOID
{
  ___set_timer_interval (0);
}

___HIDDEN void disable_timer_interrupt ___PVOID
{
  timer_enabled = 0;
}

___HIDDEN void enable_timer_interrupt ___PVOID
{
  timer_enabled = 1;
}

#endif

/*   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   */

#ifdef USE_DosStartTimer

___HIDDEN int    timer_hev     = 0;
___HIDDEN int    timer_htimer  = 0;
___HIDDEN int    timer_tid     = 0;
___HIDDEN int    timer_enabled = 0;
___HIDDEN HEV    hev    = 0;
___HIDDEN HTIMER htimer = 0;
___HIDDEN TID    tid    = 0;

___HIDDEN void APIENTRY timercode ___P((ULONG param),(param)
ULONG param;)
{
  ULONG pc = 0;
  while (1)
    {
      DosWaitEventSem (hev, SEM_INDEFINITE_WAIT);
      DosResetEventSem (hev, &pc);
      if (timer_enabled)
        {
          DosEnterCritSec ();
          ___raise_interrupt (___INTR_TIMER);
          DosExitCritSec ();
        }
    }
}

___HIDDEN void disable_timer_interrupt ___PVOID
{
  timer_enabled = 0;
}

___HIDDEN void enable_timer_interrupt ___PVOID
{
  timer_enabled = 1;
}

void ___set_timer_interval ___P((unsigned long msecs),(msecs)
unsigned long msecs;)
{
  ULONG res;
  if (!timer_hev || !timer_tid)
    return;
  if (timer_htimer)
    {
      DosStopTimer (htimer);
      timer_htimer = 0;
    }
  if (msecs && (res = DosStartTimer (msecs, (HSEM)hev, &htimer)) == NO_ERROR)
    timer_htimer = 1;
}

___HIDDEN void setup_timer_interrupt_handling ___PVOID
{
  if (DosCreateEventSem (0, &hev, DC_SEM_SHARED, FALSE) != NO_ERROR)
    return;
  timer_hev = 1;
  if (DosCreateThread (&tid, timercode, 0, CREATE_READY|STACK_COMMITTED, 4096)
      != NO_ERROR)
    return;
  timer_tid = 1;
  DosSetPriority (PRTYS_THREAD, PRTYC_NOCHANGE, PRTYD_MAXIMUM, tid);
  timer_enabled = 1;
}


___HIDDEN void cleanup_timer_interrupt_handling ___PVOID
{
  timer_enabled = 0;
  if (timer_htimer) { DosStopTimer (htimer);  timer_htimer = 0; }
  if (timer_tid)    { DosKillThread (tid);    timer_tid    = 0; }
  if (timer_hev)    { DosCloseEventSem (hev); timer_hev    = 0; }
}

#endif

/*   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   */

#ifdef USE_VInstall

___HIDDEN int timer_task_installed = 0;
___HIDDEN short timer_task_ticks;
___HIDDEN VBLTask timer_task;

___HIDDEN void timer_task_code ___PVOID
{
  ___raise_interrupt (___INTR_TIMER);
  timer_task.vblCount = timer_task_ticks;
}

#ifdef __POWERPC__

#define timer_task_proc timer_task_code

#else

___HIDDEN asm void timer_task_proc ___PVOID
{
    move.l a5,-(sp)
    move.l 0x904,a5
    jsr    timer_task_code
    move.l (sp)+,a5
    rts
}

#endif

void ___set_timer_interval ___P((unsigned long msecs),(msecs)
unsigned long msecs;)
{
  unsigned long ticks = msecs*3/50; /* convert to ticks */

  if (timer_task_installed)
    if (VRemove ((QElemPtr)&timer_task) != noErr)
      timer_task_installed = 0;

  if (!timer_task_installed && ticks > 0)
    {
      timer_task_ticks    = ticks;
      timer_task.qType    = vType;
      timer_task.vblAddr  = NewVBLProc (timer_task_proc);
      timer_task.vblCount = ticks;
      timer_task.vblPhase = 0;
      if (VInstall ((QElemPtr)&timer_task) == noErr)
        timer_task_installed = 1;
    }
}

___HIDDEN void setup_timer_interrupt_handling ___PVOID
{
}

___HIDDEN void cleanup_timer_interrupt_handling ___PVOID
{
  ___set_timer_interval (0);
}

___HIDDEN void disable_timer_interrupt ___PVOID
{
  ___set_timer_interval (0);
}

___HIDDEN void enable_timer_interrupt ___PVOID
{
  ___set_timer_interval (timer_task_ticks);
}

#endif

/*   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   */

#ifdef USE_CreateThread

___HIDDEN HANDLE timer_thread = NULL;
___HIDDEN unsigned long timer_thread_interval;
___HIDDEN int timer_thread_enabled;
___HIDDEN int timer_thread_alive;
___HIDDEN DWORD timer_thread_id;

___HIDDEN DWORD timer_thread_code ___P((LPDWORD lpdwParam),(lpdwParam)
LPDWORD lpdwParam;)
{
  while (timer_thread_alive)
    {
      Sleep (timer_thread_interval);
      ___raise_interrupt (___INTR_TIMER);
    }
  timer_thread_alive = 1; /* indicate termination */
  return 0;
}

void ___set_timer_interval ___P((unsigned long msecs),(msecs)
unsigned long msecs;)
{
  timer_thread_enabled = 0;
  timer_thread_alive = 0; /* request timer thread to terminate */

  if (timer_thread != NULL)
    {
      CloseHandle (timer_thread);
      timer_thread = NULL;
      while (!timer_thread_alive)
        Sleep (0); /* wait until timer thread actually terminates */
      timer_thread_alive = 0;
    }

  if (timer_thread == NULL && msecs > 0)
    {
      timer_thread_interval = msecs;
      timer_thread_enabled = 1;
      timer_thread_alive = 1;
      timer_thread =
        CreateThread (NULL,  /* no security attributes        */
                      1024,  /* small stack                   */
                      (LPTHREAD_START_ROUTINE) timer_thread_code,
                      NULL,  /* argument to thread function   */
                      0,     /* use default creation flags    */
                      &timer_thread_id);
    }
}

___HIDDEN void setup_timer_interrupt_handling ___PVOID
{
}

___HIDDEN void cleanup_timer_interrupt_handling ___PVOID
{
  ___set_timer_interval (0);
}

___HIDDEN void disable_timer_interrupt ___PVOID
{
  timer_thread_enabled = 0;
}

___HIDDEN void enable_timer_interrupt ___PVOID
{
  timer_thread_enabled = 1;
}

#endif

/*   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   */

#ifdef USE_timeSetEvent

___HIDDEN int timer_installed = 0;
___HIDDEN unsigned int timer_interval = 0;
___HIDDEN unsigned int timer_resolution;
___HIDDEN unsigned int timer_id;

___HIDDEN void CALLBACK timer_callback
   ___P((UINT timer_id,
         UINT message,
         DWORD user_data,
         DWORD dw1,
         DWORD dw2),
        (timer_id, message, user_data, dw1, dw2)
UINT timer_id;
UINT message;
DWORD user_data;
DWORD dw1;
DWORD dw2;)
{
  ___raise_interrupt (___INTR_TIMER);
}

void ___set_timer_interval ___P((unsigned long msecs),(msecs)
unsigned long msecs;)
{
  TIMECAPS tc;

  if (timer_installed)
    {
      timeKillEvent (timer_id);
      timeEndPeriod (timer_resolution);
      timer_installed = 0;
    }

  if (!timer_installed && msecs > 0)
    if (timeGetDevCaps (&tc, sizeof(TIMECAPS)) == TIMERR_NOERROR)
      {
        timer_interval = msecs;

        timer_resolution = 1;
        if (timer_resolution < tc.wPeriodMin)
          timer_resolution = tc.wPeriodMin;
        else if (timer_resolution > tc.wPeriodMax)
          timer_resolution = tc.wPeriodMax;

        timeBeginPeriod (timer_resolution);

        timer_id = timeSetEvent (timer_interval,
                                 timer_resolution,
                                 (LPTIMECALLBACK)timer_callback,
                                 0,
                                 TIME_PERIODIC);

        if (timer_id != 0)
          timer_installed = 1;
        else
          timeEndPeriod (timer_resolution);
      }
}

___HIDDEN void setup_timer_interrupt_handling ___PVOID
{
}

___HIDDEN void cleanup_timer_interrupt_handling ___PVOID
{
  ___set_timer_interval (0);
}

___HIDDEN void disable_timer_interrupt ___PVOID
{
  ___set_timer_interval (0);
}

___HIDDEN void enable_timer_interrupt ___PVOID
{
  install_timer_callback (timer_interval);
}

#endif

/*   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   */

___HIDDEN void setup_interrupt_handling ___PVOID
{
  setup_user_interrupt_handling ();
  setup_timer_interrupt_handling ();
}

___HIDDEN void cleanup_interrupt_handling ___PVOID
{
  cleanup_timer_interrupt_handling ();
  cleanup_user_interrupt_handling ();
}

___HIDDEN void disable_interrupts ___PVOID
{
  disable_timer_interrupt ();
  disable_user_interrupt ();
}

___HIDDEN void enable_interrupts ___PVOID
{
  enable_user_interrupt ();
  enable_timer_interrupt ();
}


/*---------------------------------------------------------------------------*/

/* OS event handling. */

___WORD ___os_event_get ___PVOID
{
  ___WORD event = ___FAL;

#ifdef BUFFERED_STDOUT

  fflush (stdout);

#endif

#ifdef USE_MACOS

#ifdef USE_mac_gui

  {
    EventRecord er;

    if (mac_gui_get_next_event (&er))
      {
        event = ___alloc_scmobj (___sU8VECTOR, sizeof(er), ___STILL);
        if (event != ___FAL)
          {
            *(EventRecord*)___BODY_AS(event,___tSUBTYPED) = er;
            ___still_obj_refcount_dec (event);
          }
      }
  }

#else

  {
    EventRecord er;
    RgnHandle cursor_rgn;

    cursor_rgn = NewRgn ();
    if (has_IdleUpdate)
      IdleUpdate ();
    if (has_WaitNextEvent ?
        WaitNextEvent (everyEvent, &er, 0, cursor_rgn) :
        GetNextEvent (everyEvent, &er))
      {
        event = ___alloc_scmobj (___sU8VECTOR, sizeof(er), ___STILL);
        if (event != ___FAL)
          {
            *(EventRecord*)___BODY_AS(event,___tSUBTYPED) = er;
            ___still_obj_refcount_dec (event);
          }
      }
    DisposeRgn (cursor_rgn);
  }

#endif

#endif

#ifdef __WATCOMC__
#ifdef __SW_BW

  _dwYield ();

#endif
#endif

  return event;
}

___WORD ___os_event_handler ___P((___WORD event),(event)
___WORD event;)
{
#ifdef USE_MACOS

#ifdef USE_mac_gui

  if (mac_gui_handle_event ((EventRecord*)___BODY_AS(event,___tSUBTYPED), NULL))
    if (mac_gui_quit ())
      exit (1);

#else

  SIOUXHandleOneEvent ((EventRecord*)___BODY_AS(event,___tSUBTYPED));

#endif

#endif

  return ___TRU;
}


/*---------------------------------------------------------------------------*/

/* Access to environment variables. */

char *___getenv ___P((char *name),(name)
char *name;)
{
  return getenv (name);
}


/*---------------------------------------------------------------------------*/

/* Shell command. */

int ___shell_command ___P((char *cmd),(cmd)
char *cmd;)
{
  int result;

  disable_interrupts ();
  result = system (cmd);
  enable_interrupts ();

  return result;
}


/*---------------------------------------------------------------------------*/

/* Dynamic loading. */

#ifdef USE_dlopen
#define DL_DESCR void *
#endif

#ifdef USE_shl_load
#define DL_DESCR shl_t
#endif

#ifdef USE_LoadLibrary
#define DL_DESCR HANDLE
#define ERROR_BUF_SIZE 256
___HIDDEN char error_buffer[ERROR_BUF_SIZE];
#endif

#ifdef USE_DosLoadModule
#define DL_DESCR HMODULE
#define ERROR_BUF_SIZE 256
___HIDDEN char error_buffer[ERROR_BUF_SIZE];
#endif

#ifdef USE_dxe_load
#define DL_DESCR void *
#endif

#ifdef USE_GetDiskFragment
#define DL_DESCR CFragConnectionID
#define ERROR_BUF_SIZE 256
___HIDDEN char error_buffer[ERROR_BUF_SIZE];
#endif

#ifdef DL_DESCR

typedef struct dl_entry
  {
    struct dl_entry *next;
    DL_DESCR descr;
  } dl_entry;

___HIDDEN dl_entry *dl_list;

#endif

___HIDDEN void setup_dynamic_load
   ___P((___setup_params_struct *setup_params),(setup_params)
___setup_params_struct *setup_params;)
{
#ifdef DL_DESCR

  dl_list = 0;

#endif
}

void *___dynamic_load
   ___P((char *path, char *name, char **errmsg),(path, name, errmsg)
char *path;
char *name;
char **errmsg;)
{
#ifndef DL_DESCR

  *errmsg = "Dynamic loading is not available on this platform";
  return 0;

#else

  void *result = 0;
  dl_entry *p;

  p = (dl_entry*)___alloc_mem (sizeof(dl_entry));
  if (p == 0)
    {
      *errmsg = "Can't allocate dynamic loader entry";
      return 0;
    }

#ifdef USE_dlopen

#ifdef RTLD_NOW
  p->descr = dlopen (path, RTLD_NOW);
#else
  p->descr = dlopen (path, 1);
#endif

  if (p->descr != 0)
    if ((result = dlsym (p->descr, name)) == 0)
      dlclose (p->descr);

  if (result == 0)
    *errmsg = (char*)dlerror ();

#endif

#ifdef USE_shl_load

  p->descr = shl_load (path, BIND_IMMEDIATE, 0);
  if (p->descr == 0)
    *errmsg = strerror (errno);
  else
    if (shl_findsym (&p->descr, name, TYPE_PROCEDURE, &result))
      {
        shl_unload (p->descr);
        result = 0;
        *errmsg = "can't find symbol";
      }

#endif

#ifdef USE_LoadLibrary

  p->descr = LoadLibrary (path);
  if (p->descr != 0)
    if ((result = GetProcAddress (p->descr, name)) == 0)
      FreeLibrary (p->descr);

  if (result == 0)
    {
      FormatMessage (FORMAT_MESSAGE_FROM_SYSTEM,
                     0,
                     GetLastError (),
                     0,
                     error_buffer,
                     ERROR_BUF_SIZE,
                     0);
      *errmsg = error_buffer;
    }

#endif

#ifdef USE_DosLoadModule

  {
    HMODULE hmodule;
    APIRET rc;
    char real_path[280];
    int i;

    i = 0;
    while (path[i] != '\0') i++;
    if (i <= 4 ||
        path[i-1] != 'l' && path[i-1] != 'L' ||
        path[i-2] != 'l' && path[i-2] != 'L' ||
        path[i-3] != 'd' && path[i-3] != 'D' ||
        path[i-4] != '.')
      {
        /* if path doesn't end in ".dll" we must find the real ".dll" path */

        FILE *f = fopen (path, "r");
        if (f == 0)
          {
            *errmsg = "Invalid Name";
            return 0;
          }
        if (!fgets (real_path, 280, f))
          {
            *errmsg = "Invalid Redirection";
            fclose (f);
            return 0;
          }
        fclose (f);
        path = real_path;
      }

    if (DosLoadModule (error_buffer, ERROR_BUF_SIZE, path, &hmodule)
        != NO_ERROR)
      *errmsg = error_buffer;
    else
      {
        p->descr = hmodule;
        rc = DosQueryProcAddr (hmodule, 0L, name, &result);
        if (rc != NO_ERROR)
          {
            switch (rc)
              {
              case ERROR_INVALID_HANDLE:
                *errmsg = "Invalid Handle";
                break;
              case ERROR_INVALID_NAME:
                *errmsg = "Invalid Name";
                break;
              case ERROR_INVALID_ORDINAL:
                *errmsg = "Invalid Ordinal";
                break;
              case ERROR_ENTRY_IS_CALLGATE:
                *errmsg = "Entry is callgate";
                break;
              default:
                *errmsg = "Unknown error";
                break;
              }
            DosFreeModule(hmodule);
          }
      }
  }

#endif

#ifdef USE_dxe_load

  result = _dxe_load (path);
  p->descr = result;

  if (result == 0)
    *errmsg = "unknown error";

#endif

#ifdef USE_GetDiskFragment

  {
    OSErr err;
    Ptr mainadr, procadr;
    Str63 ppath;
    Str255 pname;
    Str255 pmsg;
    char msg[256];
    FSSpec spec;

    if (!c2pascal (path, ppath, sizeof(ppath)-1))
      *errmsg = "path is too long";
    else if (!c2pascal (name, pname, sizeof(pname)-1))
      *errmsg = "module name is too long";
    else if (make_ResolvedFSSpec (0, 0, ppath, &spec) != noErr)
      *errmsg = "invalid path";
    else
      {
        err = GetDiskFragment (&spec, 0, kCFragGoesToEOF, ppath,
                               kPrivateCFragCopy, &p->descr, &mainadr, pmsg);
        if (err != noErr)
          {
            pascal2c (pmsg, msg, 255);
            sprintf (error_buffer,
                     "GetDiskFragment failed with error code %d on \"%s\"",
                     err,
                     msg);
            *errmsg = error_buffer;
          }
        else
          {
            if (FindSymbol (p->descr, pname, &procadr, kCodeCFragSymbol)
                != noErr)
              {
                *errmsg = "FindSymbol failed";
                CloseConnection (&p->descr);
              }
            else
              result = (void*)procadr;
          }
      }
  }

#endif

  if (result == 0)
    ___free_mem (p);
  else
    {
      p->next = dl_list;
      dl_list = p;
    }

  return result;

#endif
}

___HIDDEN void cleanup_dynamic_load ___PVOID
{
#ifdef DL_DESCR

  dl_entry *p = dl_list;
  while (p != 0)
    {
      dl_entry *next = p->next;

#ifdef USE_dlopen
      dlclose (p->descr);
#endif

#ifdef USE_shl_load
      shl_unload (p->descr);
#endif

#ifdef USE_LoadLibrary
      FreeLibrary (p->descr);
#endif

#ifdef USE_DosLoadModule
      DosFreeModule (p->descr);
#endif

#ifdef USE_dxe_load
#endif

#ifdef USE_GetDiskFragment
      CloseConnection (&p->descr);
#endif

      ___free_mem (p);
      p = next;
    }
  dl_list = 0;

#endif  
}


/*---------------------------------------------------------------------------*/

/* Dynamic C compilation. */

#ifdef USE_CWIDE

/*
 * On the Macintosh, a C module can be dynamically compiled by
 * sending AppleEvents to the CodeWarrior IDE.  This code has been
 * tested with CWPro 1 on a PowerPC based Macintosh.
 */

#define CWIDE_EVENT_CLASS 'MMPR'
#define CWIDE_SIGNATURE 'CWIE'

#define CWIDE_no_result     0
#define CWIDE_list_of_short 1
#define CWIDE_list_of_ErrM  2

enum
  {
    errShell_ActionFailed  = 1,
    errShell_FileNotFound  = 2,
    errShell_DuplicateFile = 3,
    errShell_CompileError  = 4,
    errShell_MakeFailed    = 5,
    errShell_NoOpenProject = 6,
    errShell_WindowNotOpen = 7
  };

___HIDDEN AEAddressDesc CWIDE_address;

___HIDDEN OSErr setup_CWIDE ___PVOID
{
  long signature = CWIDE_SIGNATURE;

  CWIDE_address.dataHandle = nil;

  return AECreateDesc (typeApplSignature,
                       (Ptr)&signature,
                       sizeof (long),
                       &CWIDE_address);
}

___HIDDEN OSErr cleanup_CWIDE ___PVOID
{
  if (CWIDE_address.dataHandle != nil)
    return AEDisposeDesc (&CWIDE_address);
  else
    return noErr;
}

___HIDDEN OSErr CWIDE_send_ae
   ___P((AppleEvent *ae, int res_kind, long *CWIDE_err),
        (ae, res_kind, CWIDE_err)
AppleEvent ae;
int res_kind;
long *CWIDE_err;)
{
  OSErr err;
  AppleEvent reply_ae;

  reply_ae.dataHandle = nil;

  err = AESend (ae,
                &reply_ae,
                kAEWaitReply,
                kAENormalPriority,
                kNoTimeOut,
                nil,
                nil);

  if (err == noErr && res_kind != CWIDE_no_result)
    switch (res_kind)
      {
        case CWIDE_list_of_short:
        case CWIDE_list_of_ErrM:
          {
            AEDescList list;
            list.dataHandle = nil;
            err = AEGetParamDesc (&reply_ae, keyAEResult, typeAEList, &list);
            if (err == noErr)
              if (res_kind == CWIDE_list_of_short)
                {
                  short error_code;
                  AEKeyword key;
                  DescType typ;
                  Size siz;
                  err = AEGetNthPtr (&list, 1, typeShortInteger, &key,
                                     &typ, (Ptr)&error_code, sizeof (short), &siz);
                  if (err == noErr)
                    *CWIDE_err = error_code;
                }
              else
                {
                  long nb_errors;
                  err = AECountItems (&list, &nb_errors);
                  if (err == noErr)
                    *CWIDE_err = nb_errors;
                }
            if (list.dataHandle != nil)
              AEDisposeDesc (&list);
            break;
          }
      }

  if (reply_ae.dataHandle != nil)
    AEDisposeDesc (&reply_ae);

  return err;
}

___HIDDEN OSErr CWIDE_make_ae
   ___P((AEEventClass clas, AEEventID id, AppleEvent *ae),
        (clas, id, ae)
AEEventClass clas;
AEEventID id;
AppleEvent *ae;)
{
  return AECreateAppleEvent (clas,
                             id,
                             &CWIDE_address,
                             kAutoGenerateReturnID,
                             kAnyTransactionID,
                             ae);
}

___HIDDEN OSErr CWIDE_file_param
   ___P((FSSpec *file, AEKeyword key, AppleEvent *ae),
        (file, key, ae)
FSSpec *file;
AEKeyword key;
AppleEvent *ae;)
{
  OSErr err;
  AEDescList list;

  list.dataHandle = nil;

  err = AECreateList (nil, 0, false, &list);
  if (err == noErr)
    {
      err = AEPutPtr (&list, 1, typeFSS, (Ptr)file, sizeof (FSSpec));
      if (err == noErr)
        err = AEPutParamDesc (ae, key, &list);
    }

  if (list.dataHandle != nil)
    AEDisposeDesc (&list);

  return err;
}

___HIDDEN OSErr CWIDE_no_param
   ___P((AEEventID id, int res_kind, long *CWIDE_err),
        (id, res_kind, CWIDE_err)
AEEventID id;
int res_kind;
long *CWIDE_err;)
{
  OSErr err;
  AppleEvent ae;

  ae.dataHandle = nil;

  err = CWIDE_make_ae (CWIDE_EVENT_CLASS, id, &ae);
  if (err == noErr)
    err = CWIDE_send_ae (&ae, res_kind, CWIDE_err);

  if (ae.dataHandle != nil)
    AEDisposeDesc (&ae);

  return err;
}

___HIDDEN OSErr CWIDE_open_project ___P((FSSpec *file),(file)
FSSpec *file;)
{
  long CWIDE_err;
  OSErr err;
  AppleEvent ae;

  ae.dataHandle = nil;

  err = CWIDE_make_ae ('aevt', 'odoc', &ae);
  if (err == noErr)
    {
      err = CWIDE_file_param (file, keyDirectObject, &ae);
      if (err == noErr)
        err = CWIDE_send_ae (&ae, CWIDE_no_result, &CWIDE_err);
    }

  if (ae.dataHandle != nil)
    AEDisposeDesc (&ae);

  return err;
}

___HIDDEN OSErr CWIDE_close_project ___PVOID
{
  long CWIDE_err;
  return CWIDE_no_param ('ClsP', CWIDE_no_result, &CWIDE_err);
}

___HIDDEN OSErr CWIDE_make_project ___P((long *CWIDE_err),(CWIDE_err)
long *CWIDE_err;)
{
  OSErr err;
  AppleEvent ae;

  ae.dataHandle = nil;

  err = CWIDE_make_ae (CWIDE_EVENT_CLASS, 'Make', &ae);
  if (err == noErr)
    {
      Boolean extern_editor = true;
      err = AEPutParamPtr (&ae, 'Errs', typeBoolean,
                           (Ptr)&extern_editor, sizeof (Boolean));
      if (err == noErr)
        err = CWIDE_send_ae (&ae, CWIDE_list_of_ErrM, CWIDE_err);
    }

  if (ae.dataHandle != nil)
    AEDisposeDesc (&ae);

  return err;
}

___HIDDEN OSErr CWIDE_add_file
   ___P((FSSpec *file, short segment_num, long *CWIDE_err),
        (file, segment_num, CWIDE_err)
FSSpec *file;
short segment_num;
long *CWIDE_err;)
{
  OSErr err;
  AppleEvent ae;

  ae.dataHandle = nil;

  err = CWIDE_make_ae (CWIDE_EVENT_CLASS, 'AddF', &ae);
  if (err == noErr)
    {
      err = CWIDE_file_param (file, keyDirectObject, &ae);
      if (err == noErr)
        {
          err = AEPutParamPtr (&ae, 'Segm', typeShortInteger,
                               (Ptr)&segment_num, sizeof (short));
          if (err == noErr)
            err = CWIDE_send_ae (&ae, CWIDE_list_of_short, CWIDE_err);
        }
    }

  if (ae.dataHandle != nil)
    AEDisposeDesc (&ae);

  return err;
}

___HIDDEN OSErr CWIDE_set_prefs ___P((Str255 name, AERecord *rec),(name, rec)
Str255 name;
AERecord *rec;)
{
  long CWIDE_err;
  OSErr err;
  AppleEvent ae;

  ae.dataHandle = nil;

  err = CWIDE_make_ae (CWIDE_EVENT_CLASS, 'Pref', &ae);
  if (err == noErr)
    {
      err = AEPutParamPtr (&ae, 'PNam', typeChar, (Ptr)(name+1), name[0]);
      if (err == noErr)
        {
          err = AEPutParamDesc (&ae, 'PRec', rec);
          if (err == noErr)
            err = CWIDE_send_ae (&ae, CWIDE_no_result, &CWIDE_err);
        }
    }

  if (ae.dataHandle != nil)
    AEDisposeDesc (&ae);

  return err;
}

___HIDDEN OSErr CWIDE_set_output ___P((Str255 path),(path)
Str255 path;)
{
  OSErr err;
  AERecord rec;
  err = AECreateList (nil, 0, true, &rec);
  if (err == noErr)
    {
      err = AEPutKeyPtr (&rec, 'PR02', typeChar, (Ptr)(path+1), path[0]);
      if (err == noErr)
        err = CWIDE_set_prefs ("\pPPC Project", &rec);
    }

  if (rec.dataHandle != nil)
    AEDisposeDesc (&rec);

  return err;
}

___HIDDEN int CWIDE_is_a_regular_file ___P((char *path),(path)
char *path;)
{
  struct stat sbuf;
  return (stat (path, &sbuf) == 0 && S_ISREG(sbuf.st_mode));
}

int ___dynamic_cc
   ___P((char *cc_cmd,
         char *ld_cmd,
         char *root_path,
         char *output_path,
         int ld_flip,
         char **errmsg),
        (cc_cmd, ld_cmd, root_path, output_path, ld_flip, errmsg)
char *cc_cmd;
char *ld_cmd;
char *root_path;
char *output_path;
int ld_flip;
char **errmsg;)
{
  long CWIDE_err;
  OSErr err;
  FSSpec source_spec, project_spec, model_spec;
  char source_path[___PATH_MAX_LENGTH+1];
  char project_path[___PATH_MAX_LENGTH+1];
  char model_path[___PATH_MAX_LENGTH+1];
  Str255 source_ppath;
  Str255 project_ppath;
  Str255 output_ppath;
  Str255 model_ppath;
  char *p;

  /******* we should check that we will not overflow the strings ******/

  sprintf (source_path, "%s.c", root_path);
  sprintf (project_path, "%s.prj", output_path);

  p = output_path; /* strip directory */
  while (*p != '\0')
    {
      if (DIR_SEPARATOR(*p))
        output_path = p+1;
      p++;
    }

  if (!path_expand_to_absolute (cc_cmd, model_path, ___PATH_MAX_LENGTH) ||
      !CWIDE_is_a_regular_file (model_path))
    {
      *errmsg = "The model project file is missing";
      return 1;
    }

  if (!c2pascal (source_path, source_ppath, sizeof (source_ppath)-1) ||
      !c2pascal (project_path, project_ppath, sizeof (project_ppath)-1) ||
      !c2pascal (output_path, output_ppath, sizeof (output_ppath)-1) ||
      !c2pascal (model_path, model_ppath, sizeof (model_ppath)-1) ||
      (err = make_ResolvedFSSpec (0, 0, source_ppath, &source_spec)) != noErr ||
      (err = make_ResolvedFSSpec (0, 0, project_ppath, &project_spec)) != noErr ||
      (err = make_ResolvedFSSpec (0, 0, model_ppath, &model_spec)) != noErr ||
      (err = copy_file (model_spec, project_spec)) != noErr ||
      (err = setup_CWIDE ()) != noErr)
    {
      *errmsg = "The setting up of the CodeWarrior IDE failed";
      return 1;
    }

  if ((err = CWIDE_close_project ()) != noErr ||
      (err = CWIDE_open_project (&project_spec)) != noErr)
    {
      if (err == connectionInvalid)
        *errmsg = "The CodeWarrior IDE is not running";
      else
        *errmsg = "The CodeWarrior IDE failed to open the project file";
    }
  else if ((err = CWIDE_add_file (&source_spec, 0, &CWIDE_err)) != noErr)
    *errmsg = "The CodeWarrior IDE failed to add the source file to the project";
  else if (CWIDE_err == errShell_FileNotFound)
    {
      err = fnfErr;
      *errmsg = "The source file could not be found";
    }
  else if ((err = CWIDE_set_output (output_ppath)) != noErr)
    *errmsg = "The CodeWarrior IDE failed to set the output file";
  else if ((err = CWIDE_make_project (&CWIDE_err)) != noErr)
    *errmsg = "The CodeWarrior IDE failed to make the project";
  else if (CWIDE_err > 0)
    {
      err = fnfErr; /* anything can do */
      *errmsg = "Compilation failed at C level";
    }
  else if ((err = CWIDE_close_project ()) != noErr)
    *errmsg = "The CodeWarrior IDE could not close the project file";
  else
    remove (project_path);

  cleanup_CWIDE ();

  return err != noErr;
}

#else

int ___dynamic_cc
   ___P((char *cc_cmd,
         char *ld_cmd,
         char *root_path,
         char *output_path,
         int ld_flip,
         char **errmsg),
        (cc_cmd, ld_cmd, root_path, output_path, ld_flip, errmsg)
char *cc_cmd;
char *ld_cmd;
char *root_path;
char *output_path;
int ld_flip;
char **errmsg;)
{
  char cmd[___CMD_MAX_LENGTH+1];

  /******* we should check that we will not overflow the string ******/

  sprintf (cmd, cc_cmd, root_path, root_path);

  if (___shell_command (cmd) != 0)
    {
      *errmsg = "Compilation failed at C level";
      return 1;
    }

  if (ld_flip)
    sprintf (cmd, ld_cmd, output_path, root_path);
  else
    sprintf (cmd, ld_cmd, root_path, output_path);

  if (___shell_command (cmd) != 0)
    {
      *errmsg = "Linking failed at C level";
      return 1;
    }

  return 0;
}

#endif


/*---------------------------------------------------------------------------*/

/* Setup floating point processing. */

___HIDDEN void setup_fp ___PVOID
{
#ifdef USE_control87

#define FP_EXC_MASK MCW_EM
#define FP_EXC_CW \
(EM_INVALID+EM_ZERODIVIDE+EM_OVERFLOW+EM_UNDERFLOW+EM_INEXACT+EM_DENORMAL)

  _control87 (FP_EXC_CW, FP_EXC_MASK);

#endif
}


/*---------------------------------------------------------------------------*/

/* Elapsed real time. */

void ___real_time ___P((___U64 *rt),(rt)
___U64 *rt;)
{

#ifndef USE_time
#ifndef USE_ftime
#ifndef USE_gettimeofday
#ifndef USE_clock_gettime
#ifndef USE_getclock
#ifndef USE_GetDateTime

  *rt = ___U64_init (0, 0); /* can't get time... result is 0 */

#endif
#endif
#endif
#endif
#endif
#endif

#ifdef USE_time

  time_t t;

  time (&t);

  *rt = ___U64_mul_U32 (___U64_init (0, t), 1000000000);

#endif

#ifdef USE_ftime

  struct timeb tb;

  ftime (&tb);

  *rt = ___U64_mul_U32 (___U64_add_U32 (___U64_mul_U32 (___U64_init (0, tb.time),
                                                        1000),
                                        tb.millitm),
                        1000000);

#endif

#ifdef USE_gettimeofday

  struct timeval tv;

  if (gettimeofday (&tv, NULL) == 0)
    *rt = ___U64_mul_U32 (___U64_add_U32 (___U64_mul_U32 (___U64_init (0, tv.tv_sec),
                                                          1000000),
                                          tv.tv_usec),
                          1000);
  else
    *rt = ___U64_init (0, 0);

#endif

#ifdef USE_clock_gettime

  struct timespec ts;

  if (clock_gettime (CLOCK_REALTIME, &ts) == 0)
    *rt = ___U64_add_U32 (___U64_mul_U32 (___U64_init (0, ts.tv_sec),
                                          1000000000),
                          ts.tv_nsec);
  else
    *rt = ___U64_init (0, 0);

#endif

#ifdef USE_getclock

  struct timespec ts;

  if (getclock (TIMEOFDAY, &ts) == 0)
    *rt = ___U64_add_U32 (___U64_mul_U32 (___U64_init (0, ts.tv_sec),
                                          1000000000),
                          ts.tv_nsec);
  else
    *rt = ___U64_init (0, 0);

#endif

#ifdef USE_GetDateTime

  unsigned long secs_since_1904;
  MachineLocation here;
  long offset;
  ___U32 t;

  /* Compute seconds since 01-01-1970. */

  GetDateTime (&secs_since_1904);
  t = secs_since_1904 - ((1970-1904)*365+(1970-1904)/4+1)*24*60*60;

  /* Adjust to CUT based on machine location. */

  ReadLocation (&here);
  offset = here.u.gmtDelta & 0x00ffffff;
  if (offset & 0x00800000)
    offset |= 0xff000000;
  t -= offset;

  *rt = ___U64_mul_U32 (___U64_init (0, t), 1000000000);

#endif
}


___HIDDEN void init_real_time ___PVOID
{
  /* Some systems (e.g. DJGPP) need a first call to define the time origin. */
  ___U64 rt;
  ___real_time (&rt);
}


/*---------------------------------------------------------------------------*/

/* Elapsed cpu time (user time and system time). */

void ___cpu_time ___P((___U64 *user, ___U64 *sys),(user, sys)
___U64 *user;
___U64 *sys;)
{

#ifndef USE_times
#ifndef USE_clock
#ifndef USE_getrusage
#ifndef USE_GetProcessTime
#ifndef USE_TickCount
#ifndef USE_DosQuerySysInfo

  *user = ___U64_init (0, 0); /* can't get time... result is 0 */
  *sys = ___U64_init (0, 0);

#endif
#endif
#endif
#endif
#endif
#endif

#ifdef USE_times

#ifdef CLK_TCK
#define CLOCK_TICKS_PER_SEC CLK_TCK
#else
#ifdef _SC_CLOCKS_PER_SEC
#define CLOCK_TICKS_PER_SEC sysconf (_SC_CLOCKS_PER_SEC)
#else
#ifdef _SC_CLK_TCK
#define CLOCK_TICKS_PER_SEC sysconf (_SC_CLK_TCK)
#else
@error Cannot find a definition for CLOCK_TICKS_PER_SEC
#endif
#endif
#endif

  struct tms t;

  times (&t);

  *user = ___U64_mul_U32 (___U64_init (0, t.tms_utime),
                          (___U32)(1<<2)*1000000000/(___U32)CLOCK_TICKS_PER_SEC);
  ___U64_shift_right (user, 2);

  *sys = ___U64_mul_U32 (___U64_init (0, t.tms_stime),
                         (___U32)(1<<2)*1000000000/(___U32)CLOCK_TICKS_PER_SEC);
  ___U64_shift_right (sys, 2);

#endif

#ifdef USE_clock

  *user = ___U64_mul_U32 (___U64_init (0, clock ()),
                          (___U32)(1<<2)*1000000000/(___U32)CLOCKS_PER_SEC);
  ___U64_shift_right (user, 2);

  *sys = ___U64_init (0, 0); /* fake system time */

#endif

#ifdef USE_getrusage

  struct rusage ru;

  if (getrusage (RUSAGE_SELF, &ru) == 0)
    {
      *user = ___U64_mul_U32 (___U64_add_U32 (___U64_mul_U32 (___U64_init (0, ru.ru_utime.tv_sec),
                                                              1000000),
                                              ru.ru_utime.tv_usec),
                              1000);
      *sys = ___U64_mul_U32 (___U64_add_U32 (___U64_mul_U32 (___U64_init (0, ru.ru_stime.tv_sec),
                                                             1000000),
                                             ru.ru_stime.tv_usec),
                             1000);
    }
  else
    {
      *user = ___U64_init (0, 0); /* can't get time... result is 0 */
      *sys =  ___U64_init (0, 0);
    }

#endif

#ifdef USE_GetProcessTime

  HANDLE p;
  FILETIME creation_time, exit_time, sys_time, user_time;

  p = GetCurrentProcess ();

  if (GetProcessTimes (p, &creation_time, &exit_time, &sys_time, &user_time))
    {
      *user = ___U64_mul_U32 (___U64_init (user_time.dwHighDateTime,
                                           user_time.dwLowDateTime),
                              100);
      *sys = ___U64_mul_U32 (___U64_init (sys_time.dwHighDateTime,
                                          sys_time.dwLowDateTime),
                             100);
    }
  else
    {
      *user = ___U64_init (0, 0); /* can't get time... result is 0 */
      *sys =  ___U64_init (0, 0);
    }

#endif

#ifdef USE_TickCount

  static ___U32 origin = 0;

  if (origin == 0)
    origin = TickCount ();

  *user = ___U64_mul_U32 (___U64_init (0, (___U32)TickCount () - origin),
                          (___U32)(1<<2)*1000000000/(___U32)60);
  ___U64_shift_right (user, 2);

  *sys = ___U64_init (0, 0); /* fake system time */

#endif

#ifdef USE_DosQuerySysInfo

  static ___U32 origin = 0;
  ___U32 now;

  DosQuerySysInfo (QSV_MS_COUNT, QSV_MS_COUNT, &now, sizeof(now));

  if (origin == 0)
    origin = now;

  *user = ___U64_mul_U32 (___U64_init (0, now-origin), 1000000); /* are these the right units??? */
  *sys = ___U64_init (0, 0); /* fake system time */

#endif
}


___HIDDEN void init_cpu_time ___PVOID
{
  /* Some systems (e.g. DJGPP) need a first call to define the time origin. */
  ___U64 user, sys;
  ___cpu_time (&user, &sys);
}


/*---------------------------------------------------------------------------*/

/* Virtual memory statistics. */

void ___vm_stats ___P((long *minflt, long *majflt),(minflt, majflt)
long *minflt;
long *majflt;)
{
#ifndef USE_getrusage

  *minflt = 0; /* can't get statistics... result is 0 */
  *majflt = 0;

#endif

#ifdef USE_getrusage

  struct rusage ru;

  if (getrusage (RUSAGE_SELF, &ru) == 0)
    {
      *minflt = ru.ru_minflt;
      *majflt = ru.ru_majflt;
    }
  else
    {
      *minflt = 0; /* can't get statistics... result is 0 */
      *majflt = 0;
    }

#endif
}

/*---------------------------------------------------------------------------*/

/* Filesystem path expansion. */

#ifndef USE_MACOS

#ifdef USE_DOS_PATHS

#ifdef USE_DOS
#define DIR_SEPARATOR1 '\\'
#define DIR_SEPARATOR2 '/'
#define PARENT_HOP "..\\"
#else
#define DIR_SEPARATOR1 '/'
#define DIR_SEPARATOR2 '\\'
#define PARENT_HOP "../"
#endif
#define DRIVE_SEPARATOR ':'

#define DIR_SEPARATOR(c)((c) == DIR_SEPARATOR1 || (c) == DIR_SEPARATOR2)
#define SEPARATOR(c)(DIR_SEPARATOR(c) || (c) == DRIVE_SEPARATOR)

#else

#define DIR_SEPARATOR1 '/'
#define PARENT_HOP "../"

#define DIR_SEPARATOR(c)((c) == DIR_SEPARATOR1)
#define SEPARATOR(c)DIR_SEPARATOR(c)

#endif

#ifdef USE_DOS

___HIDDEN int normalize_dir_path
   ___P((char *dir_path, char *norm_dir_path, long max_length),
        (dir_path, norm_dir_path, max_length)
char *dir_path;
char *norm_dir_path;
long max_length;)
{
  char cwd1[___PATH_MAX_LENGTH+1];
  int result = 0;
  if (getcwd (cwd1, ___PATH_MAX_LENGTH+1) != 0)
    {
      long i = 0;
      while (dir_path[i] != '\0' && !SEPARATOR(dir_path[i]))
        i++;
      if (dir_path[i] == DRIVE_SEPARATOR && dir_path[i+1] != '\0')
        {
          char cwd2[___PATH_MAX_LENGTH+1];
          char tmp = dir_path[i+1];
          dir_path[i+1] = '\0';
          if (chdir (dir_path) == 0)
            {
              if (getcwd (cwd2, ___PATH_MAX_LENGTH+1) != 0)
                {
                  dir_path[i+1] = tmp;
                  if (chdir (dir_path) == 0)
                    {
                      result = getcwd (norm_dir_path, max_length) != 0;
                      chdir (cwd2);
                    }
                }
              chdir (cwd1);
            }
        }
      else if (chdir (dir_path) == 0)
        {
          result = getcwd (norm_dir_path, max_length) != 0;
          chdir (cwd1);
        }
    }
  return result;
}

#else

___HIDDEN int normalize_dir_path
   ___P((char *dir_path, char *norm_dir_path, long max_length),
        (dir_path, norm_dir_path, max_length)
char *dir_path;
char *norm_dir_path;
long max_length;)
{
  char cwd[___PATH_MAX_LENGTH+1];
  int result = 0;
  if (getcwd (cwd, ___PATH_MAX_LENGTH+1) != 0 && chdir (dir_path) == 0)
    {
      result = getcwd (norm_dir_path, max_length) != 0;
      chdir (cwd);
    }
  return result;
}

#endif

___HIDDEN int path_expand_to_absolute
   ___P((char *path, char *new_path, long max_length),
        (path, new_path, max_length)
char *path;
char *new_path;
long max_length;)
{
  int result = 0;
  char tmp[___PATH_MAX_LENGTH+1];
  long i = 0;
  long j = 0;
  long k;

  disable_interrupts ();

  if (path[0] == '\0')
    {
      /* Empty path expands to current working directory. */

      if (getcwd (new_path, max_length) == 0)
        goto ret;

      goto terminal_dir_separator;
    }

  if (path[0] == '~')
    {
      /*
       * Path starting with a '~' refers to a user's home directory or
       * the Gambit-C installation directory.  So get into "tilde_dir"
       * the path of the appropriate directory.
       */

      char *tilde_dir;

      if (path[1] == '~')
        {
          /* "~~" or "~~/xxx..." */

          if (path[2]!='\0' && !DIR_SEPARATOR(path[2]))
            goto ret;
          tilde_dir = getenv ("GAMBCDIR");
          if (tilde_dir == 0)
#ifdef ___GAMBCDIR
            tilde_dir = ___GAMBCDIR;
#else
#ifdef USE_DOS_PATHS
            tilde_dir = "c:/gambc/";
#else
            tilde_dir = "/usr/local/share/gambc/";
#endif
#endif
          i += 2;
        }
      else
        {
          /* "~" or "~/xxx..." or "~user" or "~user/xxx..." */

          i++;
          while (path[i]!='\0' && !DIR_SEPARATOR(path[i]))
            {
              if (i-1 > ___PATH_MAX_LENGTH)
                goto ret;
              tmp[i-1] = path[i];
              i++;
            }

          if (i == 1)
            {
              tilde_dir = getenv ("HOME");
              if (tilde_dir == 0)
#ifdef ___HOME
                tilde_dir = ___HOME;
#else
                goto ret;
#endif
            }
          else
#ifdef USE_UNIX
            {
              struct passwd *p;
              tmp[i-1] = '\0';
              p = getpwnam (tmp);
              if (p == 0)
                goto ret;
              tilde_dir = p->pw_dir;
              if (tilde_dir == 0)
                goto ret;
            }
#else
            goto ret;
#endif
        }

      /* Copy "tilde_dir" to head of "tmp". */

      while (*tilde_dir != '\0')
        {
          if (j > ___PATH_MAX_LENGTH)
            goto ret;
          tmp[j++] = *tilde_dir++;
        }
    }

  k = i;

  /*
   * At this point the first "k" characters of the path have
   * been processed.  Either k==0 or path[k]=='\0' or path[k] is
   * a path separator.
   */

  /* Copy the "non-tilde" part of the path to "tmp". */

  while (path[i] != '\0')
    {
      if (j > ___PATH_MAX_LENGTH)
        goto ret;
      tmp[j++] = path[i++];
    }

  if (j > ___PATH_MAX_LENGTH)
    goto ret;

  tmp[j] = '\0';

#ifdef USE_fullpath

  if (_fullpath (new_path, tmp, max_length+1) == 0)
    goto ret;

  {
    struct stat sbuf;
    if (stat (new_path, &sbuf) == 0 && S_ISDIR(sbuf.st_mode))
      {
        /* Make sure the directory path ends in a path separator. */

        terminal_dir_separator:
        j = strlen (new_path);
        if (j==0 || !DIR_SEPARATOR(new_path[j-1]))
          {
            new_path[j++] = DIR_SEPARATOR1;
            if (j > max_length)
              goto ret;
            new_path[j] = '\0';
          }
      }
  }

#else

  /* Check if this is the path of an existing directory. */

  if (normalize_dir_path (tmp, new_path, max_length))
    {
      /* Make sure the directory path ends in a path separator. */

      terminal_dir_separator:
      j = strlen (new_path);
      if (j==0 || !DIR_SEPARATOR(new_path[j-1]))
        {
          new_path[j++] = DIR_SEPARATOR1;
          if (j > max_length)
            goto ret;
          new_path[j] = '\0';
        }
    }
  else
    {
      /*
       * The path is not that of an existing directory.  Get the path
       * of the enclosing directory by removing the part that follows
       * the last path separator.
       */

      i = strlen (path);
      while (i > k && !SEPARATOR(path[i]))
        {
          i--;
          j--;
        }

      if (SEPARATOR(path[i]))
        {
          /* Check if the enclosing directory exists. */

          i++;
          j++;
          if (j > ___PATH_MAX_LENGTH)
            goto ret;
          tmp[j] = '\0';
          if (!normalize_dir_path (tmp, new_path, max_length))
            goto ret;
        }
      else if (i == 0)
        {
          /*
           * No enclosing directory specified so use current working
           * directory instead.
           */

          if (getcwd (new_path, max_length) == 0)
            goto ret;
        }
      else
        goto ret;

      /* Append the "non-tilde" part of the path. */

      j = strlen (new_path);
      if (!DIR_SEPARATOR(new_path[j-1]))
        new_path[j++] = DIR_SEPARATOR1;

      while (path[i] != '\0')
        {
          if (j > max_length)
            goto ret;
          new_path[j++] = path[i++];
        }

      if (j > max_length)
        goto ret;

      new_path[j] = '\0';
    }

#endif

  result = 1;

  ret:

  enable_interrupts ();

  return result;
}

#endif

int ___path_absolute ___P((char *path),(path)
char *path;)
{
#ifdef USE_MACOS

  char *p = path;
  if (*p == '~')
    return 1;
  while (*p != '\0' && !SEPARATOR(*p))
    p++;
  return p != path && DIR_SEPARATOR(*p);

#else

#ifdef USE_DOS_PATHS

  char *p = path;
  if (*p == '~' || DIR_SEPARATOR(*p))
    return 1;
  while (*p != '\0' && !SEPARATOR(*p))
    p++;
  return *p == DRIVE_SEPARATOR && DIR_SEPARATOR(p[1]);

#else

  return *path == '~' || DIR_SEPARATOR(*path);
  
#endif

#endif
}

int ___path_extension
   ___P((char *path, char *ext, long max_length),(path, ext, max_length)
char *path;
char *ext;
long max_length;)
{
  char *p, *end = path;
  while (*end != '\0')
    end++;
  p = end;
  while (p > path && *p != '.' && !SEPARATOR(*p))
    p--;
  if (*p != '.')
    p = end;
  if (end - p > max_length)
    return 0;
  while (p <= end)
    *ext++ = *p++;
  return 1;
}

int ___path_strip_extension
   ___P((char *path, char *new_path, long max_length),
        (path, new_path, max_length)
char *path;
char *new_path;
long max_length;)
{
  char *p, *end = path;
  while (*end != '\0')
    end++;
  p = end;
  while (p > path && *p != '.' && !SEPARATOR(*p))
    p--;
  if (*p != '.')
    p = end;
  if (p - path > max_length)
    return 0;
  while (path < p)
    *new_path++ = *path++;
  *new_path++ = '\0';
  return 1;
}

int ___path_directory
   ___P((char *path, char *dir, long max_length),(path, dir, max_length)
char *path;
char *dir;
long max_length;)
{
  char *p = path, *end = path;
  while (*end != '\0')
    {
      end++;
      if (SEPARATOR(end[-1]))
        p = end;
    }
  if (p - path > max_length)
    return 0;
  while (path < p)
    *dir++ = *path++;
  *dir++ = '\0';
  return 1;
}

int ___path_strip_directory
   ___P((char *path, char *new_path, long max_length),
        (path, new_path, max_length)
char *path;
char *new_path;
long max_length;)
{
  char *p = path, *end = path;
  while (*end != '\0')
    {
      end++;
      if (SEPARATOR(end[-1]))
        p = end;
    }
  if (end - p > max_length)
    return 0;
  while (p <= end)
    *new_path++ = *p++;
  return 1;
}

int ___path_drive
   ___P((char *path, char *drive, long max_length),
        (path, drive, max_length)
char *path;
char *drive;
long max_length;)
{
#ifdef USE_DOS_PATHS

  char *p = path;
  while (*p != '\0' && !SEPARATOR(*p))
    p++;
  if (*p != DRIVE_SEPARATOR)
    {
      drive[0] = '\0';
      return 1;
    }
  p++;
  if (p - path > max_length)
    return 0;
  while (path < p)
    *drive++ = *path++;
  *drive++ = '\0';
  return 1;

#else

  drive[0] = '\0';
  return 1;

#endif
}

int ___path_strip_drive
   ___P((char *path, char *new_path, long max_length),
        (path, new_path, max_length)
char *path;
char *new_path;
long max_length;)
{
#ifdef USE_DOS_PATHS

  char *p = path, *end = path;
  while (*end != '\0')
    end++;
  while (*p != '\0' && !SEPARATOR(*p))
    p++;
  if (*p != DRIVE_SEPARATOR)
    p = path;
  else
    p++;
  if (end - p > max_length)
    return 0;
  while (p <= end)
    *new_path++ = *p++;
  return 1;

#else

  char *end = path;
  while (*end != '\0')
    end++;
  if (end - path > max_length)
    return 0;
  while (path <= end)
    *new_path++ = *path++;
  return 1;

#endif
}

int ___path_expand
   ___P((char *path, char *new_path, long max_length, int format),
        (path, new_path, max_length, format)
char *path;
char *new_path;
long max_length;
int format;)
{
  int expansion, subdir, nb_parent_hops;
  char *p1, *p2, *q1, *q2;
  char path_cwd[___PATH_MAX_LENGTH+1];

  if (!path_expand_to_absolute (path, new_path, max_length))
    return 0;

  if (format == ___PATH_ABSOLUTE)
    return 1;

  if (!path_expand_to_absolute ("", path_cwd, max_length))
    return 0;

#ifdef USE_DOS_PATHS

  /* return absolute path if drives are different */

  if (path_cwd[0] == '\0' ||
      path_cwd[1] != DRIVE_SEPARATOR ||
      path_cwd[2] != DIR_SEPARATOR1 ||
      new_path[0] != path_cwd[0] ||
      new_path[1] != DRIVE_SEPARATOR ||
      new_path[2] != DIR_SEPARATOR1)
    return 1;

#endif

  p1 = path_cwd;
  p2 = new_path;
  q1 = new_path;

  while (*p1 != '\0' && *p1 == *p2)
    {
      if (DIR_SEPARATOR(*p2))
        q1 = p2+1;
      p1++;
      p2++;
    }

#ifdef USE_MACOS

  /* if volume names are different an absolute path is needed */

  if (q1 == new_path)
    return 1;

#endif

  subdir = 0;
  q2 = q1;
  while (*q2 != '\0')
    {
      if (DIR_SEPARATOR(*q2))
        subdir = 1;
      q2++;
    }

  nb_parent_hops = 0;
  while (*p1 != '\0')
    {
      if (DIR_SEPARATOR(*p1))
        nb_parent_hops++;
      *p1++;
    }

#ifdef USE_MACOS

  if (nb_parent_hops > 0 || subdir)
    nb_parent_hops++;

#endif

  expansion = nb_parent_hops*(sizeof (PARENT_HOP)-1) - (q1-new_path);

  if (expansion >= 0)
    {
      if (format == ___PATH_SHORTEST)
        return 1;
      if ((q2-new_path)+expansion > max_length)
        return 0;
      *(q2+expansion) = '\0';
      while (q1 < q2)
        {
          q2--;
          *(q2+expansion) = *q2;
        }
    }
  else
    {
      while (q1 < q2)
        {
          *(q1+expansion) = *q1;
          q1++;
        }
      *(q1+expansion) = '\0';
    }

  while (nb_parent_hops > 0)
    {
      char *p = PARENT_HOP;
      while (*p != '\0')
        *new_path++ = *p++;
      nb_parent_hops--;
    }

  return 1;
}


/*---------------------------------------------------------------------------*/

/* Formatting of source code position. */

char *___format_filepos
   ___P((char *path, long filepos, ___BOOL pinpoint),(path, filepos, pinpoint)
char *path;
long filepos;
___BOOL pinpoint;)
{
#ifdef USE_MACOS

#ifdef USE_mac_gui

  if (pinpoint)
    mac_gui_highlight (path, filepos);

#endif

#endif

  return 0; /* Use default format for displaying location */
}


/*---------------------------------------------------------------------------*/

/* Low-level I/O. */

___HIDDEN ___STREAM io_stdin, io_stdout, io_stderr;

___HIDDEN int default_io_encoding;

___HIDDEN void setup_io
   ___P((___setup_params_struct *setup_params),(setup_params)
___setup_params_struct *setup_params;)
{
  int force_tty = setup_params->force_tty;
  int force_unbuffered_io = setup_params->force_unbuffered_io;

  default_io_encoding = setup_params->default_io_encoding;

  io_stdin.stdio_stream  = stdin;
  io_stdin.io_encoding   = default_io_encoding;
  io_stdin.force_tty     = force_tty;
  io_stdin.lookahead     = ___FAL;

  io_stdout.stdio_stream = stdout;
  io_stdout.io_encoding  = default_io_encoding;
  io_stdout.force_tty    = force_tty;
  io_stdout.lookahead    = ___FAL;

  io_stderr.stdio_stream = stderr;
  io_stderr.io_encoding  = default_io_encoding;
  io_stderr.force_tty    = force_tty;
  io_stderr.lookahead    = ___FAL;

  if (force_unbuffered_io)
    {
      setbuf (stdin, 0);
      setbuf (stdout, 0);
      setbuf (stderr, 0);
    }

#ifndef BUFFERED_STDOUT

  if ((force_tty || isatty (fileno (stdout))) && !force_unbuffered_io)
    setbuf (stdout, 0); /* unbuffered stdout, for running Gambit in emacs */

#endif

#ifdef USE_MACOS

#ifdef USE_mac_gui

  putchar ('\0');
  fflush (stdout); /* force call of mac_gui_setup */

  atexit (___cleanup); /* make sure cleanup is done if user quits */

#endif

#ifdef USE_SIOUX

  SIOUXSettings.initializeTB     = 1;
  SIOUXSettings.standalone       = 1;
  SIOUXSettings.setupmenus       = 1;
  SIOUXSettings.autocloseonquit  = 1;
  SIOUXSettings.asktosaveonclose = 0;
  SIOUXSettings.showstatusline   = 0;
  SIOUXSettings.tabspaces        = 8;

  putchar ('\0');
  fflush (stdout); /* force creation of window */

  SIOUXSetTitle ("\pInteraction");

  atexit (___cleanup); /* make sure cleanup is done if user quits */

#endif

  has_IdleUpdate = trap_exists (_IdleUpdate);
  has_WaitNextEvent = trap_exists (_WaitNextEvent);
  has_OSDispatch = trap_exists (_OSDispatch);

  {
    long response;

    has_FindFolder = (Gestalt (gestaltFindFolderAttr, &response) == noErr &&
                      test_bit (response, gestaltFindFolderPresent));
    has_AliasMgr = (Gestalt (gestaltAliasMgrAttr, &response) == noErr &&
                    test_bit (response, gestaltAliasMgrPresent));
    has_AppleEvents = (Gestalt (gestaltAppleEventsAttr, &response) == noErr &&
                       test_bit (response, gestaltAppleEventsPresent));
  }

#endif

#ifdef __WATCOMC__
#ifdef __SW_BW

  _dwSetAboutDlg ("About Gambit",
                  "Copyright (C) 1994-1998 by Marc Feeley, All Rights Reserved.");
  _dwSetAppTitle ("Gambit");
  _dwSetConTitle (0, "Interaction");

  atexit (___cleanup); /* make sure cleanup is done if user quits */

#endif
#endif
}


#ifdef USE_MACOS

#ifdef USE_mac_gui


___HIDDEN int mac_gui_getc (FILE *stdio_stream)
{
  if (stdio_stream == stdin)
    {
      char c;
      long i;
      i = mac_gui_read (&c, 1);
      if (i == -1) /* nothing to read? */
        {
          errno = EINTR;
          return EOF;
        }
      if (i == 0) /* end-of-file? */
        {
          stdio_stream->state.eof = 1;
          return EOF;
        }
      return c;
    }
  else
    return getc (stdio_stream);
}

#undef getc
#define getc mac_gui_getc


___HIDDEN void mac_gui_signal_interrupt (void)
{
  ___raise_interrupt (___INTR_USER);
}


short InstallConsole (short fd)
{
  mac_gui_set_signal_interrupt (mac_gui_signal_interrupt);
  return mac_gui_setup ();
}


void RemoveConsole (void)
{
  while (!mac_gui_quit ())
    while (!mac_gui_update ()) ;
  mac_gui_cleanup ();
}


long WriteCharsToConsole (char *buffer, long n)
{
  return mac_gui_write (buffer, n);
}


long ReadCharsFromConsole (char *buffer, long n)
{
  /*
   * This procedure should never get called in normal circumstances
   * because the I/O system uses "mac_gui_getc" to read characters from the console.
   * It is defined just in case there is a need to use other C library I/O functions
   * for debugging.
   */

  int i, m = 0;
  do
    {
      if (mac_gui_update () && mac_gui_quit ())
        exit (1);
      i = mac_gui_read (buffer, n-m);
      if (i < 1)
        if (m == 0) continue; else break;
      buffer += i;
      m += i;
    } while (m < n);
  return m;
}


char *__ttyname (long fd)
{
  if (fd >= 0 && fd <= 2)
    return "interaction";

  return NULL;
}


#endif

#endif


___HIDDEN void cleanup_io ___PVOID
{
}


___STREAM *___io_stdin ___PVOID
{
  return &io_stdin;
}


___STREAM *___io_stdout ___PVOID
{
  return &io_stdout;
}


___STREAM *___io_stderr ___PVOID
{
  return &io_stderr;
}


___STREAM *___io_open_file
   ___P((char *path, int direction, int encoding),
        (path, direction, encoding)
char *path;
int direction;
int encoding;)
{
  ___STREAM *stream = (___STREAM*) ___alloc_mem (sizeof (___STREAM));
  FILE *stdio_stream;
  char *dir;
  int io_encoding = (encoding == ___IO_DEFAULT_ENCODING)
                    ? default_io_encoding
                    : encoding;
  int binary = (io_encoding > ___IO_UTF8_ENCODING);

  if (stream == 0) /* couldn't allocate space for stream? */
    return 0;

  switch (direction)
    {
    case ___IO_INPUT:
      if (binary) dir = "rb"; else dir = "r";
      break;
    case ___IO_OUTPUT:
      if (binary) dir = "wb"; else dir = "w";
      break;
    default: /* ___IO_INPUT_OUTPUT */
      if (binary) dir = "r+b"; else dir = "r+";
    }

  stdio_stream = fopen (path, dir);
  
  if (stdio_stream == 0) /* stream could not be opened? */
    {
      ___free_mem (stream);
      return 0;
    }

  stream->stdio_stream = stdio_stream;
  stream->kind = ___IO_FILE;
  stream->io_encoding = io_encoding;
  stream->lookahead = ___FAL;

  return stream;
}


___STREAM *___io_open_pipe
   ___P((char *command, int direction, int encoding),
        (command, direction, encoding)
char *command;
int direction;
int encoding;)
{
  ___STREAM *stream = (___STREAM*) ___alloc_mem (sizeof (___STREAM));
  FILE *stdio_stream;
  char *dir;
  int io_encoding = (encoding == ___IO_DEFAULT_ENCODING)
                    ? default_io_encoding
                    : encoding;
  int binary = (io_encoding > ___IO_UTF8_ENCODING);

  if (stream == 0) /* couldn't allocate space for stream? */
    return 0;

  switch (direction)
    {
    case ___IO_INPUT:
      if (binary) dir = "rb"; else dir = "r";
      break;
    case ___IO_OUTPUT:
      if (binary) dir = "wb"; else dir = "w";
      break;
    default: /* ___IO_INPUT_OUTPUT */
      ___free_mem (stream);
      return 0; /* bidirectional pipes not allowed */
    }

  stdio_stream = popen (command, dir);
  
  if (stdio_stream == 0) /* stream could not be opened? */
    {
      ___free_mem (stream);
      return 0;
    }

  stream->stdio_stream = stdio_stream;
  stream->kind = ___IO_PIPE;
  stream->io_encoding = io_encoding;
  stream->lookahead = ___FAL;

  return stream;
}


___WORD ___io_close ___P((___STREAM *stream),(stream)
___STREAM *stream;)
{
  FILE *stdio_stream;
  int x;

  if (stream == 0) /* stream has been freed? */
    return ___FAL;

  stdio_stream = stream->stdio_stream;

  if (stdio_stream == 0) /* stream has been closed? */
    return ___FAL;

  if (stream->kind == ___IO_FILE)
    x = fclose (stdio_stream);
  else
    x = pclose (stdio_stream);

  if (x == EOF)
    {
      ___WORD r = ___FIX(errno);
      errno = ESUCCESS;
      clearerr (stdio_stream);
      return r;
    }

  stream->stdio_stream = 0;
  return ___FAL;
}


___WORD ___io_free ___P((___STREAM *stream),(stream)
___STREAM *stream;)
{
  if (stream == 0) /* stream has been freed? */
    return ___FAL;

  ___free_mem (stream);

  return ___FAL;
}


___HIDDEN ___WORD io_get ___P((___STREAM *stream),(stream)
___STREAM *stream;)
{
  FILE *stdio_stream = stream->stdio_stream;
  int x;

  if (stdio_stream == 0) /* stream is closed? */
    return ___EOF;

  if (feof (stdio_stream))
    {
      clearerr (stdio_stream);
      return ___EOF;
    }

  switch (stream->io_encoding)
    {
    case ___IO_CHAR_ENCODING:
      if ((x = getc (stdio_stream)) != EOF)
        return ___CHR(uchar_to_unicode(x));
      break;

    case ___IO_LATIN1_ENCODING:
    case ___IO_BYTE_ENCODING:
      if ((x = getc (stdio_stream)) != EOF)
        return ___CHR(x);
      break;

    case ___IO_UTF8_ENCODING:
      {
        unsigned char byte;
        ___UCS4 unicode;
        int bits;
        if ((x = getc (stdio_stream)) == EOF)
          goto err;
        byte = x;
        if (byte <= 0x7f)
          return ___CHR(byte);
        if (byte <= 0xbf || byte > 0xfd)
          return ___FIX(EIO);
        unicode = byte; /* upper bits are removed later */
        bits = 6;
        while (byte & 0x40)
          {
            if ((x = getc (stdio_stream)) == EOF)
              goto err;
            if (x <= 0x7f || x > 0xbf)
              return ___FIX(EIO);
            unicode = (unicode << 6) + (x & 0x3f);
            byte <<= 1;
            bits += 5;
          }
        unicode &= ((unsigned long)1<<bits)-1;
        if ((unicode > 0xd7ff && unicode <= 0xdfff) ||
            (unicode > 0xfffd && unicode <= 0xffff))
          return ___FIX(EIO);
        if ((unicode & (~(unsigned long)0<<(bits-5))) == 0)
          return ___FIX(EIO);
        if (unicode > ___MAX_CHR)
          return ___FIX(EDOM);
        return ___CHR(unicode);
      err:
        break;
      }

    case ___IO_UCS2_ENCODING:
      if ((x = getc (stdio_stream)) != EOF)
        {
          ___UCS4 unicode = x;
          if ((x = getc (stdio_stream)) != EOF)
            {
              unicode = (unicode<<8) + x;
              if ((unicode > 0xd7ff && unicode <= 0xdfff) ||
                  (unicode > 0xfffd && unicode <= 0xffff))
                return ___FIX(EIO);
              if (unicode > ___MAX_CHR)
                return ___FIX(EDOM);
              return ___CHR(unicode);
            }
          if (feof (stdio_stream))
            {
              clearerr (stdio_stream);
              return ___FIX(EIO);
            }
          clearerr (stdio_stream);
          return ___FIX(errno);
        }
      break;

    default: /* ___IO_UCS4_ENCODING */
      if ((x = getc (stdio_stream)) != EOF)
        {
          ___UCS4 unicode = x;
          if ((x = getc (stdio_stream)) != EOF)
            {
              unicode = (unicode<<8) + x;
              if ((x = getc (stdio_stream)) != EOF)
                {
                  unicode = (unicode<<8) + x;
                  if ((x = getc (stdio_stream)) != EOF)
                    {
                      unicode = (unicode<<8) + x;
                      if ((unicode > 0xd7ff && unicode <= 0xdfff) ||
                          (unicode > 0xfffd && unicode <= 0xffff))
                        return ___FIX(EIO);
                      if (unicode > ___MAX_CHR)
                        return ___FIX(EDOM);
                      return ___CHR(unicode);
                    }
                }
            }
          if (feof (stdio_stream))
            {
              clearerr (stdio_stream);
              return ___FIX(EIO);
            }
          clearerr (stdio_stream);
          return ___FIX(errno);
        }
      break;
    }

  if (feof (stdio_stream)
#ifdef __BORLANDC__
      || errno == EPIPE
#else
#ifdef __WATCOMC__
      || errno < 0 /* bug in Watcom C??? apparently errno<0 at end of pipe */
#endif
#endif
     )
    {
      clearerr (stdio_stream);
      return ___EOF;
    }

  clearerr (stdio_stream);
  return ___FIX(errno);
}

___WORD ___io_getc ___P((___STREAM *stream),(stream)
___STREAM *stream;)
{
  ___WORD r = stream->lookahead;

  if (r == ___FAL)
    return io_get (stream);

  stream->lookahead = ___FAL;
  return r;
}

___WORD ___io_peek ___P((___STREAM *stream),(stream)
___STREAM *stream;)
{
  ___WORD r = stream->lookahead;

  if (r == ___FAL)
    {
      r = io_get (stream);
      if (!___FIXNUMP(r))
        stream->lookahead = r;
    }

  return r;
}

___WORD ___io_ready ___P((___STREAM *stream),(stream)
___STREAM *stream;)
{
  return ___TRU;
}

___WORD ___io_putc ___P((___STREAM *stream, ___UCS4 unicode),(stream, unicode)
___STREAM *stream;
___UCS4 unicode;)
{
  FILE *stdio_stream = stream->stdio_stream;
  int x;

  if (stdio_stream == 0) /* stream is closed? */
    return ___FAL;

  switch (stream->io_encoding)
    {
    case ___IO_CHAR_ENCODING:
      {
        ___UCS4 c = unicode_to_uchar(unicode);
        if (c > UCHAR_MAX)
          return ___FIX(EDOM);
        x = putc (c, stdio_stream);
        break;
      }

    case ___IO_LATIN1_ENCODING:
    case ___IO_BYTE_ENCODING:
      if (unicode > 0xff)
        return ___FIX(EDOM);
      x = putc (unicode, stdio_stream);
      break;

    case ___IO_UTF8_ENCODING:
      {
        char buf[6];
        char *p1 = buf, *p2 = buf;
        ___utf8_put (&p2, unicode);
        x = EOF;
        while (p1 < p2 && (x = putc (*p1++, stdio_stream)) != EOF) ;
        break;
      }

    case ___IO_UCS2_ENCODING:
      if (unicode > 0xffff)
        return ___FIX(EDOM);
      (x = putc ((unicode>>8) & 0xff, stdio_stream)) != EOF &&
      (x = putc (unicode      & 0xff, stdio_stream));
      break;

    default: /* ___IO_UCS4_ENCODING */
      if (unicode > 0x7fffffff)
        return ___FIX(EDOM);
      (x = putc ((unicode>>24) & 0xff, stdio_stream)) != EOF &&
      (x = putc ((unicode>>16) & 0xff, stdio_stream)) != EOF &&
      (x = putc ((unicode>>8)  & 0xff, stdio_stream)) != EOF &&
      (x = putc (unicode       & 0xff, stdio_stream));
      break;
    }

  if (x == EOF)
    {
      ___WORD r = ___FIX(errno);
      errno = ESUCCESS;
      clearerr (stdio_stream);
      return r;
    }

  return ___FAL;
}

___WORD ___io_flush ___P((___STREAM *stream),(stream)
___STREAM *stream;)
{
  FILE *stdio_stream = stream->stdio_stream;

  if (stdio_stream == 0) /* stream is closed? */
    return ___FAL;

  if (fflush (stdio_stream) == EOF)
    {
      ___WORD r = ___FIX(errno);
      errno = ESUCCESS;
      clearerr (stdio_stream);
      return r;
    }

  return ___FAL;
}

___WORD ___io_width ___P((___STREAM *stream),(stream)
___STREAM *stream;)
{
  return ___FIX(79);
}

___WORD ___io_isatty ___P((___STREAM *stream),(stream)
___STREAM *stream;)
{
  FILE *stdio_stream = stream->stdio_stream;

  if (stdio_stream == 0) /* stream is closed? */
    return ___FAL;

  return ___BOOLEAN((stream->force_tty || isatty (fileno (stdio_stream))));
}


/*---------------------------------------------------------------------------*/

/* Fatal error handling. */

void (*___fatal_error) ___P((char *msg),());

/*
 * The procedure "fatal_error" is the default error handler.  It is
 * assigned to ___fatal_error if no fatal error handler is supplied
 * in the setup parameters.
 */

___HIDDEN void fatal_error ___P((char *msg),(msg)
char *msg;)
{
  fprintf (stderr, "*** FATAL ERROR -- %s\n", msg);
  exit (1);
}


/*---------------------------------------------------------------------------*/

void ___setup_os ___P((___setup_params_struct *setup_params),(setup_params)
___setup_params_struct *setup_params;)
{
  /*
   * To perform correct cleanup when the program terminates an
   * "atexit (___cleanup)" is performed in "setup_io" in certain
   * environments.  There must not be any possibility of program
   * termination through "exit (...)" between the "atexit (...)"
   * and the entry of "___setup_mem".  This guarantees that
   * "___cleanup" does not access dangling pointers.
   */

  if (setup_params->fatal_error == 0)
    ___fatal_error = fatal_error;
  else
    ___fatal_error = setup_params->fatal_error;

  setup_io (setup_params);
  setup_dynamic_load (setup_params);
  setup_interrupt_handling ();
  setup_fp ();
  init_real_time ();
  init_cpu_time ();
  ___set_timer_interval (___DEFAULT_TIMER_INTERVAL);
}


void ___cleanup_os ___PVOID
{
  cleanup_interrupt_handling ();
  cleanup_dynamic_load ();
  cleanup_io ();
}


/*---------------------------------------------------------------------------*/
