/* file: "mem.c" */

/* Copyright (C) 1994-1998 by Marc Feeley, All Rights Reserved.  */

#define ___INCLUDED_FROM_MEM
#define ___VERSION 21
#include "gambit.h"

#include "os.h"
#include "setup.h"
#include "mem.h"
#include "c_intf.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/*---------------------------------------------------------------------------*/

/*
 * Global Scheme variables needed by this module.
 */

___NEED_GLO(___G__23__23_gc_2d_report)

/*---------------------------------------------------------------------------*/

/* 
 * Defining the symbol CONSISTENCY_CHECKS will cause the GC to perform
 * checks that detect when the heap is in an inconsistent state.  This
 * is useful to detect bugs in the GC and the rest of the system.
 * The checks are very extensive and consequently are expensive.  They
 * should only be used for debugging.
 */

#undef CONSISTENCY_CHECKS

/* 
 * Defining the symbol GATHER_STATS will cause the GC to gather
 * statistics on the objects it encounters in the heap.
 */

#undef GATHER_STATS

#ifdef GATHER_STATS

#define MAX_STAT_SIZE 20

___HIDDEN long movable_pair_objs;
___HIDDEN long movable_subtyped_objs[MAX_STAT_SIZE+2];

#endif

/*---------------------------------------------------------------------------*/

/* Allocation and reclamation of aligned blocks of memory.  */


/* 
 * 'alloc_mem_aligned (words, multiplier, modulus)' allocates an
 * aligned block of memory through the '___alloc_mem' function.
 * 'words' is the size of the block in words and 'multiplier' and
 * 'modulus' specify its alignment in words.  'multiplier' must be a
 * power of two and 0<=modulus<multiplier.  The pointer returned
 * corresponds to an address that is equal to
 * (i*multiplier+modulus)*sizeof(___WORD) for some 'i'.
 */

___HIDDEN void *alloc_mem_aligned
   ___P((unsigned long words, unsigned int multiplier, unsigned int modulus),
        (words, multiplier, modulus)
unsigned long words;
unsigned int multiplier;
unsigned int modulus;)
{
  void *container;    /* pointer to block returned by ___alloc_mem */
  unsigned int extra; /* space for alignment to multiplier */

  /* Make sure alignment is sufficient for pointers */

  if (multiplier < sizeof(void*)/sizeof(___WORD))
    multiplier = sizeof(void*)/sizeof(___WORD);

  /* How many extra bytes are needed for padding */

  extra = (multiplier*sizeof(___WORD))-1;
  if (modulus < sizeof(void*)/sizeof(___WORD))
    extra += sizeof(void*);

  container = ___alloc_mem (extra + ((words+modulus)*sizeof(___WORD)));

  if (container == 0)
    return 0;
  else
    {
      void *ptr = (void*)((((long)container+extra) &
                           (-multiplier*sizeof(___WORD))) +
                          modulus*sizeof(___WORD));
      *((void**)(((long)ptr-sizeof(void*)) & (-sizeof(void*)))) = container;
      return ptr;
    }
}


/* 
 * 'free_mem_aligned (ptr)' reclaims the aligned block of memory 'ptr'
 * that was allocated using 'alloc_mem_aligned'.
 */

___HIDDEN void free_mem_aligned ___P((void *ptr),(ptr)
void *ptr;)
{
  ___free_mem (*((void**)(((long)ptr-sizeof(void*)) & (-sizeof(void*)))));
}


/*---------------------------------------------------------------------------*/

/* Allocation of movable objects.  */

/* 
 * Movable Scheme objects are allocated in an area of memory
 * distributed in multiple non-contiguous sections (collectively
 * called the "msections").  All sections are of the same size and are
 * allocated through the '___alloc_mem' function.  The number of
 * sections can expand and contract to accommodate the needs of the
 * program.
 */

typedef struct msect
  {
    int pos;            /* position in msections's 'sections' array */
    ___WORD *alloc;     /* allocation pointer, grows towards high addresses */
    struct msect *prev; /* previous section in list of sections */
    struct msect *next; /* next section in list of sections */
    ___WORD base[1];    /* content of section */
  } msection;

#define sizeof_msection(n) (sizeof(msection)+((n)-1)*sizeof(___WORD))

typedef struct
  {
    unsigned int max_nb_sections; /* actual size of 'sections' array */
    unsigned int nb_sections;     /* number of sections */
    msection *head, *tail;        /* head and tail of list of sections */
    msection *sections[1];        /* each section ordered by address */
                                  /* (increasing order if ___ALLOC_MEM_UP */
                                  /* is defined otherwise decreasing order) */
  } msections;

#define sizeof_msections(n) (sizeof(msections)+((n)-1)*sizeof(msection*))


/* 
 * 'find_msection (ms, ptr)' finds the index of the section that
 * contains the pointer 'ptr' within the msections 'ms'.  More
 * precisely, if ___ALLOC_MEM_UP is defined, it returns the integer
 * 'i' (-1<=i<=n-1) such that 'ptr' is between the start of section i
 * and section i+1.  -1 is returned if 'ptr' is lower than the lowest
 * section and 'n' is returned if 'ptr' is not lower than the highest
 * section.  If ___ALLOC_MEM_UP is not defined, it returns the integer
 * 'i' (0<=i<=n) such that 'ptr' is between the start of section i and
 * section i-1.  n is returned if 'ptr' is lower than the lowest
 * section and 0 is returned if 'ptr' is not lower than the highest
 * section.
 */

___HIDDEN int find_msection ___P((msections *ms, void *ptr),(ms, ptr)
msections *ms;
void *ptr;)
{
  unsigned int ns = ms->nb_sections;
  msection **sections = ms->sections;
#ifdef ___ALLOC_MEM_UP
  if ((ns == 0) || (ptr < (void*)sections[0]))
    return -1;
#else
  if ((ns == 0) || (ptr < (void*)sections[ns-1]))
    return ns;
#endif
  else
    {
      /* binary search */
      unsigned int lo = 0, hi = ns-1;

      /* loop invariant: lo <= find_msection (ms, ptr) <= hi */
      while (lo < hi)
        {
          unsigned int mid = (lo+hi)/2; /* lo <= mid < hi */
#ifdef ___ALLOC_MEM_UP
          if (ptr < (void*)sections[mid+1]) hi = mid; else lo = mid+1;
#else
          if (ptr < (void*)sections[mid]) lo = mid+1; else hi = mid;
#endif
        }
      return lo;
    }
}


/* 
 * 'adjust_msections (msp, n)' contracts or expands the msections
 * pointed to by 'msp' so that it contains 'n' sections.  When the
 * msections is contracted, the sections at the top of the 'sections'
 * array will be reclaimed (these are the highest sections if
 * ___ALLOC_MEM_UP is defined, otherwise they are the lowest
 * sections).  When expanding the msections there may not be enough
 * memory to allocate new sections so the operation may fail.  However
 * 'adjust_msections' will always leave the msections in a consistent
 * state and there will be at least as many sections as when the
 * expansion was started.  Failure can be detected by checking the
 * 'nb_sections' field.
 */

___HIDDEN void adjust_msections ___P((msections **msp, unsigned int n),(msp, n)
msections **msp;
unsigned int n;)
{
  unsigned int max_ns, ns;
  msections *ms = *msp;
  msection *hd, *tl;

  if (ms == 0)
    {
      max_ns = 0;
      ns = 0;
      hd = 0;
      tl = 0;
    }
  else
    {
      max_ns = ms->max_nb_sections;
      ns = ms->nb_sections;
      hd = ms->head;
      tl = ms->tail;
    }

  if ((ms == 0) || (n > max_ns))
    {
      /* must allocate a new msections structure */

      msections *new_ms;
      int i;

      while (n > max_ns) /* grow max_nb_sections until big enough */
        max_ns = 2*max_ns + 1;

      new_ms = (msections*)
        alloc_mem_aligned (___WORDS(sizeof_msections(max_ns)), 1, 0);

      if (new_ms == 0)
        return;

      new_ms->max_nb_sections = max_ns;
      new_ms->nb_sections = ns;
      new_ms->head = hd;
      new_ms->tail = tl;
      for (i=ns-1; i>=0; i--)
        new_ms->sections[i] = ms->sections[i];

      if (ms != 0)
        free_mem_aligned (ms);

      ms = new_ms;

      *msp = ms;
    }

  if (n < ns)
    {
      /* contraction of the msections */

      unsigned int j;

      while (ns > n)
        {
          msection *s = tl;
          tl = tl->prev;
          if (tl == 0)
            hd = 0;
          else
            tl->next = 0;
          for (j=s->pos; j<ns-1; j++)
            {
              ms->sections[j] = ms->sections[j+1];
              ms->sections[j]->pos = j;
            }
          free_mem_aligned (s);
          ns--;
        }

      ms->nb_sections = ns;
      ms->head = hd;
      ms->tail = tl;

      /* contraction of the msections structure is not implemented */
    }
  else
    {
      /* expansion of the msections */

      int i, j;

      while (ns < n)
        {
          msection *s = (msection*)
            alloc_mem_aligned (___WORDS(sizeof_msection(___MSECTION_SIZE)),
                               1,
                               0);
          if (s == 0)
            return;
          i = find_msection (ms, (void*)s);
#ifdef ___ALLOC_MEM_UP
          i++;
#endif
          for (j=ns; j>i; j--)
            {
              ms->sections[j] = ms->sections[j-1];
              ms->sections[j]->pos = j;
            }
          ms->sections[i] = s;
          if (tl == 0)
            hd = s;
          else
            tl->next = s;
          s->pos = i;
          s->prev = tl;
          s->next = 0;
          tl = s;
          ms->nb_sections = ++ns;
          ms->head = hd;
          ms->tail = tl;
        }
    }
}


/* 
 * 'free_msections (msp)' releases all memory associated with the
 * msections pointed to by 'msp'.
 */

___HIDDEN void free_msections ___P((msections **msp),(msp)
msections **msp;)
{
  msections *ms = *msp;
  if (ms != 0)
    {
      int i;

      for (i=ms->nb_sections-1; i>=0; i--)
        free_mem_aligned (ms->sections[i]);

      free_mem_aligned (ms);

      *msp = 0;
    }
}


/*---------------------------------------------------------------------------*/

/* Allocation of permanent objects.  */

/* 
 * Permanent objects are allocated in sections called "psections".
 * Each section contains multiple objects.  The sections are kept in a
 * list so that the storage they occupy can be reclaimed when the
 * program terminates.
 */

___HIDDEN void *psections;       /* list of psections */
___HIDDEN ___WORD *palloc_ptr;   /* allocation pointer in current psection */
___HIDDEN ___WORD *palloc_limit; /* allocation limit in current psection */


/* 
 * 'alloc_mem_aligned_psection (words, multiplier, modulus)' allocates
 * an aligned block of memory inside a new psection.  'words' is the
 * size of the block in words and 'multiplier' and 'modulus' specify
 * its alignment in words.  'multiplier' must be a power of two and
 * 0<=modulus<multiplier.  The pointer returned corresponds to an
 * address that is equal to (i*multiplier+modulus)*sizeof(___WORD) for
 * some 'i'.
 */

___HIDDEN void *alloc_mem_aligned_psection
   ___P((unsigned long words, unsigned int multiplier, unsigned int modulus),
        (words, multiplier, modulus)
unsigned long words;
unsigned int multiplier;
unsigned int modulus;)
{
  void *container;

  /* Make sure alignment is sufficient for pointers */

  if (multiplier < sizeof(void*)/sizeof(___WORD))
    multiplier = sizeof(void*)/sizeof(___WORD);

  /* Make space for psection link and modulus */

  if (modulus < sizeof(void*)/sizeof(___WORD))
    modulus += multiplier;

  /* Allocate container */

  container = alloc_mem_aligned (words+modulus, multiplier, 0);

  if (container == 0)
    return 0;
  else
    {
      *(void**)container = psections;
      psections = container;
      return (void*)(modulus+(___WORD*)container);
    }
}


/* 
 * 'alloc_mem_aligned_perm (words, multiplier, modulus)' allocates an
 * aligned block of memory inside a psection.  If there is enough free
 * space in a previously allocated psection that psection is used,
 * otherwise a new psection is allocated.  'words' is the size of the
 * block in words and 'multiplier' and 'modulus' specify its alignment
 * in words.  'multiplier' must be a power of two and
 * 0<=modulus<multiplier.  The pointer returned corresponds to an
 * address that is equal to (i*multiplier+modulus)*sizeof(___WORD) for
 * some 'i'.
 */

___HIDDEN void *alloc_mem_aligned_perm
   ___P((unsigned long words, unsigned int multiplier, unsigned int modulus),
        (words, multiplier, modulus)
unsigned long words;
unsigned int multiplier;
unsigned int modulus;)
{
  long waste;
  ___WORD *base;

  /*
   * Try to satisfy request in current psection.
   */

  if (palloc_ptr != 0)
    {
      ___WORD *new_palloc_ptr;
      base = modulus +
             (___WORD*)(((long)(palloc_ptr+multiplier-1-modulus)) &
                        (-multiplier*sizeof(___WORD)));
      new_palloc_ptr = base+words;
      if (new_palloc_ptr <= palloc_limit) /* did it fit in the psection? */
        {
          palloc_ptr = new_palloc_ptr;
          return base;
        }
      waste = palloc_limit - palloc_ptr;
    }
  else
    waste = 0;

  /*
   * Request can't be satisfied in current psection so we must
   * allocate a new psection.
   */

  if (waste > ___PSECTION_WASTE || words > ___PSECTION_SIZE)
    return alloc_mem_aligned_psection (words, multiplier, modulus);

  base = (___WORD*)
    alloc_mem_aligned_psection (___PSECTION_SIZE, multiplier, modulus);
  if (base != 0)
    {
      palloc_ptr = base+words;
      palloc_limit = base+___PSECTION_SIZE;
    }

  return base;
}


___HIDDEN void free_psections ___PVOID
{
  void *base = psections;
  psections = 0;
  while (base != 0)
    {
      void *link = *(void**)base;
      free_mem_aligned (base);
      base = link;
    }
}


___glo_struct *___alloc_global_var ___PVOID
{
  return (___glo_struct*)
    alloc_mem_aligned_perm (___WORDS(sizeof(___glo_struct)), 1, 0);
}


/*---------------------------------------------------------------------------*/

/* Constants related to representation of permanent and still objects: */

#ifdef ___USE_HANDLES
#define ___PERM_HAND_OFS 0
#define ___PERM_BODY_OFS 2
#else
#define ___PERM_HAND_OFS ___PERM_BODY_OFS
#define ___PERM_BODY_OFS 1
#endif

#define ___STILL_LINK_OFS 0
#define ___STILL_REFCOUNT_OFS 1
#define ___STILL_LENGTH_OFS 2
#define ___STILL_MARK_OFS 3
#ifdef ___USE_HANDLES
#define ___STILL_HAND_OFS 4
#define ___STILL_BODY_OFS 6
#else
#define ___STILL_HAND_OFS ___STILL_BODY_OFS
#define ___STILL_BODY_OFS (5+1)/************/
#endif

/*
 * 'normal_overflow_reserve' is the number of words reserved in the heap
 * in normal circumstances for handling heap overflows.  When the heap
 * overflows, this space will be made available to the heap overflow
 * handler.
 *
 * 'overflow_reserve' is the number of words currently reserved in the heap
 * for handling heap overflows.  If a heap overflow has occured, and there
 * has not yet been a GC that at least made 'normal_overflow_reserve' free,
 * then 'overflow_reserve' will be 0.
 */

___HIDDEN long normal_overflow_reserve, overflow_reserve;

___HIDDEN unsigned long stack_cache_size;/* size of stack non-fudge in words */
___HIDDEN unsigned long min_heap_size;/* size of heap in words (lower bound) */
___HIDDEN unsigned long max_heap_size;/* size of heap in words (upper bound) */

___HIDDEN long words_non_movable; /* words occupied by non-movable objs */

___HIDDEN long words_prev_msections; /* words occupied by movable objs */
                                     /* not including current section  */

/* total words occupied in heap including current section */
#define WORDS_MOVABLE ((words_prev_msections+(alloc_ptr-start_ptr))*2)

/* total words occupied in heap including current section */
#define WORDS_OCCUPIED (words_non_movable+WORDS_MOVABLE)

/* number of words in heap */
#define WORDS_IN_HEAP \
(the_msections->nb_sections*2*((___MSECTION_SIZE>>1)-___MSECTION_FUDGE+1) \
- overflow_reserve * 2)

___HIDDEN ___WORD still_objs;         /* list of still objects */
___HIDDEN ___WORD still_objs_to_scan; /* still objects remaining to scan */

___HIDDEN msections *the_msections; /* the msections */

___HIDDEN int tospace_at_top;      /* location of tospace in each section */

___HIDDEN msection *alloc_msection;/* section currently being allocated in */
___HIDDEN int nb_prev_msections;   /* nb of sections filled before current */
___HIDDEN ___WORD *start_ptr;      /* start of allocation in current section */
___HIDDEN ___WORD *alloc_ptr;      /* allocation pointer in current section */
___HIDDEN ___WORD *alloc_limit;    /* allocation limit in current section */

___HIDDEN msection *scan_msection; /* section currently being scanned */
___HIDDEN ___WORD *scan_ptr;       /* scan pointer in section being scanned */
___HIDDEN long words_allocated_for_cont; /* for allocation statistics */


/*
 * '___alloc_scmobj (subtype, bytes, kind)' allocates a permanent
 * or still Scheme object (depending on 'kind') of subtype
 * 'subtype' with a body containing 'bytes' bytes, and returns it
 * as an encoded Scheme object.  A permanent object is allocated
 * when 'kind' = ___PERM and a still object is allocated when
 * 'kind' = ___STILL.  The initialization of the object's body must
 * be done by the caller.  In the case of still objects this
 * initialization must be done before the next allocation is
 * requested.  The 'refcount' field of still objects is initially
 * 1.  The value ___FAL is returned if the object can not be
 * allocated.
 */

___EXP_FUNC(___WORD,___alloc_scmobj)
   ___P((unsigned int subtype, unsigned long bytes, unsigned int kind),
        (subtype, bytes, kind)
unsigned int subtype;
unsigned long bytes;
unsigned int kind;)
{
  void *ptr;
  ___processor_state pstate = ___PSTATE;
  unsigned long words = (kind==___PERM ? ___PERM_BODY_OFS : ___STILL_BODY_OFS)
                        + ___WORDS(bytes);

  alloc_ptr = pstate->hp; /* needed by 'WORDS_OCCUPIED' */
  words_non_movable += words;
  if (WORDS_OCCUPIED > WORDS_IN_HEAP)
    {
      int overflow = ___gc ();
      alloc_ptr = pstate->hp; /* needed by 'WORDS_OCCUPIED' */
      if (overflow || WORDS_OCCUPIED > WORDS_IN_HEAP)
        {
          words_non_movable -= words;
          overflow_reserve = normal_overflow_reserve;
          return ___FAL;
        }
    }

  /* 
   * Some objects, such as ___sF64VECTOR and ___sFLONUM and
   * ___sPOINTER, must have a body that is aligned on a multiple of 8
   * on some machines.  Here, we force alignment to a multiple of 8
   * even if not necessary in all cases because it is typically more
   * efficient due to a better utilization of the cache.
   */

  if (kind == ___PERM)
    ptr = alloc_mem_aligned_perm (words,
                                  8>>___LWS,
                                  (-___PERM_BODY_OFS)&((8>>___LWS)-1));
  else
    ptr = alloc_mem_aligned (words,
                             8>>___LWS,
                             (-___STILL_BODY_OFS)&((8>>___LWS)-1));

  if (ptr == 0)
    {
      words_non_movable -= words;
      return ___FAL;
    }
  else if (kind == ___PERM)
    {
      ___WORD *base = (___WORD*)ptr;

#ifdef ___USE_HANDLES
      base[___PERM_HAND_OFS] = (___WORD)(base+___PERM_BODY_OFS-___BODY_OFS);
#endif
      base[___PERM_BODY_OFS-1] = ___MAKE_HD(bytes, subtype, ___PERM);

      return ___TAG((base + ___PERM_HAND_OFS - ___BODY_OFS),
                    (subtype == ___sPAIR ? ___tPAIR : ___tSUBTYPED));
    }
  else
    {
      ___WORD *base = (___WORD*)ptr;

      base[___STILL_LINK_OFS] = still_objs;
      still_objs = (___WORD)base;
      base[___STILL_REFCOUNT_OFS] = 1;
      base[___STILL_LENGTH_OFS] = words;
#ifdef ___USE_HANDLES
      base[___STILL_HAND_OFS] = (___WORD)(base+___STILL_BODY_OFS-___BODY_OFS);
#endif
      base[___STILL_BODY_OFS-1] = ___MAKE_HD(bytes, subtype, ___STILL);

      return ___TAG((base + ___STILL_HAND_OFS - ___BODY_OFS),
                    (subtype == ___sPAIR ? ___tPAIR : ___tSUBTYPED));
    }
}


/*
 * '___make_pair (car, cdr, kind)' creates a Scheme pair having the
 * values 'car' and 'cdr' in its CAR and CDR fields.  A permanent
 * or still object is allocated, depending on 'kind' (___PERM for
 * permanent object, ___STILL for still object).  The value ___FAL
 * is returned if the object can not be allocated.
 */

___EXP_FUNC(___WORD,___make_pair)
   ___P((___WORD car, ___WORD cdr, unsigned int kind),
        (car, cdr, kind)
___WORD car;
___WORD cdr;
unsigned int kind;)
{
  ___WORD obj = ___alloc_scmobj (___sPAIR, ___PAIR_SIZE<<___LWS, kind);
  if (obj == ___FAL)
    return ___FAL;
  else
    {
      ___PAIR_CAR(obj) = car;
      ___PAIR_CDR(obj) = cdr;
      return obj;
    }
}


/*
 * '___make_vector (length, init, kind)' creates a Scheme vector of
 * length 'length' and initialized with the value 'init'.  A
 * permanent or still object is allocated, depending on 'kind'
 * (___PERM for permanent object, ___STILL for still object).  The
 * value ___FAL is returned if the object can not be allocated.
 */

___EXP_FUNC(___WORD,___make_vector)
   ___P((unsigned long length, ___WORD init, unsigned int kind),
        (length, init, kind)
unsigned long length;
___WORD init;
unsigned int kind;)
{
  if (length >= (1<<(32-___LF-___LWS)))
    return ___FAL;
  else
    {
      ___WORD obj = ___alloc_scmobj (___sVECTOR, length<<___LWS, kind);
      if (obj == ___FAL)
        return ___FAL;
      else
        {
          unsigned int i;
          for (i=0; i<length; i++)
            ___FIELD(obj, i) = init;
          return obj;
        }
    }
}


/* 
 * '___make_string (length, init, kind)' creates a Scheme string of
 * length 'length' and initialized with the character 'init'.  A
 * permanent or still object is allocated, depending on 'kind'
 * (___PERM for permanent object, ___STILL for still object).  The
 * value ___FAL is returned if the object can not be allocated.
 */

___EXP_FUNC(___WORD,___make_string)
   ___P((unsigned long length, char init, unsigned int kind),
        (length, init, kind)
unsigned long length;
char init;
unsigned int kind;)
{
  if (length >= (1<<(32-___LF-1)))
    return ___FAL;
  else
    {
      ___WORD obj = ___alloc_scmobj (___sSTRING, length<<___LCS, kind);
      if (obj == ___FAL)
        return ___FAL;
      else
        {
          unsigned int i;
          for (i=0; i<length; i++)
            ((char*)___BODY_AS(obj,___tSUBTYPED))[i] = init;
          return obj;
        }
    }
}


/*
 * '___still_obj_refcount_inc (obj)' increments the reference count
 * of the still object 'obj'.
 */

___EXP_FUNC(void,___still_obj_refcount_inc) ___P((___WORD obj),(obj)
___WORD obj;)
{
  ___UNTAG(obj)[___BODY_OFS - ___STILL_BODY_OFS + ___STILL_REFCOUNT_OFS]++;
}


/*
 * '___still_obj_refcount_dec (obj)' decrements the reference count
 * of the still object 'obj'.
 */

___EXP_FUNC(void,___still_obj_refcount_dec) ___P((___WORD obj),(obj)
___WORD obj;)
{
  ___UNTAG(obj)[___BODY_OFS - ___STILL_BODY_OFS + ___STILL_REFCOUNT_OFS]--;
}


/*---------------------------------------------------------------------------*/

/*

Object representation.

Memory allocated Scheme objects can be allocated using one of three
allocation strategies:

   Permanently allocated:
     These objects, called 'permanent objects' for short, are never
     moved or reclaimed, and all pointers to memory allocated
     objects they contain must point to permanent objects.  As a
     consequence, the GC does not have to scan permanent objects.
     Permanent objects can be allocated on the C heap, but they are
     typically allocated in C global variables and structures that are
     set up when the program starts up or when a module is dynamically
     loaded.

   Still dynamically allocated:
     These objects, called 'still objects' for short, are allocated on
     the C heap.  Still objects are never moved but they can be reclaimed by
     the GC.  A mark-and-sweep GC is used to garbage-collect still objects.

   Movable dynamically allocated:
     These objects, called 'movable objects' for short, are allocated
     in an area of memory that is managed by a compacting GC.  The GC can
     move and reclaim movable objects.

Scheme objects are encoded using integers of type ___WORD (defined as
a 'long' integer on most systems).  A ___WORD either encodes an
immediate value or encodes a pointer when the object is memory
allocated.  The two lower bits of a ___WORD contain a primary type tag
for the object and the other bits contain the immediate value or the
pointer.  Because all memory allocated objects are aligned on ___WORD
boundaries (and a ___WORD is either 4 or 8 bytes), the two lower bits
of pointers are zero and can be used to store the tag without reducing
the address space.  The four tags are:

 immediate:
   ___tFIXNUM    object is a small integer (fixnum)
   ___tSPECIAL   object is a boolean, character, or other immediate

 memory allocated:
   ___tPAIR      object is a pair
   ___tSUBTYPED  object is memory allocated but not a pair

A special type of object exists to support object finalization: 'will'
objects.  Wills contain an object (the will's testator) and may also
contain a procedure (the will's action procedure).  An object is
finalizable when all paths to the object from the root set pass through
a will.  When the GC detects that an object is finalizable the
corresponding wills are placed on a list of executable wills (for wills
with an action procedure).  Following the GC, this list is traversed to
invoke the action procedures.

All memory allocated objects, including pairs, are composed of at
least a head and a body.  The head is a single ___WORD that contains
3 "head" tag bits (the 3 lower bits), a subtype tag (the next
5 bits), and the length of the object in bytes (the remaining bits).
The head immediately precedes the body of the object, which contains the
rest of the information associated with the object.  Depending on the
subtype, the body can contain raw binary data (such as when the object
is a string) and Scheme objects (such as when the object is a vector).
Memory allocated objects have the following layout:

     _head_   _____body______
    /      \ /               \
   +--------+--------+--------+
   |llllssst|        |        |
   +--------+--------+--------+
     ^   ^ ^
     |   | |
length   | |
   subtype head tag

Of the 8 possible head tags, only 5 are currently used:

   ___PERM     (P) the object is a permanent object
   ___STILL    (S) the object is a still object
   ___MOVABLE0 (M) the object is a movable object in generation 0
   ___FORW     (F) the object has been moved by the GC (counts as 2 tags)

Permanent objects have the following layout:

     _head_   _____body______
    /      \ /               \
   +--------+--------+--------+
   |       P|        |        |
   +--------+--------+--------+

Still objects have the following layout:

     _link_   _ref__   length   _mark_   _head_   _____body______
    /      \ / count\ /      \ /      \ /      \ /               \
   +--------+--------+--------+--------+--------+--------+--------+
   |        |        |        |        |       S|        |        |
   +--------+--------+--------+--------+--------+--------+--------+

All still objects are linked in a list using the 'link' field.  The
'refcount' field contains a reference count, which counts the number
of pointers to this object that are hidden from the GC (typically
these hidden pointers are in C data structures).  When 'refcount' is
zero, the object will survive a GC only if it is pointed to by a GC
root or a live Scheme object.  The 'length' field contains the
length of the object and is only used to maintain statistics on
the space allocated.  The 'mark' field is used by the GC to
indicate that the object has been marked (at the start of a GC it is
set to -1).  The 'mark' field links all objects that have been marked
but have not yet been scanned.  It contains a pointer to the next
still object that needs to be scanned.

Movable objects have the following layout:

     _head_   _____body______
    /      \ /               \
   +--------+--------+--------+
   |       M|        |        |
   +--------+--------+--------+

When a movable object is moved by the GC, the head is replaced with a
pointer to the copy, tagged with ___FORW.

Layout of body.

     _head_   __________body__________
    /      \ /                        \
   +--------+--------+--------+--------+
   |        | field_0| field_1|  etc.  |
   +--------+--------+--------+--------+

Some types of objects have bodies that only contain pointers to other
Scheme objects.  For example, pairs have two fields (car and cdr) and
vectors have one field per element.  Other object types have bodies
that only contain raw binary data (such as strings and bignums).  The
remaining object types have bodies that contain both pointers to
Scheme objects and raw binary data.  Their layout is summarized below.

Symbols:
    subtype = ___sSYMBOL
    field_0 = name (a Scheme string)
    field_1 = hash code (fixnum)
    field_2 = C pointer to global variable (0 if none allocated)

    Note: interned symbols must be permanently allocated;
          uninterned symbols can be permanent, still or movable

Keywords:
    subtype = ___sKEYWORD
    field_0 = name (a Scheme string) not including the trailing ':'
    field_1 = hash code (fixnum)

Procedures:

  non-closures (toplevel procedures and return addresses):
    subtype = ___sPROCEDURE
    field_0 = C pointer to field_0 - ___BODY_OFS
    field_1 = C pointer to host C procedure
    field_2 = return frame descriptor

  closures:
    subtype = ___sPROCEDURE
    field_0 = C pointer to field_0 of entry procedure - ___BODY_OFS
    field_1 = free variable 1
    field_2 = free variable 2
    ...

    Note: the entry procedure must be a non-closure procedure

Wills:
    subtype = ___sWILL
    field_0 = next will in will list with special tag in lower bits
    field_1 = testator object
    field_2 = action procedure (if this field exists)

    Note: wills must be movable

Continuations:
    subtype = ___sCONTINUATION
    field_0 = first frame (C pointer to stack at first and then Scheme obj)
    field_1 = return address
    field_2 = dynamic-environment

Frame:
    subtype = ___sFRAME
    field_0 = next frame (C pointer if unscanned, otherwise Scheme obj)
    field_1 = frame slot 1
    field_2 = frame slot 2
    ...

*/

/*---------------------------------------------------------------------------*/

#ifdef CONSISTENCY_CHECKS

___HIDDEN int reference_location; /* where is offending reference located */

#define IN_OBJECT     0
#define IN_REGISTER   1
#define IN_GLOBAL_VAR 2
#define IN_WILL_LIST  3
#define IN_STACK      4

___HIDDEN ___WORD *container_body; /* pointer to body of object      */
                                   /* containing offending reference */


___HIDDEN void explain_problem ___P((___WORD obj, char *msg),(obj, msg)
___WORD obj;
char *msg;)
{
  fprintf (stderr, ">>> The object 0x%08lx %s\n", obj, msg);

  switch (reference_location)
    {
    case IN_OBJECT:
      {
        ___WORD head = container_body[-1];
        unsigned long words = ___HD_WORDS(head);
        unsigned int subtype = ___HD_SUBTYPE(head);
        unsigned int i;

        fprintf (stderr, ">>> The reference was found in ");
        if (___HD_TYP(head) == ___PERM)
          fprintf (stderr, "___PERM ");
        else if (___HD_TYP(head) == ___STILL)
          fprintf (stderr, "___STILL ");
        else if (___HD_TYP(head) == ___MOVABLE0)
          fprintf (stderr, "___MOVABLE0 ");
        else if (___TYP(head) == ___FORW)
          fprintf (stderr, "___FORW ");
        else
          fprintf (stderr, "UNKNOWN ");
        fprintf (stderr, "object with body at 0x%08lx:\n", container_body);
        fprintf (stderr, ">>>  subtype = %d\n", subtype);
        fprintf (stderr, ">>>  length  = %ld words\n", words);
        for (i=0; i<words; i++)
          fprintf (stderr, ">>>  body[%2d] = 0x%08lx\n", i, container_body[i]);
        break;
      }

    case IN_REGISTER:
      fprintf (stderr, ">>> The reference was found in a register\n");
      break;

    case IN_GLOBAL_VAR:
      fprintf (stderr, ">>> The reference was found in a global variable\n");
      break;

    case IN_WILL_LIST:
      fprintf (stderr, ">>> The reference was found in a will list\n");
      break;

    case IN_STACK:
      fprintf (stderr, ">>> The reference was found in the stack\n");
      break;
    }

  fflush (stderr);
}


___HIDDEN void bug ___P((___WORD obj, char *msg),(obj, msg)
___WORD obj;
char *msg;)
{
  fprintf (stderr, ">>> The GC has detected the following inconsistency:\n");
  explain_problem (obj, msg);
  ___fatal_error ("GC inconsistency detected");
}


___HIDDEN void validate_old_obj ___P((___WORD obj),(obj)
___WORD obj;)
{
  ___WORD *hd_ptr = ___BODY(obj)-1;
  ___WORD head;
  int i = find_msection (the_msections, hd_ptr);
  if (i >= 0 && i < the_msections->nb_sections)
    {
      long pos = hd_ptr - the_msections->sections[i]->base;
      if (pos >= 0 && pos < ___MSECTION_SIZE)
        {
          head = *hd_ptr;
          if (___TYP(head) == ___FORW)
            {
              ___WORD *hd_ptr2 = ___UNTAG_AS(head,___FORW)+___BODY_OFS-1;
              int i2 = find_msection (the_msections, hd_ptr2);
              if (i2 >= 0 && i2 < the_msections->nb_sections)
                {
                  long pos2 = hd_ptr2 - the_msections->sections[i2]->base;
                  if (tospace_at_top
                      ? (pos2 < ___MSECTION_SIZE>>1 ||
                         pos2 >= ___MSECTION_SIZE)
                      : (pos2 < 0 ||
                         pos2 >= ___MSECTION_SIZE>>1))
                    bug (obj, "was copied outside of tospace");
                  else if (___HD_TYP((*hd_ptr2)) != ___MOVABLE0)
                    bug (obj, "was copied and copy is not ___MOVABLE0");
                }
              else
                bug (obj, "was copied outside of tospace");
            }
          else if (___HD_TYP(head) != ___MOVABLE0)
            bug (obj, "should be ___MOVABLE0");
          else if (tospace_at_top
                   ? (pos >= ___MSECTION_SIZE>>1 &&
                      pos < ___MSECTION_SIZE)
                   : (pos >= 0 &&
                      pos < ___MSECTION_SIZE>>1))
            bug (obj, "is in tospace");
          return;
        }
    }
  head = *hd_ptr; /* this dereference will likely bomb if there is a bug */
  if (___HD_TYP(head) != ___PERM && ___HD_TYP(head) != ___STILL)
    bug (obj, "is not ___PERM or ___STILL");
}

#endif


___HIDDEN ___WORD *start_of_fromspace ___P((msection *s),(s)
msection *s;)
{
  if (tospace_at_top)
    return s->base;
  else
    return s->base + (___MSECTION_SIZE>>1);
}


___HIDDEN ___WORD *start_of_tospace ___P((msection *s),(s)
msection *s;)
{
  if (tospace_at_top)
    return s->base + (___MSECTION_SIZE>>1);
  else
    return s->base;
}


___HIDDEN void next_alloc_msection ___PVOID
{
  if (alloc_msection == 0)
    {
      nb_prev_msections = 0;
      words_prev_msections = 0;
      alloc_msection = the_msections->head;
    }
  else
    {
      nb_prev_msections++;
      words_prev_msections += alloc_ptr - start_ptr;
      alloc_msection->alloc = alloc_ptr;
      alloc_msection = alloc_msection->next;
    }
  if (alloc_msection == 0)
    ___fatal_error ("Heap overflow");
  start_ptr = start_of_tospace (alloc_msection);
  alloc_ptr = start_ptr;
  alloc_limit = start_ptr + (___MSECTION_SIZE>>1);
}


___HIDDEN void mark_array ___P((___WORD *start, ___WORD n),(start, n)
___WORD *start;
___WORD n;)
{
  ___WORD *alloc = alloc_ptr;
  ___WORD *limit = alloc_limit;

  while (n > 0)
    {
      ___WORD obj = *start;
      if (___MEM_ALLOCATED(obj))
        {
          ___WORD *body;
          ___WORD head;
          unsigned int head_typ;
          unsigned int subtype;

#ifdef CONSISTENCY_CHECKS
          validate_old_obj (obj);
#endif

          body = ___UNTAG(obj) + ___BODY_OFS;
          head = body[-1];
          subtype = ___HD_SUBTYPE(head);
          head_typ = ___HD_TYP(head);

          if (head_typ == ___MOVABLE0)
            {
              unsigned long words = ___HD_WORDS(head);
#if ___WS == 4
              int pad = 0;
              while (alloc + words + (subtype >= ___sF64VECTOR ? 2 : 1) > limit)
#else
              while (alloc + words + 1 > limit)
#endif
                {
                  alloc_ptr = alloc;
                  next_alloc_msection ();
                  alloc = alloc_ptr;
                  limit = alloc_limit;
                }
#if ___WS == 4
              /*
               * ___sF64VECTOR and ___sFLONUM and ___sPOINTER need to be
               * aligned on a multiple of 8.
               */

              if (subtype >= ___sF64VECTOR)
                if (((___WORD)alloc&7) == 0)
                  *alloc++ = ___MAKE_HD_WORDS(0, ___sVECTOR);
                else
                  pad = 1;
#endif
#ifdef GATHER_STATS
              if (subtype == ___sPAIR)
                movable_pair_objs++;
              else if (words <= MAX_STAT_SIZE)
                movable_subtyped_objs[words]++;
              else
                movable_subtyped_objs[MAX_STAT_SIZE+1]++;
#endif
              *alloc++ = head;
              *start = ___TAG((alloc - ___BODY_OFS), ___TYP(obj));
              body[-1] = ___TAG((alloc - ___BODY_OFS), ___FORW);
              while (words > 0)
                {
                  *alloc++ = *body++;
                  words--;
                }
#if ___WS == 4
              if (pad)
                *alloc++ = ___MAKE_HD_WORDS(0, ___sVECTOR);
#endif
            }
          else if (head_typ == ___STILL)
            {
              if (body[___STILL_MARK_OFS - ___STILL_BODY_OFS] == -1)
                {
                  body[___STILL_MARK_OFS - ___STILL_BODY_OFS]
                    = (___WORD)still_objs_to_scan;
                  still_objs_to_scan = (___WORD)(body - ___STILL_BODY_OFS);
                }
            }
          else if (___TYP(head_typ) == ___FORW)
            {
              ___WORD *copy_body = ___UNTAG_AS(head, ___FORW) + ___BODY_OFS;
              *start = ___TAG((copy_body - ___BODY_OFS), ___TYP(obj));
            }
        }
      start++;
      n--;
    }
  alloc_ptr = alloc;
}


extern ___WORD ___internal_return;

___HIDDEN void mark_continuation
   ___P((___WORD *orig_ptr, ___WORD ra),(orig_ptr, ra)
___WORD *orig_ptr;
___WORD ra;)
{
  ___processor_state pstate = ___PSTATE;
  ___WORD frame, *ptr = orig_ptr;

next_frame:

  frame = *ptr;

  if (frame != 0 && ___TYP(frame) != ___tSUBTYPED)
    {
      /* continuation's frame is in the stack */

      ___WORD *fp, frame_ra;
      int fs, link, i;

      if (ra == ___internal_return)
        {
          ___WORD fs_link = ((___label_struct*)(((___WORD*)frame)[0]-___tSUBTYPED))->flags;
          fs = (fs_link>>3)&((1<<14)-1);
          link = (fs_link>>17)&((1<<14)-1);
        }
      else
        {
          ___WORD fs_link = ((___label_struct*)(ra-___tSUBTYPED))->flags;
          fs = (fs_link>>3)&((1<<14)-1);
          link = (fs_link>>17)&((1<<14)-1);
        }
      fp = ((___WORD*)frame)+fs;
      frame_ra = fp[-link-1];

      if (___TYP(frame_ra) == ___tFIXNUM)
        *ptr = frame_ra;
      else
        {
          ___WORD *copy_body;
          long words = fs+1;

          while (alloc_ptr + words + 1 > alloc_limit)
            next_alloc_msection ();
          *alloc_ptr++ = ___MAKE_HD_WORDS(words, ___sFRAME);
          copy_body = alloc_ptr;
          alloc_ptr += words;

          /* 
           * The frame is being copied from the stack to the heap.
           * The space for this newly allocated object is counted
           * in "words_allocated_for_cont".
           */

          words_allocated_for_cont += words + 1;

          if (frame_ra == pstate->handler_break)
            {
              /* first frame of that section */

              alloc_ptr[-fs-1] = fp[1];
              ra               = fp[0];
              for (i=-fs; i<0; i++)
                alloc_ptr[i] = fp[i];

              alloc_ptr[-link-1] = ra;
            }
          else
            {
              /* not the first frame of that section */

              alloc_ptr[-fs-1] = (___WORD)fp;
              ra               = frame_ra;

              for (i=-fs; i<0; i++)
                alloc_ptr[i] = fp[i];
            }

          /* leave a forwarding pointer */
          fp[-link-1] = ___TAG((copy_body - ___BODY_OFS), ___tFIXNUM);

          *ptr = ___TAG((copy_body - ___BODY_OFS), ___tFIXNUM);

          ptr = &alloc_ptr[-fs-1];

          goto next_frame;
        }
    }

  if (*orig_ptr != 0 && ___TYP((*orig_ptr)) == ___tFIXNUM)
    *orig_ptr = ___TAG(___UNTAG_AS(*orig_ptr, ___tFIXNUM), ___tSUBTYPED);
  else
    mark_array (orig_ptr, 1);
}


___HIDDEN void move_stack ___PVOID
{
  ___processor_state pstate = ___PSTATE;
  ___WORD *fp = pstate->fp;
  ___WORD *p1 = pstate->stack_break+2;
  ___WORD *p2 = pstate->stack_base;

  if (p1 != p2)
    {
      pstate->stack_break = p2-2;
      while (p1 != fp)
        *--p2 = *--p1;
      pstate->fp = p2;
    }
}


___HIDDEN int traverse_wills; /* indicates if body of will must be inspected */
___HIDDEN ___WORD reached_floating_wills; /* wills reached by GC */


___HIDDEN unsigned long scan ___P((___WORD *body),(body)
___WORD *body;)
{
  ___WORD head = body[-1];
  unsigned long words = ___HD_WORDS(head);
  unsigned int subtype = ___HD_SUBTYPE(head);

#ifdef CONSISTENCY_CHECKS
  reference_location = IN_OBJECT;
  container_body = body;
#endif

  switch (subtype)
    {
    case ___sSTRING:
    case ___sBIGNUM:
    case ___sU8VECTOR:
    case ___sU16VECTOR:
    case ___sU32VECTOR:
    case ___sF32VECTOR:
    case ___sF64VECTOR:
    case ___sFLONUM:
    case ___sPOINTER:
      break;

    case ___sWILL:
      if (traverse_wills)
        mark_array (body+1, words-1); /* don't scan link */
      else
        {
          ___WORD link = body[0];
          if (link == ___FLOATING_WILL) /* floating will? */
            {
              /*
               * Maintain a list of all the wills reached by the GC
               * that are not in the executable or non-executable
               * will lists.
               */
              body[0] = reached_floating_wills;
              reached_floating_wills = ___TAG((body-1),___REACH_WILL);
            }
          else
            body[0] = link | ___REACH_WILL;
        }
      break;

    case ___sSYMBOL:
    case ___sKEYWORD:
      mark_array (body, 1); /* only scan name of symbols & keywords */
      break;

    case ___sCONTINUATION:
      mark_continuation (body, body[1]);
      mark_array (body+2, 1); /* only scan dynamic environment */
      break;

    case ___sFRAME:
      if (body[0] != 0 && ___TYP((body[0])) == ___tFIXNUM)
        body[0] = ___TAG(___UNTAG_AS(body[0], ___tFIXNUM), ___tSUBTYPED);
      else
        mark_array (body, 1);
      mark_array (body+1, words-1);
      break;

    case ___sPROCEDURE:
      if (___HD_TYP(head) != ___PERM) /* only scan closures */
        mark_array (body+1, words-1); /* only scan free variables */
      break;

    default:
      mark_array (body, words);
      break;
    }
  return words;
}


___HIDDEN void init_still_objs_to_scan ___PVOID
{
  ___WORD *base = (___WORD*)still_objs;
  ___WORD *to_scan = 0;
  while (base != 0)
    {
      if (base[___STILL_REFCOUNT_OFS] == 0)
        base[___STILL_MARK_OFS] = -1;
      else
        {
          base[___STILL_MARK_OFS] = (___WORD)to_scan;
          to_scan = base;
        }
      base = (___WORD*)base[___STILL_LINK_OFS];
    }
  still_objs_to_scan = (___WORD)to_scan;
}


___HIDDEN void scan_still_objs_to_scan ___PVOID
{
  ___WORD *base;
  while ((base = (___WORD*)still_objs_to_scan) != 0)
    {
      still_objs_to_scan = base[___STILL_MARK_OFS];
      scan (base + ___STILL_BODY_OFS);
    };
}


___HIDDEN void scan_movable_objs_to_scan ___PVOID
{
  while (scan_msection != alloc_msection || scan_ptr < alloc_ptr)
    {
      ___WORD *body = scan_ptr + 1;
      unsigned long words = scan (body);
      scan_ptr = body + words;
      if (scan_msection != alloc_msection &&
          scan_ptr >= scan_msection->alloc)
        {
          scan_msection = scan_msection->next;
          scan_ptr = start_of_tospace (scan_msection);
        }
    };
}


___HIDDEN void free_unmarked_still_objs ___PVOID
{
  ___WORD *last = &still_objs;
  ___WORD *base = (___WORD*)*last;
  while (base != 0)
    {
      ___WORD link = base[___STILL_LINK_OFS];
      if (base[___STILL_MARK_OFS] == -1)
        {
          ___WORD head = base[___STILL_BODY_OFS-1];
          words_non_movable -= base[___STILL_LENGTH_OFS];
          free_mem_aligned (base);
        }
      else
        {
          *last = (___WORD)base;
          last = base + ___STILL_LINK_OFS;
        }
      base = (___WORD*)link;
    }
  *last = 0;
}


___HIDDEN void free_still_objs ___PVOID
{
  ___WORD *base = (___WORD*)still_objs;
  still_objs = 0;
  while (base != 0)
    {
      ___WORD link = base[___STILL_LINK_OFS];
      free_mem_aligned (base);
      base = (___WORD*)link;
    }
}


___HIDDEN int live_percent;


int get_live_percent ___PVOID
{
  return live_percent;
}


void set_live_percent ___P((int percent),(percent)
int percent;)
{
  if (percent <= 0 || percent > 100)
    live_percent = ___DEFAULT_LIVE_PERCENT;
  else
    live_percent = percent;
}


___HIDDEN long (*adjust_hook) ___P((long avail, long live),());


___HIDDEN long default_adjust_hook ___P((long avail, long live),(avail, live)
long avail;
long live;)
{
  long target;
  if (live_percent < 100)
    target = live / live_percent * 100;
  else
    target = live + ___MSECTION_BIGGEST;
  if (target < min_heap_size)
    target = min_heap_size;
  if (max_heap_size > 0 && target > max_heap_size)
    target = max_heap_size;
  return target;
}


___HIDDEN void setup_pstate ___PVOID
{
  ___processor_state pstate = ___PSTATE;
  long unused = WORDS_IN_HEAP - WORDS_OCCUPIED;

  if (unused > 0)
    if (alloc_ptr + unused/2 > alloc_limit - ___MSECTION_FUDGE)
      pstate->heap_limit = alloc_limit - ___MSECTION_FUDGE;
    else
      pstate->heap_limit = alloc_ptr + unused/2;
  else
    pstate->heap_limit = alloc_ptr;

  pstate->hp = alloc_ptr;
}


void ___setup_mem ___P((___setup_params_struct *setup_params),(setup_params)
___setup_params_struct *setup_params;)
{
  int init_nb_sections;
  ___processor_state pstate = ___PSTATE;

  /* initialize global state */

  ___GSTATE->nb_gcs = ___U64_init (0, 0);
  ___GSTATE->gc_user_nsecs = ___U64_init (0, 0);
  ___GSTATE->gc_sys_nsecs = ___U64_init (0, 0);
  ___GSTATE->bytes_allocated_minus_occupied = ___U64_init (0, 0);

  /*
   * It is important to initialize the following pointers first so that
   * if the program terminates early (such as a shortage of free memory
   * during the allocation of the stack) the procedure ___cleanup_mem
   * will not access dangling pointers.
   */

  the_msections = 0;
  psections     = 0;
  still_objs    = 0;
  pstate->stack = 0;

  /* allocate stack */

  stack_cache_size = setup_params->stack_cache >> ___LWS;
  if (stack_cache_size == 0)
    stack_cache_size = ___DEFAULT_STACK_SIZE;
  if (stack_cache_size < ___MAX_NB_FRAME_SLOTS)
    stack_cache_size = ___MAX_NB_FRAME_SLOTS;

  pstate->stack = (___WORD*)
    alloc_mem_aligned (stack_cache_size+___STACK_FUDGE, 1, 0);
  if (pstate->stack == 0)
    ___fatal_error ("Can't allocate initial stack");

  pstate->stack_base = pstate->stack + (stack_cache_size+___STACK_FUDGE);
  pstate->stack_limit = pstate->stack + ___STACK_FUDGE;
  pstate->fp = pstate->stack_base;

  /* allocate heap */

  min_heap_size = setup_params->min_heap >> ___LWS;
  max_heap_size = setup_params->max_heap >> ___LWS;
  normal_overflow_reserve = stack_cache_size*(1+___SUBTYPED_OVERHEAD) +
                            (___STACK_FUDGE+___SUBTYPED_OVERHEAD);
  overflow_reserve = normal_overflow_reserve;

  set_live_percent (setup_params->live_percent);

  adjust_hook = setup_params->gc_hook;
  if (adjust_hook == 0)
    adjust_hook = default_adjust_hook;

  init_nb_sections = (min_heap_size +
                      overflow_reserve*2 +
                      2*((___MSECTION_SIZE>>1)-___MSECTION_FUDGE+1) - 1) /
                     (2*((___MSECTION_SIZE>>1)-___MSECTION_FUDGE+1));

  if (init_nb_sections < ___MIN_NB_MSECTIONS)
    init_nb_sections = ___MIN_NB_MSECTIONS;

  adjust_msections (&the_msections, init_nb_sections);
  if (the_msections == 0 || the_msections->nb_sections != init_nb_sections)
    ___fatal_error ("Can't allocate initial heap");

  tospace_at_top = 0;

  words_non_movable = 0;
  words_prev_msections = 0;

  alloc_msection = 0;
  next_alloc_msection ();

  palloc_ptr = 0;

  setup_pstate ();
}


void ___cleanup_mem ___PVOID
{
  ___processor_state pstate = ___PSTATE;
  free_msections (&the_msections);
  free_psections ();
  free_still_objs ();
  if (pstate->stack != 0)
    {
      free_mem_aligned (pstate->stack);
      pstate->stack = 0;
    }
}


___HIDDEN void determine_will_executability ___P((___WORD list),(list)
___WORD list;)
{
  while (___UNTAG(list) != 0)
    {
      ___WORD* will_body = ___BODY(list);
      ___WORD will_head = will_body[-1];
      ___WORD testator;
      ___WORD head;
      ___WORD *body;
      unsigned int head_typ;

      if (___TYP(will_head) == ___FORW) /* was will forwarded? */
        will_body = ___BODY_AS(will_head,___FORW);

      list = will_body[0];

      testator = will_body[1];

      if (___MEM_ALLOCATED(testator) &&
          ((head_typ = ___HD_TYP((head=(body=___BODY(testator))[-1])))
           == ___MOVABLE0 ||
           (head_typ == ___STILL &&
            body[___STILL_MARK_OFS - ___STILL_BODY_OFS] == -1)))
        {
          /* 
           * All paths to testator object from roots pass through
           * wills, so mark will as executable.
           */

          will_body[0] = list | ___EXEC_WILL;
        }
    }
}


___HIDDEN void process_wills ___PVOID
{
  ___processor_state pstate = ___PSTATE;
  ___WORD* tail_exec;
  ___WORD* tail_non_exec;
  ___WORD curr;

  determine_will_executability (pstate->non_executable_wills);
  determine_will_executability (reached_floating_wills);

  /*
   * Move executable wills to executable will list and also
   * mark all wills in case they were not reached.
   */

  tail_exec = &pstate->executable_wills;
  curr = *tail_exec;

  while (___UNTAG(curr) != 0)
    {
      ___WORD will = ___TAG(___UNTAG(curr),___tSUBTYPED);

      mark_array (&will, 1);

      *tail_exec = ___TAG(___UNTAG(will),___EXEC_WILL);
      tail_exec = &___BODY_AS(will,___tSUBTYPED)[0];
      curr = *tail_exec;
      if (curr & ___REACH_WILL) /* was will reached? */
        mark_array (tail_exec+1, ___WILL2_SIZE-1);
    }

  tail_non_exec = &pstate->non_executable_wills;
  curr = *tail_non_exec;

  while (___UNTAG(curr) != 0)
    {
      ___WORD will = ___TAG(___UNTAG(curr),___tSUBTYPED);

      mark_array (&will, 1);

      if (___BODY_AS(will,___tSUBTYPED)[0] & ___EXEC_WILL)
        {
          /* move will to executable will list */

          *tail_exec = ___TAG(___UNTAG(will),___EXEC_WILL);
          tail_exec = &___BODY_AS(will,___tSUBTYPED)[0];
          tail_exec[1] = ___FAL; /* zap testator */
          curr = *tail_exec;
          if (curr & ___REACH_WILL) /* was will reached? */
            mark_array (tail_exec+1, ___WILL2_SIZE-1);
        }
      else
        {
          /* leave will in non-executable will list */

          *tail_non_exec = ___TAG(___UNTAG(will),0);
          tail_non_exec = &___BODY_AS(will,___tSUBTYPED)[0];
          curr = *tail_non_exec;
          if (curr & ___REACH_WILL) /* was will reached? */
            mark_array (tail_non_exec+1, ___WILL2_SIZE-1);
        }
    }

  *tail_exec = ___TAG(0,___EXEC_WILL);
  *tail_non_exec = ___TAG(0,0);

  curr = reached_floating_wills;

  while (___UNTAG(curr) != 0)
    {
      ___WORD* will_body = ___BODY(curr);

      curr = will_body[0];

      if (will_body[0] & ___EXEC_WILL)
        will_body[1] = ___FAL; /* zap testator */

      mark_array (will_body+1, ___HD_WORDS(will_body[-1])-1);

      will_body[0] = ___FLOATING_WILL;
    }
}


int ___gc ___PVOID
{
  long unused;
  int target_nb_sections;
  int overflow = 0;
  ___processor_state pstate = ___PSTATE;
  ___U64 user_nsecs_start, sys_nsecs_start;
  ___U64 user_nsecs_end, sys_nsecs_end;
  ___U64 gc_user_nsecs, gc_sys_nsecs;

  ___cpu_time (&user_nsecs_start, &sys_nsecs_start);

  if ((___G__23__23_gc_2d_report.val != ___FAL) || (___debug_level == 1))
    {
      fprintf (stderr, "*** GC");
      fflush (stderr);
    }

  alloc_ptr = pstate->hp; /* needed by 'WORDS_OCCUPIED' */

  ___GSTATE->bytes_allocated_minus_occupied =
    ___U64_add_U32 (___GSTATE->bytes_allocated_minus_occupied,
                    WORDS_OCCUPIED << ___LWS);

#ifdef GATHER_STATS
  movable_pair_objs = 0;
  {
    int i;
    for (i=0; i<=MAX_STAT_SIZE+1; i++)
      movable_subtyped_objs[i] = 0;
  }
#endif

  tospace_at_top = !tospace_at_top;

  alloc_msection = 0;
  next_alloc_msection ();
  scan_msection = alloc_msection;
  scan_ptr = alloc_ptr;
  words_allocated_for_cont = 0;

  /* maintain lists of wills reached by GC */

  reached_floating_wills = ___TAG(0,___REACH_WILL);

  /* trace externally referenced still objects */

  init_still_objs_to_scan ();

  /* trace registers */

#ifdef CONSISTENCY_CHECKS
    reference_location = IN_REGISTER;
#endif

  mark_array (pstate->r, ___NB_GVM_REGS);

  /* trace global variables */

#ifdef CONSISTENCY_CHECKS
    reference_location = IN_GLOBAL_VAR;
#endif

  {
    ___WORD p = pstate->glo_list_head;

    while (p != 0)
    {
      mark_array (&((___glo_struct*)p)->val, 1);
      p = ((___glo_struct*)p)->next;
    }
  }

  /* trace stack */

#ifdef CONSISTENCY_CHECKS
    reference_location = IN_STACK;
#endif

  mark_continuation (&pstate->stack_break[1], pstate->stack_break[0]);

  mark_array (pstate->fp, pstate->stack_break-pstate->fp);

  /* mark objects reachable from marked objects */

  traverse_wills = 0; /* don't traverse wills during first pass */

again:

  if ((___WORD*)still_objs_to_scan != 0)
    scan_still_objs_to_scan ();

  if (scan_msection != alloc_msection || scan_ptr < alloc_ptr)
    {
      scan_movable_objs_to_scan ();
      goto again;
    }

  if (!traverse_wills)
    {
      /*
       * At this point all of the objects accessible from the roots
       * without having to traverse a will object have been scanned
       * by the GC.
       */

      traverse_wills = 1;

      process_wills ();

      goto again;
    }

  move_stack ();

#ifdef CONSISTENCY_CHECKS
  {
    int i, j;
    ___WORD *p;
    p = pstate->fp;
    while (p > pstate->stack)
      *--p = -1;
    for (i=0; i<the_msections->nb_sections; i++)
      {
        p = start_of_fromspace (the_msections->sections[i]);
        for (j=(___MSECTION_SIZE>>1)-1; j>=0; j--)
          *p++ = -1;
      }
  }
#endif

  free_unmarked_still_objs ();

  target_nb_sections = (adjust_hook (WORDS_IN_HEAP, WORDS_OCCUPIED) +
                        overflow_reserve*2 +
                        2*((___MSECTION_SIZE>>1)-___MSECTION_FUDGE+1) - 1) /
                       (2*((___MSECTION_SIZE>>1)-___MSECTION_FUDGE+1));

  if (target_nb_sections < nb_prev_msections+1)
    target_nb_sections = nb_prev_msections+1;

  if (target_nb_sections < ___MIN_NB_MSECTIONS)
    target_nb_sections = ___MIN_NB_MSECTIONS;

  adjust_msections (&the_msections, target_nb_sections);

  if (alloc_ptr > alloc_limit - ___MSECTION_FUDGE)
    next_alloc_msection ();

  unused = WORDS_IN_HEAP - WORDS_OCCUPIED;

  if (overflow_reserve == 0)
    {
      if (unused/2 >= normal_overflow_reserve)
        overflow_reserve = normal_overflow_reserve;
    }
  else if (unused <= 0)
    {
      overflow_reserve = 0;
      overflow = 1;
    }

  setup_pstate ();

  ___cpu_time (&user_nsecs_end, &sys_nsecs_end);
  gc_user_nsecs = ___U64_sub_U64 (user_nsecs_end, user_nsecs_start);
  gc_sys_nsecs = ___U64_sub_U64 (sys_nsecs_end, sys_nsecs_start);
  ___GSTATE->nb_gcs = ___U64_add_U32 (___GSTATE->nb_gcs, 1);
  ___GSTATE->gc_user_nsecs =
    ___U64_add_U64 (___GSTATE->gc_user_nsecs, gc_user_nsecs);
  ___GSTATE->gc_sys_nsecs =
    ___U64_add_U64 (___GSTATE->gc_sys_nsecs, gc_sys_nsecs);
  ___GSTATE->bytes_allocated_minus_occupied =
    ___U64_sub_U32 (___GSTATE->bytes_allocated_minus_occupied,
                    (WORDS_OCCUPIED-words_allocated_for_cont) << ___LWS);

  if ((___G__23__23_gc_2d_report.val != ___FAL) || (___debug_level >= 1))
    {
      ___U64 total_alloc_bytes;
      ___U64 total_gc_nsecs;
      ___U32 total_gc_msecs;

      total_alloc_bytes =
        ___U64_add_U32 (___GSTATE->bytes_allocated_minus_occupied,
                        WORDS_OCCUPIED << ___LWS);
      ___U64_shift_right (&total_alloc_bytes, 10); /* convert to Kbytes */

      total_gc_nsecs = ___U64_add_U64 (gc_user_nsecs, gc_sys_nsecs);
      ___U64_shift_right (&total_gc_nsecs, 6);
      total_gc_msecs = ___U64_to_U32 (total_gc_nsecs) / (1000000 >> 6);

      if ((___G__23__23_gc_2d_report.val != ___FAL) || (___debug_level == 1))
        fprintf (stderr,
                 ": %u ms, %uK alloc, %ldK heap, %ldK live (%ld%% %ld+%ld)\n",
                 total_gc_msecs, /* assumes GC took < 274.877 seconds */
                 ___U64_to_U32 (total_alloc_bytes), /* assumes < 2^32K */
                 WORDS_IN_HEAP>>(10-___LWS),
                 WORDS_OCCUPIED>>(10-___LWS),
                 100*WORDS_OCCUPIED/WORDS_IN_HEAP,
                 WORDS_MOVABLE<<___LWS,
                 words_non_movable<<___LWS);

      if (___debug_level == 2)
        fprintf (stderr,
                 "%u %ld\n",
                 ___U64_to_U32 (total_alloc_bytes), /* assumes < 2^32K */
                 WORDS_OCCUPIED>>(10-___LWS));
      else if (___debug_level == 3)
        {
          ___U64 non_gc_user_nsecs;
          ___U64 non_gc_sys_nsecs;
          ___U64 non_gc_nsecs;
          ___U32 non_gc_msecs;
          non_gc_user_nsecs =
            ___U64_sub_U64 (user_nsecs_end, ___GSTATE->gc_user_nsecs);
          non_gc_sys_nsecs =
            ___U64_sub_U64 (sys_nsecs_end, ___GSTATE->gc_sys_nsecs);
          non_gc_nsecs =
            ___U64_add_U64 (non_gc_user_nsecs, non_gc_sys_nsecs);
          ___U64_shift_right (&non_gc_nsecs, 6);
          non_gc_msecs = ___U64_to_U32 (non_gc_nsecs) / (1000000 >> 6);
          fprintf (stderr,
                   "%ld %u\n",
                   non_gc_msecs,
                   ___U64_to_U32 (total_alloc_bytes)); /* assumes < 2^32K */
        }
      else if (___debug_level == 4)
        {
#ifdef GATHER_STATS
          fprintf (stderr,
                   "  movable pairs                              = %ld\n",
                   movable_pair_objs);
          {
            int i;
            for (i=0; i<=MAX_STAT_SIZE; i++)
              fprintf (stderr,
                       "  movable subtyped objects (size =%3d words) = %ld\n",
                       i,
                       movable_subtyped_objs[i]);
            fprintf (stderr,
                     "  movable subtyped objects (size >%3d words) = %ld\n",
                     MAX_STAT_SIZE,
                     movable_subtyped_objs[MAX_STAT_SIZE+1]);
          }
#endif
        }
      fflush (stderr);
    }

  ___raise_interrupt (___INTR_GC); /* raise gc interrupt */

  return overflow;
}


int ___heap_limit ___PVOID
{
  ___processor_state pstate = ___PSTATE;
  long unused;

  alloc_ptr = pstate->hp; /* needed by 'WORDS_OCCUPIED' */

  unused = WORDS_IN_HEAP - WORDS_OCCUPIED;

  if (unused > 0)
    {
      next_alloc_msection ();
      if (alloc_ptr + unused/2 > alloc_limit - ___MSECTION_FUDGE)
        pstate->heap_limit = alloc_limit - ___MSECTION_FUDGE;
      else
        pstate->heap_limit = alloc_ptr + unused/2;
      pstate->hp = alloc_ptr;
      return 0;
    }

  return ___gc ();
}


/*---------------------------------------------------------------------------*/

___U64 ___bytes_allocated ___PVOID
{
  ___processor_state pstate = ___PSTATE;

  alloc_ptr = pstate->hp; /* needed by 'WORDS_OCCUPIED' */

  return ___U64_add_U32 (___GSTATE->bytes_allocated_minus_occupied,
                         WORDS_OCCUPIED << ___LWS);
}


/*---------------------------------------------------------------------------*/
