;==============================================================================

; file: "_ptree2.scm"

; Copyright (C) 1994-1998 by Marc Feeley, All Rights Reserved.

(include "fixnum.scm")

;------------------------------------------------------------------------------
;
; Parse tree manipulation module: (part 2)
; ------------------------------

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (normalize-program lst1)
  (let ((lst2 (beta-reduce (assignment-convert lst1))))
    (for-each lambda-lift! lst2)
    lst2))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; (delete-ptree ptree) removes parse tree 'ptree' from program and updates
; references and assignments to variables.

(define (delete-ptree ptree)

  (cond ((ref? ptree)
         (let ((var (ref-var ptree)))
           (var-refs-set! var (ptset-remove (var-refs var) ptree))))

        ((set? ptree)
         (let ((var (set-var ptree)))
           (var-sets-set! var (ptset-remove (var-sets var) ptree))))

        ((def? ptree)
         (let ((var (def-var ptree)))
           (var-sets-set! var (ptset-remove (var-sets var) ptree)))))

  (for-each delete-ptree (node-children ptree)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;
; Assignment conversion:
; ---------------------

; (assignment-convert lst) takes a list of parse-trees and returns a
; list where each parse-tree has been replaced with an equivalent
; parse-tree containing no assignments to non-global variables.  In
; the converted parse-tree, 'cells' are used to implement mutable
; variables and calls to the procedures:
;
;   ##make-cell
;   ##cell-ref
;   ##cell-set!
;
; are added to create and access the cells.

(define (assignment-convert lst)
  (map (lambda (ptree) (ac ptree '()))
       lst))

(define (ac ptree mut)

  (cond ((cst? ptree)
         ptree)

        ((ref? ptree)
         (let ((var (ref-var ptree)))
           (if (global? var)
             ptree
             (let ((x (assq var mut)))
               (if x
                 (let ((source (node-source ptree)))
                   (var-refs-set! var (ptset-remove (var-refs var) ptree))
                   (new-call source (node-env ptree)
                     (new-ref-extended-bindings
                      source
                      **cell-ref-sym
                      (node-env ptree))
                     (list (new-ref source (node-env ptree) (cdr x)))))
                 ptree)))))

        ((set? ptree)
         (let ((var (set-var ptree))
               (source (node-source ptree))
               (val (ac (set-val ptree) mut)))
           (if (global? var)
             (begin
               (var-sets-set! var (ptset-remove (var-sets var) ptree))
               (new-set source (node-env ptree)
                 var
                 val))
             (new-call source (node-env ptree)
               (new-ref-extended-bindings
                source
                **cell-set!-sym
                (node-env ptree))
               (list (new-ref source (node-env ptree) (cdr (assq var mut)))
                     val)))))

        ((def? ptree) ; guaranteed to be a toplevel definition
         (let ((var (def-var ptree))
               (val (ac (def-val ptree) mut)))
           (var-sets-set! var (ptset-remove (var-sets var) ptree))
           (new-def (node-source ptree) (node-env ptree)
             var
             val)))

        ((tst? ptree)
         (new-tst (node-source ptree) (node-env ptree)
           (ac (tst-pre ptree) mut)
           (ac (tst-con ptree) mut)
           (ac (tst-alt ptree) mut)))

        ((conj? ptree)
         (new-conj (node-source ptree) (node-env ptree)
           (ac (conj-pre ptree) mut)
           (ac (conj-alt ptree) mut)))

        ((disj? ptree)
         (new-disj (node-source ptree) (node-env ptree)
           (ac (disj-pre ptree) mut)
           (ac (disj-alt ptree) mut)))

        ((prc? ptree)
         (ac-proc ptree mut))

        ((app? ptree)
         (let ((oper (app-oper ptree))
               (args (app-args ptree)))
           (if (and (prc? oper) ; applying a lambda-expr is like a 'let'
                    (prc-req-and-opt-parms-only? oper)
                    (= (length (prc-parms oper)) (length args)))
             (ac-let ptree mut)
             (new-call (node-source ptree) (node-env ptree)
               (ac oper mut)
               (map (lambda (x) (ac x mut)) args)))))

        ((fut? ptree)
         (new-fut (node-source ptree) (node-env ptree)
           (ac (fut-val ptree) mut)))

        (else
         (compiler-internal-error "ac, unknown parse tree node type"))))

(define (ac-proc ptree mut)
  (let* ((mut-parms (ac-mutables (prc-parms ptree)))
         (cloned-mut-parms (clone-vars mut-parms)))

    (for-each (lambda (var) (var-sets-set! var (ptset-empty)))
              mut-parms)

    (for-each (lambda (var) (var-cell?-set! var #t))
              cloned-mut-parms)

    (new-prc (node-source ptree) (node-env ptree)
      (prc-name ptree)
      (prc-c-name ptree)
      (prc-parms ptree)
      (prc-opts ptree)
      (prc-keys ptree)
      (prc-rest? ptree)
      (new-let ptree
               ptree
               cloned-mut-parms
               (map (lambda (var)
                      (new-call (var-source var) (node-env ptree)
                        (new-ref-extended-bindings
                         (var-source var)
                         **make-cell-sym
                         (node-env ptree))
                        (list (new-ref (var-source var)
                                       (node-env ptree)
                                       var))))
                    mut-parms)
               (ac (prc-body ptree)
                   (append (pair-up mut-parms cloned-mut-parms) mut))))))

(define (ac-let ptree mut)
  (let* ((proc (app-oper ptree))
         (vals (app-args ptree))
         (vars (prc-parms proc))
         (vals-fv (varset-union-multi (map free-variables vals)))
         (mut-parms (ac-mutables vars))
         (cloned-mut-parms (clone-vars mut-parms))
         (mut (append (pair-up mut-parms cloned-mut-parms) mut)))

    (for-each (lambda (var) (var-sets-set! var (ptset-empty)))
              mut-parms)

    (for-each (lambda (var) (var-cell?-set! var #t))
              cloned-mut-parms)

    (let loop ((l1 vars)
               (l2 vals)
               (new-vars '())
               (new-vals '())
               (new-body (ac (prc-body proc) mut)))
      (if (null? l1)

        (new-let ptree proc new-vars new-vals new-body)

        (let ((var (car l1))
              (val (car l2)))

          (if (memq var mut-parms)

            (let ((src (node-source val))
                  (env (node-env val))
                  (var* (cdr (assq var mut))))

              (if (varset-member? var vals-fv)

                (loop (cdr l1)
                      (cdr l2)
                      (cons var* new-vars)
                      (cons (new-call src env
                              (new-ref-extended-bindings
                               src
                               **make-cell-sym
                               env)
                              (list (new-cst src env void-object)))
                            new-vals)
                      (new-seq src env
                        (new-call src env
                          (new-ref-extended-bindings
                           src
                           **cell-set!-sym
                           env)
                          (list (new-ref src env var*)
                                (ac val mut)))
                        new-body))

                (loop (cdr l1)
                      (cdr l2)
                      (cons var* new-vars)
                      (cons (new-call src env
                              (new-ref-extended-bindings
                               src
                               **make-cell-sym
                               env)
                              (list (ac val mut)))
                            new-vals)
                      new-body)))

            (loop (cdr l1)
                  (cdr l2)
                  (cons var new-vars)
                  (cons (ac val mut) new-vals)
                  new-body)))))))

(define (ac-mutables lst)
  (keep mutable? lst))

(define (clone-vars vars)
  (map (lambda (var)
         (let ((cloned-var
                (make-var (var-name var)
                          #t
                          (ptset-empty)
                          (ptset-empty)
                          (var-source var))))
           (var-cell?-set! cloned-var (var-cell? var))
           cloned-var))
       vars))


; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; (clone-ptree ptree) returns a fresh copy of 'ptree'.  All the
; bound variables (i.e. not free variables) in ptree are also copied.
; It is assumed that 'ptree' has already been assignment-converted.

(define (clone-ptree ptree)
  (cp ptree '()))

(define (cp ptree substs)

  (define (rename-var var)
    (let ((x (assq var substs)))
      (if x (cdr x) var)))

  (cond ((cst? ptree)
         (new-cst (node-source ptree) (node-env ptree)
           (cst-val ptree)))

        ((ref? ptree)
         (let ((var (rename-var (ref-var ptree))))
           (new-ref (node-source ptree) (node-env ptree)
             var)))

        ((set? ptree)
         (let ((var (rename-var (set-var ptree))))
           (new-set (node-source ptree) (node-env ptree)
             var
             (cp (set-val ptree) substs))))

        ((def? ptree) ; guaranteed to be a toplevel definition
         (new-def (node-source ptree) (node-env ptree)
           (def-var ptree)
           (cp (def-val ptree) substs)))

        ((tst? ptree)
         (new-tst (node-source ptree) (node-env ptree)
           (cp (tst-pre ptree) substs)
           (cp (tst-con ptree) substs)
           (cp (tst-alt ptree) substs)))

        ((conj? ptree)
         (new-conj (node-source ptree) (node-env ptree)
           (cp (conj-pre ptree) substs)
           (cp (conj-alt ptree) substs)))

        ((disj? ptree)
         (new-disj (node-source ptree) (node-env ptree)
           (cp (disj-pre ptree) substs)
           (cp (disj-alt ptree) substs)))

        ((prc? ptree)
         (let* ((parms (prc-parms ptree))
                (vars (clone-vars parms)))
           (new-prc (node-source ptree) (node-env ptree)
             (prc-name ptree)
             (prc-c-name ptree)
             vars
             (prc-opts ptree)
             (prc-keys ptree)
             (prc-rest? ptree)
             (cp (prc-body ptree)
                 (append (pair-up parms vars) substs)))))

        ((app? ptree)
         (let ((oper (app-oper ptree))
               (args (app-args ptree)))
           (if (and (prc? oper) ; applying a lambda-expr is like a 'let'
                    (prc-req-and-opt-parms-only? oper)
                    (= (length (prc-parms oper)) (length args)))
             (let* ((parms (prc-parms oper))
                    (vars (clone-vars parms))
                    (new-substs (append (pair-up parms vars) substs)))
               (new-let ptree
                        oper
                        vars
                        (map (lambda (x) (cp x new-substs)) args)
                        (cp (prc-body oper) new-substs)))
             (new-call (node-source ptree) (node-env ptree)
               (cp (app-oper ptree) substs)
               (map (lambda (x) (cp x substs)) (app-args ptree))))))

        ((fut? ptree)
         (new-fut (node-source ptree) (node-env ptree)
           (cp (fut-val ptree) substs)))

        (else
         (compiler-internal-error "cp, unknown parse tree node type"))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;
; Beta-reduction:
; --------------

; (beta-reduce lst) takes a list of parse-trees and returns a list of
; the parse-trees transformed with the following transformations:
;
;  - constant propagation
;  - copy propagation
;  - useless variable elimination
;
; It is assumed that the parse-trees have already been assignment-converted.

(define (beta-reduce lst)

  (define (process lst)
    (if (null? lst)
      '()
      (let ((ptree (car lst)))
        (if (def? ptree)
         (let ((var (def-var ptree))
               (val (br (def-val ptree) '() #f)))
           (let ((rest (process (cdr lst))))
             (var-sets-set! var (ptset-remove (var-sets var) ptree))
             (cons (new-def (node-source ptree) (node-env ptree)
                     var
                     val)
                   rest)))
         (let ((val (br ptree '() #f)))
           (let ((rest (process (cdr lst))))
             (cons val rest)))))))

  (process lst))

(define (br ptree substs expansion-limit)

  (cond ((cst? ptree)
         (new-cst (node-source ptree) (node-env ptree)
           (cst-val ptree)))

        ((ref? ptree)
         (let ((var (ref-var ptree)))
           (var-refs-set! var (ptset-remove (var-refs var) ptree))
           (let ((new-var (var-subst var substs)))
             (let ((x (var-to-val new-var substs)))
               (if (and x (or (cst? x) (ref? x)))
                 (clone-ptree x)
                 (new-ref (node-source ptree) (node-env ptree)
                   new-var))))))

        ((set? ptree) ; variable guaranteed to be a global variable
         (let ((var (set-var ptree))
               (val (br (set-val ptree) substs expansion-limit)))
           (var-sets-set! var (ptset-remove (var-sets var) ptree))
           (new-set (node-source ptree) (node-env ptree)
             var
             val)))

        ((tst? ptree)
         (let ((pre (br (tst-pre ptree) substs expansion-limit)))
           (if (cst? pre)
             (if (false-object? (cst-val pre))
               (begin
                 (delete-ptree (tst-con ptree))
                 (br (tst-alt ptree) substs expansion-limit))
               (begin
                 (delete-ptree (tst-alt ptree))
                 (br (tst-con ptree) substs expansion-limit)))
             (new-tst (node-source ptree) (node-env ptree)
               pre
               (br (tst-con ptree) substs expansion-limit)
               (br (tst-alt ptree) substs expansion-limit)))))

        ((conj? ptree)
         (let ((pre (br (conj-pre ptree) substs expansion-limit)))
           (if (cst? pre)
             (if (false-object? (cst-val pre))
               (begin
                 (delete-ptree (conj-alt ptree))
                 pre)
               (br (conj-alt ptree) substs expansion-limit))
             (new-conj (node-source ptree) (node-env ptree)
               pre
               (br (conj-alt ptree) substs expansion-limit)))))

        ((disj? ptree)
         (let ((pre (br (disj-pre ptree) substs expansion-limit)))
           (if (cst? pre)
             (if (false-object? (cst-val pre))
               (br (disj-alt ptree) substs expansion-limit)
               (begin
                 (delete-ptree (disj-alt ptree))
                 pre))
             (new-disj (node-source ptree) (node-env ptree)
               pre
               (br (disj-alt ptree) substs expansion-limit)))))

        ((prc? ptree)
         (new-prc (node-source ptree) (node-env ptree)
           (prc-name ptree)
           (prc-c-name ptree)
           (prc-parms ptree)
           (prc-opts ptree)
           (prc-keys ptree)
           (prc-rest? ptree)
           (br (prc-body ptree) substs expansion-limit)))

        ((app? ptree)
         (let ((oper (app-oper ptree))
               (args (app-args ptree)))
           (if (ref? oper)
             (let ((br-oper (br oper substs expansion-limit)))
               ; at this point (or (cst? br-oper) (ref? br-oper))
               (or (br-inline-user ptree br-oper args substs expansion-limit)
                   (br-app ptree br-oper args substs expansion-limit)))
             (br-app ptree oper args substs expansion-limit))))

        ((fut? ptree)
         (new-fut (node-source ptree) (node-env ptree)
           (br (fut-val ptree) substs expansion-limit)))

        (else
         (compiler-internal-error "br, unknown parse tree node type"))))

(define (var-subst var substs)
  (if (null? substs)
    var
    (let ((couple (car substs)))
      (if (eq? (car couple) var)
        (if (ref? (cdr couple))
          (var-subst (ref-var (cdr couple)) (cdr substs))
          var)
        (var-subst var (cdr substs))))))

(define (var-to-val var substs)
  (if (global? var)
    (global-single-def var)
    (let ((x (assq var substs)))
      (if x (cdr x) #f))))

(define (br-app ptree oper args substs expansion-limit)
  (if (and (prc? oper) ; applying a lambda-expr is like a 'let'
           (prc-req-and-opt-parms-only? oper)
           (= (length (prc-parms oper)) (length args)))
    (br-let ptree oper args substs expansion-limit)
    (new-call (node-source ptree) (node-env ptree)
      (br oper substs expansion-limit)
      (map (lambda (arg) (br arg substs expansion-limit)) args))))

(define (br-let ptree proc vals substs expansion-limit)
  (let* ((vars
          (prc-parms proc))
         (new-substs
          (br-extend-substs vars vals substs))
         (br-vals
          (map (lambda (x) (br x new-substs expansion-limit)) vals))
         (new-body
          (br (prc-body proc) new-substs expansion-limit)))

    ; remove useless bindings

    (let loop ((l1 vars)
               (l2 br-vals)
               (new-vars '())
               (new-vals '()))
      (if (null? l1)
        (new-let ptree
                 proc
                 (reverse new-vars)
                 (reverse new-vals)
                 new-body)
        (let ((var (car l1))
              (br-val (car l2)))
          (if (and (ptset-empty? (var-refs var))
                   (or (cst? br-val)
                       (ref? br-val)
                       (prc? br-val)))
            (begin
              (delete-ptree br-val)
              (loop (cdr l1)
                    (cdr l2)
                    new-vars
                    new-vals))
            (loop (cdr l1)
                  (cdr l2)
                  (cons var new-vars)
                  (cons br-val new-vals))))))))

(define (br-extend-substs vars vals substs)
  (let loop ((l1 vars)
             (l2 vals)
             (new-substs substs))
    (if (null? l1)
      new-substs
      (let ((var (car l1))
            (val (car l2)))
        (cond ((or (cst? val)
                   (and (ref? val)
                        (or (bound? (ref-var val))
                            (global-singly-bound? val)))
                   (and (prc? val)
                        (ptset-every? oper-pos? (var-refs var))))
               (loop (cdr l1)
                     (cdr l2)
                     (cons (cons var val) new-substs)))
              (else
               (loop (cdr l1)
                     (cdr l2)
                     new-substs)))))))

(define (br-inline-user ptree br-oper args substs expansion-limit)
  (and (ref? br-oper)
       (let* ((var (ref-var br-oper))
              (val (var-to-val var substs)))
         (and val
              (prc? val)
              (inline? (node-env val))
              (let* ((size-val
                       (ptree-size val))
                     (size-ptree
                       (ptree-size ptree))
                     (new-limit
                       (- (if expansion-limit
                            (car expansion-limit)
                            (quotient (* (inlining-limit (node-env ptree))
                                         size-ptree)
                                      100))
                          (- size-val 1))))
                (and (>= new-limit 0)
                     (let ((cloned-oper (clone-ptree val)))
                       (delete-ptree br-oper)
                       (br-app ptree cloned-oper args substs
                               (if expansion-limit
                                 (begin
                                   (set-car! expansion-limit new-limit)
                                   expansion-limit)
                                 (list new-limit))))))))))

(define (ptree-size ptree)
  (let loop ((lst (node-children ptree)) (n 1))
    (if (null? lst)
      n
      (loop (cdr lst) (+ n (ptree-size (car lst)))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;
; Lambda-lifting procedure:
; ------------------------

; (lambda-lift! ptree) modifies the parse-tree 'ptree' so that some
; of its procedures (i.e. lambda-expressions) are replaced with
; weaker ones (i.e. lambda-expressions having fewer or no closed variables).
; It is assumed that 'ptree' has already been assignment-converted.
; Presently, only named procedures are lambda-lifted.

(define (lambda-lift! ptree)
  (ll! ptree (varset-empty) '()))

(define (ll! ptree cst-procs env)

  (define (new-env env vars)
    (define (loop i l)
      (if (pair? l)
        (let ((var (car l)))
          (cons (cons var (cons (ptset-size (var-refs var)) i))
                (loop (+ i 1) (cdr l))))
        env))
    (loop (length env) vars))

  (cond ((or (cst? ptree)
             (ref? ptree)
             (set? ptree)
             (def? ptree) ; guaranteed to be a toplevel definition
             (tst? ptree)
             (conj? ptree)
             (disj? ptree)
             (fut? ptree))
         (for-each (lambda (child) (ll! child cst-procs env))
                   (node-children ptree)))

        ((prc? ptree)
         (ll! (prc-body ptree) cst-procs (new-env env (prc-parms ptree))))

        ((app? ptree)
         (let ((oper (app-oper ptree))
               (args (app-args ptree)))
           (if (and (prc? oper) ; applying a lambda-expr is like a 'let'
                    (prc-req-and-opt-parms-only? oper)
                    (= (length (prc-parms oper)) (length args)))
             (ll!-let ptree cst-procs (new-env env (prc-parms oper)))
             (for-each (lambda (child) (ll! child cst-procs env))
                       (node-children ptree)))))

        (else
         (compiler-internal-error "ll!, unknown parse tree node type"))))

(define (ll!-let ptree cst-procs env)
  (let* ((proc (app-oper ptree))
         (vals (app-args ptree))
         (vars (prc-parms proc))
         (var-val-map (pair-up vars vals)))

    (define (var->val var) (cdr (assq var var-val-map)))

    (define (liftable-proc-vars vars)
      (let loop ((cst-proc-vars-list
                   (keep (lambda (var)
                           (let ((val (var->val var)))
                             (and (prc? val)
                                  (lambda-lift? (node-env val))
                                  (ptset-every? oper-pos? (var-refs var)))))
                         vars)))
        (let* ((cst-proc-vars
                 (list->varset cst-proc-vars-list))
               (non-cst-proc-vars-list
                 (keep (lambda (var)
                         (let ((val (var->val var)))
                           (and (prc? val)
                                (not (varset-member? var cst-proc-vars)))))
                       vars))
               (non-cst-proc-vars
                 (list->varset non-cst-proc-vars-list))
               (cst-proc-vars-list*
                 (keep (lambda (var)
                         (not (varset-intersects?
                                (free-variables (var->val var))
                                non-cst-proc-vars)))
                       cst-proc-vars-list)))
          (if (= (length cst-proc-vars-list)
                 (length cst-proc-vars-list*))
            cst-proc-vars-list
            (loop cst-proc-vars-list*)))))

    (define (transitively-closed-free-variables vars)
      (let ((tcfv-map
              (map (lambda (var) (cons var (free-variables (var->val var))))
                   vars)))
        (let loop ()
          (let ((changed? #f))
            (for-each (lambda (var-tcfv)
                        (let ((tcfv (cdr var-tcfv)))
                          (let loop2 ((l (varset->list tcfv))
                                      (fv tcfv))
                            (if (null? l)
                              (if (not (= (varset-size fv) (varset-size tcfv)))
                                (begin
                                  (set-cdr! var-tcfv fv)
                                  (set! changed? #t)))
                              (let ((x (assq (car l) tcfv-map)))
                                (loop2 (cdr l)
                                       (if x
                                         (varset-union fv (cdr x))
                                         fv)))))))
                      tcfv-map)
            (if changed?
              (loop)
              tcfv-map)))))

    (let* ((tcfv-map
             (transitively-closed-free-variables (liftable-proc-vars vars)))
           (cst-proc-vars-list
             (map car tcfv-map))
           (cst-procs*
             (varset-union (list->varset cst-proc-vars-list) cst-procs)))

      (define (var->tcfv var) (cdr (assq var tcfv-map)))

      (define (order-vars vars)
        (map car
             (sort-list (map (lambda (var) (assq var env)) vars)
                        (lambda (x y)
                          (if (= (cadr x) (cadr y))
                            (< (cddr x) (cddr y))
                            (< (cadr x) (cadr y)))))))

      (define (lifted-vars var)
        (order-vars
          (varset->list (varset-difference (var->tcfv var) cst-procs*))))

      (define (lift-app! var)
        (let* ((val (var->val var))
               (vars (lifted-vars var)))
          (if (not (null? vars))
            (for-each (lambda (oper)
                        (let ((node (node-parent oper)))

                          (define (new-ref* var)
                            (new-ref (var-source var) (node-env node) var))

                          (node-children-set! node
                            (cons (app-oper node)
                                  (append (map new-ref* vars)
                                          (app-args node))))))
                      (ptset->list (var-refs var))))))

      (define (lift-prc! var)
        (let* ((val (var->val var))
               (vars (lifted-vars var)))
          (if (not (null? vars))
            (let ((cloned-vars (clone-vars vars)))
              (prc-parms-set! val (append cloned-vars (prc-parms val)))
              (for-each (lambda (x) (var-bound-set! x val)) cloned-vars)
              (node-fv-invalidate! val)
              (ll-rename! val (pair-up vars cloned-vars))))))

      (for-each lift-app! cst-proc-vars-list)
      (for-each lift-prc! cst-proc-vars-list)
      (for-each (lambda (node) (ll! node cst-procs* env)) vals)
      (ll! (prc-body proc) cst-procs* env))))

(define (ll-rename! ptree var-map)

  (if (ref? ptree)
    (let* ((var (ref-var ptree))
           (x (assq var var-map)))
      (if x
        (begin
          (var-refs-set! var (ptset-remove (var-refs var) ptree))
          (var-refs-set! (cdr x) (ptset-adjoin (var-refs (cdr x)) ptree))
          (ref-var-set! ptree (cdr x))))))

  (node-fv-set! ptree #t)

  (for-each (lambda (child) (ll-rename! child var-map))
            (node-children ptree)))

;------------------------------------------------------------------------------
;
; Debugging stuff:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;
; (parse-tree->expression ptree) returns the Scheme expression corresponding to
; the parse tree 'ptree'.

(define (parse-tree->expression ptree)
  (se ptree '() (list 0)))

(define (se ptree env num)

  (cond ((cst? ptree)
         (list quote-sym (cst-val ptree)))

        ((ref? ptree)
         (se-var->id (ref-var ptree) env))

        ((set? ptree)
         (list set!-sym
               (se-var->id (set-var ptree) env)
               (se (set-val ptree) env num)))

        ((def? ptree)
         (list define-sym
               (se-var->id (def-var ptree) env)
               (se (def-val ptree) env num)))

        ((tst? ptree)
         (list if-sym (se (tst-pre ptree) env num)
                      (se (tst-con ptree) env num)
                      (se (tst-alt ptree) env num)))

        ((conj? ptree)
         (list and-sym (se (conj-pre ptree) env num)
                       (se (conj-alt ptree) env num)))

        ((disj? ptree)
         (list or-sym (se (disj-pre ptree) env num)
                      (se (disj-alt ptree) env num)))

        ((prc? ptree)
         (let ((new-env (se-rename (prc-parms ptree) env num)))
           (list lambda-sym
             (se-parameters (prc-parms ptree)
                            (prc-opts ptree)
                            (prc-keys ptree)
                            (prc-rest? ptree)
                            new-env
                            num)
             (se (prc-body ptree) new-env num))))

        ((app? ptree)
         (let ((oper (app-oper ptree))
               (args (app-args ptree)))
           (if (and (prc? oper) ; applying a lambda-expr is like a 'let'
                    (prc-req-and-opt-parms-only? oper)
                    (= (length (prc-parms oper)) (length args)))
             (let ((new-env (se-rename (prc-parms oper) env num)))
               (list
                 (if (varset-intersects?
                       (list->varset (prc-parms oper))
                       (varset-union-multi (map free-variables args)))
                   letrec-sym
                   let-sym)
                 (se-bindings (prc-parms oper) args new-env num)
                 (se (prc-body oper) new-env num)))
             (map (lambda (x) (se x env num)) (cons oper args)))))

        ((fut? ptree)
         (list future-sym (se (fut-val ptree) env num)))

        (else
         (compiler-internal-error "se, unknown parse tree node type"))))

(define (se-var->id var env)
  (let ((x (assq var env)))
    (if x (cdr x) (var-name var))))

(define (se-parameters parms opts keys rest? env num)

  (define (se-required parms n)
    (if (= n 0)
      (se-opts parms)
      (let ((parm (se-var->id (car parms) env)))
        (cons parm (se-required (cdr parms) (- n 1))))))

  (define (se-opts parms)
    (if (null? opts)
      (se-rest parms)
      (cons optional-object
            (let loop ((parms parms) (opts opts))
              (if (null? opts)
                (se-rest parms)
                (let ((parm (se-var->id (car parms) env)))
                  (cons (list parm (list quote-sym (car opts)))
                        (loop (cdr parms) (cdr opts)))))))))

  (define (se-rest parms)
    (if rest?
      (let ((parm (se-var->id (car (last-pair parms)) env)))
        (if #f;*******(and (null? opts) (not keys))
          parm
          (cons rest-object (cons parm (se-keys parms)))))
      (se-keys parms)))

  (define (se-keys parms)
    (if (not keys)
      '()
      (cons key-object
            (let loop ((parms parms) (keys keys))
              (if (null? keys)
                '()
                (let ((parm (se-var->id (car parms) env)))
                  (cons (list parm (list quote-sym (cdr (car keys))))
                        (loop (cdr parms) (cdr keys)))))))))

  (se-required parms
               (- (length parms)
                  (length opts)
                  (if keys (length keys) 0)
                  (if rest? 1 0))))

(define (se-bindings vars vals env num)
  (if (null? vars)
    '()
    (cons (list (se-var->id (car vars) env) (se (car vals) env num))
          (se-bindings (cdr vars) (cdr vals) env num))))

(define (se-rename vars env num)

  (define (rename vars)
    (if (null? vars)
      env
      (cons (cons (car vars)
                  (string->symbol
                    (string-append (symbol->string (var-name (car vars)))
                                   "#"
                                   (number->string (car num)))))
            (rename (cdr vars)))))

  (set-car! num (+ (car num) 1))
  (rename vars))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; C-interface stuff:

(define (c-interface-begin module-name)
  (set! c-interface-module-name module-name)
  (set! c-interface-proc-count 0)
  (set! c-interface-types '())
  (set! c-interface-decls '())
  (set! c-interface-procs '())
  (set! c-interface-inits '())
  #f)

(define (c-interface-end)
  (let ((i (make-c-intf (reverse c-interface-decls)
                        (reverse c-interface-procs)
                        (reverse c-interface-inits))))
    (set! c-interface-module-name #f)
    (set! c-interface-proc-count #f)
    (set! c-interface-types #f)
    (set! c-interface-decls #f)
    (set! c-interface-procs #f)
    (set! c-interface-inits #f)
    i))

(define c-interface-module-name #f)
(define c-interface-proc-count #f)
(define c-interface-types #f)
(define c-interface-decls #f)
(define c-interface-procs #f)
(define c-interface-inits #f)

(define (add-c-type name type-source)
  (set! c-interface-types
    (cons (cons name type-source) c-interface-types))
  #f)

(define (add-c-decl declaration-string)
  (set! c-interface-decls
    (cons declaration-string c-interface-decls))
  #f)

(define (add-c-proc c-proc)
  (set! c-interface-procs
    (cons c-proc c-interface-procs))
  #f)

(define (add-c-init initialization-code-string)
  (set! c-interface-inits
    (cons initialization-code-string c-interface-inits))
  #f)

(define (make-c-intf decls procs inits) (vector decls procs inits))
(define (c-intf-decls c-intf)           (vector-ref c-intf 0))
(define (c-intf-decls-set! c-intf x)    (vector-set! c-intf 0 x))
(define (c-intf-procs c-intf)           (vector-ref c-intf 1))
(define (c-intf-procs-set! c-intf x)    (vector-set! c-intf 1 x))
(define (c-intf-inits c-intf)           (vector-ref c-intf 2))
(define (c-intf-inits-set! c-intf x)    (vector-set! c-intf 2 x))

(define (make-c-proc scheme-name c-name arity body)
  (vector c-proc-tag scheme-name c-name arity body))

(define c-proc-tag (list 'c-proc))

(define (c-proc? x)
  (and (vector? x)
       (> (vector-length x) 0)
       (eq? (vector-ref x 0) c-proc-tag)))

(define (c-proc-scheme-name x) (vector-ref x 1))
(define (c-proc-c-name x)      (vector-ref x 2))
(define (c-proc-arity x)       (vector-ref x 3))
(define (c-proc-body x)        (vector-ref x 4))

(define (c-define-type-expr? source)
  (and (match c-define-type-sym 2 source)
       (proper-c-type-definition? source)))

(define (proper-c-type-definition? source)
  (let* ((code (source-code source))
         (name-source (cadr code))
         (name (source-code name-source))
         (type-source (caddr code)))
    (cond ((not (symbol-object? name))
           (pt-syntax-error
             name-source
             "C type's name must be an identifier"))
          ((or (assq name scheme-to-c-notation)
               (assq name c-interface-types))
           (pt-syntax-error
             name-source
             "C type's name is already defined"))
          (else
           (check-c-type type-source #f #f))))) ; allow all types

(define (c-type-definition-name source)
  (cadr (source-code source)))

(define (c-type-definition-type source)
  (caddr (source-code source)))

(define (c-declare-expr? source)
  (and (match c-declare-sym 1 source)
       (let ((code (source-code source)))
         (or (string? (source-code (cadr code)))
             (pt-syntax-error
               source
               "Argument to 'c-declare' must be a string")))))

(define (c-declaration-body source)
  (cadr (source-code source)))

(define (c-initialize-expr? source)
  (and (match c-initialize-sym 1 source)
       (let ((code (source-code source)))
         (or (string? (source-code (cadr code)))
             (pt-syntax-error
               source
               "Argument to 'c-initialize' must be a string")))))

(define (c-initialization-body source)
  (cadr (source-code source)))

(define (c-lambda-expr? source)
  (and (match c-lambda-sym 3 source)
       (let ((code (source-code source)))
         (if (not (string? (source-code (cadddr code))))
           (pt-syntax-error
             source
             "Third argument to 'c-lambda' must be a string")
           (check-c-function-type (cadr code) (caddr code) #f)))))

(define (c-define-expr? source env)
  (and (match c-define-sym -6 source)
       (proper-c-definition? source env)))

(define (proper-c-definition? source env)
  (let* ((code (source-code source))
         (pattern-source (cadr code))
         (pattern (source-code pattern-source))
         (arg-typs-source (caddr code))
         (res-typ-source (cadddr code))
         (name-source (car (cddddr code)))
         (name (source-code name-source))
         (scope-source (cadr (cddddr code)))
         (scope (source-code scope-source)))
    (cond ((not (pair? pattern))
           (pt-syntax-error
             pattern-source
             "Ill-formed definition pattern"))
          ((not (bindable-var? (car pattern) env))
           (pt-syntax-error
             (car pattern)
             "Procedure name must be an identifier"))
          (else
           (and (check-c-function-type arg-typs-source res-typ-source #f)
                (cond ((not (string? name))
                       (pt-syntax-error
                         name-source
                         "Fourth argument to 'c-define' must be a string"))
                      ((not (valid-c-id? name))
                       (pt-syntax-error
                         name-source
                         "Ill-formed C function identifier"))
                      ((not (string? scope))
                       (pt-syntax-error
                         scope-source
                         "Fifth argument to 'c-define' must be a string"))
                      (else
                       #t)))))))

(define (c-definition-variable source)
  (let ((code (source-code source)))
    (car (source-code (cadr code)))))

(define (c-definition-value source)
  (let ((code (source-code source))
        (loc (source-locat source)))
    (make-source
      (cons (make-source lambda-sym loc)
            (cons (parms->source (cdr (source-code (cadr code))) loc)
                  (cdr (cddddr code))))
      loc)))

(define (c-definition-param-types source)
  (source-code (caddr (source-code source))))

(define (c-definition-result-type source)
  (cadddr (source-code source)))

(define (c-definition-proc-name source)
  (car (cddddr (source-code source))))

(define (c-definition-scope source)
  (cadr (cddddr (source-code source))))

(define (c-type-pt-syntax-error source err-source msg)
  (pt-syntax-error (or err-source source) msg))

(define (check-c-function-type arg-typs-source res-typ-source err-source)
  (and (check-c-arg-types arg-typs-source err-source)
       (check-c-result-type res-typ-source err-source)))

(define (check-c-arg-types arg-typs-source err-source)
  (let ((arg-typs (source-code arg-typs-source)))
    (if (not (proper-length arg-typs))
      (c-type-pt-syntax-error
        arg-typs-source
        err-source
        "Ill-terminated C function argument type list")
      (let loop ((lst arg-typs))
        (if (pair? lst)
          (and (check-c-type (car lst) err-source 'no-void-or-struct)
               (loop (cdr lst)))
          #t)))))

(define (check-c-result-type res-typ-source err-source)
  (check-c-type res-typ-source err-source 'no-struct))

(define (check-c-type typ-source err-source restriction)

  (define (ill-formed-c-type)
    (c-type-pt-syntax-error typ-source err-source "Ill-formed C type"))

  (let ((typ (source-code typ-source)))
    (cond ((pair? typ)
           (let ((length (proper-length (cdr typ))))
             (if length
               (let ((head (source-code (car typ))))
                 (cond ((or (eq? head struct-sym) (eq? head union-sym))
                        (if restriction
                          (c-type-pt-syntax-error
                            typ-source
                            err-source
                            "Ill-placed C STRUCT or UNION type")
                          (or (and (= length 1)
                                   (let ((id (source-code (cadr typ))))
                                     (and (string? id)
                                          (valid-c-id? id))))
                              (c-type-pt-syntax-error
                                typ-source
                                err-source
                                "Ill-formed C STRUCT or UNION type"))))
                       ((eq? head pointer-sym)
                        (if (= length 1)
                          (check-c-type
                            (cadr typ)
                            err-source
                            #f) ; allow all types
                          (c-type-pt-syntax-error
                            typ-source
                            err-source
                            "Ill-formed C POINTER type")))
                       ((eq? head function-sym)
                        (if (= length 2)
                          (check-c-function-type
                            (cadr typ)
                            (caddr typ)
                            err-source)
                          (c-type-pt-syntax-error
                            typ-source
                            err-source
                            "Ill-formed C FUNCTION type")))
                       (else
                        (ill-formed-c-type))))
               (c-type-pt-syntax-error
                 typ-source
                 err-source
                 "Ill-terminated C type"))))
          ((string? typ)
           (if (valid-c-id? typ)
             (or (not restriction)
                 (c-type-pt-syntax-error
                   typ-source
                   err-source
                   "Ill-placed C type identifier"))
             (c-type-pt-syntax-error
               typ-source
               err-source
               "Ill-formed C type identifier")))
          ((symbol-object? typ)
           (if (eq? typ void-sym)
             (or (not (eq? restriction 'no-void-or-struct))
                 (c-type-pt-syntax-error
                   typ-source
                   err-source
                   "Ill-placed C VOID type"))
             (let ((x (assq typ scheme-to-c-notation)))
               (or x
                   (let ((y (assq typ c-interface-types)))
                     (if y
                       (check-c-type (cdr y) typ-source restriction)
                       (c-type-pt-syntax-error
                         typ-source
                         err-source
                         "Undefined C type identifier")))))))
          (else
           (ill-formed-c-type)))))

(define (void-type? typ-source)
  (eq? (source-code typ-source) void-sym))

(define (scmobj-type? typ-source)
  (eq? (source-code typ-source) scheme-object-sym))

(define (dynamically-allocated-type? typ-source)
  (let ((typ (source-code typ-source)))
    (cond ((pair? typ)
           (let ((head (source-code (car typ))))
             (eq? head function-sym)))
          ((symbol-object? typ)
           (let ((x (assq typ scheme-to-c-notation)))
             (if x
               (vector-ref (cdr x) 2)
               (let ((y (assq typ c-interface-types)))
                 (if y
                   (dynamically-allocated-type? (cdr y))
                   #f)))))
          (else
           #f))))

(define (pt-c-lambda source env use)
  (let ((name
         (build-c-lambda
           (c-lambda-param-types source)
           (c-lambda-result-type source)
           (source-code (c-lambda-proc-name source)))))
    (new-ref source
             env
             (env-lookup-global-var env (string->symbol name)))))

(define (c-lambda-param-types source)
  (source-code (cadr (source-code source))))

(define (c-lambda-result-type source)
  (caddr (source-code source)))

(define (c-lambda-proc-name source)
  (cadddr (source-code source)))

(define (number-from-1 lst)
  (let loop ((i 1) (lst1 lst) (lst2 '()))
    (if (pair? lst1)
      (loop (+ i 1) (cdr lst1) (cons (cons (car lst1) i) lst2))
      (reverse lst2))))

(define (c-type-name typ)

  (define (err)
    (compiler-internal-error "c-type-name, unknown C type"))

  (let ((t (source-code typ)))
    (cond ((pair? t)
           (let ((head (source-code (car t))))
             (cond ((eq? head pointer-sym)
                    "POINTER")
                   ((eq? head function-sym)
                    "FUNCTION")
                   (else
                    (err)))))
          ((string? t)
           t)
          ((symbol-object? t)
           (let ((x (assq t scheme-to-c-notation)))
             (if x
               (vector-ref (cdr x) 0)
               (let ((y (assq t c-interface-types)))
                 (if y
                   (c-type-name (cdr y))
                   (err))))))
          (else
           (err)))))

(define nl-str (string #\newline))

(define (c-preproc-define id val body)
  (string-append
    "#define " id " " val nl-str
    body
    "#undef " id nl-str))

(define (c-preproc-define-default-empty id body)
  (string-append
    "#undef " id nl-str
    body
    "#ifndef " id nl-str
    "#define " id nl-str
    "#endif" nl-str))

(define (c-result cdef? scheme-side?)
  (string-append
    c-id-prefix
    (if scheme-side?
      (if cdef? "CDEF_RESULT" "CLAM_RESULT")
      "result")))

(define (c-argument scheme-side? numbered-typ)
  (let ((i (number->string (cdr numbered-typ))))
    (string-append
      c-id-prefix
      (if scheme-side? "ARG" "arg")
      i)))

(define (c-declare-argument cdef? numbered-typ body)
  (let ((c-id (c-argument #f numbered-typ))
        (scm-id (c-argument #t numbered-typ))
        (typ (car numbered-typ))
        (i (number->string (cdr numbered-typ))))
    (string-append
      c-id-prefix
      (if (scmobj-type? typ)
        (if cdef? "BEGIN_CDEF_ARG_SCMOBJ(" "BEGIN_CLAM_ARG_SCMOBJ(")
        (if cdef? "BEGIN_CDEF_ARG(" "BEGIN_CLAM_ARG("))
      i
      (if (scmobj-type? typ)
        ""
        (string-append "," (if cdef? scm-id (c-type-decl typ c-id))))
      ")" nl-str
      body
      c-id-prefix
      (if (scmobj-type? typ)
        (if cdef? "END_CDEF_ARG_SCMOBJ(" "END_CLAM_ARG_SCMOBJ(")
        (if cdef? "END_CDEF_ARG(" "END_CLAM_ARG("))
      i ")" nl-str)))

(define (c-convert-representation cdef? to-scmobj? typ from to i body)
  (let ((tail
          (string-append
            (if to-scmobj?
              (string-append (c-type-name typ) "_TO_SCMOBJ(")
              (string-append "SCMOBJ_TO_" (c-type-name typ) "("))
            from "," to
            (if i (string-append "," i) "")
            ")" nl-str)))
    (string-append
      c-id-prefix (if cdef? "BEGIN_CDEF_" "BEGIN_CLAM_") tail
      body
      c-id-prefix (if cdef? "END_CDEF_" "END_CLAM_") tail)))

(define (c-convert-argument cdef? numbered-typ body)
  (let* ((typ
          (car numbered-typ))
         (from
          (c-argument (not cdef?) numbered-typ))
         (to
          (c-argument cdef? numbered-typ))
         (i
          (number->string (cdr numbered-typ)))
         (decl
          (c-declare-argument
            cdef?
            numbered-typ
            (if (scmobj-type? typ)
              (c-preproc-define to from body)
              (c-convert-representation cdef? cdef? typ from to i body)))))
    (if cdef?
      decl
      (c-preproc-define
        from
        (string-append
          c-id-prefix
          "CLAM_ARG("
          i
          ")")
        decl))))

(define (c-set-result cdef? result-typ)
  (cond ((void-type? result-typ)
         (string-append
           c-id-prefix
           (if cdef? "CDEF_SET_RESULT_VOID" "CLAM_SET_RESULT_VOID")
           nl-str))
        ((scmobj-type? result-typ)
         (string-append
           c-id-prefix
           (if cdef? "CDEF_SET_RESULT_SCMOBJ" "CLAM_SET_RESULT_SCMOBJ")
           nl-str))
        (else
         (c-convert-representation
           cdef?
           (not cdef?)
           result-typ
           (c-result cdef? cdef?)
           (c-result cdef? (not cdef?))
           #f
           (string-append
             c-id-prefix
             (if cdef? "CDEF_SET_RESULT" "CLAM_SET_RESULT")
             nl-str)))))
          
(define (c-make-function cdef? param-typs result-typ make-body)
  (let ((free?
         (not (every? (lambda (t) (not (dynamically-allocated-type? t)))
                      param-typs))))

    (define (convert-param-list)

      (define (scmobj? numbered-typ)
        (scmobj-type? (car numbered-typ)))

      (define (not-scmobj? numbered-typ)
        (not (scmobj? numbered-typ)))

      (let ((numbered-param-typs (number-from-1 param-typs)))
        (let convert ((numbered-typs
                        (append (keep scmobj? numbered-param-typs)
                                (keep not-scmobj? numbered-param-typs))))
          (if (null? numbered-typs)
            (make-body (c-set-result cdef? result-typ) free?)
            (c-convert-argument
              cdef?
              (car numbered-typs)
              (convert (cdr numbered-typs)))))))

    (c-preproc-define
      (string-append c-id-prefix "NARGS")
      (number->string (length param-typs))
      (if (void-type? result-typ)
        (string-append
          c-id-prefix
          (if cdef? "BEGIN_CDEF_VOID" "BEGIN_CLAM_VOID") nl-str
          (convert-param-list)
          c-id-prefix
          (if cdef?
            (string-append "CDEF_ERROR_VOID(" cdef? ")")
            (if free? "CLAM_ERROR_FREE_VOID" "CLAM_ERROR_VOID"))
          nl-str
          (if cdef?
            (c-set-result cdef? result-typ)
            "")
          c-id-prefix
          (if cdef? "END_CDEF_VOID" "END_CLAM_VOID") nl-str)
        (string-append
          c-id-prefix
          (if (scmobj-type? result-typ)
            (if cdef? "BEGIN_CDEF_SCMOBJ" "BEGIN_CLAM_SCMOBJ")
            (string-append
              (if cdef? "BEGIN_CDEF(" "BEGIN_CLAM(")
              (c-type-decl result-typ (c-result cdef? #f))
              ")"))
          nl-str
          (convert-param-list)
          c-id-prefix
          (if (scmobj-type? result-typ)
            (if cdef?
              (string-append "CDEF_ERROR_SCMOBJ(" cdef? ")")
              (if free? "CLAM_ERROR_FREE_SCMOBJ" "CLAM_ERROR_SCMOBJ"))
            (if cdef?
              (string-append "CDEF_ERROR(" cdef? ")")
              (if free? "CLAM_ERROR_FREE" "CLAM_ERROR")))
          nl-str
          (if cdef?
            (c-set-result cdef? result-typ)
            "")
          c-id-prefix
          (if (scmobj-type? result-typ)
            (if cdef? "END_CDEF_SCMOBJ" "END_CLAM_SCMOBJ")
            (if cdef? "END_CDEF" "END_CLAM"))
          nl-str
          (if cdef?
            (string-append "return " (c-result cdef? #f) ";" nl-str)
            ""))))))

(define (comma-separated strs)
  (if (null? strs)
    ""
    (string-append
      (car strs)
      (apply string-append
             (map (lambda (s) (string-append "," s)) (cdr strs))))))

(define (c-type-decl typ inner)

  (define (err)
    (compiler-internal-error "c-type-decl, unknown C type"))

  (define (prefix-inner str)
    (if (= (string-length inner) 0)
      str
      (string-append str " " inner)))

  (let ((t (source-code typ)))
    (cond ((pair? t)
           (let ((head (source-code (car t))))
             (cond ((eq? head struct-sym)
                    (prefix-inner
                      (string-append "struct " (source-code (cadr t)))))
                   ((eq? head union-sym)
                    (prefix-inner
                      (string-append "union " (source-code (cadr t)))))
                   ((eq? head pointer-sym)
                    (c-type-decl (cadr t)
                                 (string-append "*" inner)))
                   ((eq? head function-sym)
                    (c-type-decl (caddr t)
                                 (string-append
                                   "(*" inner ") "
                                   (c-param-list-with-types
                                     (source-code (cadr t))))))
                   (else
                    (err)))))
          ((string? t)
           (prefix-inner t))
          ((symbol-object? t)
           (let ((x (assq t scheme-to-c-notation)))
             (if x
               (prefix-inner (vector-ref (cdr x) 1))
               (let ((y (assq t c-interface-types)))
                 (if y
                   (c-type-decl (cdr y) inner)
                   (err))))))
          (else
           (err)))))

(define (c-param-list-with-types typs)
  (if (null? typs)
    (string-append c-id-prefix "PVOID")
    (string-append
      c-id-prefix
      "P(("
      (comma-separated (map (lambda (typ) (c-type-decl typ "")) typs))
      "),())")))

(define (c-param-id numbered-typ)
  (c-argument #f numbered-typ))

(define (c-param-list-with-ids numbered-typs)
  (if (null? numbered-typs)
    (string-append c-id-prefix "PVOID")
    (string-append
      c-id-prefix
      "P(("
      (comma-separated
        (map (lambda (t) (c-type-decl (car t) (c-param-id t)))
             numbered-typs))
      "),("
      (comma-separated (map c-param-id numbered-typs))
      ")"
      (apply string-append
             (map (lambda (t)
                    (string-append
                     nl-str
                     (c-type-decl (car t) (c-param-id t))
                     ";"))
                  numbered-typs))
      ")")))

(define (c-function-decl param-typs result-typ id scope body)
  (let ((numbered-typs (number-from-1 param-typs)))
    (let ((function-decl
           (c-type-decl result-typ
                        (string-append
                          id
                          " "
                          (if body
                            (c-param-list-with-ids numbered-typs)
                            (c-param-list-with-types param-typs))))))
      (if body
        (string-append
          scope " "
          function-decl nl-str
          "{" nl-str body "}" nl-str)
        (string-append
          function-decl ";" nl-str)))))

(define (build-c-define param-typs result-typ proc-name scope)
  (let ((proc-lbl
         (string-append
           c-id-prefix "MLBL(" c-id-prefix "CDEF_LBL_" proc-name ")")))

    (define (make-body set-result-code free?)
      (string-append
        c-id-prefix "BEGIN_CDEF_BODY" nl-str
        (let convert ((numbered-typs (number-from-1 param-typs)))
          (if (null? numbered-typs)
            (string-append
              c-id-prefix
              (cond ((void-type? result-typ)
                     "CDEF_CALL_VOID(")
                    ((scmobj-type? result-typ)
                     "CDEF_CALL_SCMOBJ(")
                    (else
                     "CDEF_CALL("))
              c-id-prefix "NARGS," proc-lbl ")" nl-str)
            (let ((numbered-typ (car numbered-typs)))
              (string-append
                c-id-prefix
                "CDEF_ARG("
                (number->string (cdr numbered-typ))
                ","
                (c-argument #t numbered-typ)
                ")" nl-str
                (convert (cdr numbered-typs))))))
        set-result-code
        c-id-prefix "END_CDEF_BODY" nl-str))

    (add-c-decl
      (c-function-decl param-typs
                       result-typ
                       proc-name
                       scope
                       (c-make-function proc-lbl
                                        param-typs
                                        result-typ
                                        make-body)))))

(define (build-c-lambda param-typs result-typ proc-name)
  (let* ((index
           (number->string c-interface-proc-count))
         (scheme-name
           (string-append module-prefix c-interface-module-name "#" index))
         (c-name
           (string-append c-id-prefix (scheme-id->c-id scheme-name)))
         (arity
           (length param-typs)))

    (define (make-body set-result-code free?)
      (string-append
        c-id-prefix
        (if free? "BEGIN_CLAM_BODY_FREE" "BEGIN_CLAM_BODY")
        nl-str
        (c-preproc-define-default-empty
          (string-append c-id-prefix "AT_END")
          (if (valid-c-id? proc-name)
            (string-append
              (if (void-type? result-typ)
                ""
                (string-append (c-result #f #f) " = "))
              proc-name "("
              (comma-separated (map c-param-id (number-from-1 param-typs)))
              ");" nl-str)
            (string-append
              proc-name nl-str)))
        set-result-code
        c-id-prefix
        (if free? "END_CLAM_BODY_FREE" "END_CLAM_BODY")
        nl-str))

    (set! c-interface-proc-count (+ c-interface-proc-count 1))
    (add-c-proc
      (make-c-proc scheme-name
                   c-name
                   arity
                   (c-make-function #f
                                    param-typs
                                    result-typ
                                    make-body)))
    scheme-name))
  
(define (scheme-id->c-id s)
  (let loop1 ((i (- (string-length s) 1)) (lst '()))
    (if (>= i 0)
      (let ((c (string-ref s i)))
        (cond ((char=? c #\_)
               (loop1 (- i 1) (cons c (cons c lst))))
              ((c-id-subsequent? c)
               (loop1 (- i 1) (cons c lst)))
              (else
               (let ((n (character->unicode c)))
                 (if (= n 0)
                   (loop1 (- i 1) (cons #\_ (cons #\0 (cons #\_ lst))))
                   (let loop2 ((n n) (lst (cons #\_ lst)))
                     (if (> n 0)
                       (loop2 (quotient n 16)
                              (cons (string-ref "0123456789abcdef"
                                                (modulo n 16))
                                    lst))
                       (loop1 (- i 1) (cons #\_ lst)))))))))
      (list->str lst))))

(define (c-id-initial? c) ; c is one of #\A..#\Z, #\a..#\z, #\_
  (let ((n (character->unicode c)))
    (or (and (>= n 65) (<= n 90))
        (and (>= n 97) (<= n 122))
        (= n 95))))

(define (c-id-subsequent? c) ; c is one of #\A..#\Z, #\a..#\z, #\_, #\0..#\9
  (let ((n (character->unicode c)))
    (or (and (>= n 65) (<= n 90))
        (and (>= n 97) (<= n 122))
        (= n 95)
        (and (>= n 48) (<= n 57)))))

(define (valid-c-id? id)
  (let ((n (string-length id)))
    (and (> n 0)
         (c-id-initial? (string-ref id 0))
         (let loop ((i (- n 1)))
           (if (> i 0)
             (if (c-id-subsequent? (string-ref id i))
               (loop (- i 1))
               #f)
             #t)))))

;==============================================================================
