package biss.awt;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.LayoutManager;
import java.awt.Point;
import java.awt.Rectangle;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/**
 * auxiliary class used by Layouter. Instances hold layout specs of a
 * single widget
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
class LayoutSpec
  implements Cloneable
{
	int OffX0;
	int OffX1;
	int OffY0;
	int OffY1;
	Component Obj;
	int X0;
	int X1;
	int Y0;
	int Y1;
	int VarSides;
	boolean DUOffset;

/**
  seems to be a Cloneable error (method should not be necessary at all)
*/
public Object clone () {
	try {
		return super.clone();
	}
	catch ( CloneNotSupportedException x ) {
		return null;
	}
}

void reshape ( int parentWidth, int parentHeight, int dx, int dy ) {

	if ( X0 == 0 && X1 == 0 && OffX0 == 0 && OffX1 == 0 ) {
		Obj.hide();
	}
	else {                          // ratio specified in 1/1000 parent dimensions
		int x = (X0 * parentWidth + 500) / 1000;
		int y = (Y0 * parentHeight + 500) / 1000;
		int width = ((X1 * parentWidth + 500) / 1000) - x;
		int height = ((Y1 * parentHeight + 500) / 1000) - y;

		if ( DUOffset ) {	  // inset specified in 1/100 SysFont units
			int cx = Awt.DlgXUnit;
			int cy = Awt.DlgYUnit;
			int dx0 = (OffX0 * cx + 50) / 100;
			int dy0 = (OffY0 * cy + 50) / 100;
			x += dx0;
			y += dy0;
			width  -= dx0 - ((OffX1 * cx-50) / 100);
			height -= dy0 - ((OffY1 * cx-50) / 100);
		}
		else {                        // inset specified in pixels
			x += OffX0;
			y += OffY0;
			width  -= (OffX0 - OffX1);
			height -= (OffY0 - OffY1);
		}

		Obj.reshape( x + dx, y + dy, width, height);
		if ( !Obj.isShowing() )
			Obj.show();
	}
}

public String toString () {
	return "LayoutSpec(" + Obj.getClass().getName() + ',' 
	+ X0 + ',' + Y0 + ',' + X1 + ',' + Y1 + ", "
	+ DUOffset + ',' + OffX0 + ',' + OffY0 + ','
	+ OffX1 + ',' + OffY1 + ')';
}
}

/**
 * LayoutManager implementation to explicitly place objects.
 * Rather than utilizing a layout "policy" (like BorderLayout etc.),
 * Layouter uses explicit specifications to place the components.
 * Each size and location can be specified with two different coordinate
 * specs: ratio and offsets.
 *
 *      0------x0---------------------x1------1000
 *      |      :                      :       |
 *      |      :                      :       |
 *      y0.....:......................:.......|
 *      |      :       | oy0          :       |
 *      |      :    +--V--------+     :       |
 *      |      :--->|           |<----:       |
 *      |      :ox0 |           | -ox1:       |
 *      |      :    +--^--------+     :       |
 *      |      :       | -oy1         :       |
 *      y1.....:.......|..............:.......|
 *      |      :                      :       |
 *   1000-------------------------------------+
 *
 * The ratio boundaries are in 1/1000 units of the parent width/height.
 * The offsets have the usual Graphics orientation (right, down) and are
 * specified either in pixels or in 1/100 so called "dialog units" (usually
 * SysFontHeight, can be set via "biss.awt.DlgXUnit" in the biss.awt
 * property file).
 *
 * In addtion, the Layouter can automatically insert horizontal or
 * vertical splitbars if a border side of a Component is marked as variable.
 *
 * Another useful feature of the Layouter are alternate layouts. In addition
 * to the mandatory (unnamed) default layout, the user can specify other
 * layout schemes which can be refered to (e.g. when switching between
 * layouts) by a symbolic name. These alternate layouts may contain the
 * same and/or different components as the default layout. Alternate layouts
 * are convenient to implement "pane zooming"
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
public class Layouter
  implements LayoutManager
{
	final public static int LEFT = 1;
	final public static int TOP = 2;
	final public static int RIGHT = 4;
	final public static int BOTTOM = 8;
	static int SplitWidth;
	Vector CurrentLayout;
	Vector DefaultLayout = new Vector(10);
	Vector EditLayout;
	LayoutSpec EditLayoutSpec;
	Vector Components = new Vector(10);
	int LastHeight;
	int LastWidth;
	Dimension MinimumLayoutSize = new Dimension( 200, 200);
	Container Owner;
	Dimension PreferredLayoutSize;
	static int DUSplitWidth;
	static int DUSplitHeight;
	Vector Layouts = new Vector( 3);

static {
	SplitWidth = Awt.SplitBarWidth;
	DUSplitHeight = (SplitWidth * 100 +Awt.DlgYUnit/2)/ Awt.DlgYUnit;
	DUSplitWidth = (SplitWidth * 100 +Awt.DlgXUnit/2)/ Awt.DlgXUnit;
}

public Layouter ( Container owner ) {
	Owner = owner;

	Layouts.addElement( DefaultLayout);

	CurrentLayout = DefaultLayout;
	EditLayout = DefaultLayout;

	Owner.setLayout( this);
}

public void add ( Component obj ) {
	EditLayoutSpec = new LayoutSpec();
	EditLayoutSpec.Obj = obj;
	EditLayout.addElement( EditLayoutSpec);
	addToOwner( obj);
}

/**
 * add a component to the layout (and to the container as well if
 * it is the first spec for this component)
 * 
 * @param obj - component to be added
 * @param x0 - left boundary of the ratio rectangle (0..1000)
 * @param y0 - top boundary of the ratio rectangle (0..1000)
 * @param x1 - right boundary of the ratio rectangle (0..1000)
 * @param y1 - bottom boundary of the ratio rectangle (0..1000)
 * @param isDUOffset - true: offsets are in (1/100 dialog units,
 *                     false: offsets are in pixels
 * @param ox0 - offset (to the right) from left ratio boundary
 * @param oy0 - offset (downward) from the top ratio boundary
 * @param ox1 - offset (to the right) from the right ratio boundary
 * @param oy1 - offset (downward) from the botton ratio boundary
 * @param varSides - int encoded spec of sides which should be variable
 *              (1: left, 2: top, 4: right, 8: bottom)
 */
public void add ( Component obj, int x0,  int y0,  int x1,  int y1,
           boolean isDUOffset, int ox0, int oy0, int ox1, int oy1,
           int varSides ) {

	add( obj);
	setRatioBoundaries( x0, y0, x1, y1);
	setInsetBoundaries( isDUOffset, ox0, oy0, ox1, oy1);
	if ( varSides != 0 )
		addSplitBars( varSides);
}

public void addInvisible ( Component obj ) {
	LayoutSpec spec = new LayoutSpec();
	spec.Obj = obj;

	CurrentLayout.addElement( spec);
	if ( CurrentLayout == DefaultLayout )
		Owner.add( obj);	
}

public void addLayoutComponent ( String name, Component comp ) {
}

public void addLike ( Component obj, int loId ) {
	LayoutSpec spec = getSpecFor( layoutAt( loId), obj);
	if ( spec != null ) {
		EditLayout.addElement( spec);
		if ( spec.VarSides != 0 )
			addSplitBars( spec.VarSides);
	}
}

public void addLikeDefault ( Component obj ) {
	addLike( obj, 0);
}

public void addSplitBars ( int varSides ) {
	EditLayoutSpec.VarSides = varSides;
	LayoutSpec spec = (LayoutSpec) EditLayoutSpec.clone();
	SplitBar  split = null;
	int w;

	if ( (varSides & LEFT) != 0 ){
		split = new HSplitBar();
		spec.X1 = spec.X0;
		w = EditLayoutSpec.DUOffset ? DUSplitWidth : SplitWidth;
		EditLayoutSpec.OffX0 += w;
		spec.OffX1 = EditLayoutSpec.OffX0;
	}
	if ( (varSides & TOP) != 0 ){
		split = new VSplitBar();
		spec.Y1 = spec.Y0;
		w = EditLayoutSpec.DUOffset ? DUSplitHeight : SplitWidth;
		EditLayoutSpec.OffY0 += w;
		spec.OffY1 = EditLayoutSpec.OffY0;
	}
	if ( (varSides & RIGHT) != 0 ){
		split = new HSplitBar();
		spec.X0 = spec.X1;
		w = EditLayoutSpec.DUOffset ? DUSplitWidth : SplitWidth;
		EditLayoutSpec.OffX1 -= w;
		spec.OffX0 = EditLayoutSpec.OffX1;
	}
	if ( (varSides & BOTTOM) != 0 ){
		split = new VSplitBar();
		spec.Y0 = spec.Y1;
		w = EditLayoutSpec.DUOffset ? DUSplitHeight : SplitWidth;
		EditLayoutSpec.OffY1 -= w;
		spec.OffY0 = EditLayoutSpec.OffY1;
	}

	if ( split != null ) {
		spec.Obj = split;
		EditLayout.addElement( spec);
		addToOwner( split);
	}
}

void addToOwner ( Component obj ) {
	int n = Components.size();

	for ( int i=0; i<n; i++ )
		if ( Components.elementAt(i) == obj ) return;

	Components.addElement( obj);
	Owner.add( obj);
}

public int countLayouts () {
	return Layouts.size();
}

public int currentLayoutId () {
	return Layouts.indexOf( CurrentLayout);
}

public void forceLayout () {
	LastWidth = 0; LastHeight = 0;  // force layout
	Owner.layout();
}

public int getFirstOf ( Component obj ) {
	for ( int i=0; i<Layouts.size(); i++) {
		if ( getSpecFor( layoutAt( i), obj) != null)
			return i;
	}
	return -1;
}

LayoutSpec getSpecFor ( Vector layout, Component c ) {
	int i, n = layout.size();
	for ( i=0; i<n; i++ ) {
		LayoutSpec l = (LayoutSpec) layout.elementAt(i);
		if ( l.Obj == c )
			return l;
	}
	return null;
}

public Vector layoutAt ( int idx) {
	if ( idx == 0 || idx > Layouts.size() )
		return DefaultLayout;
	return (Vector)Layouts.elementAt( idx);
}

public void layoutContainer ( Container parent ) {
	Dimension size = parent.size();
	Insets    in   = parent.insets();
	int       i, n = CurrentLayout.size();

	int w = size.width - in.left - in.right;
	int h = size.height - in.top - in.bottom;

	if ( w <= 0 || h <= 0 ) return;			// performance filters
	if ( w == LastWidth && h == LastHeight )	return;
	LastWidth = w; LastHeight = h;

	// hide all visible components so that there are no obscured paints
	for ( i=0; i<n; i++ ) {
		LayoutSpec spec = (LayoutSpec) CurrentLayout.elementAt( i);
		spec.Obj.hide();
	}

	// components will get visible again automatically during spec.reshape
	for ( i=0; i<n; i++ ) {
		LayoutSpec spec = (LayoutSpec) CurrentLayout.elementAt( i);
		spec.reshape( w, h, in.left, in.top);
	}

	Awt.DefToolkit.sync();
		
	// give TopWindows a chance to restore the focus
	Owner.requestFocus();
}

public Dimension minimumLayoutSize ( Container parent ) {
	return MinimumLayoutSize;
}

public void newLayout () {
	EditLayout = new Vector( DefaultLayout.size());
	Layouts.addElement( EditLayout);
}

public Dimension preferredLayoutSize ( Container parent ) {
	if ( PreferredLayoutSize != null)
		return PreferredLayoutSize;
	Dimension s = Owner.size();
	if ( (s.width > 1) && (s.height > 1) )
		return s;
	return new Dimension( 200, 200);
}

public void removeLayoutComponent ( Component comp ) {
	for( Enumeration le = Layouts.elements(); le.hasMoreElements(); ) {
		Vector v = (Vector) le.nextElement();
		for ( Enumeration lc = v.elements(); lc.hasMoreElements(); ) {
			LayoutSpec spec = (LayoutSpec) lc.nextElement();
			if ( spec.Obj == comp ){
				v.removeElement( spec);
				break;
			}
		}
	}
}

public void setDefaultLayout () {
	setLayout( 0);
}

public void setInsetBoundaries ( boolean duOffset, int ox0, int oy0, int ox1, int oy1 ) {
	EditLayoutSpec.DUOffset	= duOffset;
	EditLayoutSpec.OffX0 = ox0;
	EditLayoutSpec.OffY0 = oy0;
	EditLayoutSpec.OffX1 = ox1;
	EditLayoutSpec.OffY1 = oy1;
}

public void setLayout ( int id ) {
	Vector newLayout = layoutAt( id);
	int i, j;
	int n = CurrentLayout.size();
	int m = newLayout.size();
	Insets    in   = Owner.insets();

	if ( CurrentLayout != newLayout ) {

		// check for visible Components that should be hidden (not contained in newLayout)
		next: for ( i=0; i<n; i++ ) {
			LayoutSpec s1 = (LayoutSpec) CurrentLayout.elementAt( i);

			for ( j=0; j < m; j++ ) {
				LayoutSpec s2 = (LayoutSpec) newLayout.elementAt( j);
				if ( s2.equals( s1) ) // there is a good chance these components (menubar etc.)
					continue next;      // can be drawn without being obscured by a sibling
			}

			s1.Obj.hide();
		}

		CurrentLayout = newLayout;

		for ( j=0; j<m; j++ ) {
			LayoutSpec spec = (LayoutSpec) CurrentLayout.elementAt( j);
			spec.reshape( LastWidth, LastHeight, in.left, in.top);
		}
		
		Awt.DefToolkit.sync();

		// give TopWindows a chance to restore the focus
		Owner.requestFocus();
	}
}

public void setMinimumLayoutSize ( int width, int height ) {
	MinimumLayoutSize.width = width;
	MinimumLayoutSize.height = height;
}

public boolean setNextLayout () {
	if ( Layouts.size() == 1)
		return false;

	int id = currentLayoutId();
	if ( id < Layouts.size() - 1)
		id++;
	else
		id = 0;
	setLayout( id);
	return true;
}

public void setPreferredLayoutSize ( int width, int height ) {
	PreferredLayoutSize.width = width;
	PreferredLayoutSize.height = height;
}

public boolean setPrevLayout () {
	if ( Layouts.size() == 1)
		return false;

	int id = currentLayoutId();
	if ( id > 0) id--;
	else         id = Layouts.size()-1;
	setLayout( id);
	return true;
}

public void setRatioBoundaries ( int x0,  int y0,  int x1,  int y1 ) {
	EditLayoutSpec.X0 = x0;
	EditLayoutSpec.Y0 = y0;
	EditLayoutSpec.X1 = x1;
	EditLayoutSpec.Y1 = y1;
}
}
