package biss;

import java.util.Date;

/**
 * lib class providing general static functions on Date objects
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
public class DateLib
{
	static int[] MonthLen = { 31,28,31,30,31,30,31,31,30,31,30,31 };
	final public static long DayMillis = 86400000L;
	final public static long HourMillis = 3600000L;
	final public static long MinuteMillis = 60000L;
	public static String[] MonthNames;
	public static String[] MonthShorts;
	public static String[] DayNames;
	public static String[] DayShorts;
	final static int DATE_ISO = 1;
	public static String[] GDNames = {"Sonntag","Montag","Dienstag","Mittwoch","Donnerstag","Freitag","Samstag"};
	public static String[] ADNames = {"Sunday","Monday","Tuesday","Wednesday","Thursday","Friday","Saturday"};
	public static String[] GDShorts = {"Mo","Di","Mi","Do","Fr","Sa","So"};
	public static String[] ADShorts = {"Mo","Tu","We","Th","Fr","Sa","Su"};
	public static String[] GMNames = {"Januar","Februar","Maerz","April","Mai","Juni","Juli","August","September","Oktober","November","Dezember"};
	public static String[] AMNames = {"January","February","March","April","May","June","July","August","September","October","November","December"};
	public static String[] GMShorts = {"Jan","Feb","Mar","Apr","Mai","Jun","Jul","Aug","Sep","Okt","Nov","Dez"};
	public static String[] AMShorts = {"Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"};

static {
	boolean ger = Biss.langGerman();

	DayNames = ger ? GDNames : ADNames;
	DayShorts = ger ? GDShorts : ADShorts;
	MonthNames = ger ? GMNames : AMNames;
	MonthShorts = ger ? GMShorts : AMShorts;
}

public static Date addDays ( Date dRef, int ad) {
	return addMinutes( dRef, ad*1440);
}

public static Date addHours ( Date dRef, int ad) {
	return addMinutes( dRef, ad*60);
}

public static Date addMinutes ( Date dRef, int ad) {
	int  rto = dRef.getTimezoneOffset();
	Date nd  = new Date( dRef.getTime() + MinuteMillis*ad);
	int  nto = nd.getTimezoneOffset();

	// include DST (daylight saving time) switch time deltas
	if ( rto != nto)
		nd.setTime( nd.getTime() + (nto-rto)*MinuteMillis );

	return nd;
}

public static Date clone ( Date dRef) {
	return new Date( dRef.getTime() );
}

public static Date dateFromEDF ( String s ) {

	// get date from european date format( <dd.mm.yyyy> )

	int dy=0, mo=0, ye=0;
	int idx, len=s.length();
	char c;

	for ( idx=0; idx<len; idx++) {
		c = s.charAt( idx);
		if ( c == '.') break;
		dy=dy*10+c-48;
	}

	for ( idx++; idx<len; idx++) {
		c = s.charAt( idx);
		if ( c == '.') break;
		mo=mo*10+c-48;
	}

	for ( idx++; idx<len; idx++) {
		c = s.charAt( idx);
		ye=ye*10+c-48;
	}

	try { return new Date( ye-1900, mo-1, dy); }
	catch( Throwable t) { return null; }
}

public static Date dateFromEDTF ( String s ) {

	// get date from european date time format( <dd.mm.yyy hh:mm> )

	int dy=0, mo=0, ye=0, hr=0, mn=0;
	int idx, len=s.length();
	char c;

	for ( idx=0; idx<len; idx++) {
		c = s.charAt( idx);
		if ( c == '.') break;
		dy=dy*10+c-48;
	}

	for ( idx++; idx<len; idx++) {
		c = s.charAt( idx);
		if ( c == '.') break;
		mo=mo*10+c-48;
	}

	for ( idx++; idx<len; idx++) {
		c = s.charAt( idx);
		if ( c == ' ') break;
		ye=ye*10+c-48;
	}

	for ( idx++; idx<len; idx++) {
		c = s.charAt( idx);
		if ( c == ':') break;
		hr=hr*10+c-48;
	}

	for ( idx++; idx<len; idx++) {
		c = s.charAt( idx);
		if ( c == ' ') break;
		mn=mn*10+c-48;
	}

	try { return new Date( ye-1900, mo-1, dy, hr, mn); }
	catch( Throwable t) { return null; }
}

public static int dayInMonth () {
	return dayInMonth( new Date() );
}

public static int dayInMonth ( Date d) {
	return d.getDate();
}

public static int dayInWeek () {
	return dayInWeek( new Date() );
}

public static int dayInWeek ( Date d) {
	return d.getDay();
}

public static int dayOfYear ( Date d) {
	int ye = d.getYear() + 1900;
	int mn = d.getMonth() + 1;
	int doy = 0;

	for ( int i=1; i<mn; i++)
		doy += daysInMonth( i, ye);
	doy += d.getDate();

	return doy;
}

public static int daysInMonth ( Date d) {
	return daysInMonth( d.getMonth()+1, d.getYear()+1900 );
}

public static int daysInMonth ( int mn, int ye) {
	int mof=0;
	if ( (mn<1) || (mn>12))
		return 0;
	if ( (mn==2) && (isLeapYear( ye)) )
		mof=1;
	return MonthLen[mn-1]+mof;    
}

public static Date endOfWeek ( Date dRef) {
	return addMinutes( dRef, 7*1440 - 1);
}

public static Date firstWeekDay ( Date dRef) {
	long di = (long)24*3600000;
	Date nd = new Date( dRef.getTime() );
	while( nd.getDay() != 1)
		nd.setTime( nd.getTime() - di);
	return nd;
}

public static int[][] getDayArray( int mn, int ye) {

	int[][] darr = new int[7][6];

	int maxd = daysInMonth( mn, ye);
	int sd = (new Date( ye, mn-1, 1)).getDay();

	if ( sd == 0) sd = 7;	// day starts with sunday

	for( int day=sd-1; day<maxd+sd-1; day++)
		darr[day%7][day/7] = day-sd+2;

	return darr;
}

public static boolean intersects ( Date dRefStart, Date dRefEnd,
                     Date dStart, Date dEnd) {
	if ( dRefStart.getTime() > dEnd.getTime()-1 )
		return false;
	if ( dStart.getTime() > dRefEnd.getTime()-1 )
		return false;
	return true;
}

public static boolean isLeapYear ( int year) {
	if ( year%400==0 )
		return true;
	if ( year%100==0 )
		return false;

	return (year%4)==0;
}

public static void main ( String[] args) {
	System.out.println( dateFromEDF( "12.12.1970"));
	System.out.println( dateFromEDTF( "12.12.1978 12:12"));
	System.out.println( weekNumber( today() ));
}

public static int monthToDays ( int mn) {
	return ((mn * 3057 - 3007) / 100);
}

public static Date nextDay ( Date dRef) {
	return addDays( dRef, 1);
}

public static Date nextMonth ( Date dRef) {
	int dm = daysInMonth( dRef);
	return addDays( dRef, dm - dRef.getDate() + 1);
}

public static Date nextWeek ( Date dRef) {
	return addDays( dRef, 7);
}

public static Date prevDay ( Date dRef) {
	return addDays( dRef, -1);
}

public static Date prevMonth ( Date dRef) {
	Date nd = addDays( dRef, -dRef.getDate() );
	return addDays( nd, 1 - daysInMonth( nd));
}

public static Date prevWeek ( Date dRef) {
	return addDays( dRef, -7);
}

public static String toDMString ( Date dt ) {
	int da = dt.getDate();
	int dy = dt.getDay();
	int mo = dt.getMonth();

	return (DayNames[dy] + ", " + (da/10) + (da%10) + "." + MonthNames[mo] );
}

public static String toDMYString ( Date dt ) {
	int da = dt.getDate();
	int dy = dt.getDay();
	int mo = dt.getMonth();

	return (DayNames[dy] + ", " + (da/10) + (da%10) + "." + MonthNames[mo] +
	        " " + (1900 + dt.getYear()) );
}

public static String toEDFString ( Date dt ) {

	StringBuffer b = new StringBuffer( "19.07.1996");
	int dy = dt.getDate();
	int mo = dt.getMonth()+1;
	int ye = dt.getYear()+1900;

	b.setCharAt(0, (char)(dy/10+48));    b.setCharAt(1, (char)(dy%10+48));
	b.setCharAt(3, (char)(mo/10+48));	   b.setCharAt(4, (char)(mo%10+48));
	b.setCharAt(6, (char)(ye/1000+48));  b.setCharAt(7, (char)(ye%1000/100+48));
	b.setCharAt(8, (char)(ye%100/10+48));b.setCharAt(9, (char)(ye%10+48));

	return b.toString();
}

public static String toEDTFString ( Date dt ) {

	StringBuffer b = new StringBuffer( "19.07.1996 17:15");
	int dy = dt.getDate();
	int mo = dt.getMonth()+1;
	int ye = dt.getYear()+1900;
	int hr = dt.getHours();
	int mn = dt.getMinutes();

	b.setCharAt(0, (char)(dy/10+48));    b.setCharAt(1, (char)(dy%10+48));
	b.setCharAt(3, (char)(mo/10+48));	   b.setCharAt(4, (char)(mo%10+48));
	b.setCharAt(6, (char)(ye/1000+48));  b.setCharAt(7, (char)(ye%1000/100+48));
	b.setCharAt(8, (char)(ye%100/10+48));b.setCharAt(9, (char)(ye%10+48));
	b.setCharAt(11,(char)(hr/10+48));    b.setCharAt(12,(char)(hr%10+48));
	b.setCharAt(14,(char)(mn/10+48));    b.setCharAt(15,(char)(mn%10+48));

	return b.toString();
}

public static String toHMSString ( Date dt ) {
	return toHMSString( dt, null);
}

public static String toHMSString ( Date dt, StringBuffer b ) {

	int hr = dt.getHours();
	int mn = dt.getMinutes();
	int se = dt.getSeconds();

	if ( b == null)
		b = new StringBuffer( "11:11:11" );

	b.setCharAt(0,(char)(hr/10+48));    b.setCharAt(1,(char)(hr%10+48));
	b.setCharAt(3,(char)(mn/10+48));    b.setCharAt(4,(char)(mn%10+48));
	b.setCharAt(6,(char)(se/10+48));    b.setCharAt(7,(char)(se%10+48));

	return b.toString();
}

public static String toHMString ( Date dt ) {

	StringBuffer b = new StringBuffer( "17:15");
	int hr = dt.getHours();
	int mn = dt.getMinutes();

	b.setCharAt(0,(char)(hr/10+48));    b.setCharAt(1,(char)(hr%10+48));
	b.setCharAt(3,(char)(mn/10+48));    b.setCharAt(4,(char)(mn%10+48));

	return b.toString();
}

public static String toMString ( Date dt ) {

	StringBuffer b = new StringBuffer( ":15");
	int mn = dt.getMinutes();

	b.setCharAt(1,(char)(mn/10+48));
	b.setCharAt(2,(char)(mn%10+48));

	return b.toString();
}

public static String toMYString ( Date dt ) {
	return (MonthNames[dt.getMonth()] + " " + (dt.getYear()+1900));
}

public static Date today () {
	return new Date();
}

public static void trim ( Date dt) {
	dt.setHours( 0);
	dt.setMinutes( 0);
	dt.setSeconds( 0);
}

public static int weekNumber ( Date dt) {
	int ye = dt.getYear() + 1900;
	int dn = dayOfYear( dt);
	int jd = ymdToScalar( ye, 1, 1);
	int j1 = (jd - DATE_ISO) % 7;
	int wn;

	dn += j1 - 1;
	wn = dn / 7;
	if ( DATE_ISO == 1)
		wn += (j1 < 4) ? 1 : 0;
	else
		wn ++;

	return wn;
}

public static int within ( Date dRef, Date dMin, Date dMax) {
	if ( dRef.before( dMin))
		return 1;
	if ( dRef.after( dMax))
		return -1;
	return 0;
}

public static int yearsToDays ( int ye) {
	return (ye * 365 + ye / 4 - ye / 100 + ye / 400);
}

public static int ymdToScalar ( int ye, int mn, int dy) {
	int scalar = dy + monthToDays( mn);

	if ( mn > 2)
		scalar -= isLeapYear( ye) ? 1 : 2;

	ye--;
	scalar += yearsToDays( ye);

	return scalar;
}
}
