package biss.awt;

import java.awt.Dimension;
import java.awt.Image;
import java.awt.image.ColorModel;
import java.awt.image.ImageConsumer;
import java.awt.image.ImageProducer;

/**
 * base class to implement portable Images which do not rely on a URL connection
 * or local file access. Instead of keeping Images (in various native formats
 * like GIF, JPEG) in files, these images are translated to plain vanilla
 * Java classes containing the pixel data as MemoryImages and providing
 * it to clients via the java.awt.image.ImageProducer interface.
 *
 * Instead of specifying/programming all the pixel data manually, a specialed
 * tool called 'biss.jde.ImageEditor' is used to 'paint' the contents and
 * generate the class (automatically).
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
abstract public class ClassImageSource
  implements ImageProducer
{
	ColorModel Model;
	Object Pels;
	ImageConsumer Consumer;
	Dimension Dim;

protected ClassImageSource (){
}

public ClassImageSource(int w, int h, byte[] pels){
	initialize(w, h, (Object) pels);
}

public synchronized void addConsumer(ImageConsumer ic) {
	Consumer = ic;
	try { produce(); }
	catch (Exception e) {
		if (Consumer != null)
			Consumer.imageComplete(ImageConsumer.IMAGEERROR);
	}
	Consumer = null;
}

public static Image getImage( String imgCls){
	try {
		Class c = Class.forName( imgCls);
		ClassImageSource cis = (ClassImageSource)c.newInstance();
		return Awt.DefToolkit.createImage( cis);
	}
	catch ( Exception e) {
		System.out.println( e);
	}
	return null;
}

void initialize(int w, int h, ColorModel cm, Object pels) {
	Dim = new Dimension( w, h);
	Model = cm;
	Pels = pels;
}

void initialize(int w, int h, Object pels) {
	initialize( w, h, DefaultIndexModel.Model, pels);
}

public synchronized boolean isConsumer(ImageConsumer ic) {
	return (ic == Consumer);
}

private void produce() {
	if (Consumer != null)
		Consumer.setDimensions(Dim.width, Dim.height);
	if (Consumer != null)
		Consumer.setColorModel(Model);
	if (Consumer != null)
		Consumer.setHints(ImageConsumer.TOPDOWNLEFTRIGHT |
	                  ImageConsumer.COMPLETESCANLINES |
	                  ImageConsumer.SINGLEPASS |
	                  ImageConsumer.SINGLEFRAME);
	if (Consumer != null) {
		if (Pels instanceof byte[])
			Consumer.setPixels(0, 0, Dim.width, Dim.height, Model,
		                   ((byte[]) Pels), 0, Dim.width);
		else
			Consumer.setPixels(0, 0, Dim.width, Dim.height, Model,
		                   ((int[]) Pels), 0, Dim.width);
	}
	if (Consumer != null)
		Consumer.imageComplete(ImageConsumer.STATICIMAGEDONE);
}

public synchronized void removeConsumer(ImageConsumer ic) {
	if (Consumer == ic)
		Consumer = null;
}

public void requestTopDownLeftRightResend(ImageConsumer ic) {
	// Not needed.  The data is always in TDLR format.
}

public void startProduction(ImageConsumer ic) {
	addConsumer(ic);
}
}
