/* Copyright (C) 1994 
            Olav Woelfelschneider (wosch@rbg.informatik.th-darmstadt.de)

  McInfoRequest.c

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; see the file COPYING.LIB.  If
  not, write to the Free Software Foundation, Inc., 675 Mass Ave,
  Cambridge, MA 02139, USA.
*/
#include "../config.h"
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <signal.h>
#include <unistd.h>
#include "McAlloc.h"
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>

#include "McApp.h"
#include "McGadget.h"
#include "McSlider.h"
#include "McText.h"
#include "McBitmap.h"
#include "McString.h"
#include "McSelector.h"
#include "McResource.h"
#include "McUtils.h"

#include "McInfoRequest.h"


enum { TXT_CENTERED, TXT_FLUSHLEFT, TXT_FLUSHRIGHT, };

struct line_t {
  int width;
  int len;
  int orientation;
  int offset;
  unsigned char *data;
};


static void gadget_proc(McGadget *gadget);
static int calculate_text(McInfoRequest *req, unsigned char *text,
			  int *lines, int *maxwidth, struct line_t *line);
static int textwidth(McInfoRequest *req, const unsigned char *text);
static  int event_proc(McWindow *window, XEvent *event);

McInfoRequest *McCreateInfoRequest(McApp *app,
				   const unsigned char *title,
				   const unsigned char *text,
				   const unsigned char *yes,
				   const unsigned char *middle,
				   const unsigned char *no,
				   void (*callback)(int, void *)) {
  McInfoRequest *req;
  int i, gadwidth;
  int width, height;
  McGadget *gad;
  unsigned char *message;
  struct line_t line[25];
  int lines, maxwidth, lineheight;

  if (!text) return NULL;
  message=strdup(text); /* Make it writable if it isn't */

  if (!(req=malloc(sizeof(McInfoRequest)))) return NULL;

  req->app=app;
  req->callback=callback;

  lineheight=calculate_text(req, message, &lines, &maxwidth, line);

  gadwidth=0;
  if (yes)    gadwidth+=textwidth(req,yes)+20;
  if (middle) gadwidth+=textwidth(req,middle)+20;
  if (no)     gadwidth+=textwidth(req,no)+20;
  if (maxwidth<gadwidth) maxwidth=gadwidth;

  width=maxwidth+20; height=(lines*lineheight)+60;

  /*
   * Create the window
   */
  req->mcw=McCreateSimpleWindow(app, title, width, height, width, height,
				width>>1, height-10, NULL, event_proc);

  if (yes) {
    gad= MakeButton(req->mcw, 20, -8, 0, 0, 1,
		    (unsigned char *)yes, gadget_proc);
    gad->topLeftGravity = SouthWestGravity;
    gad->customData = req;
  }

  if (middle) {
    gad= MakeButton(req->mcw, 0, -8, 0, 0, -1,
		    (unsigned char *)middle, gadget_proc);
    gad->topLeftGravity = SouthWestGravity;
    gad->customData = req;
  }

  if (no) {
    gad= MakeButton(req->mcw, -20, -8, 0, 0, 0,
		    (unsigned char *)no, gadget_proc);
    gad->topLeftGravity = SouthWestGravity;
    gad->customData = req;
  }

  req->mcw->customData = req;

  for(i=0; i<lines; i++) {
    switch(line[i].orientation) {
    case TXT_CENTERED:
      MakeText(req->mcw,
	       (req->mcw->w-line[i].width)>>1,
	       (lineheight*i)+15,
	       0, line[i].data);
      break;
    case TXT_FLUSHLEFT:
      MakeText(req->mcw, 14+line[i].offset,
	       (lineheight*i)+15, 0, line[i].data);
      break;
    case TXT_FLUSHRIGHT:
      MakeText(req->mcw,
	       (req->mcw->w-line[i].width-15),
	       (lineheight*i)+15, 0, line[i].data);
      break;
    }
  }

  /*
   * Display the window
   */
  McInitGadgets(req->mcw);
  XMapWindow(app->display, req->mcw->window);

  free(message); /* Don't need it any longer */

  /* Setup complete. Let the event-loop do the rest. */
  return req;
}

void McRemoveInfoRequest(McInfoRequest **req) {
  if (*req) {
    (*req)->mcw->flags|=MCW_CLOSEREQUEST;
    free(*req);
    *req=NULL;
  }
}

/**************************************************************************/

static void gadget_proc(McGadget *gadget) {
  McInfoRequest *req=(McInfoRequest *)gadget->customData;
  gadget->mcw->flags|=MCW_CLOSEREQUEST;
  if (req->callback) (*(req->callback))(gadget->id, req->customData);
  free(req);
}

/**************************************************************************/

static int calculate_text(McInfoRequest *req, unsigned char *text,
			  int *lines, int *maxwidth, struct line_t *line) {
  int len, i;
  unsigned char *begin, *end, *chp;
  int direction, ascent, descent, ch;
  int orientation, offset, maxoffset;
  XCharStruct overall;

  *maxwidth=0; *lines=0; orientation=TXT_CENTERED; maxoffset=0;
  begin=text;
  while(*begin) {
    offset=0;
    end=begin;
    while((*end) && (*end!='\n')) end++;
    len=end-begin;
    if (*begin=='&') {
      switch(begin[1]) {
      case 'c':
	orientation=TXT_CENTERED;
	begin+=2;
	break;
      case 'l':
	orientation=TXT_FLUSHLEFT;
	begin+=2;
	break;
      case 'r':
	orientation=TXT_FLUSHRIGHT;
	begin+=2;
	break;
      case 'L':
	orientation=TXT_FLUSHLEFT;
	begin+=2;
	ch=*begin++;
	if (len && (chp=memchr(begin, ch, len)) && (chp>begin)) {
	  XTextExtents(req->app->defaultFont, begin, chp-begin, &direction,
		       &ascent, &descent, &overall);
	  offset=overall.width;
	}
	break;

      case '&':
	begin++;
	break;
      default:
	break;
      }
    }

    XTextExtents(req->app->defaultFont, begin, len, &direction,
		 &ascent, &descent, &overall);
    if (offset>maxoffset) maxoffset=offset;
    line[*lines].width=overall.width;
    line[*lines].len=len;
    line[*lines].offset=offset;
    line[*lines].data=begin;
    line[*lines].orientation=orientation;
    (*lines)++;
    if (!*end) break;
    *end=0;
    begin=end+1;
  }

  for (i=0; i<*lines; i++) {
    line[i].offset=maxoffset-line[i].offset;
    line[i].width+=line[i].offset;
    if (line[i].width>(*maxwidth)) *maxwidth=line[i].width;
  }

  return ascent+descent+2;
}

static int textwidth(McInfoRequest *req, const unsigned char *text) {
  int direction, ascent, descent;
  XCharStruct overall;
  XTextExtents(req->app->defaultFont, text, strlen(text), &direction,
	       &ascent, &descent, &overall);
  return overall.width;
}

/**************************************************************************
 * static int event_proc(McWindow *mcw, XEvent *event)
 *
 */
static int event_proc(McWindow *mcw, XEvent *event) {
  McInfoRequest *req=(McInfoRequest *)mcw->customData;

  if(event->type==ClientMessage) {
    if ((event->xclient.format == 32) &&
	(event->xclient.data.l[0] == req->app->wmDelWin)) {
      mcw->flags|=MCW_CLOSEREQUEST;
      if (req->callback) (*(req->callback))(0, req->customData);
      return 1;
    }
  }
  return 0;
}

/***************************************************************************/

void McError(McApp *app, const char *fmt, ...) {
  const char *msg;
  va_list args;
  unsigned char buf[1024];
  va_start(args, fmt);
  vsprintf(buf, fmt, args);
  va_end(args);

#ifndef NO_FUNNY_MSG
  {
    int num;
    static const char *funny_msg[]={
      "So what?", "Doh!", "Oh no!", "Shut up!", "Never mind!", "Oh really?",
      "Wheee...",
    };
    do {
      num=((random()>>8)*(sizeof(funny_msg)/sizeof(char *)))/(RAND_MAX>>8);
    } while(num>=(sizeof(funny_msg)/sizeof(char *)));
    msg = funny_msg[num];
  }
#else
  msg = "Ok";
#endif

  McCreateInfoRequest(app, "Error", buf, NULL, msg, NULL, NULL);
} 

/***************************************************************************/

void McInfo(McApp *app, const char *fmt, ...) {
  const char *msg;
  va_list args;
  unsigned char buf[1024];
  va_start(args, fmt);
  vsprintf(buf, fmt, args);
  va_end(args);

#ifndef NO_FUNNY_MSG
  {
    int num;
    static const char *funny_msg[]={
      "Very nice!", "So?", "Wow!", "Sure!", "I don't care!", "Cool!",
    };
    do {
      num=((random()>>8)*(sizeof(funny_msg)/sizeof(char *)))/(RAND_MAX>>8);
    } while(num>=(sizeof(funny_msg)/sizeof(char *)));
    msg = funny_msg[num];
  }
#else
  msg = "Ok";
#endif

  McCreateInfoRequest(app, "Info", buf, NULL, msg, NULL, NULL);
} 

/**************************************************************************/

void McSync(McApp *app) {
  XSync(app->display, 0);
}

