/*
                             Joystick wrapper

	Private Functions:

	int JSLoadCaliberationLinux(js_data_struct *jsd)


	Public Functions:

	int JSIsAxisAllocated(js_data_struct *jsd, int n)
	int JSIsButtonAllocated(js_data_struct *jsd, int n)

	double JSGetAxisCoeff(js_data_struct *jsd, int n)
	double JSGetAxisCoeffNZ(js_data_struct *jsd, int n)
	int JSLoadCaliberationLinux(js_data_struct *jsd)

	int JSInit(
	        js_data_struct *jsd,
	        int argc, char *argv[]
	)
	int JSUpdate(js_data_struct *jsd)
	void JSClose(js_data_struct *jsd)

	---

	Note: JSLoadCaliberationLinux() is called by JSInit().

 */

#include <stdio.h>
#include <db.h>
#include <malloc.h>
#include <string.h>
#include <stdlib.h>   

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>

extern int errno;
#include <errno.h>


#include "../include/cfgfmt.h"
#include "../include/string.h"
#include "../include/strexp.h"
#include "../include/disk.h"
#include "../include/fio.h"

#include "../include/jsw.h"


/*
 *	Checks if axis n is allocated on jsd.
 */
int JSIsAxisAllocated(js_data_struct *jsd, int n)
{
	if(jsd == NULL)
	    return(0);
	else if((n < 0) || (n >= jsd->total_axises))
	    return(0);
	else if(jsd->axis[n] == NULL)
	    return(0);
	else
	    return(1);
}

/*
 *      Checks if button n is allocated on jsd.
 */
int JSIsButtonAllocated(js_data_struct *jsd, int n)
{
        if(jsd == NULL)
            return(0);
        else if((n < 0) || (n >= jsd->total_buttons))
            return(0);
        else if(jsd->button[n] == NULL)
            return(0);
        else
            return(1);
}

/*
 *	Returns coefficient value from -1 to 1 for current
 *	axis position.
 */
double JSGetAxisCoeff(js_data_struct *jsd, int n)
{
	int x, r;
	js_axis_struct *axis;

	if(JSIsAxisAllocated(jsd, n))
	{
	    axis = jsd->axis[n];
	    x = axis->cur;
	    if(x < axis->cen)
	    {
		/* Negative position. */
		r = axis->min - axis->cen;
		if(r != 0)
		    return(
		        (double)(x - axis->cen) /
		        (double)r * (double)((axis->flip) ? 1 : -1)
		    );
		else
		    return(0);
	    }
	    else
	    {
		/* Positive position. */
                r = axis->max - axis->cen;
                if(r != 0)
                    return(
                        (double)(x - axis->cen) /
                        (double)r * (double)((axis->flip) ? -1 : 1)
                    );
                else
                    return(0);
	    }
	}
	else
	{
	    return(0);
	}
}

/*
 *	Same as JSGetAxisCoefficient() except that it takes
 *	the nullzone into account.  Returning 0 if the position
 *	is within the nullzone.
 */
double JSGetAxisCoeffNZ(js_data_struct *jsd, int n)
{
        int x, r;
        js_axis_struct *axis;

        if(JSIsAxisAllocated(jsd, n))
        {
            axis = jsd->axis[n];
            x = axis->cur;

	    /* Null zone check. */
	    r = x - axis->cen;
            if((r < (axis->nz)) &&
               (r > (axis->nz * -1))
	    )
		return(0);

            if(x < axis->cen)
            {
                /* Negative position. */
                r = axis->min - axis->cen;
                if(r != 0)
                    return(
                        (double)(x - axis->cen) /
                        (double)r * (double)((axis->flip) ? 1 : -1)
                    );
                else
                    return(0);
            }   
            else
            {
                /* Positive position. */
                r = axis->max - axis->cen;
                if(r != 0)
                    return(
                        (double)(x - axis->cen) /
                        (double)r * (double)((axis->flip) ? -1 : 1)
                    );
                else
                    return(0);
            }
        }
        else 
        {
            return(0);
        }
}


#ifdef __linux__
/*
 *	This function is internally called (as neede) by JSInit()
 *	external functions need not call this.
 *
 *	Loads caliberation information from the caliberation file
 *	specified in the jsd structure and loads caliberation information
 *	into the jsd structure.
 */
int JSLoadCaliberationLinux(js_data_struct *jsd)
{
	int i, p;
	char this_device = 0;
        char *strptr, *strptr2;
  
        FILE *fp;
  
        char parm[CFG_PARAMETER_MAX];
        char val[CFG_VALUE_MAX];
        int lines_read = 0;

	int axis_num, button_num;
/*
	js_axis_struct *axis_ptr;
	js_button_struct *button_ptr;
 */


	if(jsd == NULL)
	    return(-1);

	if(jsd->device_name == NULL)
	    return(-1);
        if(jsd->caliberation_file == NULL)
            return(-1);

        fp = fopen(jsd->caliberation_file, "r");
        if(fp == NULL)
	    return(-1);


	strptr = NULL;

        while(1)
        {
            /* Free previous line and allocate/read next line. */
            free(strptr); strptr = NULL;
            strptr = FReadNextLineAllocCount(
		fp, UNIXCFG_COMMENT_CHAR, &lines_read
	    );
            if(strptr == NULL) break;

            /* Fetch parameter. */
            strptr2 = StringCfgParseParm(strptr);
            if(strptr2 == NULL) continue;
            strncpy(parm, strptr2, CFG_PARAMETER_MAX);
            parm[CFG_PARAMETER_MAX - 1] = '\0';

            /* Fetch value. */
            strptr2 = StringCfgParseValue(strptr);
            if(strptr2 == NULL) strptr2 = "0";  /* Set it to "0" if NULL. */
            strncpy(val, strptr2, CFG_VALUE_MAX);
            val[CFG_VALUE_MAX - 1] = '\0';


            /* BeginJoystick */
            if(!strcasecmp(parm, "BeginJoystick"))
            {
		/*   Joystick device name must match, so we know
		 *   that this configuration block is for this
		 *   device.
		 */
		if(strcmp(val, jsd->device_name))
		    this_device = 0;
		else
		    this_device = 1;

                while(1)
                {
                    /* Free previous line and allocate/read next line. */
                    free(strptr); strptr = NULL;
                    strptr = FReadNextLineAllocCount(
			fp, UNIXCFG_COMMENT_CHAR, &lines_read
		    );
                    if(strptr == NULL) break;

                    /* Fetch parameter. */
                    strptr2 = StringCfgParseParm(strptr);
                    if(strptr2 == NULL) continue;
                    strncpy(parm, strptr2, CFG_PARAMETER_MAX);
                    parm[CFG_PARAMETER_MAX - 1] = '\0';

                    /* Fetch value. */
                    strptr2 = StringCfgParseValue(strptr);
                    if(strptr2 == NULL) strptr2 = "0";
                    strncpy(val, strptr2, CFG_VALUE_MAX);
                    val[CFG_VALUE_MAX - 1] = '\0';

		    /* *********************************************** */
                    /* BeginAxis */
                    if(!strcasecmp(parm, "BeginAxis") &&
                       this_device
                    )
                    {
			axis_num = atoi(val);
			if(axis_num < 0)
			{
			    fprintf(
				stderr,
				"%s: Line %i: Invalid axis index %i.\n",
				jsd->caliberation_file,
				lines_read,
				axis_num
			    );
			    axis_num = 0;
			}

			/* Need to allocate more axises? */
			if(axis_num >= jsd->total_axises)
			{
			    p = jsd->total_axises;
			    jsd->total_axises = axis_num + 1;
			    jsd->axis = (js_axis_struct **)realloc(
				jsd->axis,
				jsd->total_axises * sizeof(js_axis_struct *)
			    );
			    if(jsd->axis == NULL)
			    {
				jsd->total_axises = 0;
			    }

			    for(i = p; i < jsd->total_axises; i++)
				jsd->axis[i] = NULL;
			}

			/* If axis_num is valid, allocate as needed. */
			if((axis_num >= 0) && (axis_num < jsd->total_axises))
			{
			    if(jsd->axis[axis_num] == NULL)
				jsd->axis[axis_num] = (js_axis_struct *)calloc(
				    1,
                                    sizeof(js_axis_struct)
				);
			}


                        while(1)
                        {
                            /* Free previous line and allocate/read next line. */
                            free(strptr); strptr = NULL;
                            strptr = FReadNextLineAllocCount(
				fp, UNIXCFG_COMMENT_CHAR, &lines_read
			    );
                            if(strptr == NULL) break;

                            /* Fetch parameter. */
                            strptr2 = StringCfgParseParm(strptr);
                            if(strptr2 == NULL) continue;
                            strncpy(parm, strptr2, CFG_PARAMETER_MAX);
                            parm[CFG_PARAMETER_MAX - 1] = '\0';

                            /* Fetch value. */
                            strptr2 = StringCfgParseValue(strptr);
                            if(strptr2 == NULL) strptr2 = "0";
                            strncpy(val, strptr2, CFG_VALUE_MAX);
                            val[CFG_VALUE_MAX - 1] = '\0';


			    /* Minimum */
			    if(!strcasecmp(parm, "Minimum"))
			    {
				if(JSIsAxisAllocated(jsd, axis_num))
				    jsd->axis[axis_num]->min = atoi(val);
			    }
                            /* Maximum */
                            else if(!strcasecmp(parm, "Maximum"))
                            {
                                if(JSIsAxisAllocated(jsd, axis_num))
                                    jsd->axis[axis_num]->max = atoi(val);
                            }
                            /* Center */
                            else if(!strcasecmp(parm, "Center"))
                            {
                                if(JSIsAxisAllocated(jsd, axis_num))
                                    jsd->axis[axis_num]->cen = atoi(val);
                            }
                            /* NullZone */
                            else if(!strcasecmp(parm, "NullZone"))
                            {
                                if(JSIsAxisAllocated(jsd, axis_num))
                                    jsd->axis[axis_num]->nz = atoi(val);
                            }
                            /* Flip */
                            else if(!strcasecmp(parm, "Flip") ||
                                    !strcasecmp(parm, "Flipped")
			    )
                            {
                                if(JSIsAxisAllocated(jsd, axis_num))
                                    jsd->axis[axis_num]->flip = 1;
                            }
                            /* IsHat */
                            else if(!strcasecmp(parm, "IsHat"))
                            {
                                if(JSIsAxisAllocated(jsd, axis_num))
                                    jsd->axis[axis_num]->is_hat = 1;
                            }

                            /* EndAxis */
                            else if(!strcasecmp(parm, "EndAxis"))
                            {
                                break;
                            }
			}
                    }
                    /* *********************************************** */
                    /* BeginButton */
                    else if(!strcasecmp(parm, "BeginButton") &&
                            this_device
                    )
                    {
                        button_num = atoi(val);
                        if(button_num < 0)
                        {
                            fprintf(     
                                stderr,
                                "%s: Line %i: Invalid button index %i.\n",
                                jsd->caliberation_file,
                                lines_read,
                                button_num
                            );  
                            button_num = 0;
                        }

                        /* Need to allocate more buttons? */
                        if(button_num >= jsd->total_buttons)
                        {
                            p = jsd->total_buttons;
                            jsd->total_buttons = button_num + 1;
                            jsd->button = (js_button_struct **)realloc(
                                jsd->button,
                                jsd->total_buttons * sizeof(js_button_struct *)
                            );
                            if(jsd->button == NULL)
                            {
                                jsd->total_buttons = 0;
                            }
                            
                            for(i = p; i < jsd->total_buttons; i++)
                                jsd->button[i] = NULL;
                        }
                    
                        /* If button_num is valid, allocate as needed. */
                        if((button_num >= 0) && (button_num < jsd->total_buttons))
                        {
                            if(jsd->button[button_num] == NULL)
                                jsd->button[button_num] = (js_button_struct *)calloc(
                                    1,
                                    sizeof(js_button_struct)
                                );
                        }


                        while(1)
                        {   
                            /* Free previous line and allocate/read next line. */
                            free(strptr); strptr = NULL;
                            strptr = FReadNextLineAllocCount(
				fp, UNIXCFG_COMMENT_CHAR, &lines_read
			    );
                            if(strptr == NULL) break;

                            /* Fetch parameter. */
                            strptr2 = StringCfgParseParm(strptr);
                            if(strptr2 == NULL) continue;
                            strncpy(parm, strptr2, CFG_PARAMETER_MAX);
                            parm[CFG_PARAMETER_MAX - 1] = '\0';

                            /* Fetch value. */
                            strptr2 = StringCfgParseValue(strptr);
                            if(strptr2 == NULL) strptr2 = "0";
                            strncpy(val, strptr2, CFG_VALUE_MAX);
                            val[CFG_VALUE_MAX - 1] = '\0';

                            /* EndButton */
                            if(!strcasecmp(parm, "EndButton"))
                            {
                                break;
                            }
			}
                    }
                    /* *********************************************** */
                    /* EndJoystick */
                    else if(!strcasecmp(parm, "EndJoystick"))
                    {
			break;
                    }
		}
	    }
	}

	/* Close file. */
	fclose(fp);


	return(0);
}
#endif	/* __linux__ */


/*
 *	Initializes the joystick device, the required arguments
 *	are:
 *
 *	-d <device>
 *	-f <caliberation_file>
 *
 *	Note that the proper caliberation file will be automatically
 *	loaded and its data fetched into the jsd structure.
 */
int JSInit(
	js_data_struct *jsd,
	int argc, char *argv[]
)
{
	int i;
	char non_blocking = 1;

#ifdef __linux__
	unsigned char axes = 2;
	unsigned char buttons = 2;
	int version = 0x000800;

# ifndef LINUX_JS_NAME_MAX
#  define LINUX_JS_NAME_MAX	128
# endif

	char name[LINUX_JS_NAME_MAX] = "Unknown";

	struct js_corr *corr;
#endif	/* __linux__ */



	if(jsd == NULL)
	    return(JSBadValue);


	/* Reset values. */
	jsd->axis = NULL;
	jsd->total_axises = 0;

	jsd->button = NULL;
	jsd->total_buttons = 0;

	jsd->device_name = NULL;
	jsd->caliberation_file = NULL;

	jsd->fd = -1;


	/* Parse arguments. */
	for(i = 0; i < argc; i++)
	{
	    if(argv[i] == NULL)
		continue;

	    /* Device. */
	    if(strcasepfx(argv[i], "--d") ||
	       strcasepfx(argv[i], "-d")
	    )
	    {
		i++;
		if(i < argc)
		{
		    free(jsd->device_name);
		    jsd->device_name = StringCopyAlloc(argv[i]);
		}
	    }
	    /* Caliberation file. */
            else if(strcasepfx(argv[i], "--f") ||
                    strcasepfx(argv[i], "-f")
            )
            {
                i++;
                if(i < argc)
                {
                    free(jsd->caliberation_file);
                    jsd->caliberation_file = StringCopyAlloc(argv[i]);
                }
            }
	    /* Blocking mode. */
            else if(strcasepfx(argv[i], "--b") ||
                    strcasepfx(argv[i], "-b")
	    )
	    {
		non_blocking = 0;
	    }
            /* Non blocking mode. */
            else if(strcasepfx(argv[i], "--non_b") ||
                    strcasepfx(argv[i], "--non-b") ||
                    strcasepfx(argv[i], "--nonb") ||
                    strcasepfx(argv[i], "-non_b") ||
                    strcasepfx(argv[i], "-non-b") ||
                    strcasepfx(argv[i], "-nonb")
            )
            {
                non_blocking = 1;
            }
	}


	/* Assume defaults? */
	if(jsd->device_name == NULL)
	    jsd->device_name = StringCopyAlloc(JSDefaultDevice);
	if(jsd->device_name == NULL)
	    return(JSNoBuffers);

        if(jsd->caliberation_file == NULL)
            jsd->caliberation_file = StringCopyAlloc(JSDefaultCaliberation);
        if(jsd->caliberation_file == NULL)
            return(JSNoBuffers);

#ifdef __linux__
	/* Open joystick. */
	jsd->fd = open(jsd->device_name, O_RDONLY);
	if(jsd->fd < 0)
	{
/*
	    fprintf(
		stderr,
		"%s: Cannot open device.\n",
		jsd->device_name
	    );
 */
	    JSClose(jsd);
            return(JSNoAccess);
        }

	/* Fetch device values. */
        ioctl(jsd->fd, JSIOCGVERSION, &version);

        ioctl(jsd->fd, JSIOCGAXES, &axes);	/* Total axises. */
	jsd->total_axises = axes;

        ioctl(jsd->fd, JSIOCGBUTTONS, &buttons);	/* Total buttons. */
	jsd->total_buttons = buttons;

        ioctl(jsd->fd, JSIOCGNAME(LINUX_JS_NAME_MAX), name);
	jsd->name = StringCopyAlloc(name);

	/* Allocate axises. */
	if(jsd->total_axises > 0)
	{
	    jsd->axis = (js_axis_struct **)calloc(
	        jsd->total_axises,
	        sizeof(js_axis_struct *)
	    );
	    if(jsd->axis == NULL)
	    {
	        jsd->total_axises = 0;
	        JSClose(jsd);
	        return(JSNoBuffers);
	    }
	}
	for(i = 0; i < jsd->total_axises; i++)
	{
	    jsd->axis[i] = (js_axis_struct *)calloc(
                1,
                sizeof(js_axis_struct)
            );
            if(jsd->axis == NULL)
            {
                JSClose(jsd);
                return(JSNoBuffers);
            }

	    /* Reset axis values. */
            jsd->axis[i]->cur = JSDefaultCenter;
	    jsd->axis[i]->min = JSDefaultMin;
            jsd->axis[i]->max = JSDefaultMax;
            jsd->axis[i]->cen = JSDefaultCenter;
	    jsd->axis[i]->nz = JSDefaultNullZone;
            jsd->axis[i]->flip = 0;
	    jsd->axis[i]->is_hat = 0;
	}

        /* Allocate buttons. */  
	if(jsd->total_buttons > 0)
	{
            jsd->button = (js_button_struct **)calloc(
                jsd->total_buttons,
                sizeof(js_button_struct *)
            );
            if(jsd->button == NULL)   
            {
                jsd->total_buttons = 0;
                JSClose(jsd);   
                return(JSNoBuffers);
	    }
        }
        for(i = 0; i < jsd->total_buttons; i++)
        {
            jsd->button[i] = (js_button_struct *)calloc(
                1,
                sizeof(js_button_struct)
            );
            if(jsd->button == NULL)
            {
                JSClose(jsd);
                return(JSNoBuffers);
            }

	    /* Reset button values. */
            jsd->button[i]->state = JSButtonStateOff;
        }


	/* Set joystick driver to no correction for all axises. */
	if(jsd->total_axises > 0)
	{
	    corr = (struct js_corr *)calloc(
		jsd->total_axises,
		sizeof(struct js_corr)
	    );
	    if(corr == NULL)
	    {
                JSClose(jsd);  
                return(JSNoBuffers);
            }

            for(i = 0; i < jsd->total_axises; i++)
	    {
                corr[i].type = JS_CORR_NONE;
                corr[i].prec = 0;
	    }

            if(ioctl(jsd->fd, JSIOCSCORR, corr))
                perror("setting correction");

	    free(corr);
	    corr = NULL;
	}


	/* Set to non-blocking? */
	if(non_blocking)
            fcntl(jsd->fd, F_SETFL, O_NONBLOCK);
 

	/* Load caliberation. */
	JSLoadCaliberationLinux(jsd);

#endif	/* __linux__ */


	return(JSSuccess);
}


/*
 *	Updates the information in jsd, returns JSGotEvent if there
 *	was some change or JSNoEvent if there was no change.
 *
 *	jsd needs to be previously initialized by a call to
 *	JSInit().
 */
int JSUpdate(js_data_struct *jsd)
{
	int n;
	int status = JSNoEvent;
#ifdef __linux__
	int keep_handling = 1;
	int bytes_read, cycles;
	struct js_event event;
#endif  /* __linux__ */


        if(jsd == NULL)
            return(status);

	if(jsd->fd < 0)
	    return(status);


#ifdef __linux__
	cycles = 0;
	while(keep_handling &&
              (cycles < 16)
	)
	{
	    /* Get event. */
	    bytes_read = read(
	        jsd->fd,
	        &event,
	        sizeof(struct js_event)
	    );
	    /* No more events to be read? */
	    if(bytes_read != sizeof(struct js_event))
	        return(status);

	    /* Handle by event type. */
            switch(event.type & ~JS_EVENT_INIT)
	    {
	      /* Axis event. */
	      case JS_EVENT_AXIS:
		/* Mark time. */
		jsd->time = event.time;

	        /* Get axis number. */
                n = event.number;

                /* Does axis exist? */
                if(JSIsAxisAllocated(jsd, n))
                {
                    /* Set new axis value. */
                    jsd->axis[n]->cur = event.value;
                }
/*
printf("Axis %i: %i\n", n, event.value);
 */
	        status = JSGotEvent;	/* Mark that we got event. */

	        break;

	      /* Button event. */
              case JS_EVENT_BUTTON:
                /* Mark time. */
                jsd->time = event.time;

	        /* Get button number. */
	        n = event.number;

	        /* Does button exist? */
                if(JSIsButtonAllocated(jsd, n))
	        {
		    /* Set new button state. */
                    jsd->button[n]->state = ((event.value) ?
		        JSButtonStateOn : JSButtonStateOff
		    );
	        }

	        status = JSGotEvent;	/* Mark that we got event. */

	        break;

	      /* Other event. */
	      default:
		keep_handling = 0;	/* Stop handling events. */
		break;
	    }

	    cycles++;
	}
#endif	/* __linux__ */


        return(status);
}

/*
 *	Closes the joystick and deallocates all resources.
 */
void JSClose(js_data_struct *jsd)
{
	int i;


	if(jsd == NULL)
	    return;


	/* Close descriptor to joystick. */
	if(jsd->fd > -1)
	{
	    close(jsd->fd);
	    jsd->fd = -1;
	}

	/* Free name. */
	free(jsd->name);
	jsd->name = NULL;


	/* Free all axises. */
	for(i = 0; i < jsd->total_axises; i++)
	    free(jsd->axis[i]);
	free(jsd->axis);
	jsd->axis = NULL;

	jsd->total_axises = 0;

	/* Free all buttons. */
	for(i = 0; i < jsd->total_buttons; i++)
            free(jsd->button[i]);
        free(jsd->button);
        jsd->button = NULL;
        jsd->total_buttons = 0;


	/* Free device name. */
	free(jsd->device_name);
	jsd->device_name = NULL;

	/* Free caliberation file name. */
	free(jsd->caliberation_file);
	jsd->caliberation_file = NULL;


	/* Reset time. */
	jsd->time = 0;


	return;
}


/*
 *	Dummy function, for ANSI C compliancy.
 */
int JSMain(int argc, char *argv[])
{
	int x;

	x = 0;

	return(x);
}
