#!/usr/bin/perl -w
# Copyright (C) Steve Haslam 1999. This is free software.
# Distributable under the terms of the GNU General Public License, version 2
# See the file 'COPYING' for license text.

package Debian::ThemeConverters;
use strict;
use POSIX;
use Exporter;
use Digest::MD5;
use Carp;
use Getopt::Std;
use Cwd;
use vars qw[$VERSION @ISA @EXPORT_OK];

$VERSION = "0.00";
@ISA = qw[Exporter];
@EXPORT_OK = qw[&printl &printv &printi &printw &printe &md5sumfilehex &compare_file &debwalk
	       &direntries &md5_files &fixperms &wrapsystem &main &wchdir];

my $verbose = exists $ENV{DEBUG_THEMECONVERTERS};

sub md5sumfilehex {
  my $filename = shift;
  my $md5 = new Digest::MD5;
  open(FILE, $filename) || die "Can't open $filename: $!\n";
  $md5->addfile(\*FILE);
  close FILE;
  return $md5->hexdigest;
}

sub printl {
  my $level = shift;
  my @lines = split(/\n/, join('', @_));
  my $prog = $0;
  $prog =~ s@.*/@@;
  foreach (@lines) {
    print "$prog:$level: $_\n";
  }
}

sub printv { printl('V', @_) if ($verbose) }
sub printi { printl('I', @_) }
sub printw { printl('W', @_) }
sub printe { printl('E', @_) }

sub compare_file {
  my $pathname = shift;
  my $chk_md5sum = shift;
  my $chk_size = shift;

  if (defined($chk_size)) {
    if (-s $pathname != $chk_size) {
      printe "$pathname: file sizes differ";
      return undef;
    }
  }

  if (defined($chk_md5sum)) {
    if (md5sumfilehex($pathname) ne $chk_md5sum) {
      printe "$pathname: checksums differ";
      return undef;
    }
  }

  1;
}

sub debwalk {
  my $dir = shift;
  opendir(DIR, $dir) || die "Can't open directory $dir: $!\n";
  my @files = readdir DIR;
  my @filelist;
  closedir(DIR);
  foreach (@files) {
    next if ($_ eq '.' || $_ eq '..');
    push(@filelist, "$dir/$_");
    if (-d "$dir/$_") {
      push(@filelist, debwalk("$dir/$_"));
    }
  }
  return @filelist;
}

sub direntries {
  my $dir = shift;
  opendir(DIR, $dir) || die "Can't open directory $dir: $!\n";
  my @entries = readdir DIR;
  closedir(DIR);
  @entries;
}

sub md5_files {
  my $topdir = shift;
  map { -f $_ ? (&md5sumfilehex($_)."  $_") : () } (debwalk $topdir);
}

sub fixperms {
  my $topdir = shift;
  foreach (debwalk $topdir) {
    chown(0, 0, $_) || die "$_: chown failed: $!\n";
    my $mode = (-f $_) ? 0644 : 0755;
    printv("chmod($mode, $_)\n");
    chmod($mode, $_) || die "$_: chmod $mode failed: $!\n";
  }
}

sub wrapsystem {
  my @systemargs = @_;
  my @prettyargs = map { '['.$_.']' } @systemargs;
  my $exitcode;
  if ($systemargs[0] !~ m@^/@) {
    croak "\"$systemargs[0]\" is not absolute- not allowed any more";
  }
 FORK:
  {
    my $pid = fork;
    if (!defined $pid) {
      if ($! == EAGAIN) {
	warn "fork failed: sleeping for 5 seconds";
	sleep 5;
	redo FORK;
      }
      else {
	die "fork failed: $!\n";
      }
    }
    elsif ($pid == 0) {
      # Child process
      printv "$$: @prettyargs";
      exec @systemargs;
      die "exec failed: $!\n";
    }
    else {
      die "waitpid didn't return $pid\n" if (waitpid($pid, 0) != $pid);
      $exitcode = $?;
    }
  }
  if ($exitcode) {
    my $vexitcode = '['.($exitcode&0xff).']'.($exitcode >> 8);
    printv "Exit code=$vexitcode";
  }
  return $exitcode;
}

sub readlsminfo {
  my $dir = shift;
  # Look for .lsm file
  my @lsmfiles = grep { /\.lsm$/ } direntries($dir);
  my %results;
  if (@lsmfiles == 0) {
    printv "No .lsm file. Urrgh.";
  }
  elsif (@lsmfiles > 1) {
    warn "Multiple .lsm files. Confused.";
  }
  else {
    open(LSM, "$dir/$lsmfiles[0]") || die "Can't open $dir/$lsmfiles[0]: $!\n";
    printv "Reading $lsmfiles[0]";
    while (<LSM>) {
      if (/^author\s*:\s*(.*)/i) {
	printv "Found \"Author\" field: $1";
	$results{'author'} = $1;
      }
      elsif (/^description\s*:\s*(.*)/i) {
	printv "Found \"Description\" field: $1";
	$results{'description'} = $1;
      }
    }
    close LSM;
  }
  %results;
}

sub spliceinfo {
  my $controlinfo = shift;
  my $name = shift;
  if (exists $$controlinfo{$name}) {
    my $data = $$controlinfo{$name};
    delete $$controlinfo{$name};
    return $data;
  }
  printv "$name not found for Debian header\n";
  undef;
}

sub makedeb {
  my $dpkgfile = shift;
  my $tmpdir = shift;
  my $controlinfo = shift;
  my @md5sums = md5_files($tmpdir);

  mkdir("$tmpdir/DEBIAN", 0755) || die "Unable to create directory $tmpdir/DEBIAN: $!\n";

  open(MD5SUMS, ">$tmpdir/DEBIAN/md5sums") || die "Unable to open $tmpdir/DEBIAN/md5sums: $!\n";
  foreach (@md5sums) {
    s@  $tmpdir/@  @;
    print MD5SUMS "$_\n";
  }
  close MD5SUMS;

  my $dpkgname = spliceinfo($controlinfo, 'Name');
  my $dpkgversion = spliceinfo($controlinfo, 'Version');
  my $dpkgmaintainer = spliceinfo($controlinfo, 'Maintainer');
  my $dpkgdescription = spliceinfo($controlinfo, 'Description');
  my $dpkgarch = spliceinfo($controlinfo, 'Architecture') || 'all';
  my $dpkgsection = spliceinfo($controlinfo, 'Section') || 'x11';
  my $dpkgpriority = spliceinfo($controlinfo, 'Priority') || 'extra';
  my $longdescription = spliceinfo($controlinfo, 'LongDescription');
  my $otherheaders = join("\n", map { "$_: $$controlinfo{$_}" } keys %$controlinfo);

  open(CONTROL, ">$tmpdir/DEBIAN/control") || die "Unable to open $tmpdir/DEBIAN?control: $!\n";
  print CONTROL <<EOF;
Package: $dpkgname
Version: $dpkgversion
Maintainer: $dpkgmaintainer
Section: $dpkgsection
Priority: $dpkgpriority
Architecture: $dpkgarch
Description: $dpkgdescription
 $longdescription
EOF
  close CONTROL;

  fixperms($tmpdir);

  # Create .deb
  printv "Building in $dpkgfile";
  if (wrapsystem("/usr/bin/dpkg", "--build", $tmpdir, $dpkgfile) != 0) {
    die "Failed to build Debian package\n";
  }
}

sub convert {
  my $convertsub = pop;
  my $cwd = getcwd;
  my @debfiles;
  my $themefile;

  foreach (@_) {
    $themefile = $_;
    if ($themefile !~ m@^/@) {
      $themefile = "$cwd/$themefile";
    }
    if (-f $themefile) {
      printv "Theme file: $themefile";
      my $debfile = &$convertsub($themefile, $cwd);
      push(@debfiles, $debfile) if (defined($debfile))
    }
  }

  wchdir($cwd);
  @debfiles;
}

sub wchdir {
  my $dir = shift;
  printv "Entering directory \"$dir\"";
  chdir $dir || croak "Unable to chdir to $dir: $!";
}

sub findonpath {
  my $program = shift;
  if ($program =~ m@^/@) {
    return $program;
  }
  foreach (@_) {
    if (-x "$_/$program") {
      return "$_/$program";
    }
  }
  die "Unable to find $program in @_\n";
}

sub main {
  my $convertsub = shift;
  my %opts;

  getopts('pir:R:', \%opts);

  $|=1;
  
  my $iamroot = ($> == 0);

  my @debfiles;

  if (!$iamroot && $opts{'r'}) {
    # re-execute ourselves as root to do the conversion
    my $fakeroot = findonpath($opts{'r'}, qw[/usr/local/bin /usr/bin /bin]);
    # Scan output of child process for .deb files written
    my $pid = open(CHILD, "-|");
    if (!defined $pid) {
      die "fork failed: $!\n";
    }
    if ($pid == 0) {
      # Child process
      my @args = ($fakeroot, $0, '-p', @ARGV);
      printv "$$: ".join(' ', map{"[$_]"} @args);
      exec @args;
      die;
    }
    while (<CHILD>) {
      if (/qw\[(.*)\]/) {
	@debfiles=split(/ +/, $1);
      }
      else {
	print;
      }
    }
    waitpid($pid, 0) || die "waitpid($pid,0) failed: $!\n";
    printv "$pid made [@debfiles]\n";
  }
  else {
    @debfiles = convert(@ARGV, $convertsub);
    if ($opts{p}) {
      printi "qw[@debfiles]";
    }
  }

  if ($opts{'i'} && @debfiles) {
    if (!$iamroot && ($opts{'R'} || $opts{'r'})) {
      my $sudo = findonpath(($opts{'R'} || $opts{'r'}), qw[/usr/local/bin /usr/bin /bin]);
      wrapsystem($sudo, '/usr/bin/dpkg', '-i', @debfiles);
    }
    else {
      wrapsystem('/usr/bin/dpkg', '-i', @debfiles);
    }
  }
}

1;
__END__

=head1 NAME

Debian::ThemeConvertors - Perl library for the theme convertor scripts

=head1 SYNOPSIS

    use Debian::ThemeConvertors;

    Debian::ThemeConvertors::main \&theme_specific_sub;

    sub theme_specific_sub {
      my($themepkgfilename, $destdir) = @_;
      ...
      Debian::ThemeConvertors::makedeb($dpkgfile, $tmpdir,
                                       { Name => ...
				         Version => ...
					 Maintainer => ...
				       }
    }

=head1 DESCRIPTION

This library contains the guts of the theme convertor scripts. The
script calls F<Debian::ThemeConvertors::main> with a reference to a
subroutine for the specific script. F<main()> parses the command line,
and calls the given sub for each theme mentioned on the command line,
passing in the theme filename and the destination directory.

The script's sub is then responsible for producing a directory
structure for the resultant Debian package, and determining what the
binary package's control fields should be. These are then sent to
F<Debian::ThemeConvertors::makedeb> and a .deb package is
made. F<main()> will then install the packages if the user so
requested.

Various functions are provided in the library to help with conversion
and provide consistency. The F<printv()>, F<printi()>, F<printw()> and
F<printe()> are for printing verbose, informational, warning and error
messages respectively. F<wchdir()> will change working directory, and
F<croak> if this fails. F<direntries()> will return a list of entries
in a directory. F<debwalk()> will recursively walk a directory tree,
returning a list of entries found.

=head1 CAVEATS

This library was made when I ripped out the common elements of
B<gtkthemetodeb> and B<wmakerthemetodeb> to stop version skew. It is
not well-designed, and certainly not well-documented. Sorry.

=head1 EXAMPLES

See the wmakerthemetodeb and gtkthemetodeb scripts.

=head1 SEE ALSO

wmakerthemetodeb(1), gtkthemetodeb(1)

=head1 AUTHOR

Steve Haslam <araqnid@debian.org>

=cut
