// bbpager.cc for bbpager - a pager for Blackbox.
//
//  Copyright (c) 1998-1999 by John Kennis, jkennis@chello.nl
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// (See the included file COPYING / GPL-2.0)
//

#include "bbpager.hh"
#include "version.h"

ToolWindow::ToolWindow(int argc,char **argv, struct CMDOPTIONS *options) :
  Basewindow(argc,argv,options) {
  XrmInitialize();

  resource = new Resource(this);
  wminterface = new WMInterface(this);
  windowList = new LinkedList<WindowList>;
  desktopList = new LinkedList<DesktopList>;

  current_desktop_nr=0;
  kwm_init=False;
  number_of_desktops=1;
  row_last=column_last=1;
  MakeWindow(False);
  wminterface->moduleInit();
  EventLoop();
}

ToolWindow::~ToolWindow() {
  XUnmapWindow(getDisplay(),framewin);

  /* destroy pixmaps */
  if (pixmap.frame) getImageControl()->removeImage(pixmap.frame);
  if (pixmap.desktop) getImageControl()->removeImage(pixmap.desktop);
  if (pixmap.window) getImageControl()->removeImage(pixmap.window);
  if (pixmap.focusedWindow) 
    getImageControl()->removeImage(pixmap.focusedWindow);
  if (pixmap.focusedDesktop) 
    getImageControl()->removeImage(pixmap.focusedDesktop);

  /* destroy windows */
  XDestroyWindow(getDisplay(),framewin);
  /* destroy windows */
  delete windowList;
  delete desktopList;
}

void ToolWindow::moveWinToDesktop(Window win,int desktop_nr) {

  LinkedListIterator<WindowList> win_it(windowList);
  for (; win_it.current(); win_it++)
    if ((win_it.current()->win) == win)
      break;
  if (win_it.current()) {
    if ((desktop_nr!=win_it.current()->desktop_nr) 
                    & (!win_it.current()->sticky)) {
      LinkedListIterator<DesktopList> desktop_it(desktopList);
      desktop_it.set(desktop_nr-1);

      if (desktop_it.current()) {
        XUnmapWindow(getDisplay(),win_it.current()->pager_win);
        XReparentWindow(getDisplay(),win_it.current()->pager_win,
                        desktop_it.current()->win,win_it.current()->pager_x,
                        win_it.current()->pager_y);
        XMapWindow(getDisplay(),win_it.current()->pager_win);
        win_it.current()->desktop_nr=desktop_nr;
      }
    }
  }
}

int ToolWindow::getWindowGeometry(struct WindowList  *winlist) {
  unsigned int border_width,depth;
  Window root_return,child_return;
  int x_return,y_return;
  int status;

  status = XGetGeometry(getDisplay(),winlist->win,&root_return,&x_return,&y_return,
                        &winlist->width,&winlist->height,&border_width,&depth);
  if (status) {
    XTranslateCoordinates(getDisplay(),winlist->win,root_return,x_return,y_return,
                          &winlist->x_position,&winlist->y_position,
                          &child_return);
    return 1;
  }
  return 0;
}


void ToolWindow::Reconfigure(void) {
  /* destroy pixmaps */
  getImageControl()->removeImage(pixmap.frame);
  getImageControl()->removeImage(pixmap.desktop);
  getImageControl()->removeImage(pixmap.window);
  if (pixmap.focusedWindow) 
    getImageControl()->removeImage(pixmap.focusedWindow);
  if (pixmap.focusedDesktop) 
    getImageControl()->removeImage(pixmap.focusedDesktop);

  resource->Reload();

  MakeWindow(True);
  
  LinkedListIterator<DesktopList> desktop_it(desktopList);
  LinkedListIterator<WindowList> win_it(windowList);

  desktop_nr=0;
  
  for (; desktop_it.current(); desktop_it++)
    addDesktopWindow(desktop_it.current(),True);
 
  for (; win_it.current(); win_it++)
    addFrameWindow(win_it.current(),True);

  XClearWindow(getDisplay(), framewin);
}

void ToolWindow::raiseWindow(Window win) {
  LinkedListIterator<WindowList> win_it(windowList);
  for (; win_it.current(); win_it++)
    if ((win_it.current()->win) == win)
      break;

  if (win_it.current()) {
    WindowList *tmp = win_it.current();
    XRaiseWindow(getDisplay(),tmp->pager_win);
  }
}

void ToolWindow::lowerWindow(Window win) {
  LinkedListIterator<WindowList> win_it(windowList);
  for (; win_it.current(); win_it++)
    if ((win_it.current()->win) == win)
      break;

  if (win_it.current()) {
    WindowList *tmp = win_it.current();
    XLowerWindow(getDisplay(),tmp->pager_win);
  }
}

void ToolWindow::focusWindow(Window win) {
  WindowList *tmp;


  if (focuswin) {
    LinkedListIterator<WindowList> old_it(windowList);
    for (; old_it.current(); old_it++) {
      if (old_it.current()->win == focuswin) {
        if (resource->getFocusStyle()==border)
          XSetWindowBorder(getDisplay(),old_it.current()->pager_win,
                           resource->pagerwin.inactiveColor.getPixel());
         else
          XSetWindowBackgroundPixmap(getDisplay(), 
                                     old_it.current()->pager_win,
                                     pixmap.window);

         old_it.current()->focused=False;
      
         XClearWindow(getDisplay(),old_it.current()->pager_win);
      }
    }
  }
 
  LinkedListIterator<WindowList> win_it(windowList);
  for (; win_it.current(); win_it++) {
    if ((win_it.current()->win) == win) {
      tmp = win_it.current();
  
      
      if (resource->getFocusStyle()==border)
        XSetWindowBorder(getDisplay(),tmp->pager_win,resource->pagerwin.
                         activeColor.getPixel());
      else 
        XSetWindowBackgroundPixmap(getDisplay(), tmp->pager_win,
                                   pixmap.focusedWindow);
    
      XClearWindow(getDisplay(),tmp->pager_win);
      tmp->focused=True;
      focuswin=tmp->win;
    }
  }
}

void ToolWindow::desktopChange(int desktop_nr) {
  Window win;

  if (resource->getDesktopFocusStyle()!=none) {
    LinkedListIterator<DesktopList> win_it(desktopList);
    if (current_desktop_nr) {
      win_it.set(current_desktop_nr-1);
      win = win_it.current()->win;
      if (resource->getDesktopFocusStyle()==border)
        XSetWindowBorderWidth(getDisplay(),win,0);
      else
        XSetWindowBackgroundPixmap(getDisplay(), win, pixmap.desktop);
      XClearWindow(getDisplay(),win);
    }
    win_it.set(desktop_nr-1);
    win = win_it.current()->win;
    if (resource->getDesktopFocusStyle()==border)
      XSetWindowBorderWidth(getDisplay(),win,1);
    else
      XSetWindowBackgroundPixmap(getDisplay(), win, pixmap.focusedDesktop);
     XClearWindow(getDisplay(),win);
  }
    current_desktop_nr=desktop_nr;
}

int ToolWindow::winOnDesktop(Window win) {
  LinkedListIterator<WindowList> win_it(windowList);
  for (; win_it.current(); win_it++)
    if ((win_it.current()->pager_win) == win)
      break;

  if (win_it.current())
    return(win_it.current()->desktop_nr);

  return(0);
}

void ToolWindow::removeWindow(Window win) {
  LinkedListIterator<WindowList> win_it(windowList);
  for (; win_it.current(); win_it++)
    if ((win_it.current()->win) == win)
      break;

  if (win_it.current()) {
    WindowList *tmp = win_it.current();

    if (tmp->sticky) wminterface->removeSticky(tmp->win,tmp->desktop_nr);
    windowList->remove(tmp);
    if (tmp->focused) focuswin=0;
    XDestroyWindow(getDisplay(),tmp->pager_win);
    delete tmp;
  }
}

void ToolWindow::removeDesktopWindow()
{
   /* delete last */
  DesktopList *tmp = new DesktopList;
  tmp= desktopList->last();
  LinkedListIterator<WindowList> win_it(windowList);
  for (; win_it.current(); win_it++)
    if ((win_it.current()->sticky && 
      win_it.current()->desktop_nr==tmp->desktop_nr))
      break;
              
  if (win_it.current()) {
    WindowList *sticky=win_it.current();
    windowList->remove(sticky);
    XDestroyWindow(getDisplay(),sticky->pager_win);
    delete sticky;
  }
  
  desktopList->remove(tmp);
  XUnmapWindow(getDisplay(),tmp->win);
  XDestroyWindow(getDisplay(),tmp->win);
  delete tmp;
  if (resource->position.vertical) {
   if (number_of_desktops<resource->columns)
    frame.width=(unsigned int)(resource->desktopSize.width + 
                resource->frame.bevelWidth)*
                (number_of_desktops%resource->columns) + 
                resource->frame.bevelWidth;
   else
    frame.width=(unsigned int)(resource->desktopSize.width + 
                 resource->frame.bevelWidth)*
                 resource->columns + resource->frame.bevelWidth;

    frame.height=(unsigned int)(((number_of_desktops-1)/
                  resource->columns)+1)*
                  (resource->desktopSize.height + 
                   resource->frame.bevelWidth) + 
                   resource->frame.bevelWidth;
  } else {
    frame.width=(unsigned int)((number_of_desktops-1)/resource->rows+1) *
                (resource->desktopSize.width + 
                 resource->frame.bevelWidth) + 
                 resource->frame.bevelWidth;
   
    if (number_of_desktops<resource->rows)
      frame.height=(unsigned int)(resource->desktopSize.height+
                  resource->frame.bevelWidth)*
                  (number_of_desktops%resource->rows) + 
                  resource->frame.bevelWidth;
    else
      frame.height=(unsigned int)(resource->desktopSize.height+
                  resource->frame.bevelWidth)*
                  resource->rows + resource->frame.bevelWidth;
  }

  if (resource->position.mask & XNegative) {
    frame.x = getCurrentScreenInfo()->getWidth() +
              resource->position.x - frame.width;
  }
  if (resource->position.mask & YNegative) {
    frame.y = getCurrentScreenInfo()->getHeight() +
              resource->position.y - frame.height;
  }
              
  if (!withdrawn)
    XMoveResizeWindow(getDisplay(), framewin,frame.x,frame.y,
                      frame.width,frame.height);
  else
    XResizeWindow(getDisplay(),framewin,frame.width,frame.height);

  if (!shape)
    XSetWindowBackgroundPixmap(getDisplay(), framewin, pixmap.frame);

  XClearWindow(getDisplay(),framewin);
  desktop_nr--;
}

void ToolWindow::addDesktopWindow(struct DesktopList *tmp,bool reconfigure) {
  XSetWindowAttributes attrib;
  int row,column;
  unsigned long create_mask = CWBackPixmap|CWCursor|CWEventMask|CWBorderPixel;

  attrib.background_pixmap = ParentRelative;

  attrib.border_pixel=resource->desktopwin.activeColor.getPixel();
  attrib.cursor = getSessionCursor();
#ifdef NEW_BUTTON_STYLE
  attrib.event_mask = ButtonPressMask | ButtonReleaseMask | ExposureMask |
                      FocusChangeMask | StructureNotifyMask|
                      SubstructureRedirectMask | Button2MotionMask;

#else
  attrib.event_mask = ButtonPressMask | ButtonReleaseMask | ExposureMask |
                      FocusChangeMask | StructureNotifyMask|
                      SubstructureRedirectMask | Button1MotionMask;
#endif


  tmp->width=resource->desktopSize.width;
  tmp->height=resource->desktopSize.height;
  
  if (resource->position.vertical) {
    row=desktop_nr/resource->columns;
    column=desktop_nr%resource->columns +1;

    tmp->x=(resource->frame.bevelWidth + resource->desktopSize.width)* 
            (column-1) + resource->frame.bevelWidth;
    tmp->y = resource->frame.bevelWidth +
             ((row) * (resource->desktopSize.height +
              resource->frame.bevelWidth));

  if (number_of_desktops<resource->columns)
    frame.width=(unsigned int)(resource->desktopSize.width + 
                resource->frame.bevelWidth)*
                (number_of_desktops%resource->columns) + 
                resource->frame.bevelWidth;
   else
    frame.width=(unsigned int)(resource->desktopSize.width + 
                resource->frame.bevelWidth)*
                resource->columns + resource->frame.bevelWidth;

   frame.height=(unsigned int)(((number_of_desktops-1)/resource->columns+1)*
                   (resource->desktopSize.height + 
                   resource->frame.bevelWidth) + resource->frame.bevelWidth);
  }
  else {
    column=desktop_nr/resource->rows;
    row=desktop_nr%resource->rows +1;

    tmp->x=(resource->frame.bevelWidth + resource->desktopSize.width)* 
            column + resource->frame.bevelWidth;

    tmp->y = resource->frame.bevelWidth +
             ((row-1) * (resource->desktopSize.height +
              resource->frame.bevelWidth));

    frame.width=(unsigned int)((number_of_desktops-1)/resource->rows+1) *
                 (resource->desktopSize.width + 
                  resource->frame.bevelWidth) + resource->frame.bevelWidth;
   
    if (number_of_desktops<resource->rows)
       frame.height=(unsigned int)(resource->desktopSize.height+
                  resource->frame.bevelWidth)*
                  (number_of_desktops%resource->rows) + 
                  resource->frame.bevelWidth;
    else
      frame.height=(unsigned int)(resource->desktopSize.height+
                  resource->frame.bevelWidth)*
                  resource->rows + resource->frame.bevelWidth;
  }

  if (resource->position.mask & XNegative) {
    frame.x = getCurrentScreenInfo()->getWidth() + 
              resource->position.x - frame.width;
  }
  if (resource->position.mask & YNegative) {
    frame.y = getCurrentScreenInfo()->getHeight() + 
              resource->position.y - frame.height;
  }
  if (!withdrawn)
    XMoveResizeWindow(getDisplay(), framewin,frame.x,
                    frame.y,frame.width,frame.height);
  else
    XResizeWindow(getDisplay(),framewin,frame.width,frame.height);

  if (!reconfigure)
    tmp->win = XCreateWindow(getDisplay(), framewin,tmp->x,tmp->y,tmp->width,
                           tmp->height,0, getCurrentScreenInfo()->getDepth(),
                           InputOutput, getCurrentScreenInfo()->getVisual(),
                           create_mask,&attrib);
  else
      XMoveResizeWindow(getDisplay(),tmp->win,tmp->x,tmp->y,tmp->width,tmp->height);

  getImageControl()->removeImage(pixmap.frame);
  pixmap.frame = getImageControl()->renderImage(frame.width, frame.height,
                 &resource->frame.texture);

  if (!shape)
    XSetWindowBackgroundPixmap(getDisplay(), framewin, pixmap.frame);

  XSetWindowBackgroundPixmap(getDisplay(), tmp->win, pixmap.desktop);
  XMapSubwindows(getDisplay(), framewin);
  XClearWindow(getDisplay(),framewin);
  XClearWindow(getDisplay(),tmp->win );
  desktop_nr++;
  tmp->desktop_nr=desktop_nr;
}

void ToolWindow::addFrameWindow(struct WindowList *window,bool reconfigure) {
  Window desktopWin;
  XSetWindowAttributes attrib;
  unsigned long create_mask = CWBackPixmap|CWCursor|CWBorderPixel;
  double xdiv,ydiv;

  LinkedListIterator<DesktopList> desktop_it(desktopList);
  desktop_it.set(window->desktop_nr-1);
  desktopWin = desktop_it.current()->win;

  attrib.background_pixmap = ParentRelative;
  attrib.border_pixel=resource->pagerwin.inactiveColor.getPixel();
  attrib.cursor = getSessionCursor();

  xdiv=(double)resource->desktopSize.width/
          getCurrentScreenInfo()->getWidth();
  ydiv=(double)resource->desktopSize.height/
          getCurrentScreenInfo()->getHeight();
  window->pager_x=(int)(window->x_position*xdiv);
  window->pager_y=(int)(window->y_position*ydiv);
  window->pager_width=(unsigned int)(window->width*xdiv);
  window->pager_height=(unsigned int)(window->height*ydiv);
  if (window->pager_width==0)   window->pager_width=1;
  if (window->pager_height==0) window->pager_height=1;

  
  if (!reconfigure)
    window->pager_win = XCreateWindow(getDisplay(), desktopWin,
                                    window->pager_x,window->pager_y,
                                    window->pager_width,window->pager_height,
                                    1, getCurrentScreenInfo()->getDepth(), 
                                    InputOutput,
                                    getCurrentScreenInfo()->getVisual(), 
                                    create_mask,&attrib);
  else
    XMoveResizeWindow(getDisplay(),window->pager_win,window->pager_x,window->pager_y,
                      window->pager_width,window->pager_height);
  if (!window->focused)
     XSetWindowBackgroundPixmap(getDisplay(), window->pager_win,
                             pixmap.window);
  else
     XSetWindowBackgroundPixmap(getDisplay(), window->pager_win,
                             pixmap.focusedWindow);
  if (!window->icon)
    XMapWindow(getDisplay(),window->pager_win);
  XClearWindow(getDisplay(),window->pager_win );
}

void ToolWindow::MakeWindow(bool reconfigure) {
  XSetWindowAttributes attrib;
  XWMHints wmhints;
  XClassHint classhints;


  unsigned long create_mask = CWBackPixmap|CWBorderPixel|
                              CWOverrideRedirect |CWCursor|CWEventMask;

  if (resource->position.vertical) {

    if (number_of_desktops<resource->columns)
      frame.width=(unsigned int)(resource->desktopSize.width + 
                  resource->frame.bevelWidth)*
                  (number_of_desktops%resource->columns) + 
                  resource->frame.bevelWidth;
     else
      frame.width=(unsigned int)(resource->desktopSize.width + 
                  resource->frame.bevelWidth)*
                  resource->columns + resource->frame.bevelWidth;

     frame.height=(unsigned int)(((number_of_desktops-1)/resource->columns+1)*
                     (resource->desktopSize.height + 
                     resource->frame.bevelWidth) + resource->frame.bevelWidth);

  }
  else {
    frame.width=(unsigned int)((number_of_desktops-1)/resource->rows+1) *
                (resource->desktopSize.width + 
                 resource->frame.bevelWidth) + resource->frame.bevelWidth;
   
    if (number_of_desktops<resource->rows)
       frame.height=(unsigned int)(resource->desktopSize.height+
                  resource->frame.bevelWidth)*
                  (number_of_desktops%resource->rows) + 
                  resource->frame.bevelWidth;
    else
      frame.height=(unsigned int)(resource->desktopSize.height+
                  resource->frame.bevelWidth)*
                  resource->rows + resource->frame.bevelWidth;

  }
  frame.x=resource->position.x;
  frame.y=resource->position.y;
  if (resource->position.mask & XNegative) {
    frame.x = getCurrentScreenInfo()->getWidth() + 
              resource->position.x - frame.width;
  }
  if (resource->position.mask & YNegative) {
    frame.y = getCurrentScreenInfo()->getHeight() +
              resource->position.y - frame.height;
  }

  if (withdrawn) {
    attrib.override_redirect = False;
    wmhints.initial_state = WithdrawnState;
  } else if (decorated) {
  	attrib.override_redirect = False;
	 	wmhints.initial_state = NormalState;
  }
  else {
    attrib.override_redirect = True;
    wmhints.initial_state = NormalState;
  }

  attrib.background_pixmap = ParentRelative;
  attrib.border_pixel=resource->desktopwin.activeColor.getPixel();


  pixmap.frame = getImageControl()->renderImage(frame.width, frame.height,
                 &resource->frame.texture);

  pixmap.window =	getImageControl()->renderImage(resource->desktopSize.width,
                  resource->desktopSize.height,
                  &resource->pagerwin.texture);

  if (resource->getFocusStyle()==texture)
    pixmap.focusedWindow =	
          getImageControl()->renderImage(resource->desktopSize.width,
                  resource->desktopSize.height,
                  &resource->pagerwin.focusedTexture);

  
  pixmap.desktop =	getImageControl()->renderImage(resource->desktopSize.width,
                   resource->desktopSize.height,
                   &resource->desktopwin.texture);

  if (resource->getDesktopFocusStyle()==texture)
    pixmap.focusedDesktop =	
          getImageControl()->renderImage(resource->desktopSize.width,
                  resource->desktopSize.height,
                  &resource->desktopwin.focusedTexture);

  
  attrib.cursor = getSessionCursor();
  attrib.event_mask = ButtonPressMask | ButtonReleaseMask | ExposureMask |
                      FocusChangeMask | StructureNotifyMask|
                      SubstructureRedirectMask;

  if (!reconfigure) {
    framewin = XCreateWindow(getDisplay(), 
                             getCurrentScreenInfo()->getRootWindow(), frame.x,
                             frame.y, frame.width,
                             frame.height, 0, 
                             getCurrentScreenInfo()->getDepth(),
                             InputOutput,
                             getCurrentScreenInfo()->getVisual(), 
                             create_mask, &attrib);
  } else if (!withdrawn) {
    XMoveResizeWindow(getDisplay(), framewin, resource->position.x,
                      resource->position.y, frame.width,
                      frame.height);

  } else {
    XResizeWindow(getDisplay(),framewin,frame.width,frame.height);
  }

  wmhints.flags = StateHint;//IconWindowHint | StateHint;
  wmhints.icon_window = framewin;

  classhints.res_name = "bbpager";
  classhints.res_class = "bbtools";
 
  XSetClassHint(getDisplay(),framewin,&classhints);
  XSetWMHints(getDisplay(),framewin,&wmhints);
  XSetStandardProperties(getDisplay(),framewin,BBTOOL,
                  			 BBTOOL, None,getArgv(),getArgc(),NULL);
  XSetWMProtocols (getDisplay(), framewin,&wm_delete_window, 1);

  
  if (!shape) {
    XSetWindowBackgroundPixmap(getDisplay(), framewin, pixmap.frame);
  }

  if (!withdrawn && resource->report.auto_raise) {
    XRaiseWindow(getDisplay(),framewin);
    lower=False;
  }
  else lower=True;

}

void ToolWindow::CheckConfig()
{
	struct stat file_status;

	if (stat(config_filename,&file_status)!=0)
	{
		fprintf(stderr,"Could not open config file %s\n",
        		resource->style.conf_filename);
	}
   	else if (file_status.st_mtime != resource->style.mtime)
	{
		resource->style.mtime=file_status.st_mtime;
		Reconfigure();
	}
}

void ToolWindow::EventLoop(void) {
  int xfd = ConnectionNumber(getDisplay());
  time_t lastTime = time(NULL);
  bool shutdown=False;
  bool redraw=False;
  static bool moved;
  static Window grabbedWindow;
  static Window realWindow;
  static Window pagerWindow;
  static WindowList *moveWindow;
  static int grabbed_x,grabbed_y,move_x,move_y;

  while (! shutdown) {
    if (XPending(getDisplay())) {
      XEvent Event;
      XNextEvent(getDisplay(), &Event);
      /* process events */
      switch (Event.type) {
      case ClientMessage:
        {
          if ((unsigned)Event.xclient.data.l[0]==wm_delete_window) return;
          if (Event.xclient.message_type==wminterface->getKWMModuleInit()) {
            int i;
            number_of_desktops=wminterface->getNumberOfDesktops();
            desktop_nr=0;
            for (i=0;i<number_of_desktops;i++) {
              DesktopList *tmp = new DesktopList;
              addDesktopWindow(tmp,False);
              desktopList->insert(tmp);
            }
            kwm_init=True;
            XClearWindow(getDisplay(), framewin);
            XMapWindow(getDisplay(), framewin);
            XMapSubwindows(getDisplay(), framewin);

            int current_desktop=wminterface->getCurrentDesktop();
            desktopChange(current_desktop);
          }
          else if (Event.xclient.message_type==
                  wminterface->getKWMModuleWinAdd()) {
            WindowList *tmp = new WindowList;
            tmp->win= (Window) Event.xclient.data.l[0];
            int status=getWindowGeometry(tmp);
            if (status) {
              if (wminterface->isIcon(tmp->win)==1) tmp->icon=True;
              else tmp->icon=False;
              tmp->sticky=False;
              tmp->focused=False;
              tmp->desktop_nr=wminterface->getWinOnDesktop(tmp->win);
              if (tmp->desktop_nr>0) {
                addFrameWindow(tmp,False);
                windowList->insert(tmp);
                if (wminterface->isSticky(tmp->win)==1)
                  wminterface->addSticky(tmp);
              }
              else delete tmp;
            } else {
              delete tmp;
            }
          }
          else if (Event.xclient.message_type==
              wminterface->getKWMModuleWinRemove()) {
            Window win;
            win= Event.xclient.data.l[0];
            removeWindow(win);
          }
          else if (Event.xclient.message_type==
                  wminterface->getKWMModuleWinChange()) {
            Window win;
            win= Event.xclient.data.l[0];
            wminterface->windowChange(win);
          }
          else if (Event.xclient.message_type==
                  wminterface->getKWMModuleWinActivate()) {
            if (resource->getFocusStyle()!=none) {
              Window win;
              win = Event.xclient.data.l[0];
              focusWindow(win);
            }
          }
          else if (Event.xclient.message_type==
                  wminterface->getKWMModuleWinRaise()) {
            Window win=Event.xclient.data.l[0];
            raiseWindow(win);
          }
          else if (Event.xclient.message_type==
                  wminterface->getKWMModuleWinLower()) {
            Window win=Event.xclient.data.l[0];
            lowerWindow(win);
          }

          else if (Event.xclient.message_type==
                  wminterface->getKWMModuleDesktopChange()) {
            int current_desktop= (int) Event.xclient.data.l[0];
            desktopChange(current_desktop);
            redraw=True;
          }
          else if (Event.xclient.message_type==
                  wminterface->getKWMModuleDesktopNumberChange()) {
            int old_number_of_desktops = number_of_desktops;
            number_of_desktops= (int) Event.xclient.data.l[0];
            if (number_of_desktops>old_number_of_desktops) {
              /* enter new */
              DesktopList *tmp = new DesktopList;
              addDesktopWindow(tmp,False);
              desktopList->insert(tmp);
              LinkedListIterator<WindowList> win_it(windowList);
              for (; win_it.current(); win_it++)
              if ((win_it.current()->sticky && win_it.current()->desktop_nr==1))
                break;
              
              if (win_it.current()) {
                WindowList *sticky= win_it.current();
                WindowList *copy = new WindowList;
                copy->win= sticky->win;
                copy->width=sticky->width;
                copy->height=sticky->height;
                copy->x_position=sticky->x_position;
                copy->y_position=sticky->y_position;
                copy->icon=sticky->icon;
                copy->sticky=sticky->sticky;
                copy->desktop_nr=tmp->desktop_nr;
                addFrameWindow(copy,False);
                windowList->insert(copy);
              }
            } else if (number_of_desktops<old_number_of_desktops) {
              removeDesktopWindow();
            }
          }
        }
        break;
      case ButtonPress:
        {
          if (Event.xbutton.button == (unsigned)
                                    resource->getDesktopChangeButton()) {
            if (Event.xbutton.window==framewin) {
              XRaiseWindow(getDisplay(),framewin);
              lower=False;
            }
            else {
              int i=0;
              LinkedListIterator<DesktopList> win_it(desktopList);
              for (; win_it.current(); win_it++) {
                i++;
                if ((win_it.current()->win) == Event.xbutton.window)
                  break;
              }

              if (win_it.current())
                wminterface->changeDesktop(win_it.current()->desktop_nr-1);
            }
          }
          if (Event.xbutton.button == (unsigned)
                                      resource->getWindowMoveButton()) {
            if (Event.xbutton.window==framewin) {
              XLowerWindow(getDisplay(),framewin);
              lower=True;
            }
            else {
              LinkedListIterator<WindowList> win_it(windowList);
              LinkedListIterator<DesktopList> desktop_it(desktopList);

              DesktopList *desktop = new DesktopList;
  
              for (; desktop_it.current(); desktop_it++) {
                if ((desktop_it.current()->win) == Event.xbutton.window)
                  break;
              }

              if (desktop_it.current()) {
                desktop=desktop_it.current();
              } 

              for (; win_it.current(); win_it++)
                if ((win_it.current()->pager_win) == Event.xbutton.subwindow)
                  break;

              if (win_it.current()) {
                XSetWindowAttributes attrib;
                unsigned long create_mask = CWBackPixmap|CWCursor|
                                            CWBorderPixel;
  
                attrib.background_pixmap = ParentRelative;
                attrib.border_pixel=
                  resource->pagerwin.inactiveColor.getPixel();
                attrib.cursor = getSessionCursor();

                grabbedWindow = XCreateWindow(getDisplay(), framewin,
                                          win_it.current()->pager_x+desktop->x,
                                          win_it.current()->pager_y+desktop->y,
                                          win_it.current()->pager_width,
                                          win_it.current()->pager_height,
                                          1, getCurrentScreenInfo()->getDepth(),
                                          InputOutput,
                                          getCurrentScreenInfo()->getVisual(),
                                          create_mask,&attrib);

                grabbed_x=desktop->x+win_it.current()->pager_x-
                          Event.xbutton.x;
                grabbed_y=desktop->y+win_it.current()->pager_y-
                          Event.xbutton.y;
 
                if (!win_it.current()->focused ||
                                  resource->getFocusStyle()!=texture)
                  XSetWindowBackgroundPixmap(getDisplay(), grabbedWindow,
                                           pixmap.window);
                else
                   XSetWindowBackgroundPixmap(getDisplay(), grabbedWindow,
                                           pixmap.focusedWindow);
 
                moveWindow=win_it.current();
                realWindow=win_it.current()->win;
                pagerWindow=win_it.current()->pager_win;
                moved=False;
                XMapWindow(getDisplay(),grabbedWindow);
              } else {
                grabbed_x=0;
                grabbed_y=0;
                grabbedWindow=0;
              }
            }
          }
        }
        break;
      case ButtonRelease:
        {
          if (Event.xbutton.button == (unsigned) 
                                    resource->getWindowMoveButton())
            if (grabbedWindow!=0) {
              if (!moved) {
                XDestroyWindow(getDisplay(),grabbedWindow);
                grabbedWindow=0; 
                break;
              }
              LinkedListIterator<DesktopList> desktop_it(desktopList);
                for (; desktop_it.current(); desktop_it++) {
                  if (move_x>desktop_it.current()->x -
                      desktop_it.current()->width &&
                      move_x<=desktop_it.current()->x +
                      desktop_it.current()->width &&
                      move_y>desktop_it.current()->y -
                      desktop_it.current()->height&&
                      move_y<desktop_it.current()->y +
                      desktop_it.current()->height)
                    break;
                 } 
                if (desktop_it.current()) {

                  if (!moveWindow->sticky) 
                    wminterface->sendWindowToDesktop(realWindow,
                                desktop_it.current()->desktop_nr);
                  double xdiv=(double)resource->desktopSize.width/
                              getCurrentScreenInfo()->getWidth();
                  double ydiv=(double)resource->desktopSize.height/
                              getCurrentScreenInfo()->getHeight();
 
                  int x=(int)((move_x-desktop_it.current()->x)/xdiv);
                  int y=(int)((move_y-desktop_it.current()->y)/ydiv);
                  XMoveWindow(getDisplay(),realWindow,x,y);
                  XUnmapWindow(getDisplay(),grabbedWindow);
                  XDestroyWindow(getDisplay(),grabbedWindow);
                  if (!moveWindow->sticky) 
                    moveWinToDesktop(realWindow,
                               desktop_it.current()->desktop_nr);
                  grabbedWindow=0;
                }
                else {
                  XDestroyWindow(getDisplay(),grabbedWindow);
                  grabbedWindow=0;
                }
            }
        }
        break;
      case MotionNotify:
        {
          if (grabbedWindow) {
            moved=True;
            move_x=Event.xmotion.x+grabbed_x;
            move_y=Event.xmotion.y+grabbed_y;
            XMoveWindow(getDisplay(),grabbedWindow,Event.xmotion.x+grabbed_x,
                        Event.xmotion.y+grabbed_y);
          }
        }
        break;
      case ConfigureNotify:
        {
          if (Event.xconfigure.window==framewin && Event.xconfigure.send_event)
          {
            if (!decorated)
              Reconfigure();
 //           else
 //            	XResizeWindow(getDisplay(),framewin,frame.width,
//                          frame.height);

            int parent_x,parent_y;
            Window parent_root;
            unsigned int parent_width;
            unsigned int parent_height;
            unsigned int parent_border_width;
            unsigned int parent_depth;
            if (!decorated) {
              XGetGeometry(getDisplay(),Event.xconfigure.above,&parent_root,
                           &parent_x,&parent_y,&parent_width,&parent_height,
                           &parent_border_width,&parent_depth);
              frame.x=Event.xconfigure.x+parent_x;
              frame.y=Event.xconfigure.y+parent_y;

            }
          }
//          else
//           if (Event.xconfigure.window==framewin) 
//             	XResizeWindow(getDisplay(),framewin,frame.width,
//                          frame.height);

        }
      case Expose:
        {
          if (!lower)
            XRaiseWindow(getDisplay(),framewin);
        }
      }

      if (time(NULL) - lastTime > 10) {
    		if (resource->style.auto_config && !withdrawn)
				{
					CheckConfig();
				}
        lastTime = time(NULL);
      }
    } else {
      if (time(NULL) - lastTime < 10) {
        FD_ZERO(&rfds);
        FD_SET(xfd, &rfds);

        struct timeval tv;
        tv.tv_sec = 10 - (time(NULL) - lastTime);
        tv.tv_usec = 0;

        select(xfd + 1, &rfds, 0, 0, &tv);
      } else {
       	if (resource->style.auto_config && !withdrawn) {
	  		  CheckConfig();
		  	}
        if (!kwm_init) {
          fprintf(stderr,"Cannot setup KWM module\n");
          exit(-1);
        }
        lastTime = time(NULL);
      }
    }
  }
}
