 /* 
   sitecopy, for managing remote web sites.
   Copyright (C) 1998-99, Joe Orton <joe@orton.demon.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   $Id: console_fe.c,v 1.29.2.10 1999/10/24 21:52:21 joe Exp $
*/

/* This is the console front end for sitecopy.
 * TODO: Some of the configuration stuff should be moved out of here. */

#include <config.h>

#include <sys/types.h>

#include <sys/param.h>
#include <sys/stat.h>

#include <stdio.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif 
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include <signal.h>
#include <time.h>

#include <getopt.h>
#include <basename.h>

#ifdef ENABLE_NLS
#include <libintl.h>
#define _(str) gettext(str)
#else
#define _(str) str
#endif /* ENABLE_NLS */

#include "frontend.h"
#include "sites.h"
#include "rcfile.h"
#include "common.h"

/* From lib/yesno.c */
int yesno(void);

/* The maximum number of sites which can be specified on the command
 * line */

#define MAXSITES 20

enum action_t {
    action_none,
    action_list,
    action_synch,
    action_fetch,
    action_update,
    action_catchup,
    action_init,
    action_view
} action;

const char *action_names[] = {
    NULL,
    _("Showing changes to"),
    _("Synchronizing"),
    _("Fetching"),
    _("Updating"),
    _("Catching up"),
    _("Initializing"),
    NULL
};

/* The short program name, basename(argv[0]) */
const char *progname;

const char *sitenames[MAXSITES]; /* the sites specified on the command line */
int numsites; /* the number of sites specified */
struct site_t *current_site; /* this is used to save the state if we
			      * get signalled mid-update */
bool needsite; /* Is a site name required for the given action? */

/* User-specified options */
bool allsites; /* Do they want all sites to be operated on? */
bool listflat; /* Do they want the 'flat' list style */
int quiet; /* How quiet do they want us to be? */

/* Functions prototypes */

void init( int, char ** );
int main( int, char ** );
void usage( void );
void version( void );
void site_normlist( struct site_t * );
#if 0
bool fe_ynprompt( void );
#endif
void abort_handler( int );
void site_dumpsites( void );
void init_sites( void );

void abort_handler( int sig ) {
    /* Flush out debugging messages first */
    fflush( stderr );
#ifdef HAVE_STRSIGNAL
    printf( _("\n\n%s: Caught signal: %s (%d)\n"), progname, 
	    strsignal( sig ), sig );
#else
    printf( _("\n\n%s: Caught signal %d\n"), progname, sig );
#endif
    if( action == action_update && current_site!=NULL ) {
	printf( _("%s: In site update, saving current progress... "),
		progname );
	fflush( stdout );
	site_writefiles( current_site );
	printf( _("done.\n") );
    }
    printf( _("%s: Terminating.\n"), progname );
    exit( 15 );
}

/* Produce the normal listing output for the given site.
 */
void site_normlist( struct site_t *the_site ) {
    struct site_file_t *current;
    int count;

    if( the_site->numnew > 0 ) {
	printf( _("* These items have been added since the last update:\n") );
	count = 0;
	for( current = the_site->files; current!=NULL; current=current->next) {
	    if( current->diff == file_new ) {
		if( count++ ) printf( ", " );
		if( current->dir ) printf( _("dir:") );
		    printf( "%s", current->rel_local);
	    }
	}
	putchar( '\n' );
    }
    if( the_site->numchanged > 0 ) {
	printf( _("* These items have been changed since the last update:\n"));
	count = 0;
	for( current = the_site->files; current!=NULL; current=current->next) {
	    if( current->diff == file_changed ) {
		if( count++ ) printf( ", " );
		printf( "%s", current->rel_local);
	    }
	}
	putchar( '\n' );
    }
    if( the_site->numdeleted > 0 ) {
	if( the_site->nodelete ) {
	    printf( _("* These items have been deleted, but will be left on the server:\n") );
	} else {
	    printf( _("* These items have been deleted since the last update:\n"));
	}
	count = 0;
	for (current = the_site->files; current!=NULL; current=current->next) {
	    if( current->diff == file_deleted ) {
		if( count++ ) printf( ", " );
		if( current->dir ) printf( _("dir:") );
		printf( "%s", current->rel_local );
	    }
	}
	putchar( '\n' );
    }
    if( the_site->nummoved > 0 ) {
	printf( _("* These items have been moved since the last update:\n") );
	count = 0;
	for (current = the_site->files; current!=NULL; current=current->next) {
	    if( current->diff == file_moved ) {
		if( count++ ) printf( ", " );
		printf( "%s->/%s", current->old->rel_local, 
			current->directory  );
	    }
	}
	putchar( '\n' );
    }
}

void site_dumpsites( ) {
    struct site_t *current;
    for( current=all_sites; current!=NULL; current=current->next ) {
	/* FIXME: Possibly, make sure we have the actual port number first
	 * so we don't have to mess around printing out the port */
	printf( _("Site: %s\n\tServer: %s"), current->name, current->server );
	printf( _("  Port: ") );
	if( current->port == 0 ) {
	    printf( "%s\n", current->driver->service_name );
	} else {
	    printf( "%d\n", current->port );
	}
	printf( _("\tProtocol: %s\n\tUsername: %s\n"),
		current->driver->protocol_name, current->username );
	if( ! current->ftp_pasv_mode )
	    printf( _("\tPassive mode FTP will not be used.\n") );
	printf( _("\tRemote directory: %s\n\tLocal directory: %s\n"),
		current->remote_root_user, current->local_root_user );
	printf( _("\tPermissions: %s     Symlinks: %s\n"),
		PERMSMODE( current ), SYMLINKSMODE( current ) );
	if( current->nodelete ) 
	    printf( _("\tRemote files will not be deleted.\n") );
	if( current->checkmoved ) 
	    printf( _("\tFiles will be moved remotely if moved locally.\n") );
    }
}

void init( int argc, char *argv[] ) {
    int optc, opt_index;
    bool firstlist = false;
    extern char *optarg;
    extern int optind;
#ifdef DEBUGGING
    char *shortopts = "d:acifhklp:qr:suvVy";
#else
    char *shortopts = "acifhklp:qr:suvVy";
#endif
    const struct option longopts[] = {
	/* Operation modes */
	{ "update", no_argument, NULL, 'u' },
	{ "initialize", no_argument, NULL, 'i' },
	{ "fetch", no_argument, NULL, 'f' },
	{ "synchronize", no_argument, NULL, 's' },
	{ "list", no_argument, NULL, 'l' },
	{ "flatlist", no_argument, &listflat, true },
	{ "keep-going", no_argument, NULL, 'k' },
	{ "help", no_argument, NULL, 'h' },
	{ "catchup", no_argument, NULL, 'c' },
	{ "view", no_argument, NULL, 'v' },
	/* Options */
	{ "quiet", no_argument, &quiet, 1 },
	{ "silent", no_argument, &quiet, 2 },
	{ "rcfile", required_argument, NULL, 'r' },
	{ "storepath", required_argument, NULL, 'p' },
#ifdef DEBUGGING
	{ "debug", required_argument, NULL, 'd' },
#endif
	{ "prompting", no_argument, &fe_prompting, true },
	{ "allsites", no_argument, &allsites, true },
	{ "version", no_argument, NULL, 'V' },
	{ 0, 0, 0, 0 }
    };
	
#ifdef DEBUGGING
    /* We need these for setting up the debugging mask from
     * the command line */
    extern int debug_mask;

    debug_mask = 0;
#endif
    
    /* SETACTION is used to ensure only one opmode is specified
     * on the command line at a time.
     */
#define SETACTION( newaction ) 						   \
    if( action != action_none ) {					   \
	printf( _("%s: Error: Only specify ONE operation mode at a time.\n"), \
		progname );						   \
	printf( _("Try `%s --help' for more information.\n"), progname );  \
	exit( -1 );							   \
    } else {								   \
	action = newaction;						   \
    }

    /* Defaults */
    allsites = fe_prompting = false;
    action = action_none;
    fe_prompting = false;
    progname = base_name( argv[0] );
    needsite = false;

    /* Read the cmdline args */
    while( (optc = getopt_long( argc, argv, 
				shortopts, longopts, &opt_index)) != -1) {
	switch( optc ) {
	case 0:
	    /* Make the action list mode if they gave --flatlist */
	    if( listflat == true )
		action = action_list;
	    break;
	case 'l':
	    if( firstlist == false ) {
		SETACTION( action_list );
		firstlist = true;
	    } else {
		listflat = true;
	    }
	    break;
	case 's': SETACTION( action_synch ); break;
	case 'f': SETACTION( action_fetch ); break;
	case 'i': SETACTION( action_init ); break;
	case 'c': SETACTION( action_catchup ); break;
	case 'u': SETACTION( action_update ); break;
	case 'v': 
	    SETACTION( action_view );
	    allsites = true; /* so they get checked */
	    break;
	case 'q': quiet++; break;
	case 'r': 
	    if( strlen(optarg) != 0 ) {
		rcfile = strdup( optarg );
	    } else {
		usage( );
		exit(-1);
	    }
	    break;
	case 'p':
	    if( strlen(optarg) != 0 ) {
		if( optarg[strlen(optarg)] != '/' ) {
		    copypath = malloc( strlen(optarg) + 2 );
		    strcpy( copypath, optarg );
		    strcat( copypath, "/" );
		} else {
		    copypath = strdup( optarg );
		}
	    } else {
		usage( );
		exit(-1);
	    }
	    break;
#ifdef DEBUGGING
	case 'd': /* set debugging level */
	    debug_mask = atoi(optarg);
	    break;
#endif

	case 'y':
	    fe_prompting = true;
	    break;
	case 'k':
	    site_keepgoing = true;
	    break;
	case 'a':
	    allsites = true;
	    break;
	case 'V': version( ); exit(-1);
	case 'h': usage( ); exit(-1);
	case '?': 
	default:
	    printf( _("Try `%s --help' for more information.\n"), progname );
	    exit(-1);
	}
    }

    /* Set the default action mode */
    if( action == action_none )
	action = action_list;

    /* Determine whether the specified action needs a site name
     * to operate on */
    needsite = ((action==action_list) || (action==action_update) || \
		(action==action_catchup) || (action==action_synch) || \
		(action==action_fetch) || (action==action_init));
    
    /* Get those site names off the end of the cmdline */
    for( numsites = 0 ; optind < argc; optind++ ) {
	sitenames[numsites++] = argv[optind];
	if( numsites == MAXSITES ) {
	    printf( _("%s: Warning: Only %d sites can be specified on the command line!\nExtra entries are being skipped.\n"), progname, MAXSITES );
	    break;
	}
    }

}

#if 0
/* Prompts for confirmation - returns */
bool fe_ynprompt( ) {
    char in[BUFSIZ];
    fgets( in, BUFSIZ, stdin );
    return (*in == 'y'); /* only return true if they put a 'y' first */
}
#endif

bool fe_can_update( const struct site_file_t *file ) {
    if( file->dir ) {
	if( file->diff == file_new ) {
	    printf( _("Create %s/"), file->rel_local );
	} else {
	    printf( _("Delete %s/"), file->rel_local );
	}
    } else {
	switch( file->diff ) {
	case file_changed:
	case file_new: printf( _("Upload %s"), file->rel_local ); break;
	case file_deleted: printf( _("Delete %s"), file->rel_local ); break;
	case file_moved: printf( _("Move %s->/%s"), file->old->rel_local,
				 file->directory ); break;
	default: printf( "You've really broken it now.\n" ); break;
	}
    }
    printf( _("? (y/n) ") );
    return yesno( );
}

/* Called when the given files is about to be updated */
void fe_updating( const struct site_file_t *file ) {
    switch( quiet ) {
    case 0:
	if( file->dir ) {
	    if( file->diff == file_new ) {
		printf( _("Creating %s/: "), file->rel_local+1 );
	    } else {
		printf( _("Deleting %s/: "), file->rel_local+1 );
	    }
	} else {
	    switch( file->diff ) {
	    case file_changed:
	    case file_new: printf( _("Uploading %s: ["), file->rel_local+1 ); break;
	    case file_deleted: printf( _("Deleting %s: "), file->rel_local+1 ); break;
	    case file_moved: printf( _("Moving %s->%s: "), file->old->rel_local+1,
				     file->directory ); break;
	    default: printf( "You've really broken it now.\n" ); break;
	    }
	}
	fflush( stdout );
	break;
    case 1:
	printf( "%s\n", file->rel_local+1 );
	break;
    default:
	break;
    }
}

void fe_updated( const struct site_file_t *file, const bool success,
		 const char *error ) {
    if( quiet > 0 ) {
	if( ! success ) {
	    printf( _("Failed to update %s:\n%s\n"), 
		    file->rel_local, error );
	}
	return;
    }
    if( file->dir || (file->diff!=file_changed && file->diff!=file_new) ) {
	if( success ) {
	    printf( _("done.\n") );
	} else {
	    printf( _("failed:\n%s\n"), error );
	}
    } else {
	if( success ) {
	    printf( _("] done.\n") );
	} else {
	    printf( _("] failed:\n%s\n"), error );
	}
    }
}

void fe_synching( const struct site_file_t *file ) {
    if( quiet ) {
	printf( "%s\n", file->rel_local );
    } else {
	if( file->dir ) {
	    if( file->diff != file_new ) {
		printf( _("Creating %s/: "), file->rel_local+1 );
	    } else {
		printf( _("Deleting %s/: "), file->rel_local+1 );
	    }
	} else {
	    switch( file->diff ) {
	    case file_changed:
	    case file_deleted: 
		printf( _("Downloading %s: ["), file->rel_local+1 ); 
		break;
	    case file_new: 
		printf( _("Deleting %s: "), file->rel_local+1 ); 
		break;
	    case file_moved: 
		printf( _("Moving %s->%s: "), file->rel_local+1,
			file->old->directory ); 
		break;
	    default: 
		break;
	    }
	}
	fflush( stdout );
    }
}

void fe_synched( const struct site_file_t *file, const bool success,
		 const char *error ) {
    if( quiet )
	return;
    if( file->dir || (file->diff!=file_changed && file->diff!=file_deleted) ) {
	if( success ) {
	    printf( _("done.\n") );
	} else {
	    printf( _("failed:\n%s\n"), error );
	}
    } else {
	if( success ) {
	    printf( _("] done.\n") );
	} else {
	    printf( _("] failed:\n%s\n"), error );
	}
    }
}

void fe_transfer_progress( size_t num, size_t total ) {
    if( quiet > 0 ) return;
    putchar( '.' );
    fflush( stdout );
}

void fe_connection( fe_conn_status status ) {}

void fe_fetch_found( const struct site_file_t *file ) {
    if( file->dir ) {
	printf( _("Directory: %s/\n"), file->rel_local+1 );
    } else {
	printf( _("File: %s - modified %s"), file->rel_local+1,
		ctime(&file->remotetime) );
    }
}

void init_sites( void ) {
    int ret;
    
    /* Read the rcfile */
    ret = rcfile_read( &all_sites );
    if( ret == RC_OPENFILE ) {
	printf( _("%s: Error: Could not open rcfile: %s.\n"),
		progname, rcfile );
	exit( -1 );
    } else if( ret == RC_CORRUPT ) {
	printf( _("%s: rcfile corrupt at line %d:\n%s\n"),
		progname, rcfile_linenum, rcfile_err );
	exit( -1 );
    }
}

int main( int argc, char *argv[] ) {
    int ret, num, numgoodsites;
    struct site_t *current;

    /* Initialize up */
    ret = 0;

#ifdef HAVE_SETLOCALE
    setlocale( LC_MESSAGES, "" );
#endif /* HAVE_SETLOCALE */

#ifdef ENABLE_NLS
    bindtextdomain( PACKAGE, LOCALEDIR );
    textdomain( PACKAGE );
#endif /* ENABLE_NLS */

    init( argc, argv );

    if( init_env( ) == 1 ) {
	printf( _("%s: Error: Environment variable HOME not set.\n"),
		progname );
	exit( -1 );
    }

    ret = init_paths( );

    if( ret != 0 ) {
	switch( ret ) {
	case RC_OPENFILE:
	    printf( _("%s: Error: Could not open rcfile: %s\n"), progname, rcfile );
	    break;
	case RC_PERMS:
	    printf( _("%s: Error: rcfile permissions allow other users to read your rcfile.\n"), progname );
	    printf( _("%s: Set the permissions to 0600.\n"), progname );
	    break;
	case RC_DIROPEN:
	    printf( _("%s: Error: Could not open storage directory: %s\n"), progname, copypath );
	    printf( _("%s: You need to create this directory and set the permissions to 0700.\n"), progname );
	    break;
	case RC_DIRPERMS:
	    printf( _("%s: Error: storage directory permissions incorrect.\n"), progname );
	    printf( _("%s: Set the permissions to 0700.\n"), progname );
	    break;
	case RC_NETRCPERMS:
	    printf( _("%s: Error: ~/.netrc permissions incorrect.\n"), progname );
	    printf( _("%s: Set the permissions to 0600.\n"), progname );
	    break;
	default:
	    printf( _("%s: Argh, unhandled error.\n"), progname );
	    break;
	}
	exit(-1);
    }

    if( init_netrc( ) == 1 ) {
	printf( _("%s: Error: Could not parse ~/.netrc.\n"), progname );
	exit( -1 );
    }

    ret = 0;

    init_sites( );

    /* Handle signals, so if we get killed in the middle of a
     * site update, the state gets saved properly. */
    signal( SIGTERM, abort_handler );
    signal( SIGQUIT, abort_handler );
    signal( SIGINT, abort_handler );
    signal( SIGABRT, abort_handler );
    /* And ignore SIGPIPE */
    signal( SIGPIPE, SIG_IGN );

    if( needsite && !allsites ) {
	/* Mark all the sites we are interested in */
	for( num=0; num < numsites; num++ ) {
	    current = site_find( sitenames[num] );
	    if( current == NULL ) {
		printf( _("%s: Error: No site called `%s' found - skipping.\n"), 
			progname, sitenames[num] );
	    } else {
		current->use_this = true;
	    }
	}
    }

    /* Count the number of sites that are okay */
    numgoodsites = 0;

    for( current = all_sites; current!=NULL; current=current->next ) {
	if( !current->use_this && !allsites ) continue;
	/* Check the site rcfile entry is okay */
	ret = rc_verifysite( current );
	switch( ret ) {
	case SITE_ACCESSLOCALDIR:
	    printf( _("%s: Could not read directory for `%s':\n\t%s\n"), 
		    progname, current->name, current->local_root );
	    break;
	case SITE_NOSERVER:
	    printf( _("%s: Server not specified in site `%s'.\n"), 
		    progname, current->name );
	    break;
	case SITE_NOUSERNAME:
	    printf( _("%s: Username not specified in site `%s'.\n"), 
		    progname, current->name );
	    break;
	case SITE_NOPASSWORD:
	    printf( _("%s: Password not specified in site `%s'.\n"), 
		    progname, current->name );
	    break;
	case SITE_NOREMOTEDIR:
	    printf( _("%s: Remote directory not specified in site `%s'.\n"), 
		    progname, current->name );
	    break;
	case SITE_NOLOCALDIR:
	    printf( _("%s: Local directory not specified in site `%s'.\n"), 
		    progname, current->name );
	    break;
	case SITE_INVALIDPORT:
	    printf( _("%s: Invalid port used in site `%s'.\n"),
		    progname, current->name );
	    break;
	case SITE_NOMAINTAIN:
	    printf( _("%s: %s cannot maintain symbolic links (site `%s').\n"),
		    progname, current->driver->protocol_name, current->name );
	    break;
	case SITE_NOREMOTEREL:
	    printf( _("%s: Cannot use a relative remote directory in %s (site `%s').\n"), progname, current->driver->protocol_name, current->name );
	    break;
	case SITE_NOPERMS:
	    printf( _("%s: File permissions are not supported in %s (site `%s').\n"), progname, current->driver->protocol_name, current->name );
	    break;
	case 0:
	    /* Success */
	    break;
	default:
	    printf( _("%s: Unhandled error %d in site `%s' - please contact the maintainer.\n"), progname, ret, current->name );
	    break;
	}
	if( ret != 0 ) continue;
	
	if( site_readfiles( current ) ) {
	    printf( _("%s: Error: Could not read site info file for `%s'.\n"),
		    progname, current->name );
	} else {
	    numgoodsites++;
	}
    }

    /* Handle the dump sites request */
    if( ! needsite ) {
	switch( action ) {
	case action_view:
	    site_dumpsites( );
	    ret = 0;
	    break;
	default:
	    printf( _("Unimplemented.\n") );
	    ret = -1;
	}
	return ret;
    }
    
    if( numgoodsites==0 ) {
	printf( _("%s: No sites specified.\n"), progname );
	printf( _("Try `%s --help' for more information.\n"), progname );
	exit(-1);
    }

    for( current=all_sites; current!=NULL; current=current->next ) {
	
	if( !current->use_this && !allsites ) 
	    continue;
	
	if( !listflat ) {
	    printf( _("%s: %s site `%s' (on %s in %s)\n"), 
		    progname, action_names[action], current->name, 
		    current->server, current->remote_root_user );
	}

	switch( action ) {
	case action_update:
	    if( current->is_different ) {
		current_site = current;
		ret = site_update( current, false );
		switch( ret ) {
		case SITE_OK:
		    printf( _("%s: Remote site updated successfully.\n"),
			    progname );
		    ret = 0;
		    break;
		case SITE_ERRORS:
		    printf( _("%s: Errors occurred while updating remote site.\n"),
			   progname );
		    ret = 1;
		    break;
		case SITE_LOOKUP:
		    printf( _("%s: Error: Could not resolve remote hostname.\n"),
			    progname );
		    ret = 2;
		    break;
		case SITE_CONNECT:
		    printf( _("%s: Error: Could not connect to remote host.\n"),
			    progname );
		    ret = 2;
		    break;
		case SITE_AUTH:
		    printf( _("%s: Error: Could not authorise user on server.\n"),
			    progname );
		    ret = 2;
		    break;
		}		    
		/* hope we don't get signalled here */
		current_site = NULL;
		site_writefiles( current );
	    } else {
		printf( _("%s: Nothing to do - no changes found.\n"), progname );
		ret = 0;
	    }
	    break;
	case action_list:
	    if( listflat ) {
		site_flatlist( stdout, current );
	    } else {
		site_normlist( current );
	    }
	    ret = (current->is_different)?1:0;
	    if( current->is_different ) {
		if( !listflat ) {
		    printf( _("%s: The remote site needs updating.\n"), progname);
		}
		ret = 1;
	    } else {
		if( !listflat ) {
		    printf( _("%s: The remote site does not need updating.\n"), progname );
		}
		ret = 0;
	    }
	    break;
	case action_init:
	    site_initialize( current );
	    site_writefiles( current );
	    printf( _("%s: All the files and directories are marked as NOT updated remotely.\n"), progname );
	    break;
	case action_catchup:
	    site_catchup( current );
	    site_writefiles( current );
	    printf( _("%s: All the files and and directories are marked as updated remotely.\n"), progname );
	    break;
	case action_fetch:
	    ret = site_fetch( current );
	    switch( ret ) {
	    case SITE_UNIMPL: 
		printf( _("%s: Fetch mode unsupported for %s.\n"),
			progname, current->driver->protocol_name );
		ret = -1;
		break;
	    case SITE_LOOKUP:
		printf( _("%s: Error: Could not resolve remote hostname.\n"),
			progname );
		ret = 2;
		break;
	    case SITE_CONNECT:
		printf( _("%s: Error: Could not connect to remote host.\n"),
			progname );
		ret = 2;
		break;
	    case SITE_AUTH:
		printf( _("%s: Error: Could not authorise user on server.\n"),
			progname );
		ret = 2;
		break;
	    case SITE_FAILED:
		printf( _("%s: Failed to fetch file listing.\n"), progname );
		ret = -1;
		break;
	    case SITE_OK:
		printf( _("%s: Fetch completed successfully.\n"), progname );
		site_writefiles( current );
		ret = 0;
		break;
	    } 
	    break;
	case action_synch:
	    if( current->is_different ) {
		ret = site_synch( current );
		switch( ret ) {
		case SITE_OK:
		    printf( _("%s: Synchronize completed successfully\n"),
			    progname );
		    ret = 0;
		    break;
		case SITE_ERRORS:
		    printf( _("%s: Errors occurred while synchronizing the local site.\n"), progname );
		    ret = 1;
		    break;
		}
	    } else {
		printf( _("%s: Nothing to do - no changes found.\n"), progname );
		ret = 0;
	    }
	    break;
	default:
	    printf( _("Unimplemented.\n") );
	    ret = -1;
	    break;
	}
    }
    
    return ret;
}

void usage( ) {
    version();
    printf( _("Usage: %s [OPTIONS] [MODE] [sitename]...\n"),
	     progname );
    printf( _("Options: \n") );
#ifdef DEBUGGING
    printf( _("  -d, --debug=LEVEL     Set debugging level\n"
" (LEVEL=sum of: 1=socket, 2=files, 4=rcfile, 8=WebDAV, 16=FTP, 32=XML )\n") );
#endif
    printf( _(
"  -r, --rcfile=FILE     Use alternate run control file\n"
"  -p, --storepath=PATH  Use alternate site storage directory\n"
"  -y, --prompting       Request confirmation before making each update\n"
"  -a, --allsites        Perform the operation on ALL defined sites\n"
"  -k, --keep-going      Carry on an update regardless of errors\n"
"  -q, --quiet           Be quiet while performing the operation\n"
"  -qq, --silent         Be silent while perforing the operation\n"
"Operation modes:\n"
"  -l, --list            List changes between remote and local sites (default)\n"
"  -ll, --flatlist       Flat list of changes between remote and local sites\n"
"  -v, --view            Display a list of the site definitions\n"
"  -i, --initialize      Mark all files and directories as not updated\n"
"  -f, --fetch           Find out what files are on the remote site\n"
"  -c, --catchup         Mark all files and directories as updated\n"
"  -s, --synchronize     Update the local site from the remote site\n"
"  -u, --update          Update the remote site\n"
"  -h, --help            Display this help message\n"
"  -V, --version         Display version information\n"
"Please report bugs to sitecopy@lyra.org\n") );
}
/* "Warning: directory search depth exceeded. Increase MAXDIRS in sites.c.\n\
   Directory %s is being skipped.\n" */

/* One-liner version information. */
void version( void ) {
    printf( 
	PACKAGE " " VERSION ": FTP"
#ifdef USE_DAV
	", WebDAV"
#ifdef HAVE_LIBEXPAT
	"+fetch"
#endif
#endif
#ifdef DEBUGGING
	", debugging"
#endif
#ifdef __EMX__
	", EMX/RSX platform"
#else
#ifdef __CYGWIN__
	", cygwin platform"
#else
	", Unix platform"
#endif /* __CYGWIN__ */
#endif /* __EMX__ */
#ifdef HAVE_LX22_SENDFILE
	", sendfile()"
#endif
	".\n" );
}
