/****************************************************************************/
/* ##   ##         #####   #####  ##     **        NoSQL RDBMS - pick       */
/* ###  ##        ####### ####### ##     **        $Revision: 2.1.1 $       */
/* #### ##        ###     ##   ## ##     ************************************/
/* #######  ####  #####   ##   ## ##     **      Carlo Strozzi (c) 1998     */
/* ####### ######   ##### ## # ## ##     ************************************/
/* ## #### ##  ##     ### ##  ### ##     **           Written by            */
/* ##  ### ###### ####### ######  ###### **     Maurizio Sartori Masar      */
/* ##   ##  ####   #####   #### # ###### **       email: masar@ibm.it       */
/****************************************************************************/
/*  NoSQL RDBMS, Copyright (C) 1998 Carlo Strozzi.                          */
/*  This program comes with ABSOLUTELY NO WARRANTY; for details             */
/*  refer to the GNU General Public License.                                */
/****************************************************************************

   Extract rows at selected offsets from a large, indexed table.

   Usage: nosql pick [-R|--raw-input] [-x|--hex] table < offset_list

   This operator takes a list of byte-offsets on STDIN and prints on STDOUT
   the corresponding table records from the table specified as a command
   line argument. The list of offsets can be built with the 'index'
   operator, and 'pick' expects it to be a one-column table, like this:

           lpos
           ----
           228
           1117
           518
           1225

   The column name, 'lpos' in the example, can take any name, provided
   that the list has a valid NoSQL table format. If option '-R' is
   given, then the offsets are expected to be a blank- , tab- or
   newline-separated raw list of numbers on STDIN, i.e. they are not
   expected to to have a valid NoSQL table header.

   If option '-x' is given, then the offsets are expected to be base 16
   numbers instead of the usual base 10 numbers.

   Examples :

     nosql pick table < offset_table

     or

     echo "228 1117 518 1225" | nosql pick -R table
     echo "a28 1b1F 5c8 1425" | nosql pick -R -x table


 ****************************************************************************/

/*
static char Psz_RCS [] =
    "$Id: pick.c,v 1.1 1999/04/07 17:17:41 carlos Exp $";
*/

/****************************************************************************/

#include <stdio.h>
#include <ctype.h>


/****************************************************************************/
/*                          Function prototypes                             */
/****************************************************************************/
void LineOutput( FILE * file, const char * pszStr );
void LineSkip( FILE * file, const char * pszStr );

/****************************************************************************/
/*                              LineOutput ()                               */
/****************************************************************************/

void LineOutput( FILE * file, const char * pszStr )
{
   int chChar;

   chChar = fgetc( file );
   while ( chChar != '\n' )
   {
      if ( chChar == EOF )
      {
         fprintf( stderr, pszStr );
         exit(1);
      }

      fputc( chChar, stdout );
      chChar = fgetc( file );
   }
   fputc( chChar, stdout );
}

/****************************************************************************/
/*                               LineSkip ()                                */
/****************************************************************************/
void LineSkip( FILE * file, const char * pszStr )
{
   int chChar;

   chChar = fgetc( file );
   while ( chChar != '\n' )
   {
      if ( chChar == EOF )
      {
         fprintf( stderr, pszStr );
         exit(1);
      }

      chChar = fgetc( file );
   }
}

/****************************************************************************/
/*                                  main ()                                 */
/****************************************************************************/
int main( int argc, char **argv )
{
   int    fRaw = 0;
   FILE * fileDb = NULL;
   int    iInx;
   int    chChar;
   long   lOffset;
   int    iValue[256];
   int    iBase;

   for ( iInx = 0; iInx < 256; iInx++ )
   {
      iValue[iInx] = 0;
   }

   iValue['0'] = 1;
   iValue['1'] = 2;
   iValue['2'] = 3;
   iValue['3'] = 4;
   iValue['4'] = 5;
   iValue['5'] = 6;
   iValue['6'] = 7;
   iValue['7'] = 8;
   iValue['8'] = 9;
   iValue['9'] = 10;
   iBase = 10;

   for ( iInx = 1; iInx < argc; iInx++ )
   {
      if ( !strcmp( argv[iInx], "-R" )
        || !strcmp( argv[iInx], "--raw-input" ) )
      {
         fRaw = 1;
      }
      else if ( !strcmp ( argv[iInx], "-x" )
            || !strcmp ( argv[iInx], "--hex" ) )
      {
         iValue['a'] = iValue['A'] = 11;
         iValue['b'] = iValue['B'] = 11;
         iValue['c'] = iValue['C'] = 11;
         iValue['d'] = iValue['D'] = 11;
         iValue['e'] = iValue['E'] = 11;
         iValue['f'] = iValue['F'] = 11;
         iBase = 16;
      }
      else if ( fileDb )
      {
         fprintf( stderr, "nosql pick: invalid number of parameters\n" );
         return 1;
      }
      else
      {
         fileDb = fopen( argv[iInx], "rt" );
         if ( !fileDb )
         {
            fprintf( stderr, "nosql pick: cannot open \"%s\"\n", argv[iInx] );
            return 1;
         }
      }
   }

   if ( !fileDb )
   {
      fprintf( stderr, "nosql pick: no database specified\n" );
      return 1;
   }

   /*************************************************************************/
   /* Put headers (first and second line of the database)                   */
   /*************************************************************************/
   LineOutput( fileDb, "Invalid Database Header 1\n" );
   LineOutput( fileDb, "Invalid Database Header 2\n" );

   /*************************************************************************/
   /* Skip headers if not Raw Mode                                          */
   /*************************************************************************/
   if ( !fRaw )
   {
      LineSkip( stdin, "nosql pick: invalid input header 1\n" );
      LineSkip( stdin, "nosql pick: invalid input header 2\n" );
   }

   /*************************************************************************/
   /* Process offsets                                                       */
   /*************************************************************************/
   lOffset = 0;
   chChar = fgetc( stdin );
   while ( chChar != EOF )
   {
      if (iValue [chChar])
      {
         lOffset *= iBase;
         lOffset += iValue[chChar] - 1;
      }
      else if ( isspace( chChar ) )
      {
         if ( lOffset )
         {
            if ( fseek( fileDb, lOffset, SEEK_SET ) )
            {
               fprintf( stderr, "nosql pick: cannot seek\n" );
               exit(1);
            }

            LineOutput( fileDb, "nosql pick: invalid file line\n" );
         }

         lOffset = 0;
      }
      else
      {
         fprintf( stderr, "nosql pick: invalid character \"%c\"\n", chChar );
         exit(1);
      }

      chChar = fgetc( stdin );
   }

   return 0;
}

