/* t1asm
 *
 * This program `assembles' Adobe Type-1 font programs in
 * pseudo-PostScript form into either PFB or PFA format.  The human
 * readable/editable input is charstring- and eexec-encrypted as
 * specified in the `Adobe Type 1 Font Format' version 1.1 (the `black
 * book').  There is a companion program, t1disasm, which `disassembles'
 * PFB and PFA files into a pseudo-PostScript file.
 *
 * Copyright (c) 1992 by I. Lee Hetherington, all rights reserved.
 *
 * Permission is hereby granted to use, modify, and distribute this
 * program for any purpose provided this copyright notice and the one
 * below remain intact.
 *
 * I. Lee Hetherington (ilh@lcs.mit.edu)
 *
 *======================================================================
 *
 * This program has been modified by CurveSoft, Inc. and all such
 * modifications are covered by the following copyright notice, license,
 * and disclaimer
 *
 * Copyright (C) 1997  CurveSoft, Inc. All rights reserved.
 *
 * License is hereby granted without fee, to use, copy, modify,
 * translate and distribute this software and its documentation for any
 * purpose provided that the above copyright notice is present in all
 * copies and that the name of CurveSoft not be used in advertising or
 * publicity pertaining to this software without written prior
 * permission.
 *
 * CURVESOFT PROVIDES THIS SOFTWARE "AS IS", WITHOUT ANY WARRANTIES OF
 * ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, BUT NOT LIMITED TO
 * ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, AND NONINFRINGEMENT OF THIRD PARTY RIGHTS.  THE ENTIRE RISK
 * AS TO THE QUALITY AND PERFORMANCE OF THE SOFTWARE, INCLUDING ANY DUTY
 * TO SUPPORT OR MAINTAIN, BELONGS TO THE LICENSEE.  SHOULD ANY PORTION
 * OF THE SOFTWARE PROVE DEFECTIVE, THE LICENSEE (NOT CURVESOFT) ASSUMES
 * THE ENTIRE COST OF ALL SERVICING, REPAIR AND CORRECTION.  IN NO EVENT
 * SHALL CURVESOFT BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 *======================================================================
 *
 * Log:
 *
 * Revision 1.4  92/07/10  10:53:09  ilh
 * Added support for additional PostScript after the closefile command
 * (ie., some fonts have {restore}if after the cleartomark).
 *
 * Revision 1.3  92/06/23  10:58:25  ilh
 * MSDOS porting by Kai-Uwe Herbing (herbing@netmbx.netmbx.de)
 * incoporated.
 *
 * Revision 1.2  92/05/22  11:54:45  ilh
 * Fixed bug where integers larger than 32000 could not be encoded in
 * charstrings.  Now integer range is correct for four-byte
 * twos-complement integers: -(1<<31) <= i <= (1<<31)-1.  Bug detected
 * by Piet Tutelaers (rcpt@urc.tue.nl).
 *
 * Revision 1.1  92/05/22  11:48:46  ilh
 * initial version
 *
 * Ported to Microsoft C/C++ Compiler and MS-DOS operating system by
 * Kai-Uwe Herbing (herbing@netmbx.netmbx.de) on June 12, 1992. Code
 * specific to the MS-DOS version is encapsulated with #ifdef _MSDOS
 * ... #endif, where _MSDOS is an identifier, which is automatically
 * defined, if you compile with the Microsoft C/C++ Compiler.
 */

#ifndef lint
static char rcsid[] =
 "@(#) $Id: t1asm.c,v 1.4 92/07/10 10:53:09 ilh Exp $";
static char copyright[] =
 "@(#) Copyright (c) 1992 by I. Lee Hetherington, all rights reserved.";
#ifdef _MSDOS
static char portnotice[] =
 "@(#) Ported to MS-DOS by Kai-Uwe Herbing (herbing@netmbx.netmbx.de).";
#endif
#endif  /* lint */

/* Note: this is ANSI C. */

#ifdef _MSDOS
#include <fcntl.h>
#include <getopt.h>
#include <io.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>

/* int32 must be at least 32-bit and uint16 must be at least 16-bit */
#if INT_MAX >= 0x7FFFFFFFUL
typedef int int32;
#else
typedef long int32;
#endif

#if USHRT_MAX >= 0xFFFFUL
typedef unsigned short uint16;
#else
typedef unsigned int uint16;
#endif

#define LINESIZE 256

#define MAXBLOCKLEN ((1L<<17)-6)
#define MINBLOCKLEN ((1L<<8)-6)

#define MARKER   128
#define ASCII    1
#define BINARY   2
#define DONE     3

typedef unsigned char byte;

static FILE *ifp = stdin;
static FILE *ofp = stdout;

/* flags */
static int pfb = 0;
static int active = 0;
static int start_charstring = 0;
static int in_eexec = 0;

static char line[LINESIZE];

/* lenIV and charstring start command */
static int lenIV = 4;
static char cs_start[10];

/* for charstring buffering */
static byte charstring_buf[65535];
static byte *charstring_bp;

/* for PFB block buffering */
static byte blockbuf[MAXBLOCKLEN];
static int32 blocklen = MAXBLOCKLEN;
static int32 blockpos = -1;
static int blocktyp = ASCII;

/* decryption stuff */
static uint16 er, cr;
static uint16 c1 = 52845, c2 = 22719;

/* table of charstring commands */
static struct command {
  char *name;
  int one, two;
} command_table[] = {

  { "Div", 12, 25 },                /* Type 2 */
  { "abs", 12, 9 },                 /* Type 2 */
  { "add", 12, 10 },                /* Type 2 */
  { "and", 12, 3 },                 /* Type 2 */
  { "blend", 16, -1 },              /* Type 2 */
  { "callgsubr", 29, -1 },          /* Type 2 */
  { "callothersubr", 12, 16 },
  { "callsubr", 10, -1 },
  { "closepath", 9, -1 },
  { "cntrmask", 20, -1 },           /* Type 2 */
  { "div", 12, 12 },
  { "dotsection", 12, 0 },
  { "drop", 12, 18 },               /* Type 2 */
  { "dup", 12, 27 },                /* Type 2 */
  { "endchar", 14, -1 },
  { "eq", 12, 15 },                 /* Type 2 */
  { "exch", 12, 28 },               /* Type 2 */
  { "flex", 12, 35 },               /* Type 2 */
  { "flex1", 12, 37 },              /* Type 2 */
  { "get", 12, 21 },                /* Type 2 */
  { "hflex", 12, 34 },              /* Type 2 */
  { "hflex1", 12, 36 },             /* Type 2 */
  { "hhcurveto", 27, -1 },          /* Type 2 */
  { "hintmask", 19, -1 },           /* Type 2 */
  { "hlineto", 6, -1 },
  { "hmoveto", 22, -1 },
  { "hsbw", 13, -1 },
  { "hstem", 1, -1 },
  { "hstem3", 12, 2 },
  { "hstemhm", 18, -1 },            /* Type 2 */
  { "hvcurveto", 31, -1 },
  { "ifelse", 12, 22 },             /* Type 2 */
  { "index", 12, 29 },              /* Type 2 */
  { "load", 12, 13 },               /* Type 2 */
  { "mul", 12, 24 },                /* Type 2 */
  { "neg", 12, 14 },                /* Type 2 */
  { "not", 12, 5 },                 /* Type 2 */
  { "or", 12, 4 },                  /* Type 2 */
  { "pop", 12, 17 },
  { "put", 12, 20 },                /* Type 2 */
  { "random", 12, 23 },             /* Type 2 */
  { "rcurveline", 24, -1 },         /* Type 2 */
  { "return", 11, -1 },
  { "rlinecurve", 25, -1 },         /* Type 2 */
  { "rlineto", 5, -1 },
  { "rmoveto", 21, -1 },
  { "roll", 12, 30 },               /* Type 2 */
  { "rrcurveto", 8, -1 },
  { "sbw", 12, 7 },
  { "seac", 12, 6 },
  { "setcurrentpoint", 12, 33 },
  { "sqrt", 12, 26 },               /* Type 2 */
  { "store", 12, 8 },               /* Type 2 */
  { "sub", 12, 11 },                /* Type 2 */
  { "vhcurveto", 30, -1 },
  { "vlineto", 7, -1 },
  { "vmoveto", 4, -1 },
  { "vstem", 3, -1 },
  { "vstem3", 12, 1 },
  { "vstemhm", 23, -1 },            /* Type 2 */
  { "vvcurveto", 26, -1 },          /* Type 2 */
};                                                /* alphabetical */

/*
 * Two separate encryption functions because eexec and charstring
 * encryption must proceed in parallel.
 */

/* -----------------------------------------------------------  eencrypt
 *
 * eexec encryption
 */
static byte
eencrypt(byte plain)
{
  byte cipher;

  cipher = (byte)(plain ^ (er >> 8));
  er = (uint16)((cipher + er) * c1 + c2);
  return cipher;
}                                                         /* eencrypt */

/* -----------------------------------------------------------  cencrypt
 *
 * CharString encryption
 */
static byte
cencrypt(byte plain)
{
  byte cipher;

  cipher = (byte)(plain ^ (cr >> 8));
  cr = (uint16)((cipher + cr) * c1 + c2);
  return cipher;
}                                                         /* cencrypt */

/* -------------------------------------------------------  output_block
 *
 * flush a buffered PFB block
 */
static void
output_block( void )
{
  int32 i;

  /* output four-byte block length */
  fputc((int)(blockpos & 0xff), ofp);
  fputc((int)((blockpos >> 8) & 0xff), ofp);
  fputc((int)((blockpos >> 16) & 0xff), ofp);
  fputc((int)((blockpos >> 24) & 0xff), ofp);

  /* output block data */
  for (i = 0; i < blockpos; i++)
    fputc(blockbuf[i], ofp);

  /* mark block buffer empty and uninitialized */
  blockpos =  -1;
}                                                     /* output_block */

/* --------------------------------------------------------  output_byte
 *
 * This function outputs a single byte.  If output is in PFB format then
 * output is buffered through blockbuf[].  If output is in PFA format,
 * then output will be hexadecimal if in_eexec is set, ASCII otherwise.
 */
static void
output_byte(byte b)
{
  static char *hexchar = "0123456789ABCDEF";
  static int hexcol = 0;

  if (pfb) {
    /* PFB */
    if (blockpos < 0) {
      fputc(MARKER, ofp);
      fputc(blocktyp, ofp);
      blockpos = 0;
    }
    blockbuf[blockpos++] = b;
    if (blockpos == blocklen)
      output_block();
  } else {
    /* PFA */
    if (in_eexec) {
      /* trim hexadecimal lines to 64 columns */
      if (hexcol >= 64) {
        fputc('\n', ofp);
        hexcol = 0;
      }
      fputc(hexchar[(b >> 4) & 0xf], ofp);
      fputc(hexchar[b & 0xf], ofp);
      hexcol += 2;
    } else {
      fputc(b, ofp);
    }
  }
}                                                      /* output_byte */

/* ---------------------------------------------------------  eexec_byte
 *
 * This function outputs a byte through possible eexec encryption.
 */
static void
eexec_byte(byte b)
{
    output_byte( in_eexec ? eencrypt(b) : b );
}                                                       /* eexec_byte */

/* -------------------------------------------------------  eexec_string
 *
 * This function outputs a null-terminated string through possible eexec
 * encryption.
 */
static void
eexec_string(char *string)
{
  while (*string)
    eexec_byte((byte) *string++);
}                                                     /* eexec_string */

/* --------------------------------------------------------  eexec_start
 *
 * This function gets ready for the eexec-encrypted data.  If output is
 * in PFB format then flush current ASCII block and get ready for binary
 * block.  We start encryption with four random (zero) bytes.
 */
static void
eexec_start( void )
{
  eexec_string(line);
  if (pfb) {
    output_block();
    blocktyp = BINARY;
  }

  in_eexec = 1;
  er = 55665;
  eexec_byte(0);
  eexec_byte(0);
  eexec_byte(0);
  eexec_byte(0);
}                                                      /* eexec_start */

/* ------------------------------------------------------------- getline
 *
 * This function returns an input line of characters.  A line is
 * terminated by length (including terminating null) greater than
 * LINESIZE, a newline \n, or when active (looking for charstrings) by
 * '{'.  When terminated by a newline the newline is put into line[].
 * When terminated by '{', the '{' is not put into line[], and the flag
 * start_charstring is set to 1.
 */
static void
getline( void )
{
  int c;
  char *p = line;
  int comment = 0;

  start_charstring = 0;
  while (p < line + LINESIZE - 1) {
    c = fgetc(ifp);
    if (c == EOF)
      break;
    if (c == '%')
      comment = 1;
    if (active && !comment && c == '{') {
      start_charstring = 1;
      break;
    }
    *p++ = (char) c;
    if (c == '\n')
      break;
  }  /* while */

  /* change CR/LF to LF */
  if (p[-2] == '\r') {
    p[-2] = '\n';
    --p;
  }
  *p = '\0';
}                                                          /* getline */

/* ----------------------------------------------------------- eexec_end
 *
 * This function wraps-up the eexec-encrypted data and writes ASCII
 * trailer.  If output is in PFB format then this entails flushing
 * binary block and starting an ASCII block.
 */
static void
eexec_end ( void )
{
  int i, j;

  if (pfb) {
    output_block();
    blocktyp = ASCII;
  } else {
    fputc('\n', ofp);
  }

  in_eexec = active = 0;

  for (i = 0; i < 8; i++) {
    for (j = 0; j < 64; j++)
      eexec_byte('0');
    eexec_byte('\n');
  }
  eexec_string("cleartomark\n");

  /* There may be additional code. */
  while (!feof(ifp) && !ferror(ifp)) {
    getline();
    eexec_string(line);
  }

  if (pfb) {
    output_block();
    fputc(MARKER, ofp);
    fputc(DONE, ofp);
  }
}                                                        /* eexec_end */

/* ----------------------------------------------------- command_compare
 *
 * This function is used by the binary search, bsearch(), for command
 * names in the command table.
 */
static int
command_compare(const void *key, const void *item)
{
  return strcmp((char *) key, ((struct command *) item)->name);
}                                                  /* command_compare */

/* ---------------------------------------------------------  is_integer
 *
 * This function returns 1 if the string is an integer and 0 otherwise.
 */
static int
is_integer(char *string)
{
  if ( !(isdigit(string[0]) || string[0] == '-' || string[0] == '+') ) {
    return 0;
  }
  while (*++string && isdigit(*string))
    ;   /* empty body */
  return *string ? 0 : 1 ;
}                                                       /* is_integer */

/* ---------------------------------------------------  charstring_start
 *
 * This function initializes charstring encryption.  Note that this is
 * called at the beginning of every charstring.
 */
static void
charstring_start( void )
{
  int i;

  charstring_bp = charstring_buf;
  cr = 4330;
  for (i = 0; i < lenIV; i++)
    *charstring_bp++ = cencrypt((byte) 0);
}                                                 /* charstring_start */


/* ----------------------------------------------------  charstring_byte
 *
 * This function encrypts and buffers a single byte of charstring data.
 */
static void
charstring_byte(int v)
{
  byte b = (byte)(v & 0xff);

  if (charstring_bp - charstring_buf > sizeof(charstring_buf)) {
    fprintf(stderr, "error: charstring_buf full (%d bytes)\n",
        sizeof(charstring_buf));
    exit(1);
  }
  *charstring_bp++ = cencrypt(b);
}                                                  /* charstring_byte */


/* -----------------------------------------------------  charstring_end
 *
 * This function outputs buffered, encrypted charstring data through
 * possible eexec encryption.
 */
static void
charstring_end( void )
{
  byte *bp;

  sprintf(line, "%d ", charstring_bp - charstring_buf);
  eexec_string(line);
  sprintf(line, "%s ", cs_start);
  eexec_string(line);
  for (bp = charstring_buf; bp < charstring_bp; bp++)
    eexec_byte(*bp);
}                                                   /* charstring_end */


/* -----------------------------------------------------  charstring_int
 *
 * This function generates the charstring representation of an integer.
 */
static void
charstring_int(int num)
{
  int x;

  if (num >= -107 && num <= 107) {
    charstring_byte(num + 139);
  } else if (num >= 108 && num <= 1131) {
    x = num - 108;
    charstring_byte(x / 256 + 247);
    charstring_byte(x % 256);
  } else if (num >= -1131 && num <= -108) {
    x = abs(num) - 108;
    charstring_byte(x / 256 + 251);
    charstring_byte(x % 256);
    /* can handle the Type 2 'shortint' here if necessary */
  } else if (num >= (-2147483647-1) && num <= 2147483647) {
    charstring_byte(255);
    charstring_byte(num >> 24);
    charstring_byte(num >> 16);
    charstring_byte(num >> 8);
    charstring_byte(num);
  } else {
    fprintf(stderr,
        "error: cannot format the integer %d, too large\n", num);
    exit(1);
  }
}                                                  /* charstring_int */


/* ---------------------------------------------------  parse_charstring
 *
 * This function parses an entire charstring into integers and commands,
 * outputting bytes through the charstring buffer.
 */
static void
parse_charstring( void )
{
  struct command *cp;

  charstring_start();
  while (fscanf(ifp, "%s", line) == 1) {
    if (line[0] == '%') {
      /* eat comment to end of line */
      while (fgetc(ifp) != '\n' && !feof(ifp))
        ;    /* empty body */
      continue;
    }
    if (line[0] == '}')
      break;
    if (is_integer(line)) {
      charstring_int(atoi(line));
    } else {
      cp = (struct command *)
      bsearch((void *) line, (void *) command_table,
              sizeof(command_table) / sizeof(struct command),
              sizeof(struct command),
              command_compare);
      if (cp) {
        charstring_byte(cp->one);
        if (cp->two >= 0)
          charstring_byte(cp->two);
      } else {
        fprintf(stderr, "error: cannot use `%s' in charstring\n");
        exit(1);
      }
    }
  }    /* while */
  charstring_end();
}                                                 /* parse_charstring */

/* --------------------------------------------------------------  usage
 *
 * print usage message
 */
static void
usage( void )
{
  fprintf(stderr,
    "usage: t1asm [-b] [-l block-length] [input [output]]\n"
    "\n-b means output in PFB format, otherwise PFA format.\n"
    "The block length applies to the length of blocks in the\n"
    "PFB output file; the default is to use the largest possible.\n");
  exit(1);
}                                                            /* usage */

/* -------------------------------------------------------  print_banner
 *
 * print banner
 */
static void
print_banner( void )
{
  static char rcs_revision[] = "$Revision: 1.4 $";
  static char revision[20];

  if (sscanf(rcs_revision, "$Revision: %19s", revision) != 1)
    revision[0] = '\0';
  fprintf(stderr, "This is t1asm %s (modified).\n", revision);
}                                                     /* print_banner */

/* ---------------------------------------------------------------  main
 *
 * main program
 */
int
main(int argc, char **argv)
{
  char *p, *q, *r;
  int c;

  extern char *optarg;
  extern int optind;
  extern int getopt(int argc, char **argv, char *optstring);

  print_banner();

  /* interpret command line arguments using getopt */
  while ((c = getopt(argc, argv, "bl:")) != -1)
    switch (c) {
    case 'b':
      pfb = 1;
      break;
    case 'l':
      blocklen = atoi(optarg);
      if (blocklen < MINBLOCKLEN) {
        blocklen = MINBLOCKLEN;
        fprintf(stderr, "warning: using minimum block length of %d\n",
                blocklen);
      } else if (blocklen > MAXBLOCKLEN) {
        blocklen = MAXBLOCKLEN;
        fprintf(stderr, "warning: using maximum block length of %d\n",
                blocklen);
      }
      break;
    default:
      usage();
      break;
    }    /* switch */

  if (argc - optind > 2)
    usage();

  /* possibly open input & output files */
  if (argc - optind >= 1) {
    ifp = fopen(argv[optind], "r");
    if (!ifp) {
      fprintf(stderr, "error: cannot open %s for reading\n", argv[1]);
      exit(1);
    }
  }
  if (argc - optind >= 2) {
    ofp = fopen(argv[optind + 1], "w");
    if (!ofp) {
      fprintf(stderr, "error: cannot open %s for writing\n", argv[2]);
      exit(1);
    }
  }

#ifdef _MSDOS
  /*
   * If we are processing a PFB (binary) output file, we must set its
   * file mode to binary.
   */
  if (pfb)
    _setmode(_fileno(ofp), _O_BINARY);
#endif  /* _MSDOS */

  /*
   * Finally, we loop until no more input.  Some special things to look
   * for are the `currentfile eexec' line, the beginning of the `/Subrs'
   * definition, the definition of `/lenIV', and the definition of the
   * charstring start command which has `...string currentfile...' in
   * it.
   */

  while (!feof(ifp) && !ferror(ifp)) {
    getline();
    if (strcmp(line, "currentfile eexec\n") == 0) {
      eexec_start();
      continue;
    } else if (strstr(line, "/Subrs") && isspace(line[6])) {
      active = 1;
    } else if ((p = strstr(line, "/lenIV"))) {
      sscanf(p, "%*s %d", &lenIV);
    } else if ((p = strstr(line, "string currentfile"))) {
      /* locate the name of the charstring start command */
      *p = '\0';                                  /* damage line[] */
      q = strrchr(line, '/');
      if (q) {
        r = cs_start;
        ++q;
        while (!isspace(*q) && *q != '{')
            *r++ = *q++;
        *r = '\0';
      }
      *p = 's';                                   /* repair line[] */
    } else if (strstr(line, "mark currentfile closefile")) {
      eexec_string(line);
      break;
    }
    /* output line data */
    eexec_string(line);
    if (start_charstring) {
      if (!cs_start[0]) {
        fprintf(stderr,
                "error: couldn't find charstring start command\n");
        exit(1);
      }
      parse_charstring();
    }
  }    /* while */
  eexec_end();

  fclose(ifp);
  fclose(ofp);

  return 0;
}                                                             /* main */
