%x incl
%x define
%x definition
%x title

%{
/* lexical analyser for gpasm
   Copyright (C) 1998 James Bowman

This file is part of gpasm.

gpasm is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

gpasm is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with gpasm; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"

#include "gpasm.h"
#include "gpasm.tab.h"
#include "scan.h"
#include "gperror.h"
#include "symbol.h"
#include "lst.h"

#define OPERATOR(x)  return (yylval.i = (x))

static void push_string(char *str);
static int final();
static struct symbol *current_definition;	/* Used in #define */
static int quoted; /* Used to prevent #define expansion in ifdef and ifndef... */

%}

IDENT  [A-Za-z_][A-Za-z_0-9]*
ESCCH  \\([abfnrtv\\?'"]|0[0-7]{2}|x[0-9a-fA-F]{2})
STR_QCHAR  ([^"\n]|{ESCCH})

%%
^[ \t]*include[ \t]*[<"]? { BEGIN(incl); }
<incl>[^<"\r\n]*[>"]?	 { /* got the include file name */
			   char *pc = &yytext[yyleng - 1];
			   if ((*pc == '"') || (*pc == '>'))
			     *pc = '\0';
			   open_file(yytext);
			   BEGIN(INITIAL);
			 }
<<EOF>>		 	 {
			   if (final())
			     yyterminate();

			 }
end|END		         {
			   if (final())
			     yyterminate();
			 }
^[ \t]*title[ \t]*[<"]?  { BEGIN(title); }
<title>[^<"\r\n]*[>"]?   { /* got the title text */
#define LEN sizeof(state.lst.title_name)
                           yytext[yyleng - 1] = '\0';
                           strncpy(state.lst.title_name, yytext, LEN - 1);
                           state.lst.title_name[LEN - 1] = '\0';
                           BEGIN(INITIAL);
#undef LEN
                         }
cblock|CBLOCK		 {
			   return CBLOCK;
			 }
endc|ENDC		 {
			   return ENDC;
			 }
^#define[ \t]* 		 {
                           BEGIN(define);
			 }
<define>{IDENT}		 {
                           current_definition = add_symbol(state.stDefines, yytext);
                           BEGIN(definition);
			 }
<definition>.*$          {
			   /* Should have a #define in progress */
			   assert(current_definition != NULL);
			   annotate_symbol(current_definition, strdup(yytext));
			   current_definition = NULL;
                           BEGIN(INITIAL);
                         }
high|HIGH		 {
			   yylval.i = HIGH;
			   return HIGH;
			 }
low|LOW		     	 {
			   yylval.i = LOW;
			   return LOW;
			 }
ifdef|IFDEF		 {
			     quoted = 1;
                             yylval.s = strdup(yytext);
			     return IDENTIFIER;
			 }
ifndef|IFNDEF		 {
			     quoted = 1;
                             yylval.s = strdup(yytext);
			     return IDENTIFIER;
			 }
^{IDENT}:?		 { 
			   if (yytext[strlen(yytext) - 1] == ':')
			     yytext[strlen(yytext) - 1] = '\0';
			   yylval.s = strdup(yytext);
			   return LABEL;
			 }
{IDENT}			 {
			   struct symbol *sym;

			   /* If not quoted, check for #define substitution */
			   if (!quoted &&
                               (sym = get_symbol(state.stDefines, yytext)) != NULL) {
			     char *subst = get_symbol_annotation(sym);
			     assert(subst != NULL);
			     /* Make the substitution */
			     push_string(subst);
			   } else {
                             yylval.s = strdup(yytext);
			     return IDENTIFIER;
			   }
			 }
1[26][CcFf][0-9][0-9][0-9]? {
/* Ugh.  As a special case, treat processor names, such as 16C84
as identifiers rather than as hex numbers. */
                           yylval.s = strdup(yytext);
			   return IDENTIFIER;
			 }
[DdHhOoBb]'-?[0-9a-fA-F]+' {
			   yylval.i = gpasm_number(yytext);
			   return NUMBER;
			 }
(0[Xx])?[0-9a-fA-F]+	 {
                           char *endptr;

    			   if ((yytext[0] == '0') &&
			       (tolower(yytext[1]) == 'x'))
			     yylval.i = strtol(yytext + 2, &endptr, 16);
			   else
			     yylval.i = strtol(yytext, &endptr, state.radix);
                           if ((endptr == NULL) || (*endptr != '\0')) {
    			     char complaint[80];
			 
    			     sprintf(complaint,
				     isprint(*endptr) ?
				     "Illegal character '%c' in numeric constant" :
				     "Illegal character %#x in numeric constant",
				     *endptr);
                             gperror(113, complaint);
                           }
			   return NUMBER;
			 }
[0-9a-fA-F]+[hH]         {
                           char *endptr;

                           yytext[yyleng - 1] = '\0';
                           yylval.i = strtol(yytext, &endptr, 16);
                           if ((endptr == NULL) || (*endptr != '\0')) {
                             char complaint[80];

                             sprintf(complaint,
                                     isprint(*endptr) ?
                                     "Illegal character '%c' in numeric constant " :
                                     "Illegal character %#x in numeric constant" ,
                                     *endptr);
                             gperror(113, complaint);
                           }
                           return NUMBER;
                         }
\"{STR_QCHAR}*\"?        {
			   char *pc = &yytext[yyleng - 1];
			   if (*pc == '"')
			     *pc = '\0';
                           yylval.s = strdup(yytext + 1);
			   return STRING;
                         }
'{STR_QCHAR}'            {
                           yylval.i = yytext[1];
			   return NUMBER;
                         }
A'{STR_QCHAR}'           {
                           yylval.i = yytext[2];
			   return NUMBER;
                         }
"<<"			 OPERATOR(LSH);
">>"			 OPERATOR(RSH);
">="			 OPERATOR(GREATER_EQUAL);
"<="			 OPERATOR(LESS_EQUAL);
"=="			 OPERATOR(EQUAL);
"!="			 OPERATOR(NOT_EQUAL);
"&&"			 OPERATOR(LOGICAL_AND);
"||"			 OPERATOR(LOGICAL_OR);
"+="			 OPERATOR(ASSIGN_PLUS);
"-="			 OPERATOR(ASSIGN_MINUS);
"*="			 OPERATOR(ASSIGN_MULTIPLY);
"/="			 OPERATOR(ASSIGN_DIVIDE);
"%="			 OPERATOR(ASSIGN_MODULUS);
"<<="			 OPERATOR(ASSIGN_LSH);
">>="			 OPERATOR(ASSIGN_RSH);
"&="			 OPERATOR(ASSIGN_AND);
"|="			 OPERATOR(ASSIGN_OR);
"^="			 OPERATOR(ASSIGN_XOR);
"++"			 OPERATOR(INCREMENT);
"--"			 OPERATOR(DECREMENT);
[ \t\r]*
[\n]		         {
			   quoted = 0;
 			   return yytext[0];
			 }
;.*			 {  }
.			 { 
			   yylval.i = yytext[0];
		           return yytext[0];
			 }
%%

void open_file(char *name)
{
  extern FILE *yyin;
  struct source_context *new = malloc(sizeof(*new));

  if (state.src)
    state.src->yybuf = YY_CURRENT_BUFFER;

  new->name = strdup(name);
  new->f = fopen(name, "r");
  new->f2 = fopen(name, "r");

  yyin = new->f;
  if (new->f == NULL) {
    if (state.src) {
      char complaint[BUFSIZ];

      sprintf(complaint,
	      "Unable to open file \"%s\" - %s",
	      name,
	      strerror(errno));
      state.pass = 2; /* Ensure error actually gets displayed */
      gperror(115, complaint);
    } else {
      perror(name);
    }
    exit(1);
  }

  if (state.src) {
    yy_switch_to_buffer(yy_create_buffer(yyin, YY_BUF_SIZE));
  }

  new->line_number = 1;	/* Files start at line 1, for some reason */
  new->prev = state.src;
  state.src = new;
}

static void push_string(char *str)
{
  extern FILE *yyin;
  struct source_context *new = malloc(sizeof(*new));

  assert(state.src != NULL);
  state.src->yybuf = YY_CURRENT_BUFFER;

  new->name = state.src->name;
  new->line_number = state.src->line_number;
  new->f = NULL;
  new->f2 = NULL;

  yy_scan_string(str);

  new->prev = state.src;
  state.src = new;
}

void close_file()
{
  struct source_context *old;

  old = state.src;
  state.src = state.src->prev;
  if (old->f != NULL) {
    fclose(old->f);
    fclose(old->f2);
    free(old->name);
  }
  free(old);
}

static int final()
{
  int terminate = 0;

  if (strcmp(state.src->name, SPECIAL_PATH) == 0) {
    close_file();
    open_file(state.srcfilename);
    yyrestart(yyin);
  } else {
    close_file();
    if (state.src) {
      /* Just an include file */
      yy_delete_buffer(YY_CURRENT_BUFFER);
      yy_switch_to_buffer(state.src->yybuf);
    } else {
      if (state.pass == 1) {
	/* End of first pass - rewind and start second pass */
	open_file(state.srcfilename);
	yyrestart(yyin);
	state.pass++;
	state.org = 0;
        state.cblock = 0;
        lst_init();
      } else { 
	/* End of second pass - done */
	terminate = 1;
      }
    }
  }

  return terminate;
}
