/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <stdio.h>
#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-pixmap.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-dialog-util.h>

#include "article.h"
#include "article-db.h"
#include "articlelist.h"
#include "article-thread.h"
#include "article-toolbar.h"
#include "grouplist.h"
#include "fnmatch.h"
#include "globals.h"
#include "group.h"
#include "gui.h"
#include "gui-paned.h"
#include "log.h"
#include "message-filter.h"
#include "message-window.h"
#include "prefs.h"
#include "queue.h"
#include "status-item.h"
#include "queue-item-body.h"
#include "queue-item-decode.h"
#include "queue-item-headers.h"
#include "text.h"
#include "util.h"

#include "xpm/book_open.xpm"
#include "xpm/book_closed.xpm"
#include "xpm/mini_page.xpm"
#include "xpm/greencheck.xpm"
#include "xpm/bluecheck.xpm"
#include "xpm/x.xpm"
#include "xpm/binary.xpm"
#include "xpm/binary_complete.xpm"
#include "xpm/binary_incomplete.xpm"
#include "xpm/unread_children.xpm"


/**********************************************************
 * Private Variables
 **********************************************************/

/* state */
static group_data *my_group = NULL;
static server_data *my_server = NULL;
static article_data** article_headers = NULL;
static int article_header_qty = 0;
static gboolean is_threaded = TRUE;
static int sort_type = ARTICLE_SORT_SUBJECT;
static pthread_mutex_t article_ctree_lock = PTHREAD_MUTEX_INITIALIZER;
static GHashTable *messageid_to_node = NULL;


/* by holding a reference to the active group, we can avoid a lot of open/close
 * db calls as the user reads articles.  We don't * actually do anything with
 * it here, but we know that most of the article headers being updated in the
 * db will be through the article list. */
static article_db active_group_header_db = NULL;

/* article list */
static GnomePixmap *articlelist_closed_pixmap = NULL;
static GnomePixmap *articlelist_open_pixmap = NULL;
static GnomePixmap *decode_queue_pixmap = NULL;
static GnomePixmap *download_queue_pixmap = NULL;
static GnomePixmap *mp_complete_pixmap = NULL;
static GnomePixmap *mp_incomplete_pixmap = NULL;
static GnomePixmap *error_pixmap = NULL;
static GnomePixmap *article_read_pixmap = NULL;
static GnomePixmap *binary_decoded_pixmap = NULL;
static GnomePixmap *unread_children_pixmap = NULL;

/**********************************************************
 * Public Variables
 **********************************************************/

PanCallback* articlelist_state_filter_changed = NULL;
PanCallback* articlelist_group_changed = NULL;
PanCallback* articlelist_sort_changed = NULL;
PanCallback* articlelist_thread_changed = NULL;
PanCallback* articlelist_selection_changed = NULL;


GdkColormap *cmap;
GdkColor new_color;
GdkColor read_color;
GdkColor unread_color;
GdkColor killfile_color;
GdkColor watched_color;

article_data *current_article = NULL;
GtkWidget *article_ctree_menu;

GtkStyle *killfile_style = NULL;
GtkStyle *watched_style = NULL;
GtkStyle *unread_style = NULL;
GtkStyle *read_style = NULL;
GtkStyle *new_style = NULL;

gboolean hide_mpart_child_nodes = TRUE;

/**********************************************************
 * Private Functions
 **********************************************************/

static gboolean article_passes_filter (const article_data*);
static void apply_filter_tests (void);

static GtkCTreeNode* articlelist_get_node_from_message_id (
	const char* message_id);

static void articlelist_repopulate (
	const group_data*,
	article_data**,
	int qty,
	StatusItem*);

static void build_article_ctree_recursive (
	GtkCTree*,
	GtkCTreeNode*,
	GSList*,
	StatusItem*,
	int);

static GnomePixmap* get_article_pixmap (
	const article_data*);

static void articlelist_button_press (
	GtkWidget*,
	GdkEventButton*);

static void articlelist_click_column (
	GtkCList*,
	int col_clicked);

static void article_ctree_destroy_cb (
	void);

static void articlelist_update_node_fast (
	GtkCTreeNode*,
	const article_data* );



/**********************************************************
 * Articlelist popup menu
 **********************************************************/

static GnomeUIInfo articlelist_menu_popup[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Open in Window"),
		N_("Open this message for reading."),
		message_read_window
	},
	GNOMEUIINFO_SEPARATOR,
        {
		GNOME_APP_UI_ITEM,
		N_("_Followup to newsgroup"),
		N_("Post a reply to the message on the news server."),
		message_followup_window },
	{
		GNOME_APP_UI_ITEM,
		N_("Reply by _E-mail"),
		N_("Create a mail reply to the sender."),
		message_reply_window
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Save Attachment"),
		N_("Save Attachments."),
		articlelist_selected_decode,
	  	NULL, NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'D', GDK_CONTROL_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Save Attachment As..."),
		N_("Download and attachments in the selected message, and save as..."),
		articlelist_selected_decode_as
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Open Attachment"),
		N_("Open Attachments."),
		articlelist_selected_open,
		NULL, NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'O', GDK_CONTROL_MASK, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Download Entire Thread"),
		N_("Download Entire Thread."),
		articlelist_selected_thread_download
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Delete Message"),
		N_("Delete message from database."),
		articlelist_selected_delete
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Delete Thread"),
		N_("Delete Thread from database."),
		articlelist_selected_delete_thread
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Add thread to Killfile"),
		N_("Add thread to Killfile"),
		articlelist_selected_thread_kill
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Track Thread"),
		N_("Track Thread"),
		articlelist_selected_thread_track
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Mark as Read"),
		N_("Marks current message as read."),
		articlelist_selected_mark_read
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Mark as Unread"),
		N_("Marks the current message as unread."),
		articlelist_selected_mark_unread
	},
	GNOMEUIINFO_SEPARATOR,
        {
		GNOME_APP_UI_ITEM,
		N_("Mark All Articles Read"),
		N_("Marks the entire group as read."),
		articlelist_all_mark_read
	},
	GNOMEUIINFO_END
};

static const char*
articlelist_get_row_message_id (int i)
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode* node = gtk_ctree_node_nth (tree, i);
	const article_data* adata = NULL;
	const gchar* message_id = NULL;

	if (node)
		adata = articlelist_get_adata_from_node (node);
	if (adata)
		message_id = adata->message_id;

	return message_id;
}

article_data*
articlelist_get_adata_from_node (GtkCTreeNode* node)
{
	article_data* adata = NULL;

	if (node)
		adata = ARTICLE_DATA(
				gtk_ctree_node_get_row_data (
					GTK_CTREE(Pan.article_ctree), node));

	return adata;
}

GtkCTreeNode*
articlelist_get_selected_node (void)
{
	GtkCTreeNode* node = NULL;

	const GList* list =  GTK_CLIST(Pan.article_ctree)->selection;
	if ( list != NULL )
		node = GTK_CTREE_NODE(list->data);

	return node;
}

article_data*
articlelist_get_selected_adata (void)
{
	return articlelist_get_adata_from_node (
		articlelist_get_selected_node());
}

static void
articlelist_unselect_all (void)
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);

	pan_lock();
	gtk_clist_freeze (GTK_CLIST(tree));
	gtk_ctree_unselect_recursive (tree, gtk_ctree_node_nth(tree, 0));
	gtk_clist_thaw (GTK_CLIST(tree));
	pan_unlock();
}

void
articlelist_set_selected_node (GtkCTreeNode *node)
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode* n = NULL;

	pan_lock();
	gtk_clist_freeze (GTK_CLIST(tree));
	g_assert ( node!=NULL );
	gtk_ctree_unselect_recursive (tree, gtk_ctree_node_nth(tree, 0));
	for (n=GTK_CTREE_ROW(node)->parent; n; n=GTK_CTREE_ROW(n)->parent)
		gtk_ctree_expand(tree, n);
	gtk_ctree_select (tree, node);
	gtk_clist_thaw (GTK_CLIST(tree));
	gtk_ctree_node_moveto (tree, node, 0, 0.5, 0);
	pan_unlock();
}

static void
articlelist_select_and_read_node (GtkCTreeNode *node)
{
	article_data *adata = NULL;

	if (node!=NULL)
		adata = articlelist_get_adata_from_node (node);
	if (adata!=NULL)
	{
		articlelist_set_selected_node(node);
		queue_add(QUEUE_ITEM(queue_item_body_new(
			my_server, my_group, message_new(my_server, my_group, adata))));
	}
}

void
articlelist_set_selected_message_id (const char* message_id)
{
	GtkCTreeNode* node =
		articlelist_get_node_from_message_id (message_id);
	g_return_if_fail (node!=NULL);
	articlelist_set_selected_node (node);
}

const char*
articlelist_get_selected_message_id (void)
{
	const article_data* adata = articlelist_get_selected_adata();
	return adata ? adata->message_id : NULL;
}

GtkCTreeNode*
articlelist_node_next (GtkCTreeNode *node, gboolean sibling_only)
{
	GtkCTreeNode* n;

	if (!node)
		return NULL;

	if (sibling_only)
		return GTK_CTREE_ROW(node)->sibling;

	if (GTK_CTREE_ROW(node)->children)
		return GTK_CTREE_ROW(node)->children;

	for (n=node; n!=NULL; n=GTK_CTREE_ROW(n)->parent)
		if (GTK_CTREE_ROW(n)->sibling)
			return GTK_CTREE_ROW(n)->sibling;

	return NULL;
}


GtkCTreeNode*
articlelist_node_prev (GtkCTreeNode *node, gboolean sibling_only)
{
	GtkCTreeNode *parent;
	GtkCTreeNode *sibling;
	
	if (!node)
		return NULL;

	/* get parent */	
	parent = GTK_CTREE_ROW(node)->parent;
	if (!parent)
		return NULL;

	/* parent's first child */
	sibling=GTK_CTREE_ROW(parent)->children;
	if (sibling==node) /* this is the first child */
		return parent;

	/* previous sibling of node */
	while (GTK_CTREE_ROW(sibling)->sibling != node) {
		sibling = GTK_CTREE_ROW(sibling)->sibling;
		g_assert (sibling != NULL);
	}

	if (sibling_only)
		return sibling;

	/* find the absolutely last child of the older sibling */
	for (;;) {
		GtkCTreeNode* tmp = GTK_CTREE_ROW(sibling)->children;
		if (!tmp)
			return sibling;

		/* last child of sibling */
		while (GTK_CTREE_ROW(tmp)->sibling != NULL)
			tmp = GTK_CTREE_ROW(tmp)->sibling;

		sibling = tmp;
	}

	pan_warn_if_reached ();
}

static const article_data*
articlelist_get_relative_article (const char* message_id, int step)
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode *node=NULL;
	const int row_qty = GTK_CLIST(tree)->rows;

	if ( row_qty<2 )
		return NULL;

	/* get working node */
	node = articlelist_get_node_from_message_id (message_id);
	g_assert ( node!=NULL );

	/* Find the requested node... */
	while ( step > 0 ) {
		--step;
		node = articlelist_node_next (node, FALSE);
	}
	while ( step < 0 ) {
		++step;
		node = articlelist_node_prev (node, FALSE);
	}


	/* get step node */
	return articlelist_get_adata_from_node (node);
}


const article_data*
articlelist_get_next_article (void)
{
	GtkCTree *tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode *node = NULL;

	/* sanity checks */
	g_return_val_if_fail (!my_server, NULL);
	g_return_val_if_fail (!my_group, NULL);

	/* get next node */
	node = articlelist_get_selected_node();
	if (!node)
		node = gtk_ctree_node_nth (tree, 0);
	node = articlelist_node_next (node, FALSE);
	return articlelist_get_adata_from_node (node);
}

void
articlelist_read_next_article(void)
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode* node;

	/* if nothing to read... */
	if (GTK_CLIST(tree)->rows <2)
		return;

	/* get starting point */
	node = articlelist_get_selected_node ();
	if (!node)
		node = gtk_ctree_node_nth (tree, 0);

	/* move to next node */
	node = articlelist_node_next (node, FALSE);

	/* read it */
	if ( node!=NULL )
		articlelist_select_and_read_node (node);
	else {
		articlelist_unselect_all ();
		articlelist_read_next_article();
	}
}
void
articlelist_read_next_unread_article (void)
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode* node = NULL;
	GtkCTreeNode* start = NULL;
	gboolean all_read = FALSE;

	/* get first node */	
	node = articlelist_get_selected_node ();
	if (!node)
		node = gtk_ctree_node_nth (tree, 0);

	/* move to next unread node */
	all_read = FALSE;
	start = node;
	for ( ;; ) {
		const article_data* adata =
			articlelist_get_adata_from_node (node);
		if (adata && !article_flag_on(adata, STATE_READ))
			break;
		node = articlelist_node_next(node, FALSE);
		if (!node) /* wrap */
			node = gtk_ctree_node_nth (tree, 0);
		all_read = node == start;
		if (all_read)
			break;
	}

	if (!all_read)
	{
		/* select it */
		if ( node!=NULL )
			articlelist_select_and_read_node (node);
		else {
			articlelist_unselect_all ();
			articlelist_read_next_unread_article();
		}
	}
}
void
articlelist_read_next_thread (void)
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode* top = gtk_ctree_node_nth (tree, 0);
	GtkCTreeNode* node = NULL;

	/* if nothing to read... */
	if (GTK_CLIST(tree)->rows<2)
		return;

	if (( node = articlelist_get_selected_node () ))
	{
		while (node!=top && GTK_CTREE_ROW(node)->parent!=top)
			node = GTK_CTREE_ROW(node)->parent;
		if (node==top)
			return;
		node = GTK_CTREE_ROW(node)->sibling;
	}
	else if (GTK_CLIST(tree)->rows > 1)
	{
		node = gtk_ctree_node_nth (tree, 1);
	}


	/* read it. */
	if (node!=NULL)
		articlelist_select_and_read_node (node);
	else {
		articlelist_unselect_all ();
		articlelist_read_next_thread ();
	}
}

void
articlelist_read_prev_article (void)
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode* top = gtk_ctree_node_nth (tree, 0);
	const int row_qty = GTK_CLIST(tree)->rows;
	GtkCTreeNode* node = NULL;
	gboolean wrap = FALSE;

	/* if nothing to read... */
	if (row_qty<2)
		return;

	/* get a starting node... */
	node = articlelist_get_selected_node (); 
	if ( !node || articlelist_node_prev(node,TRUE)==top ) {
		wrap = TRUE;
		node = gtk_ctree_node_nth (tree, row_qty-1 );
	}

	/* if we didn't wrap, we need to move backwards by hand */
	if ( !wrap )
		node = articlelist_node_prev (node, FALSE);

	/* read it. */
	if ( node!=NULL )
		articlelist_select_and_read_node (node);
}
void articlelist_read_prev_unread_article(void)
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode* top = gtk_ctree_node_nth (tree, 0);
	const int row_qty = GTK_CLIST(tree)->rows;
	GtkCTreeNode* node = NULL;
	GtkCTreeNode* start = NULL;
	gboolean all_read = FALSE;

	/* if nothing to read... */
	if (row_qty<2)
		return;

	/* get starting point */	
	node = articlelist_get_selected_node ();
	if (!node || articlelist_node_prev(node,TRUE)==top) {
		node = gtk_ctree_node_nth (tree, row_qty-1); /* wrap */
	} else
		node = articlelist_node_prev(node, FALSE);

	/* move to the previous unread article (if any) */
	start = node;
	all_read = FALSE;
	for (;;) {
		const article_data* adata =
			articlelist_get_adata_from_node (node);
		if (adata && !article_flag_on(adata, STATE_READ))
			break;
		node = articlelist_node_prev(node, FALSE);
		if (!node) /* wrap */
			node = gtk_ctree_node_nth (tree, row_qty-1);
		all_read = node == start;
		if (all_read)
			break;
	}

	/* read it */
	if (!all_read && node!=NULL)
		articlelist_select_and_read_node (node);
}
void articlelist_read_prev_thread(void)
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode* top = gtk_ctree_node_nth (tree, 0);
	const int row_qty = GTK_CLIST(tree)->rows;
	GtkCTreeNode* node = NULL;
	gboolean wrap = FALSE;

	if ( !row_qty )
		return;

	/* get starting point */	
	node = articlelist_get_selected_node ();
	if ( node )
		while ( node!=top && GTK_CTREE_ROW(node)->parent!=top )
			node = GTK_CTREE_ROW(node)->parent;

	/* either no node selected or we're at the first thread; wrap */
	if ( !node || articlelist_node_prev(node,TRUE)==top ) {
		wrap = TRUE;
		node = gtk_ctree_node_nth ( tree, row_qty-1 );
		while ( node!=top && GTK_CTREE_ROW(node)->parent!=top )
			node = GTK_CTREE_ROW(node)->parent;
	}

	/* if we didn't wrap around, we need to back up one. */
	if ( !wrap )
		node = articlelist_node_prev(node, TRUE);

	/* read it. */
	if ( node!=NULL )
		articlelist_select_and_read_node (node);
}


const article_data*
articlelist_get_previous_article ( void )
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);
	const char* message_id = NULL;

	/* get current message id */
	message_id = articlelist_get_selected_message_id ( );
	if (!message_id)
		message_id = articlelist_get_row_message_id (GTK_CLIST(tree)->rows-1);
	if (!message_id) return NULL;

	/* return the adata */
	return articlelist_get_relative_article (message_id, -1);
}

static void
articlelist_button_press (GtkWidget* widget, GdkEventButton* bevent)
{
	if (bevent->button != 1)
		gtk_signal_emit_stop_by_name (
			GTK_OBJECT(widget), "button_press_event");

	if (bevent->button == 1)
	{
		const article_data *a = NULL;

		if ((bevent->type==GDK_2BUTTON_PRESS) || 
		    (one_click_preview && Pan.viewmode==GUI_PANED))
		{
			int row=0, col=0;
			if (gtk_clist_get_selection_info (
				GTK_CLIST(Pan.article_ctree),
				bevent->x, bevent->y, &row, &col))
			{
				GtkCTreeNode *node = gtk_ctree_node_nth(
					GTK_CTREE(Pan.article_ctree), row);
				a = articlelist_get_adata_from_node (node);
			}
		}

		if (a)
		{
			queue_add(QUEUE_ITEM(queue_item_body_new(
				my_server, my_group,
				message_new(my_server,my_group,a))));

			/* this stops an article selection from also expanding
			   a subtree via double-click.  Use the hotspot... */
			if (bevent->type == GDK_2BUTTON_PRESS)
				gtk_signal_emit_stop_by_name (
					GTK_OBJECT(widget),
					"button_press_event");

			gui_page_set (MESSAGE_PAGE, Pan.text);
		}
	}
	else if ( bevent->button == 3 )
	{
		const gboolean sel = GTK_CLIST(Pan.article_ctree)->selection != NULL;
		const gboolean articles_available = GTK_CLIST(Pan.article_ctree)->rows != 0;

		GList* l = NULL;

		pan_lock();

		l=GTK_MENU_SHELL(article_ctree_menu)->children; /* open */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), sel); 

		l=l->next; /* separator */
		l=l->next; /* reply to sender */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), sel);
		l=l->next; /* reply to group */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), sel);

		l=l->next; /* separator */
		l=l->next; /* save attachment */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), sel);
		l=l->next; /* save attachment as */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), sel);
		l=l->next; /* open attachment */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), sel);

		l=l->next; /* separator */
		l=l->next; /* download entire thread */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), sel);

		l=l->next; /* separator */
		l=l->next; /* delete article */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), sel);
		l=l->next; /* delete thread */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), sel);

		l=l->next; /* separator */
		l=l->next; /* add to killfile */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), sel);
		l=l->next; /* add to watchfile */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), sel);

		l=l->next; /* separator */
		l=l->next; /* mark as read */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), sel);
		l=l->next; /* mark as unread */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), sel);

		l=l->next; /* separator */
		l=l->next; /* mark all read */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), articles_available);

		gtk_menu_popup (GTK_MENU (article_ctree_menu), NULL, NULL, NULL,
				NULL, bevent->button, bevent->time);
		pan_unlock();
	}
}


/*--------------------------------------------------------------------
 * clear the ctree of all messages
 *--------------------------------------------------------------------*/

static void
clear_article_buffer (void)
{
	if (article_header_qty)
	{
		int i;
		for (i=0; i<article_header_qty; ++i)
			article_free (article_headers[i]);
		g_free (article_headers);
		article_header_qty = 0;
	}
}

void
articlelist_clear (void)
{
	GtkCList *list = GTK_CLIST(Pan.article_ctree);
	const int row_qty = list->rows;

	g_hash_table_destroy (messageid_to_node);
	messageid_to_node = g_hash_table_new (g_str_hash, g_str_equal);

	if (row_qty)
	{
		pan_lock();
		gtk_clist_freeze (list);
		gtk_clist_clear (list);
		gtk_clist_thaw (list);
		pan_unlock();
	}

	if (active_group_header_db != NULL)
		ahdb_unref (active_group_header_db);
	my_group = NULL;
	my_server = NULL;
	pan_callback_call (articlelist_group_changed, Pan.article_ctree, my_group);

	clear_article_buffer();
}

/*--------------------------------------------------------------------
 * load article listing
 *--------------------------------------------------------------------*/
void
articlelist_load (
	server_data *sdata,
	group_data *gdata,
	StatusItem *item)
{
	article_db db = (article_db)0;
	GSList *alist = NULL;
	int total = 0;
	int loaded_total = 0;

	/* argument sanity check */
	g_return_if_fail (gdata != NULL);

	/* step 1: load article headers */

	total = group_get_attrib_i (sdata, gdata, "Total");
	status_item_emit_status (item, _("Loading article headers"));
	status_item_emit_init_steps (item, total*4);
	db = ahdb_ref (sdata, gdata);
	alist = ahdb_get_all (db, status_item_next_step_gfunc, item);
	ahdb_unref (db);
	/* update total because ahdb_get_all does article expiration. */
	total = group_get_attrib_i (sdata, gdata, "Total");

	/* check totals */
	loaded_total = g_slist_length (alist);
	if (total != loaded_total) {
		g_warning (
			_("Pan thought there were %d articles; instead we got %d"),
			total,
			loaded_total );
		group_set_attrib_i (sdata, gdata, "Total", total=loaded_total);
	}

	if (!alist)
	{
		choose_download_dialog (sdata, gdata);
	} 
	else
	{
		log_add_va (
			_("Loaded %d article headers for group %s"),
			total,
			gdata->name );
		articlelist_set_contents (sdata, gdata, alist, total, item);
	}

	status_item_emit_progress (item, 0);
}

static void
killfile_articles (
	article_data** buf,
	int article_qty,
	StatusItem *item )
{
	int i;

	g_return_if_fail (my_group!=NULL);
	g_return_if_fail (article_qty>=0);

	for (i=0; i!=article_qty; ++i)
	{
		article_data* adata = buf[i];

		const MessageFilterAction action =
			message_filter_get_action (my_group, adata);

		if (action == MESSAGE_FILTER_ACTION_KILL)
		{
			adata->state |= STATE_KILLFILE;
		}
		else if (action == MESSAGE_FILTER_ACTION_WATCH)
		{
			adata->state |= STATE_WATCHED;
		}
		else
		{
			adata->state &= ~(STATE_KILLFILE | STATE_WATCHED);
		}

		if (item != NULL)
			status_item_emit_next_step (item);
	}
}

void
articlelist_queue_reload (void)
{
	if (my_group != NULL)
	{
		queue_add (QUEUE_ITEM(queue_item_headers_new(
			(server_data*)my_server, TRUE,
			(group_data*)my_group,
			HEADERS_LOCAL)));
	}
}

static const gchar*
column_to_title (int col)
{
	switch (col) {
		case 0: return _("Subject");
		case 1: return _("New");
		case 2: return _("Lines");
		case 3: return _("Date");
		case 4: return _("Author");
		default: break;
	}
	
	pan_warn_if_reached();
	return _("BUG!!");
}

static int
sort_type_to_column (int type)
{
	int col = 0;
	switch (abs(type)) {
		case ARTICLE_SORT_SUBJECT: col=0; break;
		case ARTICLE_SORT_UNREAD_CHILDREN: col=1; break;
		case ARTICLE_SORT_LINES: col=2; break;
		case ARTICLE_SORT_DATE: col=3; break;
		case ARTICLE_SORT_AUTHOR: col=4; break;
		default: pan_warn_if_reached(); break;
	}

	return col;
}

static void
articlelist_set_sort_type_nosort (int type, gboolean force)
{
	gchar *pch = NULL;
	int col = 0;

	if (!force && (type==sort_type))
		return;

	/* update old column */
	if (abs(sort_type) != abs(type)) {
		col = sort_type_to_column (sort_type);
		pan_lock();
		gtk_clist_set_column_title (GTK_CLIST(Pan.article_ctree), col, column_to_title(col));
		pan_unlock();
	}

	/* update sort type, fire callbacks */
	sort_type = type;
	pan_callback_call (articlelist_sort_changed, Pan.article_ctree, GINT_TO_POINTER(type));

	/* update new column */
	col = sort_type_to_column (sort_type);
	pch = g_strdup_printf ("%c%s", (type>0?'+':'-'), column_to_title(col));
	pan_lock();
	gtk_clist_set_column_title (GTK_CLIST(Pan.article_ctree), col, pch);
	pan_unlock();
	g_free (pch);

}	


void
articlelist_set_contents (
	server_data* sdata,
	group_data* gdata,
	GSList* alist,
	int alist_total,
	StatusItem *item)
{
	int i;
	pthread_mutex_lock (&article_ctree_lock);

	/* get sorting information */
	i = group_get_attrib_i (sdata, gdata,"sort_style");
	if (!i || abs(i)>ARTICLE_SORT_TYPES_QTY)
		i = -ARTICLE_SORT_DATE;
	articlelist_set_sort_type_nosort (i, TRUE);

	/* update local state */
	my_server = sdata;
	my_group = gdata;
	if (active_group_header_db != NULL)
		ahdb_unref (active_group_header_db);
	if (my_group != NULL)
		active_group_header_db = ahdb_ref (sdata, gdata);

	pan_callback_call (articlelist_group_changed, Pan.article_ctree, my_group);

	/* thread */
	status_item_emit_status (item, _("Threading article list"));
	clear_article_buffer ();
	thread_articles (
		alist, alist_total,
		&article_headers, &article_header_qty, item);
	alist = NULL;

	/* killfile */
	status_item_emit_status (item, _("Applying Killfile"));
	killfile_articles (article_headers, article_header_qty, item);

	/* multipart state */
	check_multipart_articles (article_headers, article_header_qty);

	/* filter */
	apply_filter_tests ();

	/* sort */
	sort_articles (
		article_headers, article_header_qty,
		abs(sort_type), sort_type>0);

	/* add to tree */
	status_item_emit_status (item, _("Displaying article list"));
	articlelist_repopulate (gdata,
		(article_data**)article_headers, article_header_qty, item);

	pthread_mutex_unlock (&article_ctree_lock);
}

const server_data*
articlelist_get_current_server (void)
{
	return my_server;
}
group_data*
articlelist_get_current_group (void)
{
	return my_group;
}


//============================================================================


static void
articlelist_unread_inc (GtkCTreeNode* node, int inc)
{
	/* update the parents */
	while (node) {
		article_data *a = articlelist_get_adata_from_node (node);
		if (!a) /* top node */
			break;
		a->unread_children = MAX(a->unread_children+inc, 0);
		articlelist_update_node_fast (node, a);
		node = GTK_CTREE_ROW(node)->parent;
	}
}

static int
article_deleted_cb (
	gpointer call_obj,
	gpointer call_arg,
	gpointer client_data)
{
	int i;
	article_data* adata = ARTICLE_DATA(call_obj);
	GtkCTreeNode* node = NULL;

	/* entry assertions */
	g_return_val_if_fail (adata!=NULL, 0);

	/* if this node is in the tree, remove it */
	node = articlelist_get_node_from_message_id (adata->message_id);
	if (node != NULL) /* reparent children */
	{
		GSList* children = NULL;
		GtkCTreeNode *child = NULL;

		/* make a list of the children */
		for (child=GTK_CTREE_ROW(node)->children;
			child!=NULL;
			child=GTK_CTREE_ROW(child)->sibling)
		{
			children = g_slist_append (children, child);
		}

		/* reparent all the children to the parent */
		if (children!=NULL)
		{
			GSList *l = NULL;
			pan_lock();
			gtk_clist_freeze (GTK_CLIST(Pan.article_ctree));
			for (l=children; l!=NULL; l=l->next)
				gtk_ctree_move (
					GTK_CTREE(Pan.article_ctree),
					(GtkCTreeNode*)l->data,
					GTK_CTREE_ROW(node)->parent,
					GTK_CTREE_ROW(node)->sibling);
			gtk_clist_thaw (GTK_CLIST(Pan.article_ctree));
			pan_unlock();
			g_slist_free (children);
		}

		/* remove from hashtable */
		g_hash_table_remove (messageid_to_node, adata->message_id);

		/* don't remove the node yet; we need it
		   for article_unread_inc() below */
	}

	/* is this article in our article headers array? */
	for (i=0; i!=article_header_qty; ++i)
		if (!strcmp(adata->message_id, article_headers[i]->message_id))
			break;

	/* yes it is... */
	if (i!=article_header_qty)
	{
		/* if ours is a different copy, delete it */
		if (adata != article_headers[i])
		{
			article_isolate (article_headers[i]);
			article_free (article_headers[i]);
		}

		/* remove item from our array */
		array_shrink (
			article_headers, i,
			sizeof(article_data*), article_header_qty--);
	}

	/* if this article's unread, update the parents' unread children count */
	if (!article_flag_on (adata, STATE_READ)) {
		GtkCTreeNode *n = node;
		if (!n) {
			article_data* a;
			for (a=adata->parent; a!=NULL && !n; a=a->parent)
				n = articlelist_get_node_from_message_id (
					adata->message_id);
		}
		if (n!=NULL) 
			articlelist_unread_inc (n, -1);
	}

	/* remove this node from the tree */
	if (node) {
		pan_lock();
		gtk_ctree_remove_node (GTK_CTREE(Pan.article_ctree), node);
		pan_unlock();
	}

	return 0;
}




void
articlelist_selected_delete_thread (void)
{
	GtkCList* clist = GTK_CLIST(Pan.article_ctree);
	GList* list = g_list_copy (clist->selection);
	GList* l = NULL;

	/* if it buys us anything, lock the clist */
	if (list!=NULL) {
		pan_lock();
		gtk_clist_freeze (clist);
		pan_unlock();
	}

	for (l=list; l!=NULL; l=l->next)
	{
		article_data **articles = NULL;
		int article_qty = 0;
		int i = 0;

		/* get the thread */
		article_get_entire_thread (
			articlelist_get_selected_adata(),
			&article_qty,
			&articles);

		/* FIXME: maybe that special case in article_delete wasn't
		   such a good idea after all. */
		for (i=0; i<article_qty; ) {
			if (articles[i]->part>1)
				array_shrink (
					articles, i,
					sizeof(articles[0]), article_qty--);
			else ++i;
		}

		/* delete the thread */
		if (article_qty)
			articles_delete (
				my_server, my_group,
				articles, article_qty, TRUE);

		/* cleanup */
		g_free (articles);
	}

	/* thaw the clist if we locked it before */
	if (list!=NULL) {
		pan_lock();
		gtk_clist_thaw (clist);
		pan_unlock();
		g_list_free (list);
	}
}


void
articlelist_selected_delete (void)
{
	GtkCTree* tree = GTK_CTREE (Pan.article_ctree);
	article_data **articles = NULL;
	int article_qty = 0;
	GSList* l = NULL;
	int i=0;

	/* make a temporary buffer */
	article_qty = 0;
	for ( l=(GSList*)GTK_CLIST(tree)->selection; l!=NULL; l=l->next)
		++article_qty;
	articles = g_new (article_data*, article_qty);
	for (i=0, l=(GSList*)GTK_CLIST(tree)->selection; l!=NULL; l=l->next)
		articles[i++] = articlelist_get_adata_from_node (
			GTK_CTREE_NODE(l->data));
	g_assert (article_qty == i);

	/* delete the articles */
	gtk_clist_freeze (GTK_CLIST(tree));
	articles_delete (my_server, my_group, articles, article_qty, TRUE);
	gtk_clist_thaw (GTK_CLIST(tree));

	/* cleanup */
	g_free (articles);
}

static void
articlelist_mark_read (GSList* nodes) 
{
	GtkCTree* tree = GTK_CTREE (Pan.article_ctree);
	const int qty = g_slist_length (nodes);
	article_data** articles = NULL;
	int i;

	/* sanity checking */
	g_return_if_fail (qty>=1);

	articles = g_new (article_data*, qty);
	/* make a list of selected articles */
	for (i=0; nodes!=NULL; nodes=nodes->next, ++i)
		articles[i] = articlelist_get_adata_from_node (
			GTK_CTREE_NODE(nodes->data));

	/* mark them all as read */
	gtk_clist_freeze (GTK_CLIST(tree));
	articles_mark_read (my_server, my_group, articles, qty, TRUE);
	gtk_ctree_unselect_recursive (tree, NULL);
	gtk_clist_thaw (GTK_CLIST(tree));

	g_free (articles);
}


void
articlelist_selected_mark_read ( void )
{
	articlelist_mark_read (
		(GSList*)GTK_CLIST(Pan.article_ctree)->selection);
}

void
articlelist_all_mark_read ( void )
{
	GtkCList* list = GTK_CLIST(Pan.article_ctree);

	if (article_header_qty>0)
	{
		pan_lock();
		gtk_clist_freeze (list);
		pan_unlock();

		articles_mark_read (
			my_server, my_group,
			article_headers, article_header_qty,
			TRUE);

		pan_lock();
		gtk_clist_thaw (list);
		pan_unlock();
	}
}

void
articlelist_selected_mark_unread ( )
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);
	article_data ** articles = NULL;
	int qty = 0;
	int i = 0;
	GList* l = NULL;

	/* find the selected nodes */
	l = GTK_CLIST(tree)->selection;
	qty = g_list_length (l);
	if (qty<1)
		return;

	/* make a list of all the articles the nodes match up with */
	articles = g_new (article_data*, qty);
	for (i=0; l!=NULL; l=l->next, ++i)
		articles[i] = articlelist_get_adata_from_node (
			GTK_CTREE_NODE(l->data));
	g_assert (i==qty);
	
	/* mark them all as unread */
	gtk_clist_freeze (GTK_CLIST(tree));
	articles_mark_unread (my_server, my_group, articles, qty, TRUE);
	gtk_ctree_unselect_recursive (tree, NULL);
	gtk_clist_thaw (GTK_CLIST(tree));

	/* cleanup */
	g_free (articles);
}

static void
rekill_thread (article_data *adata)
{
	int qty = 0;
	article_data ** buf = NULL;
	article_get_entire_thread (adata, &qty, &buf);
	if (qty != 0)
	{
		int i;

		/* refresh adata flags */
		killfile_articles (buf, qty, NULL);

		/* refresh the tree */
		pan_lock ();
		gtk_clist_freeze (GTK_CLIST(Pan.article_ctree));
		pan_unlock();
		for (i=0; i!=qty; ++i)
			articlelist_update_node (buf[i]);
		pan_lock ();
		gtk_clist_thaw (GTK_CLIST(Pan.article_ctree));
		pan_unlock();

		/* cleanup */
		g_free (buf);
	}
}

void
articlelist_selected_thread_download (void)
{
	article_data* adata = NULL;
	adata = articlelist_get_selected_adata ();
	if (adata != NULL)
	{
		int i = 0;
		int qty = 0;
		article_data ** buf = NULL;
		article_get_entire_thread (adata, &qty, &buf);

		/* queue them all */
		for (i=0; i!=qty; ++i) {
			QueueItem *item = QUEUE_ITEM(queue_item_body_new (
				(server_data*)my_server, my_group,
				message_new(my_server, my_group, buf[i])));
			item->high_priority = FALSE;
			queue_add (item);
		}

		/* cleanup */
		g_free (buf);
	}
}

void
articlelist_selected_thread_track (void)
{
	article_data* adata = NULL;
	gchar *pch = NULL;

	adata = articlelist_get_selected_adata ();
	if (!adata)
		return;
	pch = article_get_thread_message_id (adata);
	if (!pch)
		return;

	/* add to killfile */
	message_filter_killfile_add (
		"", "", "", pch, 0, -1, MESSAGE_FILTER_ACTION_WATCH);
	g_free (pch);

	rekill_thread (adata);
}

void
articlelist_selected_thread_kill (void)
{
	article_data* adata = NULL;
	gchar *pch = NULL;

	adata = articlelist_get_selected_adata ();
	if (!adata)
		return;
	pch = article_get_thread_message_id (adata);
	if (!pch)
		return;

	message_filter_killfile_add (
		"", "", "", pch, 0, -1, MESSAGE_FILTER_ACTION_KILL);
	g_free (pch);

	rekill_thread (adata);
}

/**
***
***
**/

static void
decode_as_callback (gchar *s, gpointer data)
{
	QueueItemDecode *qid = QUEUE_ITEM_DECODE(data);

	/* abort or no good filename */
	if (!s || !*g_strstrip(s))
	{
		const gchar* message_id = (const gchar*) qid->message_ids->data;
		GtkCTreeNode* node = articlelist_get_node_from_message_id (message_id);
		article_data* adata = articlelist_get_adata_from_node (node);
		article_remove_flag (my_server, my_group, adata, STATE_DECODE_QUEUED, TRUE);
		pan_object_unref (PAN_OBJECT(qid));
	}
	else /* presumably-good filename */
	{
		qid->filename = g_strdup(s);
		queue_add (QUEUE_ITEM(qid));
	}
}

static void
articlelist_selected_decode_impl (gboolean open, gboolean prompt_for_filename)
{
	GtkCTree* tree = GTK_CTREE (Pan.article_ctree);
	GSList* l = NULL;

	/* sanity checks... */
	g_return_if_fail (my_server!=NULL);
	g_return_if_fail (my_group!=NULL);

	/* loop through all the selected articles */
	gtk_clist_freeze (GTK_CLIST(tree));
	for (l=(GSList*)GTK_CLIST(tree)->selection; l!=NULL; l=l->next)
	{
		GSList *tmp = NULL;
		GSList *list = NULL;
		QueueItemDecode *qid = NULL;
		GtkCTreeNode* node = GTK_CTREE_NODE(l->data);
		article_data* adata = articlelist_get_adata_from_node (node);

		/* make sure this is an article... */
		if (!adata)
			continue;

		/* is this node already queued? */
		if (article_flag_on (adata, STATE_DECODE_QUEUED)) {
			pan_error_dialog (_("Article ``%s'' already queued; skipping."), adata->subject);
			continue;
		}

		/* build an array of this node and all its children. */
		list = g_slist_append (list, adata->message_id);
		for (tmp=adata->threads; tmp!=NULL; tmp=tmp->next)
			list = g_slist_append (list, ARTICLE_DATA(tmp->data)->message_id);

		/* queue for decode */
		qid = QUEUE_ITEM_DECODE(
			queue_item_decode_new(
				(server_data*)my_server,
				(group_data*)my_group,
				adata->subject, list, open));

		/* update the list to show this is queued */
		adata->state |= STATE_DECODE_QUEUED;
		articlelist_update_node_fast (GTK_CTREE_NODE(l->data), adata);
	
		/* cleanup this loop */
		g_slist_free (list);

		if (prompt_for_filename)
		{
			gnome_request_dialog (FALSE,
				_("Save As:"),
				adata->subject, 128,
				decode_as_callback, (gpointer)qid,
				GTK_WINDOW (Pan.window));
		}
		else queue_add (QUEUE_ITEM(qid));

	}

	/* cleanup */
	gtk_ctree_unselect_recursive ( tree, NULL );
	gtk_clist_thaw ( GTK_CLIST(tree) );
}

void
articlelist_selected_decode_as (void)
{
	articlelist_selected_decode_impl (FALSE, TRUE);
}

void
articlelist_selected_decode (void)
{		
	articlelist_selected_decode_impl (FALSE, FALSE);
}

void
articlelist_selected_open (void)
{
	articlelist_selected_decode_impl (TRUE, FALSE);
}

void
articlelist_select_all ( void )
{
	pan_lock ();
	gtk_ctree_select_recursive (GTK_CTREE(Pan.article_ctree), NULL);
	pan_unlock ();
}


//============================================================================


static
GnomePixmap*
get_article_pixmap (
	const article_data* adata )
{
	g_return_val_if_fail ( adata!=NULL, NULL );

	if (article_flag_on (adata, STATE_ERROR))
		return error_pixmap;

	if (article_flag_on (adata, STATE_DECODED))
		return binary_decoded_pixmap;

	if (article_flag_on (adata, STATE_READ))
		return article_read_pixmap;

	if (article_flag_on (adata, STATE_DOWNLOAD_QUEUED))
		return download_queue_pixmap;

	if (article_flag_on (adata, STATE_DECODE_QUEUED))
		return decode_queue_pixmap;

	if (article_flag_on (adata, STATE_MULTIPART_ALL))
		return mp_complete_pixmap;

	if (article_flag_on (adata, STATE_MULTIPART_SOME))
		return mp_incomplete_pixmap;

	return NULL;
}


static void
articlelist_set_node_style (
	GtkCTreeNode* node,
	const article_data* adata)
{
	GtkCTree* tree = GTK_CTREE (Pan.article_ctree);

	pan_lock();

	if (!unread_style)
	{
		GdkFont* font = NULL;
		if (articlelist_font != NULL)
			font = gdk_fontset_load (articlelist_font);

		killfile_style = gtk_style_copy (
				gtk_widget_get_style (GTK_WIDGET (Pan.window)));
		killfile_style->fg[0] = killfile_color;

		watched_style = gtk_style_copy (
				gtk_widget_get_style (GTK_WIDGET (Pan.window)));
		watched_style->fg[0] = watched_color;

		unread_style = gtk_style_copy (
				gtk_widget_get_style (GTK_WIDGET (Pan.window)));
		unread_style->fg[0] = unread_color;

		read_style = gtk_style_copy (
				gtk_widget_get_style (GTK_WIDGET (Pan.window)));
		read_style->fg[0] = read_color;

		new_style = gtk_style_copy (
				gtk_widget_get_style (GTK_WIDGET (Pan.window)));
		new_style->fg[0] = new_color;

		if (font) {
			killfile_style->font = font;
			unread_style->font = font;
			read_style->font = font;
			new_style->font = font;
		}
	}

	if (articlelist_font)
	{
		if (article_flag_on (adata, STATE_KILLFILE))
			gtk_ctree_node_set_row_style (
				tree, node, killfile_style);

		else if (article_flag_on (adata, STATE_WATCHED))
			gtk_ctree_node_set_row_style (
				tree, node, watched_style);

		else if (article_flag_on (adata, STATE_READ) ||
		         article_flag_on (adata, STATE_DECODED))
			gtk_ctree_node_set_row_style (tree, node, read_style);

		else if (article_flag_on (adata, STATE_NEW))
			gtk_ctree_node_set_row_style (tree, node, new_style);

		else
			gtk_ctree_node_set_row_style (tree, node, unread_style);
	}	
	else /* If the font isn't specified by the user, just set the color */
	{
		if (article_flag_on (adata, STATE_KILLFILE))
			gtk_ctree_node_set_foreground (
				tree, node, &killfile_color);

		else if (article_flag_on (adata, STATE_WATCHED))
			gtk_ctree_node_set_foreground (
				tree, node, &watched_color);

		else if (article_flag_on (adata, STATE_READ) ||
		         article_flag_on (adata, STATE_DECODED))
			gtk_ctree_node_set_foreground (tree, node, &read_color);

		else if (article_flag_on (adata, STATE_NEW))
			gtk_ctree_node_set_foreground (tree, node, &new_color);

		else
			gtk_ctree_node_set_foreground (tree, node, &unread_color);
	}

	pan_unlock();
}

static GtkCTreeNode*
add_article_to_ctree (
	GtkCTree* tree,
	GtkCTreeNode* parent,
	article_data* adata,
	StatusItem *item,
	gboolean expanded)
{
	char *text[5];
	char line_count_buf[16];
	GtkCTreeNode *node;
	GnomePixmap* icon = get_article_pixmap (adata);
        char timebuf[32] = { '\0' };
        char childbuf[8] = { '\0' };
        const struct tm local_tm = *localtime (&adata->date);
	gulong lcountsum = 0;

        /* this is sortable with a simple strcmp */
        strftime (timebuf, sizeof(timebuf), "%Y/%m/%d %H:%M", &local_tm);

	/* number of children */
	if (adata->unread_children)
		sprintf (childbuf, "%d", adata->unread_children);

	/* line count */
	lcountsum = adata->linecount;
	if (article_flag_on (adata, STATE_MULTIPART_ALL)) {
		GSList *l;
		for (l=adata->threads; l; l=l->next)
			lcountsum += ((article_data *)l->data)->linecount;
	}
	commatize_ulong ( lcountsum, line_count_buf );

	/* set text */
	text[0] = adata->subject;
	text[1] = childbuf;
	text[2] = line_count_buf;
	text[3] = timebuf;
	text[4] = adata->author;

	status_item_emit_next_step (item);

	pan_lock();
	node = gtk_ctree_insert_node (
		tree, parent, NULL,
		text, 5,
		(icon ? icon->pixmap : NULL),
		(icon ? icon->mask : NULL),
		(icon ? icon->pixmap : NULL),
		(icon ? icon->mask : NULL),
		FALSE, expanded);
	g_hash_table_insert (messageid_to_node, adata->message_id, node);
	gtk_ctree_node_set_row_data (tree, node, adata);
	pan_unlock();
	articlelist_set_node_style (node, adata);

	return node;
}

static void
build_article_ctree_recursive (
	GtkCTree* tree,
	GtkCTreeNode* parent,
	GSList *list,
	StatusItem *item,
	gboolean expanded)
{
	GSList *pos = NULL;

	/* sanity check */
	g_return_if_fail (parent!=NULL);

        for (pos=list; pos!=NULL; pos=pos->next)
        {
		article_data* adata = ARTICLE_DATA(pos->data);
		GtkCTreeNode* node = NULL;

		/* don't show subtrees that don't measure up to filter */
		if (is_threaded && !adata->tree_passes_filter)
			continue;

		node = add_article_to_ctree (tree, parent, adata, item, expanded);
		if (adata->threads!=NULL)
			build_article_ctree_recursive (tree, node, adata->threads, item, TRUE);
        }
}



typedef struct
{
	const group_data *gdata;
	article_data **article_buf;
	int article_qty;
	StatusItem *item;
	int index;
	GtkCTreeNode *root;
}
BuildArticleListStruct;

static gboolean
article_passes_filter (const article_data* adata)
{
	gboolean step_passed;
	const guint state_filter = my_group->state_filter;

	const gboolean is_complete_bin =
		article_flag_on (adata, STATE_MULTIPART_ALL);
	const gboolean is_incomplete_bin =
		article_flag_on (adata, STATE_MULTIPART_SOME);
	const gboolean complete_bin_on =
		state_filter & STATE_FILTER_COMPLETE_BINARIES;
	const gboolean incomplete_bin_on =
		state_filter & STATE_FILTER_INCOMPLETE_BINARIES;
	const gboolean nonbin_on =
		state_filter & STATE_FILTER_NONBINARIES;
	const gboolean unread_on =
		state_filter & STATE_FILTER_UNREAD;
	const gboolean read_on =
		state_filter & STATE_FILTER_READ;
	const gboolean is_read =
		article_flag_on (adata, STATE_READ);

	const gboolean kill_on =
		state_filter & STATE_FILTER_KILLFILE;
	const gboolean is_killed =
		article_flag_on (adata, STATE_KILLFILE);
	const gboolean watch_on =
		state_filter & STATE_FILTER_WATCHED;
	const gboolean is_watched =
		article_flag_on (adata, STATE_WATCHED);
	const gboolean normal_rank_on =
		state_filter & STATE_FILTER_NORMAL_RANK;

	/* filter on complete/incomplete/nonbinary */
	if (!(complete_bin_on && is_complete_bin)
		&& !(incomplete_bin_on && is_incomplete_bin)
		&& !(nonbin_on && !is_complete_bin && !is_incomplete_bin))
		return FALSE;

	/* filter on read */
	if (!(unread_on && !is_read) &&
		!(read_on && is_read))
		return FALSE;

	/* filter on killfile, watch, normal rank */
	if (!(kill_on && is_killed) &&
		!(watch_on && is_watched) &&
		!(normal_rank_on && !is_killed && !is_watched))
		return FALSE;

	/* filter on decoded/queued/idle */
	step_passed = FALSE;
	if ((state_filter&STATE_FILTER_SAVED)
		&& article_flag_on (adata, STATE_DECODED))
		step_passed = TRUE;
	else if ((state_filter&STATE_FILTER_QUEUED)
		&& ((adata->state&STATE_DECODE_QUEUED)
	       		|| (adata->state&STATE_DOWNLOAD_QUEUED)))
		step_passed = TRUE;
	else if ((state_filter&STATE_FILTER_IDLE)
		&& !(adata->state&STATE_DECODED)
		&& !(adata->state&STATE_DECODE_QUEUED)
		&& !(adata->state&STATE_DOWNLOAD_QUEUED))
		step_passed = TRUE;
	if (!step_passed)
		return FALSE;

	return TRUE;
}

static gboolean
ancestor_is_binary (const article_data* a)
{
	g_return_val_if_fail (a!=NULL, FALSE);

	for (a=a->parent; a!=NULL; a=a->parent)
		if ((a->state & STATE_MULTIPART_ALL)
			|| (a->state&STATE_MULTIPART_SOME))
			return TRUE;

	return FALSE;
}

static void
apply_filter_tests (void)
{
	register int i;
	register article_data *adata;

	/* clear the filter state */
	for (i=0; i!=article_header_qty; ++i) {
		adata = article_headers[i];
		adata->self_passes_filter = FALSE;
		adata->tree_passes_filter = FALSE;
	}

	/* apply the filter tests */
	for (i=0; i!=article_header_qty; ++i)
	{
		adata = article_headers[i];
		if (article_passes_filter (adata))
		{
			adata->self_passes_filter = TRUE;
			if (ancestor_is_binary (adata))
				continue;

			while (adata && !adata->tree_passes_filter) {
				adata->tree_passes_filter = TRUE;
				adata = adata->parent;
			}
		}
	}
}

static void
add_nodes (gpointer data)
{
	BuildArticleListStruct *build = (BuildArticleListStruct*) data;
	GtkCTree* tree = GTK_CTREE (Pan.article_ctree);
	const gchar* subject_filter_str = article_toolbar_get_filter();

	while (build->index < build->article_qty)
	{
		article_data* adata = build->article_buf[build->index++];
		GtkCTreeNode* node = NULL;
		gboolean has_parent;
		gboolean filter_match;

		/* make sure we pass the state filter */
		if (is_threaded && !adata->tree_passes_filter)
			continue;
		if (!is_threaded && !adata->self_passes_filter)
			continue;

		/* make sure we pass the subject filter */
		filter_match =
			subject_filter_str==NULL ||
			!fnmatch(subject_filter_str,adata->subject,PAN_CASEFOLD);
		if (!filter_match)
			continue;

		has_parent = adata->parent != NULL;

		if ((!is_threaded && adata->part<2) || (is_threaded && !has_parent))
		{
			node = add_article_to_ctree (
				tree, build->root, adata, build->item, FALSE);
		}
		if (is_threaded && node!=NULL && adata->threads!=NULL
		    && !((adata->state&STATE_MULTIPART_ALL) && hide_mpart_child_nodes))
		{
			build_article_ctree_recursive (
				tree, node, adata->threads, build->item, TRUE);
		}
	}

	pan_lock();
	gtk_ctree_expand (tree, build->root);
	gtk_clist_thaw (GTK_CLIST(tree));
	gtk_widget_queue_resize (Pan.article_ctree);
	pan_unlock();
}

/* fires next_step article_qty times */
static void
articlelist_repopulate (
	const group_data* gdata,
	article_data** article_buf,
	int article_qty,
	StatusItem *item)
{
	GtkCTree* tree = GTK_CTREE (Pan.article_ctree);
	char *text[5];
	BuildArticleListStruct build;

	killfile_articles (article_buf, article_qty, item);

	build.gdata = gdata;
	build.article_buf = article_buf;
	build.article_qty = article_qty;
	build.item = item;
	build.index = 0;
	build.root = NULL;

	status_item_emit_status (item, _("Updating article display"));

	pan_lock();

	/* clear out old */
	g_hash_table_destroy (messageid_to_node);
	messageid_to_node = g_hash_table_new (g_str_hash, g_str_equal);
	gtk_clist_freeze (GTK_CLIST(tree));
	gtk_clist_clear (GTK_CLIST(tree));

	/* add root node */
	text[0] = gdata->name;
	text[1] = text[2] = text[3] = text[4] = NULL;
	build.root = gtk_ctree_insert_node (
		tree, NULL, NULL,
		text, 5, articlelist_closed_pixmap->pixmap,
		articlelist_closed_pixmap->mask,
		articlelist_open_pixmap->pixmap,
		articlelist_open_pixmap->mask, FALSE, FALSE );
	gtk_ctree_node_set_row_data (tree, build.root, NULL);
	pan_unlock();

	add_nodes (&build);
}

/***
****
****
***/

static void 
articlelist_update_node_fast (GtkCTreeNode* node, const article_data* adata)
{
	GnomePixmap* icon = get_article_pixmap (adata);
        char childbuf[8] = { '\0' };
	if (adata->unread_children)
		sprintf (childbuf, "%d", adata->unread_children);

	pan_lock();
	gtk_ctree_set_node_info (
		GTK_CTREE(Pan.article_ctree), node,
		adata->subject, 5,
		(icon ? icon->pixmap : NULL),
		(icon ? icon->mask : NULL),
		(icon ? icon->pixmap : NULL),
		(icon ? icon->mask : NULL),
		GTK_CTREE_ROW(node)->is_leaf,
		GTK_CTREE_ROW(node)->expanded );
	gtk_ctree_node_set_text (
		GTK_CTREE(Pan.article_ctree), node, 1, childbuf);
	pan_unlock();

	articlelist_set_node_style (node, adata);
}

void
articlelist_adata_read_changed (const article_data* adata)
{
	article_data* a = NULL;
	int inc = 0;
	GtkCTreeNode* node = NULL;

	/* sanity checks */
	g_return_if_fail (adata!=NULL);	
	inc = article_flag_on (adata, STATE_READ) ? -1 : 1;

	/* find the matching node */
	node = articlelist_get_node_from_message_id (adata->message_id);
	if (!node)
		return;

	/* update the read flag of our adata */
	a = articlelist_get_adata_from_node (node);
	if (a != adata) {
		a->state &= ~STATE_READ;
       		a->state |= (adata->state&STATE_READ);
	}

	/* update the unread child count of our adata and its ancestors */
	articlelist_unread_inc (node, inc);
}



static GtkCTreeNode*
articlelist_get_node_from_message_id (const char* message_id)
{
	return (GtkCTreeNode*) g_hash_table_lookup (messageid_to_node, message_id);
}

void
articlelist_update_node (const article_data* adata)
{
	GtkCTreeNode* node =
		articlelist_get_node_from_message_id (adata->message_id);
        if (node != NULL)
	{
		/* update the adata state */
		article_data* real_adata =
			articlelist_get_adata_from_node (node);
		real_adata->state = adata->state;

		/* repaint */
                articlelist_update_node_fast (node, real_adata);
	}
}


static void
articlelist_click_column (GtkCList* clist, int n)
{
	int type;
	switch (n)
	{
		case 0:
			type = ARTICLE_SORT_SUBJECT;
			break;
		case 1:
			type = ARTICLE_SORT_UNREAD_CHILDREN;
			break;
		case 2:
			type = ARTICLE_SORT_LINES;
			break;
		case 3:
			type = ARTICLE_SORT_DATE;
			break;
		case 4:
			type = ARTICLE_SORT_AUTHOR;
			break;
		default:
			pan_warn_if_reached();
			type = -ARTICLE_SORT_DATE;
			break;
	}

	if (type == sort_type)
		type = -type;

	articlelist_set_sort_type (type);
}

static void
article_ctree_destroy_cb (void)
{
	gtk_widget_destroy (GTK_WIDGET(articlelist_closed_pixmap));
	gtk_widget_destroy (GTK_WIDGET(articlelist_open_pixmap));
	gtk_widget_destroy (GTK_WIDGET(article_read_pixmap));
	gtk_widget_destroy (GTK_WIDGET(decode_queue_pixmap));
	gtk_widget_destroy (GTK_WIDGET(download_queue_pixmap));
	gtk_widget_destroy (GTK_WIDGET(error_pixmap));
	gtk_widget_destroy (GTK_WIDGET(binary_decoded_pixmap));
	gtk_widget_destroy (GTK_WIDGET(unread_children_pixmap));
	gtk_widget_destroy (GTK_WIDGET(mp_complete_pixmap));
	gtk_widget_destroy (GTK_WIDGET(mp_incomplete_pixmap));

	pan_callback_free (articlelist_state_filter_changed);
	pan_callback_free (articlelist_group_changed);
	pan_callback_free (articlelist_sort_changed);
	pan_callback_free (articlelist_thread_changed);
	pan_callback_free (articlelist_selection_changed);
}

static int
articlelist_group_changed_cb (
	gpointer call_obj,
	gpointer call_arg,
	gpointer user_data )
{
	const group_data* gdata = (const group_data*) call_arg;

	/* the group has changes, so the group state filter has too */
	pan_callback_call (
		articlelist_state_filter_changed,
		Pan.article_ctree,
		GUINT_TO_POINTER(gdata ? gdata->state_filter : 0));

	return 0;
}

static void
tree_select_row_cb (
	GtkCTree *tree,
	GtkCTreeNode *node,
	gint column,
	gpointer u)
{
	pan_callback_call (
		articlelist_selection_changed,
		Pan.article_ctree,
		articlelist_get_selected_adata());
}
static void
tree_unselect_row_cb (
	GtkCTree *tree,
	GtkCTreeNode *node,
	gint column,
	gpointer u)
{
	tree_select_row_cb (tree, node, column, u);
}

/*--------------------------------------------------------------------
 * generate the listing of articles, for the "Articles" tab
 *--------------------------------------------------------------------*/
GtkWidget*
create_articlelist_ctree (void)
{
	int i;
	GtkCList* list;
	GtkCTree* tree;
	GtkWidget *s_window;
	const char* titles[5];

	/* get the titles */
	for (i=0; i<5; ++i)
		titles[i] = column_to_title (i);

	/* load the pixmaps */
	articlelist_closed_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d(book_closed_xpm);
	articlelist_open_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d(book_open_xpm);
	article_read_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d(mini_page_xpm);
	decode_queue_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d(bluecheck_xpm);
	download_queue_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d(greencheck_xpm);
	error_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d(x_xpm);
	binary_decoded_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d(binary_xpm);
	unread_children_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d(unread_children_xpm);
	mp_complete_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d(binary_complete_xpm);
	mp_incomplete_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d(binary_incomplete_xpm);

	cmap = gdk_colormap_get_system ();
	if (!gdk_color_alloc (cmap, &unread_color))
		g_error ("couldn't allocate color");
	if (!gdk_color_alloc (cmap, &new_color))
		g_error ("couldn't allocate color");
	if (!gdk_color_alloc (cmap, &read_color))
		g_error ("couldn't allocate color");

	messageid_to_node = g_hash_table_new (g_str_hash, g_str_equal);

	/* create the widget */
	Pan.article_ctree = gtk_ctree_new_with_titles (5, 0, (gchar**)titles);
	tree = GTK_CTREE(Pan.article_ctree);
	list = GTK_CLIST(Pan.article_ctree);
	gtk_signal_connect (GTK_OBJECT(tree), "tree-select-row", tree_select_row_cb, NULL);
	gtk_signal_connect (GTK_OBJECT(tree), "tree-unselect-row", tree_unselect_row_cb, NULL);

	/* wrap it in a scrolled window */
	//gtk_widget_show (tree);
	s_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy (
		GTK_SCROLLED_WINDOW(s_window),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER(s_window), Pan.article_ctree);

	/* ui settings */
	gtk_ctree_set_line_style (tree, GTK_CTREE_LINES_DOTTED);
	gtk_clist_set_selection_mode (list, GTK_SELECTION_EXTENDED);
	gtk_clist_set_column_width (list, 0, 400);
	gtk_clist_set_column_auto_resize (list, 1, TRUE);
	gtk_clist_set_column_justification (list, 1, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_auto_resize (list, 2, TRUE);
	gtk_clist_set_column_justification (list, 2, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_auto_resize (list, 3, TRUE);
	gtk_clist_set_column_justification (list, 3, GTK_JUSTIFY_RIGHT);

	/* create the right click popup menu */
	article_ctree_menu = gnome_popup_menu_new (articlelist_menu_popup);

	/* connect signals */
	gtk_signal_connect (
		GTK_OBJECT(tree), "button_press_event",
		GTK_SIGNAL_FUNC(articlelist_button_press), NULL);
	gtk_signal_connect (
		GTK_OBJECT(tree), "click_column",
		GTK_SIGNAL_FUNC(articlelist_click_column), NULL);
	gtk_signal_connect (
		GTK_OBJECT(tree), "destroy",
		GTK_SIGNAL_FUNC(article_ctree_destroy_cb), NULL);

	/* callbacks */
	articlelist_group_changed = pan_callback_new ();
	articlelist_state_filter_changed = pan_callback_new ();
	articlelist_sort_changed = pan_callback_new ();
	articlelist_thread_changed = pan_callback_new ();
	articlelist_selection_changed = pan_callback_new ();

	pan_callback_add (
		articlelist_group_changed, articlelist_group_changed_cb, NULL);
	pan_callback_add (
		article_deleted_callback, article_deleted_cb, NULL);

	return s_window;
}

/*FIXME: is this used? */
//static article_db active_group_header_db = NULL;

void
articlelist_set_current_group ( server_data *s, group_data* g )
{
	if (g == my_group)
		return;

	/* OUT WITH THE OLD GROUP */

	if (my_group != NULL)
	{
		ahdb_unref (active_group_header_db);
		active_group_header_db = NULL;
	}

	/* BRING IN THE NEW GROUP */

	my_group = g;
	my_server = s;
	pan_callback_call (articlelist_group_changed, Pan.article_ctree, my_group);

	if (g != NULL)
	{
		/* open the article header database */
		active_group_header_db = ahdb_ref (s, g);

		/* load the local headers... */
		if (ahdb_exists (s,g))
			queue_add(QUEUE_ITEM(
				queue_item_headers_new(s,1,g,HEADERS_LOCAL)));
		else
			choose_download_dialog (s, g);

	}
	else
	{
		articlelist_clear ( );
	}

}


/**
***
***  ARTICLE FILTERING
***
**/

static gchar*
filter_selected_describe (const StatusItem* item)
{
	return g_strdup (_("Filtering Articles"));
}
static void
filter_changed_thread (void *data)
{
	StatusItem *item = NULL;

	if (!my_group) /* articlelist is idle */
		return;

	/* create a new status item to get sort/thread messages */
	item = STATUS_ITEM(status_item_new(filter_selected_describe));
	pan_object_sink(PAN_OBJECT(item));
	gui_add_status_item (item);

	/* filter */
	apply_filter_tests ();

	/* repopulate */
	articlelist_refresh (item);

	/* clean out the status item */
	gui_remove_status_item (item);
	pan_object_unref(PAN_OBJECT(item));
}

void
articlelist_poke_state_filter (guint flag, gboolean on)
{
	pthread_t thread;

	/* someone toggled a button, or something,
	   even though there's no group */
	if (!my_group)
		return;

	/* update the state if necessary,
	   return otherwise */
	if (on) {
		if (my_group->state_filter & flag)
			return;
		my_group->state_filter |= flag;
	} else {
		if (!(my_group->state_filter & flag))
			return;
		my_group->state_filter &= ~flag;
	}

	/* save the state */
	grouplist_save_group (my_server, my_group);

	/* notify listeners of the change */
	pan_callback_call (
		articlelist_state_filter_changed,
		Pan.article_ctree,
		GUINT_TO_POINTER(my_group->state_filter));

	pthread_create (&thread, NULL, (void*)filter_changed_thread, NULL);
	pthread_detach (thread);
}


static gchar*
sort_selected_describe (const StatusItem* item)
{
	return g_strdup ("Sorting Articles");
}

static void
sort_in_thread (void* data)
{
	const int sort = GPOINTER_TO_INT(data);
        const server_data *server = articlelist_get_current_server ();
        group_data *group = articlelist_get_current_group ();
        StatusItem *item = NULL;

        if (!server || !group) /* articlelist is empty */
                return;

	pthread_mutex_lock (&article_ctree_lock);

	if (sort != sort_type)
	{
		/* create a new status item to get sort/thread messages */
		item = STATUS_ITEM(status_item_new(sort_selected_describe));
		pan_object_sink(PAN_OBJECT(item));
		gui_add_status_item (item);

		/* tell the world what we're doing */
		status_item_emit_init_steps (item, article_header_qty*2);
		status_item_emit_status (item, _("Sorting Articles"));

		/* update the sort type */
		articlelist_set_sort_type_nosort (sort, FALSE);
		group_set_attrib_i (my_server, my_group, "sort_style", sort);
		sort_articles (article_headers, article_header_qty, abs(sort), sort>0);

		/* update articles */
		articlelist_repopulate (my_group,
			(article_data**)article_headers,
			article_header_qty, item);

		/* clean out the status item */
		gui_remove_status_item (item);
		pan_object_unref(PAN_OBJECT(item));
	}

	pthread_mutex_unlock (&article_ctree_lock);
}

void
articlelist_set_sort_type (int sort)
{
       	pthread_t thread;
       	pthread_create (&thread, NULL, (void*)sort_in_thread, GINT_TO_POINTER(sort));
       	pthread_detach (thread);
}


static gchar*
thread_selected_describe (const StatusItem* item)
{
	return g_strdup (_("Threading Articles"));
}

static void
articlelist_set_threaded_thread (void* data)
{
	const gboolean threaded_on = GPOINTER_TO_INT(data);
	StatusItem *item = NULL;

	if (!my_server || !my_group) /* articlelist is idle */
		return;

	/* create a new status item to get thread messages */
	item = STATUS_ITEM(status_item_new(thread_selected_describe));
	pan_object_sink(PAN_OBJECT(item));
	gui_add_status_item (item);

	/* update the articlelist */
	is_threaded = threaded_on;
	pan_callback_call (articlelist_thread_changed,
		Pan.article_ctree, GINT_TO_POINTER(threaded_on));
	articlelist_refresh (item);
										        /* clean out the status item */
	gui_remove_status_item (item);
	pan_object_unref(PAN_OBJECT(item));
}

void
articlelist_set_threaded (gboolean threaded_on)
{
	if (threaded_on != is_threaded)
	{
		pthread_t thread;
		pthread_create (&thread,
			NULL, (void*)articlelist_set_threaded_thread,
			GINT_TO_POINTER(threaded_on));
		pthread_detach (thread);
	}
}



void
articlelist_refresh (StatusItem* item)
{
	pthread_mutex_lock (&article_ctree_lock);

        /* tell the StatusItem what we're doing */
	status_item_emit_init_steps (item, article_header_qty*2);

	/* update articles */
	articlelist_repopulate (my_group,
		(article_data**)article_headers,
		article_header_qty, item);

	pthread_mutex_unlock (&article_ctree_lock);
}

