/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <pthread.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>

#include "acache.h"
#include "prefs.h" /* for data_dir */
#include "util.h" /* for read_file */
#include "debug.h"

/* Solaris' dirent.d_nam can be up to MAXNAMLEN */
#ifdef DIRENT_NEEDS_MAXNAMLEN
#define D_NAME_MAXLEN MAXNAMLEN
#else
#define D_NAME_MAXLEN NAME_MAX
#endif


GHashTable *files_in_use = NULL;
pthread_mutex_t fiu_lock = PTHREAD_MUTEX_INITIALIZER;

void
acache_init (void)
{
	files_in_use = g_hash_table_new (g_str_hash, g_str_equal);
}

void
acache_file_checkout (const gchar *m_id)
{
	gchar *message_id = g_strdelimit (g_strdup(m_id), "/\\", '_');
	gpointer p = NULL;

	debug (DEBUG_ACACHE, "File Checkout for message_id: %s", message_id);

	pthread_mutex_lock (&fiu_lock);
	if ((p = g_hash_table_lookup (files_in_use, message_id))) {
		gint refcount = GPOINTER_TO_INT(p);
		debug (DEBUG_ACACHE, "previous value: %d", refcount);
		g_hash_table_insert (files_in_use, message_id, GINT_TO_POINTER(++refcount));
		g_free (message_id);
	} else {
		g_hash_table_insert (files_in_use, message_id, GINT_TO_POINTER(1));
	}
	pthread_mutex_unlock (&fiu_lock);
}

void
acache_file_checkin (const gchar *m_id)
{
	gchar *message_id = g_strdelimit (g_strdup(m_id), "/\\", '_');
	gpointer p = NULL;

	debug (DEBUG_ACACHE, "File Checkin for message_id: %s", message_id);

	pthread_mutex_lock (&fiu_lock);
	if ((p = g_hash_table_lookup (files_in_use, message_id))) {
		gint refcount = GPOINTER_TO_INT(p);
		debug (DEBUG_ACACHE, "previous value: %d", refcount);
		refcount--;
		
		if (refcount <= 0) {
	        gpointer key = NULL;
	        gpointer value = NULL;
	        g_hash_table_lookup_extended (files_in_use, message_id, &key, &value);
			g_hash_table_remove (files_in_use, message_id);
	        g_free (key);
		} else {
			g_hash_table_insert (files_in_use, message_id, GINT_TO_POINTER(refcount));
		}
	}
	pthread_mutex_unlock (&fiu_lock);
	g_free (message_id);
}

static gboolean
acache_file_in_use (const gchar *m_id)
{
	gchar *message_id = g_strdelimit (g_strdup(m_id), "/\\", '_');
	gchar *q = strrchr(message_id, '>');
	gpointer p = NULL;

	/* Reduce the filename to a message_id */
	if (q)
		q[1] = '\0';

	pthread_mutex_lock (&fiu_lock);
	p = g_hash_table_lookup (files_in_use, message_id);
	pthread_mutex_unlock (&fiu_lock);
	
	debug (DEBUG_ACACHE, "file_in_use check: %s - %s in use", message_id, (p ? "Is" : "Not") );
	g_free (message_id);
	return (p ? TRUE : FALSE);
}

void
acache_delete (const gchar *m_id)
{
	gchar *path = NULL;
	gchar *message_id = g_strdelimit (g_strdup(m_id), "/\\", '_');
			
	debug (DEBUG_ACACHE, "Cache delete: %s", message_id);
	
	path = g_strdup_printf ( "/%s/%s.head", data_dir, message_id);
	unlink (path);
	g_free (path);
	path = g_strdup_printf ( "/%s/%s.body", data_dir, message_id);
	unlink (path);
	g_free (path);
	g_free (message_id);
}

static gboolean
ends_with (const gchar* str, const gchar* end)
{
	size_t str_len;
	size_t end_len;
	gboolean retval;

	g_return_val_if_fail (str!=NULL, FALSE);
	g_return_val_if_fail (end!=NULL, FALSE);

	end_len = strlen(end);
	str_len = strlen(str);

	if (end_len > str_len)
		retval = FALSE;
	else
		retval = !memcmp(end, str+str_len-end_len, end_len);
	return retval;
}

static int
acache_expire_to_size (int cache_max)
{
	int files_removed = 0;
	char *p;
	char dirpath[PATH_MAX];
	char path[PATH_MAX];
	DIR *dir_p;
	struct dirent *dirent_p;
	struct stat stat_p;
	time_t t_oldest;
	char n_oldest[D_NAME_MAXLEN] = "";
	int cache_size;

	g_snprintf (dirpath, PATH_MAX, "/%s", data_dir);
	cache_size = cache_max + 1;
	
	t_oldest = 0;
	while (cache_max < cache_size)
	{
		/* If it's too big and there is more than one entry, delete one */
		if ((cache_max < cache_size) && t_oldest) { 
			p = (char *)strrchr(n_oldest, '>');
			if (p) p[1] = '\0';
			++files_removed;
			acache_delete(n_oldest);
		}
		
		cache_size = t_oldest = 0;
		
		/* Get the size and oldest file */
   		dir_p = opendir(dirpath);
		while((dirent_p = readdir(dir_p)))
		{
			if (ends_with(dirent_p->d_name, ".head") || ends_with(dirent_p->d_name, ".body"))
			{
				if (acache_file_in_use (dirent_p->d_name))
					continue;
				g_snprintf(path, PATH_MAX, "/%s/%s", data_dir, dirent_p->d_name);
				if ((stat(path, &stat_p) == 0) && (S_ISREG(stat_p.st_mode)))
				{
					cache_size += stat_p.st_size;
					if ((stat_p.st_mtime < t_oldest) || (t_oldest == 0))
					{
						strcpy(n_oldest, dirent_p->d_name);
						t_oldest = stat_p.st_mtime;
					} 
				}
			}
		}	
		closedir (dir_p);
		debug (DEBUG_ACACHE, "Cache size: %d", cache_size);
	}

	return files_removed;
}

int 
acache_expire (void)
{
	int cache_max = gnome_config_get_int_with_default("/Pan/Cache/MaxSize=1242880", NULL);
	return acache_expire_to_size (cache_max);
}

int
acache_expire_all (void)
{
	return acache_expire_to_size(0);
}

static FILE*
acache_open_something(const article_data* adata, const char* suffix)
{
	FILE *f = NULL;
	gchar *path = NULL;
	gchar *message_id = g_strdelimit (g_strdup(adata->message_id), "/\\", '_');

	path = g_strdup_printf ( "/%s/%s.%s", data_dir, message_id, suffix);
	g_free (message_id);
	f = fopen (path, "w");
	g_free (path);
	g_assert (f != NULL);
	return f;
}

FILE *
acache_open_header (const article_data *adata)
{
	return acache_open_something(adata, "head");
}

void
acache_putline_header (FILE *f, const char *line)
{
	fputs (line, f);
}


void
acache_close_header (FILE *f)
{
	fclose (f);
}


FILE *
acache_open_body (const article_data *adata)
{
	return acache_open_something(adata, "body");
}
void
acache_putline_body (FILE *f, const char *line)
{
	/* Collapse double periods at line start, as described
	 * in RFC977 section 2.4.1 */
	if ((line[0] == '.') && (line[1] == '.')) {
		fputs (line + 1, f);
	} else {
		fputs (line, f);
	}
}
void
acache_close_body (FILE *f)
{
	//int cache_max = gnome_config_get_int ("/Pan/Cache/MaxSize=5242880");

	fclose(f);
	
	/* This will expire some, if necessary */
	acache_expire();
}

static gchar*
acache_get_existing_filename (const char *message_id, const char* suffix)
{
	struct stat st;
	gchar *msg_id = g_strdelimit (g_strdup(message_id), "/\\", '_');
	gchar* filename = g_strdup_printf ( "/%s/%s.%s", data_dir, msg_id, suffix );
	int rc = stat ( filename, &st );
	if ( !rc ) {
		g_free (msg_id);
		return filename;
	}
	g_free (filename);
	g_free (msg_id);
	return NULL;
}

gboolean
acache_body_exists (const char *message_id)
{
	gchar* filename = acache_get_existing_filename(message_id, "body");
	const gboolean filename_found = filename!=NULL;
	g_free(filename);
	return filename_found;
}

int
acache_header_exists (const char *message_id)
{
	gchar* filename = acache_get_existing_filename(message_id, "head");
	const gboolean filename_found = filename!=NULL;
	g_free(filename);
	return filename_found;
}


static char*
acache_load_something (const char *message_id, const char* suffix)
{
	char *buf = NULL;

	/* get the filename... */
	gchar* filename = acache_get_existing_filename (message_id, suffix);
	if ( filename ) {
		GArray* file = read_file ( filename );
		if ( file ) {
			buf = file->data;
			g_array_free ( file, FALSE );
		}
		g_free ( filename );
	}

	return buf;
}
char*
acache_load_body (const char *message_id)
{
	return acache_load_something(message_id, "body");
}
char*
acache_load_header (const char *message_id)
{
	return acache_load_something(message_id, "head");
}
