/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>
#include <gnome.h>

#include <sys/types.h>
#include <sys/wait.h>
#include <string.h>

#include "acache.h"
#include "articlelist.h"
#include "globals.h"
#include "grouplist.h"
#include "gui.h"
#include "message.h"
#include "message-send.h"
#include "message-window.h"
#include "prefs.h"
#include "print.h"
#include "text.h"
#include "util.h"

#include "xpm/more_headers.xpm"


extern GtkTooltips *ttips;

static void headers_cb (GtkWidget *widget, gpointer data);

static void message_window_save_ok_clicked (GtkWidget *widget, GtkWidget *file_entry);
static void message_window_save_cb (GtkWidget *widget, Message *msg);

static void message_window_cut_cb (GtkWidget *widget, Message *msg);
static void message_window_copy_cb (GtkWidget *widget, Message *msg);
static void message_window_paste_cb (GtkWidget *widget, Message *msg);

static void message_window_close (GtkWidget *widget, Message *msg);
static void message_window_appbar_toggle (GtkWidget *widget, Message *msg);

static void message_edit_external (GtkWidget *widget, Message *msg);

static GtkWidget* create_post_body_pane (MessageWindow *mw);
static GtkWidget* create_read_body_pane (GtkWidget *body_widget, Message *msg);

gboolean use_message_window_always = FALSE;

static GnomeUIInfo message_post_file_menu[] = {
//	GNOMEUIINFO_MENU_OPEN_ITEM (not_implemented, NULL),
	GNOMEUIINFO_ITEM_STOCK (N_("Send Now"), N_("Send this message now."), 
				send_cb, GNOME_STOCK_PIXMAP_MAIL_SND),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_SAVE_AS_ITEM (message_window_save_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_PRINT_ITEM (not_implemented, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_CLOSE_ITEM (message_window_close, NULL),	
	GNOMEUIINFO_END
};

static GnomeUIInfo message_read_file_menu[] = {
//	GNOMEUIINFO_MENU_OPEN_ITEM (not_implemented, NULL),
	GNOMEUIINFO_MENU_SAVE_AS_ITEM (message_window_save_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_PRINT_ITEM (print_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_CLOSE_ITEM (message_window_close, NULL),	
	GNOMEUIINFO_END
};

static GnomeUIInfo message_edit_menu[] = {
	GNOMEUIINFO_MENU_CUT_ITEM (message_window_cut_cb, NULL),
	GNOMEUIINFO_MENU_COPY_ITEM (message_window_copy_cb, NULL),
	GNOMEUIINFO_MENU_PASTE_ITEM (message_window_paste_cb, NULL),
	GNOMEUIINFO_END
};

static GnomeUIInfo message_view_menu[] = {
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_ ("Show Status Bar"), 0,
		message_window_appbar_toggle, 0, 0, 0, 0, 0, 0, 0
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo message_read_message_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("_Post to newsgroup"),
		N_("Post a new message to the current group."),
		message_post_window
	},
	{
		GNOME_APP_UI_ITEM,
		N_("_Followup to newsgroup"),
		N_("Post a reply to t he message on the news server."),
		message_followup_window
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Reply by _E-mail"),
		N_("Create a mail reply to the sender."),
		message_reply_window
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Reply with External Mailer"),
		N_("Create a mail reply to the sender with your favorite MUA."),
		message_reply_external,
		NULL
       	},
	GNOMEUIINFO_END
};

static GnomeUIInfo message_post_message_menu[] = {
	{ GNOME_APP_UI_ITEM, N_("New"), N_("Create a new message."), message_post_window },
	GNOMEUIINFO_END
};

static GnomeUIInfo message_read_main_menu[] = {
	GNOMEUIINFO_MENU_FILE_TREE (message_read_file_menu),
	GNOMEUIINFO_MENU_EDIT_TREE (message_edit_menu),
	GNOMEUIINFO_SUBTREE (N_("_View"), &message_view_menu),
	GNOMEUIINFO_SUBTREE (N_("_Message"), &message_read_message_menu),
	GNOMEUIINFO_MENU_HELP_TREE (help_menu),
	GNOMEUIINFO_END
};

static GnomeUIInfo message_post_main_menu[] = {
	GNOMEUIINFO_MENU_FILE_TREE (message_post_file_menu),
	GNOMEUIINFO_MENU_EDIT_TREE (message_edit_menu),
	GNOMEUIINFO_SUBTREE (N_("_View"), &message_view_menu),
	GNOMEUIINFO_SUBTREE (N_("_Message"), &message_post_message_menu),
	GNOMEUIINFO_MENU_HELP_TREE (help_menu),
	GNOMEUIINFO_END
};

static GnomeUIInfo message_read_toolbar[] = {
	GNOMEUIINFO_ITEM_STOCK(
		N_("Post Followup"),
		N_("Post a reply to the message on the news server."),
		message_followup_window,
		GNOME_STOCK_PIXMAP_MAIL_NEW),
	GNOMEUIINFO_ITEM_STOCK(
		N_("E-mail Reply"),
		N_("Create a mail reply to the sender."),
		message_reply_window,
		GNOME_STOCK_PIXMAP_MAIL_RPL),
//	GNOMEUIINFO_ITEM_STOCK (N_("Forward"), N_("Forward this Message"), 
//				NULL, GNOME_STOCK_PIXMAP_MAIL_FWD),
	GNOMEUIINFO_SEPARATOR,
//	GNOMEUIINFO_ITEM_STOCK (N_("Spelling"), N_("Check spelling"),
//				NULL, GNOME_STOCK_PIXMAP_SPELLCHECK),
	GNOMEUIINFO_ITEM_STOCK (N_("Print"), N_("Print this message"),
				print_cb, GNOME_STOCK_PIXMAP_PRINT),

//	GNOMEUIINFO_ITEM_STOCK (N_("Previous"), N_("Read the previous message"),
//				message_previous, GNOME_STOCK_PIXMAP_UP),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK (N_("Close"), N_("Close this Window"),
				message_window_close, GNOME_STOCK_PIXMAP_CLOSE),
	GNOMEUIINFO_END
};

static GnomeUIInfo message_post_toolbar[] = {
	GNOMEUIINFO_ITEM_STOCK (N_("Send"), N_("Send this message"), 
				send_cb, GNOME_STOCK_PIXMAP_MAIL_SND),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK (N_("Spelling"), N_("Check spelling"),
				NULL, GNOME_STOCK_PIXMAP_SPELLCHECK),
	GNOMEUIINFO_ITEM_STOCK (N_("Print"), N_("Print this message"),
				not_implemented, GNOME_STOCK_PIXMAP_PRINT),
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM, N_("Headers"), N_("Show More Headers"),
		headers_cb, NULL, NULL,
		GNOME_APP_PIXMAP_DATA, more_headers_xpm,
		0, (GdkModifierType)0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK (N_("Open Editor"), N_("Edit this in your favorite editor"),
				message_edit_external, GNOME_STOCK_PIXMAP_DOWN),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK (N_("Close"), N_("Close this Window without Sending Message"),
				message_window_close, GNOME_STOCK_PIXMAP_CLOSE),
	GNOMEUIINFO_END
};


static void
message_window_appbar_toggle (GtkWidget *widget, Message *msg)
{
	pan_lock();
	if ( !msg->window->appbar ) {
		msg->window->appbar = gnome_appbar_new (FALSE, TRUE, GNOME_PREFERENCES_USER );
		gnome_app_set_statusbar (GNOME_APP (msg->window->window), GTK_WIDGET (msg->window->appbar));
		if (msg->type == NNTP_ARTICLE_READ)
			gnome_app_install_menu_hints (GNOME_APP (msg->window->window), message_read_main_menu);
		else
			gnome_app_install_menu_hints (GNOME_APP (msg->window->window), message_post_main_menu);
	}
	else if (GTK_IS_WIDGET (msg->window->appbar->parent)) {
		gtk_widget_ref (msg->window->appbar);
		gtk_container_remove (GTK_CONTAINER (msg->window->appbar->parent), msg->window->appbar);
	}
	else {
		gtk_box_pack_start (GTK_BOX (GNOME_APP (msg->window->window)->vbox), msg->window->appbar, FALSE, FALSE, 0);
		gtk_widget_unref (msg->window->appbar);
	}
	pan_unlock();
}


static void
message_window_save_ok_clicked (GtkWidget *widget, GtkWidget *file_entry)
{
	gchar *filename;
	FILE *msg_dump;
	gchar *header;
	gchar *body;
	gchar *msg_id;

	pan_lock();
	msg_id = gtk_object_get_data (GTK_OBJECT (file_entry), "msg_id");
	filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (file_entry));
	pan_unlock();

	header = acache_load_header (msg_id);
	body = acache_load_body (msg_id);

	msg_dump = fopen (filename, "w");
	if ( !msg_dump ) {
		g_print ("fopen() failed to save message");
		g_free ( header );
		g_free ( body );
		return;
	}

	fprintf (msg_dump, "%s\n%s", header, body);
	fclose (msg_dump);

	g_free (header);
	g_free (body);

	pan_lock();
	gtk_widget_destroy (file_entry);
	pan_unlock();
}

static void
message_edit_external (GtkWidget *widget, Message *msg)
{
	not_implemented ();
}

static void
headers_cb (GtkWidget *widget, gpointer data)
{
	Message *msg = (Message*) data;
	MessageWindow *mw = msg->window;

	if (!mw->followup_to)
	{
		GtkWidget *table = mw->users->parent;
		GtkWidget *w = NULL;
		GtkWidget *eventbox = NULL; 

		gtk_table_resize (GTK_TABLE(table), 6, 2);

		w = gtk_label_new ( _("Followup-To:") );
		gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
		eventbox = gtk_event_box_new ();
		gtk_container_add (GTK_CONTAINER(eventbox), w);
		gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), eventbox,
			_("The newsgroup or newsgroups where replies to "
			  "your posted message should go.  This is only "
			  "needed if it differs from the \"Post To Groups\" "
			  "header. "
			  "\nTo direct all replies to your email address, "
			  "use \"Followup-To: poster\""), "");
		gtk_table_attach (GTK_TABLE(table), eventbox, 0, 1, 4, 5,
				  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
		mw->followup_to = gtk_entry_new ();
		gtk_table_attach (GTK_TABLE(table), mw->followup_to, 1, 2, 4, 5,
				  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

		w = gtk_label_new ( _("Reply-To:") );
		gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
		eventbox = gtk_event_box_new ();
		gtk_container_add (GTK_CONTAINER(eventbox), w);
		gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), eventbox,
			_("The e-mail account where mail replies to "
			  "your posted message should go.  This is only "
			  "needed if it differs from the \"From\" "
			  "header."), "");
		gtk_table_attach (GTK_TABLE(table), eventbox, 0, 1, 5, 6,
				  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
		mw->reply_to = gtk_entry_new ();
		gtk_table_attach (GTK_TABLE(table), mw->reply_to, 1, 2, 5, 6,
				  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

		gtk_widget_show_all (table);
	}
}

static void
message_window_save_cb (GtkWidget *widget, Message *msg)
{
	GtkWidget *file_entry = NULL;
	
	pan_lock();
	
	file_entry = gtk_file_selection_new (_("Save message to file"));

	gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (file_entry));

	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (file_entry)->cancel_button),
				   "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (file_entry));

	gtk_object_set_data (GTK_OBJECT (file_entry), "msg_id", msg->message_id);

	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (file_entry)->ok_button),
				"clicked",
				GTK_SIGNAL_FUNC (message_window_save_ok_clicked),
				file_entry);

	gtk_widget_show_all (file_entry);
	pan_unlock();
}

static void
message_window_cut_cb (GtkWidget *widget, Message *msg)
{
	g_return_if_fail (msg->window->body);

	pan_lock();
	gtk_editable_cut_clipboard (GTK_EDITABLE(msg->window->body));
	pan_unlock();
}


static void
message_window_copy_cb (GtkWidget *widget, Message *msg)
{
	g_return_if_fail ( msg->window->body );

	pan_lock();
	gtk_editable_copy_clipboard (GTK_EDITABLE(msg->window->body));
	pan_unlock();
}


static void
message_window_paste_cb (GtkWidget *widget, Message *msg)
{
	pan_lock();
	gtk_editable_paste_clipboard (GTK_EDITABLE(msg->window->body));
	pan_unlock();
}


static void
message_window_destroy (MessageWindow *mw)
{
	message_free (mw->message);

	pan_lock();
	gtk_container_foreach (GTK_CONTAINER (mw->window), GTK_SIGNAL_FUNC (gtk_widget_destroy), NULL);
	gtk_widget_destroy (mw->window);
	pan_unlock();

	g_free (mw);
}


static void
message_window_close (GtkWidget *widget, Message *msg)
{
	message_window_destroy (msg->window);
}

/*---[ create_read_info_pane ]----------------------------------------
 * From:
 * Organization:
 * Reply-To: (optional)
 * Newsgroups:
 * Followup-To: (optional)
 * Date:
 * Subject:
 *--------------------------------------------------------------------*/
static void
populate_read_info_pane (MessageWindow *mw, GtkTable* table, const Message* msg)
{
	GtkWidget *label;
        int row = 0;

	pan_lock();

	/* from */
	label = gtk_label_new ( _("From:") );
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, row, row+1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->from = gtk_label_new (msg->from);
	gtk_misc_set_alignment (GTK_MISC (mw->from), 0.0, 0.0);
	gtk_table_attach (GTK_TABLE (table), mw->from, 1, 2, row, row+1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

	/* organization */
	++row;
	label = gtk_label_new ( _("Organization:") );
	gtk_misc_set_alignment( GTK_MISC (label), 0.0, 0.0);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, row, row+1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	/* FIXME: should actually get the Organization: from the message,
	   not say None */
	mw->organization = gtk_label_new ("None");
	gtk_misc_set_alignment (GTK_MISC (mw->organization), 0.0, 0.0);
	gtk_table_attach (GTK_TABLE (table), mw->organization, 1, 2, row, row+1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

        /* reply-to */
        if ( !*msg->reply_to && !strcmp(msg->from,msg->reply_to) ) {
		++row;
		label = gtk_label_new ( _("Reply-To:") );
		gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
		gtk_table_attach (GTK_TABLE(table), label, 0, 1, row,row+1,
				  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
		mw->reply_to = gtk_label_new (msg->reply_to);
		gtk_misc_set_alignment (GTK_MISC (mw->reply_to), 0.0, 0.0);
		gtk_table_attach (GTK_TABLE (table), mw->reply_to, 1, 2, row, row+1,
				  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);
	}

        /* newsgroups */
	++row;
	label = gtk_label_new ( _("Newsgroups:") );
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, row, row+1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->newsgroups = gtk_label_new (msg->newsgroups);
	gtk_misc_set_alignment (GTK_MISC(mw->newsgroups), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL(mw->newsgroups), TRUE);
	gtk_table_attach (GTK_TABLE (table), mw->newsgroups, 1, 2, row, row+1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);


        /* followup-to */
        if ( !*msg->followup_to && !strcmp(msg->newsgroups,msg->followup_to) ) {
		++row;
		label = gtk_label_new ( _("Followup-To:") );
		gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
		gtk_table_attach (GTK_TABLE(table), label, 0, 1, row,row+1,
				  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
		mw->followup_to = gtk_label_new (msg->followup_to);
		gtk_misc_set_alignment (GTK_MISC (mw->followup_to), 0.0, 0.5);
		gtk_table_attach (GTK_TABLE (table), mw->followup_to, 1, 2, row, row+1,
				  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);
	}

        /* date */
	++row;
	label = gtk_label_new ( _("Date:") );
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, row, row+1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->date = gtk_label_new (msg->date);
	gtk_misc_set_alignment (GTK_MISC (mw->date), 0.0, 0.0);
	gtk_table_attach (GTK_TABLE (table), mw->date, 1, 2, row, row+1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

        /* subject */
	++row;
	label = gtk_label_new ( _("Subject:") );
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, row, row+1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->subject = gtk_label_new (msg->subject);
	gtk_misc_set_alignment (GTK_MISC (mw->subject), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL(mw->subject), TRUE);
	gtk_table_attach (GTK_TABLE (table), mw->subject, 1, 2, row, row+1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

	pan_unlock();
}

static GtkWidget *
create_read_info_pane (MessageWindow *mw)
{
	GtkWidget *table;

	pan_lock();
	table = gtk_table_new (7, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	pan_unlock();

	populate_read_info_pane ( mw, GTK_TABLE(table), mw->message );

	return table;
}


static void
update_title_with_subject (GtkWidget *widget, Message *msg)
{
	gtk_window_set_title (GTK_WINDOW (msg->window->window),
				  gtk_entry_get_text (GTK_ENTRY (widget)));
}


/*--------------------------------------------------------------------
 * Newsgroups:
 * CC:
 * Subject:
 *--------------------------------------------------------------------*/
static GtkWidget*
create_post_info_pane (MessageWindow *mw)
{
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *eventbox;
	gchar *p;
	const group_data *gdata;
	Message *msg = mw->message;

	pan_lock();

	table = gtk_table_new (4, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);

	label = gtk_label_new ( _("From:") );
	gtk_misc_set_alignment (GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	mw->from = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(table), mw->from, 1, 2, 0, 1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

	if ((p = gnome_config_get_string ("/Pan/User/Email"))) {
		gtk_entry_set_text (GTK_ENTRY(mw->from), p);
		g_free(p);
	}

#if 0
	label = gtk_label_new (_("Organization:"));
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,
			 GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	mw->organization = gtk_entry_new();
	gtk_table_attach (GTK_TABLE(table), mw->organization, 1, 2, 1, 2,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

	if ((p = gnome_config_get_string ("/Pan/User/Organization"))) {
		gtk_entry_set_text(GTK_ENTRY(mw->organization), p);
		g_free(p);
	}
#endif

	label = gtk_label_new ( _("Post To Groups:") );
	gtk_misc_set_alignment (GTK_MISC(label), 0.0, 0.5);
	eventbox = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER(eventbox), label);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), eventbox,
						  _("Groups to Post this message to"), "");
	gtk_table_attach (GTK_TABLE(table), eventbox, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->newsgroups = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(table), mw->newsgroups, 1, 2, 1, 2,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);



	label = gtk_label_new ( _("Mail To:") );
	gtk_misc_set_alignment (GTK_MISC(label), 0.0, 0.5);
	eventbox = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER(eventbox), label);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), eventbox,
						  _("Users to Mail this message to"), "");
	gtk_table_attach (GTK_TABLE(table), eventbox, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->users = gtk_entry_new();
	gtk_table_attach (GTK_TABLE(table), mw->users, 1, 2, 2, 3,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);



	/* set the newsgroups/users entry field... */
	p = NULL;
	if (msg && msg->subject) /* if we've got a message */
	{
		if (msg->type==NNTP_ARTICLE_REPLY) /* follow-up post */
		{
			if (msg->followup_to && *msg->followup_to) /* if followup-to specified */
			{
				if (!g_strcasecmp ("poster", msg->followup_to)) /* Followup-To: poster */
				{
					if (msg->reply_to && *msg->reply_to)
						gtk_entry_set_text (GTK_ENTRY(mw->users), msg->reply_to);
					else
						gtk_entry_set_text (GTK_ENTRY(mw->users), msg->from);
					gnome_ok_dialog_parented (
						_("``Followup-To: poster'': sending email to author."),
						GTK_WINDOW(mw->window));
				}
				else
				{
					gtk_entry_set_text (GTK_ENTRY(mw->newsgroups), msg->followup_to);
				}
			}
			else if (msg->newsgroups && *msg->newsgroups)
			{
				gtk_entry_set_text (GTK_ENTRY(mw->newsgroups), msg->newsgroups);
			}
		}
		else if (msg->type==EMAIL_REPLY || msg->type==EMAIL_POST) /* we're sending e-mail */
		{
			if (msg->reply_to && *msg->reply_to)
				gtk_entry_set_text (GTK_ENTRY(mw->users), msg->reply_to);
			else if (msg->from && *msg->from)
				gtk_entry_set_text (GTK_ENTRY(mw->users), msg->from);
		}
	}
	else if (((gdata = articlelist_get_current_group())) != NULL)
	{
		/* if nothing else, try the current group.  This may be email, if
		we were to let the user send email cold.  The but we have no way
		of knowing if the user here wants to send mail or post a message.
		This should be fixed by associating the MessageWindowType with
		a MessageWindow, rather than a Message.  FIXME(jason) */
		gtk_entry_set_text (GTK_ENTRY(mw->newsgroups), gdata->name);
	}


	label = gtk_label_new ( _("Subject:") );
	gtk_misc_set_alignment (GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, 3, 4,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	/* set the subject of the new message */
	mw->subject = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(table), mw->subject, 1, 2, 3, 4,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);
	/* if this is a reply... */
	if (mw->message && mw->message->subject)
	{
		if (!g_strncasecmp ("Re: ", mw->message->subject, 4))
		{
			gtk_entry_set_text (GTK_ENTRY(mw->subject), mw->message->subject);
		}
		else
		{
			gchar* pch = g_strconcat ( "Re: ", mw->message->subject, NULL );
			gtk_entry_set_text(GTK_ENTRY(mw->subject), pch);
			g_free (pch);
		}
	}

	gtk_signal_connect (GTK_OBJECT (mw->subject), "changed",
				update_title_with_subject, mw->message);

	pan_unlock();

	return table;
}

static GdkColor quoted_col = {0, 0, 0, 0xbbbb};

static void
ensure_quoted_col_inited (void)
{
	static gboolean quoted_col_inited = FALSE;

	if (!quoted_col_inited)
	{
		GdkColormap *cmap = NULL;

		pan_lock ();
		cmap = gdk_colormap_get_system ();
		if (gdk_color_alloc (cmap, &quoted_col))
			quoted_col_inited = TRUE;
		else
			g_error (_("couldn't allocate \"reply\" color"));
		pan_unlock ();
	}
}

static GtkWidget*
create_post_body_pane (MessageWindow *mw)
{
	GtkWidget *scrolled_window;
	gchar* sig_file;

	ensure_quoted_col_inited ();

	pan_lock();

//	if ( !mw->body )
		mw->body = gtk_text_new (NULL, NULL);

	gtk_text_set_word_wrap (GTK_TEXT (mw->body), TRUE);
	gtk_text_set_editable (GTK_TEXT (mw->body), TRUE);
	widget_set_font (GTK_WIDGET(mw->body), message_body_font);

	gtk_text_freeze (GTK_TEXT (mw->body));

	/* insert message body */
	if (mw->message->type==NNTP_ARTICLE_REPLY || mw->message->type==EMAIL_REPLY)
	{
		gchar *attrib = NULL;
		gchar **sd = NULL;
		gchar *pch=NULL, *tmp=NULL;
		int i = 0;

		/* precede quoted material with attribution
		   line containing original author */
		pch = pan_substitute (attribution_line, "%i", mw->message->message_id);
		tmp = pan_substitute (pch, "%a", mw->message->from);
		g_free (pch);
		pch = tmp;
		tmp = pan_substitute (pch, "%d", mw->message->date);
		g_free (pch);
		pch = tmp;
		attrib = g_strdup_printf ("%s\n", pch);
		gtk_text_insert (GTK_TEXT (mw->body),
			NULL, &mw->body->style->black, NULL,
			attrib, -1);
		g_free (attrib);
		g_free (pch);

		/* quote the original message.
		   It's good practice for newsreaders to use '> ' as the quote
		   prefix for newly-quoted, and '>' for repetitively quoted
		   text.   Also, included text should NOT include the original
		   article's signature unless explicitly requested by the user.
		   Finally, strip out the carriage returns as the only thing
		   they do is show up badly on some fonts.
		   FIXME: "quote sig" to prefs */
		sd = g_strsplit (mw->message->body, "\n", -1);
		for (i=0; sd[i]; ++i) {
			const gchar *prefix = (*sd[i]=='>') ? ">" : "> ";
			gchar *line = NULL;
			size_t len = strlen(sd[i]);
			if (len && (sd[i][len-1]=='\r')) /* trip out linefeeds */
				sd[i][--len] = '\0';
			if (len==3 && !strcmp(sd[i],"-- ")) /* sig reached... */
				break;
			line = g_strdup_printf ("%s%s\n", prefix, sd[i]);
			gtk_text_insert (GTK_TEXT (mw->body), NULL, &quoted_col, NULL, line, -1);
			g_free (line);
		}
		g_strfreev (sd);

		gtk_text_insert (GTK_TEXT (mw->body), NULL, &mw->body->style->black, NULL, "\n\n", 1);
	}

	/* attach signature */
	sig_file = gnome_config_get_string ( "/Pan/User/Signature_File" );
	if (sig_file && strlen(sig_file))
	{
		GArray* sig = read_file (sig_file);
		if (sig!=NULL)
		{
			gtk_text_insert (GTK_TEXT (mw->body), NULL, &mw->body->style->black, NULL, "\n-- \n\n", 5);
			gtk_text_insert (GTK_TEXT (mw->body), NULL, &mw->body->style->black, NULL, sig->data, sig->len);
			gtk_text_set_point (GTK_TEXT (mw->body), 0);
			g_array_free ( sig, TRUE );
		}
	}
	g_free (sig_file);

	gtk_text_thaw (GTK_TEXT (mw->body));

	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (scrolled_window), mw->body);

	pan_unlock();

	return scrolled_window;
}


/*---[ create_read_body_pane ]----------------------------------------
 * will be called from Reading code
 * This should NOT be called within a pan lock
 *--------------------------------------------------------------------*/
static GtkWidget *
create_read_body_pane (GtkWidget *body_widget, Message *msg)
{
	gchar* sig_file;

	ensure_quoted_col_inited ();

	pan_lock();

	if (!body_widget)
		body_widget = gtk_text_new (NULL, NULL);
	widget_set_font (GTK_WIDGET(body_widget), message_body_font);
	/*gtk_text_set_word_wrap (GTK_TEXT (body_widget), TRUE);*/

	gtk_text_freeze (GTK_TEXT (body_widget));

	/* insert message body */
	if (msg->type == NNTP_ARTICLE_READ)
	{
		int i;
		gchar **sd = g_strsplit (msg->body, "\n", -1);
		for (i=0; sd[i]; ++i) {
			gchar *line = sd[i];
			size_t len = strlen(line);
			GdkColor *pFG = *line=='>' ? &quoted_col : &body_widget->style->black;
			if (line[len-1] == '\r') /* strip out carriage returns */
				line[--len] = '\0';
			line[len++] = '\n';
			gtk_text_insert (GTK_TEXT(body_widget), NULL, pFG, NULL, line, len);
		}
		g_strfreev (sd);
		gtk_text_set_point (GTK_TEXT(body_widget), 0);
	}

	/* attach signature */
	if (msg->type != NNTP_ARTICLE_READ)
	{
		sig_file = gnome_config_get_string ( "/Pan/User/Signature_File" );
		if ( sig_file && strlen(sig_file) )
		{
			GArray* sig = read_file ( sig_file );
			if ( sig != NULL )
			{
				gtk_text_insert (GTK_TEXT (body_widget), NULL, &body_widget->style->black, NULL, "\n-- \n\n", 5);
				gtk_text_insert (GTK_TEXT (body_widget), NULL, &body_widget->style->black, NULL, sig->data, sig->len);
				gtk_text_set_point (GTK_TEXT (body_widget), 0);
				g_array_free ( sig, TRUE );
			}
			g_free ( sig_file );
		}
	}

	gtk_text_thaw (GTK_TEXT (body_widget));

	gtk_widget_show (body_widget);

	pan_unlock();

	return body_widget;
}

#if 0
void
update_info_pane (MessageWindow *mw, Message *msg)
{
	//GList* l;

	gtk_label_set_text (GTK_LABEL (mw->from), msg->from);
	gtk_label_set_text (GTK_LABEL (mw->organization), "None");
	gtk_label_set_text (GTK_LABEL (mw->date), msg->date);
	gtk_label_set_text (GTK_LABEL (mw->newsgroups), msg->newsgroups);
	gtk_label_set_text (GTK_LABEL (mw->subject), msg->subject);
        gtk_label_set_text (GTK_LABEL (mw->followup_to), msg->followup_to);
        gtk_label_set_text (GTK_LABEL (mw->reply_to), msg->reply_to);
	gtk_window_set_title (GTK_WINDOW (mw->window), msg->subject);

}
#endif


/*---[ update_body_pane ]---------------------------------------------
 * will only be called for messages being "read"
 *--------------------------------------------------------------------*/
void
update_body_pane (GtkWidget *body_widget, const gchar *body)
{
	GtkText *text = NULL;

	/* sanity checks */
	g_return_if_fail (body_widget!=NULL);
	g_return_if_fail (GTK_IS_TEXT(body_widget));

	ensure_quoted_col_inited ();
	text = GTK_TEXT(body_widget);

	pan_lock();
	gtk_text_set_point (text, 0);
	gtk_editable_delete_text (GTK_EDITABLE(body_widget), 0, -1);
	if (body != NULL)
	{
		gint i = 0;
		gchar **sd = g_strsplit (body, "\n", -1);
		gtk_text_freeze (text);

		/* insert new text.
		 * + colorize quoted lines
		 * + strip out carriage returns
		 */
		for (i=0; sd[i]; ++i) {
			gchar *line = sd[i];
			size_t len = strlen(line);
			GdkColor *pFG = *line=='>' ? &quoted_col : &body_widget->style->black;
			if (len && (line[len-1] == '\r'))
				line[--len] = '\0';
			line[len++] = '\n';
			gtk_text_insert (text, NULL, pFG, NULL, line, len);
		}

		gtk_text_thaw (text);
		g_strfreev (sd);
	}

	pan_unlock();
}

#if 0
static void
message_window_update (MessageWindow *mw, Message *msg)
{
	if (mw && mw->body)
	{
		update_info_pane (mw, msg);
		update_body_pane (mw->body, msg->body);
	}
	else
	{
		update_body_pane (text, msg->body);
	}
}
#endif


void
message_post_window (GtkWidget *widget, Message *msg)
{
	if ( !msg || msg->window )
		msg = message_new (grouplist_get_current_server (), NULL, NULL);
	
	msg->type = NNTP_ARTICLE_POST;
	message_window_new (msg);
}


static void
message_post_window_create (MessageWindow *mw)
{
	GtkWidget *vbox = NULL;
	GtkWidget *info_post_pane = NULL;
	GtkWidget *body_post_pane = NULL;

	pan_lock();
	mw->window = gnome_app_new ("Post", "New Message");
	vbox = gtk_vbox_new (FALSE, 0);
	gnome_app_set_contents (GNOME_APP (mw->window), vbox);
	gtk_window_set_default_size (GTK_WINDOW (mw->window), 550, 410);
	gnome_app_create_menus_with_data (GNOME_APP (mw->window), message_post_main_menu, mw->message);
	gnome_app_create_toolbar_with_data (GNOME_APP (mw->window), message_post_toolbar, mw->message);
	pan_unlock();

	info_post_pane = create_post_info_pane (mw);
	body_post_pane = create_post_body_pane(mw);

	pan_lock();
	gtk_box_pack_start (GTK_BOX(vbox), info_post_pane, FALSE, FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(vbox), gtk_hseparator_new(), FALSE, FALSE, 2);
	gtk_box_pack_start (GTK_BOX(vbox), body_post_pane, TRUE, TRUE, 0);
	//gtk_widget_show_all (mw->window);
	pan_unlock();
}


void
message_read_window (GtkWidget *widget, Message *msg)
{
	if ( !msg ) {
		const char *message_id = articlelist_get_selected_message_id ();
		text_set_from_message_id ( articlelist_get_current_server (),
				articlelist_get_current_group (),
				message_id, NNTP_ARTICLE_READ, TRUE );
		return;
	}

	msg->type = NNTP_ARTICLE_READ;
	message_window_new (msg);
}


/*--------------------------------------------------------------------
 *
 *--------------------------------------------------------------------*/
static void
message_read_window_create (MessageWindow *mw)
{
	GtkWidget *vbox = NULL;
	GtkWidget *read_body_pane = NULL;
	GtkWidget *scrolled_window = NULL;

	/* create a new window */
	pan_lock();
	mw->window = gnome_app_new ("Read", mw->message->subject);
	vbox = gtk_vbox_new (FALSE, 0);
	gnome_app_set_contents (
		GNOME_APP (mw->window), vbox);
	gtk_window_set_default_size (
		GTK_WINDOW (mw->window), 550, 410);
	gnome_app_create_menus_with_data (
		GNOME_APP(mw->window), message_read_main_menu, mw->message);
	gnome_app_create_toolbar_with_data (
		GNOME_APP(mw->window), message_read_toolbar, mw->message);
	pan_unlock();

	/* create read info pane */
	mw->read_info_pane = create_read_info_pane (mw);

	/* create read body pane */
	read_body_pane = create_read_body_pane (NULL, mw->message);

	pan_lock();

	/* add info pane */
	gtk_box_pack_start (
		GTK_BOX (vbox), mw->read_info_pane,
		FALSE, FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (
		GTK_BOX (vbox), gtk_hseparator_new(),
		FALSE, FALSE, 2);

	/* add scrolled body window */
	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (
		GTK_SCROLLED_WINDOW (scrolled_window),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC);
	gtk_container_add (
		GTK_CONTAINER (scrolled_window),
		read_body_pane);
	gtk_box_pack_start (
		GTK_BOX(vbox), scrolled_window,
		TRUE, TRUE, 0);
		
	pan_unlock();
}


/*--------------------------------------------------------------------
 *
 *--------------------------------------------------------------------*/

void
message_reply_external (GtkWidget *widget, Message *msg)
{
	gchar *exec = NULL;
	gchar *subject = NULL;
	gchar *author = NULL;
	gchar *body = NULL;
	gchar *filename = NULL;
	gchar *p = NULL;
	const article_data *adata = NULL;

	/* What are we replying to anyway ? */
	if (msg) {
		subject = g_strdup (msg->subject);
		author = g_strdup (msg->from);
		body = acache_load_body (msg->message_id);
	} else if ((adata = articlelist_get_selected_adata ())) {
		subject = g_strdup (adata->subject);
		author = g_strdup (adata->author);
		body = acache_load_body (adata->message_id);
	} else {
		return;
	}
	
	if (!(body && author && subject)) {
		return;	
	}
	
	exec = g_strdup (external_mailer);

	/* If %t is being used, write the body to a temporary file */
	if ((strstr(external_mailer, "%t")) != NULL) {
		FILE *f = NULL;
		gchar **sd = NULL;
		gint i;

		if ((filename = pan_make_temp (&f)) == NULL) {
			return;
		}

		/* Quote the body */
		sd = g_strsplit (body, "\n", -1);
		for (i=0; sd[i]; ++i) {
			const gchar *prefix = (*sd[i]=='>') ? ">" : "> ";
			gchar *line = NULL;
			size_t len = strlen(sd[i]);
			if (len && (sd[i][len-1]=='\r')) /* trip out linefeeds */
				sd[i][--len] = '\0';
			if (len==3 && !strcmp(sd[i],"-- ")) /* sig reached... */
				break;
			line = g_strdup_printf ("%s%s\n", prefix, sd[i]);
			fputs (line, f);
			g_free (line);
		}
		g_strfreev (sd);
		fclose (f);
		
		p = pan_substitute (exec, "%t", filename);
		g_free (filename);
		g_free (exec);
		exec = p;
	}

	/* Add the Re: to the subject if necessary */
	p = add_re (subject);
	g_free (subject);
	subject = p;

	/* Expand the rest of the command string */
	p = pan_substitute (exec, "%r", author);
	g_free (exec);
	exec = p;
	p =  pan_substitute (exec, "%s", subject);
	g_free (exec);
	exec = p;

	/* Launch the command */
	gnome_execute_shell (NULL, exec);
	
	/* Clean 'em up */
	g_free (subject);
	g_free (author);
	g_free (body);
	g_free (exec);
}


void
message_reply_window (GtkWidget *widget, Message *msg)
{
	if (!msg) {
		const char *message_id = articlelist_get_selected_message_id ();
		if (message_id)
			text_set_from_message_id (
				articlelist_get_current_server (),
				articlelist_get_current_group (),
				message_id, EMAIL_REPLY, TRUE );
		return;
	}

	msg->type = EMAIL_REPLY;
	message_window_new (message_copy (msg));
}

void
message_followup_window (GtkWidget *widget, Message *msg)
{
	if ( !msg ) {
		const char *message_id = articlelist_get_selected_message_id ();
		text_set_from_message_id (
			articlelist_get_current_server (),
			articlelist_get_current_group (),
			message_id, NNTP_ARTICLE_REPLY, TRUE );
		return;
	}

	msg->type = NNTP_ARTICLE_REPLY;
	message_window_new (message_copy (msg));
}

static void
message_reply_window_create (MessageWindow *mw)
{
	GtkWidget *vbox = NULL;
	GtkWidget *post_info_pane = NULL;
	GtkWidget *post_body_pane = NULL;
	gchar* pch = NULL;

	if ( !g_strncasecmp ( "Re: ", mw->message->subject, 4 ) )
	{
		pch = g_strdup (mw->message->subject);
	}
	else
		pch = g_strconcat ( "Re: ", mw->message->subject, NULL );

	pan_lock();
	mw->window = gnome_app_new ("Reply", pch);
	vbox = gtk_vbox_new (FALSE, 0);
	gnome_app_set_contents (GNOME_APP (mw->window), vbox);
	gtk_window_set_default_size (GTK_WINDOW (mw->window), 550, 410);
	pan_unlock();

	post_info_pane = create_post_info_pane (mw);
	post_body_pane = create_post_body_pane (mw);

	pan_lock ();
	gnome_app_create_menus_with_data (GNOME_APP (mw->window), message_post_main_menu, mw->message);
	gnome_app_create_toolbar_with_data (GNOME_APP (mw->window), message_post_toolbar, mw->message);
	gtk_box_pack_start (GTK_BOX(vbox), post_info_pane, FALSE, FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(vbox), gtk_hseparator_new(), FALSE, FALSE, 2);
	gtk_box_pack_start (GTK_BOX(vbox), post_body_pane, TRUE, TRUE, 0);
	//gtk_widget_show_all (mw->window);
	pan_unlock();

	g_free (pch);
}

void
message_window_new (Message *msg)
{
	MessageWindow *mw = g_malloc0 (sizeof(MessageWindow));

	mw->message = msg;
	msg->window = mw;

	switch (msg->type) {
	case NNTP_ARTICLE_READ:
		message_read_window_create (mw);
		break;
	case NNTP_ARTICLE_POST:
		message_post_window_create (mw);
		break;
	case NNTP_ARTICLE_REPLY:
	case EMAIL_REPLY:
		message_reply_window_create (mw);
		break;
	case EMAIL_FORWARD:
	default:
		break;
	}

	if (mw->window)
		gui_popup_draw (mw->window, Pan.window);
}
