/*
 *  Copyright (C) 1998-99 Luca Deri <deri@unipi.it>
 *                      
 *  			  Centro SERRA, University of Pisa
 *  			  http://www-serra.unipi.it/
 *  					
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "ntop.h"

/* #define DEBUG  */

typedef union {
  int32_t al;
  char ac;
} align;


/* Extern */
extern char domainName[MAXHOSTNAMELEN];
extern unsigned short alternateColor, maxNameLen;
extern unsigned int webMode;

#ifdef ASYNC_ADDRESS_RESOLUTION
unsigned int addressQueueLen, maxAddressQueueLen, addressQueueHead, addressQueueTail;
extern TrafficCounter droppedAddresses;
#ifdef MULTITHREADED
extern pthread_mutex_t addressQueueMutex;
extern ConditionalVariable queueAddressCondvar;
#ifdef USE_SEMAPHORES
extern sem_t queueAddressSem;
#endif
#endif
struct hnamemem *addressQueue[ADDRESS_QUEUE_LENGTH+1];
#endif

extern void updateHostNameInfo(unsigned long numeric, char* symbolic);
extern unsigned char isLocalAddress(struct in_addr *addr);
extern unsigned char isBroadcastAddress(struct in_addr *addr);

#ifdef HAVE_GDBM_H
extern GDBM_FILE gdbm_file;
#ifdef MULTITHREADED
extern pthread_mutex_t gdbmMutex;
#endif
#else /* HAVE_GDBM_H */
extern struct hnamemem* hnametable[HASHNAMESIZE];
#endif

/* Forward */
char* intoa(struct in_addr addr);
char* savestr(const char *str);

/* Global */
struct enamemem enametable[HASHNAMESIZE];
static char hex[] = "0123456789ABCDEF";

/* ************************************ */


#if defined(MULTITHREADED) && defined(ASYNC_ADDRESS_RESOLUTION)


void queueAddress(struct hnamemem* elem) {
  /****************************
   - If the queue is full then wait until a slot is freed
   - If the queue is getting full then periodically wait
     until a slot is freed
  *****************************/
  if(addressQueueLen >= ADDRESS_QUEUE_LENGTH) {
#ifdef DEBUG
    printf("Dropping address!!! [addr queue=%d/max=%d]\n", 
	   addressQueueLen, maxAddressQueueLen);
#endif
    /* The address is kept in numerical form
       (i.e. no conversion will take place)
    */
    strcpy(elem->name, intoa(elem->addr));

    droppedAddresses++;    
#ifdef HAVE_SCHED_H
    sched_yield(); /* Allow other threads (dequeue) to run */
#endif
  } else {
    accessMutex(&addressQueueMutex);
    addressQueue[addressQueueHead] = elem;
    addressQueueHead = ((addressQueueHead+1) % ADDRESS_QUEUE_LENGTH);
    addressQueueLen++;
    if(addressQueueLen > maxAddressQueueLen) {
      maxAddressQueueLen = addressQueueLen;
#ifdef DEBUG
	  printf("Max queue len: %ld\n", maxAddressQueueLen);
#endif
	}

    releaseMutex(&addressQueueMutex);
#ifdef DEBUG
    printf("Queued address... [addr queue=%d/max=%d]\n", 
	   addressQueueLen, maxAddressQueueLen);
#endif

#ifdef DEBUG_THREADS
    printf("+ [addr queue=%d/max=%d]\n", addressQueueLen, maxAddressQueueLen);
#endif

#ifdef USE_SEMAPHORES
    incrementSem(&queueAddressSem);
#else
    signalCondvar(&queueAddressCondvar);
#endif
  }
}

/* ************************************ */

void resolveAddress(char* symAddr, struct in_addr *hostAddr, 
		    short keepAddressNumeric) {
  int addr, raddr, i;
  struct hostent *hp = NULL;
  char* res;
#ifdef HAVE_GDBM_H
  datum key_data;
  datum data_data;
  char tmpBuf[32];
#endif

  /* This is necessary due to BIG/LITTLE endian crap */
  raddr = ntohl(hostAddr->s_addr);
  addr = hostAddr->s_addr; 

#ifdef HAVE_GDBM_H
  sprintf(tmpBuf, "%u", addr);
  key_data.dptr = tmpBuf;
  key_data.dsize = strlen(tmpBuf)+1;
  
#ifdef MULTITHREADED
  accessMutex(&gdbmMutex);
#endif 

  data_data = gdbm_fetch(gdbm_file, key_data);

#ifdef MULTITHREADED
  releaseMutex(&gdbmMutex);
#endif 

  if(data_data.dptr != NULL) {
#ifdef GDBM_DEBUG
    printf("Fetched data (2): '%s' [%s]\n", data_data.dptr, tmpBuf);
#endif
    strcpy(symAddr, data_data.dptr);
    updateHostNameInfo(addr, data_data.dptr);
    free(data_data.dptr);
    return;
  } else {
#ifdef GDBM_DEBUG
    printf("Unable to retrieve %s\n", tmpBuf);
#endif
  }
#endif

  if(!keepAddressNumeric) {
#ifdef DNS_DEBUG
    fprintf(stdout, "Resolving %s...\n", intoa(*hostAddr)); fflush(stdout);
#endif
    hp = (struct hostent*)gethostbyaddr((char*)&raddr, 4, AF_INET);
#ifdef DNS_DEBUG
    fprintf(stdout, "Resolved.\n"); fflush(stdout);
#endif

    if (hp && (hp->h_name)) {
      char *dotp = (char*)hp->h_name;    

      if(domainName[0] != '\0') {
	int len = strlen(dotp)-strlen(domainName);
	  
	if((len > 0) && (strcmp(&dotp[len], domainName) == 0))
	  dotp[len-1] = '\0';
      }
      res = dotp;
    } else
      res = intoa(*hostAddr);
  } else 
    res = intoa(*hostAddr);

  if(strlen(res) > maxNameLen) {
    strncpy(symAddr, res, maxNameLen-3);
    symAddr[maxNameLen] = '\0';
    symAddr[maxNameLen-1] = '.';
    symAddr[maxNameLen-2] = '.';
    symAddr[maxNameLen-3] = '.';
  } else
    strcpy(symAddr, res);

  for(i=0; symAddr[i] != '\0'; i++)
    symAddr[i] = tolower(symAddr[i]);

#ifdef HAVE_GDBM_H
  /* key_data has been set already */
  data_data.dptr = symAddr;
  data_data.dsize = strlen(symAddr)+1;

  updateHostNameInfo(addr, symAddr);

#ifdef MULTITHREADED
  accessMutex(&gdbmMutex);
#endif 
  
  if(gdbm_store(gdbm_file, key_data, data_data, GDBM_REPLACE) != 0)
    printf("Error while adding '%s'\n.\n", symAddr);
  else {
#ifdef GDBM_DEBUG
    printf("Added data: '%s' [%s]\n", symAddr, tmpBuf);
#endif
  }

#ifdef MULTITHREADED
  releaseMutex(&gdbmMutex);
#endif 

#endif /* HAVE_GDBM_H */
}


/* ************************************ */

#ifdef MULTITHREADED

void* dequeueAddress(void* notUsed) {
  struct hnamemem *elem;

  while(1) {
#ifdef DEBUG
    printf("Waiting for address...\n");
#endif

/* LUCA !!!!!!!!!!!!! */
/* #ifdef WIN32 */
	while(addressQueueLen == 0) 
/* #endif */
	{
#ifdef USE_SEMAPHORES
	    waitSem(&queueAddressSem);
#else
		waitCondvar(&queueAddressCondvar);
#endif
	}

    accessMutex(&addressQueueMutex);
    elem = addressQueue[addressQueueTail];

    if(elem == NULL) {
      printf("Internal error: queued address is NULL (%d/%d)\n",
	     addressQueueTail, addressQueueLen);
      releaseMutex(&addressQueueMutex);
      continue;
    }
    addressQueue[addressQueueTail] = NULL; /* Just to keep the table clean */
    addressQueueTail = ((addressQueueTail+1) % ADDRESS_QUEUE_LENGTH);
    addressQueueLen--;
    releaseMutex(&addressQueueMutex);
#ifdef DEBUG_THREADS
    printf("- [addr queue=%d/max=%d]\n", addressQueueLen, maxAddressQueueLen);
#endif

#ifdef DEBUG
    printf("Processing address... [addr queue=%d/max=%d]: %s\n", 
	   addressQueueLen, maxAddressQueueLen, intoa(elem->addr));
#endif

    /* 
       If the queue is full enough then keep 
       remote addresses numeric
    */
    if((addressQueueLen > (ADDRESS_QUEUE_LENGTH/2))
       && (!isLocalAddress(&elem->addr)))
      resolveAddress(elem->name, &elem->addr, 1); /* Keep address numeric */
    else
      resolveAddress(elem->name, &elem->addr, 0);
  
#ifdef DEBUG
    printf("Resolved address %s\n", elem->name);
#endif

#ifdef HAVE_GDBM_H
    /* 
       Free up bucket 
       
       NOTE: do not free elem->name because it is used
       by ipaddr2str.
    */
    free(elem); 
#endif
  } /* endless loop */
	
  return(NULL); /* NOTREACHED */
}

#endif /* defined(MULTITHREADED) && defined(ASYNC_ADDRESS_RESOLUTION) */

#endif

/* ************************************ */

/*
 * A faster replacement for inet_ntoa().
 */
char* intoa(struct in_addr addr)
{
  char *cp, *retStr;
  u_int byte;
  int n;
  static char buf[sizeof(".xxx.xxx.xxx.xxx")];
 
  /*
  if(addr.s_addr == 0x0)
    return("0.0.0.0");
  else if(addr.s_addr == INADDR_BROADCAST) {
    if(!webMode)
      return("<broadcast>");
    else
      return("&lt;broadcast&gt;");   
  }  else if(isBroadcastAddress(&addr)) {
    if(!webMode)
      return("<subnet broadcast>");
    else
      return("&lt;subnet&nbsp;broadcast&gt;");
  }
*/
  cp = &buf[sizeof buf];
  *--cp = '\0';

  n = 4;
  do {
    byte = addr.s_addr & 0xff;
    *--cp = byte % 10 + '0';
    byte /= 10;
    if (byte > 0) {
      *--cp = byte % 10 + '0';
      byte /= 10;
      if (byte > 0)
	*--cp = byte + '0';
    }
    *--cp = '.';
    addr.s_addr >>= 8;
  } while (--n > 0);

  /* Convert the string to lowercase */
  retStr = (char*)(cp+1);
  for(n=0; retStr[n] != '\0'; n++)
    retStr[n] = tolower(retStr[n]);

  return(cp+1);
}

/* ******************************* */

/*
  This function returns a brand new pointer that
  will be handled (e.e. deallocated) by functions 
  other than this
*/
char* ipaddr2str(struct in_addr hostIpAddress) {
  unsigned int addr = hostIpAddress.s_addr, i;
#ifdef HAVE_GDBM_H
  datum key_data;
  datum data_data;
  char tmpBuf[32];

#else
  int idx = addr % HASHNAMESIZE;
#endif
  struct hnamemem *p;
#if !(defined(MULTITHREADED) && defined(ASYNC_ADDRESS_RESOLUTION))
  struct hostent *hp = NULL;
#endif

  /* printf("Searching for %s.\n", intoa(hostIpAddress));  */

  if((addr == INADDR_BROADCAST) || (addr == 0x0))
    return(intoa(hostIpAddress));

  for(;;) {
#ifndef HAVE_GDBM_H
    p = hnametable[idx];

    if(p != NULL) {
      if(p->addr.s_addr == addr) {
#ifdef DEBUG
	printf("Cached %s->%s\n", intoa(hostIpAddress), p->name);
#endif
	break; /* entry found */
      }
    }
#else /* We have installed GDBM */

  sprintf(tmpBuf, "%u", hostIpAddress.s_addr);
  key_data.dptr = tmpBuf;
  key_data.dsize = strlen(tmpBuf)+1;
  
#ifdef MULTITHREADED
  accessMutex(&gdbmMutex);
#endif 

  data_data = gdbm_fetch(gdbm_file, key_data);

#ifdef MULTITHREADED
 releaseMutex(&gdbmMutex);
#endif 

  if(data_data.dptr != NULL) {
#ifdef GDBM_DEBUG
    printf("Fetched data (1): %s [%s]\n", data_data.dptr, tmpBuf);
#endif
    updateHostNameInfo(hostIpAddress.s_addr, data_data.dptr);
    return(data_data.dptr);
  } else {
#ifdef GDBM_DEBUG
    printf("Unable to retrieve %s\n", tmpBuf);
#endif
    p = NULL;
  }

#endif /* HAVE_GDBM_H */

  if(p == NULL) { /* New entry */
#if !(defined(MULTITHREADED) && defined(ASYNC_ADDRESS_RESOLUTION))
    char theAddr[5];
#endif

      p = (struct hnamemem*)malloc(sizeof(struct hnamemem));
      memset(p, 0, sizeof(struct hnamemem));
#ifndef HAVE_GDBM_H
      hnametable[idx] = p;      
#endif
      p->addr.s_addr = addr;

#if defined(MULTITHREADED) && defined(ASYNC_ADDRESS_RESOLUTION)
       p->name = (char*)malloc(maxNameLen);
	   p->name[0] = '\0';
#ifdef WIN32
      /* 
	 Under WIN32 local addresses are resolved immediately
	 because this is much more efficient under this OS 
      */
      if(isLocalAddress(&hostIpAddress))
		resolveAddress(p->name, &hostIpAddress, 0);
      else {
		sprintf(p->name, "*%s*", intoa(hostIpAddress));
		queueAddress(p);
	  }
#else
      sprintf(p->name, "*%s*", intoa(hostIpAddress));
      queueAddress(p);
#endif /* WIN32 */
	  

#else
      memcpy(theAddr, &addr, 4);
      theAddr[0] = '\0';
      addr = ntohl(addr); /* This is necessary due to BIG/LITTLE endian crap */
      hp = (struct hostent*)gethostbyaddr(theAddr, 4, AF_INET);

      if (hp && (hp->h_name)) {
	char dotp[64];

	strcpy(dotp, (char*)hp->h_name);

	if(domainName[0] != '\0') {
	  int len = strlen(dotp)-strlen(domainName);
		  
	  if((len > 0) && (strcmp(&dotp[len], domainName) == 0))
	    dotp[len-1] = '\0';
	}

	p->name = savestr(dotp);

	/* Add '...' at the end of each host name if the name
	 * is longer than maxNameLen chars 
	 * (maxNameLen is the max host name length)
	 * Benedikt Heinen <Benedikt.Heinen@infrasys.ascom.ch>
	 */
	if(strlen(p->name) > maxNameLen) { 
	  p->name[maxNameLen] = '\0';
	  p->name[maxNameLen-1] = '.';
	  p->name[maxNameLen-2] = '.';
	  p->name[maxNameLen-3] = '.';
	}
      } else {
	p->name = savestr(intoa(hostIpAddress));
	/* printf("Unable to find IP address for %s\n", p->name); */
      }
#endif
      break;
    }

#ifndef HAVE_GDBM_H    
    idx = (idx+1)%HASHNAMESIZE;
#else
    break;
#endif
  }

  for(i=0; p->name[i] != '\0'; i++)
      p->name[i] = tolower(p->name[i]);

  return(p->name);
}

/* ************************************ */

static struct enamemem* lookup_emem(const u_char *ep)
{
  u_int i, j, k, len;
  struct enamemem *tp;

  k = (ep[0] << 8) | ep[1];
  j = (ep[2] << 8) | ep[3];
  i = (ep[4] << 8) | ep[5];

  tp = &enametable[(i ^ j) & (HASHNAMESIZE-1)];
  while (tp->e_nxt)
    if (tp->e_addr0 == i &&
	tp->e_addr1 == j &&
	tp->e_addr2 == k)
      return tp;
    else
      tp = tp->e_nxt;
  tp->e_addr0 = i;
  tp->e_addr1 = j;
  tp->e_addr2 = k;
  len = sizeof(*tp);
  tp->e_nxt = (struct enamemem *)malloc(len);
  memset(tp->e_nxt, 0, len);
  if(tp->e_nxt == NULL) {
    printf("lookup_emem: malloc");
    exit(-1);
  }

  return tp;
}

/* ************************************ */

char* etheraddr_string(const u_char *ep)
{
  u_int i, j;
  char *cp;
  struct enamemem *tp;
  char buf[sizeof("00:00:00:00:00:00")];

  tp = lookup_emem(ep);

  if (tp->e_name)
    return (tp->e_name);

  cp = buf;

  if ((j = *ep >> 4) != 0)
    *cp++ = hex[j];
  else
    *cp++ = '0';

  *cp++ = hex[*ep++ & 0xf];

  for(i = 5; (int)--i >= 0;) {
    *cp++ = ':';
    if ((j = *ep >> 4) != 0)
      *cp++ = hex[j];
    else
      *cp++ = '0';

    *cp++ = hex[*ep++ & 0xf];
  }

  *cp = '\0';
  tp->e_name = savestr(buf);
  return (tp->e_name);
}

/* ************************************ */

/* Return a zero'ed hnamemem struct and cuts down on malloc() overhead */
struct hnamemem * newhnamemem(void)
{
  struct hnamemem *p;
  static struct hnamemem *ptr = NULL;
  static u_int num = 0;

  if (num  <= 0) {
    int len;

    num = 64;
    len = num*sizeof(*ptr);
    ptr = (struct hnamemem *)malloc(len);
    memset(ptr, 0, len);
    if (ptr == NULL) {
      printf("newhnamemem: malloc");
      exit(-1);
    }
  }
  --num;
  p = ptr++;
  return (p);
}

/* ************************************ */


char* llcsap_string(u_char sap)
{
  register char *cp;
  static char buf[sizeof("sap 00")];

  cp = buf;
  (void)strcpy(cp, "sap ");
  cp += strlen(cp);
  *cp++ = hex[sap >> 4 & 0xf];
  *cp++ = hex[sap & 0xf];
  *cp++ = '\0';

  /* printf("%s\n", buf); */
  return(buf);
}

/* ************************************ */

/*
  The FDDI code below has been grabbed from
  tcpdump
*/

static u_char fddi_bit_swap[] = {
  0x00, 0x80, 0x40, 0xc0, 0x20, 0xa0, 0x60, 0xe0,
  0x10, 0x90, 0x50, 0xd0, 0x30, 0xb0, 0x70, 0xf0,
  0x08, 0x88, 0x48, 0xc8, 0x28, 0xa8, 0x68, 0xe8,
  0x18, 0x98, 0x58, 0xd8, 0x38, 0xb8, 0x78, 0xf8,
  0x04, 0x84, 0x44, 0xc4, 0x24, 0xa4, 0x64, 0xe4,
  0x14, 0x94, 0x54, 0xd4, 0x34, 0xb4, 0x74, 0xf4,
  0x0c, 0x8c, 0x4c, 0xcc, 0x2c, 0xac, 0x6c, 0xec,
  0x1c, 0x9c, 0x5c, 0xdc, 0x3c, 0xbc, 0x7c, 0xfc,
  0x02, 0x82, 0x42, 0xc2, 0x22, 0xa2, 0x62, 0xe2,
  0x12, 0x92, 0x52, 0xd2, 0x32, 0xb2, 0x72, 0xf2,
  0x0a, 0x8a, 0x4a, 0xca, 0x2a, 0xaa, 0x6a, 0xea,
  0x1a, 0x9a, 0x5a, 0xda, 0x3a, 0xba, 0x7a, 0xfa,
  0x06, 0x86, 0x46, 0xc6, 0x26, 0xa6, 0x66, 0xe6,
  0x16, 0x96, 0x56, 0xd6, 0x36, 0xb6, 0x76, 0xf6,
  0x0e, 0x8e, 0x4e, 0xce, 0x2e, 0xae, 0x6e, 0xee,
  0x1e, 0x9e, 0x5e, 0xde, 0x3e, 0xbe, 0x7e, 0xfe,
  0x01, 0x81, 0x41, 0xc1, 0x21, 0xa1, 0x61, 0xe1,
  0x11, 0x91, 0x51, 0xd1, 0x31, 0xb1, 0x71, 0xf1,
  0x09, 0x89, 0x49, 0xc9, 0x29, 0xa9, 0x69, 0xe9,
  0x19, 0x99, 0x59, 0xd9, 0x39, 0xb9, 0x79, 0xf9,
  0x05, 0x85, 0x45, 0xc5, 0x25, 0xa5, 0x65, 0xe5,
  0x15, 0x95, 0x55, 0xd5, 0x35, 0xb5, 0x75, 0xf5,
  0x0d, 0x8d, 0x4d, 0xcd, 0x2d, 0xad, 0x6d, 0xed,
  0x1d, 0x9d, 0x5d, 0xdd, 0x3d, 0xbd, 0x7d, 0xfd,
  0x03, 0x83, 0x43, 0xc3, 0x23, 0xa3, 0x63, 0xe3,
  0x13, 0x93, 0x53, 0xd3, 0x33, 0xb3, 0x73, 0xf3,
  0x0b, 0x8b, 0x4b, 0xcb, 0x2b, 0xab, 0x6b, 0xeb,
  0x1b, 0x9b, 0x5b, 0xdb, 0x3b, 0xbb, 0x7b, 0xfb,
  0x07, 0x87, 0x47, 0xc7, 0x27, 0xa7, 0x67, 0xe7,
  0x17, 0x97, 0x57, 0xd7, 0x37, 0xb7, 0x77, 0xf7,
  0x0f, 0x8f, 0x4f, 0xcf, 0x2f, 0xaf, 0x6f, 0xef,
  0x1f, 0x9f, 0x5f, 0xdf, 0x3f, 0xbf, 0x7f, 0xff,
};

void extract_fddi_addrs(const struct fddi_header *fddip, 
			char *fsrc, char *fdst)
{
  register int i;
  
  if (1) {
    for (i = 0; i < 6; ++i)
      fdst[i] = fddi_bit_swap[fddip->dhost[i]];
    for (i = 0; i < 6; ++i)
      fsrc[i] = fddi_bit_swap[fddip->shost[i]];
  } else {
    memcpy(fdst, (char *)fddip->dhost, 6);
    memcpy(fsrc, (char *)fddip->shost, 6);
  }
}

/* *************************************

   Code "inherited" from nslookup

   ************************************* */

#define NS_INT16SZ      2       /* #/bytes of data in a u_int16_t */

#ifndef NS_GET16
#define NS_GET16(s, cp) { \
        register u_char *t_cp = (u_char *)(cp); \
        (s) = ((u_int16_t)t_cp[0] << 8) \
            | ((u_int16_t)t_cp[1]) \
            ; \
        (cp) += NS_INT16SZ; \
}
#endif

u_int ns_get16(const u_char *src) {
  u_int dst;
  
  NS_GET16(dst, src);
  return (dst);
}

int printable(int ch) {
  return (ch > 0x20 && ch < 0x7f);
}

int special(int ch) {
  switch (ch) {
  case 0x22: /* '"' */
  case 0x2E: /* '.' */
  case 0x3B: /* ';' */
  case 0x5C: /* '\\' */
    /* Special modifiers in zone files. */
  case 0x40: /* '@' */
  case 0x24: /* '$' */
    return (1);
  default:
    return (0);
  }
}

int ns_name_ntop(const u_char *src, char *dst, size_t dstsiz) {
  const u_char *cp;
  char *dn, *eom;
  u_char c;
  u_int n;
  static char digits[] = "0123456789";

  cp = src;
  dn = dst;
  eom = dst + dstsiz;

  while ((n = *cp++) != 0) {
    if ((n & NS_CMPRSFLGS) != 0) {
      /* Some kind of compression pointer. */
      errno = EMSGSIZE;
      return (-1);
    }
    if (dn != dst) {
      if (dn >= eom) {
	errno = EMSGSIZE;
	return (-1);
      }
      *dn++ = '.';
    }
    if (dn + n >= eom) {
      errno = EMSGSIZE;
      return (-1);
    }
    for ((void)NULL; n > 0; n--) {
      c = *cp++;
      if (special(c)) {
	if (dn + 1 >= eom) {
	  errno = EMSGSIZE;
	  return (-1);
	}
	*dn++ = '\\';
	*dn++ = (char)c;
      } else if (!printable(c)) {
	if (dn + 3 >= eom) {
	  errno = EMSGSIZE;
	  return (-1);
	}
	*dn++ = '\\';
	*dn++ = digits[c / 100];
	*dn++ = digits[(c % 100) / 10];
	*dn++ = digits[c % 10];
      } else {
	if (dn >= eom) {
	  errno = EMSGSIZE;
	  return (-1);
	}
	*dn++ = (char)c;
      }
    }
  }
  if (dn == dst) {
    if (dn >= eom) {
      errno = EMSGSIZE;
      return (-1);
    }
    *dn++ = '.';
  }
  if (dn >= eom) {
    errno = EMSGSIZE;
    return (-1);
  }
  *dn++ = '\0';
  return (dn - dst);
}

int dn_skipname(const u_char *ptr, const u_char *eom); /* forward */
int ns_name_uncompress(const u_char *msg, 
		       const u_char *eom, const u_char *src,
		       char *dst, size_t dstsiz);

char* res_skip_rr(char *cp, char *eom)
{
  int tmp;
  int dlen;

  if ((tmp = dn_skipname((u_char *)cp, (u_char *)eom)) == -1)
    return (NULL);			/* compression error */
  cp += tmp;
  if ((cp + RRFIXEDSZ) > eom)
    return (NULL);
  cp += INT16SZ;	/* 	type 	*/
  cp += INT16SZ;	/* 	class 	*/
  cp += INT32SZ;	/* 	ttl 	*/
  dlen = ns_get16((u_char*)cp);
  cp += INT16SZ;	/* 	dlen 	*/
  cp += dlen;
  if (cp > eom)
    return (NULL);
  return (cp);
}

int dn_expand(const u_char *msg, const u_char *eom, const u_char *src,
	      char *dst, int dstsiz)
{
  int n = ns_name_uncompress(msg, eom, src, dst, (size_t)dstsiz);

  if (n > 0 && dst[0] == '.')
    dst[0] = '\0';
  return (n);
}

int ns_name_unpack(const u_char *msg, 
		   const u_char *eom, const u_char *src,
		   u_char *dst, size_t dstsiz)
{
  const u_char *srcp, *dstlim;
  u_char *dstp;
  int n, len, checked;

  len = -1;
  checked = 0;
  dstp = dst;
  srcp = src;
  dstlim = dst + dstsiz;
  if (srcp < msg || srcp >= eom) {
    errno = EMSGSIZE;
    return (-1);
  }
  /* Fetch next label in domain name. */
  while ((n = *srcp++) != 0) {
    /* Check for indirection. */
    switch (n & NS_CMPRSFLGS) {
    case 0:
      /* Limit checks. */
      if (dstp + n + 1 >= dstlim || srcp + n >= eom) {
	errno = EMSGSIZE;
	return (-1);
      }
      checked += n + 1;
      *dstp++ = n;
      memcpy(dstp, srcp, n);
      dstp += n;
      srcp += n;
      break;

    case NS_CMPRSFLGS:
      if (srcp >= eom) {
	errno = EMSGSIZE;
	return (-1);
      }
      if (len < 0)
	len = srcp - src + 1;
      srcp = msg + (((n & 0x3f) << 8) | (*srcp & 0xff));
      if (srcp < msg || srcp >= eom) {  /* Out of range. */
	errno = EMSGSIZE;
	return (-1);
      }
      checked += 2;
      /*
       * Check for loops in the compressed name;
       * if we've looked at the whole message,
       * there must be a loop.
       */
      if (checked >= eom - msg) {
	errno = EMSGSIZE;
	return (-1);
      }
      break;

    default:
      errno = EMSGSIZE;
      return (-1);			/* flag error */
    }
  }
  *dstp = '\0';
  if (len < 0)
    len = srcp - src;
  return (len);
}


int ns_name_uncompress(const u_char *msg, 
		       const u_char *eom, const u_char *src,
		       char *dst, size_t dstsiz)
{
  u_char tmp[NS_MAXCDNAME];
  int n;
	
  if ((n = ns_name_unpack(msg, eom, src, tmp, sizeof tmp)) == -1)
    return (-1);
  if (ns_name_ntop(tmp, dst, dstsiz) == -1)
    return (-1);
  return (n);
}

int ns_name_skip(const u_char **ptrptr, const u_char *eom) {
  const u_char *cp;
  u_int n;

  cp = *ptrptr;
  while (cp < eom && (n = *cp++) != 0) {
    /* Check for indirection. */
    switch (n & NS_CMPRSFLGS) {
    case 0:			/* normal case, n == len */
      cp += n;
      continue;
    case NS_CMPRSFLGS:	/* indirection */
      cp++;
      break;
    default:		/* illegal type */
      errno = EMSGSIZE;
      return (-1);
    }
    break;
  }
  if (cp > eom) {
    errno = EMSGSIZE;
    return (-1);
  }
  *ptrptr = cp;
  return (0);
}

int dn_skipname(const u_char *ptr, const u_char *eom) {
  const u_char *saveptr = ptr;

  if (ns_name_skip(&ptr, eom) == -1)
    return (-1);
  return (ptr - saveptr);
}


char* res_skip(char *msg, int numFieldsToSkip,
	       char *eom)
{
  register char *cp;
  register HEADER *hp;
  register int tmp;
  register int n;

  /*
   * Skip the header fields.
   */
  hp = (HEADER *)msg;
  cp = msg + HFIXEDSZ;

  /*
   * skip question records.
   */
  n = (int)ntohs(hp->qdcount);
  if (n > 0) {
    while (--n >= 0 && cp < eom) {
      tmp = dn_skipname((u_char *)cp, (u_char *)eom);
      if (tmp == -1) return(NULL);
      cp += tmp;
      cp += INT16SZ;	/* type 	*/
      cp += INT16SZ;	/* class 	*/
    }
  }
  if (--numFieldsToSkip <= 0) return(cp);

  /*
   * skip authoritative answer records
   */
  n = (int)ntohs(hp->ancount);
  if (n > 0) {
    while (--n >= 0 && cp < eom) {
      cp = res_skip_rr(cp, eom);
      if (cp == NULL) return(NULL);
    }
  }
  if (--numFieldsToSkip == 0) return(cp);

  /*
   * skip name server records
   */
  n = (int)ntohs(hp->nscount);
  if (n > 0) {
    while (--n >= 0 && cp < eom) {
      cp = res_skip_rr(cp, eom);
      if (cp == NULL) return(NULL);
    }
  }
  if (--numFieldsToSkip == 0) return(cp);

  /*
   * skip additional records
   */
  n = (int)ntohs(hp->arcount);
  if (n > 0) {
    while (--n >= 0 && cp < eom) {
      cp = res_skip_rr(cp, eom);
      if (cp == NULL) return(NULL);
    }
  }

  return(cp);
}

#define GetShort(cp)	ns_get16(cp); cp += INT16SZ;

void handleDNSpacket(const u_char *ipPtr, 
		     u_short displ, 
		     DNSHostInfo *hostPtr,
		     short n)
{
  querybuf answer;
  register u_char	*cp;
  char		**aliasPtr;
  u_char	*eom, *bp;
  char		**addrPtr;
  int		type, class, queryType = T_A;
  int		qdcount, ancount, arcount, nscount, buflen;
  int		origClass;
  int		numAliases = 0;
  int		numAddresses = 0;
  int		i;
  int		len;
  int		dlen;
  char		haveAnswer;
  char		printedAnswers = FALSE;
  char *host_aliases[MAXALIASES];
  int   host_aliases_len[MAXALIASES];
  u_char  hostbuf[MAXDNAME];
  char *addr_list[MAXADDRS + 1];

  /* Never forget to copy the buffer !!!!! */
  memcpy(&answer, ipPtr+displ, sizeof(answer));

  if (answer.qb1.rcode != NOERROR) {
    return;
  }

  eom = (u_char *)&answer+n;

  qdcount = (int)ntohs(answer.qb1.qdcount);
  ancount = (int)ntohs(answer.qb1.ancount);
  arcount = (int)ntohs(answer.qb1.arcount);
  nscount = (int)ntohs(answer.qb1.nscount);

  /*
   * If there are no answer, n.s. or additional records
   * then return with an error.
   */
  if (ancount == 0 && nscount == 0 && arcount == 0) {
    return;
  }


  bp	   = hostbuf;
  buflen = sizeof(hostbuf);
  cp	   = (u_char *) &answer+HFIXEDSZ;

  /* Skip over question section. */
  while (qdcount-- > 0) {
    n = dn_skipname(cp, eom);
    if (n < 0)
      return;
    cp += n + QFIXEDSZ;
    if (cp > eom)
      return;
  }

  aliasPtr	= host_aliases;
  addrPtr	= addr_list;
  haveAnswer	= FALSE;

  while (--ancount >= 0 && cp < eom) {
    n = dn_expand(answer.qb2, eom, cp, (char *)bp, buflen);
    if (n < 0)
      return;
    cp += n;
    if (cp + 3 * INT16SZ + INT32SZ > eom)
      return;
    type  = GetShort(cp);
    class = GetShort(cp);
    cp   += INT32SZ;	/* skip TTL */
    dlen  = GetShort(cp);
    if (cp + dlen > eom)
      return;
    if (type == T_CNAME) {
      /*
       * Found an alias.
       */
      cp += dlen;
      if (aliasPtr >= &host_aliases[MAXALIASES-1]) {
	continue;
      }
      *aliasPtr++ = (char *)bp;
      n = strlen((char *)bp) + 1;
      host_aliases_len[numAliases] = n;
      numAliases++;
      bp += n;
      buflen -= n;
      continue;
    } else if (type == T_PTR) {
      /*
       *  Found a "pointer" to the real name.
       */
      n = dn_expand(answer.qb2, eom, cp, (char *)bp, buflen);
      if (n < 0) {
	cp += n;
	continue;
      }
      cp += n;
      len = strlen((char *)bp) + 1;
      memcpy(hostPtr->name, bp, len);
      haveAnswer = TRUE;
      break;
    } else if (type != T_A) {
      cp += dlen;
      continue;
    }
    if (dlen != INADDRSZ)
      return;
    if (haveAnswer) {
      /*
       * If we've already got 1 address, we aren't interested
       * in addresses with a different length or class.
       */
      if (dlen != hostPtr->addrLen) {
	cp += dlen;
	continue;
      }
      if (class != origClass) {
	cp += dlen;
	continue;
      }
    } else {
      /*
       * First address: record its length and class so we
       * only save additonal ones with the same attributes.
       */
      hostPtr->addrLen = dlen;
      origClass = class;
      hostPtr->addrType = (class == C_IN) ? AF_INET : AF_UNSPEC;
      len = strlen((char *)bp) + 1;
      memcpy(hostPtr->name, bp, len);
    }
    bp += (((u_int32_t)bp) % sizeof(align));

    if (bp + dlen >= &hostbuf[sizeof(hostbuf)]) {
      break;
    }
    if (numAddresses >= MAXADDRS) {
      cp += dlen;
      continue;
    }
    memcpy(*addrPtr++ = (char *)bp, cp, dlen);
    bp += dlen;
    cp += dlen;
    numAddresses++;
    haveAnswer = TRUE;
  }
	

  if ((queryType == T_A || queryType == T_PTR) && haveAnswer) {

    /*
     *  Go through the alias and address lists and return them
     *  in the hostPtr variable.
     */

    if (numAliases > 0) {
      for (i = 0; i < numAliases; i++) {
	memcpy(hostPtr->aliases[i], host_aliases[i],
	       host_aliases_len[i]);
      }
      hostPtr->aliases[i][0] = '\0';
    }
    if (numAddresses > 0) {
      for (i = 0; i < numAddresses; i++) {
	memcpy(&hostPtr->addrList[i], addr_list[i], hostPtr->addrLen);
      }
      hostPtr->addrList[i] = 0;
    }
    return;
  }

  /*
   * At this point, for the T_A query type, only empty answers remain.
   * For other query types, additional information might be found
   * in the additional resource records part.
   */

  if (!answer.qb1.aa && (queryType != T_A) && (nscount > 0 || arcount > 0)) {
    if (printedAnswers) {
      putchar('\n');
    }
  }

  cp = (u_char *)res_skip((char *)&answer, 2, (char *)eom);

  while (--nscount >= 0 && cp < eom) {
    /*
     *  Go through the NS records and retrieve the names of hosts
     *  that serve the requested domain.
     */

    n = dn_expand(answer.qb2, eom, cp, (char *)bp, buflen);
    if (n < 0) {
      return;
    }
    cp += n;
    len = strlen((char *)bp) + 1;

    if (cp + 3 * INT16SZ + INT32SZ > eom)
      return;
    type  = GetShort(cp);
    class = GetShort(cp);
    cp   += INT32SZ;	/* skip TTL */
    dlen  = GetShort(cp);
    if (cp + dlen > eom)
      return;

    if (type != T_NS) {
      cp += dlen;
    }
  }
}


