!**************************************************************************
!*
!* Boot-ROM-Code to load an operating system across a TCP/IP network.
!*
!* Module:  rom.S
!* Purpose: Decompress the rom image
!* Entries: Called by BIOS at offset 0, getbyte, putbyte
!*
!**************************************************************************
!*
!* Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
!*
!*  This program is free software; you can redistribute it and/or modify
!*  it under the terms of the GNU General Public License as published by
!*  the Free Software Foundation; either version 2 of the License, or
!*  any later version.
!*
!*  This program is distributed in the hope that it will be useful,
!*  but WITHOUT ANY WARRANTY; without even the implied warranty of
!*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!*  GNU General Public License for more details.
!*
!*  You should have received a copy of the GNU General Public License
!*  along with this program; if not, write to the Free Software
!*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!*


!
!**************************************************************************
!
! Load assembler macros.
!
#include <macros.inc>
#include <memory.inc>
#include <version.h>


!
!**************************************************************************
!
! Equates for this module:
!
! maximum size of floppy image
FLOSIZE		equ	((FLOPEND - FLOPMEM) << 4) - 1

! maximum size of decompressed rom image
IMGSIZE		equ	((FLOPMEM - LOWMEM) << 4) - 1

! an uncompressed floppy image has to be able to go into the
! destination memory area
if FLOSIZE > IMGSIZE
	.FAIL	!Invalid memory specification - check memory.inc
endif

LOADADR		equ	FLOPEND		! segment where to load the boot code
BOOTLOAD	equ	$7C00		! boot load area in segment 0
BIOSSEG		equ	$0040		! BIOS data segment
BOOTSEG		equ	$FFFF		! BIOS segment to do cold boot


!
!**************************************************************************
!
! The physical rom starts here. At the beginning there is some
! miscellaneous data.
!
	.text

	entry	start			! define entry point for rom code

	extrn	_end_of_text
	extrn	inflate

	public	getbyte
	public	putbyte

	.word	$AA55			! rom signature
romlen:	.byte	0			! rom size (filled in by checksum prg)

start:	jmp	romstart		! the BIOS calls us here


! The following variables contain the size of the bootrom kernel image and
! the program version number.

	.align	DATAOFS
romsiz:	.word	0			! rom size
	.byte	MAJOR_VER		! major version number
	.byte	MINOR_VER		! minor version number
	.word	0			! serial number
botvec:	.word	0			! offset to bootrom interrupt vector
	.asciz	COPYRIGHT		! copyright message


!
!**************************************************************************
!
! Now start with the actual boot rom code. The startup code just installs
! a new bootrom interrupt vector, and then returns to the BIOS. This is
! necessary so that the BIOS gets a chance to also initialize other installed
! roms. The bootrom vector is used for the ROM BASIC in original IBM PCs
! and gets called by the BIOS whenever there is no bootable disk in any
! of the drives. Note that the floppy driver depends on the value returned
! by this routine in SI.
!
romstart:

	cld
	cli				! do not allow interrupts when changing
	push	ds			! the interrupt vector table
	xor	ax,ax
	mov	ds,ax
	seg	cs
	mov	si,botvec		! save old interrupt vector
	or	si,si
	jnz	romst1
	mov	si,#BOOTROM * 4		! load default for floppy loader
romst1:
#ifdef IS386
	mov	eax,dword ptr [si]
	mov	dword ptr [OLDBOOTROM * 4],eax
#else
	mov	ax,word ptr [si + 0]
	mov	word ptr [OLDBOOTROM * 4 + 0],ax
	mov	ax,word ptr [si + 2]
	mov	word ptr [OLDBOOTROM * 4 + 2],ax
#endif
	mov	word ptr [si + 0],#int18
	mov	word ptr [si + 2],cs
	pop	ds
	sti				! thats it
	retf


!
!**************************************************************************
!
! The bootrom interrupt routine decompresses the rom image and starts it.
!
int18:	sti
	seg	cs
	mov	ax,botvec
	cmp	ax,#BOOTSTRAP * 4	! check if we are simulating the
	jne	dorom			! bootstrap vector

! We have been started as the bootstrap vector. Therefore we first check
! to see if we have a floppy in drive A, and boot from it if we have.

	xor	dx,dx
	xor	ax,ax
	int	$13			! reset the disk system
	mov	di,#5			! loop counter
	mov	es,dx
floop:	mov	ax,#$0201
	mov	bx,#BOOTLOAD
	mov	cx,#$0001		! try to load first sector of first
	int	$13			! track on floppy drive A
	jnc	doflop			! if error, we dont have a floppy
	dec	di
	jnz	floop			! try up to 5 times
	jmp	dorom			! if not successful, we have no floppy
doflop:	jmpi	BOOTLOAD,0		! call floppy boot sector

! Start the bootrom by first setting up the stack and copying the whole
! loader into RAM (if its not there already)

dorom:	cld
	cli
	mov	bx,#LOADADR			! set temporary stack pointer
	mov	ss,bx
	mov	sp,#$FFFE
	sti

	mov	cx,#_end_of_text + $000F	! determine size to copy
	and	cx,#$FFF0			! adjust to paragraph boundary
	shr	cx,#1
	mov	ax,cs
	mov	ds,ax
	mov	es,bx				! move boot loader into lower
	xor	si,si				! ram
	xor	di,di
	rep
	movsw
	mov	ds,bx				! set the new data segment
	jmpi	#nocopy,LOADADR			! jump to new copy

! We are now in RAM and can save the values necessary to access the compressed
! image. Also compute the last usable segment and check that we have enough
! memory. Then actually uncompress the image.

nocopy:	mov	word ptr [inptr+0],si		! set pointer to compressed img
	mov	word ptr [inptr+2],ax
	int	$12				! get base kB using BIOS
	shift	(shl,ax,6)			! convert it into segment number
	mov	si,#memmsg
	cmp	ax,#HIGHMEM			! check if its enough for us
	jb	hderr

	mov	si,#inimsg
	call	prnstr
	call	inflate				! decompress the block
	or	al,al				! check for error
	jz	doimg
	mov	si,#infmsg
hderr:	jmp	error

! Finally print the end message, restore the bootrom vector and start
! the new image

doimg:	mov	si,#endmsg
	call	prnstr
	cli
	push	ds
	xor	ax,ax
	mov	ds,ax
	seg	cs
	mov	si,botvec		! restore old interrupt vector
	or	si,si
	jnz	doimg2
	mov	si,#BOOTROM * 4		! load default for floppy loader
doimg2:
#ifdef IS386
	mov	eax,dword ptr [OLDBOOTROM * 4]
	mov	dword ptr [si],eax
#else
	mov	ax,word ptr [OLDBOOTROM * 4 + 0]
	mov	word ptr [si + 0],ax
	mov	ax,word ptr [OLDBOOTROM * 4 + 2]
	mov	word ptr [si + 2],ax
#endif
	pop	ds
	sti
	jmpi	0,LOWMEM		! call decompressed image


! Messages. Dont change these because romcheck uses them to identify
! the netboot bootrom.

inimsg:	.asciz	"Uncompressing... "
endmsg:	.ascii	"done"
	.byte	$0D,$0A,0

infmsg:	.asciz	"format"
memmsg:	.asciz	"memory"


!
!**************************************************************************
!
! Return the next byte from the compressed image.
! Input:  none
! Output: AL  -  next byte
!         Zero flag set if end of file, AL is zero in that case
! Changed registers: AX
!
getbyte:

	mov	ax,romsiz
	or	ax,ax			! check if at end of file
	jz	getb9
	dec	romsiz
	push	si
	push	ds
	lds	si,inptr
	lodsb
	pop	ds
	pop	si
	inc	word ptr [inptr+0]	! this will clear the zero flag
getb9:	ret


!
!**************************************************************************
!
! Write a byte to the decompressed image.
! Input:  AL  -  output byte
! Output: none
! Registers changed: none
!
putbyte:

	push	di
	push	es
	mov	di,#LOWMEM
	mov	es,di
	mov	di,outptr		! simply write the byte into the
	stosb				! output buffer area
	mov	outptr,di
	pop	es
	pop	di
	ret


!
!**************************************************************************
!
! Print an error message onto the console screen
! Input:  DS:SI  -  Pointer to message
! Output: none
! Registers changed: routine does not return
!
error:	push	si
	mov	si,#er1msg
	call	prnstr
	pop	si
	call	prnstr
	mov	si,#er2msg
	call	prnstr
	xor	ax,ax			! wait until user presses a key
	int	$16
	mov	ax,#BIOSSEG
	mov	ds,ax
	mov	word ptr [$0072],#$1234	! indicates warm boot
	jmpi	0,BOOTSEG		! reboot the system


! Messages:

er1msg:	.byte	$0D,$0A
	.asciz	"Error: "
er2msg:	.byte	$0D,$0A
	.asciz	"Press a key to reboot"


!
!**************************************************************************
!
! Print a string onto the console screen
! Input:  DS:SI  -  Pointer to string
! Output: none
! Registers changed: AX, BX, SI
!
prnstr:	cld
prnst1:	lodsb				! get next character
	or	al,al
	jz	prnst2
	mov	ah,#$0E			! BIOS command for character output
	mov	bx,#$0007		! screen page and colors
	int	$10			! send character to screen
	jmp	prnst1
prnst2:	ret


!
!**************************************************************************
!
! Define the data segment
!

outptr:		.word	0	! ptr into decompressed memory (rel. to LOWMEM)
inptr:		.long	0	! ptr into compressed memory (rel. to CS)


!
!**************************************************************************
!
	end
