/*
 *  ncplib.c
 *
 *  Copyright (C) 1995, 1996 by Volker Lendecke
 *
 */

#if 0
#define ncp_dprintf(X...)	printf(X)
#else
#define ncp_dprintf(X...)
#endif
#include "config.h"
#include "ncplib_i.h"
/* due to NWVerifyObjectPassword... */
#include <ncp/nwcalls.h>
#ifdef SIGNATURES
#include "ncpsign.h"
#endif
#ifdef NDS_SUPPORT
extern int bindery_only;
#include <ncp/ndslib.h>
#endif
#include <ncp/nwnet.h>

#include <sys/ioctl.h>
#include <sys/wait.h>
/* extern pid_t wait(int *); */
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <ncp/ext/socket.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>
#include <ncp/kernel/route.h>
#include <sys/param.h>
#include <stdlib.h>
#ifdef NCP_KERNEL_NCPFS_AVAILABLE
#include <mntent.h>
#endif
#include <pwd.h>
#include <sys/stat.h>
#include <stdarg.h>
#ifdef CONFIG_NATIVE_IP
#include <netdb.h>
#endif

#include <libintl.h>
#define _(X) dgettext(PACKAGE, (X))
#define N_(X) (X)

#ifndef ENOPKG
#define ENOPKG	ENOSYS
#endif

#define NCP_DEFAULT_BUFSIZE	1024
#define NCP_MAX_BUFSIZE		1024
#ifdef SIGNATURES
#define NCP_DEFAULT_OPTIONS	2
int in_options = NCP_DEFAULT_OPTIONS;
#endif

/* return number of bytes in packet */
static inline int
ncp_packet_size(struct ncp_conn *_conn)
{
	return _conn->current_point - _conn->packet;
}

static long
 ncp_negotiate_buffersize(struct ncp_conn *conn,
			  size_t size, size_t *target);

#ifdef SIGNATURES
static long
 ncp_negotiate_size_and_options(struct ncp_conn *conn,
			  size_t size, int options, 
			  size_t *ret_size, int *ret_options);
#endif

static long
 ncp_login_object(struct ncp_conn *conn,
		  const unsigned char *username,
		  int login_type,
		  const unsigned char *password);

static long
ncp_do_close(struct ncp_conn *conn);

#ifdef CONFIG_NATIVE_IPX
static long
ncp_find_server_ipx(const char **server_name, int type, struct sockaddr_ipx* addr);
#endif

void
str_upper(char *name)
{
	while (*name)
	{
		*name = toupper(*name);
		name = name + 1;
	}
}

#ifdef NCP_TRACE_ENABLE

static FILE *logfile = 0;

inline FILE *
get_logfile()
{
	if (!logfile)
		logfile = fopen("/var/log/ncplib.trc", "aw");

	return logfile;
}

void
__ncp_trace(char *module, int line, char *p,...)
{
	FILE *log = get_logfile();
	va_list ap;

	if (!log)
		return;

	fprintf(log, "%s(%d)->", module, line);

	va_start(ap, p);
	vfprintf(log, p, ap);
	va_end(ap);
	fprintf(log, "\n");
	fflush(log);
}

void
__dump_hex(const char *_msg, const unsigned char *_buf, int _len)
{
	static char sym[] = "0123456789ABCDEF";
	FILE *log = get_logfile();
	if (!log)
		return;

	fprintf(log, "len = %d:msg->%s", _len, _msg);
	fflush(log);
	for (; _len > 0; _len--, _buf++)
	{
		putc(sym[(*_buf) >> 4], log);
		putc(sym[(*_buf) & 0x0F], log);
	}
	putc('\n', log);
	fflush(log);
}
#endif /*def NCP_TRACE_ENABLE */

#if 0

static int debug_level = 5;
static FILE *logfile = stderr;

static void
dprintf(int level, char *p,...)
{
	va_list ap;

	if (level > debug_level)
	{
		return;
	}
	va_start(ap, p);
	vfprintf(logfile, p, ap);
	va_end(ap);
	fprintf(logfile, "\n");
	fflush(logfile);
}

#endif

/* I know it's terrible to include a .c file here, but I want to keep
   the file nwcrypt.c intact and separate for copyright reasons */
#include "nwcrypt.c"

void
ipx_fprint_node(FILE * file, IPXNode node)
{
	fprintf(file, "%02X%02X%02X%02X%02X%02X",
		(unsigned char) node[0],
		(unsigned char) node[1],
		(unsigned char) node[2],
		(unsigned char) node[3],
		(unsigned char) node[4],
		(unsigned char) node[5]
	    );
}

void
ipx_fprint_network(FILE * file, IPXNet net)
{
	fprintf(file, "%08X", (u_int32_t)ntohl(net));
}

void
ipx_fprint_port(FILE * file, IPXPort port)
{
	fprintf(file, "%04X", ntohs(port));
}

void
ipx_print_node(IPXNode node)
{
	ipx_fprint_node(stdout, node);
}

void
ipx_print_network(IPXNet net)
{
	ipx_fprint_network(stdout, net);
}

void
ipx_print_port(IPXPort port)
{
	ipx_fprint_port(stdout, port);
}

int
ipx_node_equal(CIPXNode n1, CIPXNode n2)
{
	return memcmp(n1, n2, IPX_NODE_LEN) == 0;
}

int
ipx_sscanf_node(char *buf, unsigned char node[6])
{
	int i;
	int n[6];

	if ((i = sscanf(buf, "%2x%2x%2x%2x%2x%2x",
			&(n[0]), &(n[1]), &(n[2]),
			&(n[3]), &(n[4]), &(n[5]))) != 6)
	{
		return i;
	}
	for (i = 0; i < 6; i++)
	{
		node[i] = n[i];
	}
	return 6;
}

#ifdef CONFIG_NATIVE_IPX
void
ipx_fprint_saddr(FILE * file, struct sockaddr_ipx *sipx)
{
	ipx_fprint_network(file, sipx->sipx_network);
	fprintf(file, ":");
	ipx_fprint_node(file, sipx->sipx_node);
	fprintf(file, ":");
	ipx_fprint_port(file, sipx->sipx_port);
}

void
ipx_print_saddr(struct sockaddr_ipx *sipx)
{
	ipx_fprint_saddr(stdout, sipx);
}

int
ipx_sscanf_saddr(char *buf, struct sockaddr_ipx *target)
{
	char *p;
	struct sockaddr_ipx addr;
	unsigned long sipx_network;

	addr.sipx_family = AF_IPX;
	addr.sipx_type = NCP_PTYPE;

	if (sscanf(buf, "%lx", &sipx_network) != 1)
	{
		return 1;
	}
	addr.sipx_network = htonl(sipx_network);
	if ((p = strchr(buf, ':')) == NULL)
	{
		return 1;
	}
	p += 1;
	if (ipx_sscanf_node(p, addr.sipx_node) != 6)
	{
		return 1;
	}
	if ((p = strchr(p, ':')) == NULL)
	{
		return 1;
	}
	p += 1;
	if (sscanf(p, "%hx", &addr.sipx_port) != 1)
	{
		return 1;
	}
	addr.sipx_port = htons(addr.sipx_port);
	*target = addr;
	return 0;
}
#endif	/* CONFIG_NATIVE_IPX */

static int
x_recvfrom(int sock, void *buf, int len, unsigned int flags,
	   struct sockaddr *sender, socklen_t* addrlen, int timeout,
	   long *err)
{
	fd_set rd, wr, ex;
	struct timeval tv;
	int result;

	FD_ZERO(&rd);
	FD_ZERO(&wr);
	FD_ZERO(&ex);
	FD_SET(sock, &rd);

	tv.tv_sec = timeout;
	tv.tv_usec = 0;

	if ((result = select(sock + 1, &rd, &wr, &ex, &tv)) == -1)
	{
		*err = errno;
		return -1;
	}
	if (FD_ISSET(sock, &rd))
	{
		result = sender?recvfrom(sock, buf, len, flags,
				  sender, addrlen):
				recv(sock, buf, len, flags);
	} else
	{
		result = -1;
		errno = ETIMEDOUT;
	}
	if (result < 0)
	{
		*err = errno;
	}
	return result;
}

static int
x_recv(int sock, void *buf, int len, unsigned int flags, int timeout,
	 long *err)
{
	return x_recvfrom(sock, buf, len, flags, NULL, 0,
			    timeout, err);
}

#ifdef CONFIG_NATIVE_IPX
#ifdef __MAKE_SULIB__
static long
ipx_sap_find_nearest(int server_type, struct sockaddr_ipx *result,
		     char server_name[NCP_BINDERY_NAME_LEN])
{
	struct sockaddr_ipx addr;
	char data[1024];
	int sock;
	int opt;
	int packets;
	int len;

	struct sap_server_ident *ident;

	if ((sock = socket(PF_IPX, SOCK_DGRAM, IPXPROTO_IPX)) < 0)
	{
		return errno;
	}
	opt = 1;
	/* Permit broadcast output */
	if (setsockopt(sock, SOL_SOCKET, SO_BROADCAST, &opt, sizeof(opt)) == -1)
	{
		goto finished;
	}
	memset(&addr, 0, sizeof(addr));
	addr.sipx_family = AF_IPX;
	addr.sipx_network = htonl(0x0);
	addr.sipx_port = htons(0x0);
	addr.sipx_type = IPX_SAP_PTYPE;

	if (bind(sock, (struct sockaddr *) &addr, sizeof(addr)) == -1)
	{
		goto finished;
	}
	WSET_HL(data, 0, IPX_SAP_NEAREST_QUERY);
	WSET_HL(data, 2, server_type);

	memset(&addr, 0, sizeof(addr));
	addr.sipx_family = AF_IPX;
	addr.sipx_port = htons(IPX_SAP_PORT);
	addr.sipx_type = IPX_SAP_PTYPE;
	addr.sipx_network = htonl(0x0);
	ipx_assign_node(addr.sipx_node, IPX_BROADCAST_NODE);

	if (sendto(sock, data, 4, 0,
		   (struct sockaddr *) &addr, sizeof(addr)) < 0)
	{
		goto finished;
	}
	packets = 5;
	do
	{
		long err;
		len = x_recv(sock, data, 1024, 0, 1, &err);
		if (len < 66)
		{
			packets = packets - 1;
			continue;
		}
	}
	while ((ntohs(*((u_int16_t *) data)) != IPX_SAP_NEAREST_RESPONSE)
	       && (packets > 0));

	if (packets == 0)
	{
		close(sock);
		return NWE_SERVER_NOT_FOUND;
	}
	ident = (struct sap_server_ident *) (data + 2);

	result->sipx_family = AF_IPX;
	result->sipx_network = ident->server_network;
	result->sipx_port = ident->server_port;
	result->sipx_type = NCP_PTYPE;
	ipx_assign_node(result->sipx_node, ident->server_node);

	memcpy(server_name, ident->server_name, sizeof(ident->server_name));

	errno = 0;

      finished:
	close(sock);
	return errno;
}
#endif	/* __MAKE_SULIB__ */

#ifndef __MAKE_SULIB__
static int
exec_nwsfind(char* reply, size_t replyLen, char* request[]) {
	int fildes[2];
	int err;
	
	err = pipe(fildes);
	if (err)
		return errno;
	signal(SIGCHLD, SIG_DFL);
	err = fork();
	if (err < 0)
		return errno;
	if (err == 0) {
		close(0);
		close(1);
		close(2);
		close(fildes[0]);
		open("/dev/null", O_RDONLY); /* fd 0 */
		dup2(fildes[1], 1);
		dup2(0, 2);
		request[0] = NWSFIND;
		execv(request[0], request);
		/* OK, something bad happen... Drop it to /dev/null */
		exit(127);
	} else {
		int rd;
		
		close(fildes[1]);
		replyLen--;
		do {
			rd = read(fildes[0], reply, 1);
			if ((rd == -1) && (errno == EINTR)) continue;
			if (rd <= 0) break;
			if ((*reply == '\n') || (*reply == '\r')) 
				*reply = 0;
			reply += 1;
			replyLen -= 1;
		} while (replyLen > 0);
		close(fildes[0]);
		waitpid(err, &rd, WNOHANG);
		*reply = 0;
	}
	return 0;
}
#endif /* !__MAKE_SULIB__ */

static int
ipx_make_reachable(const struct sockaddr_ipx* target)
{
#ifdef __MAKE_SULIB__
	IPXNet	network = target->sipx_network;
	struct rtentry rt_def;
	/* Router */
	struct sockaddr_ipx *sr = (struct sockaddr_ipx *) &rt_def.rt_gateway;
	/* Target */
	struct sockaddr_ipx *st = (struct sockaddr_ipx *) &rt_def.rt_dst;

	struct ipx_rip_packet rip;
	struct sockaddr_ipx addr;
	socklen_t addrlen;
	int sock;
	int opt;
	int res = -1;
	int i;
	int packets;

	if (geteuid() != 0)
	{
		errno = EPERM;
		return -1;
	}
	memset(&rip, 0, sizeof(rip));

	sock = socket(PF_IPX, SOCK_DGRAM, IPXPROTO_IPX);

	if (sock == -1)
	{
		return errno;
	}
	opt = 1;
	/* Permit broadcast output */
	if (setsockopt(sock, SOL_SOCKET, SO_BROADCAST, &opt, sizeof(opt)) != 0)
	{
		goto finished;
	}
	memset(&addr, 0, sizeof(addr));
	addr.sipx_family = AF_IPX;
	addr.sipx_network = htonl(0x0);
	addr.sipx_port = htons(0x0);
	addr.sipx_type = IPX_RIP_PTYPE;

	if (bind(sock, (struct sockaddr *) &addr, sizeof(addr)) != 0)
	{
		goto finished;
	}
	addr.sipx_family = AF_IPX;
	addr.sipx_port = htons(IPX_RIP_PORT);
	addr.sipx_type = IPX_RIP_PTYPE;
	addr.sipx_network = htonl(0x0);
	ipx_assign_node(addr.sipx_node, IPX_BROADCAST_NODE);

	rip.operation = htons(IPX_RIP_REQUEST);
	rip.rt[0].network = network;

	if (sendto(sock, &rip, sizeof(rip), 0,
		   (struct sockaddr *) &addr, sizeof(addr)) < 0)
	{
		goto finished;
	}
	packets = 3;
	do
	{
		long err;
		int len;

		if (packets == 0)
		{
			goto finished;
		}
		addrlen = sizeof(struct sockaddr_ipx);

		len = x_recvfrom(sock, &rip, sizeof(rip), 0, (struct sockaddr*)sr, &addrlen, 1,
				   &err);

		if (len < sizeof(rip))
		{
			packets = packets - 1;
			continue;
		}
	}
	while (ntohs(rip.operation) != IPX_RIP_RESPONSE);

	if (rip.rt[0].network != network)
	{
		goto finished;
	}
	rt_def.rt_flags = RTF_GATEWAY;
	st->sipx_network = network;
	st->sipx_family = AF_IPX;
	sr->sipx_family = AF_IPX;

	i = 0;
	do
	{
		res = ioctl(sock, SIOCADDRT, &rt_def);
		i++;
	}
	while ((i < 5) && (res < 0) && (errno == EAGAIN));

      finished:
	close(sock);

	if (res != 0)
	{
		errno = ENETUNREACH;
	}
	return res;
#else
	int err;
	char buf[40];
	char *buf2[4];
	
	buf2[1] = "-a";
	buf2[2] = buf;
	buf2[3] = NULL;
    
	sprintf(buf, "%08x:%02x%02x%02x%02x%02x%02x:%04x",
		(u_int32_t)ntohl(target->sipx_network),
		((const unsigned char*)target->sipx_node)[0],
		((const unsigned char*)target->sipx_node)[1],
		((const unsigned char*)target->sipx_node)[2],
		((const unsigned char*)target->sipx_node)[3],
		((const unsigned char*)target->sipx_node)[4],
		((const unsigned char*)target->sipx_node)[5],
		ntohs(target->sipx_port));

	err = exec_nwsfind(buf, sizeof(buf), buf2);
	if (err)
		return err;
	return 0;
#endif
}

void
ipx_assign_node(IPXNode dest, CIPXNode src)
{
	memcpy(dest, src, IPX_NODE_LEN);
}

static int
install_wdog(struct ncp_conn *conn)
{
	int parent_pid = getpid();
	int pid;
	int sock = conn->wdog_sock;

	char buf[1024];
	struct sockaddr_ipx sender;
	int sizeofaddr = sizeof(struct sockaddr_ipx);
	int pktsize;


	if ((pid = fork()) < 0)
	{
		return -1;
	}
	if (pid != 0)
	{
		/* Parent, should go on as usual */
		conn->wdog_pid = pid;
		return 0;
	}
	while (1)
	{
		long err;
		/* every 120 seconds we look if our parent is
		   still alive */
		pktsize = x_recvfrom(sock, buf, sizeof(buf), 0,
				     (struct sockaddr*)&sender, &sizeofaddr, 120, &err);

		if (getppid() != parent_pid)
		{
			/* our parent has died, so nothing to do
			   anymore */
			exit(0);
		}
		if ((pktsize != 2)
		    || (buf[1] != '?'))
		{
			continue;
		}
		buf[1] = 'Y';
		pktsize = sendto(sock, buf, 2, 0,
				 (struct sockaddr *) &sender,
				 sizeof(sender));
	}
}
#endif	/* CONFIG_NATIVE_IPX */

static void
 assert_conn_locked(struct ncp_conn *conn);

static void
assert_conn_not_locked(struct ncp_conn *conn)
{
	if (conn->lock != 0)
	{
		ncp_printf(_("ncpfs: connection already locked!\n"));
	}
}

static void
ncp_lock_conn(struct ncp_conn *conn)
{
	ncpt_mutex_lock(&conn->buffer_mutex);
	assert_conn_not_locked(conn);
	conn->lock = 1;
}

void
ncp_unlock_conn(struct ncp_conn *conn)
{
	assert_conn_locked(conn);
	conn->lock = 0;
	ncpt_mutex_unlock(&conn->buffer_mutex);
}

static struct ncp_conn *
ncp_alloc_conn(void) {
	struct ncp_conn* conn;
	
	conn = (struct ncp_conn*)malloc(sizeof(*conn));
	if (conn) {
		memset(conn, 0, sizeof(*conn));
		ncpt_atomic_set(&conn->use_count, 1);
		ncpt_atomic_set(&conn->store_count, 0);
		INIT_LIST_HEAD(&conn->nds_ring);
		ncpt_mutex_init(&conn->buffer_mutex);
		ncpt_mutex_init(&conn->serverInfo.mutex);
		conn->serverInfo.valid = 0;
	}
	return conn;
}

static long
do_ncp_call(struct ncp_conn *conn, int request_size)
{
	struct ncp_request_header request;

	int result;
	int retries = 20;
	int len;
	long err;

	memcpy(&request, conn->packet, sizeof(request));
#ifdef SIGNATURES
	if (conn->sign_active)
	{
		sign_packet(conn, &request_size);
	}
#endif	
	while (retries > 0)
	{
		struct ncp_reply_header reply;

		retries -= 1;

		result = send(conn->ncp_sock, conn->packet,
			      request_size, 0);

		if (result < 0)
		{
			return errno;
		}
	      re_select:
		len = x_recv(conn->ncp_sock,
			     (char *) &reply, sizeof(reply),
			     MSG_PEEK, 3, &err);

		if ((len < 0) && (err == ETIMEDOUT))
		{
			continue;
		}
		if (len < 0)
		{
			return err;
		}
		if (
		/* Did the sender send a positive acknowledge? */
			      ((len == sizeof(reply))
			       && (reply.type == NCP_POSITIVE_ACK))

		/* Did we get a bogus answer? */
			   || ((len < sizeof(reply))
			       || (reply.type != NCP_REPLY)
			       || ((request.type != NCP_ALLOC_SLOT_REQUEST)
				   && ((reply.sequence != request.sequence)
				       || (reply.conn_low != request.conn_low)
				 || (reply.conn_high != request.conn_high)))))
		{
			/* Then throw away the packet */
			x_recv(conn->ncp_sock, (char *) &reply, sizeof(reply),
				 0, 1, &err);
			goto re_select;
		}
		len = x_recv(conn->ncp_sock, conn->packet, NCP_PACKET_SIZE,
			 0, 1, &err);
		conn->reply_size = len;
		return 0;
	}
	return ETIMEDOUT;
}

static inline void
fill_size(struct ncp_conn *_conn)
{
	if (_conn->has_subfunction != 0)
	{
		register word sz = ncp_packet_size(_conn) - (2 + sizeof(struct ncp_request_header));
		WSET_HL(_conn->packet, 7, sz);
	}
}

static int
ncp_mount_request(struct ncp_conn *conn, int function)
{
#ifdef NCP_KERNEL_NCPFS_AVAILABLE
	struct ncp_ioctl_request request;
	int result;

	assert_conn_locked(conn);

	fill_size(conn);
	request.function = function;
	request.size = ncp_packet_size(conn);
	request.data = conn->packet;

	if ((result = ioctl(conn->mount_fid, NCP_IOC_NCPREQUEST, &request)) < 0)
	{
		return errno;
	}
	conn->ncp_reply_size = result - sizeof(struct ncp_reply_header);
	
	result = BVAL(conn->packet, 6);
	conn->conn_status = BVAL(conn->packet, 7);

	if (result && (conn->verbose != 0))
	{
		ncp_printf(_("ncp_request_error: %d\n"), result);
	}
	return (result == 0) ? 0 : (NWE_SERVER_ERROR | result);
#else
	return ENOPKG;
#endif	/* NCP_KERNEL_NCPFS_AVAILABLE */
}

static long
ncp_temp_request(struct ncp_conn *conn, int function)
{
	int err;

	assert_conn_locked(conn);

	conn->sequence += 1;

	WSET_LH(conn->packet, 0, NCP_REQUEST);
	BSET(conn->packet, 2, conn->sequence);
	BSET(conn->packet, 3, (conn->i.connection) & 0xff);
	BSET(conn->packet, 5, (conn->i.connection) >> 8);
	BSET(conn->packet, 4, 1);
	BSET(conn->packet, 6, function);

	fill_size(conn);
	if ((err = do_ncp_call(conn, ncp_packet_size(conn))) != 0)
	{
		return err;
	}
	err = BVAL(conn->packet, 6);
	conn->conn_status = BVAL(conn->packet, 7);
	conn->ncp_reply_size =
	    conn->reply_size - sizeof(struct ncp_reply_header);

	if (err && (conn->verbose != 0))
	{
		ncp_printf(_("ncp_request_error: %d\n"), err);
	}
	return (err == 0) ? 0 : (NWE_SERVER_ERROR | err);
}

long
ncp_renegotiate_connparam(struct ncp_conn *conn, int buffsize, int in_options)
{
	size_t neg_buffsize;
	int err;
#ifdef SIGNATURES
	int options;
	
	/* TODO: Signatures active? I am not sure whether we can unnegotiate them */
	if (conn->sign_active) in_options |= 2;

	if ((err = ncp_negotiate_size_and_options(conn, buffsize, in_options, 
	 &neg_buffsize, &options)) == 0)
	{
		if ((options & 2) != (in_options & 2)) 
		{
			err = ncp_negotiate_size_and_options(conn,
			 buffsize, options & 2, &neg_buffsize, &options);
		}
	}
	else
#endif
	{
#ifdef SIGNATURES
		options = 0;
#endif
		err = ncp_negotiate_buffersize(conn, NCP_DEFAULT_BUFSIZE,
			      &neg_buffsize);
	}			  
	if (err) {
		return err;
	}
	if ((neg_buffsize < 512) || (neg_buffsize > NCP_MAX_BUFSIZE))
		return -1;
	conn->i.buffer_size = neg_buffsize;
#ifdef SIGNATURES	
	conn->sign_wanted = (options & 2) ? 1:0;
	if (conn->is_connected == CONN_PERMANENT) {
#ifdef NCP_KERNEL_NCPFS_AVAILABLE	
		int cursign;
		
		if (ioctl(conn->mount_fid, NCP_IOC_SIGN_WANTED, &cursign)) {
		/* ncpfs does not support SIGN_WANTED -> current sign level = 0 */
			cursign = 0;
		}
		if (cursign) cursign = 1;
		if (cursign != conn->sign_wanted) {
			int newsign = conn->sign_wanted ? -1 : 0;
			
			err = ioctl(conn->mount_fid, NCP_IOC_SET_SIGN_WANTED, &newsign);
		}
#endif	/* NCP_KERNEL_NCPFS_AVAILABLE */		
	}
#endif
	return 0;
}

#ifdef CONFIG_NATIVE_IPX
static long
ncp_connect_ipx_addr(struct ncp_conn *conn, const struct sockaddr_ipx *target,
		 int wdog_needed)
{
	struct sockaddr_ipx addr;
	socklen_t addrlen;

	int ncp_sock, wdog_sock;
	long err;

	conn->is_connected = NOT_CONNECTED;
	conn->verbose = 0;

	if ((ncp_sock = socket(PF_IPX, SOCK_DGRAM, IPXPROTO_IPX)) == -1)
	{
		return errno;
	}
	if ((wdog_sock = socket(PF_IPX, SOCK_DGRAM, IPXPROTO_IPX)) == -1)
	{
		close(ncp_sock);
		return errno;
	}
	addr.sipx_family = AF_IPX;
	addr.sipx_port = htons(0x0);
	addr.sipx_type = NCP_PTYPE;
	addr.sipx_network = IPX_THIS_NET;
	ipx_assign_node(addr.sipx_node, IPX_THIS_NODE);

	addrlen = sizeof(addr);

	if ((bind(ncp_sock, (struct sockaddr *) &addr, sizeof(addr)) == -1)
	|| (getsockname(ncp_sock, (struct sockaddr *) &addr, &addrlen) == -1))
	{
		int saved_errno = errno;
		close(ncp_sock);
		close(wdog_sock);
		return saved_errno;
	}
	addr.sipx_port = htons(ntohs(addr.sipx_port) + 1);

	if (bind(wdog_sock, (struct sockaddr *) &addr, sizeof(addr)) == -1)
	{
		int saved_errno = errno;
		close(ncp_sock);
		close(wdog_sock);
		return saved_errno;
	}
	conn->ncp_sock = ncp_sock;
	conn->wdog_sock = wdog_sock;

	conn->sequence = 0;
	conn->i.addr.ipx = *target;

	if (connect(ncp_sock, (const struct sockaddr*)target, sizeof(*target)) == -1)
	{
		int saved_errno = errno;

		if ((saved_errno != ENETUNREACH) || ipx_make_reachable(target) 
		  || connect(ncp_sock, (const struct sockaddr*)target, sizeof(*target))) {;
			close(ncp_sock);
			close(wdog_sock);
			return saved_errno;
		}
	}

	WSET_LH(conn->packet, 0, NCP_ALLOC_SLOT_REQUEST);
	BSET(conn->packet, 2, conn->sequence);
	BSET(conn->packet, 3, 0xff);
	BSET(conn->packet, 4, 1);
	BSET(conn->packet, 5, 0xff);
	BSET(conn->packet, 6, 0);

	if ((err = do_ncp_call(conn, sizeof(struct ncp_request_header))) != 0)
	{
		if ((err != ENETUNREACH)
		    || (ipx_make_reachable(target) != 0)
		    || ((err =
			 do_ncp_call(conn,
				     sizeof(struct ncp_request_header))) != 0))
		{
			close(ncp_sock);
			close(wdog_sock);
			return err;
		}
	}
	if (wdog_needed != 0)
	{
		install_wdog(conn);
	} else
	{
		conn->wdog_pid = 0;
	}

	conn->sequence = 0;
	conn->i.connection =
	    BVAL(conn->packet, 3) + (BVAL(conn->packet, 5) << 8);
	conn->is_connected = CONN_TEMPORARY;

#ifdef SIGNATURES
	conn->sign_active = 0;
	conn->sign_wanted = 0;
	err = ncp_renegotiate_connparam(conn, NCP_DEFAULT_BUFSIZE, in_options);
#else
	err = ncp_renegotiate_connparam(conn, NCP_DEFAULT_BUFSIZE, 0);
#endif

	if (err != 0)   
	{
		return -1;
	}
	return 0;
}
#endif	/* CONFIG_NATIVE_IPX */

#ifdef CONFIG_NATIVE_IP
static long
ncp_connect_in_addr(struct ncp_conn *conn, const struct sockaddr_in *target,
		 int wdog_needed)
{
	struct sockaddr_in addr;
	socklen_t addrlen;

	int ncp_sock;
	long err;

	conn->is_connected = NOT_CONNECTED;
	conn->verbose = 0;

	if ((ncp_sock = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP)) == -1)
	{
		return errno;
	}
	addr.sin_family = AF_INET;
	addr.sin_port = htons(0x0);
	addr.sin_addr.s_addr = INADDR_ANY;

	addrlen = sizeof(addr);

	if (bind(ncp_sock, (struct sockaddr *) &addr, sizeof(addr)) == -1)
	{
		int saved_errno = errno;
		close(ncp_sock);
		return saved_errno;
	}

	conn->ncp_sock = ncp_sock;
	conn->wdog_sock = -1;

	conn->sequence = 0;
	memcpy(&conn->i.addr, target, sizeof(*target));

	if (connect(ncp_sock, (const struct sockaddr*)target, sizeof(*target)) == -1)
	{
		int saved_errno = errno;
		close(ncp_sock);
		return saved_errno;
	}

	WSET_LH(conn->packet, 0, NCP_ALLOC_SLOT_REQUEST);
	BSET(conn->packet, 2, conn->sequence);
	BSET(conn->packet, 3, 0xff);
	BSET(conn->packet, 4, 1);
	BSET(conn->packet, 5, 0xff);
	BSET(conn->packet, 6, 0);

	if ((err = do_ncp_call(conn, sizeof(struct ncp_request_header))) != 0)
	{
		/* ? request route ? */
		close(ncp_sock);
		return err;
	}

	conn->wdog_pid = 0;

	conn->sequence = 0;
	conn->i.connection =
	    BVAL(conn->packet, 3) + (BVAL(conn->packet, 5) << 8);
	conn->is_connected = CONN_TEMPORARY;

#ifdef SIGNATURES
	conn->sign_active = 0;
	conn->sign_wanted = 0;
	err = ncp_renegotiate_connparam(conn, NCP_DEFAULT_BUFSIZE, in_options);
#else
	err = ncp_renegotiate_connparam(conn, NCP_DEFAULT_BUFSIZE, 0);
#endif

	if (err != 0)   
	{
		return -1;
	}
	return 0;
}
#endif	/* CONFIG_NATIVE_IP */

static long
ncp_connect_addr(struct ncp_conn *conn, const struct sockaddr *target,
		 int wdog_needed) {
#ifdef CONFIG_NATIVE_IPX
	if (target->sa_family == AF_IPX) {
		return ncp_connect_ipx_addr(conn, (const struct sockaddr_ipx*)target, wdog_needed);
	}
#endif	/* CONFIG_NATIVE_IPX */
#ifdef CONFIG_NATIVE_IP
	if (target->sa_family == AF_INET) {
		return ncp_connect_in_addr(conn, (const struct sockaddr_in*)target, wdog_needed);
	}
#endif	/* CONFIG_NATIVE_IP */
	return EAFNOSUPPORT;
}

static long
ncp_connect_any(struct ncp_conn *conn, int wdog_needed)
{
#ifdef CONFIG_NATIVE_IPX
#ifndef __MAKE_SULIB__
	const char* server = NULL;
#else /* __MAKE_SULIB__ */
	char server[NCP_BINDERY_NAME_LEN];
#endif /* __MAKE_SULIB__ */
	struct sockaddr_ipx addr;
	long result;

#ifndef __MAKE_SULIB__
	if ((result = ncp_find_server_ipx(&server, NCP_BINDERY_FSERVER, &addr)) != 0)
#else /* __MAKE_SULIB__ */
	if ((result = ipx_sap_find_nearest(IPX_SAP_FILE_SERVER,
					   &addr, server)) != 0)
#endif /* __MAKE_SULIB__ */
	{
		return result;
	}
	if ((result = ncp_connect_ipx_addr(conn, &addr, wdog_needed)) != 0)
	{
		return result;
	}
	return 0;
#else
	return EAFNOSUPPORT;
#endif	/* CONFIG_NATIVE_IPX */
}

#ifdef CONFIG_NATIVE_IPX
static long
ncp_find_server_ipx(const char **server_name, int type, struct sockaddr_ipx* server_addr)
{
	char server[NCP_BINDERY_NAME_LEN + 1];
#ifndef __MAKE_SULIB__
	static char buf[128];
	int res;
	char *n;
#else
	long err;
	static char nearest[NCP_BINDERY_NAME_LEN + 1];
	struct nw_property prop;
	struct prop_net_address *n_addr = (struct prop_net_address *) &prop;
	struct ncp_conn* conn;

#endif /* __MAKE_SULIB__ */

	memset(server, 0, sizeof(server));

#ifndef __MAKE_SULIB__
	if (*server_name != NULL)
	{
		strncpy(server, *server_name, sizeof(server) - 1);
		str_upper(server);
	}
	{
		char *buf2[5];
		char b1[10];
		int err;
		
		sprintf(b1, "%d", type);
		buf2[1] = "-t";
		buf2[2] = b1;
		buf2[3] = server[0]?server:NULL;
		buf2[4] = NULL;
		
		err = exec_nwsfind(buf, sizeof(buf), buf2);
		if (err)
			return err;
		res = strlen(buf);
		if (ipx_sscanf_saddr(buf, server_addr) != 0)
		{
			return (*server_name != NULL)
			    ? NWE_SERVER_UNKNOWN : NWE_SERVER_NOT_FOUND;
		}
		if (*server_name == NULL)
		{
			if ((n = strchr(buf, ' ')) == NULL)
			{
				return (*server_name != NULL)
				    ? NWE_SERVER_UNKNOWN : NWE_SERVER_NOT_FOUND;
			}
			*server_name = n;
		}
	}
	return 0;
#else /* __MAKE_SULIB__ */
	memset(nearest, 0, sizeof(nearest));

	if (*server_name != NULL)
	{
		if (strlen(*server_name) >= sizeof(server))
		{
			return ENAMETOOLONG;
		}
		strcpy(server, *server_name);
		str_upper(server);
	}
	if ((err = ipx_sap_find_nearest(type, server_addr, nearest)) != 0)
	{
		return err;
	}
	/* We have to ask the nearest server for our wanted server */
	conn = ncp_alloc_conn();
	if (!conn)
		return ENOMEM;
	
	if ((err = ncp_connect_ipx_addr(conn, server_addr, 0)) != 0)
	{
		ncp_close(conn);
		return err;
	}
	if (*server_name == NULL)
	{
		*server_name = nearest;
		ncp_close(conn);
		return 0;
	}
	/* The following optimization should have been done before
	   ncp_connect_addr. This would be convenient if there was a
	   simple way to find out whether there is a route to the
	   server. Parsing /proc/net/ipx_route is not too nice, so we
	   just connect to the server and immediately disconnect
	   again. This way we also find out if the server still has
	   free connection slots. */
	if (strcmp(server, nearest) == 0)
	{
		/* Our wanted server answered the SAP GNS request, so
		   use it */
		ncp_close(conn);
		return 0;
	}
	if (ncp_read_property_value(conn, type, server, 1,
				    "NET_ADDRESS", &prop) != 0)
	{
		ncp_close(conn);
		return NWE_SERVER_UNKNOWN;
	}
	if ((err = ncp_close(conn)) != 0)
	{
		return err;
	}
	server_addr->sipx_family = AF_IPX;
	server_addr->sipx_network = n_addr->network;
	server_addr->sipx_port = n_addr->port;
	ipx_assign_node(server_addr->sipx_node, n_addr->node);

	/* To make the final server reachable, we connect again. See
	   above. (When can we rely on all users running ipxd??? :-)) */
	conn = ncp_alloc_conn();
	if (!conn)
		return ENOMEM;
	err = ncp_connect_ipx_addr(conn, server_addr, 0);
	ncp_close(conn);
	return err;
#endif /* __MAKE_SULIB__ */
}
#endif	/* CONFIG_NATIVE_IPX */

long 
ncp_find_server(const char **server_name, int type, struct sockaddr* addr, socklen_t len) {
	long err = NWE_SERVER_UNKNOWN;
#ifdef CONFIG_NATIVE_IPX
	if (len >= sizeof(struct sockaddr_ipx)) {
		err = ncp_find_server_ipx(server_name, type, (struct sockaddr_ipx*)addr);
		if (!err) return 0;
	}
#endif	/* CONFIG_NATIVE_IPX */
	return err;
}

long
ncp_find_fileserver(const char *server_name, struct sockaddr* addr, socklen_t len)
{
	return ncp_find_server(&server_name, NCP_BINDERY_FSERVER, addr, len);
}

static int
ncp_login_conn(struct ncp_conn* conn, const char* object_name, int object_type, const char* password) {
	int err;

#ifdef NDS_SUPPORT
	if (!nds_get_tree_name(conn, NULL, 0))
	{
		err = nds_login_auth(conn, object_name, password);
		if (!err) return 0;
		if (err == NWE_PASSWORD_EXPIRED) {
			fprintf(stderr, _("Your password has expired\n"));
			return 0;
		}
		fprintf(stderr, _("Unable to NDS log-in (error %d [0x%04X]), trying bindery...\n"), err, err);
	}
#endif    	
	if (ncp_login_object(conn, object_name, object_type, password) != 0)
	{
		return EACCES;
	}
	return 0;
}

static long
ncp_open_temporary(struct ncp_conn *conn,
		   const char* server)
{
	struct sockaddr addr;
	long err;

	if (server == NULL)
	{
		return ncp_connect_any(conn, 1);
	}
	if ((err = ncp_find_fileserver(server, &addr, sizeof(addr))) != 0)
	{
		return err;
	}
	if ((err = ncp_connect_addr(conn, &addr, 1)) != 0)
	{
		return err;
	}
	return 0;
}

static long
ncp_open_temporary2(struct ncp_conn *conn,
		     const char* address)
{
#ifdef CONFIG_NATIVE_IP
	struct sockaddr_in addr;
	long err;
	struct hostent* h;

	if (!address) return EFAULT;

	h = gethostbyname(address);
	if (!h) return NWE_SERVER_UNKNOWN;
	if (h->h_addrtype != AF_INET) return EAFNOSUPPORT;
	if (h->h_length != 4) return EAFNOSUPPORT;
	addr.sin_family = AF_INET;
	addr.sin_port = htons(0x020C);
	memcpy(&addr.sin_addr.s_addr, h->h_addr, 4);

	if ((err = ncp_connect_addr(conn, (struct sockaddr*)&addr, 1)) != 0)
	{
		return err;
	}
	return 0;
#else	/* CONFIG_NATIVE_IP */
	(void)conn;
	(void)address;
	return EAFNOSUPPORT;
#endif	/* CONFIG_NATIVE_IP */
}

char *
ncp_find_permanent(const struct ncp_conn_spec *spec)
{
#ifdef NCP_KERNEL_NCPFS_AVAILABLE
	FILE *mtab;
	struct ncp_conn_ent *conn_ent;
	char *result = NULL;
	int mount_fid;
	struct ncp_fs_info i;

	if ((mtab = fopen(MOUNTED, "r")) == NULL)
	{
		return NULL;
	}
	while ((conn_ent = ncp_get_conn_ent(mtab)) != NULL)
	{
		if (spec != NULL)
		{
			if ((conn_ent->uid != spec->uid)
			    || ((strlen(spec->server) != 0)
				&& (strcasecmp(conn_ent->server,
					       spec->server) != 0))
			    || ((strlen(spec->user) != 0)
				&& (strcasecmp(conn_ent->user,
					       spec->user) != 0)))
			{
				continue;
			}
		}
		mount_fid = open(conn_ent->mount_point, O_RDONLY, 0);
		if (mount_fid < 0)
		{
			continue;
		}
		i.version = NCP_GET_FS_INFO_VERSION;

		if (ioctl(mount_fid, NCP_IOC_GET_FS_INFO, &i) < 0)
		{
			close(mount_fid);
			continue;
		}
		close(mount_fid);
		result = conn_ent->mount_point;
		break;
	}

	fclose(mtab);
	errno = (result == NULL) ? ENOENT : 0;
	return result;
#else
	return NULL;
#endif	/* NCP_KERNEL_NCPFS_AVAILABLE */
}

#ifdef NCP_KERNEL_NCPFS_AVAILABLE
#ifdef SIGNATURES
static void
ncp_sign_init_perm(struct ncp_conn *conn)
{
	if (ioctl(conn->mount_fid, NCP_IOC_SIGN_WANTED, 
	          &conn->sign_wanted) != 0)
		conn->sign_wanted = 0;
	conn->sign_active = 0;
}
#endif
#endif

static int
ncp_open_permanent(const struct ncp_conn_spec *spec, struct ncp_conn** conn)
{
#ifdef NCP_KERNEL_NCPFS_AVAILABLE
	char *mount_point;
	int err;

	if ((mount_point = ncp_find_permanent(spec)) == NULL)
	{
		return -1;
	}
	err = ncp_open_mount(mount_point, conn);
	if (!err) {
		(*conn)->serverInfo.serverName = NULL;
		(*conn)->user = NULL;
	}
	return err;
#else
	return -1;
#endif	/* NCP_KERNEL_NCPFS_AVAILABLE */
}

static struct ncp_conn *
ncp_open_2(const struct ncp_conn_spec *spec, long *err, const char* address)
{
	struct ncp_conn *result;

	if (ncp_open_permanent(spec, &result) == 0) {
		return result;
	}

	result = ncp_alloc_conn();
	if (result == NULL)
	{
		*err = ENOMEM;
		return NULL;
	}
	if (spec) {
		*err = ncp_open_temporary2(result, address);
		if (*err) *err = ncp_open_temporary(result, spec->server);
	} else {
		*err = ncp_connect_any(result, 1);
	}
	if (*err) {
		ncp_close(result);
		return NULL;
	}
	if (spec && (strlen(spec->user) != 0)) {
		*err = ncp_login_conn(result, spec->user, spec->login_type, spec->password);
		if (*err) {
			ncp_close(result);
			return NULL;
		}
		result->user = strdup(spec->user);
	}
	return result;
}

struct ncp_conn *
ncp_open(const struct ncp_conn_spec *spec, long *err) {
	return ncp_open_2(spec, err, NULL);
}

static int ncp_do_open_fd(int fd, struct ncp_conn** conn) {
#ifdef NCP_KERNEL_NCPFS_AVAILABLE
	struct ncp_conn *result;

	*conn = NULL;

	result = ncp_alloc_conn();
	if (result == NULL) {
		return ENOMEM;
	}
	result->mount_fid = fd;
	result->is_connected = CONN_PERMANENT;

	result->i.version = NCP_GET_FS_INFO_VERSION;

	/* on kernels <= 2.2.10 & 2.3.8 directory_id can be set incorrectly */
	/* do stat(mount_point) before if you want to use it... */
	if (ioctl(result->mount_fid, NCP_IOC_GET_FS_INFO, &(result->i)) != 0)
	{
		free(result);
		return EINVAL;
	}

#ifdef SIGNATURES
	ncp_sign_init_perm(result);
#endif	
	*conn = result;
	return 0;
#else
	return ENOPKG;	/* better error code, anyone? */
#endif	/* NCP_KERNEL_NCPFS_AVAILABLE */
}

int ncp_open_fd(int fd, struct ncp_conn** conn) {
	int fd2;
	int err;

	fd2 = dup(fd);
	if (fd2 == -1)
		return errno;
	err = ncp_do_open_fd(fd2, conn);
	if (err) 
		close(fd2);
	return err;
}

int
ncp_open_mount(const char *mount_point, struct ncp_conn** conn)
{
	int fd;
	int err;
	
	fd = open(mount_point, O_RDONLY, 0);
	if (fd == -1)
		return errno;
	err = ncp_do_open_fd(fd, conn);
	if (err) {
		close(fd);
		return err;
	}
	/* returning NULL is not critical */
	(*conn)->mount_point = strdup(mount_point);
	return 0;
}

static long
ncp_user_disconnect(struct ncp_conn *conn)
{
	long result;

	conn->sequence += 1;

	WSET_LH(conn->packet, 0, NCP_DEALLOC_SLOT_REQUEST);
	BSET(conn->packet, 2, conn->sequence);
	BSET(conn->packet, 3, (conn->i.connection) & 0xff);
	BSET(conn->packet, 4, 1);
	BSET(conn->packet, 5, (conn->i.connection) >> 8);
	BSET(conn->packet, 6, 0);

	if ((result = do_ncp_call(conn, sizeof(struct ncp_request_header))) != 0)
	{
		return result;
	}
	close(conn->ncp_sock);
	if (conn->wdog_sock != -1) {
		close(conn->wdog_sock);
	}

	if (conn->wdog_pid != 0)
	{
		kill(conn->wdog_pid, SIGTERM);
		wait(NULL);
	}
	return 0;
}

static long
ncp_do_close(struct ncp_conn *conn)
{
	long result;

	switch (conn->is_connected)
	{
	case CONN_PERMANENT:
		result = close(conn->mount_fid);
		list_del(&conn->nds_ring);
		/* call NWDSReleaseDSConnection... */
		conn->state++;
		break;

	case CONN_TEMPORARY:
		result = ncp_user_disconnect(conn);
		list_del(&conn->nds_ring);
		/* call NWDSReleaseDSConnection... */
		conn->state++;
		break;

	case NOT_CONNECTED:
		result = 0;
		break;
		
	default:
		result = -1;
		break;
	}
	conn->is_connected = NOT_CONNECTED;

	if (ncpt_atomic_read(&conn->store_count))
		return 0;
	if (conn->mount_point) {
		free(conn->mount_point);
		conn->mount_point = NULL;
	}
	if (conn->serverInfo.serverName) {
		free(conn->serverInfo.serverName);
		conn->serverInfo.serverName = NULL;
	}
	if (conn->user) {
		free(conn->user);
		conn->user = NULL;	/* To be safe */
	}
	ncpt_mutex_destroy(&conn->serverInfo.mutex);
	ncpt_mutex_destroy(&conn->buffer_mutex);
	free(conn);
	return result;
}

long
ncp_conn_release(struct ncp_conn *conn) {
	if (!ncpt_atomic_dec_and_test(&conn->store_count))
		return 0;
	if (ncpt_atomic_read(&conn->use_count))
		return 0;
	return ncp_do_close(conn);
}

long
ncp_close(struct ncp_conn *conn)
{
	if (conn == NULL)
	{
		return 0;
	}
	if (ncpt_atomic_read(&conn->use_count)) {
		if (!ncpt_atomic_dec_and_test(&conn->use_count))
			return 0;
		return ncp_do_close(conn);
	} else
		return NWE_REQUESTER_FAILURE;	/* buggg ! */
}

int 
ncp_get_mount_uid(int fid, uid_t* uid)
{
#ifdef NCP_KERNEL_NCPFS_AVAILABLE
	__kernel_uid_t k_uid;
	int err;

	err = ioctl(fid, NCP_IOC_GETMOUNTUID, &k_uid);
	if (err) return err;
	*uid = k_uid;
	return 0;
#else
	errno = ENOPKG;
	return -1;
#endif	/* NCP_KERNEL_NCPFS_AVAILABLE */
}

struct ncp_conn_ent *
ncp_get_conn_ent(FILE * filep)
{
#ifdef NCP_KERNEL_NCPFS_AVAILABLE
	static struct ncp_conn_ent entry;
	static char server[2 * NCPFS_MAX_CFG_USERNAME];
	char *user;
	struct mntent *mnt_ent;
	int fid;

	memset(server, 0, sizeof(server));
	memset(&entry, 0, sizeof(entry));

	while ((mnt_ent = getmntent(filep)) != NULL)
	{
		if (strcmp(mnt_ent->mnt_type, "ncpfs") != 0)
		{
			continue;
		}
		if (strlen(mnt_ent->mnt_fsname) >= sizeof(server))
		{
			continue;
		}
		strcpy(server, mnt_ent->mnt_fsname);
		user = strchr(server, '/');
		if (user == NULL)
		{
			continue;
		}
		*user = '\0';
		user += 1;
		entry.user = user;
		if ((strlen(server) >= sizeof(entry.server))
		    || (strlen(mnt_ent->mnt_dir) >= sizeof(entry.mount_point)))
		{
			continue;
		}
		strcpy(entry.server, server);
		strcpy(entry.mount_point, mnt_ent->mnt_dir);

		fid = open(entry.mount_point, O_RDONLY, 0);

		if (fid == -1)
		{
			continue;
		}
		if (ncp_get_mount_uid(fid, &entry.uid) != 0)
		{
			close(fid);
			continue;
		}
		close(fid);
		return &entry;
	}
#else
	errno = ENOPKG;
#endif
	return NULL;
}

static struct ncp_conn_spec *
ncp_get_nwc_ent(FILE * nwc)
{
	static struct ncp_conn_spec spec;
	char line[512];
	int line_len;
	char *user;
	char *password;

	memset(&spec, 0, sizeof(spec));
	spec.uid = getuid();

	while (fgets(line, sizeof(line), nwc) != NULL)
	{
		if ((line[0] == '\n')
		    || (line[0] == '#'))
		{
			continue;
		}
		line_len = strlen(line);
		if (line[line_len - 1] == '\n')
		{
			line[line_len - 1] = '\0';
		}
		user = strchr(line, '/');
		password = strchr(user != NULL ? user : line, ' ');

		if (password != NULL)
		{
			*password = '\0';
			password += 1;
		}
		if (user != NULL)
		{
			*user = '\0';
			user += 1;
			if (strlen(user) >= sizeof(spec.user))
			{
				continue;
			}
			strcpy(spec.user, user);
		}
		if (strlen(line) >= sizeof(spec.server))
		{
			continue;
		}
		strcpy(spec.server, line);

		if (password != NULL)
		{
			while (*password == ' ')
			{
				password += 1;
			}

			if (strlen(password) >= sizeof(spec.password))
			{
				continue;
			}
			strcpy(spec.password, password);
		}
		return &spec;
	}
	return NULL;
}

FILE *
ncp_fopen_nwc(const char *user, const char *mode, long *err)
{
	char path[MAXPATHLEN];
	char *home = NULL;
	struct stat st;

	if (mode == NULL)
	{
		mode = "r";
	}
	if (user == NULL)
	{
		home = getenv("HOME");
	} else
	{
		struct passwd *pwd;

		if ((pwd = getpwnam(user)) != NULL)
		{
			home = pwd->pw_dir;
		}
	}

	if ((home == NULL)
	    || (strlen(home) + sizeof(NWCLIENT) + 2 > sizeof(path)))
	{
		*err = ENAMETOOLONG;
		return NULL;
	}
	strcpy(path, home);
	strcat(path, "/");
	strcat(path, NWCLIENT);

	if (stat(path, &st) != 0)
	{
		*err = errno;
		return NULL;
	}
	if ((st.st_mode & (S_IRWXO | S_IRWXG)) != 0)
	{
		*err = NCPLIB_INVALID_MODE;
		return NULL;
	}
	return fopen(path, mode);
}

struct ncp_conn_spec *
ncp_find_conn_spec2(const char *server, const char *user, const char *password,
		   int login_necessary, uid_t uid, int allow_multiple_conns, long *err)
{
	static struct ncp_conn_spec spec;

	FILE *nwc;
	struct ncp_conn_spec *nwc_ent;

	*err = 0;
	memset(&spec, 0, sizeof(spec));
	spec.uid = getuid();

	if (server != NULL)
	{
		if (strlen(server) >= sizeof(spec.server))
		{
			*err = ENAMETOOLONG;
			return NULL;
		}
		strcpy(spec.server, server);
	} else
	{
		if ((nwc = ncp_fopen_nwc(NULL, NULL, err)) == NULL)
		{
			*err = NWE_SERVER_UNKNOWN;
			return NULL;
		}
		nwc_ent = ncp_get_nwc_ent(nwc);
		fclose(nwc);

		if (nwc_ent == NULL)
		{
			*err = NWE_SERVER_NO_CONN;
			return NULL;
		}
		strcpy(spec.server, nwc_ent->server);
		strcpy(spec.user, nwc_ent->user);
	}

	str_upper(spec.server);

	if (login_necessary == 0)
	{
		memset(spec.user, 0, sizeof(spec.user));
		memset(spec.password, 0, sizeof(spec.password));
		return &spec;
	}
	if (user != NULL)
	{
		if (strlen(user) >= sizeof(spec.user))
		{
			*err = ENAMETOOLONG;
			return NULL;
		}
		strcpy(spec.user, user);
	}
	str_upper(spec.user);
	spec.login_type = NCP_BINDERY_USER;

	if (!allow_multiple_conns) {
		struct ncp_conn* conn;

		if (ncp_open_permanent(&spec, &conn) == 0)
		{
			ncp_close(conn);
			return &spec;
		}
	}
	if (password != NULL)
	{
		if (strlen(password) >= sizeof(spec.password))
		{
			*err = ENAMETOOLONG;
			return NULL;
		}
		strcpy(spec.password, password);
	} else
	{
		if ((nwc = ncp_fopen_nwc(NULL, NULL, err)) != NULL)
		{
			while ((nwc_ent = ncp_get_nwc_ent(nwc)) != NULL)
			{
				if ((strcasecmp(spec.server,
						nwc_ent->server) != 0)
				    || ((*spec.user != '\0')
					&& (strcasecmp(spec.user,
						       nwc_ent->user) != 0)))
				{
					continue;
				}
				strcpy(spec.user, nwc_ent->user);
				strcpy(spec.password, nwc_ent->password);
				break;
			}
			fclose(nwc);
		}
	}

	if (strlen(spec.user) == 0)
	{
		*err = NWE_USER_NO_NAME;
		return NULL;
	}
	if ((strlen(spec.password) == 0) && (password == NULL))
	{
		char *password;
		if (!(isatty(0) && isatty(1)))
		{
			return NULL;
		}
		printf(_("Logging into %s as %s\n"),
		       spec.server, spec.user);

		password = getpass(_("Password: "));
		if (strlen(password) > sizeof(spec.password))
		{
			return NULL;
		}
		strcpy(spec.password, password);
	} else
	{
		if (strcmp(spec.password, NWC_NOPASSWORD) == 0)
		{
			*spec.password = '\0';
		}
	}

	str_upper(spec.server);
	str_upper(spec.user);
	str_upper(spec.password);
	return &spec;
}

struct ncp_conn_spec *
ncp_find_conn_spec(const char *server, const char *user, const char *password,
		   int login_necessary, uid_t uid, long *err) {
	return ncp_find_conn_spec2(server, user, password, login_necessary,
				  uid, 0, err);
}

struct ncp_conn *
ncp_initialize_2(int *argc, char **argv, int login_necessary, 
		 int login_type, long *err, int required)
{
	const char *server = NULL;
	const char *user = NULL;
	const char *password = NULL;
	const char *address = NULL;
	struct ncp_conn_spec *spec;
	int i = 1;

	int get_argument(int arg_no, const char **target)
	{
		int count = 1;

		if (target != NULL)
		{
			if (arg_no + 1 >= *argc)
			{
				/* No argument to switch */
				errno = EINVAL;
				return -1;
			}
			*target = argv[arg_no + 1];
			count = 2;
		}
		/* Delete the consumed switch from the argument list
		   and decrement the argument count */
		while (count + arg_no < *argc)
		{
			argv[arg_no] = argv[arg_no + count];
			arg_no += 1;
		}
		*argc -= count;
		return 0;
	}

	*err = EINVAL;

	while (i < *argc)
	{
		if ((argv[i][0] != '-')
		    || (strlen(argv[i]) != 2))
		{
			i += 1;
			continue;
		}
		switch (argv[i][1])
		{
		case 'S':
			if (get_argument(i, &server) != 0)
			{
				return NULL;
			}
			continue;
		case 'U':
			if (get_argument(i, &user) != 0)
			{
				return NULL;
			}
			continue;
		case 'P':
			if (get_argument(i, &password) != 0)
			{
				return NULL;
			}
			continue;
		case 'n':
			if (get_argument(i, NULL) != 0)
			{
				return NULL;
			}
			password = NWC_NOPASSWORD;
			continue;
#ifdef NDS_SUPPORT
		case 'b':
			if (get_argument(i, NULL) != 0)
			{
				return NULL;
			}
			bindery_only = 1;
			continue;
#endif
#ifdef CONFIG_NATIVE_IP
		case 'A':
			if (get_argument(i, &address) != 0)
			{
				return NULL;
			}
			continue;
#endif	/* CONFIG_NATIVE_IP */
		}
		i += 1;
	}

	if ((!required) && !(server || user || password || address))
		return NULL;
	spec = ncp_find_conn_spec(server, user, password, login_necessary,
				  getuid(), err);

	if (spec == NULL)
	{
		if (login_necessary != 0)
		{
			return NULL;
		} else
		{
			return ncp_open(NULL, err);
		}
	}
	spec->login_type = login_type;

	if (login_necessary == 0)
	{
		spec->user[0] = '\0';
	}
	return ncp_open_2(spec, err, address);
}

struct ncp_conn *
ncp_initialize_as(int *argc, char **argv, int login_necessary,
		  int login_type, long *err)
{
	return ncp_initialize_2(argc, argv, login_necessary,
		  login_type, err, 1);
}

struct ncp_conn *
ncp_initialize(int *argc, char **argv,
	       int login_necessary, long *err)
{
	return ncp_initialize_as(argc, argv, login_necessary,
				 NCP_BINDERY_USER, err);
}

long
ncp_request(struct ncp_conn *conn, int function)
{
	long result = ENOTCONN;
#ifdef NCP_TRACE_ENABLE
	{
		char *ppacket = conn->packet + sizeof(struct ncp_request_header);
		if (conn->has_subfunction)
		{
			NCP_TRACE("ncp_request: fn=%d subfn=%d", function, (int) ppacket[2]);
		} else
		{
			NCP_TRACE("ncp_request: fn=%d subfn?%d", function, (int) *ppacket);
		}
		DUMP_HEX("ncp_request: packet=", ppacket, ncp_packet_size(conn));
	}
#endif /*def NCP_TRACE_ENABLE */
	switch (conn->is_connected)
	{
	case CONN_PERMANENT:
		result = ncp_mount_request(conn, function);
		break;
	case CONN_TEMPORARY:
		result = ncp_temp_request(conn, function);
		break;
	default:
	}
#ifdef NCP_TRACE_ENABLE
	NCP_TRACE("ncp_request: reply %ld", result);
	if (result == 0)
	{
		DUMP_HEX("ncp_request->reply=",
			 conn->packet + sizeof(struct ncp_reply_header),
			 conn->ncp_reply_size);
	}
#endif /*def NCP_TRACE_ENABLE */
	return result;
}

/****************************************************************************/
/*                                                                          */
/* Helper functions                                                         */
/*                                                                          */
/****************************************************************************/

struct nw_time_buffer
{
	u_int8_t year __attribute__((packed));
	u_int8_t month __attribute__((packed));
	u_int8_t day __attribute__((packed));
	u_int8_t hour __attribute__((packed));
	u_int8_t minute __attribute__((packed));
	u_int8_t second __attribute__((packed));
	u_int8_t wday __attribute__((packed));
};

static time_t
nw_to_ctime(struct nw_time_buffer *source)
{
	struct tm u_time;

	memset(&u_time, 0, sizeof(u_time));
	u_time.tm_sec = source->second;
	u_time.tm_min = source->minute;
	u_time.tm_hour = source->hour;
	u_time.tm_mday = source->day;
	u_time.tm_mon = source->month - 1;
	u_time.tm_year = source->year;

	if (u_time.tm_year < 80)
	{
		u_time.tm_year += 100;
	}
	return mktime(&u_time);
}

void ncp_add_pstring(struct ncp_conn *conn, const char *s) {
	int len = strlen(s);
	assert_conn_locked(conn);
	if (len > 255)
	{
		ncp_printf(_("ncpfs: string too long: %s\n"), s);
		len = 255;
	}
	assert_ncp_add_byte(conn, len);
	ncp_add_mem(conn, s, len);
	return;
}

void
ncp_init_request(struct ncp_conn *conn)
{
	ncp_lock_conn(conn);

	conn->current_point = conn->packet + sizeof(struct ncp_request_header);
	conn->has_subfunction = 0;
}

void
ncp_init_request_s(struct ncp_conn *conn, int subfunction)
{
	ncp_init_request(conn);
	ncp_add_word_lh(conn, 0);	/* preliminary size */

	ncp_add_byte(conn, subfunction);

	conn->has_subfunction = 1;
}

/* Here the ncp calls begin
 */

static long
ncp_negotiate_buffersize(struct ncp_conn *conn,
			 size_t size, size_t *target)
{
	long result;

	ncp_init_request(conn);
	ncp_add_word_hl(conn, size);

	if ((result = ncp_request(conn, 33)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*target = min(ncp_reply_word_hl(conn, 0), size);

	ncp_unlock_conn(conn);
	return 0;
}

#ifdef SIGNATURES
static long
 ncp_negotiate_size_and_options(struct ncp_conn *conn,
			  size_t size, int options, 
			  size_t *ret_size, int *ret_options)
{
	long result;

	ncp_init_request(conn);
	ncp_add_word_hl(conn, size);
	ncp_add_byte(conn, options);

	if ((result = ncp_request(conn, 0x61)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	if (ncp_reply_word_hl(conn, 0) == 0) 
		*ret_size = size;
	else 
		*ret_size = min(ncp_reply_word_hl(conn, 0), size);
	*ret_options = ncp_reply_byte(conn, 4);

	ncp_unlock_conn(conn);
	return 0;
}
#endif

#ifndef __MAKE_SULIB__
long
ncp_get_file_server_description_strings(struct ncp_conn *conn,
					char target[512])
{
	long result;

	ncp_init_request_s(conn, 201);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(target, ncp_reply_data(conn, 0), 512);
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_get_file_server_time(struct ncp_conn *conn, time_t * target)
{
	long result;

	ncp_init_request(conn);

	if ((result = ncp_request(conn, 20)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*target = nw_to_ctime((struct nw_time_buffer *) ncp_reply_data(conn, 0));
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_set_file_server_time(struct ncp_conn *conn, time_t * source)
{
	long result;
	int year;
	struct tm *utime = localtime(source);

	year = utime->tm_year;
	if (year > 99)
	{
		year -= 100;
	}
	ncp_init_request_s(conn, 202);
	ncp_add_byte(conn, year);
	ncp_add_byte(conn, utime->tm_mon + 1);
	ncp_add_byte(conn, utime->tm_mday);
	ncp_add_byte(conn, utime->tm_hour);
	ncp_add_byte(conn, utime->tm_min);
	ncp_add_byte(conn, utime->tm_sec);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}
#endif	/* MAKE_SULIB */

long
ncp_get_file_server_information(struct ncp_conn *conn,
				struct ncp_file_server_info *target)
{
	long result;
	ncp_init_request_s(conn, 17);
	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(target, ncp_reply_data(conn, 0), sizeof(*target));
	target->MaximumServiceConnections
	    = htons(target->MaximumServiceConnections);
	target->ConnectionsInUse
	    = htons(target->ConnectionsInUse);
	target->MaxConnectionsEverUsed
	    = htons(target->MaxConnectionsEverUsed);
	target->NumberMountedVolumes
	    = htons(target->NumberMountedVolumes);
	ncp_unlock_conn(conn);
	return 0;
}

#ifndef MAKE_SULIB
long
ncp_get_connlist(struct ncp_conn *conn,
		 NWObjectType object_type, const char *object_name,
		 int *returned_no, u_int8_t conn_numbers[256])
{
	long result;

	ncp_init_request_s(conn, 21);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*returned_no = ncp_reply_byte(conn, 0);
	memcpy(conn_numbers, ncp_reply_data(conn, 1), (*returned_no));
	ncp_unlock_conn(conn);
	return 0;
}
#endif	/* MAKE_SULIB */

long
ncp_get_stations_logged_info(struct ncp_conn *conn,
			     u_int32_t connection,
			     struct ncp_bindery_object *target,
			     time_t * login_time)
{
	long result;
	ncp_init_request_s(conn, 28);
	ncp_add_dword_lh(conn, connection);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	if (target) {
		memset(target, 0, sizeof(*target));
		target->object_id = ncp_reply_dword_hl(conn, 0);
		target->object_type = ncp_reply_word_hl(conn, 4);
		memcpy(target->object_name, ncp_reply_data(conn, 6),
	       		sizeof(target->object_name));
	}
	if (login_time)
		*login_time = nw_to_ctime((struct nw_time_buffer *)
				  ncp_reply_data(conn, 54));
	ncp_unlock_conn(conn);
	return 0;
}

#ifndef MAKE_SULIB
long
ncp_get_internet_address(struct ncp_conn *conn,
			 u_int32_t connection,
			 struct sockaddr *target,
			 u_int8_t * conn_type)
{
	long result;
	u_int8_t ct;
	union ncp_sockaddr* t2 = (union ncp_sockaddr*)target;
	ncp_init_request_s(conn, 26);
	ncp_add_dword_lh(conn, connection);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memset(target, 0, sizeof(*target));
	ct = ncp_reply_byte(conn, 12);
	if (conn_type)
		*conn_type = ct;
	if (ct == 11) {	/* 11... why 11? Nobody knows... */
#ifdef NCP_IN_SUPPORT
		t2->inet.sin_family = AF_INET;
		memcpy(&(t2->inet.sin_addr.s_addr), ncp_reply_data(conn, 0), 4);
		/* I'm not sure about port... it is always 0 on Netwares I checked */
		memcpy(&(t2->inet.sin_port), ncp_reply_data(conn, 4), 2);
#endif
	} else {	/* 2... IPX, 3... internal network (doc lies Appletalk DDP) */
#ifdef NCP_IPX_SUPPORT
		t2->ipx.sipx_family = AF_IPX;
		memcpy(&(t2->ipx.sipx_network), ncp_reply_data(conn, 0), 4);
		memcpy(&(t2->ipx.sipx_node), ncp_reply_data(conn, 4), 6);
		memcpy(&(t2->ipx.sipx_port), ncp_reply_data(conn, 10), 2);
#endif
	}
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_send_broadcast(struct ncp_conn *conn,
		   u_int8_t no_conn, const u_int8_t * connections,
		   const char *message)
{
	long result;

	if (strlen(message) > 58)
	{
		return NWE_SERVER_FAILURE;
	}
	ncp_init_request_s(conn, 0);
	ncp_add_byte(conn, no_conn);
	ncp_add_mem(conn, connections, no_conn);
	ncp_add_pstring(conn, message);

	result = ncp_request(conn, 21);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_send_broadcast2(struct ncp_conn *conn,
		    unsigned int conns, const unsigned int* connlist, 
		    const char* message)
{
	int i;
	long result;

	i = strlen(message);
	if (i > 255)
	{
		return NWE_SERVER_FAILURE;
	}
	if (conns > 350)	/* max pkt len ~ 1KB */
				/* maybe do it by handshaked length ? */
		return NWE_SERVER_FAILURE;
		
	ncp_init_request_s(conn, 0x0A);
	ncp_add_word_lh(conn, conns);
	for (;conns; --conns)
		ncp_add_dword_lh(conn, *connlist++);
	ncp_add_byte(conn, i);
	ncp_add_mem(conn, message, i);
	result = ncp_request(conn, 0x15);
	ncp_unlock_conn(conn);
	return result;
}
#endif /* not __MAKE_SULIB__ */

/*
 * target is a 8-byte buffer
 */
long
ncp_get_encryption_key(struct ncp_conn *conn,
		       char *target)
{
	long result;

	ncp_init_request_s(conn, 23);
	 /* fn: 23 , subfn: 23 */    
	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	if (conn->ncp_reply_size < 8)
	{
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	memcpy(target, ncp_reply_data(conn, 0), 8);
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_get_bindery_object_id(struct ncp_conn *conn,
			  NWObjectType object_type,
			  const char *object_name,
			  struct ncp_bindery_object *target)
{
	long result;
	ncp_init_request_s(conn, 53);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	if (conn->ncp_reply_size < 54)
	{
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	target->object_id = ncp_reply_dword_hl(conn, 0);
	target->object_type = ncp_reply_word_hl(conn, 4);
	memcpy(target->object_name, ncp_reply_data(conn, 6), 48);
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_get_bindery_object_name(struct ncp_conn *conn,
			    u_int32_t object_id,
			    struct ncp_bindery_object *target)
{
	long result;
	ncp_init_request_s(conn, 54);
	ncp_add_dword_hl(conn, object_id);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	target->object_id = ncp_reply_dword_hl(conn, 0);
	target->object_type = ncp_reply_word_hl(conn, 4);
	memcpy(target->object_name, ncp_reply_data(conn, 6), 48);
	ncp_unlock_conn(conn);
	return 0;
}

#ifndef __MAKE_SULIB__
long
ncp_scan_bindery_object(struct ncp_conn *conn,
			u_int32_t last_id, NWObjectType object_type, char *search_string,
			struct ncp_bindery_object *target)
{
	long result;
	ncp_init_request_s(conn, 55);
	ncp_add_dword_hl(conn, last_id);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, search_string);
	/* fn: 23 , subfn: 55 */
	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	target->object_id = ncp_reply_dword_hl(conn, 0);
	target->object_type = ncp_reply_word_hl(conn, 4);
	memcpy(target->object_name, ncp_reply_data(conn, 6),
	       NCP_BINDERY_NAME_LEN);
	target->object_flags = ncp_reply_byte(conn, 54);
	target->object_security = ncp_reply_byte(conn, 55);
	target->object_has_prop = ncp_reply_byte(conn, 56);

	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_create_bindery_object(struct ncp_conn *conn,
			  NWObjectType object_type,
			  const char *object_name,
			  u_int8_t object_security,
			  u_int8_t object_status)
{
	long result;
	ncp_init_request_s(conn, 50);
	ncp_add_byte(conn, object_status);
	ncp_add_byte(conn, object_security);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}


long
ncp_delete_bindery_object(struct ncp_conn *conn,
			  NWObjectType object_type,
			  const char *object_name)
{
	long result;
	ncp_init_request_s(conn, 51);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}
#endif /* not __MAKE_SULIB__ */

long
ncp_read_property_value(struct ncp_conn *conn,
			NWObjectType object_type, const char *object_name,
			int segment, const char *prop_name,
			struct nw_property *target)
{
	long result;
	ncp_init_request_s(conn, 61);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_byte(conn, segment);
	ncp_add_pstring(conn, prop_name);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(&(target->value), ncp_reply_data(conn, 0), 128);
	target->more_flag = ncp_reply_byte(conn, 128);
	target->property_flag = ncp_reply_byte(conn, 129);
	ncp_unlock_conn(conn);
	return 0;
}

#ifndef __MAKE_SULIB__
long
ncp_scan_property(struct ncp_conn *conn,
		  NWObjectType object_type, const char *object_name,
		  u_int32_t last_id, const char *search_string,
		  struct ncp_property_info *property_info)
{
	long result;
	ncp_init_request_s(conn, 60);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_dword_hl(conn, last_id);
	ncp_add_pstring(conn, search_string);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(property_info->property_name, ncp_reply_data(conn, 0), 16);
	property_info->property_flags = ncp_reply_byte(conn, 16);
	property_info->property_security = ncp_reply_byte(conn, 17);
	property_info->search_instance = ncp_reply_dword_hl(conn, 18);
	property_info->value_available_flag = ncp_reply_byte(conn, 22);
	property_info->more_properties_flag = ncp_reply_byte(conn, 23);
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_add_object_to_set(struct ncp_conn *conn,
		      NWObjectType object_type, const char *object_name,
		      const char *property_name,
		      NWObjectType member_type,
		      const char *member_name)
{
	long result;
	ncp_init_request_s(conn, 65);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_pstring(conn, property_name);
	ncp_add_word_hl(conn, member_type);
	ncp_add_pstring(conn, member_name);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_change_property_security(struct ncp_conn *conn,
			     NWObjectType object_type, const char *object_name,
			     const char *property_name,
			     u_int8_t property_security)
{
	long result;
	ncp_init_request_s(conn, 59);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_byte(conn, property_security);
	ncp_add_pstring(conn, property_name);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_create_property(struct ncp_conn *conn,
		    NWObjectType object_type, const char *object_name,
		    const char *property_name,
		    u_int8_t property_flags, u_int8_t property_security)
{
	long result;
	ncp_init_request_s(conn, 57);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_byte(conn, property_flags);
	ncp_add_byte(conn, property_security);
	ncp_add_pstring(conn, property_name);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_delete_object_from_set(struct ncp_conn *conn,
			   NWObjectType object_type, const char *object_name,
			   const char *property_name,
			   NWObjectType member_type,
			   const char *member_name)
{
	long result;
	ncp_init_request_s(conn, 66);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_pstring(conn, property_name);
	ncp_add_word_hl(conn, member_type);
	ncp_add_pstring(conn, member_name);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_delete_property(struct ncp_conn *conn,
		    NWObjectType object_type, const char *object_name,
		    const char *property_name)
{
	long result;
	ncp_init_request_s(conn, 58);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_pstring(conn, property_name);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_write_property_value(struct ncp_conn *conn,
			 NWObjectType object_type, const char *object_name,
			 const char *property_name,
			 u_int8_t segment,
			 const struct nw_property *property_value)
{
	long result;
	ncp_init_request_s(conn, 62);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_byte(conn, segment);
	ncp_add_byte(conn, property_value->more_flag);
	ncp_add_pstring(conn, property_name);
	ncp_add_mem(conn, property_value->value, 128);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_get_big_ncp_max_packet_size(struct ncp_conn *conn,
				u_int16_t proposed_max_size,
				u_int8_t proposed_security_flag,
				u_int16_t * accepted_max_size,
				u_int16_t * echo_socket,
				u_int8_t * accepted_security_flag)
{
	long result;
	ncp_init_request(conn);
	ncp_add_word_hl(conn, proposed_max_size);
	ncp_add_byte(conn, proposed_security_flag);

	if ((result = ncp_request(conn, 97)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*accepted_max_size = ncp_reply_word_hl(conn, 0);
	*echo_socket = ncp_reply_word_hl(conn, 2);
	*accepted_security_flag = ncp_reply_byte(conn, 4);
	ncp_unlock_conn(conn);
	return 0;
}

#if 0
static long
ncp_verify_object_password(struct ncp_conn *conn,
		           const char *objName, NWObjectType objType,
                           const char *objPassword)
{
	long result;
	ncp_init_request_s(conn, 63);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_pstring(conn, passwd);
	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}
#endif

static long
ncp_keyed_verify_password(struct ncp_conn *conn,
			  const struct ncp_bindery_object *object,
			  const unsigned char *key,
			  const unsigned char *passwd)
{
	dword tmpID = htonl(object->object_id);
	unsigned char buf[128];
	unsigned char encrypted[8];
	long result;

	shuffle((byte *) & tmpID, passwd, strlen(passwd), buf);
	nw_encrypt(key, buf, encrypted);

	ncp_init_request_s(conn, 74);
	ncp_add_mem(conn, encrypted, 8);
	ncp_add_word_hl(conn, object->object_type);
	ncp_add_pstring(conn, object->object_name);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

NWCCODE NWVerifyObjectPassword(NWCONN_HANDLE conn,
			       const char* objName,
			       NWObjectType objType,
			       const char* objPassword) {
	NWCCODE result;
	u_int8_t ncp_key[8];
	struct ncp_bindery_object user;

	if ((result = ncp_get_encryption_key(conn, ncp_key)) != 0)
	{
#if 0
		if (!ncp_conn_trusted(conn))
			result = ncp_verify_object_password(conn, objName, 
						objType, objPassword);
#endif						  
		return result;
	}
	result = ncp_get_bindery_object_id(conn, objType, objName, &user);
	if (result)
		return result;
	result = ncp_keyed_verify_password(conn, &user, ncp_key, objPassword);
	return result;
}
#endif /* not __MAKE_SULIB__ */

long
ncp_login_encrypted(struct ncp_conn *conn,
		    const struct ncp_bindery_object *object,
		    const unsigned char *key,
		    const unsigned char *passwd)
{
	dword tmpID = htonl(object->object_id);
	unsigned char buf[128];
	unsigned char encrypted[8];
	long result;

	shuffle((byte *) & tmpID, passwd, strlen(passwd), buf);
	nw_encrypt(key, buf, encrypted);

	ncp_init_request_s(conn, 24);
	ncp_add_mem(conn, encrypted, 8);
	ncp_add_word_hl(conn, object->object_type);
	ncp_add_pstring(conn, object->object_name);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	if ((result == 0) || (result == NWE_PASSWORD_EXPIRED))
	{
		conn->state++;
#ifdef SIGNATURES	
		memcpy(buf + 16, key, 8);
		sign_init(buf, buf);
		result = ncp_sign_start(conn, buf);
#endif	
	}
	return result;
}

long
ncp_login_unencrypted(struct ncp_conn *conn,
		      NWObjectType object_type, const char *object_name,
		      const unsigned char *passwd)
{
	long result;
	ncp_init_request_s(conn, 20);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_pstring(conn, passwd);
	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	if ((result == 0) || (result == NWE_PASSWORD_EXPIRED))
		conn->state++;
	return result;
}

#ifndef __MAKE_SULIB__
long
ncp_change_login_passwd(struct ncp_conn *conn,
			const struct ncp_bindery_object *object,
			const unsigned char *key,
			const unsigned char *oldpasswd,
			const unsigned char *newpasswd)
{
	long id = htonl(object->object_id);
	unsigned char cryptkey[8];
	unsigned char newpwd[16];	/* new passwd as stored by server */
	unsigned char oldpwd[16];	/* old passwd as stored by server */
	unsigned char len;
	long result;

	memcpy(cryptkey, key, 8);
	shuffle((byte *) & id, oldpasswd, strlen(oldpasswd), oldpwd);
	shuffle((byte *) & id, newpasswd, strlen(newpasswd), newpwd);
	nw_encrypt(cryptkey, oldpwd, cryptkey);
	newpassencrypt(oldpwd, newpwd, newpwd);
	newpassencrypt(oldpwd + 8, newpwd + 8, newpwd + 8);
	if ((len = strlen(newpasswd)) > 63)
	{
		len = 63;
	}
	len = ((len ^ oldpwd[0] ^ oldpwd[1]) & 0x7f) | 0x40;

	ncp_init_request_s(conn, 75);
	ncp_add_mem(conn, cryptkey, 8);
	ncp_add_word_hl(conn, object->object_type);
	ncp_add_pstring(conn, object->object_name);
	ncp_add_byte(conn, len);
	ncp_add_mem(conn, newpwd, 16);
	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}
#endif /* not __MAKE_SULIB__ */

long
ncp_login_user(struct ncp_conn *conn,
	       const unsigned char *username,
	       const unsigned char *password)
{
	return ncp_login_object(conn, username, NCP_BINDERY_USER, password);
}

static long
ncp_login_object(struct ncp_conn *conn,
		 const unsigned char *username,
		 int login_type,
		 const unsigned char *password)
{
	long result;
	unsigned char ncp_key[8];
	struct ncp_bindery_object user;

	if ((result = ncp_get_encryption_key(conn, ncp_key)) != 0)
	{
		return ncp_login_unencrypted(conn, login_type, username,
					     password);
	}
	if ((result = ncp_get_bindery_object_id(conn, login_type,
						username, &user)) != 0)
	{
		return result;
	}
	if ((result = ncp_login_encrypted(conn, &user,
					  ncp_key, password)) != 0)
	{
		struct nw_property p;
		struct ncp_prop_login_control *l
		= (struct ncp_prop_login_control *) &p;

		if (result != NWE_PASSWORD_EXPIRED)
		{
			return result;
		}
		fprintf(stderr, _("Your password has expired\n"));

		if ((result = ncp_read_property_value(conn, NCP_BINDERY_USER,
						      username, 1,
						      "LOGIN_CONTROL",
						      &p)) == 0)
		{
			fprintf(stderr, _("You have %d login attempts left\n"),
				l->GraceLogins);
		}
	}
	return 0;
}

#ifdef SIGNATURES
long
ncp_sign_start(struct ncp_conn *conn, const char *sign_root)
{
	static const char init_last[16]=
			   {0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,
	                    0xfe,0xdc,0xba,0x98,0x76,0x54,0x32,0x10};
	if (conn->sign_wanted)
	{
		memcpy(conn->sign_data.sign_root, sign_root, 8);
		memcpy(conn->sign_data.sign_last, init_last, 16);
		conn->sign_active = 1;
		if (conn->is_connected == CONN_PERMANENT)
		{
#ifdef NCP_KERNEL_NCPFS_AVAILABLE
			if (ioctl(conn->mount_fid, NCP_IOC_SIGN_INIT, 
			          &conn->sign_data))
#endif				  
			          return NWE_SIGNATURE_LEVEL_CONFLICT;
		}
	}
	return 0;
}
#endif

#ifdef NDS_SUPPORT
long
ncp_send_nds_frag(struct ncp_conn *conn,
		  int ndsverb,
		  const char *inbuf, size_t inbuflen,
		  char *outbuf, size_t outbufsize, size_t *outbuflen)
{
	long result;
	size_t  sizeleft, i;
	size_t	maxdatasize = 514;
	int first = 1;
	int firstReply = 1;
	int fraghnd = -1;
	int32_t	ndsCode = -399;
	size_t	replyLen = 0;
	size_t	fragLen;

	if (outbuflen) *outbuflen = 0;
	do
	{
		sizeleft = maxdatasize;
		ncp_init_request(conn);
		ncp_add_byte(conn, 2);
		ncp_add_dword_lh(conn, fraghnd);
		if (first)
		{
			ncp_add_dword_lh(conn, maxdatasize - 8);
			ncp_add_dword_lh(conn, inbuflen + 12);
			ncp_add_dword_lh(conn, 0);
			ncp_add_dword_lh(conn, ndsverb);
			ncp_add_dword_lh(conn, outbufsize);
			sizeleft -= 25;
			first = 0;
		}
		else
			sizeleft -= 5;
		i = (sizeleft > inbuflen) ? inbuflen : sizeleft;
		if (i) ncp_add_mem(conn, inbuf, i);
		inbuflen -= i;
		inbuf += i;
		if ((result = ncp_request(conn, 0x68)) != 0)
		{
			ncp_unlock_conn(conn);
			ncp_dprintf(_("Error in ncp_request\n"));
			return result;
		}
		fragLen = ncp_reply_dword_lh(conn, 0);
		if (fragLen < 4) {
			ncp_unlock_conn(conn);
			ncp_dprintf(_("Fragment too short\n"));
			return NWE_INVALID_NCP_PACKET_LENGTH;
		}
		fraghnd = ncp_reply_dword_lh(conn, 4);
		fragLen -= 4;
		if (fragLen) {
			int hdr;

			if (firstReply) {
				ndsCode = ncp_reply_dword_lh(conn, 8);
				hdr = 12;
				fragLen -= 4;
				firstReply = 0;
			} else {
				hdr = 8;
			}
			if (fragLen > outbufsize) {
				ncp_unlock_conn(conn);
				ncp_dprintf(_("Fragment too large, len=%d, max=%d\n"), fragLen, outbufsize);
				return NWE_BUFFER_OVERFLOW;
			}
			if (outbuf) {
				memcpy(outbuf, ncp_reply_data(conn, hdr), fragLen);
				outbuf += fragLen;
			}
			replyLen += fragLen;
		} else {
			/* if reply len == 0 then we must have something to transmit */
			/* otherwise it can cause endless loop */
			if ((fraghnd != -1) && (inbuflen == 0)) {
				ncp_unlock_conn(conn);
				ncp_dprintf(_("Why next fragment?\n"));
				return NWE_SERVER_FAILURE;
			}
		}
		ncp_unlock_conn(conn);
		if (fraghnd != -1) {
			ncp_dprintf(_("Fragmented\n"));
		}
	} while (fraghnd != -1);
	if (inbuflen || firstReply) {
		ncp_dprintf(_("InBufLen after request=%d, FirstReply=%d\n"), inbuflen, firstReply);
		return NWE_SERVER_FAILURE;
	}
	if (outbuflen) *outbuflen = replyLen;
	if (ndsCode != 0) {
		ncp_dprintf(_("NDS error %d\n"), ndsCode);
		if ((ndsCode < 0) && (ndsCode > -256))
			return -ndsCode | NWE_SERVER_ERROR;
		else
			return ndsCode;
	}
	return 0;
}

long
ncp_send_nds(struct ncp_conn *conn, int fn,
	     const char *data_in, size_t data_in_len, 
	     char *data_out, size_t data_out_max, size_t *data_out_len)
{
	size_t i;
	long err;

	ncp_init_request(conn);
	ncp_add_byte(conn, fn);
	if (data_in) ncp_add_mem(conn, data_in, data_in_len);
	if (!(err = ncp_request(conn, 0x68)))
	{
		i = conn->ncp_reply_size;
		if (i > data_out_max) i = data_out_max;
		if (data_out)
			memcpy(data_out, ncp_reply_data(conn, 0), i);
		if (data_out_len) *data_out_len = i;
	}
	else
		if (data_out_len) *data_out_len = 0;
	ncp_unlock_conn(conn);
	return err;
}

long
ncp_change_conn_state(struct ncp_conn *conn, int new_state)
{
	long err;

	ncp_init_request_s(conn, 0x1d);
	ncp_add_dword_lh(conn, new_state);
	err = ncp_request(conn, 0x17);
	ncp_unlock_conn(conn);
	return err;
}
#endif	/* NDS_SUPPORT */

int
ncp_attach_by_addr(const struct sockaddr *target, struct ncp_conn** result) 
{
	struct ncp_conn *conn;
	int err;

	*result = NULL;
	conn = ncp_alloc_conn();
	if (!conn) {
		return ENOMEM;
	}
	err = ncp_connect_addr(conn, target, 1);
	if (err) {
		ncp_close(conn);
		return err;
	}
	*result = conn;
	return 0;
}

struct ncp_conn *
ncp_open_addr(const struct sockaddr *target, long *err) {
	struct ncp_conn *conn;

	*err = ncp_attach_by_addr(target, &conn);
	return conn;
}

long
ncp_get_broadcast_message(struct ncp_conn *conn, char message[256])
{
	long result;
	int length;

	ncp_init_request_s(conn, 0x0B);
	if ((result = ncp_request(conn, 0x15)) != 0)
	{
		ncp_unlock_conn(conn);
		ncp_init_request_s(conn, 0x01);
		if ((result = ncp_request(conn, 0x15)) != 0)
		{
			ncp_unlock_conn(conn);
			return result;
		}
	}
	length = ncp_reply_byte(conn, 0);
	message[length] = 0;
	memcpy(message, ncp_reply_data(conn, 1), length);
	ncp_unlock_conn(conn);
	return 0;
}

int
ncp_get_conn_type(struct ncp_conn* conn) {
	if (conn) {
		switch (conn->is_connected) {
			case CONN_PERMANENT:
					return NCP_CONN_PERMANENT;
			case CONN_TEMPORARY:
					return NCP_CONN_TEMPORARY;
			default:;
		}
	}
	return NCP_CONN_INVALID;
}

int
ncp_get_conn_number(struct ncp_conn* conn) {
	int type;
	
	type = ncp_get_conn_type(conn);
	if (conn == NCP_CONN_INVALID)
		return -ENOTCONN;
	return conn->i.connection;
}

int ncp_get_fid(struct ncp_conn* conn) {
	if (ncp_get_conn_type(conn) != NCP_CONN_PERMANENT)
		return -1;
	return conn->mount_fid;
}

NWCCODE
ncp_get_dentry_ttl(struct ncp_conn* conn, unsigned int* ttl) {
#ifdef NCP_KERNEL_NCPFS_AVAILABLE
	int fd = ncp_get_fid(conn);
	u_int32_t kernel_ttl;
	
	if (fd == -1)
		return NWE_REQUESTER_FAILURE;
	if (ioctl(fd, NCP_IOC_GETDENTRYTTL, &kernel_ttl)) {
		if (errno != EINVAL)
			return errno;
		kernel_ttl = 0;	/* old kernel... */
	}
	*ttl = kernel_ttl;
	return 0;
#else
	return NWE_REQUESTER_FAILURE;
#endif
}

NWCCODE
ncp_set_dentry_ttl(struct ncp_conn* conn, unsigned int ttl) {
#ifdef NCP_KERNEL_NCPFS_AVAILABLE
	int fd = ncp_get_fid(conn);
	u_int32_t kernel_ttl;
	
	if (fd == -1)
		return NWE_REQUESTER_FAILURE;
	kernel_ttl = ttl;
	if (ioctl(fd, NCP_IOC_SETDENTRYTTL, &kernel_ttl))
		return errno;
	return 0;
#else
	return NWE_REQUESTER_FAILURE;
#endif
}

struct errxlat {
	int ecode;
	char* estr;
};

static const char*
strdserror(int err)
{
	static const struct errxlat dserrors[] = {
		/* keep list sorted! Maybe I'll use bsearch someday */
		{ERR_NOT_ENOUGH_MEMORY,		/* -301 */
			N_("Not enough memory")},
		{ERR_BAD_KEY,			/* -302 */
			N_("Bad key passed to NWDS{Get|Set}Context")},
		{ERR_BAD_CONTEXT,		/* -303 */
			N_("Invalid context handle")},
		{ERR_BUFFER_FULL,		/* -304 */
			N_("Buffer full")},
		/* ... */
		{ERR_BUFFER_EMPTY,		/* -307 */
			N_("Buffer empty")}, /* Buffer underflow? Buffer exhausted? */
		/* ... */
		{ERR_EXPECTED_IDENTIFIER,	/* -309 */
			N_("Expected identifier")},
		/* ... */
		{ERR_ATTR_TYPE_EXPECTED,	/* -311 */
			N_("Attribute type expected")},
		{ERR_ATTR_TYPE_NOT_EXPECTED,	/* -312 */
			N_("Attribute type not expected")},
		/* ... */
		{ERR_INVALID_OBJECT_NAME,	/* -314 */
			N_("Invalid object name")},
		{ERR_EXPECTED_RDN_DELIMITER,	/* -315 */
			N_("Expected RDN delimiter")},
		{ERR_TOO_MANY_TOKENS,		/* -316 */
			N_("Too many tokens")},
		{ERR_INCONSISTENT_MULTIAVA,	/* -317 */
			N_("Inconsistent multiava")}, /* multi attribute value?! */
		{ERR_COUNTRY_NAME_TOO_LONG,	/* -318 */
			N_("Country name too long")},
		{ERR_SYSTEM_ERROR,		/* -319 */
			N_("System error")},
		/* ... */
		{ERR_CONTEXT_CREATION,		/* -328 */
			N_("Cannot create context")},
		/* ... */
		{ERR_INVALID_SERVER_RESPONSE,	/* -330 */
			N_("Invalid server response")},
		{ERR_NULL_POINTER,		/* -331 */
			N_("NULL pointer seen")},
		/* ... */
		{ERR_NO_CONNECTION,		/* -333 */
			N_("No connection exists")},
		/* ... */
		{ERR_DUPLICATE_TYPE,		/* -335 */
			N_("Duplicate type")},
		/* ... */
		{ERR_INVALID_PASSWORD_CHARS,	/* -338 */
			N_("Invalid password characters")},
		/* ... */
		{ERR_TRANSPORT,			/* -340 */
			N_("Bad transport")},
		{ERR_NO_SUCH_SYNTAX,		/* -341 */
			N_("No such syntax")},
		{ERR_INVALID_DS_NAME,		/* -342 */
			N_("Invalid DS name")},
		/* ... */
		{ERR_UNICODE_FILE_NOT_FOUND,	/* -348 */
			N_("Required unicode translation not available")},
		/* ... */
		{ERR_DN_TOO_LONG,		/* -353 */
			N_("DN too long")},
		/* ... */
		{ERR_NO_SUCH_ENTRY,		/* -601 */
			N_("No such entry")},
		/* ... */
		{ERR_ALL_REFERRALS_FAILED,	/* -626 */
			N_("All referrals failed")},
		/* ... */
		{ERR_NO_REFERRALS,		/* -634 */
			N_("No referrals")},
		/* ... */
		{ERR_FAILED_AUTHENTICATION,	/* -669 */
			N_("Invalid password")},
		/* ... */
		{-9999,
			NULL}};
	const struct errxlat* eptr = dserrors;
	char* msg = N_("Unknown NDS error");
		
	if (err > -9999) {
		while (eptr->ecode > err) eptr++;
		if (eptr->ecode == err)
			msg = eptr->estr;
	}
	{
		static char sbuf[256];

		sprintf(sbuf, "%s (%d)", _(msg), err);
		return sbuf;
	}
}

static const char*
strrqerror(int err)
{
	static const struct errxlat rqerrors[] = {
		/* keep list sorted! Maybe I'll use bsearch someday */
		{NWE_BUFFER_OVERFLOW,		/* 880E */
			N_("Server reply too long")},
		{NWE_SERVER_NO_CONN,		/* 880F */
			N_("Connection to specified server does not exist")},
		/* ... */
		{NWE_INVALID_NCP_PACKET_LENGTH,	/* 8816 */
			N_("Invalid NCP packet length")},
		/* ... */
		{NWE_USER_NO_NAME,		/* 8834 */
			N_("User name is not specified")},
		/* ... */
		{NWE_SERVER_NOT_FOUND,		/* 8847 */
			N_("Server not found")},
		/* ... */
		{NWE_SIGNATURE_LEVEL_CONFLICT,	/* 8861 */
			N_("Signature level conflict")},
		/* ... */
		{0x10000,
			NULL}};
	const struct errxlat* eptr = rqerrors;
	char* msg = N_("Unknown Requester error");
		
	if (err < 0x10000) {
		while (eptr->ecode < err) eptr++;
		if (eptr->ecode == err)
			msg = eptr->estr;
	}
	{
		static char sbuf[256];

		sprintf(sbuf, "%s (0x%04X)", _(msg), err);
		return sbuf;
	}
}

static const char*
strsrverror(int err)
{
	static const struct errxlat srverrors[] = {
		/* keep list sorted! Maybe I'll use bsearch someday */
		{NWE_LOGIN_MAX_EXCEEDED,		/* 89D9 */
			N_("Connection limit count exceeded")},
		{NWE_LOGIN_UNAUTHORIZED_TIME,		/* 89DA */
			N_("Unauthorized time")},
		{NWE_LOGIN_UNAUTHORIZED_STATION,	/* 89DB */
			N_("Unauthorized station")},
		{NWE_ACCT_DISABLED,			/* 89DC */
			N_("Account disabled")},
		/* ... */
		{NWE_PASSWORD_INVALID,			/* 89DE */
			N_("Password really expired")}, /* no grace logins left */
		{NWE_PASSWORD_EXPIRED,			/* 89DF */
			N_("Password expired")},
		/* ... */
		{NWE_SERVER_UNKNOWN,			/* 89FC */
			N_("Unknown server")},
		/* ... */
		{NWE_SERVER_FAILURE,			/* 89FF */
			N_("Server failure")},
		/* ... */
		{0x10000,
			NULL}};
	const struct errxlat* eptr = srverrors;
	char* msg = N_("Unknown Server error");
		
	if (err < 0x10000) {
		while (eptr->ecode < err) eptr++;
		if (eptr->ecode == err)
			msg = eptr->estr;
	}
	{
		static char sbuf[256];

		sprintf(sbuf, "%s (0x%04X)", _(msg), err);
		return sbuf;
	}
}

const char*
strnwerror(int err)
{
	if (err < 0)
		return strdserror(err);
	else if (err < 0x8800)
		return strerror(err);
	else if (err < 0x8900)
		return strrqerror(err);
	else if (err < 0x8A00)
		return strsrverror(err);
	else {
		static char sbuf[100];
		
		sprintf(sbuf, _("Unknown error %d (0x%X)"), err, err);
		return sbuf;
	}
}

void
com_err(const char* prog, int err, const char* msg, ...)
{
	if (prog)
		fprintf(stderr, "%s: ", prog);
	fprintf(stderr, "%s ", strnwerror(err));
	if (msg) {
		va_list va;
		
		va_start(va, msg);
		vfprintf(stderr, msg, va);
		va_end(va);
	}
	fprintf(stderr, "\n");
}
