/*  Protocol compatible masqdialer server written in C
    Copyright (C) 1998 Charles P. Wright 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <ctype.h>

#include "mserver.h"

extern FILE *outsock;
FILE *logfile, *wvlog;

int cspeed;

bool parse_pppd(void);
bool parse_pap(void);
bool parse_wvdial(void);
bool parse_ipppd(void);

bool parse_pppd_init(void);
bool parse_ipppd_init(void);
bool parse_pap_init(void);
bool parse_wvdial_init(void);

bool parse_pppd_backend(const char *, const char *, bool);

bool parse(const char *cname)
{
	char parsetype[1024];

	if (config_getvalue_bool("wvdial", false))
	{
		if (!parse_wvdial())
		{
			fclose(wvlog);
			fclose(logfile);
			return false;
		}
		fclose(wvlog);
	}

	config_getvalue_cname_specified("parselog", parsetype, cname, 1024);
	if(!strncmp(parsetype, "pppd", 1024))
	{
		return parse_pppd();
	}
	else if(!strncmp(parsetype, "ipppd", 1024))
	{
		return parse_ipppd();
	}
	else if(!strncmp(parsetype, "pap", 1024))
	{
		return parse_pap();
	}
	else if (!strncmp(parsetype, "none", 1024))
	{
		return true;
	}
	else
	{
		syslog(LOG_ERR, "Unknown parselog action: %s", parsetype);
		return false;
	}
}

bool parse_init(const char *cname)
{
	char parsetype[1024];

	config_getvalue_cname_specified("parselog", parsetype, cname, 1024);

	if (config_getvalue_bool("wvdial", false))
		if (!parse_wvdial_init()) return false;

	if(!strncmp(parsetype, "pppd", 1024))
	{
		return parse_pppd_init();
	}
	else if(!strncmp(parsetype, "ipppd", 1024))
	{
		return parse_ipppd_init();
	}
	else if(!strncmp(parsetype, "pap", 1024))
	{
		return parse_pap_init();
	}
	else if (!strncmp(parsetype, "none", 1024))
	{
		return true;
	}
	else
	{
		syslog(LOG_ERR, "Unknown parselog action: %s", parsetype);
		return false;
	}
}

bool parse_ipppd_init(void)
{
	return parse_pppd_init();
}

bool parse_pap_init(void)
{
	return parse_pppd_init();
}

bool parse_wvdial_init(void)
{
	char temp[1024];

	config_getvalue("wvlogfile", temp, 1024);

	wvlog = fopen(temp, "r");

	if (wvlog == NULL)
	{
		syslog(LOG_WARNING, "Could not open wvdial log file: %s", temp);

		fprintf(outsock, "ERROR: Could not open wvdial log file: %s\n", temp);
		fprintf(outsock,"END\n");
		return false;
	}

	(void) fseek(wvlog, 0L, SEEK_END);

	return true;
}


bool parse_pppd_init(void)
{
	char temp[1024];

	config_getvalue("logfile", temp, 1024);

	logfile = fopen(temp, "r");

	if (logfile == NULL)
	{
		syslog(LOG_WARNING, "Could not open pppd log file: %s", temp);

		fprintf(outsock, "ERROR: Could not open pppd log file: %s\n", temp);
		fprintf(outsock,"END\n");
		return false;
	}

	(void) fseek(logfile, 0L, SEEK_END);
	
	return true;
}

bool parse_pppd(void)
{
	char cstring[1024];
	config_getvalue_default("connectstr", cstring, "CONNECT", 1024);
	return parse_pppd_backend(cstring, "Connect: ppp", config_getvalue_bool("speedsearch", true));
}

bool parse_pap(void)
{
	char cstring[1024];
	config_getvalue_default("connectstr", cstring, "CONNECT", 1024);
	return parse_pppd_backend(cstring, "local  IP address", config_getvalue_bool("speedsearch", true));
}

bool parse_ipppd(void)
{
	return parse_pppd_backend("PHASE_WAIT -> PHASE_ESTABLISHED", "local  IP address", false);
}

bool get_connect_speed(char *str)
{
	char *start;
	char charspeed[10240];
	int i, n, l;

	char *validspeeds[100];
	char *tok;
	int numspeeds;
	

	config_getvalue_default("validspeed", charspeed, "28800,26400,24000,12600,19200,16800,14400,12000,9600", 10240);

	tok = strtok(charspeed, ",");

	validspeeds[0] = strdup(tok);
	numspeeds = 1;

	while((tok = strtok(NULL, ",")))
	{
		validspeeds[numspeeds++] = strdup(tok);
	}

	charspeed[0] = '\0';
	
	for(l = 0; l < numspeeds; l++)
	{
		if((start = strstr(str, validspeeds[l])) != NULL)
		{
			i = n = 0;
	
			while (isdigit(start[i]))
			{
				charspeed[n++] = start[i++];
			}
			charspeed[n] = '\0';
	
			cspeed = atoi(charspeed);
			syslog (LOG_INFO, "Connection speed: %d", cspeed);

			for(l = 0; l < numspeeds; l++)
			{	
				free(validspeeds[l]);
			}	
			return true;
		}
		
	}
	for(l = 0; l < numspeeds; l++)
	{	
		free(validspeeds[l]);
	}	
	return false;
}

bool parse_pppd_backend (const char *modemconnected, const char *pppup, bool speedsearch)
{
	int sleep_sec;
	int log_timeout;

	char temp[1024];
	
	char *tok;
	char toktemp[1024];

	char pppdstr[1024];
	char chatstr[1024];

	bool speedcheck;

	config_getvalue_default("pppdstr", pppdstr, "pppd[", 1024);	
	config_getvalue_default("chatstr", chatstr, "pppd[", 1024);	

	log_timeout = config_getvalue_int("logtimeout", 120);
	sleep_sec = 0;


	cspeed = config_getvalue_int("cspeed", 28800);
	speedcheck = false;

	while (true)
	{
		if (fgets(temp, 1024, logfile) == NULL)
		{
			util_sleep (1);
			sleep_sec++;
			if (sleep_sec > log_timeout)
			{
				fprintf (outsock, "Timeout parsing logfile\n");
				fclose(logfile);
				return false;
			}
		}
		else
		{
			if (strstr(temp, pppdstr) != NULL)
			{
				if (strstr(temp, "started") != NULL)
				{
					tok = strtok(temp, "[");
					tok = strtok(NULL, "[");
					strncpy(toktemp, tok, 1024);
					tok = strtok(toktemp, "]");
					shmem_set_pppdpid((pid_t) atoi(tok));

					syslog(LOG_INFO, "PPPD pid is %d", ((int) shmem_get_pppdpid()));
					
					fprintf(outsock, "PPPD Process Started\n");
					util_fflush(outsock);
				}
				else if (strstr(temp, modemconnected))
				{
					fprintf(outsock, "Modem connected\n");
					util_fflush(outsock);
				}
				else if (strstr(temp, "Serial connection established") != NULL)
				{
					fprintf (outsock, "Chat script successful\n");
					util_fflush(outsock);
				}
				else if (strstr(temp, pppup) != NULL)
				{
					fprintf (outsock, "PPP Link Established\n");
					fclose(logfile);
					return true;
				}
				else if (strstr(temp, "PAP authentication failed") != NULL)
				{
					fprintf (outsock, "PAP authentication failed");
				}
				else if (strstr(temp, "Host not responding to PAP authenticate requests") != NULL)
				{
					fprintf (outsock, "PAP authentication failed");
					mserver_kill_silent();
				}
				else if (strstr(temp, "Remote message: Login Succeeded"))
				{
					fprintf (outsock, "Authentication sucessful\n");
				}
				else if (strstr(temp, "Connect script failed") != NULL)
				{
					fprintf (outsock, "Connect script failed\n");
					fclose(logfile);
					return false;
				}
				else if (strstr(temp, "Modem hangup") != NULL)
				{
					fprintf (outsock, "Modem hungup\n");
					fclose(logfile);
					return false;
				}
				else if (strstr(temp, "Connection terminated") != NULL)
				{
					fprintf (outsock, "Connection terminated\n");
					fclose(logfile);
					return false;
				}
				else if (strstr(temp, "Exit.") != NULL)
				{
					fprintf (outsock, "PPPD Exited!\n");
					fclose(logfile);
					return false;
				}
			}
			else if (strstr(temp, chatstr) != NULL)
			{
				if (!strstr(temp, "abort on") && !strstr(temp, "expect"))
				{
					if (strstr(temp, "send"))
					{
						if (strstr(temp, "ATZ"))
						{
							fprintf(outsock, "Initializing Modem\n");
						}
						else if (strstr(temp, "ATDT"))
						{
							fprintf(outsock, "Dialing Modem\n");
						}
					}
					else if (strstr(temp, modemconnected))
					{
						fprintf(outsock, "Modem connected\n");
						if (speedsearch) speedcheck = true;
					}
					else if (strstr(temp, "assword:"))
					{
						fprintf(outsock, "Sending Password\n");
					}
					else if (strstr(temp, "ogin:"))
					{
						fprintf(outsock, "Sending Login\n");
					}
					else if (strstr(temp, "sername:"))
					{
						fprintf(outsock, "Sending Username\n");
					}
					else if (strstr(temp, "BUSY"))
					{	
						fprintf(outsock, "Phone Line Busy\n");
						
					}
					else if (strstr(temp, "NO CARRIER"))
					{
						fprintf(outsock, "No carrier\n");
					}
					else if (strstr(temp, "NO DIALTONE"))
					{
						fprintf(outsock, "No dialtone\n");
					}
					else if (strstr(temp, "Invalid Login"))
					{
						fprintf(outsock, "Invalid Login\n");
					}
					else if (strstr(temp, "Login incorrect"))
					{
						fprintf(outsock, "Login incorrect\n");
					}
					else if (strstr(temp, "Failed"))
					{
						fprintf(outsock, "Chat script failed!\n");
						fclose(logfile);
						return false;
					}
				}
	
				if (speedcheck)
				{
					if (get_connect_speed(temp))
					{
						speedsearch = false;
						fprintf(outsock, "Modem Connected at %d\n", cspeed);
					}
				}
			}
			util_fflush(outsock);
		}
	}

	fclose(logfile);
}

bool parse_wvdial(void)
{
	int sleep_sec;
	int log_timeout;

	char temp[1024];

	log_timeout = config_getvalue_int("logtimeout", 120);
	sleep_sec = 0;

	cspeed = config_getvalue_int("cspeed", 28800);

	while (true)
	{
		if (fgets(temp, 1024, wvlog) == NULL)
		{
			util_sleep (1);
			sleep_sec++;
			if (sleep_sec > log_timeout)
						{
				fprintf (outsock, "Timeout parsing wvlog\n");
				return false;
						}
					}
		else
		{
			if (!strncmp(temp, "--> ", 4))
			{
				if (strstr(temp, "Disconnecting") != NULL)
				{
					fprintf(outsock, "%s", temp+4);
					return false;
				}
				if (strstr(temp, "Waiting for carrier") != NULL)
				{
					fprintf(outsock, "Dialing Modem\n");
				}
				else if (strstr(temp, "Starting pppd") != NULL)
				{
					return true;
				}
				else if (strstr(temp, "Sending:") == NULL)
				{
					fprintf(outsock, "%s", temp+4);
					util_fflush(outsock);
				}
			}
			else
			{
				if (!strncmp(temp, "CONNECT ", 8))
					get_connect_speed(temp+8);
			}
		}
	}
}

/*
 * Function to parse the listen_on attribute in mserver.conf.    *
 * First parameter is the string to be parsed.  Second and third *
 * parameters are output arrays of IP addresses and ports.ut     *
 */
int parse_listenon(char * input, char iplist[MAXINTERFACES][1024],
		   char portlist[MAXINTERFACES][1024])
{
        char *listenon;
	int i, j, length;
	char thisport[1024];
	char thisip[1024];
	int num = 0;
	
	listenon = strtok(input, ",");
	do
	{
		length = strlen(listenon);
		for (i = 0; i < length; i++)
		{
			if (listenon[i] == ':')
			{
				thisip[i++] = '\0';	
				for (j = 0; i < length; i++, j++)
				{
				        thisport[j] = listenon[i];
				}
				thisport[j] = '\0';
			}
			else
			{
				thisip[i] = listenon[i];
			}
		}
		strncpy(iplist[num], thisip, 1024);
		strncpy(portlist[num], thisport, 1024);
		num++;
	}
	while( (listenon = strtok(NULL, ",")) != NULL);

	return num;
}
