%{
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sysexits.h>
#include <netinet/in.h>
#include <unistd.h>

#include "fragment.h"
#include "packets.h"

int yylex();
void yyerror(char *);
static struct pktnest *pktnest_new(char *type, int offset);
static void bpfinsn(struct bpf_insn *, u_int16_t, u_int8_t, u_int8_t, int32_t);
static struct fragment *addinsn(struct fragment *, u_int16_t, u_int8_t,
																u_int8_t, int32_t);
static struct fragment *make_loadfrag(int, int);

struct pktnest {
	char *type;
	int offset;
};

#define YYERROR_VERBOSE

struct fragment *topfrag; 

struct finfo {
	char *name;
	int len;
};

static void finfo_free(struct finfo *);
static struct finfo *finfo_make(char *, int);

struct base {
	char *name;
	int base;
} *bases = NULL;
int numbases = 0;
 
static int base_get(char *);
static void base_add(char *, int);
 
%}

%union {
	char *cp;
	long num;
	struct fields *fs;
	struct fragment *frag;
	struct pktnest *pn;
	struct finfo *fi;
}

%token T_PKT T_ACCEPT T_DENY T_IF T_BASE
%token T_LOR T_LAND
%token T_EQ T_GT T_GE T_LT T_LE T_NE
%token T_LSH T_RSH

%token <cp> T_ID
%token <num> T_NUM T_SIZE

%type <fs> fields
%type <frag> terminal directive conditional offset
%type <frag> avalue abit ashift aadd amul aatom
%type <frag> boolean orbool andbool comparison
%type <pn> pkttype pkt
%type <num> comp negcomp bitsize
%type <num> ivalue ibit ishift iadd imul iatom
%type <fi> field

%%

tops:		top
	|	tops top
	;

top:		packet
	|	base
	|	directive
	{
		frag_append(topfrag, $1);
	}
	;

base:		T_BASE T_ID bitsize ';'
	{
		base_add($2, $3);
	}
	;

packet:		T_PKT T_ID '{' fields '}' end
	{
		char * err;

		if((err = packet_add($2, $4)))
			yyerror(err);
		free($2);
	}
	;

fields:		field
	{
		char *err;

		$$ = fields_new();
		if((err = field_add($$, $1->name, $1->len)))
			yyerror(err);
		finfo_free($1);
	}
	|	fields field
	{
		char *err;

		if((err = field_add($1, $2->name, $2->len)))
			yyerror(err);
		finfo_free($2);
	}
	;

field:		T_SIZE T_ID ';'
	{
		$$ = finfo_make($2, $1);
	}
	|	T_SIZE T_ID ':' T_NUM end
	{
		$$ = finfo_make($2, $4);
	}
	;

end:
	{ /* nothing in particular */ }
	|	';'
	;

directive:	terminal
	|	conditional
	;

conditional:	T_IF '(' boolean ')' directive
	{
		frag_fixupt($3, 0);
		frag_fixupf($3, $5->numinsns);
		frag_append($3, $5);
		$$ = $3;
	}
	;

iatom:		T_NUM
	|	'-' iatom
	{	$$ = -$2;	}
	|	'(' ivalue ')'
	{	$$ = $2;	}
	;

imul:		iatom
	|	imul '*' iatom
	{	$$ = $1 * $3;	}
	|	imul '/' iatom
	{	$$ = $1 / $3;	}
	;

iadd:		imul
	|	iadd '+' imul
	{	$$ = $1 + $3;	}
	|	iadd '-' imul
	{	$$ = $1 - $3;	}
	;

ishift:		iadd
	|	ishift T_LSH iadd
	{	$$ = $1 << $3;	}
	|	ishift T_RSH iadd
	{	$$ = $1 >> $3;	}
	;

ibit:		ishift
	|	ibit '&' ishift
	{	$$ = $1 & $3;	}
	|	ibit '^' ishift
	{	$$ = $1 ^ $3;	}
	|	ibit '|' ishift
	{	$$ = $1 | $3;	}
	;

ivalue:		ibit
	;

aatom:		offset
	|	'(' avalue ')'
	{	$$ = $2;	}
	|	'-' aatom
	{	$$ = addinsn($2, BPF_ALU|BPF_NEG, 0, 0, 0);	}
	;

amul:		aatom
	|	amul '*' iatom
	{	$$ = addinsn($1, BPF_ALU|BPF_MUL|BPF_K, 0, 0, $3);	}
	|	amul '/' iatom
	{	$$ = addinsn($1, BPF_ALU|BPF_DIV|BPF_K, 0, 0, $3);	}
	;

aadd:		amul
	|	aadd '+' imul
	{	$$ = addinsn($1, BPF_ALU|BPF_ADD|BPF_K, 0, 0, $3);	}
	|	aadd '-' imul
	{	$$ = addinsn($1, BPF_ALU|BPF_SUB|BPF_K, 0, 0, $3);	}
	;


ashift:		aadd
	|	ashift T_LSH iadd
	{	$$ = addinsn($1, BPF_ALU|BPF_LSH|BPF_K, 0, 0, $3);	}
	|	ashift T_RSH iadd
	{	$$ = addinsn($1, BPF_ALU|BPF_RSH|BPF_K, 0, 0, $3);	}
	;

abit:		ashift
	|	abit '&' ishift
	{	$$ = addinsn($1, BPF_ALU|BPF_AND|BPF_K, 0, 0, $3);	}
	|	abit '|' ishift
	{	$$ = addinsn($1, BPF_ALU|BPF_OR|BPF_K, 0, 0, $3);	}
	;

avalue:		abit
	;

comparison:	avalue comp ivalue
	{
		$$ = addinsn($1, BPF_JMP|$2|BPF_K, 0, 0, $3);
		frag_sett($$);
		frag_setf($$);
	}
	|	avalue negcomp ivalue
	{
		$$ = addinsn($1, BPF_JMP|$2|BPF_K, 0, 0, $3);
		frag_setft($$);
		frag_settf($$);
	}
	|	'(' boolean ')'
	{	$$ = $2;	}
	;

andbool:	comparison
	|	andbool T_LAND comparison
	{
		frag_fixupt($1, 0);
		frag_append($1, $3);
		frag_free($3);
		$$ = $1;
	}
	;

orbool:		andbool
	|	orbool T_LOR andbool
	{
		frag_fixupf($1, 0);
		frag_append($1, $3);
		frag_free($3);
		$$ = $1;
	}
	;

boolean:	orbool
	;

comp:		T_EQ
	{	$$ = BPF_JEQ;	}
	|	T_GT
	{	$$ = BPF_JGT;	}
	|	T_GE
	{	$$ = BPF_JGE;	}
	;

negcomp:	T_NE
	{	$$ = BPF_JEQ;	}
	|	T_LT
	{	$$ = BPF_JGE;	}
	|	T_LE
	{	$$ = BPF_JGT;	}
	;

terminal:	T_ACCEPT ';'
	{
		$$ = addinsn(frag_new(), BPF_LD|BPF_W|BPF_LEN, 0, 0, 0);
		$$ = addinsn($$, BPF_RET|BPF_A, 0, 0, 0);
	}
	|	T_ACCEPT ivalue ';'
	{	$$ = addinsn(frag_new(), BPF_RET|BPF_K, 0, 0, $2);	}
	|	T_DENY ';'
	{	$$ = addinsn(frag_new(), BPF_RET|BPF_K, 0, 0, 0);	}
	;

offset:		pkttype '[' bitsize ',' bitsize ']'
	{
		$$ = make_loadfrag($1->offset + $3, $5);
		free($1->type);
		free($1);
	}
	|	pkttype '[' T_ID ']'
	{
		int offset, len;
			
		if(packet_offset($1->type, $3, &offset, &len))
			printf("No such field %s in %s\n", $3, $1->type);
		$$ = make_loadfrag($1->offset - packet_size($1->type) + offset, len);
		free($1->type);
		free($1);
	}
	;

pkttype:	pkttype '.' pkt
	{
		if($1->type)
			free($1->type);
		$1->type = $3->type;
		$3->type = NULL;
		$1->offset += $3->offset;
		free($3);
		$$ = $1;
	}
	|	pkt
	{
		$1->offset += base_get($1->type);
		$$ = $1;
	}
	;

pkt:		T_ID
	{
		int len;
		len = packet_size($1);
		if(len < 0)
			yyerror("Unknown packet type");
		$$ = pktnest_new($1, len);
	}
	;

bitsize:	ivalue
	{	$$ = $1 * 8;	}
	|	ivalue '.' ivalue
	{	$$ = $1 * 8 + $3;	}
	;

%%

void parse(int fd) {
	topfrag = frag_new();
	yyparse();
	write(fd, topfrag->insns, sizeof(*topfrag->insns) * topfrag->numinsns);
}

void yyerror(char *c) { fprintf(stderr, "%s\n", c); }

static struct pktnest *pktnest_new(char *type, int offset) {
	struct pktnest *tmp;

	tmp = malloc(sizeof(*tmp));
	tmp->type = type;
	tmp->offset = offset;
	return tmp;
}

static void bpfinsn(struct bpf_insn *i, u_int16_t code, u_int8_t jt,
										u_int8_t jf, int32_t k) {
	i->code = code;
	i->jt = jt;
	i->jf = jf;
	i->k = k;
}

static struct fragment *addinsn(struct fragment *f, u_int16_t code,
																u_int8_t jt, u_int8_t jf, int32_t k) {
		struct bpf_insn i;

		bpfinsn(&i, code, jt, jf, k);
		frag_add(f, &i);
		return f;
}

static struct finfo *finfo_make(char *name, int len) {
	struct finfo *f;

	f = malloc(sizeof(*f));
	f->name = name;
	f->len = len;
	return f;
}

static void finfo_free(struct finfo *f) {
	if(f->name)
		free(f->name);
	free(f);
}

static void base_add(char *name, int base) {
	int index = numbases;

	bases = realloc(bases, ++numbases * sizeof(*bases));
	bases[index].name = name;
	bases[index].base = base;
}

static int base_get(char *name) {
	int i, ret = 0;

	for(i = 0; i < numbases; i++)
		if(!strcmp(name, bases[i].name))
			ret = bases[i].base;
	return ret;
}

static struct fragment *make_loadfrag(int offset, int len) {
	struct fragment *f;
	int shift, mask, load;
	
	shift = (offset + len) & 7;
	mask = offset & 7;
	if((shift + len) <= 8) {
		load = BPF_B;
	}	else if((shift + len) <= 16) {
		load = BPF_H;
	} else if((shift + len) <= 32) {
		load = BPF_W;
		if((shift + len) <= 24)
			shift -= 8;
	} else {
		yyerror("Warning - cannot load data element spanning over four bytes");
	}
	f = addinsn(frag_new(), BPF_LD|load|BPF_ABS, 0, 0, offset >> 3);
	if(shift)
		addinsn(f, BPF_ALU|BPF_RSH|BPF_K, 0, 0, 8 - shift);
	if(mask)
		addinsn(f, BPF_ALU|BPF_AND|BPF_K, 0, 0, ~(~0 << len));
	return f;
}
