;;; liece-xemacs.el --- XEmacs specific routines.
;; Copyright (C) 1998, 1999 Daiki Ueno

;; Author: Daiki Ueno <ueno@ueda.info.waseda.ac.jp>
;; Created: 1998-09-28
;; Revised: 1999-08-22
;; Keywords: emulation

;; This file is part of Liece.

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Code:

(eval-when-compile 
  (require 'liece-inlines)
  (require 'liece-crypt)
  (require 'liece-commands))

(autoload 'liece-command-dcc-send "liece-dcc")
(defvar liece-nick-popup-menu)

(defmacro liece-xemacs-icon-path (file)
  `(or (and liece-icon-directory
	    (expand-file-name ,file liece-icon-directory))
       (let ((path (liece-find-path ,file "icons")))
	 (when path 
	   (setq liece-icon-directory
		 (file-name-directory path)))
	 path)))

(define-widget 'liece-toolbar-icon 'list
  "Edit toolbar spec entries"
  :match (lambda (widget value)
	   (valid-plist-p value))
  :convert-widget 'liece-toolbar-icon-convert)

(defun liece-toolbar-icon-convert (widget)
  (let* ((up '(const :format "" :value :up))
	 (down '(const :format "" :value :down))
	 (disabled '(const :format "" :value :disabled))
	 (cap-up '(const :format "" :value :cap-up))
	 (cap-down '(const :format "" :value :cap-down))
	 (cap-disabled '(const :format "" :value :cap-disabled))
	 (none '(const :tag "none" nil))
	 (up 
	  `(group :inline t :format "%t: %v" :tag "Up"
		  ,up file))
	 (down 
	  `(group :inline t :format "%t: %v" :tag "Down"
		  ,down (radio ,none file)))
	 (disabled 
	  `(group :inline t :format "%t: %v" :tag "Disabled"
		  ,disabled (radio ,none file)))
	 (cap-up 
	  `(group :inline t  :format "%t: %v" :tag "Captured Up"
		  ,cap-up (radio ,none file)))
	 (cap-down 
	  `(group :inline t :format "%t: %v" :tag "Captured Down"
		  ,cap-down (radio ,none file)))
	 (cap-disabled 
	  `(group :inline t :format "%t: %v" :tag "Captured Disabled"
		  ,cap-disabled (radio ,none file)))
	 (args `(,up ,down ,disabled ,cap-up ,cap-down ,cap-disabled)))
    (widget-put widget :args args)
    widget))

(defgroup liece-toolbar nil
  "Toolbar of your XEmacs"
  :tag "Toolbar"
  :group 'liece)

(defgroup liece-toolbar-icons nil
  "Toolbar Icons of your XEmacs"
  :tag "Toolbar Icons"  
  :prefix "liece-toolbar-"
  :group 'liece)

(defcustom liece-use-toolbar (if (featurep 'toolbar)
				  'default-toolbar
				nil)
  "*If nil, do not use a toolbar.
If it is non-nil, it must be a toolbar.  The five valid values are
`default-toolbar', `top-toolbar', `bottom-toolbar',
`right-toolbar', and `left-toolbar'."
  :type '(choice (const default-toolbar)
                 (const top-toolbar) (const bottom-toolbar)
                 (const left-toolbar) (const right-toolbar)
                 (const :tag "no toolbar" nil))
  :group 'liece-toolbar)

(defcustom liece-toolbar-back-icon '(:up "back.xpm")
  "Back button"
  :type 'liece-toolbar-icon
  :group 'liece-toolbar-icons)

(defcustom liece-toolbar-forward-icon '(:up "forward.xpm")
  "Forward button"
  :type 'liece-toolbar-icon
  :group 'liece-toolbar-icons)

(defcustom liece-toolbar-reload-icon '(:up "reload.xpm")
  "Reload button"
  :type 'liece-toolbar-icon
  :group 'liece-toolbar-icons)

(defcustom liece-toolbar-home-icon '(:up "home.xpm")
  "Home button"
  :type 'liece-toolbar-icon
  :group 'liece-toolbar-icons)

(defcustom liece-toolbar-search-icon '(:up "search.xpm")
  "Search button"
  :type 'liece-toolbar-icon
  :group 'liece-toolbar-icons)

(defcustom liece-toolbar-location-icon '(:up "location.xpm")
  "Location button"
  :type 'liece-toolbar-icon
  :group 'liece-toolbar-icons)

(defcustom liece-toolbar-crypt-active-icon '(:up "encrypt.xpm")
  "Crypt button (active)"
  :type 'liece-toolbar-icon
  :group 'liece-toolbar-icons)

(defcustom liece-toolbar-crypt-inactive-icon '(:up "crypt.xpm")
  "Crypt button (inactive)"
  :type 'liece-toolbar-icon
  :group 'liece-toolbar-icons)

(defcustom liece-toolbar-crypt-icon 
  liece-toolbar-crypt-inactive-icon
  "Crypt button"
  :type 'liece-toolbar-icon
  :group 'liece-toolbar-icons)

(defcustom liece-toolbar-stop-icon '(:up "stop.xpm")
  "Stop button"
  :type 'liece-toolbar-icon
  :group 'liece-toolbar-icons)

;;; @ internal variables
;;; 
(defvar liece-glyph-cache nil)
(defvar liece-toolbar-position 'top)

(defvar liece-toolbar-back-glyph nil)
(defvar liece-toolbar-forward-glyph nil)
(defvar liece-toolbar-reload-glyph nil)
(defvar liece-toolbar-home-glyph nil)
(defvar liece-toolbar-search-glyph nil)
(defvar liece-toolbar-location-glyph nil)
(defvar liece-toolbar-crypt-glyph nil)
(defvar liece-toolbar-crypt-active-glyph nil)
(defvar liece-toolbar-crypt-inactive-glyph nil)
(defvar liece-toolbar-stop-glyph nil)

(defvar liece-toolbar-spec-list 
  '([liece-toolbar-back-glyph
     liece-command-previous-channel t "Previous Channel"]
    [liece-toolbar-forward-glyph
     liece-command-next-channel t "Next Channel"]
    [liece-toolbar-reload-glyph
     liece-command-list t "List Channel"]
    [liece-toolbar-home-glyph
     liece-switch-to-channel-no-1 t "Go Home Channel"]
    [liece-toolbar-search-glyph
     liece-command-finger t "Finger"]
    [liece-toolbar-location-glyph
     liece-command-join t "Join Channel"]
    [liece-toolbar-crypt-glyph 
     liece-toolbar-toggle-crypt t "Toggle Crypt Mode"]
    [liece-toolbar-stop-glyph
     liece-command-quit t "Quit IRC"]))

;;; @ toolbar icons
;;; 
(defun liece-toolbar-icon-plist-get (spec prop)
  (let ((icon (plist-get spec prop)))
    (if icon (liece-xemacs-icon-path icon))))

(defun liece-toolbar-map-button-list (plist)
  (apply #'toolbar-make-button-list
	 (mapcar
	  (lambda (prop)
	    (liece-toolbar-icon-plist-get plist prop))
	  '(:up :down :disabled :cap-up :cap-down :cap-disabled))))

(defun liece-xemacs-setup-toolbar (bar &optional force)
  (let ((dir (liece-xemacs-icon-path "")) icon plist)
    (set-default-toolbar-position 
     (or liece-toolbar-position default-toolbar-position))
    (when (and dir (file-directory-p dir))
      (dolist (spec bar)
	(setq icon (aref spec 0)
	      plist (symbol-value
		     (intern (concat 
			      (substring (prin1-to-string icon) -5 0) 
			      "icon"))))
	(when (or force (not (symbol-value icon)))
	  (set icon (liece-toolbar-map-button-list plist))))
      (run-hooks 'liece-xemacs-setup-toolbar-hook)
      t)))

(add-hook 'liece-xemacs-setup-toolbar-hook 'liece-toolbar-setup-crypt-glyph)

(defun liece-toolbar-setup-crypt-glyph ()
  (setq liece-toolbar-crypt-active-glyph
	(liece-toolbar-map-button-list liece-toolbar-crypt-active-icon)
	liece-toolbar-crypt-inactive-glyph
	(liece-toolbar-map-button-list liece-toolbar-crypt-inactive-icon)))

(defun liece-toolbar-toggle-crypt ()
  (interactive)
  (liece-command-toggle-crypt)
  (setq liece-toolbar-crypt-glyph 
	(if liece-crypt-mode-active
	    liece-toolbar-crypt-active-glyph
	  liece-toolbar-crypt-inactive-glyph))
  (and liece-use-toolbar
       (set-specifier (symbol-value liece-use-toolbar)
		      (cons (current-buffer) liece-toolbar-spec-list))))

;;; @ modeline decoration
;;; 
(defun liece-xemacs-hide-modeline ()
  (set-specifier has-modeline-p (cons (current-buffer) nil)))

(when (featurep 'scrollbar)
  (defun liece-xemacs-hide-scrollbars ()
    (static-cond
     ((boundp 'horizontal-scrollbar-visible-p)
      (set-specifier horizontal-scrollbar-visible-p nil
		     (current-buffer)))
     ((boundp 'scrollbar-height)
      (set-specifier scrollbar-height (cons (current-buffer) 0)))))
  (add-hook 'liece-nick-mode-hook 'liece-xemacs-hide-scrollbars)
  (add-hook 'liece-channel-list-mode-hook 'liece-xemacs-hide-scrollbars))

(add-hook 'liece-nick-mode-hook 'liece-xemacs-hide-modeline)
(add-hook 'liece-channel-list-mode-hook 'liece-xemacs-hide-modeline)

(defvar liece-xemacs-modeline-left-extent
  (let ((ext (copy-extent modeline-buffer-id-left-extent)))
    ext))

(defvar liece-xemacs-modeline-right-extent
  (let ((ext (copy-extent modeline-buffer-id-right-extent)))
    ext))

(add-hook 'liece-command-mode-hook 'liece-setup-toolbar)

(defun liece-setup-toolbar ()
  (and liece-use-toolbar
       (liece-xemacs-setup-toolbar liece-toolbar-spec-list)
       (set-specifier (symbol-value liece-use-toolbar)
		      (cons (current-buffer) liece-toolbar-spec-list))))

(defun liece-xemacs-modeline-glyph ()
  (let* ((file-xpm 
	  (liece-xemacs-icon-path "liece-pointer.xpm"))
	 (file-xbm 
	  (liece-xemacs-icon-path "liece-pointer.xbm"))
	 (glyph (make-glyph
		 (if (memq (console-type) '(x mswindows))
		     (static-cond 
		      ((featurep 'xpm)
		       `[xpm :file ,file-xpm])
		      ((featurep 'xbm)
		       `[xbm :file ,file-xbm])
		      (t [string :data "Liece:"]))
		   [string :data "Liece:"]))))
    (set-glyph-face glyph 'modeline-buffer-id)
    glyph))

(defun liece-xemacs-mode-line-buffer-identification (line)
  (let ((line (car line)) chop)
    (cond
     ((not (stringp line))
      (list line))
     ((not (string-match "^Liece:" line))
      (list line))
     (t
      (setq chop (match-end 0))
      (list
       (let ((glyph (liece-xemacs-modeline-glyph)))
	 (if glyph
	     (cons liece-xemacs-modeline-left-extent glyph)
	   (cons liece-xemacs-modeline-left-extent (substring line 0 chop))))
       (cons liece-xemacs-modeline-right-extent (substring line chop)))))))

;;; @ menus
;;; 
(defun liece-xemacs-nick-popup-menu (widget &optional event)
  (let ((pos (widget-event-point event)))
    (when pos
      (goto-char pos)
      (liece-nick-update-region)
      (let ((menu (cdr liece-nick-popup-menu)))
	(setq menu (nconc (list "IRCHAT" ; title: not displayed
				"     IRC commands"
				"--:shadowDoubleEtchedOut")
			  (mapcar (lambda (spec) 
				    (if (stringp spec) 
					"--:shadowEtchedOut"
				      spec))
				  menu)))
	(let (popup-menu-titles)
	  (popup-menu menu))))))

(fset 'liece-nick-popup-menu 'liece-xemacs-nick-popup-menu)

;;; @ nick buffer decoration
;;; 
(defun liece-xemacs-create-nick-glyph (file &optional str)
  (setq str (or str ""))
  (or 
   (cdr-safe (assoc file liece-glyph-cache))
   (let* ((file-xpm (liece-xemacs-icon-path file))
	  (glyph (make-glyph
		  (if (memq (console-type) '(x mswindows))
		      (static-if (featurep 'xpm) 
			  `[xpm :file ,file-xpm]
			`[string :data ,str])
		    `[string :data ,str]))))
     (when glyph
       (push (cons file glyph) liece-glyph-cache)
       (set-glyph-face glyph 'default))
     glyph)))

(defun liece-xemacs-glyph-nick-region (start end)
  (save-excursion
    (goto-char start)
    (beginning-of-line)
    (setq start (point))

    (goto-char end)
    (beginning-of-line 2)
    (setq end (point))
      
    (save-restriction
      (narrow-to-region start end)
      (let ((buffer-read-only nil)
	    (inhibit-read-only t)
	    (case-fold-search nil)
	    mark file glyph ext ant)
	(map-extents
	 (lambda (e void)
	   (when (or 
		  (extent-property 
		   e 'liece-xemacs-glyph-nick-extent)
		  (extent-property 
		   e 'liece-xemacs-glyph-nick-annotation))
	     (delete-extent e)))
	 (current-buffer) start end)
	(dolist (entry liece-nick-image-alist)
	  (setq mark (car entry)
		file (cdr entry)
		glyph (liece-xemacs-create-nick-glyph 
		       file (char-to-string mark)))
	  (when glyph
	    (goto-char start)
	    (while (not (eobp))
	      (when (eq (char-after) mark)
		(mapcar 'delete-annotation 
			(annotations-at (1+ (point))))
		(setq ext (make-extent (point) (1+ (point)))
		      ant (make-annotation glyph (1+ (point)) 'text))
		(set-extent-property ext 'end-open t)
		(set-extent-property ext 'start-open t)
		(set-extent-property ext 'invisible t)
		(set-extent-property ext 'intangible t)
		(set-extent-property 
		 ant 'liece-xemacs-glyph-nick-extent ext)
		(set-extent-property 
		 ext 'liece-xemacs-glyph-nick-annotation ant)
		)
	      (beginning-of-line 2)
	      )))
	))))

(defun liece-xemacs-set-drop-functions (start end)
  "Initialize drag and drop DCC:
This function needs window system independent drag and drop 
support (21.0 b39 or later)"
  (interactive "r")
  (liece-xemacs-set-drop-functions-buffer 
   (current-buffer) start end)
  (goto-char end))

(defun liece-xemacs-set-drop-functions-buffer (&optional buffer start end)
  "Initialize drag and drop DCC:
This function needs window system independent drag and drop 
support (21.0 b39 or later)"
  (interactive)
  (when (and (featurep 'x) (featurep 'dragdrop))
    (save-excursion
      (when buffer
	(set-buffer buffer))
      (setq start (or start (point-min))
	    end (or end (point-max)))
      (goto-char start)
      (setq start (line-beginning-position))
      (goto-char end)
      (setq end (line-beginning-position))
      (goto-char end)
      (when (not (eobp))
	(beginning-of-line 2)
	(setq end (point)))
      (save-restriction
	(narrow-to-region start end)
	(let (buffer-read-only case-fold-search)
	  (map-extents
	   (function 
	    (lambda (e void)
	      (when (extent-property e 'liece-xemacs-drop-extent)
		(delete-extent e))))
	   buffer start end)
	  (goto-char start)
	  (let (st nd nick func)
	    (while (not (eobp))
	      (forward-char)
	      (setq st (point)
		    nd (line-end-position)
		    nick (buffer-substring st nd))
	      (mapcar 'delete-annotation (annotations-at nd))
	      (setq func (intern (concat "liece-xemacs-drop-function-" nick)))
	      (fset func 
		    (list 'lambda (list 'object)
			  (list 'liece-xemacs-drop-function 'object nick)))
	      (let ((ext (make-extent st nd)))
		(set-extent-property ext 'liece-xemacs-drop-extent t)
		(set-extent-property ext 'dragdrop-drop-functions (list func)))
	      (beginning-of-line 2))))
	))))

(defun liece-xemacs-drop-function (object nick)
  (if (and (eq (car object) 'dragdrop_URL)
	   (stringp (cdr object))
	   (string-match "^[^:]*:\\(.*\\)" (cdr object)))
      (let ((filename (match-string 1 (cdr object))))
	(liece-command-dcc-send filename nick))))

(defadvice easy-menu-add-item
  (around liece-fix-menu-path-switch-buffer activate)
  (save-excursion
    (set-buffer liece-command-buffer)
    (add-menu-button 
     (cons (car (ad-get-arg 0)) (ad-get-arg 1))
     (ad-get-arg 2) (ad-get-arg 3))))

(eval-and-compile
  (setq liece-x-face-insert-function 
	(function liece-x-face-insert-with-xemacs))

  (defun liece-x-face-insert-with-xemacs (buffer str nick)
    (save-excursion
      (let ((glyph (cdr-safe (assoc nick liece-glyph-cache))))
	(unless glyph
	  (setq glyph (make-glyph 
		       (cond
			((and (featurep 'xface)
			      (memq (console-type) '(x mswindows)))
			 `[xface :data ,str])
			(t `[string :data ,str]))))
	  (when glyph
	    (push (cons nick glyph) liece-glyph-cache)
	    (set-glyph-face glyph 'default)))
	(set-buffer buffer)
	(goto-char (point-max))
	(when glyph
	  (set-extent-end-glyph (make-extent (point) (point)) glyph)))))
  )

;;; @ startup splash 
;;; 
(eval-when-compile
  (defvar filename)
  (setq load-path 
	`(,(if (and (boundp 'filename)
		    (stringp filename)
		    (file-exists-p filename))
	       (file-name-directory filename)
	     default-directory)
	  ,@load-path)))

(when (featurep 'xpm)
  (eval-when-compile
    (defmacro liece-xemacs-logo ()
      (let ((logo "liece.xpm")
	    (dir (if (and (boundp 'filename)
			  (stringp filename)
			  (file-exists-p filename))
		     (file-name-directory filename)
		   default-directory)))
	(setq logo (expand-file-name logo dir))
	(if (file-exists-p logo)
	    (let ((buffer (generate-new-buffer " *liece-logo*"))
		  (coding-system-for-read (quote binary))
		  buffer-file-format format-alist
		  insert-file-contents-post-hook
		  insert-file-contents-pre-hook)
	      (prog1
		  (save-excursion
		    (set-buffer buffer)    
		    (insert-file-contents logo)
		    (buffer-string))
		(kill-buffer buffer)))
	  (progn
	    (byte-compile-warn
	     "Warning: file \"%s\" not found." logo)
	    (sit-for 2)
	    nil))))
    ))

(defconst liece-xemacs-logo
  (when (featurep 'xpm)
    (liece-xemacs-logo)))

(defun liece-xemacs-splash-at-point (&optional height)
  (or (bolp) (insert "\n"))
  (let ((bow (point))
	(glyph (make-glyph `[xpm :data ,liece-xemacs-logo]))
	(lh (/ (window-pixel-height) (window-height)))
	(lw (/ (window-pixel-width) (window-width)))
        bov)
    
    (insert-char ?\n (max 0 (/ (- (or height (window-height)) 
				  (/ (glyph-height glyph) lh))
			       2)))
    (insert-char ?\  (max 0 (/ (- (window-width) 
				  (/ (glyph-width glyph) lw))
			       2)))
    (when (and (featurep 'xpm) (memq (console-type) '(x mswindows)))
      (set-extent-end-glyph
       (make-extent (point) (point))
       glyph))
    (insert "\n")
    (insert-char ?\  (max 0 (/ (- (window-width) (length liece-version)) 2)))
    (setq bov (point))
    (insert liece-version)
    (and (find-face 'bold-italic)
         (put-text-property bov (point) 'face 'bold-italic))
    (goto-char bow)
    (set-window-start (get-buffer-window (current-buffer)) (point))
    (redisplay-frame)))

(defun liece-xemacs-splash (&optional arg)
  (interactive "P")
  (and liece-xemacs-logo
       (let ((frame (selected-frame))
             config buffer)
         (and frame
              (unwind-protect
                  (progn
                    (setq config (current-window-configuration))
                    (switch-to-buffer
                     (setq buffer (generate-new-buffer
                                   (concat (if arg "*" " *")
                                           liece-version "*"))))
                    (delete-other-windows)
                    (liece-xemacs-splash-at-point)
                    (set-buffer-modified-p nil)
                    (or arg (sleep-for 2)))
                (unless arg
                  (kill-buffer buffer)
                  (set-window-configuration config)
                  (redisplay-frame frame)))))))

(or (eq 'stream (device-type))
    (liece-xemacs-splash))


;;; @ emulation functions
;;; 
(defun liece-xemacs-add-text-properties (start end props &optional object)
  (add-text-properties start end props object)
  (put-text-property start end 'start-closed nil object))

(defun liece-xemacs-put-text-property (start end prop value &optional object)
  (put-text-property start end prop value object)
  (put-text-property start end 'start-closed nil object))

(defun liece-xemacs-move-overlay (extent start end &optional buffer)
  (set-extent-endpoints extent start end buffer))

(defun liece-xemacs-kill-all-overlays ()
  "Delete all extents in the current buffer."
  (map-extents 
   (function
    (lambda (extent ignore)
      (delete-extent extent)
      nil))))

(defun liece-xemacs-overlays-at (pos)
  (let ((ext (extent-at pos)))
    (if ext (list ext))))
      
(defmacro liece-xemacs-get-buffer-window-list (buf)
  `(windows-of-buffer ,buf))

(fset 'liece-mode-line-buffer-identification 
      'liece-xemacs-mode-line-buffer-identification)
(fset 'liece-get-buffer-window-list
      'liece-xemacs-get-buffer-window-list)

(fset 'liece-make-overlay 'make-extent)
(fset 'liece-delete-overlay 'delete-extent)
(fset 'liece-overlay-put 'set-extent-property)
(fset 'liece-move-overlay 'liece-xemacs-move-overlay)
(fset 'liece-overlay-end 'extent-end-position)
(fset 'liece-overlay-get 'extent-property)
(fset 'liece-overlays-at 'liece-xemacs-overlays-at)
(fset 'liece-kill-all-overlays 'liece-xemacs-kill-all-overlays)
(fset 'liece-put-text-property 'liece-xemacs-put-text-property)
(fset 'liece-add-text-property 'liece-xemacs-add-text-property)

(defun liece-xemacs-run-at-time (time repeat function &rest args)
  (start-itimer
   "liece-run-at-time"
   `(lambda ()
      (,function ,@args))
   time repeat))

(fset 'liece-run-at-time 'liece-xemacs-run-at-time)
(fset 'liece-cancel-timer 'delete-itimer)

(add-hook 'liece-nick-insert-hook 'liece-xemacs-glyph-nick-region)
(add-hook 'liece-nick-insert-hook 'liece-xemacs-set-drop-functions)

(add-hook 'liece-nick-replace-hook 'liece-xemacs-glyph-nick-region)
(add-hook 'liece-nick-replace-hook 'liece-xemacs-set-drop-functions)

(defadvice liece-find-etc-directory 
  (around liece-locate-data-directory activate)
  (or (locate-data-directory (ad-get-arg 0))
      ad-do-it))
  
(provide 'liece-xemacs)

;;; liece-xemacs.el ends here

