;;; liece-minibuf.el --- Minibuffer custom completion.
;; Copyright (C) 1998, 1999 Daiki Ueno

;; Author: Daiki Ueno <ueno@ueda.info.waseda.ac.jp>
;; Created: 1999-02-02
;; Revised: 1999-02-02
;; Keywords: minibuffer, completion

;; This file is part of Liece.

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Code:

(require 'liece-compat)
(require 'liece-intl)
(require 'liece-nick)

(defvar liece-minibuffer-map nil)
(defvar liece-minibuffer-complete-function nil)

(unless liece-minibuffer-map
  (setq liece-minibuffer-map
	(let ((map (make-sparse-keymap)))
	  (liece-set-keymap-parents map (list minibuffer-local-map))
	  (define-key map " " nil)
	  (define-key map "\t" 'liece-minibuffer-complete)
	  (define-key map "\r" 'exit-minibuffer)
	  (define-key map "\n" 'exit-minibuffer)
	  map)))

(defun liece-minibuffer-complete ()
  (interactive)
  (if (and liece-minibuffer-complete-function 
	   (fboundp liece-minibuffer-complete-function))
      (funcall liece-minibuffer-complete-function)))

(defun liece-minibuffer-parse-modes ()
  (save-excursion
    (let (preceding-char (state 'flag) type)
      (beginning-of-buffer)
      (while (not (eobp))
	(forward-char)
	(setq preceding-char (char-before))
	(cond
	 ((and (memq state '(flag arg))
	       (or (char-equal preceding-char ?+) 
		   (char-equal preceding-char ?-)))
	  (setq state 'mode
		type nil))
	 ((and (eq state 'mode) (char-equal preceding-char ? ))
	  (setq state 'arg))
	 ((and (eq state 'mode) (memq preceding-char '(?o ?v)))
	  (setq type (nconc type (list 'nick preceding-char
				       (char-before (1- (point))))))
	  )
	 ((and (eq state 'mode) (eq preceding-char ?b))
	  (setq type (nconc type (list 'ban (char-before (1- (point))))))
	  )))
      (cons state type))))
	
(defun liece-minibuffer-prepare-candidate ()
  (let ((point (point)))
    (skip-syntax-backward "^ ")
    (prog1 (buffer-substring (point) point)
      (goto-char point))))

(defun liece-minibuffer-delete-candidate ()
  (let ((point (point)))
    (skip-syntax-backward "^ ")
    (delete-region (point) point)))

(defun liece-minibuffer-finalize-completion (completion pattern all)
  (cond 
   ((eq completion t))
   ((null completion)
    (temp-minibuffer-message (_ "[No match]")))
   ((not (string= pattern completion))
    (liece-minibuffer-delete-candidate)
    (insert completion))
   (t
    (with-output-to-temp-buffer "*Completions*"
      (funcall completion-display-completion-list-function
	       (sort all (function (lambda (x y)
				     (string-lessp
				      (or (car-safe x) x)
				      (or (car-safe y) y))))))
      ))))

(defun liece-minibuffer-complete-channel-modes ()
  (let* ((preceding-char (char-before)) completion candidate all
	 (modes (mapconcat 
		 (function car) 
		 liece-supported-channel-mode-alist ""))
	 (nicks (liece-channel-get-nicks))
	 uahs
	 (context (liece-minibuffer-parse-modes)) 
	 (state (car context)) (type (cdr context)))
    (cond 
     ((memq state '(flag mode))
      (temp-minibuffer-message
       (format (_ "[Modes are: %s]") modes)))
     ((and (eq state 'arg) (memq 'ban type))
      (if (memq ?- type)
	  (setq uahs (list-to-alist (liece-channel-get-bans)))
	(setq uahs (mapcar 
		    (function
		     (lambda (nick) 
		       (list (concat nick "!" 
				     (liece-nick-get-user-at-host nick)))
		       ))
		    nicks)))
      (setq candidate (liece-minibuffer-prepare-candidate)
	    completion (try-completion candidate uahs)
	    all (all-completions candidate uahs)))
     ((and (eq state 'arg) (memq 'nick type))
      (let* ((masks (cond ((memq ?o type) (liece-channel-get-operators))
			  ((memq ?v type) (liece-channel-get-voices))))
	     (nicks
	      (if (memq ?- type)
		  masks
		(remove-if
		 (` (lambda (item) 
		      (and (stringp item) 
			   (string-list-member-ignore-case item '(, masks)))
		      ))
		 nicks))))
	(setq nicks (mapcar (function list) nicks)
	      candidate (liece-minibuffer-prepare-candidate)
	      completion (try-completion candidate nicks)
	      all (all-completions candidate nicks)))))
    (liece-minibuffer-finalize-completion completion candidate all)))

(defun liece-minibuffer-complete-user-modes ()
  (temp-minibuffer-message
   (format 
    (_ "[Modes are: %s]") 
    (mapconcat (function car) liece-supported-user-mode-alist ""))))

(defun liece-minibuffer-completing-default-read 
  (prompt table &optional predicate require-match initial-input)
  "completing-read w/ default argument like in 'kill-buffer'"
  (let ((default-read
	  (completing-read
	   (if initial-input
	       (format "%s(default %s) " prompt initial-input)
	     prompt)
	   table predicate require-match nil)))
    (if (and (string= default-read "") initial-input)
	initial-input
      default-read)))

(defun liece-minibuffer-completing-sequential-read 
  (prompt &optional count table predicate require-match multiple-candidate)
  "Execute completing-read w/ default argument consequently"
  (let ((count (or count 0)) string result)
    (while (progn 
	     (setq string
		   (completing-read
		    (format "%s (%d): " prompt (incf count))
		    table predicate require-match nil))
	     (or multiple-candidate
		 (remove-alist 'table string))
	     (not (string= "" string)))
      (push string result))
    result))

(provide 'liece-minibuf)

;;; liece-minibuf.el ends here.
