/*
 settings.c : Irssi settings

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include <default-config.h>

OPTCATEGORY_REC option_categories[] =
{
    { "misc", misc_options },
    { "history", history_options },
    { "flood", flood_options },
    { "completion", completion_options },
    { "lookandfeel", lookandfeel_options },
    { "ircproxy", ircproxy_options },
    { "dcc", dcc_options },
    { "appearance", appearance_options },
    { "server", server_options }
};

static GHashTable *settings;

void setup_themes_init(void);
void setup_themes_deinit(void);

void setup_servers_init(void);
void setup_servers_deinit(void);

void setup_ircnets_init(void);
void setup_ircnets_deinit(void);

void setup_channels_init(void);
void setup_channels_deinit(void);

void setup_lists_init(void);
void setup_lists_deinit(void);

void setup_keyboard_init(void);
void setup_keyboard_deinit(void);

void init_userinfo(void);

/* misc */
static gchar *url_www_client;
static gchar *url_ftp_client;
static gchar *url_mail_client;

static gchar *ctcp_version_reply;
static gchar *default_quit_message;
static gchar *default_user_mode;

static gint min_lag_check_time;
static gint max_lag_before_disconnect;

static gint knockout_time; /* How many seconds to keep /knockouted ban */

static gchar *awaylog_file;
static gchar *awaylog_level;

SETUP_REC misc_options[] =
{
    { SETUP_TYPE_ENTRY, &url_www_client, "url_www_client", "gnome-moz-remote --newwin %s" },
    { SETUP_TYPE_ENTRY, &url_ftp_client, "url_ftp_client", "Eterm -e ncftp %s" },
    { SETUP_TYPE_ENTRY, &url_mail_client, "url_mail_client", "Eterm -e pine %s" },

    { SETUP_TYPE_ENTRY, &ctcp_version_reply, "ctcp_version_reply", PACKAGE" v%s - running on %s %s" },
    { SETUP_TYPE_ENTRY, &default_quit_message, "default_quit_message", "leaving" },
    { SETUP_TYPE_ENTRY, &default_user_mode, "default_user_mode", "+iw" },

    { SETUP_TYPE_SPIN, &min_lag_check_time, "min_lag_check_time", GINT_TO_POINTER(10) },
    { SETUP_TYPE_SPIN, &max_lag_before_disconnect, "max_lag_before_disconnect", GINT_TO_POINTER(90) },

    { SETUP_TYPE_SPIN, &knockout_time, "knockout_time", GINT_TO_POINTER(300) },

    { SETUP_TYPE_ENTRY, &awaylog_file, "awaylog_file", "~/.irssi/away.log" },
    { SETUP_TYPE_ENTRY, &awaylog_level, "awaylog_level", "-all +msgs +hilight" },

    { 0, NULL, NULL }
};

/* history */
static gint max_command_history;
static gint max_textwidget_lines;
static gint rawlog_lines;
static gint block_remove_lines;

static gboolean toggle_window_history;

SETUP_REC history_options[] =
{
    { SETUP_TYPE_SPIN, &max_textwidget_lines, "max_textwidget_lines", GINT_TO_POINTER(1000) },
    { SETUP_TYPE_SPIN, &block_remove_lines, "block_remove_lines", GINT_TO_POINTER(20) },

    { SETUP_TYPE_SPIN, &max_command_history, "max_command_history", GINT_TO_POINTER(100) },
    { SETUP_TYPE_SPIN, &rawlog_lines, "rawlog_lines", GINT_TO_POINTER(200) },

    { SETUP_TYPE_TOGGLEBUTTON, &toggle_window_history, "toggle_window_history", GINT_TO_POINTER(FALSE) },

    { 0, NULL, NULL }
};

/* nick completion */
static gchar *completion_char;
static gboolean completion_disable_auto;
static gint completion_keep_publics;
static gint completion_keep_ownpublics;
static gint completion_keep_privates;

SETUP_REC completion_options[] =
{
    { SETUP_TYPE_ENTRY, &completion_char, "completion_char", ":" },
    { SETUP_TYPE_TOGGLEBUTTON, &completion_disable_auto, "completion_disable_auto", GINT_TO_POINTER(FALSE) },
    { SETUP_TYPE_SPIN, &completion_keep_publics, "completion_keep_publics", GINT_TO_POINTER(30) },
    { SETUP_TYPE_SPIN, &completion_keep_ownpublics, "completion_keep_ownpublics", GINT_TO_POINTER(60) },
    { SETUP_TYPE_SPIN, &completion_keep_privates, "completion_keep_privates", GINT_TO_POINTER(10) },

    { 0, NULL, NULL }
};

/* flood protection */
static gint flood_timecheck; /* Flood check timeout */
static gint flood_max_msgs; /* Max msgs in FLOOD_TIMECHECK msecs before considered as flooding */
static gint autoignore_time; /* How many seconds to keep someone autoignored */
static gint ctcp_timecheck; /* CTCP reply send timeout */
static gint max_ctcp_queue; /* Max CTCP reply queue length */
static gint cmd_queue_speed; /* Minimum timeout before sending the next command to server */

SETUP_REC flood_options[] =
{
    { SETUP_TYPE_SPIN, &flood_timecheck, "flood_timecheck", GINT_TO_POINTER(1000) },
    { SETUP_TYPE_SPIN, &flood_max_msgs, "flood_max_msgs", GINT_TO_POINTER(5) },
    { SETUP_TYPE_SPIN, &autoignore_time, "autoignore_time", GINT_TO_POINTER(300) },
    { SETUP_TYPE_SPIN, &ctcp_timecheck, "ctcp_timecheck", GINT_TO_POINTER(2000) },
    { SETUP_TYPE_SPIN, &max_ctcp_queue, "max_ctcp_queue", GINT_TO_POINTER(5) },
    { SETUP_TYPE_SPIN, &cmd_queue_speed, "cmd_queue_speed", GINT_TO_POINTER(2000) },

    { 0, NULL, NULL }
};

/* look and feel */
static gboolean toggle_show_menubar;
static gboolean toggle_show_toolbar;
static gboolean toggle_show_statusbar;
static gboolean toggle_show_nicklist;
static gboolean toggle_show_timestamps;
static gboolean toggle_show_msgs_timestamps;
static gboolean toggle_hide_text_style;
static gboolean toggle_bell_beeps;
static gboolean toggle_actlist_moves;
static gboolean toggle_privmsg_beeps;

static gboolean toggle_use_status_window;
static gboolean toggle_use_msgs_window;
static gboolean toggle_autoraise_msgs_window;
static gboolean toggle_autocreate_query;
static gboolean toggle_notifylist_popups;
static gboolean toggle_use_tabbed_windows;
static gint tab_orientation;

SETUP_REC lookandfeel_options[] =
{
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_show_menubar, "toggle_show_menubar", GINT_TO_POINTER(TRUE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_show_toolbar, "toggle_show_toolbar", GINT_TO_POINTER(FALSE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_show_statusbar, "toggle_show_statusbar", GINT_TO_POINTER(TRUE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_show_nicklist, "toggle_show_nicklist", GINT_TO_POINTER(TRUE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_show_timestamps, "toggle_show_timestamps", GINT_TO_POINTER(FALSE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_show_msgs_timestamps, "toggle_show_msgs_timestamps", GINT_TO_POINTER(FALSE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_hide_text_style, "toggle_hide_text_style", GINT_TO_POINTER(FALSE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_bell_beeps, "toggle_bell_beeps", GINT_TO_POINTER(FALSE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_actlist_moves, "toggle_actlist_moves", GINT_TO_POINTER(FALSE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_privmsg_beeps, "toggle_privmsg_beeps", GINT_TO_POINTER(FALSE) },

    { SETUP_TYPE_TOGGLEBUTTON, &toggle_use_status_window, "toggle_use_status_window", GINT_TO_POINTER(FALSE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_use_msgs_window, "toggle_use_msgs_window", GINT_TO_POINTER(TRUE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_autoraise_msgs_window, "toggle_autoraise_msgs_window", GINT_TO_POINTER(FALSE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_autocreate_query, "toggle_autocreate_query", GINT_TO_POINTER(TRUE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_notifylist_popups, "toggle_notifylist_popups", GINT_TO_POINTER(FALSE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_use_tabbed_windows, "toggle_use_tabbed_windows", GINT_TO_POINTER(TRUE) },
    { SETUP_TYPE_INT_OBJECT,   &tab_orientation, "tab_orientation", GINT_TO_POINTER(3) },

    { 0, NULL, NULL }
};

/* irc proxy */
static gboolean toggle_use_ircproxy;
static gchar *proxy_address;
static gint proxy_port;
static gchar *proxy_string;

SETUP_REC ircproxy_options[] =
{
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_use_ircproxy, "toggle_use_ircproxy", GINT_TO_POINTER(FALSE) },
    { SETUP_TYPE_ENTRY, &proxy_address, "proxy_address", "" },
    { SETUP_TYPE_SPIN, &proxy_port, "proxy_port", GINT_TO_POINTER(6667) },
    { SETUP_TYPE_ENTRY, &proxy_string, "proxy_string", "CONNECT %s %d" },

    { 0, NULL, NULL }
};

/* dcc */
static gboolean toggle_dcc_autodisplay_dialog;
static gboolean toggle_dcc_autoget;
static gboolean toggle_dcc_autorename;
static gint dcc_file_create_mode;
static gint dcc_max_autoget_size;
static gchar *dcc_download_path;
static gchar *dcc_autoget_masks;
static gchar *dcc_autochat_masks;

static gboolean toggle_dcc_fast_send;
static gchar *dcc_upload_path;

static gboolean toggle_dcc_mirc_ctcp;
static gint dcc_block_size;
static gint dcc_port;
static gint dcc_timeout;

SETUP_REC dcc_options[] =
{
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_dcc_autorename, "toggle_dcc_autorename", GINT_TO_POINTER(FALSE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_dcc_autoget, "toggle_dcc_autoget", GINT_TO_POINTER(FALSE) },
    { SETUP_TYPE_SPIN, &dcc_max_autoget_size, "dcc_max_autoget_size", GINT_TO_POINTER(1000) },
    { SETUP_TYPE_ENTRY, &dcc_download_path, "dcc_download_path", "~" },
    { SETUP_TYPE_SPIN, &dcc_file_create_mode, "dcc_file_create_mode", GINT_TO_POINTER(644) },
    { SETUP_TYPE_ENTRY, &dcc_autoget_masks, "dcc_autoget_masks", "" },
    { SETUP_TYPE_ENTRY, &dcc_autochat_masks, "dcc_autochat_masks", "" },

    { SETUP_TYPE_TOGGLEBUTTON, &toggle_dcc_fast_send, "toggle_dcc_fast_send", GINT_TO_POINTER(TRUE) },
    { SETUP_TYPE_ENTRY, &dcc_upload_path, "dcc_upload_path", "~" },

    { SETUP_TYPE_TOGGLEBUTTON, &toggle_dcc_mirc_ctcp, "toggle_dcc_mirc_ctcp", GINT_TO_POINTER(FALSE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_dcc_autodisplay_dialog, "toggle_dcc_autodisplay_dialog", GINT_TO_POINTER(TRUE) },
    { SETUP_TYPE_SPIN, &dcc_block_size, "dcc_block_size", GINT_TO_POINTER(2048) },
    { SETUP_TYPE_SPIN, &dcc_port, "dcc_port", GINT_TO_POINTER(0) },
    { SETUP_TYPE_SPIN, &dcc_timeout, "dcc_timeout", GINT_TO_POINTER(300) },

    { 0, NULL, NULL }
};

/* appearance */
static gboolean toggle_buggy_gtkthemes;
static gboolean toggle_use_itext;
static gboolean toggle_background_transparent;

static gint panel_max_channels;

SETUP_REC appearance_options[] =
{
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_buggy_gtkthemes, "toggle_buggy_gtkthemes", GINT_TO_POINTER(FALSE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_use_itext, "toggle_use_itext", GINT_TO_POINTER(TRUE) },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_background_transparent, "toggle_background_transparent", GINT_TO_POINTER(FALSE) },

    { SETUP_TYPE_SPIN,         &panel_max_channels, "panel_max_channels", GINT_TO_POINTER(-1) },

    { 0, NULL, NULL }
};

void setup_set_data(SETUP_REC *rec, gchar *section, gchar *value)
{
    proplist_t sectprop, pkey, pvalue;

    switch (rec->type)
    {
        case SETUP_TYPE_TOGGLEBUTTON:
            {
                gboolean b = toupper(*value) == 'T' || toupper(*value) == 'Y';

                *((gboolean *) rec->data) = b;
            }
            break;
        case SETUP_TYPE_SPIN:
        case SETUP_TYPE_INT_OBJECT:
            {
                gint i;

                if (sscanf(value, "%d", &i) == 1)
                    *((gint *) rec->data) = i;
            }
            break;
        case SETUP_TYPE_ENTRY:
            {
                if (*((gchar **) rec->data) != NULL)
                    g_free(*((gchar **) rec->data));
                *((gchar **) rec->data) = g_strdup(value);
            }
            break;
    }

    /* First get the right section */
    pkey = PLMakeString(section);
    sectprop = PLGetDictionaryEntry(cprop, pkey);
    if (sectprop == NULL)
    {
	/* not there, create. */
	sectprop = PLMakeDictionaryFromEntries(NULL, NULL);
	cprop = PLInsertDictionaryEntry(cprop, pkey, sectprop);
    }
    PLRelease(pkey);

    /* add the data to dictionary */
    pkey = PLMakeString(rec->tag); pvalue = PLMakeString(value);
    sectprop = PLInsertDictionaryEntry(sectprop, pkey, pvalue);
    PLRelease(pkey); PLRelease(pvalue);
}

gchar *setup_tag_data(SETUP_REC *rec)
{
    static gchar digit[50];

    switch (rec->type)
    {
        case SETUP_TYPE_TOGGLEBUTTON:
            return *((gboolean *) rec->data) ? "yes" : "no";

        case SETUP_TYPE_ENTRY:
            return *((gchar **) rec->data);

        case SETUP_TYPE_SPIN:
            sprintf(digit, "%d", *((gint *) rec->data));
            return digit;

        default:
            return NULL;
    }
}

void setup_read(gchar *section, SETUP_REC *options)
{
    proplist_t sectprop, key;
    gint num;

    key = PLMakeString(section);
    sectprop = PLGetDictionaryEntry(cprop, key);
    PLRelease(key);

    for (num = 0; options[num].tag != NULL; num++)
    {
	/* add to settings hash table */
	if (g_hash_table_lookup(settings, options[num].tag) == NULL)
            g_hash_table_insert(settings, options[num].tag, options[num].data);

        switch (options[num].type)
        {
            case SETUP_TYPE_TOGGLEBUTTON:
                *((gboolean *) options[num].data) = config_get_bool(sectprop, options[num].tag, GPOINTER_TO_INT(options[num].def));
                break;
            case SETUP_TYPE_SPIN:
            case SETUP_TYPE_INT_OBJECT:
                *((gint *) options[num].data) = config_get_int(sectprop, options[num].tag, GPOINTER_TO_INT(options[num].def));
                break;
            case SETUP_TYPE_ENTRY:
                {
                    gchar *ret;

                    ret = config_get_str(sectprop, options[num].tag, (gchar *) options[num].def);
		    if (ret != NULL)
		    {
			if (*((gchar **) options[num].data) != NULL)
			    g_free(*((gchar **) options[num].data));
			*((gchar **) options[num].data) = g_strdup(ret);
		    }
                }
                break;
        }
    }
}

void setup_free(SETUP_REC *options)
{
    gint num;

    for (num = 0; options[num].tag != NULL; num++)
    {
	g_hash_table_remove(settings, options[num].tag);

	if (options[num].type == SETUP_TYPE_ENTRY)
        {
            if (*((gchar **) options[num].data) != NULL)
                g_free(*((gchar **) options[num].data));
        }
    }
}

void init_configfile(void)
{
    struct stat statbuf;
    proplist_t fname;
    gchar *config_file, *str;

    str = g_strdup_printf("%s/.irssi", g_get_home_dir());
    if (stat(str, &statbuf) != 0)
    {
	/* ~/.irssi not found, create it. */
	if (mkdir(str, 0700) != 0)
	    g_error("Couldn't create %s/.irssi directory", g_get_home_dir());
    }
    g_free(str);

    config_file = g_strdup_printf("%s/.irssi/config", g_get_home_dir());
    if (stat(config_file, &statbuf) != 0)
    {
	/* user configuration file not found, copy the default one from
	   sysconfdir */
	if (!copyfile(SYSCONFDIR"/irssi/config", config_file))
	{
	    /* no configuration file in sysconfdir .. use the build-in
	       configuration */
            FILE *f;
	    f = fopen(config_file, "w");
	    if (f != NULL)
	    {
		fwrite(default_config, strlen(default_config), 1, f);
		fclose(f);
	    }
	}
    }

    /* Open configuration file */
    cprop = PLGetProplistWithPath(config_file);

    if (cprop == NULL)
    {
	/* not found, create empty dictionary */
	cprop = PLMakeDictionaryFromEntries(NULL, NULL);

	fname = PLMakeString(config_file);
	cprop = PLSetFilename(cprop, fname);
	PLRelease(fname);
    }
    g_free(config_file);
}

gchar *setup_get_str(gchar *option)
{
    gchar **p;

    g_return_val_if_fail(option != NULL, NULL);

    p = g_hash_table_lookup(settings, option);
    if (p == NULL)
    {
	g_warning("setup_get_str() : %s not found", option);
	return NULL;
    }
    return *p;
}

gint setup_get_int(gchar *option)
{
    gint *p;

    g_return_val_if_fail(option != NULL, 0);

    p = g_hash_table_lookup(settings, option);
    if (p == NULL)
    {
	g_warning("setup_get_int() : %s not found", option);
	return 0;
    }
    return *p;
}

gboolean setup_get_bool(gchar *option)
{
    gboolean *p;

    g_return_val_if_fail(option != NULL, FALSE);

    p = g_hash_table_lookup(settings, option);
    if (p == NULL)
    {
	g_warning("setup_get_bool() : %s not found", option);
	return FALSE;
    }
    return *p;
}

gpointer setup_get_ptr(gchar *option)
{
    g_return_val_if_fail(option != NULL, NULL);

    return g_hash_table_lookup(settings, option);
}

void settings_init(void)
{
    settings = g_hash_table_new((GHashFunc) g_str_hash, (GCompareFunc) g_str_equal);

    init_configfile();

    setup_lists_init();

    setup_read("ircproxy", ircproxy_options);
    setup_read("dcc", dcc_options);
    setup_read("flood", flood_options);
    setup_read("history", history_options);
    setup_read("completion", completion_options);
    setup_read("misc", misc_options);
    setup_read("lookandfeel", lookandfeel_options);
    setup_read("appearance", appearance_options);
    setup_read("server", server_options);

    setup_servers_init();
    setup_ircnets_init();
    setup_channels_init();
    setup_themes_init();
    setup_keyboard_init();

    init_userinfo();
    signal_emit("startup settings read", 0);
}

void settings_deinit(void)
{
    setup_lists_deinit();

    setup_free(ircproxy_options);
    setup_free(dcc_options);
    setup_free(flood_options);
    setup_free(history_options);
    setup_free(completion_options);
    setup_free(misc_options);
    setup_free(lookandfeel_options);
    setup_free(appearance_options);
    setup_free(server_options);

    setup_keyboard_deinit();
    setup_themes_deinit();
    setup_channels_deinit();
    setup_ircnets_deinit();
    setup_servers_deinit();

    if (cprop != NULL) PLRelease(cprop);

    g_hash_table_destroy(settings);
}
