/*
 plugins.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "../nls.h"

GList *plugins;

static GModule *plugin_open(gchar *name)
{
    GModule *module;
    gchar *str, *path;

    if (*name == '/')
        str = g_strdup(name);
    else
    {
	str = g_module_build_path(PLUGINSDIR, name);
        module = g_module_open(str, G_MODULE_BIND_LAZY);
        g_free(str);
        if (module != NULL) return module;

	/* Plugin not found from global plugin dir, check from home dir */
        path = g_strdup_printf("%s/.irssi/plugins", g_get_home_dir());
	str = g_module_build_path(path, name);
	g_free(path);
    }

    module = g_module_open(str, G_MODULE_BIND_LAZY);
    g_free(str);
    return module;
}

static void plugin_menu_unload(gpointer widget, PLUGIN_REC *plugin)
{
    g_return_if_fail(plugin != NULL);

    plugin_destroy(plugin);
}

gboolean plugin_load(gchar *name, gchar *args)
{
    GModule *module;
    PLUGIN_REC *rec;
    gchar *path, *ptr;

    PLUGIN_INIT_FUNC plugin_init;
    PLUGIN_DESC_FUNC plugin_description;
    PLUGIN_VERSION_FUNC plugin_version;

    g_return_val_if_fail(name != NULL, FALSE);

    if (!g_module_supported())
	return FALSE;

    path = name;
    if (*name == '/')
    {
        ptr = strrchr(name, '/');
        if (ptr != NULL) name = ptr+1;
    }

    if (strncmp(name, "lib", 3) == 0)
        name += 3;

    name = g_strdup(name);
    ptr = strstr(name, ".so");
    if (ptr != NULL) *ptr = '\0';

    if (plugin_find(name))
    {
	/* plugin already loaded */
	signal_emit("plugin error already loaded", 1, name);
        g_free(name);
        return FALSE;
    }

    module = plugin_open(path);
    if (module == NULL)
    {
	/* Plugin not found.. */
	signal_emit("plugin error load", 2, name, g_module_error());
        g_free(name);
        return FALSE;
    }

    if (!g_module_symbol(module, "plugin_init", (gpointer *) &plugin_init) ||
	!g_module_symbol(module, "plugin_description", (gpointer *) &plugin_description) ||
	!g_module_symbol(module, "plugin_version", (gpointer *) &plugin_version))
    {
	signal_emit("plugin error invalid", 1, name);
	g_module_close(module);
        g_free(name);
        return FALSE;
    }

    /* check that version is right */
    if (plugin_version() != PLUGIN_LAYER_VERSION)
    {
	signal_emit("plugin error version", 1, name);
        g_module_close(module);
        g_free(name);
        return FALSE;
    }

    rec = g_new0(PLUGIN_REC, 1);
    plugins = g_list_append(plugins, rec);
    rec->module = module;
    rec->name = g_strdup(name);
    rec->description = g_strdup(plugin_description());
    g_free(name);

    signal_emit("plugin created", 1, rec);

    /* create unload menu item for all plugins */
    signal_emit("plugin add menu", 3, _("Unload"), plugin_menu_unload, rec);
    signal_emit("plugin add menu", 3, NULL, NULL, rec);

    args = g_strdup(args);
    if (!plugin_init(irssi_gui, rec, args))
    {
	plugins = g_list_remove(plugins, rec);
	signal_emit("plugin destroyed", 1, rec);

        g_free(args);
        g_module_close(module);
        g_free(rec->description);
        g_free(rec->name);
        g_free(rec);
        return FALSE;
    }
    g_free(args);

    signal_emit("plugin loaded", 1, rec);
    return TRUE;
}

gchar *plugin_get_description(gchar *name)
{
    GModule *module;
    gchar *str;

    PLUGIN_DESC_FUNC plugin_description;

    g_return_val_if_fail(name != NULL, FALSE);

    if (!g_module_supported()) return NULL;

    str = NULL;
    module = plugin_open(name);
    if (module != NULL)
    {
	if (g_module_symbol(module, "plugin_description", (gpointer *) &plugin_description))
	{
	    str = plugin_description();
	    if (str != NULL) str = g_strdup(str);
	}
	g_module_close(module);
    }
    return str;
}

void plugin_destroy(PLUGIN_REC *plugin)
{
    PLUGIN_DEINIT_FUNC plugin_deinit;

    g_return_if_fail(plugin != NULL);

    plugins = g_list_remove(plugins, plugin);

    g_module_symbol(plugin->module, "plugin_deinit", (gpointer *) &plugin_deinit);
    plugin_deinit(plugin);
    g_module_close(plugin->module);

    signal_emit("plugin destroyed", 1, plugin);

    g_free(plugin->name);
    if (plugin->description != NULL) g_free(plugin->description);
    g_free(plugin);
}

PLUGIN_REC *plugin_find(gchar *name)
{
    GList *tmp;

    g_return_val_if_fail(name != NULL, NULL);

    for (tmp = g_list_first(plugins); tmp != NULL; tmp = tmp->next)
    {
        PLUGIN_REC *rec = tmp->data;

        if (g_strcasecmp(name, rec->name) == 0)
            return rec;
    }

    return NULL;
}

static gboolean cmd_load(gchar *data)
{
    gchar *params, *cmd, *args;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &cmd, &args);
    if (*cmd == '\0') cmd_return_error(CMDERR_NOT_ENOUGH_PARAMS);
    plugin_load(cmd, args);
    g_free(params);
    return TRUE;
}

static gboolean cmd_unload(gchar *data)
{
    PLUGIN_REC *plugin;

    g_return_val_if_fail(data != NULL, FALSE);
    if (*data == '\0') cmd_return_error(CMDERR_NOT_ENOUGH_PARAMS);

    plugin = plugin_find(data);
    if (plugin != NULL)
	plugin_destroy(plugin);
    else
        signal_emit("plugin error not loaded", 1, data);

    return TRUE;
}

void plugins_init(void)
{
    command_bind("load", NULL, (SIGNAL_FUNC) cmd_load);
    command_bind("unload", NULL, (SIGNAL_FUNC) cmd_unload);
}

void plugins_deinit(void)
{
    command_unbind("load", (SIGNAL_FUNC) cmd_load);
    command_unbind("unload", (SIGNAL_FUNC) cmd_unload);

    while (plugins != NULL)
        plugin_destroy(plugins->data);
}
