/*

 ignore.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static gint ignore_tag;

static void autoignore_remove_rec(SERVER_REC *server, AUTOIGNORE_REC *rec)
{
    g_return_if_fail(server != NULL);
    g_return_if_fail(rec != NULL);

    signal_emit("autoignore remove", 2, server, rec);

    g_free(rec->nick);
    g_free(rec);

    server->ignorelist = g_list_remove(server->ignorelist, rec);
}

static AUTOIGNORE_REC *autoignore_find(SERVER_REC *server, gchar *mask)
{
    GList *tmp;

    g_return_val_if_fail(server != NULL, NULL);
    g_return_val_if_fail(mask != NULL, NULL);

    for (tmp = g_list_first(server->ignorelist); tmp != NULL; tmp = tmp->next)
    {
        AUTOIGNORE_REC *rec = tmp->data;

        if (g_strcasecmp(rec->nick, mask) == 0) return rec;
    }

    return NULL;
}

/* timeout function: unignore old ignores.. */
static void autoignore_timeout_server(SERVER_REC *server)
{
    GList *tmp, *next;
    time_t t;

    g_return_if_fail(server != NULL);

    t = time(NULL);
    t -= server->ignore_lastcheck;

    for (tmp = g_list_first(server->ignorelist); tmp != NULL; tmp = next)
    {
        AUTOIGNORE_REC *rec = tmp->data;

        next = tmp->next;
        if (rec->timeleft > t)
            rec->timeleft -= t;
        else
        {
            autoignore_remove_rec(server, rec);
        }
    }

    server->ignore_lastcheck = time(NULL);
}

static gint autoignore_timeout(void)
{
    g_list_foreach(servers, (GFunc) autoignore_timeout_server, NULL);
    return 1;
}

/* Initialize auto ignoring */
static gboolean autoignore_init_server(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    server->ignorelist = NULL;
    server->ignore_lastcheck = time(NULL)-AUTOIGNORE_TIMECHECK;

    return TRUE;
}

/* Deinitialize auto ignoring */
static gboolean autoignore_deinit_server(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    while (server->ignorelist)
        autoignore_remove_rec(server, (AUTOIGNORE_REC *) server->ignorelist->data);
    return TRUE;
}

void autoignore_add(SERVER_REC *server, gint type, gchar *nick)
{
    AUTOIGNORE_REC *rec;
    gint igtime;

    g_return_if_fail(server != NULL);
    g_return_if_fail(nick != NULL);

    igtime = setup_get_int("autoignore_time");
    if (igtime <= 0) return;

    rec = autoignore_find(server, nick);
    if (rec != NULL)
    {
        /* already being ignored */
        rec->level |= type;
        rec->timeleft = igtime;
        return;
    }

    rec = g_new(AUTOIGNORE_REC, 1);
    rec->nick = g_strdup(nick);
    rec->level = type;
    rec->timeleft = igtime;

    server->ignorelist = g_list_append(server->ignorelist, rec);

    signal_emit("autoignore new", 2, server, rec);
}

gboolean autoignore_remove(SERVER_REC *server, gchar *mask, gchar *level)
{
    AUTOIGNORE_REC *rec;
    gint type;

    g_return_val_if_fail(server != NULL, FALSE);
    g_return_val_if_fail(mask != NULL, FALSE);

    if (level == NULL)
        type = MSGLEVEL_ALL;
    else
    {
        type = level2bits(level);
        if (type == 0) type = MSGLEVEL_ALL;
    }

    rec = autoignore_find(server, mask);
    if (rec != NULL && (type & rec->level))
    {
        rec->level &= ~type;
        if (rec->level == 0) autoignore_remove_rec(server, rec);
        return TRUE;
    }

    return FALSE;
}

/* Check if nick is in ignore list */
gboolean ignore_check(SERVER_REC *server, gchar *nick, gchar *host, gint type)
{
    GList *tmp;
    gint ret;

    g_return_val_if_fail(nick != NULL, 0);

    if (server != NULL)
    {
        /* check autoignoring */
        AUTOIGNORE_REC *rec;

        rec = autoignore_find(server, nick);
        if (rec != NULL && type & rec->level) return TRUE;
    }

    for (tmp = g_list_first(ignores); tmp != NULL; tmp = tmp->next)
    {
        LIST_REC *rec = tmp->data;

	ret = ((host == NULL || *host == '\0')) ?
	    match_wildcards(rec->key, nick) :
	    irc_mask_match_address(rec->key, nick, host);

	if (ret)
        {
            if (type & level2bits(rec->value))
                return TRUE;
        }
    }

    return FALSE;
}

LIST_REC *ignore_add(gchar *mask, gchar *level)
{
    LIST_REC *rec;

    rec = list_find(ignores, mask);
    if (rec != NULL)
    {
        gchar *str;

        str = g_strconcat(rec->value, " ", level, NULL);
        list_set_value(rec, str);
        g_free(str);
        signal_emit("ignore change", 1, rec);
    }
    else
    {
        rec = list_add(&ignores, mask, level);
        signal_emit("ignore new", 1, rec);
    }

    return rec;
}

LIST_REC *ignore_remove_rec(LIST_REC *rec, gchar *level)
{
    guint oldlevel, newlevel;

    oldlevel = level2bits(level);
    if (oldlevel == 0) oldlevel = MSGLEVEL_ALL;

    newlevel = level2bits(rec->value) & (~oldlevel);
    if (newlevel == 0)
    {
        /* unignored everything */
        list_remove(&ignores, rec);
        signal_emit("ignore remove", 1, rec);
        return NULL;
    }

    /* unignore just some ignore levels.. */
    level = bits2level(newlevel);
    list_set_value(rec, level);
    g_free(level);
    signal_emit("ignore change", 1, rec);

    return rec;
}

LIST_REC *ignore_remove(gchar *mask, gchar *level)
{
    LIST_REC *rec;

    rec = list_find(ignores, mask);
    if (rec == NULL) return NULL;

    return ignore_remove_rec(rec, level);
}

void ignore_init(void)
{
    ignore_tag = gui_timeout_add(AUTOIGNORE_TIMECHECK, (GUITimeoutFunction) autoignore_timeout, NULL);

    signal_add("server connected", (SIGNAL_FUNC) autoignore_init_server);
    signal_add("server disconnected", (SIGNAL_FUNC) autoignore_deinit_server);
}

void ignore_deinit(void)
{
    gui_timeout_remove(ignore_tag);
    signal_remove("server connected", (SIGNAL_FUNC) autoignore_init_server);
    signal_remove("server disconnected", (SIGNAL_FUNC) autoignore_deinit_server);
}
