/*
 gui-events-list.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

typedef struct
{
    gchar *channel;
    gchar *users;
    gchar *topic;
}
CMDLIST_REC;

static void sig_join(GtkCList *clist)
{
    SERVER_REC *server;
    GList *list;
    gchar *servertag, *channel;

    g_return_if_fail(clist != NULL);

    servertag = gtk_object_get_data(GTK_OBJECT(clist), "server");
    list = gtk_object_get_data(GTK_OBJECT(clist), "selections");

    server = server_find_tag(servertag);
    if (server == NULL)
    {
        gui_dialog(DIALOG_ERROR, _("Not connected to server '%s' anymore"), servertag);
        return;
    }

    for (; list != NULL; list = list->next)
    {
        gtk_clist_get_text(clist, GPOINTER_TO_INT(list->data), 0, &channel);
        if (channel != NULL)
            signal_emit("command join", 3, channel, server, cur_channel);
    }

    gtk_clist_unselect_all(clist);
}

static void sig_select_row(GtkCList *clist, gint row)
{
    GList *list;

    g_return_if_fail(clist != NULL);

    list = gtk_object_get_data(GTK_OBJECT(clist), "selections");
    list = g_list_append(list, GINT_TO_POINTER(row));
    gtk_object_set_data(GTK_OBJECT(clist), "selections", list);
}

static void sig_unselect_row(GtkCList *clist, gint row)
{
    GList *list;

    g_return_if_fail(clist != NULL);

    list = gtk_object_get_data(GTK_OBJECT(clist), "selections");
    list = g_list_remove(list, GINT_TO_POINTER(row));
    gtk_object_set_data(GTK_OBJECT(clist), "selections", list);
}

static void clist_click_column(GtkCList *clist, gint column)
{
    if (column != clist->sort_column)
	gtk_clist_set_sort_column(clist, column);
    else
    {
	if (clist->sort_type == GTK_SORT_ASCENDING)
	    clist->sort_type = GTK_SORT_DESCENDING;
	else
	    clist->sort_type = GTK_SORT_ASCENDING;
    }

    gtk_clist_sort(clist);
}

static gboolean event_list(gchar *data, SERVER_REC *server)
{
    gchar *params, *channel, *users, *topic;
    GUI_SERVER_REC *gui;
    CMDLIST_REC *rec;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(server != NULL, FALSE);

    params = event_get_params(data, 4, NULL, &channel, &users, &topic);
    gui = SERVER_GUI(server);

    /* add row to list */
    rec = g_new(CMDLIST_REC, 1);
    gui->list = g_list_append(gui->list, rec);

    rec->channel = g_strdup(channel);
    rec->users = g_strdup(users);
    rec->topic = g_strdup(topic);

    g_free(params);
    return FALSE;
}

static void list_free(CMDLIST_REC *rec)
{
    g_free(rec->channel);
    g_free(rec->topic);
    g_free(rec);
}

static void sig_destroy(GtkWidget *dialog)
{
    GList *list;

    list = gtk_object_get_data(GTK_OBJECT(dialog), "list");
    g_list_foreach(list, (GFunc) list_free, NULL);
    g_list_free(list);
}

static void sig_search(GtkEntry *entry, GtkWidget *dialog)
{
    gchar *text, *texts[3];
    GtkCList *clist;
    GList *list;

    text = gtk_entry_get_text(entry);

    /* redraw the list */
    clist = gtk_object_get_data(GTK_OBJECT(dialog), "clist");
    gtk_clist_freeze(clist);
    gtk_clist_clear(clist);

    list = gtk_object_get_data(GTK_OBJECT(dialog), "list");
    for (; list != NULL; list = list->next)
    {
	CMDLIST_REC *rec = list->data;

	if (*text != '\0' &&
	    stristr(rec->channel, text) == NULL &&
	    stristr(rec->users, text) == NULL &&
	    stristr(rec->topic, text) == NULL)
	    continue;

	/* add record to list */
	texts[0] = rec->channel;
	texts[1] = rec->users;
	texts[2] = rec->topic;

	gtk_clist_append(clist, texts);

    }
    gtk_clist_thaw(clist);
}

static gboolean event_end_of_list(gchar *data, SERVER_REC *server)
{
    GtkWidget *dialog, *label, *entry, *clist, *scrollbox;
    gchar *texts[3], *servertag, *str;
    GUI_SERVER_REC *gui;
    GList *tmp;

    g_return_val_if_fail(server != NULL, FALSE);

    gui = SERVER_GUI(server);
    if (gui->list == NULL)
    {
	gui_dialog(DIALOG_OK, _("No channels found for /LIST"));
	return TRUE;
    }

    servertag = g_strdup(server->tag);
    dialog = gnome_dialog_new(PACKAGE, "Join", GNOME_STOCK_BUTTON_CLOSE, NULL);
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, TRUE, FALSE);
    gtk_widget_set_usize(dialog, 500, 350);

    /* Create clist widget */
    scrollbox = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
				   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), scrollbox, TRUE, TRUE, 0);

    texts[0] = _("Channel"); texts[1] = _("Users"); texts[2] = _("Topic");
    clist = gtk_clist_new_with_titles(3, texts);
    gtk_container_add(GTK_CONTAINER(scrollbox), clist);
    gtk_signal_connect_object(GTK_OBJECT(clist), "destroy",
			      GTK_SIGNAL_FUNC(g_free), (GtkObject *) servertag);
    gtk_signal_connect_object(GTK_OBJECT(clist), "destroy",
			      GTK_SIGNAL_FUNC(sig_destroy), GTK_OBJECT(dialog));
    gtk_signal_connect(GTK_OBJECT(clist), "select_row",
		       GTK_SIGNAL_FUNC(sig_select_row), NULL);
    gtk_signal_connect(GTK_OBJECT(clist), "unselect_row",
		       GTK_SIGNAL_FUNC(sig_unselect_row), NULL);
    gtk_signal_connect(GTK_OBJECT(clist), "click_column",
		       GTK_SIGNAL_FUNC(clist_click_column), NULL);
    gtk_object_set_data(GTK_OBJECT(dialog), "clist", clist);
    gtk_object_set_data(GTK_OBJECT(dialog), "list", gui->list);
    gtk_object_set_data(GTK_OBJECT(clist), "server", servertag);

    gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_EXTENDED);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 2, TRUE);

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(sig_join), GTK_OBJECT(clist));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));
    gtk_widget_show_all(dialog);

    /* add the records to list */
    gtk_clist_freeze(GTK_CLIST(clist));
    for (tmp = gui->list; tmp != NULL; tmp = tmp->next)
    {
	CMDLIST_REC *rec = tmp->data;

	texts[0] = rec->channel;
	texts[1] = rec->users;
	texts[2] = rec->topic;

	gtk_clist_append(GTK_CLIST(clist), texts);
    }
    gtk_clist_thaw(GTK_CLIST(clist));

    /* Create the search widget */
    entry = gui_create_labelentry(GNOME_DIALOG(dialog)->vbox,
				  _("Search"), NULL, FALSE);
    gtk_signal_connect(GTK_OBJECT(entry), "changed",
		       GTK_SIGNAL_FUNC(sig_search), dialog);
    gtk_widget_grab_focus(entry);

    /* Display total number of channels */
    str = g_strdup_printf(_("Total of %d channels"), GTK_CLIST(clist)->rows);
    label = gtk_label_new(str);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), label, FALSE, FALSE, 0);
    g_free(str);

    /* Make columns resizeable */
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 0, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 1, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 2, TRUE);
    gtk_widget_show_all(dialog);

    gui->list = NULL;
    return FALSE;
}

static gboolean signal_server_disconnected(SERVER_REC *server)
{
    if (SERVER_GUI(server)->list != NULL)
	event_end_of_list(NULL, server);
    return TRUE;
}

static gboolean cmd_list(gchar *data, SERVER_REC *server)
{
    gchar *params, *args, *str;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 2 | PARAM_FLAG_OPTARGS | PARAM_FLAG_GETREST, &args, &str);

    if (*str == '\0' && stristr(args, "-yes") == NULL)
	cmd_param_error(CMDERR_NOT_GOOD_IDEA);

    if (stristr(args, "-nogui") != NULL)
    {
	g_free(params);
	return TRUE;
    }

    /* Use GUI list */
    server_redirect_event(server, NULL, 1,
			  "event 323", "gui event 323", -1,
			  "event 321", "event empty", -1,
			  "event 322", "gui event 322", -1, NULL);

    str = g_strdup_printf("LIST %s", str);
    irc_send_cmd(server, str);
    g_free(str);

    g_free(params);
    return FALSE;
}

void gui_event_list_init(void)
{
    command_bind("list", "GUI commands", (SIGNAL_FUNC) cmd_list);
    signal_add("gui event 322", (SIGNAL_FUNC) event_list);
    signal_add("gui event 323", (SIGNAL_FUNC) event_end_of_list);
    signal_add("server disconnected", (SIGNAL_FUNC) signal_server_disconnected);
}

void gui_event_list_deinit(void)
{
    command_unbind("list", (SIGNAL_FUNC) cmd_list);
    signal_remove("gui event 322", (SIGNAL_FUNC) event_list);
    signal_remove("gui event 323", (SIGNAL_FUNC) event_end_of_list);
    signal_remove("server disconnected", (SIGNAL_FUNC) signal_server_disconnected);
}
